#!/bin/bash
#
# Copyright (c) 2012, 2013, Oracle and/or its affiliates. All rights reserved.
# DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
#
# This code is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 2 only, as
# published by the Free Software Foundation.
#
# This code is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# version 2 for more details (a copy is included in the LICENSE file that
# accompanied this code).
#
# You should have received a copy of the GNU General Public License version
# 2 along with this work; if not, write to the Free Software Foundation,
# Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
# or visit www.oracle.com if you need additional information or have any
# questions.
#

# This script is processed by configure before it's usable. It is run from 
# the root of the build directory.


##########################################################################################

# Check that we are run via the wrapper generated by configure
if [ -z "$SRC_ROOT" ]; then
    echo "Error: You must run this script using build/[conf]/compare.sh"
    exit 1
fi

if [ "$OPENJDK_TARGET_OS" = "macosx" ]; then
    FULLDUMP_CMD="$OTOOL -v -V -h -X -t -d"
    LDD_CMD="$OTOOL -L"
    DIS_CMD="$OTOOL -v -t"
    STAT_PRINT_SIZE="-f %z"
elif [ "$OPENJDK_TARGET_OS" = "windows" ]; then
    FULLDUMP_CMD="$DUMPBIN -all"
    LDD_CMD="$DUMPBIN -dependants | $GREP .dll"
    DIS_CMD="$DUMPBIN -disasm:nobytes"
    STAT_PRINT_SIZE="-c %s"
else
    FULLDUMP_CMD="$READELF -a"
    LDD_CMD="$LDD"
    DIS_CMD="$OBJDUMP -d"
    STAT_PRINT_SIZE="-c %s"
fi

UNARCHIVE="$UNZIP -q"

COMPARE_EXCEPTIONS_INCLUDE="$SRC_ROOT/common/bin/compare_exceptions.sh.incl"
if [ ! -e "$COMPARE_EXCEPTIONS_INCLUDE" ]; then
    echo "Error: Cannot locate the exceptions file, it should have been here: $COMPARE_EXCEPTIONS_INCLUDE"
    exit 1
fi
# Include exception definitions
. "$COMPARE_EXCEPTIONS_INCLUDE"

##########################################################################################
# Compare text files and ignore specific differences:
#
#  * Timestamps in Java sources generated by idl2java
#  * Sorting order and cleanup style in .properties files

diff_text() {
    OTHER_FILE=$1
    THIS_FILE=$2

    SUFFIX="${THIS_FILE##*.}"

    TMP=1

    if [[ "$THIS_FILE" = *"META-INF/MANIFEST.MF" ]]; then
        TMP=$(LC_ALL=C $DIFF $OTHER_FILE $THIS_FILE | \
            $GREP '^[<>]' | \
            $SED -e '/[<>] Ant-Version: Apache Ant .*/d' \
	         -e '/[<>] Created-By: .* (Oracle Corporation).*/d')
    fi
    if test "x$SUFFIX" = "xjava"; then
        TMP=$(LC_ALL=C $DIFF $OTHER_FILE $THIS_FILE | \
            $GREP '^[<>]' | \
            $SED -e '/[<>] \* from.*\.idl/d' \
                 -e '/[<>] \*.*[0-9]\{4\}_[0-9]\{2\}_[0-9]\{2\}_[0-9]\{2\}_[0-9]\{2\}-b[0-9]\{2\}.*/d' \
                 -e '/[<>] \*.*[0-9]\{4\} [0-9][0-9]*:[0-9]\{2\}:[0-9]\{2\}.*/d' \
                 -e '/\/\/ Generated from input file.*/d' \
                 -e '/\/\/ This file was generated AUTOMATICALLY from a template file.*/d' \
                 -e '/\/\/ java GenerateCharacter.*/d')
    fi
    # Ignore date strings in class files.
    # On Macosx the system sources for generated java classes produce different output on 
    # consequtive invokations seemingly randomly.
    # For example a method parameter randomly named "thePoint" or "aPoint". Ignore this.
    if test "x$SUFFIX" = "xclass"; then
        # To improve performance when large diffs are found, do a rough filtering of classes
        # elibeble for these exceptions
        if $GREP -R -e '[0-9]\{4\}_[0-9]\{2\}_[0-9]\{2\}_[0-9]\{2\}_[0-9]\{2\}-b[0-9]\{2\}' \
	        -e '[0-9]\{2\}/[0-9]\{2\}/[0-9]\{4\}' \
	        -e thePoint -e aPoint -e setItemsPtr ${THIS_FILE} > /dev/null; then
            $JAVAP -c -constants -l -p ${OTHER_FILE} >  ${OTHER_FILE}.javap
            $JAVAP -c -constants -l -p ${THIS_FILE} > ${THIS_FILE}.javap
            TMP=$($DIFF ${OTHER_FILE}.javap ${THIS_FILE}.javap | \
                $GREP '^[<>]' | \
                $SED -e '/[<>].*[0-9]\{4\}_[0-9]\{2\}_[0-9]\{2\}_[0-9]\{2\}_[0-9]\{2\}-b[0-9]\{2\}.*/d' \
		     -e '/[0-9]\{2\}\/[0-9]\{2\}\/[0-9]\{4\}/d' \
 	             -e '/[<>].*Point   Lcom\/apple\/jobjc\/foundation\/NSPoint;/d' \
	             -e '/[<>].*public com\.apple\.jobjc\.Pointer<com\.apple\.jobjc\..*itemsPtr();/d' \
	             -e '/[<>].*public void setItemsPtr(com\.apple\.jobjc\.Pointer<com\.apple\.jobjc\..*);/d')
        fi
    fi
    if test "x$SUFFIX" = "xproperties"; then
        # Run through nawk to add possibly missing newline at end of file.
        $CAT $OTHER_FILE | $NAWK '{ print }' | LC_ALL=C $SORT > $OTHER_FILE.cleaned
# Disable this exception since we aren't changing the properties cleaning method yet.
#        $CAT $OTHER_FILE | $SED -e 's/\([^\\]\):/\1\\:/g' -e  's/\([^\\]\)=/\1\\=/g' -e 's/#.*/#/g' \
#            | $SED -f "$SRC_ROOT/common/makefiles/support/unicode2x.sed" \
#  	    | $SED -e '/^#/d' -e '/^$/d' \
#            -e :a -e '/\\$/N; s/\\\n//; ta' \
#  	    -e 's/^[ \t]*//;s/[ \t]*$//' \
#	    -e 's/\\=/=/' | LC_ALL=C $SORT > $OTHER_FILE.cleaned
        TMP=$(LC_ALL=C $DIFF $OTHER_FILE.cleaned $THIS_FILE)
    fi
    if test -n "$TMP"; then
        echo Files $OTHER_FILE and $THIS_FILE differ
        return 1
    fi

    return 0
}

##########################################################################################
# Compare directory structure

compare_dirs() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    mkdir -p $WORK_DIR

    (cd $OTHER_DIR && $FIND . -type d | $SORT > $WORK_DIR/dirs_other)
    (cd $THIS_DIR && $FIND . -type d | $SORT > $WORK_DIR/dirs_this)

    $DIFF $WORK_DIR/dirs_other $WORK_DIR/dirs_other > $WORK_DIR/dirs_diff
    
    echo -n Directory structure...
    if [ -s $WORK_DIR/dirs_diff ]; then
        echo Differences found.
        REGRESSIONS=true
        # Differences in directories found.
        ONLY_OTHER=$($GREP '<' $WORK_DIR/dirs_diff)
        if [ "$ONLY_OTHER" ]; then
            echo Only in $OTHER
            $GREP '<' $WORK_DIR/dirs_diff | $SED 's|< ./|    |g'
        fi
        ONLY_THIS=$($GREP '>' $WORK_DIR/dirs_diff)
        if [ "$ONLY_THIS" ]; then
            echo Only in $THIS
            $GREP '>' $WORK_DIR/dirs_diff | $SED 's|> ./|    |g'
        fi
    else
        echo Identical!
    fi
}


##########################################################################################
# Compare file structure

compare_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    $MKDIR -p $WORK_DIR

    (cd $OTHER_DIR && $FIND . ! -type d | $SORT > $WORK_DIR/files_other)
    (cd $THIS_DIR && $FIND . ! -type d | $SORT > $WORK_DIR/files_this)
    
    $DIFF $WORK_DIR/files_other $WORK_DIR/files_this > $WORK_DIR/files_diff

    echo -n File names...
    if [ -s $WORK_DIR/files_diff ]; then
        echo Differences found.
        REGRESSIONS=true
        # Differences in files found.
        ONLY_OTHER=$($GREP '<' $WORK_DIR/files_diff)
        if [ "$ONLY_OTHER" ]; then
            echo Only in $OTHER
            $GREP '<' $WORK_DIR/files_diff | $SED 's|< ./|    |g'
        fi
        ONLY_THIS=$($GREP '>' $WORK_DIR/files_diff)
        if [ "$ONLY_THIS" ]; then
            echo Only in $THIS
            $GREP '>' $WORK_DIR/files_diff | $SED 's|> ./|    |g'
        fi
    else
        echo Identical!
    fi
}


##########################################################################################
# Compare permissions

compare_permissions() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    mkdir -p $WORK_DIR

    echo -n Permissions...
    found=""
    for f in `cd $OTHER_DIR && $FIND . -type f`
    do
        if [ ! -f ${OTHER_DIR}/$f ]; then continue; fi
        if [ ! -f ${THIS_DIR}/$f ]; then continue; fi
        OP=`ls -l ${OTHER_DIR}/$f | awk '{printf("%.10s\n", $1);}'`
        TP=`ls -l ${THIS_DIR}/$f | awk '{printf("%.10s\n", $1);}'`
        if [ "$OP" != "$TP" ]
        then
	    if [ -z "$found" ]; then echo ; found="yes"; fi
	    $PRINTF "\told: ${OP} new: ${TP}\t$f\n"
        fi
    done
    if [ -z "$found" ]; then 
        echo "Identical!"
    else
        REGRESSIONS=true
    fi
}

##########################################################################################
# Compare file command output

compare_file_types() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    $MKDIR -p $WORK_DIR

    echo -n File types...
    found=""
    for f in `cd $OTHER_DIR && $FIND . ! -type d`
    do
        if [ ! -f ${OTHER_DIR}/$f ]; then continue; fi
        if [ ! -f ${THIS_DIR}/$f ]; then continue; fi
        OF=`cd ${OTHER_DIR} && $FILE -h $f`
        TF=`cd ${THIS_DIR} && $FILE -h $f`
        if [ "$f" = "./src.zip" ] || [[ "$f" = *"/Home/src.zip" ]] || [[ "$f" = *"/lib/JObjC.jar" ]]
        then
	    if [ "`echo $OF | $GREP -ic zip`" -gt 0 -a "`echo $TF | $GREP -ic zip`" -gt 0 ]
	    then
	        # the way we produces zip-files make it so that directories are stored in old file
	        # but not in new (only files with full-path)
	        # this makes file-5.09 report them as different
	        continue;
	    fi
        fi
        
        if [ "$OF" != "$TF" ]
        then
	    if [ -z "$found" ]; then echo ; found="yes"; fi
	    $PRINTF "\tother: ${OF}\n\tthis : ${TF}\n"
        fi
    done
    if [ -z "$found" ]; then 
        echo "Identical!"
    else
        REGRESSIONS=true
    fi
}

##########################################################################################
# Compare the rest of the files

compare_general_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3
    
    GENERAL_FILES=$(cd $THIS_DIR && $FIND . -type f ! -name "*.so" ! -name "*.jar" ! -name "*.zip" \
        ! -name "*.debuginfo" ! -name "*.dylib" ! -name "jexec" \
        ! -name "ct.sym" ! -name "*.diz" ! -name "*.dll" \
        ! -name "*.pdb" ! -name "*.exp" ! -name "*.ilk" \
        ! -name "*.lib" ! -name "*.war" ! -name "JavaControlPanel" \
        | $GREP -v "./bin/"  | $SORT | $FILTER)

    echo General files...
    for f in $GENERAL_FILES
    do
        if [ -e $OTHER_DIR/$f ]; then
            SUFFIX="${f##*.}"
            if [ "$(basename $f)" = "release" ]; then
                # Ignore differences in change numbers in release file.
                OTHER_FILE=$WORK_DIR/$f.other
                THIS_FILE=$WORK_DIR/$f.this
                $MKDIR -p $(dirname $OTHER_FILE)
                $MKDIR -p $(dirname $THIS_FILE)
                $CAT $OTHER_DIR/$f | $SED 's/\:[0-9a-f]\{12,12\}/:CHANGE/g' > $OTHER_FILE
                $CAT $THIS_DIR/$f  | $SED 's/\:[0-9a-f]\{12,12\}/:CHANGE/g' > $THIS_FILE
            elif [ "x$SUFFIX" = "xhtml" ]; then
                # Ignore time stamps in docs files
                OTHER_FILE=$WORK_DIR/$f.other
                THIS_FILE=$WORK_DIR/$f.this
                $MKDIR -p $(dirname $OTHER_FILE)
                $MKDIR -p $(dirname $THIS_FILE)
                #Note that | doesn't work on mac sed.
                $CAT $OTHER_DIR/$f | $SED -e 's/\(-- Generated by javadoc \).*\( --\)/\1(removed)\2/' \
                                          -e 's/\(<meta name="date" content="\).*\(">\)/\1(removed)\2/' \
                                          -e 's/[A-Z][a-z]*, [A-Z][a-z]* [0-9][0-9]*, [12][0-9]* [0-9][0-9:]* [AMP]\{2,2\} [A-Z][A-Z]*/(removed)/' \
                                          -e 's/[A-Z][a-z]* [A-Z][a-z]* [0-9][0-9] [0-9][0-9:]* [A-Z][A-Z]* [12][0-9]*/(removed)/' \
                                          -e 's/^\( from \).*\(\.idl\)$/\1(removed)\2/' \
                    > $OTHER_FILE
                $CAT $THIS_DIR/$f  | $SED -e 's/\(-- Generated by javadoc \).*\( --\)/\1(removed)\2/' \
                                          -e 's/\(<meta name="date" content="\).*\(">\)/\1(removed)\2/' \
                                          -e 's/[A-Z][a-z]*, [A-Z][a-z]* [0-9][0-9]*, [12][0-9]* [0-9][0-9:]* [AMP]\{2,2\} [A-Z][A-Z]*/(removed)/' \
                                          -e 's/[A-Z][a-z]* [A-Z][a-z]* [0-9][0-9] [0-9][0-9:]* [A-Z][A-Z]* [12][0-9]*/(removed)/' \
                                          -e 's/^\( from \).*\(\.idl\)$/\1(removed)\2/' \
                    > $THIS_FILE
            else
                OTHER_FILE=$OTHER_DIR/$f
                THIS_FILE=$THIS_DIR/$f
            fi
            DIFF_OUT=$($DIFF $OTHER_FILE $THIS_FILE 2>&1)
            if [ -n "$DIFF_OUT" ]; then
                echo $f
                REGRESSIONS=true
                if [ "$SHOW_DIFFS" = "true" ]; then
                    echo "$DIFF_OUT"
                fi
            fi
        fi
    done


}

##########################################################################################
# Compare zip file

compare_zip_file() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3
    ZIP_FILE=$4
    # Optionally provide different name for other zipfile
    OTHER_ZIP_FILE=$5

    THIS_ZIP=$THIS_DIR/$ZIP_FILE
    if [ -n "$OTHER_ZIP_FILE" ]; then
        OTHER_ZIP=$OTHER_DIR/$OTHER_ZIP_FILE
    else
        OTHER_ZIP=$OTHER_DIR/$ZIP_FILE
    fi

    THIS_SUFFIX="${THIS_ZIP##*.}"
    OTHER_SUFFIX="${OTHER_ZIP##*.}"
    if [ "$THIS_SUFFIX" != "$OTHER_SUFFIX" ]; then
        echo The files do not have the same suffix type!
        return 2
    fi

    TYPE="$THIS_SUFFIX"

    if $CMP $OTHER_ZIP $THIS_ZIP > /dev/null
    then
        return 0
    fi
    # Not quite identical, the might still contain the same data.
    # Unpack the jar/zip files in temp dirs
    
    THIS_UNZIPDIR=$WORK_DIR/$ZIP_FILE.this
    OTHER_UNZIPDIR=$WORK_DIR/$ZIP_FILE.other
    $RM -rf $THIS_UNZIPDIR $OTHER_UNZIPDIR
    $MKDIR -p $THIS_UNZIPDIR
    $MKDIR -p $OTHER_UNZIPDIR
    (cd $THIS_UNZIPDIR && $UNARCHIVE $THIS_ZIP)
    (cd $OTHER_UNZIPDIR && $UNARCHIVE $OTHER_ZIP)

    # Find all archives inside and unzip them as well to compare the contents rather than
    # the archives. pie.jar.pack.gz i app3.war is corrupt, skip it.
    EXCEPTIONS="pie.jar.pack.gz"
    for pack in $($FIND $THIS_UNZIPDIR \( -name "*.pack" -o -name "*.pack.gz" \) -a ! -name pie.jar.pack.gz); do
        ($UNPACK200 $pack $pack.jar)
        # Filter out the unzipped archives from the diff below.
        EXCEPTIONS="$EXCEPTIONS $pack $pack.jar"
    done
    for pack in $($FIND $OTHER_UNZIPDIR \( -name "*.pack" -o -name "*.pack.gz" \) -a ! -name pie.jar.pack.gz); do
        ($UNPACK200 $pack $pack.jar)
        EXCEPTIONS="$EXCEPTIONS $pack $pack.jar"
    done
    for zip in $($FIND $THIS_UNZIPDIR -name "*.jar" -o -name "*.zip"); do
        $MKDIR $zip.unzip
        (cd $zip.unzip && $UNARCHIVE $zip)
        EXCEPTIONS="$EXCEPTIONS $zip"
    done
    for zip in $($FIND $OTHER_UNZIPDIR -name "*.jar" -o -name "*.zip"); do
        $MKDIR $zip.unzip
        (cd $zip.unzip && $UNARCHIVE $zip)
        EXCEPTIONS="$EXCEPTIONS $zip"
    done

    CONTENTS_DIFF_FILE=$WORK_DIR/$ZIP_FILE.diff
    # On solaris, there is no -q option.
    if [ "$OPENJDK_TARGET_OS" = "solaris" ]; then
        LC_ALL=C $DIFF -r $OTHER_UNZIPDIR $THIS_UNZIPDIR \
            | $GREP -v -e "^<" -e "^>" -e "^Common subdirectories:" \
            > $CONTENTS_DIFF_FILE
    else
        LC_ALL=C $DIFF -rq $OTHER_UNZIPDIR $THIS_UNZIPDIR > $CONTENTS_DIFF_FILE
    fi

    ONLY_OTHER=$($GREP "^Only in $OTHER_UNZIPDIR" $CONTENTS_DIFF_FILE)
    ONLY_THIS=$($GREP "^Only in $THIS_UNZIPDIR" $CONTENTS_DIFF_FILE)

    return_value=0

    if [ -n "$ONLY_OTHER" ]; then
        echo "        Only OTHER $ZIP_FILE contains:"
        echo "$ONLY_OTHER" | sed "s|Only in $OTHER_UNZIPDIR|            |"g | sed 's|: |/|g'
        return_value=1
    fi

    if [ -n "$ONLY_THIS" ]; then
        echo "        Only THIS $ZIP_FILE contains:"
        echo "$ONLY_THIS" | sed "s|Only in $THIS_UNZIPDIR|            |"g | sed 's|: |/|g'
        return_value=1
    fi

    if [ "$OPENJDK_TARGET_OS" = "solaris" ]; then
        DIFFING_FILES=$($GREP -e "differ$" -e "^diff " $CONTENTS_DIFF_FILE \
            | $CUT -f 3 -d ' ' | $SED "s|$OTHER_UNZIPDIR/||g")
    else
        DIFFING_FILES=$($GREP -e "differ$" $CONTENTS_DIFF_FILE \
            | $CUT -f 2 -d ' ' | $SED "s|$OTHER_UNZIPDIR/||g")
    fi

    $RM -f $WORK_DIR/$ZIP_FILE.diffs
    for file in $DIFFING_FILES; do
	if [[ "$ACCEPTED_JARZIP_CONTENTS $EXCEPTIONS" != *"$file"* ]]; then
            diff_text $OTHER_UNZIPDIR/$file $THIS_UNZIPDIR/$file >> $WORK_DIR/$ZIP_FILE.diffs
	fi
    done

    if [ -s "$WORK_DIR/$ZIP_FILE.diffs" ]; then
        return_value=1
        echo "        Differing files in $ZIP_FILE"
        $CAT $WORK_DIR/$ZIP_FILE.diffs | $GREP differ | cut -f 2 -d ' ' | \
            $SED "s|$OTHER_UNZIPDIR|            |g" > $WORK_DIR/$ZIP_FILE.difflist
        $CAT $WORK_DIR/$ZIP_FILE.difflist

        if [ -n "$SHOW_DIFFS" ]; then
            for i in $(cat $WORK_DIR/$ZIP_FILE.difflist) ; do
                if [ -f "${OTHER_UNZIPDIR}/$i.javap" ]; then
                    LC_ALL=C $DIFF ${OTHER_UNZIPDIR}/$i.javap ${THIS_UNZIPDIR}/$i.javap
                elif [ -f "${OTHER_UNZIPDIR}/$i.cleaned" ]; then
                    LC_ALL=C $DIFF ${OTHER_UNZIPDIR}/$i.cleaned ${THIS_UNZIPDIR}/$i
                else
                    LC_ALL=C $DIFF ${OTHER_UNZIPDIR}/$i ${THIS_UNZIPDIR}/$i
                fi
            done
        fi
    fi

    return $return_value
}


##########################################################################################
# Compare all zip files

compare_all_zip_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    ZIPS=$(cd $THIS_DIR && $FIND . -type f -name "*.zip" | $SORT | $FILTER )

    if [ -n "$ZIPS" ]; then
        echo Zip files...

        return_value=0
        for f in $ZIPS; do
            if [ -f "$OTHER_DIR/$f" ]; then
                compare_zip_file $THIS_DIR $OTHER_DIR $WORK_DIR $f
                if [ "$?" != "0" ]; then
                    return_value=1
                    REGRESSIONS=true
                fi
            fi
        done
    fi

    return $return_value
}

##########################################################################################
# Compare all jar files

compare_all_jar_files() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    # TODO filter?
    ZIPS=$(cd $THIS_DIR && $FIND . -type f -name "*.jar" -o -name "*.war" | $SORT | $FILTER)

    if [ -n "$ZIPS" ]; then
        echo Jar files...

        return_value=0
        for f in $ZIPS; do
            if [ -f "$OTHER_DIR/$f" ]; then
                compare_zip_file $THIS_DIR $OTHER_DIR $WORK_DIR $f
                if [ "$?" != "0" ]; then
                    return_value=1
                    REGRESSIONS=true
                fi
            fi
        done
    fi

    return $return_value
}

##########################################################################################
# Compare binary (executable/library) file

compare_bin_file() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3
    BIN_FILE=$4

    THIS_FILE=$THIS_DIR/$BIN_FILE
    OTHER_FILE=$OTHER_DIR/$BIN_FILE
    NAME=$(basename $BIN_FILE)
    WORK_FILE_BASE=$WORK_DIR/$BIN_FILE
    FILE_WORK_DIR=$(dirname $WORK_FILE_BASE)

    $MKDIR -p $FILE_WORK_DIR

    ORIG_THIS_FILE="$THIS_FILE"
    ORIG_OTHER_FILE="$OTHER_FILE"

    if [[ "$STRIP_BEFORE_COMPARE" = *"$BIN_FILE"* ]]; then
        THIS_STRIPPED_FILE=$FILE_WORK_DIR/this/$NAME
        OTHER_STRIPPED_FILE=$FILE_WORK_DIR/other/$NAME
        $MKDIR -p $FILE_WORK_DIR/this $FILE_WORK_DIR/other
        $CP $THIS_FILE $THIS_STRIPPED_FILE
        $CP $OTHER_FILE $OTHER_STRIPPED_FILE
        $STRIP $THIS_STRIPPED_FILE
        $STRIP $OTHER_STRIPPED_FILE
        THIS_FILE="$THIS_STRIPPED_FILE"
        OTHER_FILE="$OTHER_STRIPPED_FILE"
    fi

    if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
	unset _NT_SYMBOL_PATH
	# On windows we need to unzip the debug symbols, if present
	OTHER_FILE_BASE=${OTHER_FILE/.dll/}
	OTHER_FILE_BASE=${OTHER_FILE_BASE/.exe/}
	DIZ_NAME=$(basename $OTHER_FILE_BASE).diz
        # java.exe and java.dll diz files will have the same name. Have to
	# make sure java.exe gets the right one. This is only needed for 
	# OTHER since in the new build, all pdb files are left around.
	if [ "$NAME" = "java.exe" ] && [ -f "$OTHER/tmp/java/java/obj64/java.diz" ]; then
	    OTHER_DIZ_FILE="$OTHER/tmp/java/java/obj64/java.diz"
	elif [ -f "${OTHER_FILE_BASE}.diz" ]; then
	    OTHER_DIZ_FILE=${OTHER_FILE_BASE}.diz
	else
            # Some files, jli.dll, appears twice in the image but only one of
	    # thme has a diz file next to it.
	    OTHER_DIZ_FILE="$($FIND $OTHER_DIR -name $DIZ_NAME | $SED 1q)"
	    if [ ! -f "$OTHER_DIZ_FILE" ]; then
		# As a last resort, look for diz file in the whole build output
		# dir.
		OTHER_DIZ_FILE="$($FIND $OTHER -name $DIZ_NAME | $SED 1q)"
	    fi
	fi
	if [ -n "$OTHER_DIZ_FILE" ]; then
	    $MKDIR -p $FILE_WORK_DIR/other
	    (cd $FILE_WORK_DIR/other ; $UNARCHIVE -o $OTHER_DIZ_FILE)
	    export _NT_SYMBOL_PATH="$FILE_WORK_DIR/other"
	fi
	THIS_FILE_BASE=${THIS_FILE/.dll/}
	THIS_FILE_BASE=${THIS_FILE_BASE/.exe/}
	if [ -f "${THIS_FILE/.dll/}.diz" ]; then
	    THIS_DIZ_FILE=${THIS_FILE/.dll/}.diz
	else
	    THIS_DIZ_FILE="$($FIND $THIS_DIR -name $DIZ_NAME | $SED 1q)"
	    if [ ! -f "$THIS_DIZ_FILE" ]; then
		# As a last resort, look for diz file in the whole build output
		# dir.
		THIS_DIZ_FILE="$($FIND $THIS -name $DIZ_NAME | $SED 1q)"
	    fi
	fi
	if [ -n "$THIS_DIZ_FILE" ]; then
	    $MKDIR -p $FILE_WORK_DIR/this
	    (cd $FILE_WORK_DIR/this ; $UNARCHIVE -o $THIS_DIZ_FILE)
	    export _NT_SYMBOL_PATH="$_NT_SYMBOL_PATH;$FILE_WORK_DIR/this"
	fi
    fi

    if [ -z "$SKIP_BIN_DIFF" ]; then
        if cmp $OTHER_FILE $THIS_FILE > /dev/null; then
        # The files were bytewise identical.
            if [ -n "$VERBOSE" ]; then
                echo "        :           :         :         :          : $BIN_FILE"
            fi
            return 0
        fi
        BIN_MSG=" diff "
        if [[ "$ACCEPTED_BIN_DIFF" != *"$BIN_FILE"* ]]; then
            DIFF_BIN=true
            if [[ "$KNOWN_BIN_DIFF" != *"$BIN_FILE"* ]]; then
                BIN_MSG="*$BIN_MSG*"
                REGRESSIONS=true
            else
                BIN_MSG=" $BIN_MSG "
            fi
        else
            BIN_MSG="($BIN_MSG)"
            DIFF_BIN=
        fi
    fi

    if [ -n "$STAT" ]; then
        THIS_SIZE=$($STAT $STAT_PRINT_SIZE "$THIS_FILE")
        OTHER_SIZE=$($STAT $STAT_PRINT_SIZE "$OTHER_FILE")
    else
        THIS_SIZE=$(ls -l "$THIS_FILE" | awk '{ print $5 }')
        OTHER_SIZE=$(ls -l "$OTHER_FILE" | awk '{ print $5 }')
    fi
    if [ $THIS_SIZE -ne $OTHER_SIZE ]; then
        DIFF_SIZE_NUM=$($EXPR $THIS_SIZE - $OTHER_SIZE)
        DIFF_SIZE_REL=$($EXPR $THIS_SIZE \* 100 / $OTHER_SIZE)
        SIZE_MSG=$($PRINTF "%3d%% %4d" $DIFF_SIZE_REL $DIFF_SIZE_NUM)
        if [[ "$ACCEPTED_SMALL_SIZE_DIFF" = *"$BIN_FILE"* ]] && [ "$DIFF_SIZE_REL" -gt 98 ] \
	    && [ "$DIFF_SIZE_REL" -lt 102 ]; then
            SIZE_MSG="($SIZE_MSG)"
            DIFF_SIZE=
        elif [ "$OPENJDK_TARGET_OS" = "windows" ] \
	    && [[ "$ACCEPTED_SMALL_SIZE_DIFF" = *"$BIN_FILE"* ]] \
	    && [ "$DIFF_SIZE_NUM" = 512 ]; then
	    # On windows, size of binaries increase in 512 increments.
            SIZE_MSG="($SIZE_MSG)"
            DIFF_SIZE=
        elif [ "$OPENJDK_TARGET_OS" = "windows" ] \
	    && [[ "$ACCEPTED_SMALL_SIZE_DIFF" = *"$BIN_FILE"* ]] \
	    && [ "$DIFF_SIZE_NUM" = -512 ]; then
	    # On windows, size of binaries increase in 512 increments.
            SIZE_MSG="($SIZE_MSG)"
            DIFF_SIZE=
        else
            if [[ "$ACCEPTED_SIZE_DIFF" != *"$BIN_FILE"* ]]; then
                DIFF_SIZE=true
                if [[ "$KNOWN_SIZE_DIFF" != *"$BIN_FILE"* ]]; then
                    SIZE_MSG="*$SIZE_MSG*"
                    REGRESSIONS=true
                else
                    SIZE_MSG=" $SIZE_MSG "
                fi
            else
                SIZE_MSG="($SIZE_MSG)"
                DIFF_SIZE=
            fi
        fi
    else
        SIZE_MSG="           "
        DIFF_SIZE=
        if [[ "$KNOWN_SIZE_DIFF $ACCEPTED_SIZE_DIFF" = *"$BIN_FILE"* ]]; then
            SIZE_MSG="     !     "
        fi
    fi

    if [[ "$SORT_SYMBOLS" = *"$BIN_FILE"* ]]; then
        SYM_SORT_CMD="sort"
    else
        SYM_SORT_CMD="cat"
    fi

    # Check symbols
    if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
        # The output from dumpbin on windows differs depending on if the debug symbol
        # files are still around at the location the binary is pointing too. Need
	# to filter out that extra information.
	$DUMPBIN -exports $OTHER_FILE | $GREP  -E '^ +[0-9A-F]+ +[0-9A-F]+ [0-9A-F]+' | sed 's/ = .*//g' | cut -c27- | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.other
	$DUMPBIN -exports $THIS_FILE  | $GREP  -E '^ +[0-9A-F]+ +[0-9A-F]+ [0-9A-F]+' | sed 's/ = .*//g' | cut -c27- | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.this
    elif [ "$OPENJDK_TARGET_OS" = "solaris" ]; then
        # Some symbols get seemingly random 15 character prefixes. Filter them out.
        $NM -a $ORIG_OTHER_FILE 2> /dev/null | $GREP -v $NAME | $AWK '{print $2, $3, $4, $5}' | $SED 's/^\([a-zA-Z] [\.\$]\)[a-zA-Z0-9_\$]\{15,15\}\./\1./g' | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.other
	$NM -a $ORIG_THIS_FILE  2> /dev/null | $GREP -v $NAME | $AWK '{print $2, $3, $4, $5}' | $SED 's/^\([a-zA-Z] [\.\$]\)[a-zA-Z0-9_\$]\{15,15\}\./\1./g' | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.this
    else
	$NM -a $ORIG_OTHER_FILE 2> /dev/null | $GREP -v $NAME | $AWK '{print $2, $3, $4, $5}' | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.other
	$NM -a $ORIG_THIS_FILE  2> /dev/null | $GREP -v $NAME | $AWK '{print $2, $3, $4, $5}' | $SYM_SORT_CMD > $WORK_FILE_BASE.symbols.this
    fi
    
    LC_ALL=C $DIFF $WORK_FILE_BASE.symbols.other $WORK_FILE_BASE.symbols.this > $WORK_FILE_BASE.symbols.diff
    if [ -s $WORK_FILE_BASE.symbols.diff ]; then
        SYM_MSG=" diff  "
        if [[ "$ACCEPTED_SYM_DIFF" != *"$BIN_FILE"* ]]; then
            DIFF_SYM=true
            if [[ "$KNOWN_SYM_DIFF" != *"$BIN_FILE"* ]]; then
                SYM_MSG="*$SYM_MSG*"
                REGRESSIONS=true
            else
                SYM_MSG=" $SYM_MSG "
            fi
        else
            SYM_MSG="($SYM_MSG)"            
            DIFF_SYM=
        fi
    else
        SYM_MSG="         "
        DIFF_SYM=
        if [[ "$KNOWN_SYM_DIFF $ACCEPTED_SYM_DIFF" = *"$BIN_FILE"* ]]; then
            SYM_MSG="    !    "
        fi
    fi

    # Check dependencies
    if [ -n "$LDD_CMD" ]; then
	(cd $FILE_WORK_DIR && $CP $OTHER_FILE . && $LDD_CMD $NAME 2>/dev/null | $AWK '{ print $1;}' | $SORT | $TEE $WORK_FILE_BASE.deps.other | $UNIQ > $WORK_FILE_BASE.deps.other.uniq)
	(cd $FILE_WORK_DIR && $CP $THIS_FILE . && $LDD_CMD $NAME 2</dev/null | $AWK '{ print $1;}' | $SORT | $TEE $WORK_FILE_BASE.deps.this | $UNIQ > $WORK_FILE_BASE.deps.this.uniq)
	(cd $FILE_WORK_DIR && $RM -f $NAME)
	
	LC_ALL=C $DIFF $WORK_FILE_BASE.deps.other $WORK_FILE_BASE.deps.this > $WORK_FILE_BASE.deps.diff
	LC_ALL=C $DIFF $WORK_FILE_BASE.deps.other.uniq $WORK_FILE_BASE.deps.this.uniq > $WORK_FILE_BASE.deps.diff.uniq
	
	if [ -s $WORK_FILE_BASE.deps.diff ]; then
            if [ -s $WORK_FILE_BASE.deps.diff.uniq ]; then
		DEP_MSG=" diff  "
            else
		DEP_MSG=" redun "
            fi
            if [[ "$ACCEPTED_DEP_DIFF" != *"$BIN_FILE"* ]]; then
		DIFF_DEP=true
		if [[ "$KNOWN_DEP_DIFF" != *"$BIN_FILE"* ]]; then
                    DEP_MSG="*$DEP_MSG*"
                    REGRESSIONS=true
		else
                    DEP_MSG=" $DEP_MSG "
		fi
            else
		DEP_MSG="($DEP_MSG)"
		DIFF_DEP=
            fi
	else
	    DEP_MSG="         "
	    DIFF_DEP=
            if [[ "$KNOWN_DEP_DIFF $ACCEPTED_DEP_DIFF" = *"$BIN_FILE"* ]]; then
                DEP_MSG="     !      "
            fi
	fi
    else
	DEP_MSG="    -    "
    fi
    
    # Compare fulldump output
    if [ -n "$FULLDUMP_CMD" ] && [ -z "$SKIP_FULLDUMP_DIFF" ]; then
        $FULLDUMP_CMD $OTHER_FILE > $WORK_FILE_BASE.fulldump.other 2>&1
        $FULLDUMP_CMD $THIS_FILE > $WORK_FILE_BASE.fulldump.this 2>&1
        LC_ALL=C $DIFF $WORK_FILE_BASE.fulldump.other $WORK_FILE_BASE.fulldump.this > $WORK_FILE_BASE.fulldump.diff
        
        if [ -s $WORK_FILE_BASE.fulldump.diff ]; then
            ELF_DIFF_SIZE=$(ls -n $WORK_FILE_BASE.fulldump.diff | awk '{print $5}')
            ELF_MSG=$($PRINTF "%8d" $ELF_DIFF_SIZE)
            if [[ "$ACCEPTED_ELF_DIFF" != *"$BIN_FILE"* ]]; then
                DIFF_ELF=true
                if [[ "$KNOWN_ELF_DIFF" != *"$BIN_FILE"* ]]; then
                    ELF_MSG="*$ELF_MSG*"
                    REGRESSIONS=true
                else
                    ELF_MSG=" $ELF_MSG "
                fi
            else
                ELF_MSG="($ELF_MSG)"
                DIFF_ELF=
            fi
        else
            ELF_MSG="          "
            DIFF_ELF=
            if [[ "$KNOWN_DEP_DIFF $ACCEPTED_DEP_DIFF" = *"$BIN_FILE"* ]]; then
                ELF_MSG="    !    "
            fi
        fi
    fi

    # Compare disassemble output
    if [ -n "$DIS_CMD" ] && [ -z "$SKIP_DIS_DIFF" ]; then
	if [ -z "$DIS_DIFF_FILTER" ]; then
	    DIS_DIFF_FILTER="$CAT"
	fi
        $DIS_CMD $OTHER_FILE | $GREP -v $NAME | $DIS_DIFF_FILTER > $WORK_FILE_BASE.dis.other 2>&1
        $DIS_CMD $THIS_FILE  | $GREP -v $NAME | $DIS_DIFF_FILTER > $WORK_FILE_BASE.dis.this  2>&1
        
        LC_ALL=C $DIFF $WORK_FILE_BASE.dis.other $WORK_FILE_BASE.dis.this > $WORK_FILE_BASE.dis.diff
        
        if [ -s $WORK_FILE_BASE.dis.diff ]; then
            DIS_DIFF_SIZE=$(ls -n $WORK_FILE_BASE.dis.diff | awk '{print $5}')
            DIS_MSG=$($PRINTF "%8d" $DIS_DIFF_SIZE)
            if [[ "$ACCEPTED_DIS_DIFF" != *"$BIN_FILE"* ]]; then
                DIFF_DIS=true
                if [[ "$KNOWN_DIS_DIFF" != *"$BIN_FILE"* ]]; then
                    DIS_MSG="*$DIS_MSG*"
                    REGRESSIONS=true
                else
                    DIS_MSG=" $DIS_MSG "
                fi
            else
                DIS_MSG="($DIS_MSG)"
                DIFF_DIS=
            fi
        else
            DIS_MSG="          "
            DIFF_DIS=
            if [[ "$KNOWN_DEP_DIFF $ACCEPTED_DEP_DIFF" = *"$BIN_FILE"* ]]; then
                DIS_MSG="    !    "
            fi
        fi
    fi


    if [ -n "$DIFF_BIN$DIFF_SIZE$DIFF_SYM$DIFF_DEP$DIFF_ELF$DIFF_DIS" ] || [ -n "$VERBOSE" ]; then
        if [ -n "$BIN_MSG" ]; then echo -n "$BIN_MSG:"; fi
        if [ -n "$SIZE_MSG" ]; then echo -n "$SIZE_MSG:"; fi
        if [ -n "$SYM_MSG" ]; then echo -n "$SYM_MSG:"; fi
        if [ -n "$DEP_MSG" ]; then echo -n "$DEP_MSG:"; fi
        if [ -n "$ELF_MSG" ]; then echo -n "$ELF_MSG:"; fi
        if [ -n "$DIS_MSG" ]; then echo -n "$DIS_MSG:"; fi
        echo " $BIN_FILE"
        if [ "$SHOW_DIFFS" = "true" ]; then
            if [ -s "$WORK_FILE_BASE.symbols.diff" ]; then
                echo "Symbols diff:"
                $CAT $WORK_FILE_BASE.symbols.diff
            fi
            if [ -s "$WORK_FILE_BASE.deps.diff" ]; then
                echo "Deps diff:"
                $CAT $WORK_FILE_BASE.deps.diff
            fi
            if [ -s "$WORK_FILE_BASE.fulldump.diff" ]; then
                echo "Fulldump diff:"
                $CAT $WORK_FILE_BASE.fulldump.diff
            fi
            if [ -s "$WORK_FILE_BASE.dis.diff" ]; then
                echo "Disassembly diff:"
                $CAT $WORK_FILE_BASE.dis.diff
            fi
        fi
        return 1
    fi
    return 0
}

##########################################################################################
# Print binary diff header

print_binary_diff_header() {
    if [ -z "$SKIP_BIN_DIFF" ]; then echo -n " Binary :"; fi
    if [ -z "$SKIP_SIZE_DIFF" ]; then echo -n "   Size    :"; fi
    if [ -z "$SKIP_SYM_DIFF" ]; then echo -n " Symbols :"; fi
    if [ -z "$SKIP_DEP_DIFF" ]; then echo -n "  Deps   :"; fi
    if [ -z "$SKIP_FULLDUMP_DIFF" ]; then echo -n " Fulldump :"; fi
    if [ -z "$SKIP_DIS_DIFF" ]; then echo -n " Disass   :"; fi
    echo
}

##########################################################################################
# Compare all libraries

compare_all_libs() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    LIBS=$(cd $THIS_DIR && $FIND . -type f \( -name 'lib*.so' -o -name '*.dylib' -o -name '*.dll' -o -name 'JavaControlPanel' \) | $SORT | $FILTER)

    if [ -n "$LIBS" ]; then
        echo Libraries...
        print_binary_diff_header
        for l in $LIBS; do
            if [ -f "$OTHER_DIR/$l" ]; then
                compare_bin_file $THIS_DIR $OTHER_DIR $WORK_DIR $l
                if [ "$?" != "0" ]; then
                    return_value=1
                fi
            fi
        done
    fi

    return $return_value
}

##########################################################################################
# Compare all executables

compare_all_execs() {
    THIS_DIR=$1
    OTHER_DIR=$2
    WORK_DIR=$3

    if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
        EXECS=$(cd $THIS_DIR && $FIND . -type f -name '*.exe' | $SORT | $FILTER)
    else
        EXECS=$(cd $THIS_DIR && $FIND . -name db -prune -o -type f -perm -100 \! \
            \( -name '*.so' -o -name '*.dylib' -o -name '*.dll' -o -name '*.cgi' \
            -o -name '*.jar' -o -name '*.diz' -o -name 'jcontrol' -o -name '*.properties' \
            -o -name '*.data' -o -name '*.bfc' -o -name '*.src' -o -name '*.txt' \
            -o -name '*.cfg' -o -name 'meta-index' -o -name '*.properties.ja' \
            -o -name 'classlist' \) | $SORT | $FILTER)
    fi

    if [ -n "$EXECS" ]; then
        echo Executables...
        print_binary_diff_header
        for e in $EXECS; do
            if [ -f "$OTHER_DIR/$e" ]; then
                compare_bin_file $THIS_DIR $OTHER_DIR $WORK_DIR $e
                if [ "$?" != "0" ]; then
                    return_value=1
                fi
            fi
        done
    fi

    return $return_value
}

##########################################################################################
# Initiate configuration

COMPARE_ROOT=/tmp/cimages.$USER
$MKDIR -p $COMPARE_ROOT
if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
    if [ "$(uname -o)" = "Cygwin" ]; then
	COMPARE_ROOT=$(cygpath -msa $COMPARE_ROOT)
    fi
fi

THIS="$( cd "$( dirname "$0" )" && pwd )"
echo "$THIS"
THIS_SCRIPT="$0"

if [ -z "$1" ] || [ "$1" = "-h" ] || [ "$1" = "-?" ] || [ "$1" = "/h" ] || [ "$1" = "/?" ] || [ "$1" = "-help" ] || [ "$1" = "--help" ]; then
    echo "bash ./compare.sh [OPTIONS] [FILTER]"
    echo ""
    echo "-all                Compare all files in all known ways"
    echo "-names              Compare the file names and directory structure"
    echo "-perms              Compare the permission bits on all files and directories"
    echo "-types              Compare the output of the file command on all files"
    echo "-general            Compare the files not convered by the specialized comparisons"
    echo "-zips               Compare the contents of all zip files"
    echo "-jars               Compare the contents of all jar files"
    echo "-libs               Compare all native libraries"
    echo "-execs              Compare all executables"
    echo "-v                  Verbose output, does not hide known differences"
    echo "-vv                 More verbose output, shows diff output of all comparisons"
    echo "-o [OTHER]          Compare with build in other directory. Will default to the old build directory"
    echo ""
    echo "[FILTER]            List filenames in the image to compare, works for jars, zips, libs and execs"
    echo "Example:"
    echo "bash ./common/bin/compareimages.sh CodePointIM.jar"
    echo ""
    echo "-2zips <file1> <file2> Compare two zip files only"
    echo ""
    exit 10
fi

CMP_NAMES=false
CMP_PERMS=false
CMP_TYPES=false
CMP_GENERAL=false
CMP_ZIPS=false
CMP_JARS=false
CMP_LIBS=false
CMP_EXECS=false

while [ -n "$1" ]; do
    case "$1" in
        -v)
            VERBOSE=true
            ;;
        -vv)
            VERBOSE=true
            SHOW_DIFFS=true
            ;;
        -o)
            OTHER="$2"
            shift
            ;;
        -all)
            CMP_NAMES=true
            if [ "$OPENJDK_TARGET_OS" != "windows" ]; then
                CMP_PERMS=true
            fi
            CMP_TYPES=true
            CMP_GENERAL=true
            CMP_ZIPS=true
            CMP_JARS=true
            CMP_LIBS=true
            CMP_EXECS=true
            ;;
        -names)
            CMP_NAMES=true
            ;;
        -perms)
            CMP_PERMS=true
            ;;
        -types)
            CMP_TYPES=true
            ;;
        -general)
            CMP_GENERAL=true
            ;;
        -zips)
            CMP_ZIPS=true
            ;;
        -jars)
            CMP_JARS=true
            ;;
        -libs)
            CMP_LIBS=true
            ;;
        -execs)
            CMP_EXECS=true
            ;;
        -2zips)
            CMP_2_ZIPS=true
            THIS_FILE=$2
            OTHER_FILE=$3
            shift
            shift
            ;;
        *)
            CMP_NAMES=false
            CMP_PERMS=false
            CMP_TYPES=false
            CMP_ZIPS=true
            CMP_JARS=true
            CMP_LIBS=true
            CMP_EXECS=true
            
            if [ -z "$FILTER" ]; then
                FILTER="$GREP"
            fi
            FILTER="$FILTER -e $1"
            ;;
    esac
    shift
done

if [ "$CMP_2_ZIPS" = "true" ]; then
    THIS_DIR="$(dirname $THIS_FILE)"
    THIS_DIR="$(cd "$THIS_DIR" && pwd )"
    OTHER_DIR="$(dirname $OTHER_FILE)"
    OTHER_DIR="$(cd "$OTHER_DIR" && pwd )"
    THIS_FILE_NAME="$(basename $THIS_FILE)"
    OTHER_FILE_NAME="$(basename $OTHER_FILE)"
    echo Comparing $THIS_DIR/$THIS_FILE_NAME and $OTHER_DIR/$OTHER_FILE_NAME
    compare_zip_file $THIS_DIR $OTHER_DIR $COMPARE_ROOT/2zips $THIS_FILE_NAME $OTHER_FILE_NAME
    exit
fi

if [ "$CMP_NAMES" = "false" ] && [ "$CMP_TYPES" = "false" ] && [ "$CMP_PERMS" = "false" ] && [ "$CMP_GENERAL" = "false" ] && [ "$CMP_ZIPS" = "false" ] && [ "$CMP_JARS" = "false" ] && [ "$CMP_LIBS" = "false" ] && [ "$CMP_EXECS" = "false" ]; then
    CMP_NAMES=true
    CMP_PERMS=true
    CMP_TYPES=true
    CMP_GENERAL=true
    CMP_ZIPS=true
    CMP_JARS=true
    CMP_LIBS=true
    CMP_EXECS=true
fi

if [ -z "$FILTER" ]; then
    FILTER="$CAT"
fi

if [ -z "$OTHER" ]; then
    OTHER="$THIS/../$LEGACY_BUILD_DIR"
    if [ -d "$OTHER" ]; then
        OTHER="$( cd "$OTHER" && pwd )"
    else
        echo "Default old build directory does not exist:"
        echo "$OTHER"
        exit 1
    fi
    echo "Comparing to default old build:"
    echo "$OTHER"
    echo
else
    if [ ! -d "$OTHER" ]; then
        echo "Other build directory does not exist:"
        echo "$OTHER"
        exit 1
    fi
    OTHER="$( cd "$OTHER" && pwd )"
    echo "Comparing to:"
    echo "$OTHER"
    echo
fi


# Figure out the layout of the this build. Which kinds of images have been produced
if [ -d "$THIS/install/j2sdk-image" ]; then
    THIS_J2SDK="$THIS/install/j2sdk-image"
    THIS_J2RE="$THIS/install/j2re-image"
    echo "Selecting install images in this build"
elif [ -d "$THIS/deploy/j2sdk-image" ]; then
    THIS_J2SDK="$THIS/deploy/j2sdk-image"
    THIS_J2RE="$THIS/deploy/j2re-image"
    echo "Selecting deploy images in this build"
elif [ -d "$THIS/images/j2sdk-image" ]; then
    THIS_J2SDK="$THIS/images/j2sdk-image"
    THIS_J2RE="$THIS/images/j2re-image"
    echo "Selecting jdk images in this build"
fi

if [ -d "$THIS/images/j2sdk-overlay-image" ]; then
    if [ -d "$THIS/install/j2sdk-image" ]; then
        # If there is an install image, prefer that, it's also overlay
        THIS_J2SDK_OVERLAY="$THIS/install/j2sdk-image"
        THIS_J2RE_OVERLAY="$THIS/install/j2re-image"
        echo "Selecting install overlay images in this build"
    else
        THIS_J2SDK_OVERLAY="$THIS/images/j2sdk-overlay-image"
        THIS_J2RE_OVERLAY="$THIS/images/j2re-overlay-image"
        echo "Selecting jdk overlay images in this build"
    fi
fi

if [ -d "$THIS/images/j2sdk-bundle" ]; then
    THIS_J2SDK_BUNDLE="$THIS/images/j2sdk-bundle"
    THIS_J2RE_BUNDLE="$THIS/images/j2re-bundle"
    echo "Selecting bundles in this build"
fi

# Figure out the layout of the other build (old or new, normal or overlay image)
if [ -d "$OTHER/j2sdk-image" ]; then
    if [ -f "$OTHER/j2sdk-image/LICENSE" ]; then
        OTHER_J2SDK="$OTHER/j2sdk-image"
        OTHER_J2RE="$OTHER/j2re-image"
        echo "Selecting old-style images in other build"
    else
        OTHER_J2SDK_OVERLAY="$OTHER/j2sdk-image"
        OTHER_J2RE_OVERLAY="$OTHER/j2re-image"
        echo "Selecting overlay images in other build"
    fi
elif [ -d "$OTHER/install/j2sdk-image" ]; then
    OTHER_J2SDK="$OTHER/install/j2sdk-image"
    OTHER_J2RE="$OTHER/install/j2re-image"
    echo "Selecting install images in other build"
elif [ -d "$OTHER/deploy/j2sdk-image" ]; then
    OTHER_J2SDK="$OTHER/deploy/j2sdk-image"
    OTHER_J2RE="$OTHER/deploy/j2re-image"
    echo "Selecting deploy images in other build"
elif [ -d "$OTHER/images/j2sdk-image" ]; then
    OTHER_J2SDK="$OTHER/images/j2sdk-image"
    OTHER_J2RE="$OTHER/images/j2re-image"
    echo "Selecting jdk images in other build"
fi

if [ -d "$OTHER/j2sdk-bundle" ]; then
    OTHER_J2SDK_BUNDLE="$OTHER/j2sdk-bundle"
    OTHER_J2RE_BUNDLE="$OTHER/j2re-bundle"
    echo "Selecting bundles in other build"
elif [ -d "$OTHER/images/j2sdk-bundle" ]; then
    OTHER_J2SDK_BUNDLE="$OTHER/images/j2sdk-bundle"
    OTHER_J2RE_BUNDLE="$OTHER/images/j2re-bundle"
    echo "Selecting jdk bundles in other build"
fi

if [ -z "$THIS_J2SDK" ] || [ -z "$THIS_J2RE" ]; then
    if [ -z "$THIS_J2SDK_OVERLAY" ]; then
        echo "Cannot locate images for this build. Are you sure you have run 'make images'?"
        exit 1
    fi
fi

if [ -z "$OTHER_J2SDK" ] && [ -n "$OTHER_J2SDK_OVERLAY" ] && [ -z "$THIS_J2SDK_OVERLAY" ]; then
    echo "OTHER build only has an overlay image while this build does not. Nothing to compare!"
    exit 1
fi

if [ -z "$THIS_J2SDK_BUNDLE" ] && [ -n "$OTHER_J2SDK_BUNDLE" ]; then
    echo "WARNING! OTHER build has bundles built while this build does not."
    echo "Skipping bundle compare!"
fi

if [ -d "$THIS/docs" ]; then
    THIS_DOCS="$THIS/docs"
fi

if [ -d "$OTHER/docs" ]; then
    OTHER_DOCS="$OTHER/docs"
fi

if [ -z "$THIS_DOCS" ]; then
    echo "WARNING! Docs haven't been built and won't be compared."
fi

if [ -z "$OTHER_DOCS" ]; then
    echo "WARNING! Other build doesn't contain docs, skipping doc compare."
fi

if [ -d "$OTHER/images" ]; then
    OTHER_SEC_DIR="$OTHER/images"
else
    OTHER_SEC_DIR="$OTHER/tmp"
fi
OTHER_SEC_BIN="$OTHER_SEC_DIR/sec-bin.zip"
THIS_SEC_DIR="$THIS/images"
THIS_SEC_BIN="$THIS_SEC_DIR/sec-bin.zip"
if [ "$OPENJDK_TARGET_OS" = "windows" ]; then
    if [ "$OPENJDK_TARGET_CPU" = "x86_64" ]; then
        JGSS_WINDOWS_BIN="jgss-windows-x64-bin.zip"
    else
        JGSS_WINDOWS_BIN="jgss-windows-i586-bin.zip"
    fi
    OTHER_SEC_WINDOWS_BIN="$OTHER_SEC_DIR/sec-windows-bin.zip"
    OTHER_JGSS_WINDOWS_BIN="$OTHER_SEC_DIR/$JGSS_WINDOWS_BIN"
    THIS_SEC_WINDOWS_BIN="$THIS_SEC_DIR/sec-windows-bin.zip"
    THIS_JGSS_WINDOWS_BIN="$THIS_SEC_DIR/$JGSS_WINDOWS_BIN"
fi

##########################################################################################
# Do the work

if [ "$CMP_NAMES" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        echo -n "J2SDK "
        compare_dirs $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
        echo -n "J2RE  "
        compare_dirs $THIS_J2RE $OTHER_J2RE $COMPARE_ROOT/j2re
        
        echo -n "J2SDK "
        compare_files $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
        echo -n "J2RE  "
        compare_files $THIS_J2RE $OTHER_J2RE $COMPARE_ROOT/j2re
    fi
    if [ -n "$THIS_J2SDK_OVERLAY" ] && [ -n "$OTHER_J2SDK_OVERLAY" ]; then
        echo -n "J2SDK Overlay "
        compare_dirs $THIS_J2SDK_OVERLAY $OTHER_J2SDK_OVERLAY $COMPARE_ROOT/j2sdk-overlay
        echo -n "J2RE  Overlay "
        compare_dirs $THIS_J2RE_OVERLAY $OTHER_J2RE_OVERLAY $COMPARE_ROOT/j2re-overlay
        
        echo -n "J2SDK Overlay "
        compare_files $THIS_J2SDK_OVERLAY $OTHER_J2SDK_OVERLAY $COMPARE_ROOT/j2sdk-overlay
        echo -n "J2RE  Overlay "
        compare_files $THIS_J2RE_OVERLAY $OTHER_J2RE_OVERLAY $COMPARE_ROOT/j2re-overlay
    fi
    if [ -n "$THIS_J2SDK_BUNDLE" ] && [ -n "$OTHER_J2SDK_BUNDLE" ]; then
        echo -n "J2SDK Bundle "
        compare_dirs $THIS_J2SDK_BUNDLE $OTHER_J2SDK_BUNDLE $COMPARE_ROOT/j2sdk-bundle
        echo -n "J2RE  Bundle "
        compare_dirs $THIS_J2RE_BUNDLE $OTHER_J2RE_BUNDLE $COMPARE_ROOT/j2re-bundle
        
        echo -n "J2SDK Bundle "
        compare_files $THIS_J2SDK_BUNDLE $OTHER_J2SDK_BUNDLE $COMPARE_ROOT/j2sdk-bundle
        echo -n "J2RE  Bundle "
        compare_files $THIS_J2RE_BUNDLE $OTHER_J2RE_BUNDLE $COMPARE_ROOT/j2re-bundle
    fi
    if [ -n "$THIS_DOCS" ] && [ -n "$OTHER_DOCS" ]; then
        echo -n "Docs "
        compare_dirs $THIS_DOCS $OTHER_DOCS $COMPARE_ROOT/docs
        echo -n "Docs "
        compare_files $THIS_DOCS $OTHER_DOCS $COMPARE_ROOT/docs
    fi
fi

if [ "$CMP_PERMS" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        echo -n "J2SDK "
        compare_permissions $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
        echo -n "J2RE  "
        compare_permissions $THIS_J2RE $OTHER_J2RE $COMPARE_ROOT/j2re
    fi
    if [ -n "$THIS_J2SDK_OVERLAY" ] && [ -n "$OTHER_J2SDK_OVERLAY" ]; then
        echo -n "J2SDK Overlay "
        compare_permissions $THIS_J2SDK_OVERLAY $OTHER_J2SDK_OVERLAY $COMPARE_ROOT/j2sdk-overlay
        echo -n "J2RE  Overlay "
        compare_permissions $THIS_J2RE_OVERLAY $OTHER_J2RE_OVERLAY $COMPARE_ROOT/j2re-overlay
    fi
    if [ -n "$THIS_J2SDK_BUNDLE" ] && [ -n "$OTHER_J2SDK_BUNDLE" ]; then
        echo -n "J2SDK Bundle "
        compare_permissions $THIS_J2SDK_BUNDLE $OTHER_J2SDK_BUNDLE $COMPARE_ROOT/j2sdk-bundle
        echo -n "J2RE  Bundle "
        compare_permissions $THIS_J2RE_BUNDLE $OTHER_J2RE_BUNDLE $COMPARE_ROOT/j2re-bundle
    fi
fi

if [ "$CMP_TYPES" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        echo -n "J2SDK "
        compare_file_types $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
        echo -n "J2RE  "
        compare_file_types $THIS_J2RE $OTHER_J2RE $COMPARE_ROOT/j2re
    fi
    if [ -n "$THIS_J2SDK_OVERLAY" ] && [ -n "$OTHER_J2SDK_OVERLAY" ]; then
        echo -n "J2SDK Overlay "
        compare_file_types $THIS_J2SDK_OVERLAY $OTHER_J2SDK_OVERLAY $COMPARE_ROOT/j2sdk-overlay
        echo -n "J2RE  Overlay "
        compare_file_types $THIS_J2RE_OVERLAY $OTHER_J2RE_OVERLAY $COMPARE_ROOT/j2re-overlay
    fi
    if [ -n "$THIS_J2SDK_BUNDLE" ] && [ -n "$OTHER_J2SDK_BUNDLE" ]; then
        echo -n "J2SDK Bundle "
        compare_file_types $THIS_J2SDK_BUNDLE $OTHER_J2SDK_BUNDLE $COMPARE_ROOT/j2sdk-bundle
        echo -n "J2RE  Bundle "
        compare_file_types $THIS_J2RE_BUNDLE $OTHER_J2RE_BUNDLE $COMPARE_ROOT/j2re-bundle
    fi
fi

if [ "$CMP_GENERAL" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        echo -n "J2SDK "
        compare_general_files $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
        echo -n "J2RE  "
        compare_general_files $THIS_J2RE $OTHER_J2RE $COMPARE_ROOT/j2re
    fi
    if [ -n "$THIS_J2SDK_OVERLAY" ] && [ -n "$OTHER_J2SDK_OVERLAY" ]; then
        echo -n "J2SDK Overlay "
        compare_general_files $THIS_J2SDK_OVERLAY $OTHER_J2SDK_OVERLAY $COMPARE_ROOT/j2sdk-overlay
        echo -n "J2RE  Overlay "
        compare_general_files $THIS_J2RE_OVERLAY $OTHER_J2RE_OVERLAY $COMPARE_ROOT/j2re-overlay
    fi
    if [ -n "$THIS_J2SDK_BUNDLE" ] && [ -n "$OTHER_J2SDK_BUNDLE" ]; then
        echo -n "J2SDK Bundle "
        compare_general_files $THIS_J2SDK_BUNDLE $OTHER_J2SDK_BUNDLE $COMPARE_ROOT/j2sdk-bundle
        echo -n "J2RE  Bundle "
        compare_general_files $THIS_J2RE_BUNDLE $OTHER_J2RE_BUNDLE $COMPARE_ROOT/j2re-bundle
    fi
    if [ -n "$THIS_DOCS" ] && [ -n "$OTHER_DOCS" ]; then
        echo -n "Docs "
        compare_general_files $THIS_DOCS $OTHER_DOCS $COMPARE_ROOT/docs
    fi
fi

if [ "$CMP_ZIPS" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        compare_all_zip_files $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
    fi
    if [ -n "$THIS_SEC_BIN" ] && [ -n "$OTHER_SEC_BIN" ]; then
        if [ -n "$(echo $THIS_SEC_BIN | $FILTER)" ]; then
            echo "sec-bin.zip..."
            compare_zip_file $THIS_SEC_DIR $OTHER_SEC_DIR $COMPARE_ROOT/sec-bin sec-bin.zip
        fi
    fi
    if [ -n "$THIS_SEC_WINDOWS_BIN" ] && [ -n "$OTHER_SEC_WINDOWS_BIN" ]; then
        if [ -n "$(echo $THIS_SEC_WINDOWS_BIN | $FILTER)" ]; then
            echo "sec-windows-bin.zip..."
            compare_zip_file $THIS_SEC_DIR $OTHER_SEC_DIR $COMPARE_ROOT/sec-bin sec-windows-bin.zip
        fi
    fi
    if [ -n "$THIS_JGSS_WINDOWS_BIN" ] && [ -n "$OTHER_JGSS_WINDOWS_BIN" ]; then
        if [ -n "$(echo $THIS_JGSS_WINDOWS_BIN | $FILTER)" ]; then
            echo "$JGSS_WINDOWS_BIN..."
            compare_zip_file $THIS_SEC_DIR $OTHER_SEC_DIR $COMPARE_ROOT/sec-bin $JGSS_WINDOWS_BIN
        fi
    fi
fi

if [ "$CMP_JARS" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        compare_all_jar_files $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
    fi
fi

if [ "$CMP_LIBS" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        echo -n "J2SDK "
        compare_all_libs $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
        if [ "$OPENJDK_TARGET_OS" = "macosx" ]; then
            echo -n "J2RE  "
            compare_all_libs $THIS_J2RE $OTHER_J2RE $COMPARE_ROOT/j2re
        fi
    fi
    if [ -n "$THIS_J2SDK_OVERLAY" ] && [ -n "$OTHER_J2SDK_OVERLAY" ]; then
        echo -n "Bundle   "
        compare_all_libs $THIS_J2SDK_OVERLAY $OTHER_J2SDK_OVERLAY $COMPARE_ROOT/j2sdk-overlay
    fi
fi

if [ "$CMP_EXECS" = "true" ]; then
    if [ -n "$THIS_J2SDK" ] && [ -n "$OTHER_J2SDK" ]; then
        compare_all_execs $THIS_J2SDK $OTHER_J2SDK $COMPARE_ROOT/j2sdk
    fi
    if [ -n "$THIS_J2SDK_OVERLAY" ] && [ -n "$OTHER_J2SDK_OVERLAY" ]; then
        echo -n "Overlay "
        compare_all_execs $THIS_J2SDK_OVERLAY $OTHER_J2SDK_OVERLAY $COMPARE_ROOT/j2sdk-overlay
    fi
fi

echo

if [ -n "$REGRESSIONS" ]; then
    echo "REGRESSIONS FOUND!"
    echo
    exit 1
else
    echo "No regressions found"
    echo
    exit 0
fi
