///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file SnappingManager.h 
 * \brief Contains the definition of the Core::SnappingManager class. 
 */
 
#ifndef __OVITO_SNAPPING_MANAGER_H
#define __OVITO_SNAPPING_MANAGER_H

#include <core/Core.h>
#include "SnappingProvider.h"

namespace Core {

class Viewport;		// defined in Viewport.h
class ActionProxy;	// defined in ActionProxy.h

/// \def SNAPPING_MANAGER
/// \brief The predefined instance of the Core::SnappingManager class.
/// 
/// Always use this macro to access the Core::SnappingManager class instance.
#define SNAPPING_MANAGER		(*SnappingManager::getSingletonInstance())

/**
 * \brief This class controls the snapping in the viewports.
 * 
 * This is a singleton class with only one predefined instance of this class. 
 * You can access the instance of this class using the SNAPPING_MANAGER macro.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT SnappingManager : public QObject
{
	Q_OBJECT

public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the SnappingManager singleton class.
	/// \note You should use the SNAPPING_MANAGER macro to access the SnappingManager instance instead
	///       of this method.
	inline static SnappingManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "SnappingManager::getSingletonInstance", "SnappingManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Gets the current snapping threshold. 
	/// \return The maximum distance in pixels that the mouse cursor may be away from a
	///         snapping site.
	/// \sa setSnapStrength()
	int snapStrength() const { return _snapStrength; }
	
	/// \brief Controls the snapping threshold. 
	/// \param strength The maximum distance in pixels that the mouse cursor may be away from a
	///                 snapping site.
	/// \sa snapStrength()
	void setSnapStrength(int strength) { 
		OVITO_ASSERT(strength >= 0); 
		_snapStrength = strength;
	}

	/// \brief Gets the snapping step size for angles.
	/// \return The angle snapping step size in radians. 
	///         Rotation angles are snapped to the nearest integer multiple of this value.
	/// \sa setAngleSnapStepSize() 
	FloatType angleSnapStepSize() const { return _angleSnapStepSize; }
	
	/// \brief Sets the angle snapping step size for angles.
	/// \param angleStepSize The new step size in radians.
	///         Rotation angles are snapped to the nearest integer multiple of this value.
	/// \sa angleSnapStepSize()
	void setAngleSnapStepSize(FloatType angleStepSize) {
		OVITO_ASSERT(angleStepSize > 0);
		_angleSnapStepSize = angleStepSize;
	}

	/// \brief Gets the snapping step size for percentage values.
	/// \return The snapping steps size in the range 0 to 1.
	/// \sa setPercentSnapStepSize()
	FloatType percentSnapStepSize() const { return _percentSnapStepSize; }
	
	/// \brief This controls the snapping step size for percentage values.
	/// \param percentStepSize The new snapping steps size in the range 0 to 1.
	/// \sa percentSnapStepSize()
	void setPercentSnapStepSize(FloatType percentStepSize) {
		OVITO_ASSERT(percentStepSize > 0 && percentStepSize <= 1);
		_percentSnapStepSize = percentStepSize; 
	}

	/// \brief Rounds an angle value to the nearest multiple of the current angle snap stepping size.
	/// \param angle The input angle in radians to be rounded.
	/// \return The input angle rounded to the nearest multiple of angleSnapStepSize() or, if angle snapping is disabled, 
	///         the unmodified input angle.
	/// \sa angleSnapStepSize()
	FloatType snapAngle(FloatType angle);

	/// \brief Rounds a percent value to the nearest multiple of the current percent snap stepping size.
	/// \param p The input value in the range 0 to 1.
	/// \return The input value rounded to the nearest multiple of percentSnapStepSize() or, if percent snapping is disabled, 
	///         the unmodified input value.
	/// \sa percentSnapStepSize()
	FloatType snapPercent(FloatType p);

	/// Flags that can be passed to snapPoint().
	enum SnappingFlag {
		NoFlags = 0,
	};
	Q_DECLARE_FLAGS(SnappingFlags, SnappingFlag)

	/// \brief Performs a snap in a viewport.
	/// \param[in] viewport The viewport to use.
	/// \param[in] screenPos The input mouse coordinates in pixels.
	/// \param[out] result The resulting snap point in world space when the method returns \c true.
	/// \param[in] snappingPlane The construction plane used to do snapping.
	/// \param[in] snapPointFlags Bitwise combination of flags that controls the behaviour of this method.
	/// \return \c true if a snap point has been found and is returned in \a result;
	///         \c false otherwise.
	///
	/// The result of this method is cached in an internal SnappingRecord instance. This instance
	/// can later be retrieved using lastSnapPoint().
	///
	/// The DefaultSceneRenderer renders the snapping maker of the last SnappingRecord created by snapPoint()
	/// in the viewports.
	///
	/// \sa lastSnapPoint()
	bool snapPoint(Viewport* viewport, const Point2I& screenPos, Point3& result, 
		const AffineTransformation& snappingPlane, SnappingFlags snapPointFlags = NoFlags);

	/// \brief Returns the snap point found by the last call to snapPoint().
	/// \return The result of the last snapping operation. Whether a snap point has been found can
	///         be checked via the SnappingRecord::isValid() method.
	/// \sa snapPoint()
	const SnappingRecord& lastSnapPoint() const { return _lastSnapPoint; }

	/// \brief Clears the last hit marker and invalidates the viewports.
	/// 
	/// This resets the internal SnappingRecord created by the last call to snapPoint().
	/// The viewports are updated to remove any snapping markers. 
	void clearLastSnapPoint();

public:

	Q_PROPERTY(int snapStrength READ snapStrength WRITE setSnapStrength)
	Q_PROPERTY(FloatType angleSnapStepSize READ angleSnapStepSize WRITE setAngleSnapStepSize)
	Q_PROPERTY(FloatType percentSnapStepSize READ percentSnapStepSize WRITE setPercentSnapStepSize)

public Q_SLOTS:

	/// Resets the snap settings to their initial values.
	void reset();

private:
    
	/// The snapping threshold in pixels.
	int _snapStrength;

	/// Angles will snap to a multiple of this value.
	FloatType _angleSnapStepSize;

	/// Percent values will snap to a multiple of this value.
	FloatType _percentSnapStepSize;

	/// Contains the last hit found.
	SnappingRecord _lastSnapPoint;

	/// The list of active SnappingProvider classes.
	QVector<SnappingProvider::SmartPtr> _activeProviders;

	/// Controls if snapping to object/grid is enabled.
	ActionProxy* objectSnappingEnabledAction;

	/// Controls if angle snapping is enabled.
	ActionProxy* angleSnappingEnabledAction;

	/// Controls if percent snapping is enabled.
	ActionProxy* percentSnappingEnabledAction;

private:

	/// Private constructor.
	/// This is a singleton class; no public instances are allowed.
	SnappingManager();

	/// Initializes the SnappingManager.
	/// This is called at program startup.
	static void initialize() { 
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new SnappingManager();
	}
	
	/// SnappingManager shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}
	
	/// The singleton instance of this class.
	static SnappingManager* _singletonInstance;

	friend class ApplicationManager;
};

Q_DECLARE_OPERATORS_FOR_FLAGS(SnappingManager::SnappingFlags)

};

#endif // __OVITO_SNAPPING_MANAGER_H
