/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2007: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 26/Sep/2007                        */
/*************************************************** */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <rsbac/types.h>
#include <rsbac/aci_data_structures.h>
#include <rsbac/getname.h>
#include <rsbac/pax_getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

 union rsbac_attribute_value_t value;
 enum rsbac_switch_target_t module;
 enum rsbac_target_t target;
 enum rsbac_attribute_t attr;
 char * progname;
 char * target_n;
int verbose=0;
int recurse=0;
int inherit=0;
rsbac_list_ta_number_t ta_number = 0;

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [switches] module target-type attribute file/dirname(s)\n"), progname);
      printf(gettext(" -v = verbose, -e = show effective (maybe inherited) value, not real\n"));
      printf(gettext(" -r = recurse into subdirs, -n = list all requests\n"));
      printf(gettext(" -a = list attributes and values\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
      printf(gettext(" module = GEN, MAC, FC, SIM, PM, MS, FF, RC or AUTH\n"));
      printf(gettext(" target-type = FILE, DIR, FIFO, SYMLINK, DEV or FD\n"));
      printf(gettext(" (FD: let %s decide between FILE, DIR, FIFO and SYMLINK, no DEV)\n"), progname);
    }

int process(char * name)
  {
    int res = 0;
    char tmp1[RSBAC_MAXNAMELEN];
    struct stat buf;

    if(verbose)
      printf(gettext("Processing %s '%s', attribute %s\n"),
             target_n,
             name,
             get_attribute_name(tmp1,attr));
    value.dummy = -1;
    res = rsbac_get_attr_n(ta_number, module, target, name, attr, &value, inherit);
    if(res)
      {
        if(   verbose
           || (errno != RSBAC_EINVALIDTARGET)
          ) {
          get_error_name(tmp1,res);
          fprintf(stderr, gettext("%s: %s\n"), name, tmp1);
        }
      }
    else
      switch (attr)
        {
          case A_log_array_low:
          case A_log_array_high:
          case A_log_program_based:
            printf(gettext("%s: Returned value: %s\n"),
                   name, u64tostrlog(tmp1,value.log_array_low));
            break;
          case A_mac_categories:
            printf(gettext("%s: Returned value: %s\n"),
                   name, u64tostrmac(tmp1,value.mac_categories));
            break;
          case A_ff_flags:
            printf(gettext("%s: Returned value: %u\n"),
                   name, value.ff_flags);
            break;
          case A_mac_auto:
          case A_mac_prop_trusted:
          case A_mac_file_flags:
          case A_pm_role:
          case A_daz_do_scan:
          case A_daz_scanner:
          case A_daz_scanned:
          case A_auth_may_setuid:
          case A_security_level:
          case A_symlink_add_remote_ip:
          case A_symlink_add_uid:
          case A_symlink_add_rc_role:
          case A_linux_dac_disable:
          case A_fake_root_uid:
          case A_cap_ld_env:
            printf(gettext("%s: Returned value: %u\n"),
                   name, value.u_char_dummy);
            break;
          case A_min_caps:
          case A_max_caps:
            kcaptostrcap(tmp1,value.min_caps);
            printf(gettext("%s: Returned value: %s\n"),
                   name, tmp1);
            break;
          case A_rc_type:
          case A_rc_type_fd:
          case A_rc_force_role:
          case A_rc_initial_role:
          case A_rc_role:
          case A_rc_def_role:
            printf(gettext("%s: Returned value: %u\n"),
                   name, value.rc_type);
            break;
          case A_pax_flags:
            pax_print_flags(tmp1,value.pax_flags);
            printf(gettext("%s: Returned value: %s\n"),
                   name, tmp1);
            break;
          case A_auid_exempt:
          case A_vset:
            printf("%s: Returned value: %u\n",
                   name, value.u_dummy);
            break;
          default:
            printf(gettext("%s: Returned value: %i\n"),
                   name, value.dummy);
        }

    if(   !lstat(name,&buf)
       && S_ISDIR(buf.st_mode)
       && recurse)
      {
        DIR * dir_stream_p;
        struct dirent * dirent_p;
        char name2[PATH_MAX];

        if(S_ISLNK(buf.st_mode))
          return(0);
        if(!(dir_stream_p = opendir(name)))
          {
            fprintf(stderr, gettext("opendir for dir %s returned error: %s\n"),
                   name,
                   strerror(errno));
            return(-2);
          }
        while((dirent_p = readdir(dir_stream_p)))
          {
            if(   (strcmp(".",dirent_p->d_name))
               && (strcmp("..",dirent_p->d_name)) )
              {
                strcpy(name2,name);
                strcat(name2,"/");
                strcat(name2,dirent_p->d_name);
                process(name2);
              }
          }
        closedir(dir_stream_p);
      }
    return(0);
  }


int main(int argc, char ** argv)
{
  enum rsbac_attribute_t attr_list[RSBAC_FD_NR_ATTRIBUTES] = RSBAC_FD_ATTR_LIST;
  enum rsbac_attribute_t attr_list_dev[RSBAC_DEV_NR_ATTRIBUTES] = RSBAC_DEV_ATTR_LIST;
  int res = 0;
  char tmp1[RSBAC_MAXNAMELEN],tmp2[RSBAC_MAXNAMELEN],tmp3[RSBAC_MAXNAMELEN];
  int i,j;

  locale_init();
  
  progname = argv[0];
  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'h':
                use();
                return 0;
              case 'v':
                verbose++;
                break;
              case 'r':
                recurse=1;
                break;
              case 'e':
                inherit=1;
                break;
              case 'n':
                {
                  char tmp[80];
                  int i;

                  for(i=0; i<R_NONE; i++)
                    printf("%s\n", get_request_name(tmp, i));
                  exit(0);
                }
              case 'a':
              case 'A':
                if(   (argc > 2)
                   && ((attr = get_attribute_nr(argv[2])) != A_none)
                  )
                  {
                    get_switch_target_name(tmp1, get_attr_module(attr));
                    get_attribute_name(tmp2, attr);
                    get_attribute_param(tmp3, attr);
                    printf("[%-4s] %s\n\t%s\n",tmp1,tmp2,tmp3);
                    exit(0);
                  }
                printf(gettext("- attribute (string) and returned value = see following lists:\n"));
                printf(gettext("- FILE, DIR, FIFO and SYMLINK:\n"));
                for (j=0;j<RSBAC_FD_NR_ATTRIBUTES;j++)
                  {
                    get_switch_target_name(tmp1, get_attr_module(attr_list[j]));
                    get_attribute_name(tmp2,attr_list[j]);
                    get_attribute_param(tmp3,attr_list[j]);
                    printf("[%-4s] %s\n\t%s\n",tmp1,tmp2,tmp3);
                  }
                printf("\n- DEV:\n");
                for (j=0;j<RSBAC_DEV_NR_ATTRIBUTES;j++)
                  {
                    get_switch_target_name(tmp1, get_attr_module(attr_list_dev[j]));
                    get_attribute_name(tmp2,attr_list_dev[j]);
                    get_attribute_param(tmp3,attr_list_dev[j]);
                    printf("[%-4s] %s\n\t%s\n",tmp1,tmp2,tmp3);
                  }
                exit(0);
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;
              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if(argc > 1)
    {
      module = get_switch_target_nr(argv[1]);
      if(module != SW_NONE)
        {
          argv++;
          argc--;
        }
    }
  if (argc > 3)
    {
      if(verbose)
        printf(gettext("%s: %i targets\n\n"), progname, argc - 3);
      target = get_target_nr(argv[1]);
      if(target == T_NONE)
        {
          fprintf(stderr,
                  gettext("%s: invalid target type %s\n"), progname, argv[1]);
          exit(1);
        }
      target_n = argv[1];
      attr = get_attribute_nr(argv[2]);
      for (i=3;i < (argc);i++)
        {
          process(argv[i]);
        }
    }
  else
    {
      use();
      return 1;
    }
  return (res);
}
