# ifndef _SKIT_OUTHB_H
# define _SKIT_OUTHB_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
//  Several user I/O functions for Harwell-Boeing files are provided,
//  together with various utility functions.
//  For a description of the Harwell-Boeing standard, see:
//
//            Duff, et al.,  ACM TOMS Vol.15, No.1, March 1989
//
//  The writeHB function writes the specified matrix and optional
//  right-hand-side(s) to that file in Harwell-Boeing format.
//
# include "rheolef/outhb.h"
# include "rheolef/vec.h"
# include "rheolef/iotrick.h" // TODO: delete it !
namespace rheolef { 

template <
	class Size, 
	class IteratorValue, 
	class IteratorSize>
void
print_harwell_boeing (
        std::ostream& s,
        Size nptr,
        Size nidx,
        Size nnz,
        IteratorValue val,
        IteratorSize  ptr,
        IteratorSize  idx,
        Size nrhs,
        const char* type, 
        const char* title, 
        const char* key,
        const char* rhstype)
{
	// NOTE: offset for storage for indexes
	//       = 0 for fortran index style (1..nrow,1..ncol) ; this is the default
	//       = 1 for C/C++   index style (0..nrow-1,0..ncol-1)
	const int offset = 1;

	Size ptrcrd = (nptr+1)/8;
	if ( (nptr+1)%8 != 0) ptrcrd++;

	Size idxcrd = nnz/8;
	if ( nnz % 8 != 0) idxcrd++;

	Size valcrd = nnz/4;
	if ( nnz % 4 != 0) valcrd++;

	Size rhscrd = nrhs*nidx/4; 
	if ( nrhs*nidx%4 != 0) rhscrd++;

	Size solcrd = nrhs*nptr/4; 
	if ( nrhs*nptr%4 != 0) solcrd++;
	if (rhstype [1] == 'G') rhscrd += solcrd;
	if (rhstype [2] == 'X') rhscrd += solcrd;

	Size totcrd = ptrcrd+idxcrd+valcrd+rhscrd;

	const char *ptrfmt = "(8I10)          ";
	const char *idxfmt = ptrfmt;
	const char *valfmt = "(4E20.16)           ";
	const char *rhsfmt = valfmt;

	//  Print header information
        skit_form(s, "%-72s",  title);
        skit_form(s, "%-8s\n", key);
        skit_form(s, "%14d",   int(totcrd));
        skit_form(s, "%14d",   int(ptrcrd));
        skit_form(s, "%14d",   int(idxcrd));
        skit_form(s, "%14d",   int(valcrd));
        skit_form(s, "%14d\n", int(rhscrd));
        
        skit_form(s, "%3s",    type);
        skit_form(s, "%11s",   "          ");
        skit_form(s, "%14d",   int(nidx));
        skit_form(s, "%14d",   int(nptr));
        skit_form(s, "%14d\n", int(nnz));
        
        skit_form(s, "%16s",   ptrfmt);
        skit_form(s, "%16s",   idxfmt);
        skit_form(s, "%20s",   valfmt);
        skit_form(s, "%20s\n", rhsfmt);
        if (nrhs != 0) {
        
            // Print optional fifth header line for right-hand-side information
            skit_form(s, "%-3s           ", rhstype);
            skit_form(s, "%d\n", int(nrhs));
        }
	//  Print row pointers:
	for (Size i = 0; i < nptr+1; i++) {

	    Size entry = ptr[i] + offset;
	    skit_form(s, "%10d",int(entry));
	    if ( (i+1)%8 == 0 ) s << std::endl;
	}
	if ( (nptr+1) % 8 != 0 ) s << std::endl;

	//  Print col indices
	for (Size i = 0; i < nnz; i++) {

	    Size entry = idx[i] + offset;
	    skit_form(s, "%10d", int(entry));
	    if ( (i+1)%8 == 0 ) s << std::endl;
	}
	if ( nnz % 8 != 0 ) s << std::endl;

	//  Print values
    	for (Size i = 0; i < nnz; i++) {

   	    skit_form(s, "% 20.12E", double(val[i]));
       	    if ( (i+1)%4 == 0 ) s << std::endl;
    	}
    	if ( nnz % 4 != 0 ) s << std::endl;
}
template <class Size, class Size2, class Size3, class IteratorValue>
void
print_harwell_boeing (
        std::ostream& s,
        IteratorValue u,                             
        Size  size,
        Size2 ivec,
        Size3 nrhs)
{
	// accumulate indexes for end-of-line output
	Size nprec = ivec*size;
	for (Size i = 0; i < size; i++) {
          
	    skit_form(s, "% 20.12E", double(u[i]));
            if ((nprec+i+1) % 4 == 0) 
		s << std::endl;
    	}
        if (ivec + 1 == nrhs && (nprec+size) % 4 != 0) {
		// end of a vector block: add an eoln
		s << std::endl;
	}
}
}// namespace rheolef
# endif // _SKIT_OUTHB_H
