#!/usr/bin/python

"""Read power meters available via IPMI."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import discovery
import pwrkap_data
import datetime
import os
import popen2

class ipmi_power_meter(pwrkap_data.power_meter):
	"""Driver for power meters available via IPMI."""

	def __init__(self, sensor_name, ipmitool_args = ""):
		self.sensor_name = sensor_name
		self.latency = None
		self.ipmitool_args = ipmitool_args
		self.meter_name = "ipmitool"

	def read(self):
		try:
			before = datetime.datetime.utcnow()
			proc = popen2.Popen4("ipmitool %s sensor get '%s'" % \
					     (self.ipmitool_args, self.sensor_name), 512)
			res = proc.wait()
			if res != 0:
				return None
			input = proc.fromchild
			for line in input:
				if line.strip().startswith("Sensor Reading"):
					foo = line.split()
					if foo[3] == "na":
						return None
					else:
						return float(foo[3])
			return None
		finally:
			after = datetime.datetime.utcnow()
			if self.latency == None:
				self.latency = (after - before)
			else:
				self.latency = (8 * self.latency + 2 * (after - before)) / 10

	def get_latency(self):
		return self.latency

	def inventory(self):
		return (self.meter_name, {"name": self.sensor_name})

def ipmi_meter_discover():
	"""Discover IPMI meters."""

	os.system("modprobe -q ipmi-si")
	os.system("modprobe -q ipmi-devintf")

	proc = popen2.Popen4("ipmitool sensor", 512)
	# Don't wait for output; apparently RHEL5 drops all pipe data
	# after the process terminates.
	#res = proc.wait()
	#if res != 0:
	#	return
	input = proc.fromchild
	for line in input:
		foo = line.split("|")
		if len(foo) > 2 and foo[2].strip() == "Watts":
			meter = ipmi_power_meter(foo[0].strip())
			discovery.PWRKAP_POWER_METERS.append(meter)

def ipmi_init():
	"""Set up IPMI discovery functions."""
	discovery.PWRKAP_METER_DISCOVERY.append(ipmi_meter_discover)

	return True

ipmi_init()
