/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __POWERD_INTERNAL_H__
#define __POWERD_INTERNAL_H__

#include <uuid/uuid.h>
#include <glib.h>
#include <gio/gio.h>

#include "powerd-object.h"
#include "powerd.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Length of a UUID string, including trailing nul */
#define UUID_STR_LEN 37

struct powerd_display_request {
    uuid_t cookie;
    enum powerd_display_state state;
    guint32 flags;
};

enum powerd_override_reason {
    POWERD_OVERRIDE_REASON_POWER_BUTTON,
    POWERD_OVERRIDE_REASON_PROXIMITY,

    POWERD_NUM_OVERRIDE_REASONS
};

struct DbusNameWatch {
    uint watch_id;
    uint ref_count;
};

/* Used to track call state */
struct call_data {
    char *obj_name;
    GDBusProxy *ofono_proxy;
};

void powerd_shutdown(void);
void powerd_exit(int exit_code);
void powerd_hal_signal_activity(void);
void powerd_reset_activity_timer(int add);
void powerd_dbus_init_complete(void);
int powerd_is_mainloop(void);

/* Autobrightness functions */
void powerd_new_als_event(double lux);
void powerd_autobrightness_enable(void);
void powerd_autobrightness_disable(void);
gboolean powerd_autobrightness_available(void);
int powerd_autobrightness_init(void);
void powerd_autobrightness_deinit(void);

/* Backlight functions */
int powerd_backlight_init(void);
void powerd_backlight_deinit(void);
int powerd_get_brightness(void);
int powerd_get_max_brightness(void);
int powerd_set_brightness(int brightness);
int powerd_set_user_brightness(int brightness);
void powerd_dim_screen(void);
gboolean handle_get_brightness_params(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation);

/* Display functions */
void powerd_brightness_set_value(gint value);
gboolean powerd_display_enabled(void);
gboolean display_set_power_mode(int display, const char *powerd_mode);
void powerd_set_display_state(struct powerd_display_request *req);
int powerd_display_init(void);
void powerd_proximity_event(gboolean near);
void powerd_display_set_override(enum powerd_override_reason reason);
void powerd_display_clear_override(enum powerd_override_reason reason);
gboolean handle_user_autobrightness_enable(PowerdSource *obj,
                                           GDBusMethodInvocation *invocation,
                                           gboolean enable);
gboolean handle_set_user_brightness(PowerdSource *obj,
                                    GDBusMethodInvocation *invocation,
                                    gint brightness);

/* Display request functions */
int powerd_add_display_request(struct powerd_display_request *request,
                               const char *name);
int powerd_update_display_request(struct powerd_display_request *request);
int powerd_remove_display_request(uuid_t cookie);
void display_request_init(void);
void display_request_deinit(void);
gboolean handle_add_display_request(PowerdSource *obj,
                                    GDBusMethodInvocation *invocation,
                                    const char *name, int state, guint32 flags);
gboolean handle_update_display_request(PowerdSource *obj,
                                       GDBusMethodInvocation *invocation,
                                       const gchar *ext_cookie, int state,
                                       guint32 flags);
gboolean handle_clear_display_request(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation,
                                      const gchar *ext_cookie);
gboolean handle_list_display_requests(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation);
void clear_disp_state_by_owner(const char *owner);
void powerd_display_exit_suspend(void);

/* System power state requests */
gboolean handle_list_sys_requests(PowerdSource *obj,
                                  GDBusMethodInvocation *invocation);
gboolean handle_request_sys_state(PowerdSource *obj,
                                  GDBusMethodInvocation *invocation,
                                  const char *name, int state);
gboolean handle_clear_sys_state(PowerdSource *obj,
                                GDBusMethodInvocation *invocation,
                                gchar *cookie);
gboolean request_sys_state_internal(const char *name, int state,
                                    uuid_t *cookie, const char *owner);
gboolean clear_sys_state_internal(uuid_t cookie);
void power_request_transition_acked(void);
void update_system_state(void);
void power_request_init(void);
void power_request_deinit(void);
enum SysPowerStates current_system_power_state(void);
const gchar * state_to_string(int state);
void powerd_sys_state_signal_emit(enum SysPowerStates state);
void powerd_display_state_signal_emit(enum powerd_display_state state,
                                      guint32 flags);
void clear_sys_state_by_owner(const char *owner);
gboolean powerd_suspend_active(void);

/* dbus name watch functions */
void dbus_name_watch_init(void);
void dbus_name_watch_deinit(void);
void powerd_dbus_name_watch_add(const char *owner);
void powerd_dbus_name_watch_remove(const char *owner);

/* dbus callbacks */
void powerd_bus_acquired_cb(GDBusConnection *connection, const gchar *name,
                            gpointer user_data);
void powerd_name_acquired_cb(GDBusConnection *connection, const gchar *name,
                             gpointer user_data);
void powerd_name_lost_cb(GDBusConnection *connection, const gchar *name,
                         gpointer user_data);
void powerd_name_vanished_cb(GDBusConnection *connection, const gchar *name,
                             gpointer user_data);
void ofono_get_modems_cb(GObject *source_object, GAsyncResult *res,
                         gpointer user_data);
void ofono_voicecall_get_props_cb(GObject *source_object, GAsyncResult *res,
                         gpointer user_data);

/* dbus signal handlers */
void on_ofono_manager_signal(GDBusProxy *proxy, gchar *sender_name,
    gchar *signal_name, GVariant *parameters, gpointer user_data);
void on_ofono_voicecall_signal(GDBusProxy *proxy, gchar *sender_name,
    gchar *signal_name, GVariant *parameters, gpointer user_data);
void on_ofono_voicecallmanager_signal(GDBusProxy *proxy, gchar *sender_name,
    gchar *signal_name, GVariant *parameters, gpointer user_data);
void on_ofono_messagemanager_signal(GDBusProxy *proxy, gchar *sender_name,
    gchar *signal_name, GVariant *parameters, gpointer user_data);
void on_ofono_ussd_signal(GDBusProxy *proxy, gchar *sender_name,
    gchar *signal_name, GVariant *parameters, gpointer user_data);

/* dbus proxy async setup */
void ofono_manager_proxy_connect_cb(GObject *source_object, GAsyncResult *res,
    gpointer user_data);
void ofono_voicecall_proxy_connect_cb(GObject *source_object, GAsyncResult *res,
    gpointer user_data);
void ofono_proxy_connect_cb(GObject *source_object, GAsyncResult *res,
    gpointer user_data);

/* Sensor functions */
void powerd_sensors_proximity_enable(void);
void powerd_sensors_proximity_disable(void);
void powerd_sensors_als_enable(void);
void powerd_sensors_als_disable(void);

/* Power source functions */
int powerd_ps_init(void);
void powerd_ps_deinit(void);

/* Client functions */
int powerd_client_register(const char *dbus_name, const char *name);
void powerd_client_unregister(const char *dbus_name);
gboolean powerd_client_transition_start(int state);
void powerd_client_transition_finish(int state);
int powerd_client_ack(const char *dbus_name, int state);
int powerd_client_init(void);
void powerd_client_deinit(void);

/* Statistics functions */
void powerd_account_request_sys_state(const char *dbus_name, const char *name);
void powerd_account_clear_sys_state(const char *dbus_name, const char *name);
void powerd_account_add_display_req(const char *dbus_name, const char *name,
                                    const struct powerd_display_request *req);
void powerd_account_update_display_req(const char *dbus_name,
                                       const char *name,
                                       const struct powerd_display_request *req);
void powerd_account_clear_display_req(const char *dbus_name, const char *name);
void powerd_log_stats(void);
int powerd_stats_init(void);
void powerd_stats_deinit(void);
gboolean handle_get_sys_request_stats(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation);
gboolean handle_get_disp_request_stats(PowerdSource *obj,
                                       GDBusMethodInvocation *invocation);

/* Utility functions */
int powerd_run_mainloop_sync(int (*func)(gpointer), gpointer data);
guint powerd_uuid_hash(gconstpointer key);
gboolean powerd_uuid_equal(gconstpointer a, gconstpointer b);

#ifdef __cplusplus
}
#endif

#endif
