PROGRAM calc_errorfield
!
!****	calc_errorfield
!
!	Purpose:
!	--------
!	This program read an interpolated field and calculates the error
!       wrt the source analytical field
!       
!***	History:
!       -------
!       Version   Programmer      Date        Description
!       -------   ----------      ----        -----------
!         1.0     Sophie Valcke   2006/03/28  Creation
!*----------------------------------------------------------------
!** ++ calling argument
!       1- source grid acronym (CHARACTER(len=4))
!       2- analytical field (1, 2, or 3, CHARACTER(len=1))
!             1) F = 2 - cos[Pi*acos(cos(lat)cos(lon)]
!             2) F = 2 + [(cos(lat))**2]*cos(2*lon)
!             3) F = 2 + [(sin(2*lon))**16]*cos(16lon)
!
!** ++ modules and includes
!
  IMPLICIT NONE
  INCLUDE 'netcdf.inc'
!
!** ++ declarations
!
  CHARACTER(len=4)       :: cl_grd
  CHARACTER(len=1)       :: cl_fld
  CHARACTER(len=8)       :: cl_nam
  CHARACTER(len=3)       :: cl_msk
  INTEGER                :: il_fileid, il_lonid, il_latid, il_mskid 
  INTEGER                :: il_ficid, il_fldid
  INTEGER                :: il_ndims, il_i, il_j, il_ij, il_type, il_b, il_c
  INTEGER, DIMENSION(:), ALLOCATABLE :: il_dimids, il_msk, il_i_dimid, il_j_dimid
  REAL*4,  DIMENSION(:), ALLOCATABLE :: rl_lon
  REAL*4,  DIMENSION(:), ALLOCATABLE :: rl_lat
  REAL*4,  DIMENSION(:), ALLOCATABLE           :: rl_fld
  REAL*4,  DIMENSION(:), ALLOCATABLE           :: rl_fldint
  REAL*4,  DIMENSION(:), ALLOCATABLE           :: rl_err
  DOUBLE PRECISION, PARAMETER                 :: two = 2.
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_lon
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_lat
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_fld
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_fldint
  DOUBLE PRECISION, DIMENSION(:), ALLOCATABLE :: dl_err
  DOUBLE PRECISION :: dl_min_err, dl_max_err, dl_sum_err
  DOUBLE PRECISION, PARAMETER    :: dp_pi=3.14159265359
  DOUBLE PRECISION, PARAMETER    :: dp_length= 1.2*dp_pi
  DOUBLE PRECISION, PARAMETER    :: dp_conv = dp_pi/180.
  LOGICAL                :: ll_dbl
  INTEGER, PARAMETER :: ip_single_p = SELECTED_REAL_KIND(6,37)
  INTEGER, PARAMETER :: ip_double_p = SELECTED_REAL_KIND(12,307)
!
!*----------------------------------------------------------------
!
! Get argument giving grid acronym
!
  CALL getarg(1,cl_grd)
  PRINT *, 'Target grid acronym = ', cl_grd  
  CALL getarg(2,cl_fld)
  PRINT *, 'Analytical field number = ', cl_fld
  CALL getarg(3,cl_msk)
  PRINT *, ' Mask error field with target mask = ', cl_msk
!
! Open and read file of interpolated field
!
  CALL hdlerr(NF_OPEN('fldou.nc', NF_NOWRITE, il_fileid))
!  
  cl_nam=cl_grd//".lon"
  WRITE(*,*) 'cl_nam lon', cl_nam
  CALL hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_lonid))
  WRITE(*,*) 'il_lonid', il_lonid
  cl_nam=cl_grd//".lat" 
WRITE(*,*) 'cl_nam lat', cl_nam
  CALL hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_latid))
WRITE(*,*) 'il_latid', il_latid
  cl_nam="field_ou"
WRITE(*,*) 'cl_nam fld', cl_nam
  CALL hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_fldid))
WRITE(*,*) 'il_fldid', il_fldid
!
  CALL hdlerr(NF_INQ_VARNDIMS(il_fileid, il_lonid, il_ndims))
WRITE(*,*) 'il_ndims =', il_ndims
  ALLOCATE (il_dimids(il_ndims))
  CALL hdlerr(NF_INQ_VARDIMID(il_fileid, il_lonid, il_dimids))
WRITE(*,*) 'il_dimids =', il_dimids
  CALL hdlerr(NF_INQ_VARTYPE(il_fileid, il_lonid, il_type))
  ll_dbl = .false.
  IF (il_type == NF_DOUBLE) ll_dbl = .TRUE.
WRITE(*,*) 'il_dimids =', il_dimids
  IF (il_ndims == 1) THEN
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(1), il_ij))
!      WRITE(*,*) 'il_ij= ', il_ij
  ELSE
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(1), il_i))
!      WRITE(*,*) 'il_i= ', il_i
      CALL hdlerr(NF_INQ_DIMLEN(il_fileid,il_dimids(2), il_j))
!      WRITE(*,*) 'il_j= ', il_j
      il_ij = il_i*il_j
!      WRITE(*,*) 'il_ij= ', il_ij
  ENDIF
  IF (ll_dbl) THEN
      ALLOCATE (dl_lon(il_ij))
      ALLOCATE (dl_lat(il_ij))
      ALLOCATE (dl_fldint(il_ij))
      CALL hdlerr(NF_GET_VAR_DOUBLE (il_fileid, il_lonid, dl_lon))
      CALL hdlerr(NF_GET_VAR_DOUBLE (il_fileid, il_latid, dl_lat))
      CALL hdlerr(NF_GET_VAR_DOUBLE (il_fileid, il_fldid, dl_fldint))
  ELSE
      ALLOCATE (rl_lon(il_ij))
      ALLOCATE (rl_lat(il_ij))
      ALLOCATE (rl_fldint(il_ij))
      CALL hdlerr(NF_GET_VAR_REAL (il_fileid, il_lonid, rl_lon))
      CALL hdlerr(NF_GET_VAR_REAL (il_fileid, il_latid, rl_lat))
      CALL hdlerr(NF_GET_VAR_REAL (il_fileid, il_fldid, rl_fldint))
      WRITE(78,*) rl_fldint
  ENDIF
  CALL hdlerr(NF_CLOSE(il_fileid))
  !
!
! Open mask.nc file and get mask file
!
  CALL hdlerr(NF_OPEN('masks.nc', NF_NOWRITE, il_fileid))
!  
  cl_nam=cl_grd//".msk" 
!  write(*,*) 'cl_nam ', cl_nam
  call hdlerr(NF_INQ_VARID(il_fileid, cl_nam, il_mskid))
!  write(*,*) 'il_mskid', il_mskid
!
  ALLOCATE (il_msk(il_ij))
  CALL hdlerr(NF_GET_VAR_INT (il_fileid, il_mskid, il_msk))
  CALL hdlerr(NF_CLOSE(il_fileid))
!
! Create analytical field on target grid and apply mask
!
  IF (ll_dbl) THEN
      dl_lat = dl_lat * dp_conv
      dl_lon = dl_lon * dp_conv
      ALLOCATE (dl_fld(il_ij))
      IF (cl_fld == '1') THEN
          dl_fld =  two - COS(dp_pi*(ACOS(COS(dl_lat)*COS(dl_lon))/dp_length))
      ELSE IF (cl_fld == '2') THEN
          dl_fld = two + COS(dl_lat)**2*COS(two*dl_lon)
      ELSE IF (cl_fld == '3') THEN
          dl_fld = two + SIN(two*dl_lat)**16*COS(16.*dl_lon)
      ELSE
          WRITE(*,*) 'Bad analytical field number (2nd calling argument)'
          WRITE(*,*) 'should be either "1", "2" or "3"'
          STOP
      ENDIF

!!$      WHERE (dl_fld .GT. 1.9 .AND. dl_fld .LT. 2.1)
!!$       dl_fld = 0.00
!!$      END WHERE   
!
  ELSE
      rl_lat = rl_lat * dp_conv
      rl_lon = rl_lon * dp_conv     
      ALLOCATE (rl_fld(il_ij))
      IF (cl_fld == '1') THEN
          rl_fld =  two - COS(dp_pi*(ACOS(-COS(rl_lat)*COS(rl_lon))/dp_length))
      ELSE IF (cl_fld == '2') THEN
          rl_fld = two + COS(rl_lat)**2*COS(two*rl_lon)
      ELSE IF (cl_fld == '3') THEN
          rl_fld = two + SIN(two*rl_lat)**16*COS(16.*rl_lon)
      ELSE
          WRITE(*,*) 'Bad analytical field number (2nd calling argument)'
          WRITE(*,*) 'should be either "1", "2" or "3"'
          STOP
      ENDIF
!
  ENDIF
!
! Calculate difference between analytical and interpolated fields
!
  IF (ll_dbl) THEN
      ALLOCATE (dl_err(il_ij))
      WHERE (abs(dl_fld) .GT. epsilon(1.0_ip_double_p))
          dl_err = (dl_fldint - dl_fld)/dl_fld
      ELSE WHERE
	  dl_err =  1.0
      END WHERE
  ELSE
      ALLOCATE (rl_err(il_ij))
      WHERE (abs(rl_fld) .GT. epsilon(1.0_ip_single_p))
          rl_err = (rl_fldint - rl_fld)/rl_fld
      ELSE WHERE
	  rl_err =  1.0
      END WHERE
  ENDIF
!
  IF (cl_msk == "YES") THEN
      ! Calculate and print statistics on non masked points
      IF (ll_dbl) THEN
          WHERE (il_msk == 1) 
              dl_err = -9999999.
          ELSE WHERE
              dl_err =  dl_err
          END WHERE
      ELSE
          WHERE (il_msk == 1) 
              rl_err = -9999999.
          ELSE WHERE
              rl_err =  rl_err
          END WHERE
      ENDIF
      WRITE(*,*)'ll_dbl=', ll_dbl
      WRITE(*,*)'Statistics on non masked points:'
      WRITE(*,*)'Min and max of the error field on non masked points:'
      IF (ll_dbl) THEN
          il_c = 0
          dl_sum_err = 0.0
          dl_min_err = 100000000.0
          dl_max_err = 0.0
          DO il_b=1,il_ij
            IF (dl_err(il_b) > -9999998.) THEN
                il_c = il_c + 1
!!$                WRITE(*,*) 'il_c = ', il_c
                dl_sum_err = dl_sum_err + abs(dl_err(il_b))
                IF (dl_err(il_b) < dl_min_err) dl_min_err = dl_err(il_b)
!!$                WRITE(*,*) 'dl_min_err = ', dl_min_err
                IF (dl_err(il_b) > dl_max_err) dl_max_err = dl_err(il_b)
!!$                WRITE(*,*) 'dl_max_err = ', dl_max_err     
            ENDIF
          ENDDO
          dl_sum_err = dl_sum_err / il_c
          WRITE(*,*) dl_min_err, dl_max_err
          WRITE(*,*)'Error mean on non masked points: ', dl_sum_err
          WRITE(*,*) 
      ENDIF
  ELSE IF (cl_msk == "NOT") THEN
      ! Calculate and print statistics on all target points
      WRITE(*,*)'Statistics on all target points:'
      WRITE(*,*)'Min and max of the error field on all target points:'
      IF (ll_dbl) THEN
          WRITE(*,*) MINVAL(dl_err), MAXVAL(dl_err)
          WRITE(*,*)'Error mean on all target points: '
          WRITE(*,*) SUM(ABS(dl_err))/(il_ij)
      ELSE
          WRITE(*,*) MINVAL(rl_err), MAXVAL(rl_err)
          WRITE(*,*)'Error mean on all target points: '
          WRITE(*,*) SUM(ABS(rl_err))/(il_ij)
      ENDIF
  ELSE
      WRITE(*,*) 'Bad choice for 3rd calling argument'
      WRITE(*,*) 'should be either "YES" or "NOT"'
      STOP
  ENDIF
!
! Create file and write the field
!
! Create file
  CALL hdlerr(NF_CREATE('error.nc', 0, il_ficid))
!
! Create dimensions
  IF (il_ndims == 1) THEN
      CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_ij', il_ij, il_dimids(1)))
  ELSE
      CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_i', il_i,il_dimids(1)))
      CALL hdlerr(NF_DEF_DIM(il_ficid, 'il_j', il_j,il_dimids(2)))
  ENDIF
!
! Create variables
  CALL hdlerr(NF_DEF_VAR (il_ficid, 'error', il_type, il_ndims, &
     il_dimids, il_fldid))
  cl_nam=cl_grd//".lon" 
  CALL hdlerr(NF_DEF_VAR (il_ficid, cl_nam, il_type, il_ndims, il_dimids, &
     il_lonid))
WRITE(*,*) 'il_lonid = ', il_lonid
  cl_nam=cl_grd//".lat" 
  CALL hdlerr(NF_DEF_VAR (il_ficid, cl_nam, il_type, il_ndims, il_dimids, &
     il_latid))
WRITE(*,*) 'il_latid = ', il_latid
!
! End of definition phase
  CALL hdlerr(NF_ENDDEF(il_ficid))
!
! Write the field
  IF (ll_dbl) THEN
      CALL hdlerr(NF_PUT_VAR_DOUBLE (il_ficid, il_lonid, dl_lon)) 
      CALL hdlerr(NF_PUT_VAR_DOUBLE (il_ficid, il_latid, dl_lat)) 
      CALL hdlerr(NF_PUT_VAR_DOUBLE (il_ficid, il_fldid, dl_err))
  ELSE
      CALL hdlerr(NF_PUT_VAR_REAL (il_ficid, il_lonid, rl_lon)) 
      CALL hdlerr(NF_PUT_VAR_REAL (il_ficid, il_latid, rl_lat))
      CALL hdlerr(NF_PUT_VAR_REAL (il_ficid, il_fldid, rl_err))
  ENDIF
!
! Close the file
  CALL hdlerr(NF_CLOSE(il_ficid))
!
!*----------------------------------------------------------------
!
END PROGRAM calc_errorfield
SUBROUTINE hdlerr(istatus)

  INTEGER                 :: istatus
  INCLUDE 'netcdf.inc'

  IF (istatus .ne. NF_NOERR) THEN
      print *, NF_STRERROR(istatus)
      stop 'stopped'
  ENDIF

  RETURN

END SUBROUTINE hdlerr
!
!*----------------------------------------------------------------
!*----------------------------------------------------------------
!
