/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_FILTER_H
#define MEDIASCANNER_FILTER_H

// Standard Library
#include <string>
#include <memory>

// Media Scanner
#include "mediascanner/property.h"

namespace Lucene {
typedef boost::shared_ptr<class Query> QueryPtr;
typedef boost::shared_ptr<class QueryParser> QueryParserPtr;
}

namespace mediascanner {

class Filter {
protected:
    class Private;

    explicit Filter(Private *d);

public:
    Filter();
    ~Filter();

    Filter(const Filter &other);
    Filter &operator=(const Filter &other);

    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr parser,
                                std::wstring *error_message) const;

protected:
    template<typename T> T* data() {
        return static_cast<T *>(d.get());
    }

private:
    std::shared_ptr<Private> d;
};

////////////////////////////////////////////////////////////////////////////////

class QueryStringFilter : public Filter {
    class Private;

public:
    explicit QueryStringFilter(const std::wstring &text);
};

class SubStringFilter : public Filter {
    class Private;

public:
    explicit SubStringFilter(const std::wstring &text);
};

class FullTextFilter : public Filter {
    class Private;

public:
    explicit FullTextFilter(const std::wstring &text);
};

////////////////////////////////////////////////////////////////////////////////

class PrefixFilter : public Filter {
    class Private;

public:
    explicit PrefixFilter(const StringProperty &key, const std::wstring &value);
};

class ValueFilter : public Filter {
    class Private;

public:
    explicit ValueFilter(const Property::BoundValue &value);
    explicit ValueFilter(const Property &property,
                         const Property::Value &value);
};

class RangeFilter : public Filter {
    class Private;

public:
    explicit RangeFilter(const Property &property,
                         const Property::Value &lower_value,
                         const Property::Value &upper_value);

    explicit RangeFilter(const Property &property,
                         const Property::Value &lower_value,
                         const Property::Boundary &lower_boundary,
                         const Property::Value &upper_value,
                         const Property::Boundary &upper_boundary);
};

////////////////////////////////////////////////////////////////////////////////

class BooleanFilter : public Filter {
    class Private;

public:
    enum Occur { MUST,  SHOULD, MUST_NOT };

    class Clause {
    public:
        Clause(const Filter &filter, Occur occur)
            : filter_(filter)
            , occur_(occur) {
        }

        Filter filter() const {
            return filter_;
        }

        Occur occur() const {
            return occur_;
        }

    private:
        Filter filter_;
        Occur occur_;
    };

    BooleanFilter();

    void add_clause(const Filter &filter, Occur occur) {
        add_clause(Clause(filter, occur));
    }

    void add_clause(const Clause &clause);
};

} // namespace mediascanner

#endif // MEDIASCANNER_FILTER_H
