/*
 * Kylin-video
 *
 * Copyright (C) 2021, Tianjin KYLIN Information Technology Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Liu Cong <liucong1@kylinos.cn>
 *
 */

#include <QApplication>
#include <QDBusConnection>
#include <QDBusInterface>
#include <QDBusReply>
#include <QFileInfo>
#include <QScreen>
#include <QDebug>

#include <ukui-log4qt.h>

#include "mainwidget.h"
#include "global/global.h"
#include "global/paths.h"
#include "global/functions.h"

#include <syslog.h>
#include <fcntl.h>
#include <unistd.h>

#include "elog.h"
#include "elog_file.h"

using namespace Global;
static ElogFileCfg *cfg = new ElogFileCfg;

void initLog(){
    // 初始化 log
    elog_init();
    cfg->name = (char*)Paths::logPath().append("/kylin-video.log").toUtf8().constData();
    cfg->max_size = 1024 * 1024;
    cfg->max_rotate = 5;
    elog_file_config(cfg);
    elog_set_fmt(ELOG_LVL_ASSERT,   ELOG_FMT_ALL);
    elog_set_fmt(ELOG_LVL_ERROR,    ELOG_FMT_LVL | ELOG_FMT_TAG | ELOG_FMT_TIME );
    elog_set_fmt(ELOG_LVL_WARN,     ELOG_FMT_LVL | ELOG_FMT_TAG | ELOG_FMT_TIME );
    elog_set_fmt(ELOG_LVL_INFO,     ELOG_FMT_LVL | ELOG_FMT_TAG | ELOG_FMT_TIME );
    elog_set_fmt(ELOG_LVL_DEBUG,    ELOG_FMT_ALL & ~ELOG_FMT_FUNC);
    elog_set_fmt(ELOG_LVL_VERBOSE,  ELOG_FMT_ALL & ~ELOG_FMT_FUNC);
    elog_set_text_color_enabled(false);
    elog_start();

    if (Functions::isKirinCpu()) {
        int logfd = open((char*)Paths::logPath().append("/terminal.log").toUtf8().constData(), O_RDWR | O_CREAT | O_APPEND, 0644);
        assert(-1 != logfd);
        close(STDERR_FILENO);
        dup2(logfd, STDERR_FILENO);
        close(logfd);
        openlog(NULL, LOG_PERROR, LOG_DAEMON);
    }

    initUkuiLog4qt("kylin-video");
}

int main(int argc, char **argv)
{
    initLog();

    if(QString(qgetenv("XDG_SESSION_TYPE")) == "wayland") {
        qputenv("QT_QPA_PLATFORM", "wayland");
    }
    else {
#if (QT_VERSION >= QT_VERSION_CHECK(5, 6, 0))
    QCoreApplication::setAttribute(Qt::AA_EnableHighDpiScaling);
    QCoreApplication::setAttribute(Qt::AA_UseHighDpiPixmaps);
#endif
    }

    QApplication a(argc, argv);

    a.setWindowIcon(QIcon::fromTheme("kylin-video"));

    Global::global_init();

    QFile qss(":/qss/style.qss");
    if (!qss.open(QIODevice::ReadOnly)) {
        qWarning("Can't open the style sheet file: :/qss/res/style.qss.");
    }
    else {
        qApp->setStyleSheet(qss.readAll());
        qss.close();
    }

    QStringList arg_list;
    for(int i=1; i<argc; i++)
    {
        QFileInfo f(QString(argv[i]));
        if(f.isFile())
            arg_list << f.absoluteFilePath();
    }

    MainWidget w(arg_list);
    w.show();

    // wayland 窗口无边框
    if(isWayland)
    {
        UKUIDecorationManager::getInstance()->removeHeaderBar(w.windowHandle());
        UKUIDecorationManager::getInstance()->setCornerRadius(w.windowHandle(), 12, 12, 12, 12);

        // 显示在屏幕中间
#if 0
        QDBusInterface wayland("org.ukui.SettingsDaemon",
                               "/org/ukui/SettingsDaemon/wayland",
                               "org.ukui.SettingsDaemon.wayland");
        QDBusReply<int> reX = wayland.call("x");
        QDBusReply<int> reWidth = wayland.call("width");
        QDBusReply<int> reHeight = wayland.call("height");
        PlasmaShellManager::getInstance()->setPos(w.windowHandle(), QPoint(reX + (reWidth-w.width())/2, (reHeight-w.height())/2));
#else
        int sw = QGuiApplication::primaryScreen()->availableGeometry().width();
        int sh = QGuiApplication::primaryScreen()->availableGeometry().height();
        PlasmaShellManager::getInstance()->setPos(w.windowHandle(), QPoint((sw-w.width())/2, (sh-w.height())/2));
#endif
    }

    return a.exec();
}
