//**************************************************************************************************
//                                        PrcSimEngBase.cpp                                        *
//                                       -------------------                                       *
// Started     : 2004-04-25                                                                        *
// Last Update : 2014-12-17                                                                        *
// Copyright   : (C) 2004 by MSWaters                                                              *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "PrcSimEngBase.hpp"

//**************************************************************************************************
// Constructor.

PrcSimEngBase::PrcSimEngBase( void ) : PrcBase( wxPROCESS_REDIRECT )
{
  m_eSimEng = eSIMR_NONE;
}

//**************************************************************************************************
// Destructor.

PrcSimEngBase::~PrcSimEngBase( )
{
}

//**************************************************************************************************
// Set the simulation results file name.
//
// Argument List :
//   rosFName - A string containing the full path and file name
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcSimEngBase::bSetResultsFile( const wxString & rosFName )
{
  wxFileName  ofn1;

  ofn1 = rosFName;
  if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxT(".") );

  if( ! ofn1.IsOk( ) ) return( false );

  m_ofnResults = ofn1;

  return( true );
}

//**************************************************************************************************
// Save the console output to file.
//
// Argument List :
//   rosFName - The save file name
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcSimEngBase::bSaveOutput( const wxString & rosFName )
{
  wxOutputStream * poStdOut;
  wxInputStream  * poStdIn;
  wxInputStream  * poStdErr;
  wxFileName       ofn1;
  wxString         os1, os2;
  wxChar           oc1;

  // Check the validity of the save file name
  if( rosFName.IsEmpty( ) )         return( false );
  ofn1 = rosFName;
  if( ! ofn1.IsOk( ) )              return( false );

  // Open the file
  wxTextFile  oFileCct( ofn1.GetFullPath( ) );
  if( oFileCct.Exists( ) )
       { if( ! oFileCct.Open( ) )   return( false ); }
  else { if( ! oFileCct.Create( ) ) return( false ); }

  // Clear the file if it contains lines
  oFileCct.Clear( );

  wxMicroSleep( 100 ); // 23/03/2011 ??? What's this for?

  // Get input stream objects to standard out and standard error
  poStdOut = GetOutputStream( );
  poStdIn  = GetInputStream( );
  poStdErr = GetErrorStream( );

  // Save the simulator output to file
  while( bIsExec( ) || poStdIn->CanRead( ) || poStdErr->CanRead( ) )
  {
    // Input characters from stdout
    while( poStdIn->CanRead( ) )
    {
      // Get a character from the input stream
      oc1 = poStdIn->GetC( );

      // Build each string
      if( oc1==wxT('\t') || (oc1>31 && oc1!=127) )
        os1 << oc1;

      // Append completed strings to the log file
      if( oc1 == wxT('\n') )
      {
        oFileCct.AddLine( os1 );
        os1 = wxT("");
      }
    }

    // Input characters from stderr
    while( poStdErr->CanRead( ) )
    {
      // Get a character from the input stream
      oc1 = poStdErr->GetC( );

      // Build each string
      if( oc1==wxT('\t') || (oc1>31 && oc1!=127) )
        os2 << oc1;

      // Append completed strings to the log file
      if( oc1 == wxT('\n') )
      {
        oFileCct.AddLine( os2 );
        os2 = wxT("");
      }
    }

    // Release the processor for other tasks
    wxTheApp->Yield( );
  }

  // Delete the stream objects
  delete poStdOut;
  delete poStdIn;
  delete poStdErr;
  SetPipeStreams( NULL, NULL, NULL );

  // Save any unterminated strings
  if( os1.Length( ) > 0 ) oFileCct.AddLine( os1 );
  if( os2.Length( ) > 0 ) oFileCct.AddLine( os2 );
  oFileCct.AddLine( wxT("") );

  oFileCct.Write( ); // Save the changes to disk
  oFileCct.Close( ); // Close the results file

  return( true );
}

//**************************************************************************************************
// Make the process argument list based on the contents of a simulation object.
//
// Argument List :
//   roSimn - The simulation object
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcSimEngBase::bMakeArgLst( SimnBase & roSimn )
{
  wxString  os1;
  int       i1;

  // Clear error attributes
  m_osErrMsg.Empty( );

  // Only proceed if the simulator isn't already running
  if( bIsExec( ) )
  {
    SetErrMsg( wxT("Simulator already running") );
    return( false );
  }

  // Create the results file name
  os1 = roSimn.roGetLoadFile( ).GetPath( ) + wxT('/')
      + roSimn.roGetLoadFile( ).GetName( );
  i1 = os1.Find( wxT(".gspiceui") );
  if( i1 > 0 ) os1 = os1.Truncate( (size_t) i1 );
  os1 << wxT('.') << roGetBinary( ).GetName( );
  switch( roSimn.eGetAnaType( ) )
  {
    case eCMD_OP : os1 << wxT(".op"); break;
    case eCMD_DC : os1 << wxT(".dc"); break;
    case eCMD_AC : os1 << wxT(".ac"); break;
    case eCMD_TR : os1 << wxT(".tr"); break;
    case eCMD_FO : os1 << wxT(".fo"); break;
    case eCMD_DI : os1 << wxT(".di"); break;
    case eCMD_NO : os1 << wxT(".no"); break;
    case eCMD_PZ : os1 << wxT(".pz"); break;
    case eCMD_SE : os1 << wxT(".se"); break;
    case eCMD_TF : os1 << wxT(".tf"); break;
    default      : return( false );
  }

  // Set the results file name
  if( ! PrcSimEngBase::bSetResultsFile( os1 ) )
  {
    SetErrMsg( wxT("Couldn't set results file name") );
    return( false );
  }

  // Construct the default argument list
  os1 = roSimn.roGetSaveFile( ).GetFullPath( );
  if( ! bSetArgLst( os1 ) )
  {
    SetErrMsg( wxT("Couldn't set argument list") );
    return( false );
  }

  return( true );
}

//**************************************************************************************************
// Execute a simulation.
//
// Argument List :
//   roSimn - The simulation object
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcSimEngBase::bExec( void )
{
  wxString  os1, os2;

  // Execute the simulation
  if( ! PrcBase::bExecAsync( ) ) return( false );

  // Save the simulation results to the log file
  os1 = roGetLogFile( ).GetFullPath( );
  if( ! bSaveOutput( os1 ) )     return( false );

  // Copy the log file to the results file
  os2 = roGetResultsFile( ).GetFullPath( );
  if( ! wxCopyFile( os1, os2 ) ) return( false );

  return( true );
}

//**************************************************************************************************
