/****************************************************************************
** Dooble - The Secure Internet Web Browser
**
** Copyright (c) 2008, 2009 Alexis Megas, 
** Gunther van Dooble, and the Dooble Team. 
** All rights reserved.
**
**
** License: GPL2 only: 
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; version 2 of the License only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** or see here: http://www.gnu.org/licenses/gpl.html
**
**
** For the WebKit library, please see: http://webkit.org.
**
** THE CODE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY
** EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
** IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS APPLICATION, EVEN IF ADVISED
** OF THE POSSIBILITY OF SUCH DAMAGE.
**
** Please report all praise, requests, bugs, and problems to the project
** team and administrators: http://sf.net/projects/dooble.
**
** You can find us listed at our project page. New team members are welcome.
** The name of the authors should not be used to endorse or promote products
** derived from Dooble without specific prior written permission.
** If you use this code for other projects, please let us know.
**
** Web sites:
**   http://dooble.net/projects/dooble
**   http://dooble.sf.net
**   http://www.getadooble.com
****************************************************************************/


#include <QUrl>
#include <QStyle>
#include <QSettings>
#include <QResizeEvent>

#include "durlwidget.h"

durlwidget::durlwidget(QWidget *parent):QLineEdit(parent)
{
  goToolButton = new QToolButton(this);
  goToolButton->setToolTip(tr("Load this URL from the Web."));
  goToolButton->setIconSize(QSize(16, 16));
  goToolButton->setCursor(Qt::ArrowCursor);
  goToolButton->setStyleSheet("QToolButton {"
			      "border: none; "
			      "}");
  indexToolButton = new QToolButton(this);
  indexToolButton->setToolTip(tr("peek-a-dooble: "
                                 "Swift translation - "
                                 "Load this URL "
				 "from your local cache "
                                 "(under construction)."));
  indexToolButton->setIconSize(QSize(16, 16));
  indexToolButton->setCursor(Qt::ArrowCursor);
  indexToolButton->setStyleSheet("QToolButton {"
				 "border: none; "
				 "}");
  pulldownToolButton = new QToolButton(this);
  pulldownToolButton->setIconSize(QSize(16, 16));
  pulldownToolButton->setCursor(Qt::ArrowCursor);
  pulldownToolButton->setStyleSheet("QToolButton {"
				    "border: none; "
#ifdef Q_WS_MAC
				    "padding-right: 10px; "
#else
				    "padding-right: 15px; "
#endif
				    "}");
  slotSetIcons();
  connect(indexToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotPeekADooble(void)));
  connect(pulldownToolButton, SIGNAL(clicked(void)), this,
	  SLOT(slotPulldownClicked(void)));

  QMenu *menu = new QMenu();

  menu->setStyleSheet("QMenu::separator {"
		      "height: 1px; "
		      "background: lightgray; "
		      "margin-left: 0px; "
		      "margin-right: 0px; "
		      "}");
  pulldownToolButton->setMenu(menu);

  int frameWidth = style()->pixelMetric(QStyle::PM_DefaultFrameWidth);

  setStyleSheet
    (QString("QLineEdit {padding-right: %1px; padding-left: %2px;}").arg
     (indexToolButton->sizeHint().width() + goToolButton->sizeHint().width() +
      frameWidth + 1).arg
     (pulldownToolButton->sizeHint().width() + frameWidth + 1));
#if QT_VERSION >= 0x040500
  setMinimumHeight(sizeHint().height() + 5);
#endif
}

void durlwidget::addItem(const QString &text)
{
  if(pulldownToolButton->menu()->actions().size() > 0)
    pulldownToolButton->menu()->addSeparator();

  pulldownToolButton->menu()->addAction(text);
}

void durlwidget::setIcon(const QIcon &icon)
{
  if(icon.isNull())
    {
      QSettings settings;
      QSettings cfgSettings
	(settings.value("iconSet").toString(), QSettings::IniFormat);

      pulldownToolButton->setIcon
	(QIcon(cfgSettings.value("urlWidget/emptyIcon").toString()));
    }
  else
    pulldownToolButton->setIcon(icon);
}

void durlwidget::setText(const QString &text)
{
  QLineEdit::setText(text);
  setCursorPosition(0);
}

void durlwidget::selectAll(void)
{
  setCursorPosition(text().length());
  cursorBackward(true, text().length());
}

void durlwidget::resizeEvent(QResizeEvent *event)
{
  Q_UNUSED(event);
  int frameWidth = style()->pixelMetric(QStyle::PM_DefaultFrameWidth);
  QSize size1 = goToolButton->sizeHint();
  QSize size2 = indexToolButton->sizeHint();
  QSize size3 = pulldownToolButton->sizeHint();

  goToolButton->move
    (rect().right() - frameWidth - size1.width(),
     (rect().bottom() + 2 - size1.height()) / 2);
  indexToolButton->move
    (rect().right() - frameWidth - size1.width() - size2.width(),
     (rect().bottom() + 2 - size2.height()) / 2);
  pulldownToolButton->move
    (frameWidth - rect().left() + 1,
     (rect().bottom() + 2 - size3.height()) / 2);
  pulldownToolButton->menu()->setStyleSheet
    (pulldownToolButton->menu()->styleSheet() +
     QString("QMenu {"
 	     "width: %1px; "
	     "}").arg(width()));

  if(selectedText().isEmpty())
    setCursorPosition(0);
}

void durlwidget::setItemIcon(const int index , const QIcon &icon)
{
  QList<QAction *> actions = pulldownToolButton->menu()->actions();

  if(index < actions.size())
    {
      if(icon.isNull())
	{
	  QSettings settings;
	  QSettings cfgSettings
	    (settings.value("iconSet").toString(), QSettings::IniFormat);

	  actions.at(index)->setIcon
	    (QIcon(cfgSettings.value("urlWidget/emptyIcon").toString()));
	}
      else
	actions.at(index)->setIcon(icon);
    }
}

void durlwidget::setItemText(const int index, const QString &text)
{
  QList<QAction *> actions = pulldownToolButton->menu()->actions();

  if(index < actions.size())
    actions.at(index)->setText(text);
}

void durlwidget::slotPulldownClicked(void)
{
#ifdef Q_WS_MAC
  pulldownToolButton->menu()->exec(mapToGlobal(QPoint(0, height() + 1)));
#else
  pulldownToolButton->menu()->exec(mapToGlobal(QPoint(0, height())));
#endif
}

int durlwidget::findText(const QString &text)
{
  int index = -1;
  QList<QAction *> actions = pulldownToolButton->menu()->actions();

  for(int i = 0; i < actions.size(); i++)
    if(actions.at(i)->text() == text)
      {
	index = i;
	break;
      }

  return index;
}

QMenu *durlwidget::menu(void)
{
  return pulldownToolButton->menu();
}

QToolButton *durlwidget::goButton(void)
{
  return goToolButton;
}

void durlwidget::slotSetIcons(void)
{
  QSettings settings;
  QSettings cfgSettings
    (settings.value("iconSet").toString(), QSettings::IniFormat);

  cfgSettings.beginGroup("urlWidget");
  goToolButton->setIcon(QIcon(cfgSettings.value("goToolButton").toString()));
  indexToolButton->setIcon
    (QIcon(cfgSettings.value("indexToolButton").toString()));
  pulldownToolButton->setIcon
    (QIcon(cfgSettings.value("pulldownToolButton").toString()));
}

void durlwidget::slotPeekADooble(void)
{
  emit loadPageSignal
    (QUrl
     (QString("http://translate.google.com/translate?&sl=fr&tl=en&u=%1").arg
      (text().trimmed())));
}
