/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
   \file cdw_input_line.c

   File implements widget that allows entering string values
   into single line input field. The widget also sports a message
   area where an error message is printed. The message is printed
   whenever entered string contains insecure characters.

   The widget doesn't accept string with insecure characters,
   as defined in cdw_string.h -> CDW_STRING_UNSAFE_CHARS_STRING.
*/

#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "cdw_input_line.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_window.h"



static void     cdw_input_line_driver_message(CDW_INPUT_LINE *input_line, char *insecure);
static cdw_rv_t cdw_input_line_new_form_setup(CDW_INPUT_LINE *input_line, const char *initial_string);
static int      cdw_input_line_form_driver(CDW_INPUT_LINE *input_line);
static cdw_rv_t cdw_input_line_get_content(CDW_INPUT_LINE *input_line, char **output_buffer);


/**
   \brief Create new input line widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Create widget that consists of a single line entry field,
   and single line message field.
   User enters data in entry field, and input line displays
   error message in message field (if necessary).
   Both fields have the same width (\p n_cols).

   The entry field has width \p n_cols, and is positioned in \p parent
   window in place specified by \p begin_y and \p begin_x.
   The field may be pre-filled with \p initial_string.
   You may put a restriction on number of characters that can be
   entered into the field by setting \p chars_max to value > 0.
   Input behavior may be tuned by \p input_type:
   none/hidden/integer/numeric, see cdw_ncurses.h for specifics.

   This widget is just two lines (entry + message), you have to
   add prompt and other elements separately.

   \param parent - window in which to display the widget
   \param begin_y - y coordinate of beginning of the widget in parent window
   \param begin_x - x coordinate of beginning of the widget in parent window
   \param n_cols - width of the input line
   \param initial_string - initial data displayed in input line
   \param input_type - type of input
   \param chars_max - limit on number of chars that can be entered by user

   \return NULL pointer on errors
   \return pointer to new widget on success
*/
CDW_INPUT_LINE *cdw_input_line_new(WINDOW *parent, int begin_y, int begin_x, int n_cols, const char *initial_string, int input_type, int chars_max)
{
	CDW_INPUT_LINE *input_line = (CDW_INPUT_LINE *) malloc(sizeof(CDW_INPUT_LINE));
	if (!input_line) {
		cdw_vdm ("ERROR: failed to allocate memory for input line widget\n");
		return (CDW_INPUT_LINE *) NULL;
	}

	input_line->parent = parent;
	input_line->input_type = input_type;
	input_line->chars_max = chars_max;
	input_line->n_cols = n_cols;

	input_line->save_on_tab = false;

	int n_lines = 2; /* one line for input window, second line for message area */
	input_line->win = derwin(input_line->parent, n_lines, n_cols, begin_y, begin_x);
	if (!(input_line->win)) {
		cdw_vdm ("ERROR: failed to create input line subwindow\n");
		cdw_input_line_delete(&input_line);
		return (CDW_INPUT_LINE *) NULL;
	}

	cdw_rv_t crv = cdw_input_line_new_form_setup(input_line, initial_string);
	if (crv != CDW_OK) {
		cdw_input_line_delete(&input_line);
		return (CDW_INPUT_LINE *) NULL;
	} else {
		wrefresh(input_line->win);
		return input_line;
	}
}





/**
   \brief Wrapper for code creating field of input line

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Input line is basically a form with one field (+ line for messages,
   but this is irrelevant in description of this function). This
   function calls several other functions that create and configure
   a form field.
   This is just a convenient wrapper.
   Call this function in cdw_input_line_new().

   \param input_line - allocated input line, to be configured further by this function
   \param initial_string - string to be put initially in the input line

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_input_line_new_form_setup(CDW_INPUT_LINE *input_line, const char *initial_string)
{
	input_line->field[1] = (FIELD *) NULL; /* obligatory ending element */
	input_line->field[0] = cdw_ncurses_new_input_field(1, input_line->n_cols, /* 1 - one line */
							   0, 0,                  /* begin_y, begin_x */
							   initial_string, (size_t) input_line->chars_max,
							   input_line->input_type, CDW_COLORS_INPUT);

	if (!(input_line->field[0])) {
		cdw_vdm ("ERROR: failed to create form field\n");
		return CDW_ERROR;
	}
	input_line->form = cdw_ncurses_new_form(input_line->parent, input_line->win, input_line->field);
	if (!(input_line->form)) {
		cdw_vdm ("ERROR: failed to create form with new_form()\n");
		return CDW_ERROR;
	}
	int r = set_current_field(input_line->form, input_line->field[0]);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set_current_field with set_current_field(), error code is %s\n", cdw_ncurses_error_string(r));
		return CDW_ERROR;
	}
	r = form_driver(input_line->form, REQ_END_LINE);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed in form_driver(), the function returns %s\n", cdw_ncurses_error_string(r));
		return CDW_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Delete an input line widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Deallocate all memory associated with \p input_line,
   free the \p input_line itself, set the widget to NULL.

   \param input_line - pointer to input line widget
*/
void cdw_input_line_delete(CDW_INPUT_LINE **input_line)
{
	cdw_assert (input_line, "ERROR: passing NULL pointer to the function\n");

	if (!(*input_line)) {
		cdw_vdm ("WARNING: passed NULL widget to the function\n");
		return;
	}

	/* From ncurses docs: "The functions free_field() and
	   free_form are available to free field and form objects.  It
	   is an error to attempt to free a field connected to a form,
	   but not vice-versa; thus, you will generally free your form
	   objects first.".

	   So first unpost form, then free form, then free field. */
	if ((*input_line)->form) {
		int rv = unpost_form((*input_line)->form);
		cdw_vdm_n ("unpost_form", rv, "");

		rv = free_form((*input_line)->form);
		cdw_vdm_n ("free_form", rv, "");

		(*input_line)->form = (FORM *) NULL;
	}

	if ((*input_line)->field[0]) {
		int rv = free_field((*input_line)->field[0]);
		cdw_vdm_n ("free_field", rv, "");
		(*input_line)->field[0] = (FIELD *) NULL;
	}

	cdw_window_delete(&(*input_line)->win);

	free(*input_line);
	*input_line = (CDW_INPUT_LINE *) NULL;

	return;
}





/**
   \brief Top-level driver for input line widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Driver for input line widget - accepts regular keys, handles
   special keys (Enter, Escape, Tab).
   Handling of Tab key depends on value of CDW_INPUT_LINE->save_on_tab.
   If the value is true then Tab key is treated just as Enter key.

   Pressing Enter or Tab keys is treated as:
   \li confirmation of entered data, and
   \li attempt to leave input line, with
   \li intent to save entered data in \p buffer.

   \param input_line - widget to be controlled
   \param buffer - pointer to buffer in which to store result data
   \param attempts_max - how many times prompt for valid data

   \return CDW_KEY_ENTER or CDW_KEY_TAB if user pressed Enter or Tab key
   \return CDW_KEY_ESCAPE if user pressed Escape key
   \return KEY_EXIT on errors
*/
int cdw_input_line_driver(CDW_INPUT_LINE *input_line, char **buffer, int attempts_max)
{
	int key = 'a';
	for (int i = 0; i < attempts_max; i++) {
		curs_set(1);

		/* FIXME: add validation */
		key = cdw_input_line_form_driver(input_line);

		curs_set(0);

		if (key == CDW_KEY_ENTER
		    || (key == CDW_KEY_TAB && input_line->save_on_tab)) {

			cdw_rv_t crv = cdw_input_line_get_content(input_line, buffer);
			if (crv == CDW_OK) {
				return key;
			} else {
				if (i == attempts_max - 1) {
					/* too many attempts to enter correct string;
					   break loop to avoid printing "try again"
					   error message and to go straight to final
					   error message */
					key  = CDW_KEY_ESCAPE;
					break;
				} else {
					; /* key = key */
				}
			}
		} else if (key == KEY_EXIT) {
			/* invalid form content */
			cdw_vdm ("ERROR: failed to get field buffer\n");
			key = KEY_EXIT;
		} else if (key == CDW_KEY_ESCAPE) {
			break;
		} else if (key == CDW_KEY_TAB) {
			break;
		} else {
			cdw_assert (0, "ERROR: incorrect key from one line form driver: %d\n", key);
			key = KEY_EXIT;
			break;
		}

	}
	return key;
}





/**
   \brief Print (in line's message field) message about insecure character

   \date Function's top-level comment reviewed on 2012-01-10
   \date Function's body reviewed on 2012-01-10

   If user enters insecure character into input field, the input line
   widget won't accept entered string. The widget has to inform user
   why string hasn't been accepted, and it does it by printing
   "insecure character X" message in message field.

   \param input_line - input line in which to print a message
   \param insecure - string representing insecure character
*/
void cdw_input_line_driver_message(CDW_INPUT_LINE *input_line, char *insecure)
{
	/* error message can't be longer than input line's width */
	int len_max = input_line->n_cols;
	char *message = (char *) malloc((size_t) len_max + 1);
	if (!message) {
		cdw_vdm ("ERROR: failed to malloc() message buffer\n");
		return;
	}

	/* let's have N variants of error message: the same meaning,
	   but decreasing lengths; let's also hope that at least one
	   of them will be no longer than width of message field; */
#define N_SAFETY_MESSAGES 3
	char *messages[N_SAFETY_MESSAGES] = {
		/* 2TRANS: this is error message in dialog window, "%s" is
		   string with single insecure character; keep short */
		_("ERROR: insecure char %s "),
		/* 2TRANS: this is error message in dialog window, "%s" is
		   string with single insecure character; keep short */
		_("Insecure char %s "),
		/* 2TRANS: this is error message in dialog window, "%s" is
		   string with single insecure character; keep short */
		_("Insec. char %s")};

	for (int i = 0; i < N_SAFETY_MESSAGES; i++) {
		/* +1 for ending '\0'; no format string, so will result in COMPILER WARNING */
		int n = snprintf(message, (size_t) len_max + 1, messages[i], insecure);
		if (n < 0) {
			cdw_vdm ("ERROR: snprintf() returns negative value for string #%d = \"%s\"\n", i, messages[i]);
		} else if (n >= 0 && n <= len_max) {
			/* snprintf() printed at most len_max chars
			   (excluding ending '\0'); message #i was
			   short enough, accept it */
			break;
		} else {
			cdw_vdm ("ERROR: snprintf() truncates message; limit of printable chars = %d, would print %d chars\n",
				 len_max, n);
			cdw_vdm ("ERROR: attempted to print string \"%s\"\n",
				 messages[i]);
			cdw_vdm ("ERROR: printed string            \"%s\"\n",
				 message);
		}
	}

	/* at this point we have in "message" one of three messages[]
	   string without truncations, or - as last resort - last of
	   messages[] string with truncation */

	(void) wattrset(input_line->win, COLOR_PAIR(CDW_COLORS_WARNING));
	mvwprintw(input_line->win, 1, 0, message);
	free(message);
	message = (char *) NULL;
	wrefresh(input_line->win);
	/* error message about insecure character is displayed under
	   input field only for X seconds; during this time input from
	   user is blocked */
	sleep(2);

	/* TODO: it would seem that COLOR_PAIR(input_line->colors) would work,
	   but it doesn't */
        (void) wattrset(input_line->win, COLOR_PAIR(CDW_COLORS_DIALOG));
	mvwhline(input_line->win, 1, 0, ' ', len_max);
	wrefresh(input_line->win);
	return;
}





/**
   \brief Put a string into input line

   \date Function's top-level comment reviewed on 2012-01-10
   \date Function's body reviewed on 2012-01-10

   You can pass initial string to widget's constructor, but it is sometimes
   also necessary to update the widget with a string after the widget has
   been created. The function does just that - enters new content into
   given input line.

   \param input_line - widget to update
   \param string - string to put into input line
*/
void cdw_input_line_refresh_with_string(CDW_INPUT_LINE *input_line, const char *string)
{
	cdw_assert (string, "ERROR: string passed to function is NULL\n");
	int rv = set_field_buffer(input_line->field[0], 0, string);

#ifndef NDEBUG
	cdw_assert (rv == E_OK, "ERROR: failed to set buffer of input line, error = %s\n", cdw_ncurses_error_string(rv));
	char *buffer_string = cdw_string_rtrim(field_buffer(input_line->field[0], 0));
	cdw_assert(!strcmp(buffer_string, string),
		   "ERROR: buffer string and input string are different:\nbuffer string: \"%s\"\ninput string: \"%s\"\n",
		   buffer_string, string);
#endif

	rv = form_driver(input_line->form, REQ_END_LINE);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: failed call to form_driver(), error = %s\n", cdw_ncurses_error_string(rv));
		assert (0);
	}
	wrefresh(input_line->win);

	return;
}





/**
   \brief Low-level input line driver

   \date Function's top-level comment reviewed on 2012-01-10
   \date Function's body reviewed on 2012-01-10

   Low level driver used by cdw_input_line_driver(). It deals directly
   with ncurses form that is the heart of input line.

   Function returns when user presses Enter, Tab or Escape.

   Enter key is usually considered as "save value from
   input line, and exit the widget".
   Escape key is considered to mean "don't save value from
   input line, and exit the widget".

   Default reaction to Tab key is "exit the widget" - this is because
   input line may be embedded in a form or multi-widget window. Tab is
   then necessary to switch focus, and input line must recognize the key.
   In some applications it may be useful to treat Tab key the same
   way as Enter key: "save value from input line and exit the widget".
   Set CDW_INPUT_LINE->save_on_tab to true to achieve this.


   Function returns CDW_KEY_ESCAPE if user pressed Escape key.
   Function returns CDW_KEY_TAB if user pressed Tab key, and save_on_tab
   is not set.
   Function returns CDW_KEY_TAB if user pressed Tab key, and save_on_tab
   is set (but see note below).
   Function returns CDW_KEY_ENTER if user pressed Enter key (but see
   note below).

   Note:
   Function returns CDW_KEY_EXIT if user pressed Enter or Tab, but
   value entered in form was invalid (value contained unsafe characters).

   \param input_line - widget to be controlled

   \return CDW_KEY_ESCAPE
   \return CDW_KEY_ENTER
   \return CDW_KEY_TAB
   \return KEY_EXIT
*/
int cdw_input_line_form_driver(CDW_INPUT_LINE *input_line)
{
	cdw_assert (input_line->form, "ERROR: can't operate on NULL form");
	WINDOW *window = form_win(input_line->form);
	cdw_assert (window, "ERROR: can't get window of a form with form_win()\n");

	bool loop = true;
	int key = 0;
	do {
		int r = E_OK;
		key = wgetch(window);
		switch (key) {
			case KEY_LEFT:
				r = form_driver(input_line->form, REQ_PREV_CHAR);
				break;
			case KEY_RIGHT:
				r = form_driver(input_line->form, REQ_NEXT_CHAR);
				break;
			case KEY_BACKSPACE:
				r = form_driver(input_line->form, REQ_DEL_PREV);
				break;
			case KEY_DC:
				r = form_driver(input_line->form, REQ_DEL_CHAR);
				break;
		        case KEY_HOME:
				r = form_driver(input_line->form, REQ_BEG_LINE);
				break;
		        case KEY_END:
				r = form_driver(input_line->form, REQ_END_LINE);
				break;
			case CDW_KEY_ESCAPE:
		        case CDW_KEY_ENTER:
		        case CDW_KEY_TAB:
				loop = false;
				break;
			default:
				/* push user input to form */
				r = form_driver(input_line->form, key);
				if (r != E_OK) {
					cdw_vdm ("WARNING: form_driver(..., key) returns %s for key %s\n",
						 cdw_ncurses_error_string(r), cdw_ncurses_key_label(key));
				}
				wrefresh(window);
				break;
		} /* switch () */
	} while (loop);

	if (key == CDW_KEY_ESCAPE) {
		return key;
	}
	if (key == CDW_KEY_TAB
	    && !(input_line->save_on_tab)) {
		return key;
	}

	/* key treated as "exit form and save value" */

	/* flush form buffer */
	int r = form_driver(input_line->form, REQ_VALIDATION);
	if (r != E_OK) {
		cdw_vdm ("WARNING: form_driver(..., REQ_VALIDATION) returns %s for key %s\n",
			 cdw_ncurses_error_string(r), cdw_ncurses_key_label(key));
		return KEY_EXIT;
	} else {
		/* return Enter or Tab, letting caller decide
		   how to handle Tab+save_on_tab */
		return key;
	}
}





/**
   \brief Try to get string from input line, with security check

   \date Function's top-level comment reviewed on 2012-01-10
   \date Function's body reviewed on 2012-01-10

   Try to get string entered into \p input_line, check if content of
   the string is safe, and save it into \p output_buffer.

   Print error message in \p input_line if string in the widget is
   unsafe, and return CDW_ERROR.
   Return CDW_OK if string is safe.

   \param input_line - line to get a string from
   \param output_buffer - pointer to buffer, in which to store the string

   \return CDW_OK on success
   \return CDW_ERROR on problems
*/
cdw_rv_t cdw_input_line_get_content(CDW_INPUT_LINE *input_line, char **output_buffer)
{
	char *local_buffer = (char *) NULL;
	/* all that can go wrong at this point is malloc()
	   failure - return value will be CDW_ERROR */
	cdw_rv_t crv = cdw_ncurses_get_field_buffer(input_line->field[0], &local_buffer, (size_t) input_line->chars_max);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get field buffer with cdw_ncurses_get_field_buffer()\n");
		return CDW_ERROR;
	}

	/* TODO:
	   placing security check here makes input line
	   less versatile, as I can imagine places
	   where security checking may be unwanted, or
	   may need different implementation; I have to
	   update implementation of input line so that
	   it is more configurable and universal; */

	char insecure[2];
	cdw_rv_t sec = cdw_string_security_parser(local_buffer, insecure);
	if (sec == CDW_OK) {
		cdw_vdm ("INFO: \"%s\" is secure\n", local_buffer);
		crv = cdw_string_set(output_buffer, local_buffer);
		cdw_assert (crv == CDW_OK, "ERROR: failed to save final string \"%s\"\n", local_buffer);
		cdw_string_delete(&local_buffer);
		/* key = key */
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: insecure char in \"%s\"\n", local_buffer);
		cdw_input_line_driver_message(input_line, insecure);
		cdw_string_delete(&local_buffer);
		return CDW_ERROR;
	}
}
