/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "main.h"
#include "cdw_processwin.h"
#include "gettext.h"
#include "cdw_thread.h" /* PIPE_BUFFER_SIZE */
#include "cdw_regex_dispatch.h"
#include "cdw_xorriso_regex.h"
#include "cdw_disc.h"
#include "cdw_task.h"
#include "cdw_debug.h"
#include "cdw_config.h"
#include "cdw_drive.h"
#include "cdw_utils.h"
#include "cdw_logging.h"

extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];

/* time captured at the beginning of process */
extern time_t time0;

extern cdw_config_t global_config;
extern cdw_task_t *thread_task;


/* stdout handlers */
#if 0
static int cdw_xorriso_handle_atip_speeds(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_atip_supported_modes(regex_t *regex, regmatch_t *matches);
#endif
static int cdw_xorriso_handle_media_product(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_media_summary(regex_t *regex, regmatch_t *matches);


/* stderr handlers */
static int cdw_xorriso_handle_writing(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_writing_cdrecord(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_writing_completed(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_media_current(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_media_status(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_blanking(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_blanking_done(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_simulation_not_supported(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_exceeds_free_space(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_cannot_read_iso_image_tree(regex_t *regex, regmatch_t *matches);
static int cdw_xorriso_handle_not_a_known_option(regex_t *regex, regmatch_t *matches);

static int previous_percent = 0;

/* did xorriso display "Blanking done" because it was really erasing a disc?
   or only because it was removing content of overwritten ISO image file? */
static bool real_blanking = false;

static cdw_regex_t stdout_regex[] = {
#if 0
	{ "atip speeds", 1005,
	  /* "  1T speed low:  4 1T speed high: 10" */
	  "1T speed low:( +)([0-9]+)( +)1T speed high:( +)([0-9]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_atip_speeds },

	{ "atip supported modes", 1006,
	  /* "Supported modes: SAO TAO";
	     currently there is no way to pass sao/tao information to xorriso */
	  "Supported modes:(.+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_atip_supported_modes },
#endif

	{ "media product", 1007,
	  /* "Media product: 97m10s00f/79m59s74f , TDK / Ritek"
	     f (frame) = 2048 bytes sector
	     s (sector) = 57 frames
	     m (minute) = 60 seconds

	     (f + 75*s + 60*75*m) * 2048 / (1024 * 1024) = disc size [MB] */
	  "Media product:.+/([0-9]+)m([0-9]+)s([0-9]+)f",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_media_product },

	{ "media summary", 1008,
	  /* sometimes "media product" doesn't contain necessary
	     information, so check "media summary" too;
	     "Media summary: 0 sessions, 0 data blocks, 0 data,  703m free"
	     "Media summary: 1 session, 165648 data blocks,  324m data,     0 free"
	     "Media summary: 1 session, 316880 data blocks,  619m data, 3860m free" */

	  "Media summary:([a-zA-Z0-9, ]+),([ ]+)([0-9]+)(|m|g) data,([ ]+)([0-9]+)(|m|g) free",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_media_summary },


	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};




static cdw_regex_t stderr_regex[] = {
	{ "writing", 1001,
	  /* "xorriso : UPDATE : Writing:      72603s   50.7%   fifo   0%  buf  50%   10.4xD" */
	  //"xorriso : UPDATE : Writing:[ ]+([0-9]+)s[ ]+([0-9]+)\\.([0-9]+)%[ ]+fifo[ ]+([0-9]+)%[ ]+buf[ ]+([0-9]+)%[ ]+([0-9]+)\\.([0-9]+)xD",
	  "xorriso : UPDATE : Writing:[ ]+([0-9]+)s[ ]+([0-9]+)\\.([0-9]+)%[ ]+fifo[ ]+([0-9]+)%", //[ ]+buf[ ]+([0-9]+)%[ ]+([0-9]+)\\.([0-9]+)xD",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_writing },

	{ "writing", 1001,
	  /* "xorriso : UPDATE :   49 of   64 MB written (fifo 97%) [buf 100%]  10.0x. */
	  /* update text printed when xorriso burns data while emulating cdrecord */
	  "xorriso : UPDATE : ([ ]*)([0-9]+) of ([ ]*)([0-9]+) MB written \\(fifo([ ]*)([0-9]+)%\\)([ ]+)\\[buf([ ]*)([0-9]+)%\\]([ ]+)([0-9]+)\\.([0-9]+)x",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_writing_cdrecord },

	{ "writing completed", 1002,
	  /* "Writing to '/tmp/33.iso' completed sucessfully." */
	  "Writing to .+ completed suc",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_writing_completed },

	{ "media current", 1003,
	  /* "Media current: CD-RW" */
	  "Media current: ([CDVRW+-]+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_media_current },

	{ "media status", 1004,
	  /* "Media status : is written , is appendable" */
	  "Media status : (.+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_media_status },

	{ "blanking", 1005,
	  /* "xorriso : UPDATE : Blanking  ( 26.5% done in 175 seconds )" */
	  "xorriso : UPDATE : Blanking  \\(( *)([0-9]+)\\.([0-9]+)% done in ([0-9]+) seconds \\)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_blanking },

	{ "blanking done", 1006,
	  /* "Blanking done"
	     Displayed at the end of blanking process; xorriso may not
	     display "100%" at the end of the process, so capturing the
	     string is necessary to print 100% in progress window; */
	  "Blanking done",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_blanking_done },

	{ "simulation not supported", 1007,
	  /* "SAO: simulation of write job not supported by drive and media"
	     printed when I attempted to simulate burning to DVD+R
	     ("-dummy") option */
	  "simulation of .+ job not supported by drive and media",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_simulation_not_supported },

	{ "simulation not supported", 1008,
	  /* "xorriso : FAILURE : Image size 1262877s exceeds free space on media 1217680s"
	     not enough free space on disc to write selected files
	     (this should work for burning image as well) */
	  "Image size .+ exceeds free space on media",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_exceeds_free_space },

	{ "cannot read ISO image tree", 1009,
	  /*
	    xorriso : NOTE : Loading ISO image tree from LBA 0
	    libisofs: SORRY : Damaged RR/SUSP information.
	    libisofs: FAILURE : Wrong or damaged RR entry
	    xorriso : FAILURE : Cannot read ISO image tree
	    xorriso : HINT : You might get a partial or altered ISO image tree by option -error_behavior 'image_loading' 'best_effort' if -abort_on is set to be tolerant enough.
	    xorriso : aborting : -abort_on 'FAILURE' encountered 'FAILURE'"

	    Displayed when xorriso can't read a disc information */
	  "Cannot read ISO image tree",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_cannot_read_iso_image_tree },

	{ "FAILURE : Not a known option", 1010,
	  /* Unknown option passed to xorriso */
	  "FAILURE : Not a known option: (.+)",
	  (regex_t *) NULL, (regmatch_t *) NULL, REG_EXTENDED,
	  cdw_xorriso_handle_not_a_known_option },


	{ "table guard", -1,
	  /* guard: debug_id = -1 */
	  "",
	  (regex_t *) NULL, (regmatch_t *) NULL, 0,
	  (cdw_regex_handler_t) NULL }
};





/**
   \brief Do some initialization and call cdw_regex_prepare_regexes_table() for stdout regexes table
*/
void cdw_xorriso_stdout_regex_prepare(void)
{
	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stdout_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stdout\n");
	}
	return;
}





/**
   \brief Do some initialization and call cdw_regex_execute_regexes_table()
   for stdout regexes table and stdout pipe buffer.
*/
void cdw_xorriso_stdout_regex_execute(void)
{
	stdout_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stdout_regex, stdout_pipe_buffer);

	return;
}





/**
   \brief Call cdw_regex_clean_up_regexes_table() for stdout regexes table
*/
void cdw_xorriso_stdout_regex_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stdout_regex);
	return;
}





/**
   \brief Call cdw_regex_prepare_regexes_table() for stderr regexes table
*/
void cdw_xorriso_stderr_regex_prepare(void)
{
	previous_percent = 0;
	real_blanking = false;

	cdw_rv_t crv = cdw_regex_prepare_regexes_table(stderr_regex);
	if (crv != CDW_OK) {
		cdw_vdm("ERROR: failed to prepare regexes table for stderr\n");
	}
	return;
}





/**
   \brief Do some initialization and call cdw_regex_execute_regexes_table()
   for stderr regexes table and stderr pipe buffer.
*/
void cdw_xorriso_stderr_regex_execute(void)
{
	stderr_pipe_buffer[PIPE_BUFFER_SIZE] = '\0';
	cdw_regex_execute_regexes_table(stderr_regex, stderr_pipe_buffer);

	return;
}





/**
   \brief Call cdw_regex_clean_up_regexes_table() for stderr regexes table
*/
void cdw_xorriso_stderr_regex_destroy(void)
{
	cdw_regex_clean_up_regexes_table(stderr_regex);
	return;
}





int cdw_xorriso_handle_writing(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information about writing image:
	    "xorriso : UPDATE : Writing:      72603s   50.7%   fifo   0%  buf  50%   10.4xD"
                                               1            2          3                    4                   5            6          7
	    "xorriso : UPDATE : Writing:[ ]+([0-9]+)s[ ]+([0-9]+)\\.([0-9]+)%[ ]+fifo[ ]+([0-9]+)%[ ]+buf[ ]+([0-9]+)%[ ]+([0-9]+)\\.([0-9]+)xD" */
	cdw_regex_assert_subex_number(regex->re_nsub, 4);

	int segments = 0;
	int progress_decimal = 0, progress_fract = 0;

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE + 1];
		int len = cdw_regex_get_submatch(matches, i, stderr_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 1) {
			segments = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, segments = \"%s\" -> %d\n", i, submatch, segments);
		} else if (i == 2) {
			progress_decimal = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, progress decimal = \"%s\" -> %d\n", i, submatch, progress_decimal);
		} else if (i == 3) {
			progress_fract = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, progress fractional = \"%s\" -> %d\n", i, submatch, progress_fract);
		} else {
			;
		}

	}
#if 0
	char text_info2_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window */
	snprintf(text_info2_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("segment: %d"), segments);
	cdw_processwin_display_sub_info(text_info2_string);

	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window,
	   %d.%d is a number with fractional part, describing amount
	   of data (in percents) processed */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("%d.%d %%"), progress_decimal, progress_fract);
#endif
	cdw_processwin_display_progress_conditional(progress_decimal, 100, (char *) NULL);
	cdw_processwin_wrefresh();

	return 0;

}





int cdw_xorriso_handle_writing_completed(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	char text_info2_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window;
	   the task is fully completed */
	snprintf(text_info2_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("Completed"));
	cdw_processwin_display_sub_info(text_info2_string);

	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window,
	   writing is now completed in 100% */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("100.0%%"));
	cdw_processwin_display_progress_conditional(100, 100, current_value_string);
	cdw_processwin_wrefresh();


	return 0;
}





int cdw_xorriso_handle_media_current(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information about current disc type:
	                        1
	   "Media current: ([CDVRW+-]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	unsigned int sub = 1;
	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, sub, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	if (!strcmp(submatch, "CD-RW")) {
		thread_task->disc->type = CDW_CD_RW;
	} else if (!strcmp(submatch, "CD-R")) {
		thread_task->disc->type = CDW_CD_R;
	} else if (!strcmp(submatch, "DVD-R")) {
		thread_task->disc->type = CDW_DVD_R;
	} else if (!strcmp(submatch, "DVD+R")) {
		thread_task->disc->type = CDW_DVD_RP;
	} else {
		/* thread_task->disc->type = CDW_DISC_TYPE_UNKNOWN */
		cdw_vdm ("WARNING: submatch = \"%s\" => disc type = unsupported\n", submatch);
	}
	cdw_vdm ("INFO: disc type string \"%s\" -> disc type %d\n", submatch, thread_task->disc->type);

	return 0;
}





int cdw_xorriso_handle_media_status(regex_t *regex, regmatch_t *matches)
{
	if (thread_task->id == CDW_TASK_CREATE_IMAGE) {
		/* xorriso output for ISO files (no disc):
		   "Drive current: -outdev '/tmp/image.iso'
		   Media current: stdio file, overwriteable
		   Media status : is blank"

		   Obviously we shouldn't proceed with handling media
		   status when there is no media involved (i.e. when
		   creating standalone image file) */
		return 0;
	}

	/* regex for capturing information about current disc status:
                            1
	  "Media status : (.+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	unsigned int sub = 1;
	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, sub, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	if (strstr(submatch, "is written")) {
		cdw_vdm ("INFO: disc is written\n");
		thread_task->disc->xorriso_info.is_written = CDW_TRUE;
	}
	if (strstr(submatch, "is blank")) {
		cdw_vdm ("INFO: disc is blank\n");
		thread_task->disc->xorriso_info.is_blank = CDW_TRUE;
	}

	if (strstr(submatch, "is appendable")) {
		cdw_vdm ("INFO: disc is appendable\n");
		thread_task->disc->xorriso_info.is_appendable = CDW_TRUE;
	}
	if (strstr(submatch, "is closed")) {
		cdw_vdm ("INFO: disc is closed\n");
		thread_task->disc->xorriso_info.is_closed = CDW_TRUE;
	}

	return 0;

}




#if 0

/* unfortunately xorriso does not provide option for listing supported
   write speeds, so we have use cdrecord emulation */
int cdw_xorriso_handle_atip_speeds(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information about writing speeds:
	                   1    2      3                 4    5
	   "1T speed low:( +)([0-9]+)( +)1T speed high:( +)([0-9]+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 5);

	unsigned int sub;
	char submatch[PIPE_BUFFER_SIZE + 1];
	sub = 2;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		thread_task->disc->write_speeds.speeds[0] = atoi(submatch);
		cdw_vdm ("INFO: lower speed = %d\n", thread_task->disc->write_speeds.speeds[0]);
	}

	sub = 5;
	len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		thread_task->disc->write_speeds.speeds[1] = atoi(submatch);
		cdw_vdm ("INFO: upper speed = %d\n", thread_task->disc->write_speeds.speeds[1]);
	}

	return 0;
}




int cdw_xorriso_handle_atip_supported_modes(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information about supported modes:
                              1
	   "Supported modes:(.+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	unsigned int sub = 1;
	char submatch[PIPE_BUFFER_SIZE + 1];
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	}

	cdw_vdm ("INFO: supported modes submatch: \"%s\"\n", submatch);
	const char *delim = " ";
	char *mode = strtok(submatch, delim);
	while (mode != (char *) NULL) {
		if (!strcmp(mode, "TAO")) {
			cdw_vdm ("INFO: drive supports TAO\n");
			thread_task->disc->accepted_disc_modes |= CDW_DISC_MODE_TAO;
		} else if (!strcmp(mode, "DAO")) {
			cdw_vdm ("INFO: drive supports DAO\n");
			thread_task->disc->accepted_disc_modes |= CDW_DISC_MODE_DAO;
		} else if (!strcmp(mode, "SAO")) {
			cdw_vdm ("INFO: drive supports SAO\n");
			thread_task->disc->accepted_disc_modes |= CDW_DISC_MODE_SAO;
		} else {
			;
		}
		mode = strtok((char *) NULL, delim);
	}

	return 0;
}
#endif





int cdw_xorriso_handle_media_product(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing media product information:
                                1        2        3
	   "Media product:.+/([0-9]+)m([0-9]+)s([0-9]+)f"  */
	cdw_regex_assert_subex_number(regex->re_nsub, 3);

	int minutes = 0, seconds = 0, frames = 0;

	unsigned int sub;
	char submatch[PIPE_BUFFER_SIZE + 1];
	sub = 1;
	int len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		minutes = atoi(submatch);
		cdw_vdm ("INFO: %d minutes\n", minutes);
	}
	sub = 2;
	len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		seconds = atoi(submatch);
		cdw_vdm ("INFO: %d seconds\n", seconds);
	}
	sub = 3;
	len = cdw_regex_get_submatch(matches, sub, stdout_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", sub, len);
		return -1;
	} else {
		frames = atoi(submatch);
		cdw_vdm ("INFO: %d frames\n", frames);
	}

	thread_task->disc->xorriso_info.sectors_total = frames + 75 * seconds + 60 * 75 * minutes;

	return 0;

}





int cdw_xorriso_handle_writing_cdrecord(regex_t *regex, regmatch_t *matches)
{
	/* regex for capturing information:
                                 1      2          3      4                         5      6          7           8      9          10     11         12
	   "xorriso : UPDATE : ([ ]*)([0-9]+) of ([ ]*)([0-9]+) MB written \\(fifo([ ]*)([0-9]+)%\\)([ ]+)\\[buf([ ]*)([0-9]+)%\\]([ ]+)([0-9]+)\\.([0-9]+)x" */
	cdw_regex_assert_subex_number(regex->re_nsub, 12);

	int fifo = 0;
	int total_size = 0, done_size = 0;
	int speed_decimal = 0, speed_fract = 0;

	for (unsigned int i = 0; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE + 1];
		int len = cdw_regex_get_submatch(matches, i, stderr_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}

		if (i == 2) {
			done_size = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, done size = \"%s\" -> %d\n", i, submatch, done_size);
		} else if (i == 4) {
			total_size = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, total size = \"%s\" -> %d\n", i, submatch, total_size);
		} else if (i == 6) {
			fifo = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, fifo = \"%s\" -> %d\n", i, submatch, fifo);
		} else if (i == 11) {
			speed_decimal = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed_decimal = \"%s\" -> %d\n", i, submatch, speed_decimal);
		} else if (i == 12) {
			speed_fract = atoi(submatch);
			cdw_vdm ("INFO: submatch %d, speed_fract = \"%s\" -> %d\n", i, submatch, speed_fract);
		} else {
			;
		}
	}

	cdw_processwin_display_fifo_and_speed(fifo, speed_decimal, speed_fract);

	int percent = (done_size * 100) / total_size;
	if ((percent >= 0) && (percent <= 100)) {
		if (previous_percent != percent) {
			cdw_sdm ("INFO: previous_percent = %d, percent = %d\n", previous_percent, percent);

			double time_from_start = difftime(time(NULL), time0);
			int todo_size = total_size - done_size;
			int eta = (int) (((1.0 * todo_size) / done_size) * time_from_start);

			char eta_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
			cdw_utils_eta_calculations(eta_string, eta, PROCESSWIN_MAX_RTEXT_LEN);
			previous_percent = percent;
			cdw_processwin_display_eta(eta_string);

			cdw_sdm ("INFO: eta = %d, eta_string = %s\n", eta, eta_string);
		}
	}

	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];
	/* 2TRANS: this is label displayed in progress window,
	   first %d is amount of data already written to CD,
	   second %d is total amount of data to write */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("%d/%d MB"), done_size, total_size);
	cdw_processwin_display_progress_conditional(done_size, total_size, current_value_string);

	cdw_processwin_wrefresh();

	// cdw_xorriso_burning_performed = true;

	return 0;
}




int cdw_xorriso_handle_blanking(regex_t *regex, regmatch_t *matches)
{
	/* regexp for catching blanking
	                                     1     2          3                 4
	   xorriso : UPDATE : Blanking  \\(( *)([0-9]+)\\.([0-9]+)% done in ([0-9]+) seconds \\) */
	cdw_regex_assert_subex_number(regex->re_nsub, 4);

	int perc_decimal = 0, perc_fract = 0;
	//int seconds = 0;
	real_blanking = true;

	for (unsigned int i = 2; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE];
		int len = cdw_regex_get_submatch(matches, i, stderr_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}
		if (i == 2) { /* percentage - decimal part */
			if (!strcmp(submatch, "??")) {
				perc_decimal = 0;
			} else {
				perc_decimal = atoi((char *) &submatch);
			}
			cdw_vdm ("INFO: submatch %d, decimal part = \"%s\" -> %d\n", i, submatch, perc_decimal);
		} else if (i == 3) { /* percentage - fractional part */
			if (!strcmp(submatch, "??")) {
				perc_fract = 0;
			} else {
				perc_fract = atoi((char *) &submatch);
			}
			cdw_vdm ("INFO: submatch %d, fractional part = \"%s\" -> %d\n", i, submatch, perc_fract);
		} else if (i == 4) {
			// seconds = atoi((char *) &submatch);
		} else {
			;
		}
	}

	/* float percent = perc_a + 0.1 * perc_b; */
	/* NOTE: consider implementing
	   conditional_processwin_display_progress_float(float done, long total, char *string) */

	/* display data */
	cdw_processwin_display_progress_conditional(perc_decimal, 100, (char *) NULL);

	return 0;
}





int cdw_xorriso_handle_blanking_done(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* regexp for catching "blanking done" message (no subexpressions)
	   "Blanking done" */

	/* "Blanking done" may be also displayed when creating an image;
	   "-blank fast" option is passed to "erase" content of overwritten
	   ISO9660 image file */

	if (real_blanking) {
		/* display "100%" */
		cdw_processwin_display_progress(100, 100, (char *) NULL);
	}

	return 0;
}





int cdw_xorriso_handle_media_summary(regex_t *regex, regmatch_t *matches)
{
	/* regexp for catching media summary
	                          1           2     3      4           5      6      7
	   "Media summary:([a-zA-Z0-9, ]+),([ ]+)([0-9]+)(|m|g) data,([ ]+)([0-9]+)(|m|g) free", */
	cdw_regex_assert_subex_number(regex->re_nsub, 7);

	long int data_size = 0;
	long int free_area_size = 0;

	int data_unit = 0; /* 0 = none, 1 = bytes, 2 = mb, 3 = gb */
	int free_unit = 0; /* 0 = none, 1 = bytes, 2 = mb, 3 = gb */

	for (unsigned int i = 3; i <= regex->re_nsub; i++) {
		char submatch[PIPE_BUFFER_SIZE];
		int len = cdw_regex_get_submatch(matches, i, stdout_pipe_buffer, submatch);
		if (len == -1) {
			cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
			continue;
		}
		if (i == 3) { /* data size */
			data_size = strtol(submatch, (char **) NULL, 10);
			cdw_vdm ("INFO: ooo data size = %ld\n", data_size);

		} else if (i == 4) { /* data size unit */
			if (!strcmp(submatch, "m")) {
				cdw_vdm ("INFO: ooo data size unit = m\n");
				data_unit = 2;
			} else if (!strcmp(submatch, "g")) {
				cdw_vdm ("INFO: ooo data size unit = g\n");
				data_unit = 3;
			} else {
				if (!strlen(submatch)) {
					data_unit = 0;
				} else {
					cdw_vdm ("ERROR: untested data size unit \"%s\"\n", submatch);
					return -1;
				}
			}
		} else if (i == 6) {
			free_area_size = strtol(submatch, (char **) NULL, 10);
			cdw_vdm ("INFO: ooo free area size = %ld\n", free_area_size);
		} else if (i == 7) { /* free area size unit */
			if (!strcmp(submatch, "m")) {
				cdw_vdm ("INFO: ooo data size unit = m\n");
				free_unit = 2;
			} else if (!strcmp(submatch, "g")) {
				cdw_vdm ("INFO: ooo data size unit = g\n");
				free_unit = 3;
			} else {
				if (!strlen(submatch)) {
					cdw_assert (free_area_size == 0, "ERROR: no free area size unit, but free area size != 0\n");
					free_unit = 0;
				} else {
					cdw_vdm ("ERROR: untested data size unit \"%s\"\n", submatch);
					return -1;
				}
			}
		} else {
			;
		}
	}

	long int data_coeff = 0;
	if (data_unit == 0) {
		data_coeff = 1;
	} else if (data_unit == 2) {
		data_coeff = 1024 * 1024;
	} else if (data_unit == 3) {
		data_coeff = 1024 * 1024 * 1024;
	} else {
		cdw_vdm ("ERROR: untested data unit %d\n", data_unit);
		return -1;
	}
	long int free_area_coeff = 0;
	if (free_unit == 0) {
		free_area_coeff = 1;
	} else if (free_unit == 2) {
		free_area_coeff = 1024 * 1024;
	} else if (free_unit == 3) {
		free_area_coeff = 1024 * 1024 * 1024;
	} else {
		cdw_vdm ("ERROR: untested free area unit %d\n", free_unit);
		return -1;
	}

	const int sector_size_b = 2048;

	thread_task->disc->xorriso_info.sectors_used =
		(data_size * data_coeff) / sector_size_b;

	thread_task->disc->xorriso_info.sectors_total =
		thread_task->disc->xorriso_info.sectors_used
		+ (free_area_size * free_area_coeff) / sector_size_b;

	cdw_vdm ("INFO: ooo used %ldd, free %ldd\n", thread_task->disc->xorriso_info.sectors_used, (free_area_size * free_area_coeff) / sector_size_b);

	return 0;
}





int cdw_xorriso_handle_simulation_not_supported(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* "-dummy" option has not been accepted by drive;
	   this happened when I attempted to burn data,
	   but perhaps it may also be useful for erasing */

	thread_task->tool_status.xorriso |= CDW_TOOL_STATUS_XORRISO_SIMULATION_NOT_SUPPORTED;

	return 0;
}





int cdw_xorriso_handle_exceeds_free_space(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	/* not enough space available on target disc */

	thread_task->tool_status.xorriso |= CDW_TOOL_STATUS_XORRISO_NOT_ENOUGH_SPACE;

	return 0;
}





int cdw_xorriso_handle_cannot_read_iso_image_tree(__attribute__((unused)) regex_t *regex, __attribute__((unused)) regmatch_t *matches)
{
	thread_task->tool_status.xorriso |= CDW_TOOL_STATUS_XORRISO_CANNOT_READ_ISO_IMAGE_TREE;
	return 0;
}





int cdw_xorriso_handle_not_a_known_option(regex_t *regex, regmatch_t *matches)
{
	/* regexp for catching unknown option
	                                    1
	   "FAILURE : Not a known option: (.+)" */
	cdw_regex_assert_subex_number(regex->re_nsub, 1);

	char submatch[PIPE_BUFFER_SIZE];
	unsigned int i = 1;
	int len = cdw_regex_get_submatch(matches, i, stderr_pipe_buffer, submatch);
	if (len == -1) {
		cdw_vdm ("ERROR: len of subexpr %d is negative: %d\n", i, len);
		return -1;
	}
	char *option = cdw_string_trim(submatch);
	if (option) {
		/* 2TRANS: this is a message printed to log file;
		   '%s' is a string with unknown option passed to xorriso */
		cdw_logging_write(_("ERROR: unknown xorriso option \"%s\"\n"), option);
		free(option);
		option = (char *) NULL;
	} else {
		/* 2TRANS: this is a message printed to log file */
		cdw_logging_write(_("ERROR: unknown xorriso option\n"));
	}

	thread_task->tool_status.xorriso |= CDW_TOOL_STATUS_XORRISO_NOT_A_KNOWN_OPTION;
	return 0;
}
