//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/ParticleLayoutItem.cpp
//! @brief     Implements class ParticleLayoutItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/ParticleLayoutItem.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Sample/CompoundItem.h"
#include "GUI/Model/Sample/InterferenceItems.h"
#include "GUI/Model/Sample/Lattice2DItems.h"
#include "GUI/Model/Sample/MesocrystalItem.h"
#include "GUI/Model/Sample/ParticleItem.h"

namespace {
namespace Tag {

const QString OwnDensity("OwnDensity");
const QString InterferenceFunction("InterferenceFunction");
const QString Particle("Particle");
const QString ExpandLayoutGroupbox("ExpandLayoutGroupbox");
const QString ExpandInterferenceGroupbox("ExpandInterferenceGroupbox");

} // namespace Tag
} // namespace

ParticleLayoutItem::ParticleLayoutItem(const MaterialModel* materials)
    : m_materialModel(materials)
{
    m_ownDensity.init("Total particle density",
                      "Number of particles per area (particle surface density).\n "
                      "Should be defined for disordered and 1d-ordered particle collections.",
                      0.0005, Unit::nanometerMinus2, 6 /* decimals */, 0.0001 /* step */,
                      RealLimits::nonnegative(), "density");

    m_interference.init("Interference function", "");
}

double ParticleLayoutItem::totalDensityValue() const
{
    if (!totalDensityIsDefinedByInterference())
        return m_ownDensity.value();

    ASSERT(m_interference.currentItem());

    if (const auto* interLatticeItem =
            dynamic_cast<const Interference2DAbstractLatticeItem*>(m_interference.currentItem())) {
        Lattice2DItem* latticeItem = interLatticeItem->latticeTypeItem();
        try {
            const double area = latticeItem->unitCellArea();
            return area == 0.0 ? 0.0 : 1.0 / area;
        } catch (const std::exception&) {
            // nothing to do here; new exception will be caught during job execution
            return 0.0;
        }
    }

    if (const auto* hd =
            dynamic_cast<const InterferenceHardDiskItem*>(m_interference.currentItem()))
        return hd->density();

    ASSERT(false);
}

QVector<ItemWithParticles*> ParticleLayoutItem::itemsWithParticles() const
{
    return m_particles.toQVector();
}

SelectionProperty<ItemWithParticlesCatalog>&
ParticleLayoutItem::addItemWithParticleSelection(ItemWithParticles* particle)
{
    m_particles.push_back(particle);
    return m_particles.back();
}

void ParticleLayoutItem::removeItemWithParticle(ItemWithParticles* particle)
{
    m_particles.delete_element(particle);
}

QVector<ItemWithParticles*> ParticleLayoutItem::containedItemsWithParticles() const
{
    QVector<ItemWithParticles*> result;
    for (const auto& sel : m_particles)
        result << sel.currentItem() << sel.currentItem()->containedItemsWithParticles();
    return result;
}

const SelectionProperty<InterferenceItemCatalog>& ParticleLayoutItem::interferenceSelection() const
{
    return m_interference;
}

bool ParticleLayoutItem::totalDensityIsDefinedByInterference() const
{
    return dynamic_cast<const Interference2DAbstractLatticeItem*>(m_interference.currentItem())
           || dynamic_cast<const InterferenceHardDiskItem*>(m_interference.currentItem());
}

void ParticleLayoutItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // own density
    w->writeStartElement(Tag::OwnDensity);
    m_ownDensity.writeTo(w);
    w->writeEndElement();

    // interference function
    w->writeStartElement(Tag::InterferenceFunction);
    m_interference.writeTo(w);
    w->writeEndElement();

    // particles
    for (const auto& sel : m_particles) {
        w->writeStartElement(Tag::Particle);
        sel.writeTo(w);
        w->writeEndElement();
    }

    // layout groupbox: is expanded?
    w->writeStartElement(Tag::ExpandLayoutGroupbox);
    XML::writeAttribute(w, XML::Attrib::value, m_expandParticleLayout);
    w->writeEndElement();

    // interference groupbox: is expanded?
    w->writeStartElement(Tag::ExpandInterferenceGroupbox);
    XML::writeAttribute(w, XML::Attrib::value, m_expandInterference);
    w->writeEndElement();
}

void ParticleLayoutItem::readFrom(QXmlStreamReader* r)
{
    m_particles.clear();

    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // own density
        if (tag == Tag::OwnDensity) {
            m_ownDensity.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // interference function
        } else if (tag == Tag::InterferenceFunction) {
            m_interference.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // particle
        } else if (tag == Tag::Particle) {
            addItemWithParticleSelection(nullptr).readFrom(r, m_materialModel);
            XML::gotoEndElementOfTag(r, tag);

            // layout groupbox: is expanded?
        } else if (tag == Tag::ExpandLayoutGroupbox) {
            XML::readAttribute(r, XML::Attrib::value, &m_expandParticleLayout);
            XML::gotoEndElementOfTag(r, tag);

            // interference groupbox: is expanded?
        } else if (tag == Tag::ExpandInterferenceGroupbox) {
            XML::readAttribute(r, XML::Attrib::value, &m_expandInterference);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
