// ---------------------------------------------------------------------------
// - Cell.cpp                                                                -
// - afnix:sps module - cell class implementation                            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2012 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cell.hpp"
#include "Spssid.hxx"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "InputStream.hpp"
#include "OutputStream.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // this procedure returns a new cell object for deserialization
  static Serial* mksob (void) {
    return new Cell;
  }
  // register this cell serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_CELL_ID, mksob);

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create an empty cell
  
  Cell::Cell (void) {
    d_quark = 0;
    p_cobj  = nilp;
    d_const = false;
  }

  // create a cell with a literal
  
  Cell::Cell (Literal* cobj) {
    d_quark = 0;
    Object::iref (p_cobj = cobj);
    d_const = false;
  }

  // create a cell by name with a literal
  
  Cell::Cell (const String& name, Literal* cobj) {
    d_quark = name.toquark ();
    Object::iref (p_cobj = cobj);
    d_const = false;
  }

  // copy construct this cell

  Cell::Cell (const Cell& that) {
    that.rdlock ();
    try {
      d_quark = that.d_quark;
      d_const = that.d_const;
      Object::iref (p_cobj = that.p_cobj);
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this cell

  Cell::~Cell (void) {
    Object::dref (p_cobj);
  }

  // return the class name

  String Cell::repr (void) const {
    return "Cell";
  }

  // return a clone of this object

  Object* Cell::clone (void) const {
    return new Cell (*this);
  }

  // return the cell serial code

  t_byte Cell::serialid (void) const {
    return SERIAL_CELL_ID;
  }

  // serialize this cell

  void Cell::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      const String& name = String::qmap (d_quark);
      name.wrstream (os);
      if (p_cobj == nilp) {
	Serial::wrnilid (os);
      } else {
	p_cobj->serialize (os);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this cell

  void Cell::rdstream (InputStream& is) {
    wrlock ();
    try {
      String  sval;
      sval.rdstream (is);
      Object* cobj = Serial::deserialize (is);
      d_quark = sval.toquark ();
      p_cobj  = dynamic_cast <Literal*> (cobj);
      if ((cobj != nilp) && (p_cobj == nilp)) {
	throw  Exception ("type-error", "invalid cell type object", 
			  Object::repr (cobj));
      }
      Object::iref (p_cobj);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    } 
  }

  // assign a cell to this one

  Cell& Cell::operator = (const Cell& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      d_quark = that.d_quark;
      d_const = that.d_const;
      Object::iref (that.p_cobj);
      Object::dref (p_cobj);
      p_cobj = that.p_cobj;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // match a quark against the cell

  bool Cell::operator ==  (const long quark) const {
    rdlock ();
    try {
      bool result = (d_quark == quark);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // match a literal against the cell

  bool Cell::operator ==  (const Literal& lobj) const {
    rdlock ();
    try {
      bool result = (p_cobj == nilp) ? false : (*p_cobj == lobj);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the cell name

  String Cell::getname (void) const {
    rdlock ();
    try {
      String result = String::qmap (d_quark);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    } 
  }

  // set the cell name

  void Cell::setname (const String& name) {
    wrlock ();
    try {
      d_quark = name.toquark ();
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    } 
  }

  // return the cell literal value

  Literal* Cell::get (void) const {
    rdlock ();
    try {
      Literal* result = p_cobj;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    } 
  }

  // set the cell value

  void Cell::set (Literal* cobj) {
    wrlock ();
    try {
      if (d_const == true) {
	throw Exception ("const-error", "cell const violation");
      } 
      Object::iref (cobj);
      Object::dref (p_cobj);
      p_cobj = cobj;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    } 
  }

  // map the cell literal to a string

  String Cell::tostring (void) const {
    rdlock ();
    try {
      String result = (p_cobj == nilp) ? "nil" : p_cobj->tostring ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GET      = zone.intern ("get");
  static const long QUARK_SET      = zone.intern ("set");
  static const long QUARK_SETNAME  = zone.intern ("set-name");
  static const long QUARK_TOSTRING = zone.intern ("to-string");

  // create a new object in a generic way

  Object* Cell::mknew (Vector* argv) {
    // get number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Cell;
    // check for 1 argument
    if (argc == 1) {
      Object*  lobj = argv->get (0);
      Literal* cobj = dynamic_cast <Literal*> (lobj);
      if ((lobj != nilp) && (cobj == nilp))
	throw Exception ("type-error", "invalid object for cell constructor",
			 Object::repr (lobj));
      return new Cell (cobj);
    }
    // check for 2 arguments
    if (argc == 2) {
      String name = argv->getstring (0);
      Object*  lobj = argv->get (1);
      Literal* cobj = dynamic_cast <Literal*> (lobj);
      if ((lobj != nilp) && (cobj == nilp))
	throw Exception ("type-error", "invalid object for cell constructor",
			 Object::repr (lobj));
      return new Cell (name, cobj);
    }    
    throw Exception ("argument-error", "too many arguments with cell");
  }

  // return true if the given quark is defined

  bool Cell::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Persist::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // set an object to this object

  Object* Cell::vdef (Runnable* robj, Nameset* nset, Object* object) {
    wrlock ();
    try {
      Literal* lobj = dynamic_cast <Literal*> (object);
      if (lobj != nilp) {
	set (lobj);
	robj->post (lobj);
	unlock ();
	return lobj;
      }
      throw Exception ("type-error", "invalid object with cell vdef",
		       Object::repr (object));
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark

  Object* Cell::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_TOSTRING) return new String (tostring ());
      if (quark == QUARK_GET) {
	rdlock ();
	Object* result = get ();
	robj->post (result);
	unlock ();
	return result;
      }
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nilp;
      }
      if (quark == QUARK_SET) {
	Object*   obj = argv->get (0);
	Literal* cobj = dynamic_cast <Literal*> (obj);
	if ((obj != nilp) && (cobj == nilp)) 
	  throw Exception ("type-error", "invalid object to set in cell",
			   Object::repr (obj));
	set (cobj);
	return nilp;
      }
    }
    // call the persist method
    return Persist::apply (robj, nset, quark, argv);
  }
}
