!{\src2tex{textfont=tt}}
!!****f* ABINIT/ortho_phi
!! NAME
!! ortho_phi
!! 
!! FUNCTION
!! Re-compute and re-orthogonalize partial waves (Phi), pseudo partial waves (tPhi)
!! and projectors (tProj) according to the integration scheme "ctrap" from Abinit
!!
!! COPYRIGHT
!! Copyright (C) 1998-2005 ABINIT group (FJ, MT)
!! This file is distributed under the terms of the
!! GNU General Public License, see ~ABINIT/Infos/copyright
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~ABINIT/Infos/contributors .
!!
!! INPUTS
!!  option= 0: nothing is recomputed
!!          1: Phi, tPhi and tProj are re-orthogonalized
!!          2: tProj are re-computed using a N. Holzwarth scheme
!!             Phi, tPhi and tProj are re-orthogonalized
!!  pawarray
!!    %shapefunc(mesh_size)= Normalized shape function
!!  pawdata
!!    %rad_(big_meshsz)= rad(:) + AA where rad(i)=AA*(exp[BB*(i-1)]-1)
!!  pshead
!!    %basis_size= Number of elements for the paw nl basis
!!    %lmax= Maximum value of l
!!    %log_step= Logarithmic step corresponding to radial mesh
!!    %orbitals(basis_size)= Quantum number l for each basis function
!!    %prj_meshsz= Dimension of radial mesh for tProj inherited from uspp
!!    %sph_meshsz= Dimension of radial mesh for spheres
!!    %wav_meshsz=Mesh size for partial waves and projectors inherited from uspp
!!
!! SIDE EFFECTS
!!  pawps
!!    %phi(mesh_size,basis_size)= PAW atomic wavefunctions
!!                                on the radial grid
!!    %tphi(mesh_size,basis_size)= PAW atomic pseudo-wavefunctions
!!                                 on the radial grid
!!    %tproj(prj_msz_max,basis_size)= PAW projectors
!!                                    on the radial grid
!!
!! PARENTS
!!      uspp2abinit
!!
!! CHILDREN
!!      ctrap
!!
!! SOURCE

 subroutine ortho_phi(pawarray,pawdata,pawps,pshead,usdata,option)

 use defs_basis
 use defs_pawps

 implicit none

!Arguments ---------------------------------------------
 integer :: option
!These types are defined in defs_pawps
 type(pawarray_type) :: pawarray
 type(pawdata_type)  :: pawdata
 type(pawps_type)    :: pawps
 type(pshead_type)   :: pshead
 type(usdata_type)   :: usdata

!Local variables ---------------------------------------
 integer :: ib,ib1,ib2,il,msize,psize,wsize
 real(dp) :: f2l,intg1,intg2
 integer :: iref(2,4),nref(4)
 real(dp), allocatable :: ff(:)

! *************************************************************************

 if (option==0) return
 if (option/=1.and.option/=2) stop 'Bug in ortho_phi !'

!Recompute tproj from tphi
!-------------------------
 if (option==2) then
  msize=pshead%sph_meshsz;allocate(ff(msize))
  do ib=1,pshead%basis_size
!  Compute <tPhi_i|k|tPhi_i>
   ff(1:msize)=pawarray%shapefunc(1:msize)*pawps%tphi(1:msize,ib)**2 &
&             *pawdata%rad_(1:msize)
   call ctrap(msize,ff,pshead%log_step,intg1)
!  Compute k(r).tPhi_i(r)/<tPhi_i|k|tPhi_i>
   pawps%tproj(:,ib)=0.d0
   pawps%tproj(1:msize,ib)=pawarray%shapefunc(1:msize)*pawps%tphi(1:msize,ib)/intg1
  enddo
  deallocate(ff)
  pshead%prj_meshsz=pshead%sph_meshsz
 endif

!Search for links between angular momenta and partial waves
 nref=0
 do il=1,pshead%lmax+1
  do ib=1,pshead%basis_size
   if (pshead%orbitals(ib)==(il-1)) then
    if (nref(il)==0) then
     nref(il)=1;iref(1,il)=ib
    else
     nref(il)=nref(il)+1
     if (nref(il)>2) then
      write(6,'(/,a,/,a,i1,a,/,a)') 'Info:',&
&           '  Too much partial waves corrresponding to l=',il,' !',&
&           '  Program stopped.'
      stop
     endif
     iref(nref(il),il)=ib
    endif
   endif
  enddo
 enddo
 do il=1,pshead%lmax+1
  if (nref(il)==2) then
   ib1=iref(1,il);ib2=iref(2,il)
   if (usdata%refkeys(ib2)>0) then
    if (usdata%refkeys(ib2)==il) then
     iref(1,il)=ib2
     iref(2,il)=ib1
    endif
   endif
  endif
 enddo

!Re-orthogonalize Phi, tPhi and tProj
!for each l with 2 partial waves
!------------------------------------
 wsize=pshead%wav_meshsz
 psize=pshead%prj_meshsz
 allocate(ff(psize))
 do il=1,pshead%lmax+1
  if (nref(il)==2) then
   ib1=iref(1,il);ib2=iref(2,il)
!  Compute sqrt(1-<tPhi_2l|tProj_1l><tPhi_1l|tProj_2l>)
   ff(1:psize)=pawps%tphi(1:psize,ib2)*pawps%tproj(1:psize,ib1) &
&             *pawdata%rad_(1:psize)
   call ctrap(psize,ff,pshead%log_step,intg1)
   ff(1:psize)=pawps%tphi(1:psize,ib1)*pawps%tproj(1:psize,ib2) &
&             *pawdata%rad_(1:psize)
   call ctrap(psize,ff,pshead%log_step,intg2)
   f2l=1.d0/sqrt(1.d0-intg1*intg2)
!  Compute <tProj_1l|tPhi_2l>
   ff(1:psize)=pawps%tproj(1:psize,ib1)*pawps%tphi(1:psize,ib2) &
&             *pawdata%rad_(1:psize)
   call ctrap(psize,ff,pshead%log_step,intg1)
!  Compute <tPhi_1l|tProj_2l>
   ff(1:psize)=pawps%tphi(1:psize,ib1)*pawps%tproj(1:psize,ib2) &
&             *pawdata%rad_(1:psize)
   call ctrap(psize,ff,pshead%log_step,intg2)
!  Compute Phi_2l, tPhi_2l, tProj_2l
   pawps%phi  (1:wsize,ib2)=f2l*(pawps%phi  (1:wsize,ib2)-pawps%phi  (1:wsize,ib1)*intg1)
   pawps%tphi (1:wsize,ib2)=f2l*(pawps%tphi (1:wsize,ib2)-pawps%tphi (1:wsize,ib1)*intg1)
   pawps%tproj(1:psize,ib2)=f2l*(pawps%tproj(1:psize,ib2)-pawps%tproj(1:psize,ib1)*intg2)
  endif
 enddo
 deallocate(ff)

 end subroutine
!!***
