/*
 * easy_spice front end for spice
 * Copyright (C) 2000,2001 Routoure JM
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "easyspice.h"


/*---------------------------------------------------------------------- */
/*  Menu callbacks                                                       */ 
/*---------------------------------------------------------------------- */
void
on_menu_file_new_activate              (GtkMenuItem     *menuitem,
					 EasySpice *es)
{
  GtkFileSelection *file_dialog;
  
  file_dialog=(GtkFileSelection*)gtk_file_selection_new(_("Select new project name"));
  g_object_set_data(G_OBJECT(file_dialog), "easyspice", (gpointer) es);
  
  gtk_signal_connect_object(GTK_OBJECT(file_dialog->ok_button),"clicked",
			    GTK_SIGNAL_FUNC(on_new_file_select),
			    GTK_OBJECT(file_dialog));
  gtk_signal_connect_object_after(GTK_OBJECT(file_dialog->ok_button),"clicked",
				  (GtkSignalFunc) gtk_widget_destroy,
				  GTK_OBJECT(file_dialog));
  gtk_signal_connect_object(GTK_OBJECT(file_dialog->cancel_button),"clicked",
			    (GtkSignalFunc) gtk_widget_destroy,
			    GTK_OBJECT(file_dialog));
  gtk_widget_show(GTK_WIDGET(file_dialog));
}

void
on_new_file_select(GtkWidget *file_dialog, gpointer user_data)
{
  EasySpice *es;
  gchar *path, *file, *dir, *file2;
  
  es=g_object_get_data(G_OBJECT(file_dialog), "easyspice");
  path=g_strdup(gtk_file_selection_get_filename
		(GTK_FILE_SELECTION(file_dialog)));

  if (VERBOSE2)
    printf("newfile: filename:%s\n", path);

  dir=g_path_get_dirname(path);
  if (g_file_test(dir, G_FILE_TEST_IS_DIR) == FALSE) {
    write_log("directory does not exist");
  }
  else {
    file=g_path_get_basename(path);
    /* change to the new directory */
    chdir(dir);
    /* strip of the extention and set the new projectname */
    if (g_str_has_suffix(file, PROJECT_EXTENTION)) {
      file2=g_strndup(file, strlen(file)-strlen(PROJECT_EXTENTION));
      es->project=update_str(es->project, file2);
      g_free(file2);
    }
    else {
      es->project=update_str(es->project, file);
    }
    project_set_title(es);
    easyspice_filename_update(es);
    g_free(file);
    /* clear the old data */
    spicelibrary_delete(es->spicelibrary);
    postprocessing_delete(es->postproc);
    spice_variables_default(es->spice_vars);
    /* load local configuration file */
    localconfiguration_file_read(es);
    /* copy the content to the GUI */
    spice_variables_set_entries(es->win_main, es->spice_vars);
  }
  g_free(path); g_free(dir);
}


void
on_menu_file_open_activate             (GtkMenuItem     *menuitem,
                                        EasySpice *es)
{
  GtkFileSelection *file_dialog;
  
  file_dialog=(GtkFileSelection*)gtk_file_selection_new(_("Open a project"));
  g_object_set_data(G_OBJECT(file_dialog), "easyspice", (gpointer) es);
  
  gtk_signal_connect_object(GTK_OBJECT(file_dialog->ok_button),"clicked",
			    GTK_SIGNAL_FUNC(on_open_file_select),
			    GTK_OBJECT(file_dialog));
  gtk_signal_connect_object_after(GTK_OBJECT(file_dialog->ok_button),"clicked",
				  (GtkSignalFunc) gtk_widget_destroy,
				  GTK_OBJECT(file_dialog));
  gtk_signal_connect_object(GTK_OBJECT(file_dialog->cancel_button),"clicked",
			    (GtkSignalFunc) gtk_widget_destroy,
			    GTK_OBJECT(file_dialog));
  gtk_widget_show(GTK_WIDGET(file_dialog));
}

void
on_open_file_select(GtkWidget *file_dialog, gpointer user_data)
{
  EasySpice *es;
  gchar *path, *file, *dir, *file2;
  
  es=g_object_get_data(G_OBJECT(file_dialog), "easyspice");
  path=g_strdup(gtk_file_selection_get_filename
		(GTK_FILE_SELECTION(file_dialog)));

  if (VERBOSE2)
    printf("open_file: filename:%s\n", path);
  
  if (g_file_test(path, G_FILE_TEST_IS_REGULAR) == FALSE) {
    write_log("file does not exist");
  }
  else {
    dir=g_path_get_dirname(path);
    file=g_path_get_basename(path);
    /* change to the new directory */
    chdir(dir);
    g_free(dir);
    /* strip of the extention and set the new projectname */
    if (g_str_has_suffix(file, PROJECT_EXTENTION)) {
      file2=g_strndup(file, strlen(file)-strlen(PROJECT_EXTENTION));
      es->project=update_str(es->project, file2);
      g_free(file2);
    }
    else {
      es->project=update_str(es->project, file);
    }
    project_set_title(es);
    easyspice_filename_update(es);
    g_free(file);
    /* clear the old data */
    spicelibrary_delete(es->spicelibrary);
    postprocessing_delete(es->postproc);
    spice_variables_default(es->spice_vars);
    /* load the file contents */
    localconfiguration_file_read(es);
    configuration_file_read(es, path);
    /* copy the content to the GUI */
    spice_variables_set_entries(es->win_main, es->spice_vars);
  }
  g_free(path); 
}


void
localconfiguration_file_read(EasySpice *es)
{
  gchar *local_rcfile;

  if (getenv("HOME") != NULL) {
    local_rcfile=g_strdup_printf("%s/%s",getenv("HOME"),PROJECT_RCFILENAME);
    if (g_file_test(local_rcfile, G_FILE_TEST_IS_REGULAR) == FALSE) {
      printf(_("No local configuration file found: %s\n"),local_rcfile);
    }
    else {
      configuration_file_read(es, local_rcfile);
    }
    g_free(local_rcfile);
  }
}

void
configuration_file_read(EasySpice *es, gchar *filename)
{
  FILE *file;
  gchar **split;
  gchar line[FICHIER_MAX_LINE];
  gint i;
  gboolean old_file=FALSE;
  
  if ((file = fopen(filename, "r")) == NULL ) {
    write_log(_("Error: No project file found.\n"));
    return;
  }

  /* guess whether we have an old-style colon-separated configuration file */
  if (fgets(line, FICHIER_MAX_LINE, file) != NULL) {
    split=g_strsplit(line,":",0);
    for (i=0; split[i] != NULL; i++)
      ; /* void */
    if (i>=3)
      old_file = TRUE;
  }
  fclose(file);
	
  if (old_file == TRUE)
    configuration_file_read_old(es, filename);
  else
    configuration_file_read_new(es, filename);
}

void
configuration_file_read_new(EasySpice *es, gchar *filename)
{
  GKeyFile *key_file;
  GError *error = NULL;
  gboolean flag;
  SpiceVariables *sv;
  GedaSettings *gs;
  ExternalProgramms *ep;
  gint i=0, ei=0;
  GList *iter;
  gchar *lib, *libname;

  sv = es->spice_vars;
  
  key_file = g_key_file_new();
  flag = g_key_file_load_from_file(key_file, filename, G_KEY_FILE_NONE, &error);
  if (flag == FALSE) {
    printf(_("Error: Loading configuration file %s failed.\n"), filename);
    printf(_("  GError: %s"), error->message);
  }

  if (g_key_file_has_group(key_file, "Easyspice") == TRUE) {
  /* TODO: file version check */
    
  }

  if (g_key_file_has_group(key_file, "Easyspice_Settings") == TRUE) {
    sv->op_show_netlist= g_key_file_get_boolean(key_file, "Easyspice_Settings", "OP_SHOW_NETLIST", &error);
    sv->op_show_models= g_key_file_get_boolean(key_file, "Easyspice_Settings", "OP_SHOW_MODELS", &error);
    sv->op_show_results_schematic= g_key_file_get_boolean(key_file, "Easyspice_Settings", "OP_SHOW_SCHEMATIC", &error);
    sv->param_activ = g_key_file_get_boolean(key_file, "Easyspice_Settings", "PARAM_ACTIV", &error);
    sv->plot_autoupdate = g_key_file_get_boolean(key_file, "Easyspice_Settings", "PLOT_AUTOUPDATE", &error);
  }
  
  if (g_key_file_has_group(key_file, "Spice_Variables") == TRUE) {
    set_str(&(sv->dc_source1), g_key_file_get_string(key_file, "Spice_Variables", "DC_SOURCE1", &error));
    set_str(&(sv->dc_start1), g_key_file_get_string(key_file, "Spice_Variables", "DC_START1", &error));
    set_str(&(sv->dc_stop1), g_key_file_get_string(key_file, "Spice_Variables", "DC_STOP1", &error));
    set_str(&(sv->dc_inc1), g_key_file_get_string(key_file, "Spice_Variables", "DC_INC1", &error));
    set_str(&(sv->dc_source2), g_key_file_get_string(key_file, "Spice_Variables", "DC_SOURCE2", &error));
    set_str(&(sv->dc_start2), g_key_file_get_string(key_file, "Spice_Variables", "DC_START2", &error));
    set_str(&(sv->dc_stop2), g_key_file_get_string(key_file, "Spice_Variables", "DC_STOP2", &error));
    set_str(&(sv->dc_inc2), g_key_file_get_string(key_file, "Spice_Variables", "DC_INC2", &error));
  
    sv->ac_type= g_key_file_get_integer(key_file, "Spice_Variables", "AC_TYPE", &error);
    set_str(&(sv->ac_points), g_key_file_get_string(key_file, "Spice_Variables", "AC_POINTS", &error));
    set_str(&(sv->ac_fmin), g_key_file_get_string(key_file, "Spice_Variables", "AC_FMIN", &error));
    set_str(&(sv->ac_fmax), g_key_file_get_string(key_file, "Spice_Variables", "AC_FMAX", &error));
    
    set_str(&(sv->noise_node), g_key_file_get_string(key_file, "Spice_Variables", "NOISE_NODE", &error));
    set_str(&(sv->noise_source), g_key_file_get_string(key_file, "Spice_Variables", "NOISE_SOURCE", &error));
    sv->noise_type= g_key_file_get_integer(key_file, "Spice_Variables", "NOISE_TYPE", &error);
    set_str(&(sv->noise_points), g_key_file_get_string(key_file, "Spice_Variables", "NOISE_POINTS", &error));
    set_str(&(sv->noise_fmin), g_key_file_get_string(key_file, "Spice_Variables", "NOISE_FMIN", &error)); 
    set_str(&(sv->noise_fmax), g_key_file_get_string(key_file, "Spice_Variables", "NOISE_FMAX",  &error));
    
    set_str(&(sv->tran_tstop), g_key_file_get_string(key_file, "Spice_Variables", "TRAN_TSTOP", &error));
    set_str(&(sv->tran_tstep), g_key_file_get_string(key_file, "Spice_Variables", "TRAN_TSTEP", &error));
    set_str(&(sv->tran_tmax), g_key_file_get_string(key_file, "Spice_Variables", "TRAN_TMAX", &error));
    set_str(&(sv->tran_tstart), g_key_file_get_string(key_file, "Spice_Variables", "TRAN_TSTART", &error));
    
    set_str(&(sv->tf_output), g_key_file_get_string(key_file, "Spice_Variables", "TF_OUTPUT", &error));
    set_str(&(sv->tf_source), g_key_file_get_string(key_file, "Spice_Variables", "TF_SOURCE", &error));
    
    set_str(&(sv->param_name), g_key_file_get_string(key_file, "Spice_Variables", "PARAM_NAME", &error));
    set_str(&(sv->param_param), g_key_file_get_string(key_file, "Spice_Variables", "PARAM_PARAM", &error));
    sv->param_type= g_key_file_get_integer(key_file, "Spice_Variables", "PARAM_TYPE", &error);
    set_str(&(sv->param_start), g_key_file_get_string(key_file, "Spice_Variables", "PARAM_START", &error));
    set_str(&(sv->param_stop), g_key_file_get_string(key_file, "Spice_Variables", "PARAM_STOP", &error));
    set_str(&(sv->param_step), g_key_file_get_string(key_file, "Spice_Variables", "PARAM_STEP", &error));
    set_str(&(sv->param_values), g_key_file_get_string(key_file, "Spice_Variables", "PARAM_VALUES", &error));
    
    set_str(&(sv->plot_pre), g_key_file_get_string(key_file, "Spice_Variables", "PLOT_PREPROCESSING", &error));
    sv->plot_xdefault= g_key_file_get_boolean(key_file, "Spice_Variables", "PLOT_XDEFAULT", &error);
    set_str(&(sv->plot_xvalue), g_key_file_get_string(key_file, "Spice_Variables", "PLOT_XVALUE", &error));
    sv->plot_xlogscale= g_key_file_get_boolean(key_file, "Spice_Variables", "PLOT_XLOGSCALE", &error);
    sv->plot_xlinscale= g_key_file_get_boolean(key_file, "Spice_Variables", "PLOT_XLINSCALE", &error);
    set_str(&(sv->plot_yvalues), g_key_file_get_string(key_file, "Spice_Variables", "PLOT_YVALUES", &error));
    sv->plot_ylogscale= g_key_file_get_boolean(key_file, "Spice_Variables", "PLOT_YLOGSCALE", &error);

    set_str(&(es->postproc->commands), g_key_file_get_string(key_file, "Spice_Variables", "SIMULATION_POSTPROCESSING", &error));
  }

  if (g_key_file_has_group(key_file, "gEDA_Settings") == TRUE) {
    gs =es->geda;
    gs->include_cwd= g_key_file_get_boolean(key_file, "gEDA_Settings", "GEDA_INCLUDE_CWD", &error);
    set_str(&(gs->include_dir), g_key_file_get_string(key_file, "gEDA_Settings", "GEDA_INCLUDE_DIR", &error));
    gs->pinlabel_attr= g_key_file_get_boolean(key_file, "gEDA_Settings", "GEDA_PINLABEL_ATTR", &error);
    set_str(&(gs->attr), g_key_file_get_string(key_file, "gEDA_Settings", "GEDA_ATTR", &error));
    set_str(&(gs->gschem_textsize), g_key_file_get_string(key_file, "gEDA_Settings", "GSCHEM_TEXTSIZE", &error));
    set_str(&(gs->gschem_gridsize), g_key_file_get_string(key_file, "gEDA_Settings", "GSCHEM_GRIDSIZE", &error));
    gs->gnetlist_priority_node= g_key_file_get_boolean(key_file, "gEDA_Settings", "GNETLIST_PRIORITY_NODE", &error);
    gs->gnetlist_hierarchy_allow= g_key_file_get_boolean(key_file, "gEDA_Settings", "GNETLIST_HIERARCHY_ALLOW", &error);
    set_str(&(gs->gnetlist_command_line), g_key_file_get_string(key_file, "gEDA_Settings", "GNETLIST_COMMAND_LINE", &error));
  }

  if (g_key_file_has_group(key_file, "Commands") == TRUE) {
    ep= es->commands;
    set_str(&(ep->editor), g_key_file_get_string(key_file, "Commands", "EDITOR", &error));
    set_str(&(ep->plot_command), g_key_file_get_string(key_file, "Commands", "PLOT_COMMAND", &error));
    set_str(&(ep->spice_command), g_key_file_get_string(key_file, "Commands", "SPICE_COMMAND", &error));
    set_str(&(ep->browser), g_key_file_get_string(key_file, "Commands", "BROWSER", &error));
    set_str(&(ep->html_files), g_key_file_get_string(key_file, "Commands", "SPICE_DOCFILE", &error));
    set_str(&(ep->spicehelp_command), g_key_file_get_string(key_file, "Commands", "SPICE_HELP", &error));
  }

  if (g_key_file_has_group(key_file, "Spice_Library") == TRUE) {
    i=0; ei=0;
    while (ei < 100) { /* this prevents load stopping if there are gaps in the enumering */
      libname = g_strdup_printf("library%i",i);
      lib = g_key_file_get_string(key_file, "Spice_Library", libname, &error);
      if (lib == NULL) {
	ei++;
	g_error_free(error);
	error = NULL;
      }
      else {
	/* check the library list for dulicate entries */
	for (iter=es->spicelibrary->libs; iter != NULL; iter = g_list_next(iter)) {
	  if (strcmp(lib, iter->data) == 0) {
	    g_free(lib);
	    break;
	  }
	}
	if (iter == NULL) {
	  es->spicelibrary->libs = g_list_append(es->spicelibrary->libs, lib);
	}
      }
      g_free(libname);
      i++;
    }
  }
}

void
configuration_file_read_old(EasySpice *es, gchar *filename)
{
  FILE *file;
  gchar **split;
  gchar line[FICHIER_MAX_LINE];
  gint i;
  SpiceVariables *sv;
  
  sv=es->spice_vars;

  if ((file = fopen(filename, "r")) == NULL ) {
    write_log(_("Error: No project file found.\n"));
    return;
  }

  while (fgets(line, FICHIER_MAX_LINE, file) != NULL) {

    split=g_strsplit(line,":",0);
    for (i=0; split[i] != NULL; i++)
      ; /* void */
    if (i<3) {
      printf("config file problem: \"%s\"", line);
      continue;
    }

    if (strcmp(split[0],"win_prin")==0) {
      readspice_value(&(sv->dc_source1), "dc_source1", split[1], split[2]);
      readspice_value(&(sv->dc_start1), "dc_start1", split[1], split[2]);
      readspice_value(&(sv->dc_stop1), "dc_stop1", split[1], split[2]);
      readspice_value(&(sv->dc_inc1), "dc_inc1", split[1], split[2]);
      readspice_value(&(sv->dc_source2), "dc_source2", split[1], split[2]);
      readspice_value(&(sv->dc_start2), "dc_start2", split[1], split[2]);
      readspice_value(&(sv->dc_stop2), "dc_stop2", split[1], split[2]);
      readspice_value(&(sv->dc_inc2), "dc_inc2", split[1], split[2]);
      readspice_value(&(sv->ac_points), "ac_nombre", split[1], split[2]);
      readspice_value(&(sv->ac_fmin), "ac_fmin", split[1], split[2]);
      readspice_value(&(sv->ac_fmax), "ac_fmax", split[1], split[2]);
      readspice_value(&(sv->tran_tstep), "tran_tstep", split[1], split[2]);
      readspice_value(&(sv->tran_tmax), "tran_tmax", split[1], split[2]);
      readspice_value(&(sv->tran_tstart), "tran_tstart", split[1], split[2]);
      readspice_value(&(sv->tran_tstop), "tran_tstop", split[1], split[2]);
      readspice_value(&(sv->tf_output), "tf_output", split[1], split[2]);
      readspice_value(&(sv->tf_source), "tf_source", split[1], split[2]);
      readspice_value(&(sv->noise_node), "noise_node", split[1], split[2]);
      readspice_value(&(sv->noise_source), "noise_source", split[1], split[2]);
      readspice_value(&(sv->noise_points), "noise_nombre", split[1], split[2]);
      readspice_value(&(sv->noise_fmin), "noise_fmin", split[1], split[2]);
      readspice_value(&(sv->noise_fmax), "noise_fmax", split[1], split[2]);
      readspice_value(&(sv->param_name), "param_name", split[1], split[2]);
      readspice_value(&(sv->param_param), "param_param", split[1], split[2]);
      readspice_value(&(sv->param_start), "param_start", split[1], split[2]);
      readspice_value(&(sv->param_stop), "param_stop", split[1], split[2]);
      readspice_value(&(sv->param_step), "param_step", split[1], split[2]);
      readspice_value(&(sv->param_values), "param_values", split[1], split[2]);
      readspice_value(&(sv->plot_yvalues), "plot_y", split[1], split[2]);

      if (strcmp(split[1],"ac_type")==0) {
	sv->ac_type=atoi(split[2]);
      }
      else if (strcmp(split[1],"noise_type")==0) {
	sv->noise_type=atoi(split[2]);
      }
      else if (strcmp(split[1],"param_type")==0) {
	sv->param_type=atoi(split[2]);
      }
      else if (strcmp(split[1],"plot_pre_text")==0) {
	sv->plot_pre=
	  update_str(sv->plot_pre, 
		     g_strdup_printf("%s%s\n", sv->plot_pre, split[2]));
      }
      else {
	; //printf("unknown name: \"%s\"\n",split[1]);
      }
    } /* endif win_prin */
    else if (strcmp(split[0],"win_post")==0) {
      es->postproc->commands=
	update_str(es->postproc->commands, 
		   g_strdup_printf("%s%s\n", es->postproc->commands,split[2]));
    }
    else if (strcmp(split[0],"win_lib")==0) {
      es->spicelibrary->libs=g_list_append(es->spicelibrary->libs,
					   g_strdup(split[2]));
    }
    g_strfreev(split);
  }
}

void
localconfiguration_file_write(EasySpice *es)
{
  gchar *local_rcfile;

  if (getenv("HOME") != NULL) {
    local_rcfile=g_strdup_printf("%s/%s",getenv("HOME"), PROJECT_RCFILENAME);
    /* TODO: insert GTK26 check */
    configuration_file_write_new(es, local_rcfile, TRUE);
    g_free(local_rcfile);
  }
}

void
configuration_file_write_new(EasySpice *es, gchar *filename, gboolean configfile)
{
  GKeyFile *key_file;
  GError *error=NULL;
  gchar *content;
  gsize content_size;
  FILE *file;
  SpiceVariables *sv;
  GedaSettings *gs;
  ExternalProgramms *ep;
  GList *iter;
  gint i;
  gchar *libname;

  sv = es->spice_vars;
  
  if (configfile == TRUE) {
    printf("Writing local configuration file\n");
  }
  else {
    printf("Writing project configration file\n");
  }

  key_file= g_key_file_new();

  g_key_file_set_string(key_file, "Easyspice", "VERSION", VERSION);
  g_key_file_set_integer(key_file, "Easyspice", "FILE_FORMAT_VERSION", FILE_FORMAT_VERSION);
			 
  g_key_file_set_boolean(key_file, "Easyspice_Settings", "OP_SHOW_NETLIST", sv->op_show_netlist);
  g_key_file_set_boolean(key_file, "Easyspice_Settings", "OP_SHOW_MODELS", sv->op_show_models);
  g_key_file_set_boolean(key_file, "Easyspice_Settings", "OP_SHOW_SCHEMATIC", sv->op_show_results_schematic);
  g_key_file_set_boolean(key_file, "Easyspice_Settings", "PARAM_ACTIV", sv->param_activ);
  g_key_file_set_boolean(key_file, "Easyspice_Settings", "PLOT_AUTOUPDATE", sv->plot_autoupdate);

  g_key_file_set_string(key_file, "Spice_Variables", "DC_SOURCE1", sv->dc_source1);
  g_key_file_set_string(key_file, "Spice_Variables", "DC_START1", sv->dc_start1);
  g_key_file_set_string(key_file, "Spice_Variables", "DC_STOP1", sv->dc_stop1);
  g_key_file_set_string(key_file, "Spice_Variables", "DC_INC1", sv->dc_inc1);
  g_key_file_set_string(key_file, "Spice_Variables", "DC_SOURCE2", sv->dc_source2);
  g_key_file_set_string(key_file, "Spice_Variables", "DC_START2", sv->dc_start2);
  g_key_file_set_string(key_file, "Spice_Variables", "DC_STOP2", sv->dc_stop2);
  g_key_file_set_string(key_file, "Spice_Variables", "DC_INC2", sv->dc_inc2);

  g_key_file_set_integer(key_file, "Spice_Variables", "AC_TYPE", sv->ac_type);
  g_key_file_set_string(key_file, "Spice_Variables", "AC_POINTS", sv->ac_points);
  g_key_file_set_string(key_file, "Spice_Variables", "AC_FMIN", sv->ac_fmin);
  g_key_file_set_string(key_file, "Spice_Variables", "AC_FMAX", sv->ac_fmax);

  g_key_file_set_string(key_file, "Spice_Variables", "NOISE_NODE", sv->noise_node);
  g_key_file_set_string(key_file, "Spice_Variables", "NOISE_SOURCE", sv->noise_source);
  g_key_file_set_integer(key_file, "Spice_Variables", "NOISE_TYPE", sv->noise_type);
  g_key_file_set_string(key_file, "Spice_Variables", "NOISE_POINTS", sv->noise_points);
  g_key_file_set_string(key_file, "Spice_Variables", "NOISE_FMIN", sv->noise_fmin);
  g_key_file_set_string(key_file, "Spice_Variables", "NOISE_FMAX", sv->noise_fmax);

  g_key_file_set_string(key_file, "Spice_Variables", "TRAN_TSTOP", sv->tran_tstop);
  g_key_file_set_string(key_file, "Spice_Variables", "TRAN_TSTEP", sv->tran_tstep);
  g_key_file_set_string(key_file, "Spice_Variables", "TRAN_TMAX", sv->tran_tmax);
  g_key_file_set_string(key_file, "Spice_Variables", "TRAN_TSTART", sv->tran_tstart);

  g_key_file_set_string(key_file, "Spice_Variables", "TF_OUTPUT", sv->tf_output);
  g_key_file_set_string(key_file, "Spice_Variables", "TF_SOURCE", sv->tf_source);

  g_key_file_set_string(key_file, "Spice_Variables", "PARAM_NAME", sv->param_name);
  g_key_file_set_string(key_file, "Spice_Variables", "PARAM_PARAM", sv->param_param);
  g_key_file_set_integer(key_file, "Spice_Variables", "PARAM_TYPE", sv->param_type);
  g_key_file_set_string(key_file, "Spice_Variables", "PARAM_START", sv->param_start);
  g_key_file_set_string(key_file, "Spice_Variables", "PARAM_STOP", sv->param_stop);
  g_key_file_set_string(key_file, "Spice_Variables", "PARAM_STEP", sv->param_step);
  g_key_file_set_string(key_file, "Spice_Variables", "PARAM_VALUES", sv->param_values);

  g_key_file_set_string(key_file, "Spice_Variables", "PLOT_PREPROCESSING", sv->plot_pre);
  g_key_file_set_boolean(key_file, "Spice_Variables", "PLOT_XDEFAULT", sv->plot_xdefault);
  g_key_file_set_string(key_file, "Spice_Variables", "PLOT_XVALUE", sv->plot_xvalue);
  g_key_file_set_boolean(key_file, "Spice_Variables", "PLOT_XLOGSCALE", sv->plot_xlogscale);
  g_key_file_set_boolean(key_file, "Spice_Variables", "PLOT_XLINSCALE", sv->plot_xlinscale);
  g_key_file_set_string(key_file, "Spice_Variables", "PLOT_YVALUES", sv->plot_yvalues);
  g_key_file_set_boolean(key_file, "Spice_Variables", "PLOT_YLOGSCALE", sv->plot_ylogscale);

  g_key_file_set_string(key_file, "Spice_Variables", "SIMULATION_POSTPROCESSING", es->postproc->commands);

  /* enumerate the libraries and store them into the keyfile */
  for (i=0, iter=es->spicelibrary->libs; iter != NULL; iter = g_list_next(iter), i++) {
    libname = g_strdup_printf("library%i",i);
    g_key_file_set_string(key_file, "Spice_Library", libname, iter->data);
    g_free(libname);
  }

  gs =es->geda;
  g_key_file_set_boolean(key_file, "gEDA_Settings", "GEDA_INCLUDE_CWD", gs->include_cwd);
  g_key_file_set_string(key_file, "gEDA_Settings", "GEDA_INCLUDE_DIR", gs->include_dir);
  g_key_file_set_boolean(key_file, "gEDA_Settings", "GEDA_PINLABEL_ATTR", gs->pinlabel_attr);
  g_key_file_set_string(key_file, "gEDA_Settings", "GEDA_ATTR", gs->attr);
  g_key_file_set_string(key_file, "gEDA_Settings", "GSCHEM_TEXTSIZE", gs->gschem_textsize);
  g_key_file_set_string(key_file, "gEDA_Settings", "GSCHEM_GRIDSIZE", gs->gschem_gridsize);
  g_key_file_set_boolean(key_file, "gEDA_Settings", "GNETLIST_PRIORITY_NODE", gs->gnetlist_priority_node);
  g_key_file_set_boolean(key_file, "gEDA_Settings", "GNETLIST_HIERARCHY_ALLOW", gs->gnetlist_hierarchy_allow);
  g_key_file_set_string(key_file, "gEDA_Settings", "GNETLIST_COMMAND_LINE", gs->gnetlist_command_line);
  
  /* don't store the commands and gEDA-Settings into the project file */
  if (configfile == FALSE) {
    ep = es->commands;
    g_key_file_set_string(key_file, "Commands", "EDITOR", ep->editor);
    g_key_file_set_string(key_file, "Commands", "PLOT_COMMAND", ep->plot_command);
    g_key_file_set_string(key_file, "Commands", "SPICE_COMMAND", ep->spice_command);
    g_key_file_set_string(key_file, "Commands", "BROWSER", ep->browser);
    g_key_file_set_string(key_file, "Commands", "SPICE_DOCFILE", ep->html_files);
    g_key_file_set_string(key_file, "Commands", "SPICE_HELP", ep->spicehelp_command);
  }
  
  content = g_key_file_to_data(key_file, &content_size, &error);
  file = fopen(filename,"wb");
  fwrite(content, content_size, 1, file);
  fclose(file);

  g_free(content);
  g_key_file_free(key_file);
}

void
configuration_file_write_old(EasySpice *es, gchar *filename)
{

  FILE *file;
  SpiceVariables *sv;
  GList *tmp;

  sv=es->spice_vars;

  clean_log(es);
  if (strlen(es->project) == 0) {
    write_log(_("Error: No project name specified\n"));
    write_log(_("Error: Stopped saving the project\n"));
    return;
  }
  spice_variables_get_entries(es->win_main, es->spice_vars);
  
  if ((file = fopen(es->file_easyspice_project, "wt")) ==NULL) {
    write_log(_("Error: unable to write the file"));
    return;
  }
  
  fprintf(file,"win_prin:dc_source1:%s:\n",sv->dc_source1);
  fprintf(file,"win_prin:dc_start1:%s:\n",sv->dc_start1);
  fprintf(file,"win_prin:dc_stop1:%s:\n",sv->dc_stop1);
  fprintf(file,"win_prin:dc_inc1:%s:\n",sv->dc_inc1);
  
  fprintf(file,"win_prin:dc_source2:%s:\n",sv->dc_source2);
  fprintf(file,"win_prin:dc_start2:%s:\n",sv->dc_start2);
  fprintf(file,"win_prin:dc_stop2:%s:\n",sv->dc_stop2);
  fprintf(file,"win_prin:dc_inc2:%s:\n",sv->dc_inc2);
  
  fprintf(file,"win_prin:ac_type:%d:\n",sv->ac_type);
  fprintf(file,"win_prin:ac_nombre:%s:\n",sv->ac_points);
  fprintf(file,"win_prin:ac_fmin:%s:\n",sv->ac_fmin);
  fprintf(file,"win_prin:ac_fmax:%s:\n",sv->ac_fmax);
  
  fprintf(file,"win_prin:tran_tstep:%s:\n",sv->tran_tstep);
  fprintf(file,"win_prin:tran_tmax:%s:\n",sv->tran_tmax);
  fprintf(file,"win_prin:tran_tstart:%s:\n",sv->tran_tstart);
  fprintf(file,"win_prin:tran_tstop:%s:\n",sv->tran_tstop);
  
  fprintf(file,"win_prin:tf_output:%s:\n",sv->tf_output);
  fprintf(file,"win_prin:tf_source:%s:\n",sv->tf_source);
  
  fprintf(file,"win_prin:noise_node:%s:\n",sv->noise_node);
  fprintf(file,"win_prin:noise_source:%s:\n",sv->noise_source);
  fprintf(file,"win_prin:noise_nombre:%s:\n",sv->noise_points);
  fprintf(file,"win_prin:noise_fmin:%s:\n",sv->noise_fmin);
  fprintf(file,"win_prin:noise_fmax:%s:\n",sv->noise_fmax);
  
  fprintf(file,"win_prin:noise_type:%d:\n",sv->noise_type);
  
  fprintf(file,"win_prin:param_name:%s:\n",sv->param_name);
  fprintf(file,"win_prin:param_param:%s:\n",sv->param_param);
  fprintf(file,"win_prin:param_start:%s:\n",sv->param_start);
  fprintf(file,"win_prin:param_stop:%s:\n",sv->param_stop);
  fprintf(file,"win_prin:param_step:%s:\n",sv->param_step);
  fprintf(file,"win_prin:param_values:%s:\n",sv->param_values);
  fprintf(file,"win_prin:param_type:%d:\n",sv->param_type);
  
  fprintf(file,"win_prin:plot_y:%s:\n",sv->plot_yvalues);
  
  write_multiline_text(file,"win_prin","plot_pre_text",sv->plot_pre);

  /* postproc commands */
  write_multiline_text(file,"win_post","text_post_trait_common",
		       es->postproc->commands);
  
  /* save the libraries */
  for (tmp=g_list_first(es->spicelibrary->libs);
       tmp != NULL; tmp=g_list_next(tmp)) {
    fprintf(file,"win_lib:chemin_bib:%s:\n",(gchar*) tmp->data);
  }
  
  fclose(file);
}

void 
write_multiline_text(FILE *file, gchar *section, gchar *name, gchar *data)
{
  gint i;
  gchar **split;

  if (data!=NULL) {
    split=g_strsplit(data,"\n",0); /* tokenize the data*/
    for (i=0; split[i]!=NULL; i++) {
      if (DEBUG) 
	printf("write_multiline_text: line=%d, value=%s\n", i, split[i]);
      fprintf(file,"%s:%s:%s:\n",section, name, split[i]);
    }
    g_strfreev(split);
  }
}

void
on_menu_file_save_activate              (GtkMenuItem     *menuitem,
					 EasySpice *es)
{ 
  /* TODO: insert gtk26 check */
  if (1)
    configuration_file_write_new(es, es->file_easyspice_project, FALSE);
  else
    configuration_file_write_old(es, es->file_easyspice_project);
}

void
on_menu_file_saveas_activate              (GtkMenuItem     *menuitem,
					   EasySpice *es)
{
  GtkFileSelection *file_dialog;
  
  file_dialog=(GtkFileSelection*)gtk_file_selection_new(_("Save the project as ..."));
  g_object_set_data(G_OBJECT(file_dialog), "easyspice", (gpointer) es);
  
  gtk_signal_connect_object(GTK_OBJECT(file_dialog->ok_button),"clicked",
			    GTK_SIGNAL_FUNC(on_saveas_file_select),
			    GTK_OBJECT(file_dialog));
  gtk_signal_connect_object_after(GTK_OBJECT(file_dialog->ok_button),"clicked",
				  (GtkSignalFunc) gtk_widget_destroy,
				  GTK_OBJECT(file_dialog));
  gtk_signal_connect_object(GTK_OBJECT(file_dialog->cancel_button),"clicked",
			    (GtkSignalFunc) gtk_widget_destroy,
			    GTK_OBJECT(file_dialog));
  gtk_widget_show(GTK_WIDGET(file_dialog));
}

void
on_saveas_file_select (GtkWidget *file_dialog, gpointer user_data)
{
  EasySpice *es;
  gchar *path, *file, *dir, *file2;
  
  es=g_object_get_data(G_OBJECT(file_dialog), "easyspice");
  path=g_strdup(gtk_file_selection_get_filename
		(GTK_FILE_SELECTION(file_dialog)));

  if (VERBOSE2)
    printf("saveas: filename:%s\n", path);

  /* TODO: Check filename and filelocation */
  /* strip of the extention and set the new projectname */
  /* change to the new directory if it has changed */
  /* finaly save the file using the normal file->save function */

  dir=g_path_get_dirname(path);
  if (g_file_test(dir, G_FILE_TEST_IS_DIR) == FALSE) {
    write_log("directory does not exist");
  }
  else {
    file=g_path_get_basename(path);
    /* change to the new directory */
    chdir(dir);
    /* strip of the extention and set the new projectname */
    if (g_str_has_suffix(file, PROJECT_EXTENTION)) {
      file2=g_strndup(file, strlen(file)-strlen(PROJECT_EXTENTION));
      es->project=update_str(es->project, file2);
      g_free(file2);
    }
    else {
      es->project=update_str(es->project, file);
    }
    project_set_title(es);
    easyspice_filename_update(es);
    g_free(file);

    on_menu_file_save_activate(NULL, es);
  }
  g_free(path); g_free(dir);
}


void
on_menu_file_quit_activate                   (GtkMenuItem     *menuitem,
					      EasySpice *es)
{
  GtkWidget *dialog;
  dialog=create_verify_quit();
  gtk_widget_show(dialog);
}

gboolean    
on_main_window_destroy_event                  (GtkWidget *widget,
				GdkEvent *event,
				EasySpice *es)
{
  on_menu_file_quit_activate(NULL, es);
  return TRUE;
}

/*  ----------------------------------------------------------------------- */
/*                  Menu options                                            */
/*  ----------------------------------------------------------------------- */

void
on_menu_options_commands_activate      (GtkMenuItem     *menuitem,
                                        ExternalProgramms *prog)
{
  if (prog->dialog == NULL) {
    prog->dialog=create_command_setup(prog);
    commands_set_entries(prog->dialog, prog);
    gtk_widget_show(prog->dialog);
  }
  else {
    gtk_window_present(GTK_WINDOW(prog->dialog));
  }
}

void
on_menu_options_library_activate       (GtkMenuItem     *menuitem,
                                        SpiceLibrary    *spicelib)
{
  GList *tmp;
  GtkTreeIter iter;

  if (spicelib->dialog == NULL) {
    spicelib->store=gtk_list_store_new(1, G_TYPE_STRING);
    for (tmp=g_list_first(spicelib->libs); tmp != NULL;
	 tmp=g_list_next(tmp)) {
      gtk_list_store_append(spicelib->store, &iter);
      gtk_list_store_set(spicelib->store, &iter,
			 0, (gchar*)(tmp->data),
			 -1);
    }
    spicelib->dialog=create_library_dialog(spicelib);
    gtk_widget_show(spicelib->dialog);
  }
  else {
    gtk_window_present(GTK_WINDOW(spicelib->dialog));
  }
}

void
on_menu_options_gschem_activate        (GtkMenuItem     *menuitem,
                                        GedaSettings    *geda)
{
  if (geda->dialog == NULL) {
    geda->dialog=create_geda_setup(geda);
    geda_settings_set_entries(geda->dialog, geda); 
    gtk_widget_show(geda->dialog);
  }
  else {
    gtk_window_present(GTK_WINDOW(geda->dialog));
  }
}

void
on_menu_options_save_activate    (GtkMenuItem *menuitem,
				  EasySpice *es)
{
  spice_variables_get_entries(es->win_main, es->spice_vars);
  localconfiguration_file_write(es);
}


/*  ---------------------------------------------------------------------- */
/*                  Menu window                                            */
/*  ---------------------------------------------------------------------- */

void
on_menu_windows_simulcommands_activate  (GtkMenuItem     *menuitem,
					 EasySpice  *es)
{
  clean_log(es);
  if (strlen(es->project) == 0) {
    write_log(_("Error: No project name specified\n"));
    write_log(_("Error: Stopped showing the simulation commands\n"));
    return;
  }
  textfile_view(es->textviewer, es->file_spice_simul);
}


void
on_menu_windows_plotcommands_activate   (GtkMenuItem     *menuitem,
                                        EasySpice *es)
{
  clean_log(es);
  if (strlen(es->project) == 0) {
    write_log(_("Error: No project name specified\n"));
    write_log(_("Error: Stopped showing the plotting commands\n"));
    return;
  }
  textfile_view(es->textviewer, es->file_spice_plot);
}

void
on_menu_windows_spiceinteractive_activate  (GtkMenuItem     *menuitem,
                                            ExternalProgramms *ep)
{
  char *buf;
  buf=g_strdup_printf("xterm -e %s &", ep->spice_command);
  system(buf);
  g_free(buf);
}

void
on_menu_windows_postproc_activate           (GtkMenuItem     *menuitem,
					     PostProcessing  *postproc)
{
  if (postproc->dialog == NULL) {
    postproc->dialog=create_postproc(postproc);
    postprocessing_set_entries(postproc); 
    gtk_widget_show(postproc->dialog);
  }
  else {
    gtk_window_present(GTK_WINDOW(postproc->dialog));
  }
}


/*  ----------------------------------------------------------------------- */
/*                  Menu help                                               */
/*  ----------------------------------------------------------------------- */

void
on_menu_help_spiceman_activate            (GtkMenuItem     *menuitem,
                                        ExternalProgramms *ep)
{
  char *buf;
  buf=g_strdup_printf("xterm -e man %s &", ep->spice_command);
  system(buf);
  g_free(buf);
}


void
on_menu_help_spicehtml_activate          (GtkMenuItem     *menuitem,
					  ExternalProgramms *ep)
{ 
  char *buf;
  buf=g_strdup_printf("%s %s &", ep->browser, ep->html_files);
  system(buf);
  g_free(buf);
}

void
on_menu_help_spicehelp_activate         (GtkMenuItem     *menuitem,
					 ExternalProgramms *ep)
{
  gchar *buf;
  buf=g_strdup_printf("%s &",ep->spicehelp_command);
  system(buf);
  g_free(buf);
}

void
on_menu_help_about_activate           (GtkMenuItem     *menuitem,
	                               gpointer         user_data)
{
  char *buf;
  buf=g_strdup_printf("easy_spice version %s \n\n"
		      "JM Routoure\n"
		      "    (c) 2000-2006 University of Caen\n"
		      "Olivier Hervieu\n"
		      "Werner Hoch", VERSION);
  create_quickmessage(_("About"), buf);
  g_free(buf);
}
