# Copyright 2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Model for a filesystem group. Contains a set of filesystems that create
a virtual block device. E.g. LVM Volume Group."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    'FilesystemGroup',
    ]

from django.db.models import (
    CharField,
    ForeignKey,
)
from maasserver import DefaultMeta
from maasserver.enum import (
    CACHE_MODE_TYPE_CHOICES,
    FILESYSTEM_GROUP_TYPE_CHOICES,
)
from maasserver.models.cacheset import CacheSet
from maasserver.models.cleansave import CleanSave
from maasserver.models.timestampedmodel import TimestampedModel

# Size of LVM physical extent. 4MiB
LVM_PE_SIZE = 4 * 1024 * 1024


class FilesystemGroup(CleanSave, TimestampedModel):
    """A filesystem group. Contains a set of filesystems that create
    a virtual block device. E.g. LVM Volume Group.

    :ivar uuid: UUID of the filesystem group.
    :ivar group_type: Type of filesystem group.
    :ivar name: Name of the filesytem group.
    :ivar create_params: Parameters that can be passed during the create
        command when the filesystem group is created.
    """

    class Meta(DefaultMeta):
        """Needed for South to recognize this model."""

    uuid = CharField(
        max_length=36, unique=True, null=False, blank=False, editable=False)

    group_type = CharField(
        max_length=20, null=False, blank=False,
        choices=FILESYSTEM_GROUP_TYPE_CHOICES)

    name = CharField(
        max_length=255, null=False, blank=False)

    create_params = CharField(
        max_length=255, null=True, blank=True)

    cache_mode = CharField(
        max_length=20, null=True, blank=True,
        choices=CACHE_MODE_TYPE_CHOICES)

    cache_set = ForeignKey(
        CacheSet, null=True, blank=True)
