/* cmd_parser.y
   
   Bison grammar for a parser of Network Expect Tcl commands. This
   grammar basically parses an argv[] vector. Probably overkill but
   it sure makes CLI arguments parsing a breeze.

   The entry point to the command parser is cmd_parseargs(). This
   function received a command description block structure that is
   filled-in by the parser to send information back to the caller.

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

%pure-parser

%error-verbose

%parse-param {Tcl_Interp *interp}
%parse-param {struct cdb *cdb}
%parse-param {void *scanner}
%lex-param   {void *scanner}

/*************************************************************************
 **                               Tokens                                **
 *************************************************************************/

%union {
    char *sval;
}

%token GHOST GHOSTS WITH IP AND MAC ON INTERFACE INFO KILL
%token SHOW VERS DISSECTION VARS PACKET
%token <sval> STRING

%{
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <dnet.h>
#include <tcl.h>

#include "nexp_cmd_parser.h"
#include "util.h"
#include "util-tcl.h"
#include "xstrdup.h"

static void cmderror(Tcl_Interp *, struct cdb *, void *, char const *);
extern void *cmd_scan_string(const char *, void *);
extern int cmd_delete_buffer(void *, void *);
extern int cmdlex_init(void **);
extern int cmdlex_destroy(void *);
extern int cmdlex(YYSTYPE *, void *); /* The token analyzer.
					 Produced by Flex. */

%}

%%

cmddef:		  ghostcmd_def
		| showcmd_def
		;

ghostcmd_def:	  GHOST WITH IP STRING AND MAC STRING ON INTERFACE STRING
    {
	struct addr a;

	if (addr_pton($4, &a) == -1 || a.addr_type != ADDR_TYPE_IP) {
	    nexp_error(interp, "Invalid IP address \"%s\"", $4);
	    free($4);
	    free($7);
	    free($10);
	    YYABORT;
	}

	cdb->code = CMD_GHOST_CREATE;
	cdb->_ghost.l3addr = a;
	if (parse_mac_addr($7, &cdb->_ghost.l2addr) == -1) {
	    nexp_error(interp, "Invalid MAC address \"%s\"", $7);
	    free($4);
	    free($7);
	    free($10);
	    YYABORT;
	}
	cdb->_ghost.iface = $10;

	free($4);
	free($7);
    }
		| GHOST STRING STRING STRING /* short version */
    {
	struct addr a;

	if (addr_pton($2, &a) == -1 || a.addr_type != ADDR_TYPE_IP) {
	    nexp_error(interp, "Invalid IP address \"%s\"", $2);
	    free($2);
	    free($3);
	    free($4);
	    YYABORT;
	}

	cdb->code = CMD_GHOST_CREATE;
	cdb->_ghost.l3addr = a;
	if (parse_mac_addr($3, &cdb->_ghost.l2addr) == -1) {
	    nexp_error(interp, "Invalid hostname \"%s\"", $3);
	    free($2);
	    free($3);
	    free($4);
	    YYABORT;
	}
	cdb->_ghost.iface = $4;

	free($2);
	free($3);
    }
		| GHOST INFO
    {
	cdb->code = CMD_GHOST_INFO;
    }
		| GHOST KILL		/* Kill most recently created ghost */
    {
	cdb->code = CMD_GHOST_KILL;
	cdb->_ghost.name = NULL;
    }
		| GHOST KILL STRING	/* Kill a specific ghost */
    {
	cdb->code = CMD_GHOST_KILL;
	cdb->_ghost.name = $3;
    }
		;

showcmd_def:	  SHOW VERS
    {
	cdb->code = CMD_SHOW_VERSION;
    }
		| SHOW DISSECTION VARS
    {
	cdb->code = CMD_SHOW_DISSECTION_VARS;
    }
		| SHOW PACKET	/* Not implemented yet */
    {
	cdb->code = CMD_SHOW_PACKET;
    }
		| SHOW GHOSTS
    {
	cdb->code = CMD_SHOW_GHOSTS;
    }
		;

%%

void
cmderror(Tcl_Interp *interp, struct cdb *cdb _U_, void *scanner _U_,
	 char const *msg)
{
    nexp_error(interp, "%s", msg);
}

int
cmd_parseargs(Tcl_Interp *interp, const char *cmd, struct cdb *cdb)
{
    void *yybuf;
    void *scanner;
    int retval;

    cmdlex_init(&scanner);

    yybuf = cmd_scan_string(cmd, scanner);

    retval = cmdparse(interp, cdb, scanner);

    cmd_delete_buffer(yybuf, scanner);

    cmdlex_destroy(scanner);

    return retval;
}
