/*
 * File    : MediaBag.java
 * Created : 19-dec-2000 15:49
 * By      : fbusquets
 *
 * JClic - Authoring and playing system for educational activities
 *
 * Copyright (C) 2000 - 2005 Francesc Busquets & Departament
 * d'Educacio de la Generalitat de Catalunya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details (see the LICENSE file).
 */

package edu.xtec.jclic.bags;

import edu.xtec.jclic.PlayStation;
import edu.xtec.jclic.edit.Editable;
import edu.xtec.jclic.edit.Editor;
import edu.xtec.jclic.fileSystem.FileSystem;
import edu.xtec.jclic.project.JClicProject;
import edu.xtec.jclic.skins.Skin;
import edu.xtec.util.Domable;
import edu.xtec.util.ExtendedByteArrayInputStream;
import edu.xtec.util.FontCheck;
import edu.xtec.util.JDomUtility;
import edu.xtec.util.StreamIO;
import java.awt.Font;
import java.io.InputStream;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;


/**
 * This class stores and manages all the media components (images, sounds, animations,
 * video, MIDI files, etc.) needed to run the activities of a
 * {@link edu.xtec.jclic.project.JClicProject}. The main member of the class is a
 * {@link java.util.Vector} that stores {@link edu.xtec.jclic.bags.MediaBagElement}
 * objects. It defines also a {@link edu.xtec.jclic.bags.MediaBag.Listener} interface
 * to allow other objects to be informed about changes in the media collection.
 * @author Francesc Busquets (fbusquets@xtec.net)
 * @version 1.0
 */
public class MediaBag extends Object implements Editable, Domable, StreamIO.InputStreamProvider {
    
    /**
     * The project this <CODE>MediaBag</CODE> belongs to
     */    
    protected JClicProject project;
    /**
     * Vector containing all the {@link edu.xtec.jclic.bags.MediaBagElement} objects of
     * this <CODE>MediaBag</CODE>.
     */    
    protected Vector elements;
    protected HashSet listeners;
    
    /** Creates new MediaBag */
    public MediaBag(JClicProject project) {
        this.project=project;
        elements=new Vector(30);
        listeners=new HashSet(1);
    }
    
    public JClicProject getProject(){
        return project;
    }
    
    public static String ELEMENT_NAME="mediaBag";
    
    public Vector getElements(){
        return new Vector(elements);
    }
    
    public void clear(){
        elements.clear();
    }
    
    public Vector getElementsByName(){
        Vector v=getElements();
        Collections.sort(v, new Comparator(){
            public int compare(Object o1, Object o2){
                return ((MediaBagElement)o1).getName().compareToIgnoreCase(((MediaBagElement)o2).getName());
            }
        });
        return v;
    }
    
    public Vector getElementsByType(){
        final StringBuffer sb1=new StringBuffer(200);
        final StringBuffer sb2=new StringBuffer(200);
        Vector v=getElements();
        Collections.sort(v, new Comparator(){
            public int compare(Object o1, Object o2){
                
                sb1.setLength(0);
                MediaBagElement mbe=(MediaBagElement)o1;
                String fName=mbe.getFileName();
                String name=mbe.getName();
                int dot=fName.lastIndexOf('.');
                sb1.append(dot>0 ? fName.substring(dot) : ".zzz");
                sb1.append(name);
                
                sb2.setLength(0);
                mbe=(MediaBagElement)o2;
                fName=mbe.getFileName();
                name=mbe.getName();
                dot=fName.lastIndexOf('.');
                sb2.append(dot>0 ? fName.substring(dot) : ".zzz");
                sb2.append(name);
                
                return sb1.substring(0).compareToIgnoreCase(sb2.substring(0));
            }
        });
        return v;
    }
    
    public org.jdom.Element getJDomElement(){
        org.jdom.Element e=new org.jdom.Element(ELEMENT_NAME);
        org.jdom.Element child;
        Iterator it=getElementsByType().iterator();
        while(it.hasNext()){
            e.addContent(((MediaBagElement)it.next()).getJDomElement());
        }
        return e;
    }
    
    public void clearData(){
        Iterator it=elements.iterator();
        while(it.hasNext()){
            ((MediaBagElement)it.next()).clearData();
        }
    }
    
    public void setProperties(org.jdom.Element e, Object aux) throws Exception{
        JDomUtility.checkName(e, ELEMENT_NAME);
        Iterator it=e.getChildren(MediaBagElement.ELEMENT_NAME).iterator();
        while(it.hasNext()){
            elements.add(MediaBagElement.getMediaBagElement((org.jdom.Element)it.next()));
        }
    }
    
    public boolean addElement(MediaBagElement mbe){
        boolean result=mbe!=null && (getElement(mbe.getName())==null);
        if(result)
            elements.add(mbe);
        return result;
    }
    
    public MediaBagElement getElement(String name){
        MediaBagElement result=null;
        if(name!=null){
            for(int i=0; i<elements.size(); i++){
                MediaBagElement mbe=(MediaBagElement)elements.get(i);
                if(name.equals(mbe.getName())){
                    result=mbe;
                    break;
                }
            }
        }
        return result;
    }
    
    public MediaBagElement getElementByFileName(String fileName){
        MediaBagElement result=null;
        if(fileName!=null){
            for(int i=0; i<elements.size(); i++){
                MediaBagElement mbe=(MediaBagElement)elements.get(i);
                if(fileName.equals(mbe.getFileName())){
                    result=mbe;
                    break;
                }
            }
        }
        return result;
    }
    
    public MediaBagElement registerElement(String name, String fileName){
        MediaBagElement result=getElement(name);
        if(result==null){
            result=new MediaBagElement(FileSystem.stdFn(fileName==null ? name : fileName), null, name);
            elements.add(result);
        }
        return result;
    }
    
    public boolean removeElement(MediaBagElement mbe){
        return elements.remove(mbe);
    }
    
    public Object getMediaDataSource(String name) throws Exception{
        Object result=null;
        String normalizedName=project.getFileSystem().getCanonicalNameOf(name);
        MediaBagElement mbe=registerElement(normalizedName, null);
        if(mbe!=null){
            if(mbe.getData()!=null){
                if(mbe.getData() instanceof ExtendedByteArrayInputStream)
                    result=((ExtendedByteArrayInputStream)mbe.getData()).duplicate();
                else
                    result=mbe.getData();
            }
            else{
                result=project.getFileSystem().getMediaDataSource(mbe.getFileName());
                mbe.setData(result);
            }
        }
        return result;
    }
    
    public InputStream  getInputStream(String name) throws Exception{
        InputStream result=null;
        String normalizedName=project.getFileSystem().getCanonicalNameOf(name);
        MediaBagElement mbe=registerElement(normalizedName, null);
        if(mbe!=null){
            if(mbe.getData() instanceof ExtendedByteArrayInputStream){
                ExtendedByteArrayInputStream ebais=(ExtendedByteArrayInputStream)mbe.getData();
                mbe.setData(ebais);
                result=(ebais).duplicate();
            }
            else{
                result=project.getFileSystem().getInputStream(mbe.getFileName());
                if(result instanceof ExtendedByteArrayInputStream)
                    mbe.setData(result);
            }
        }
        return result;
    }
    
    public MediaBagElement getImageElement(String name) throws Exception{
        MediaBagElement result=null;
        String normalizedName=project.getFileSystem().getCanonicalNameOf(name);
        result=registerElement(normalizedName, null);
        if(result!=null)
            result = result.prepareImage(project.getFileSystem()) ? result : null;
            return result;
    }
    
    public void buildFonts() {
        
        // count currently empty font elements
        HashMap fonts=new HashMap();
        Iterator it=elements.iterator();
        while(it.hasNext()){
            MediaBagElement mbe=(MediaBagElement)it.next();
            String name=mbe.getName();
            if(mbe!=null && mbe.getData()==null && mbe.getFileName().endsWith(".ttf")){
                fonts.put(name, mbe);
            }
        }
        if(!fonts.isEmpty()){
            //String[] fontList=GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames();
            String[] fontList=FontCheck.getFontList(false);
            int nFontsList=fontList.length;
            it=fonts.keySet().iterator();
            while(it.hasNext()){
                String name=(String)it.next();
                int i=0;
                for(i=0; i<nFontsList; i++)
                    if(fontList[i].equalsIgnoreCase(name))
                        break;
                if(i==nFontsList){
                    MediaBagElement mbe=(MediaBagElement)fonts.get(name);
                    try{
                        mbe.setData(FontCheck.buildNewFont(mbe.getFileName(), this, name));
                    } catch(Exception ex){
                        System.err.println("Unable to create font:\n"+ex);
                    }
                    /*
                    // code adapted from java.awt.Font
                    try{
                        String tmpDir=System.getProperty("java.io.tmpdir");
                        if(tmpDir==null){
                            System.err.println("Unable to create fonts: Unknown temp dir");
                            return;
                        }
                        File fontFile=new File(tmpDir+File.separator+TMP_FONT_PREFIX+mbe.getFileName());
                        if(!fontFile.exists()){
                            StreamIO.writeStreamToFile(getInputStream(name), fontFile);
                        }
                        sun.java2d.SunGraphicsEnvironment env =
                        (sun.java2d.SunGraphicsEnvironment)GraphicsEnvironment
                        .getLocalGraphicsEnvironment();
                        String createName = env.createFont(fontFile);
                        if(createName == null) {
                            System.err.println("Unable to create font - bad font data");
                            return;
                        }
                        Font fnt=new Font(createName, Font.PLAIN, 1);
                        mbe.setData(fnt);
                    } catch(Exception ex){
                        System.err.println("Unable to create font:\n"+ex);
                    }
                     */
                }
            }
        }
    }
    
    public Skin getSkinElement(String name, PlayStation ps){
        Skin sk=null;
        String normalizedName=name;
        if(name.startsWith(Skin.INTERNAL_SKIN_PREFIX)){
            try{
                return Skin.getSkin(name, project.getFileSystem(), ps);
            }
            catch(Exception e){
                System.err.println("Error loading skin \"" + name + "\":\n"+e);
            }
        }
        else{
            normalizedName=project.getFileSystem().getCanonicalNameOf(name);
            MediaBagElement mbe=registerElement(normalizedName, null);
            if(mbe!=null){
                if(mbe.getData()==null || !(mbe.getData() instanceof Skin)){
                    try{
                        sk=Skin.getSkin(mbe.getFileName(), project.getFileSystem(), ps);
                        mbe.setData(sk);
                    }
                    catch(Exception e){
                        System.err.println("Error loading skin \"" + mbe.getFileName() + "\":\n"+e);
                    }
                }
                return (Skin)mbe.getData();
            }
        }
        return null;
    }
    
    public boolean isWaitingForImages(){
        Iterator it=elements.iterator();
        while(it.hasNext()){
            MediaBagElement mbe=(MediaBagElement)it.next();
            if(mbe!=null && mbe.isWaitingForImage())
                return true;
        }
        return false;
    }
    
    public void waitForAllImages(){
        while(isWaitingForImages()){
            try{
                Thread.currentThread().sleep(100);
            } catch(Exception ex){
                System.err.println("error waiting for images!\n"+ex);
                return;
            }
        }
    }
    
    public Editor getEditor(Editor parent) {
        return Editor.createEditor(getClass().getName()+"Editor", this, parent);
    }
    
    public interface Listener{
        public void listReferences(String type, HashMap map);
        public void listReferencesTo(String name, String type, HashMap map);
    }
    
    public void addListener(Listener lst){
        if(!listeners.contains(lst))
            listeners.add(lst);
    }
    
    public void removeListener(Listener lst){
        listeners.remove(lst);
    }
    
    public void listReferencesTo(String name, String type, HashMap map){
        Iterator it=listeners.iterator();
        while(it.hasNext())
            ((Listener)it.next()).listReferencesTo(name, type, map);
    }
    
}