/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "fakedata.h"

#include <QDebug>
#include <QTransform>

#include <cmath>

const QLatin1String fakeImageBack = QLatin1String("/usr/share/fakeCamera/image_back.jpg");
const QLatin1String fakeImageFront = QLatin1String("/usr/share/fakeCamera/image_front.jpg");
const int backFrameWidth = 1250;
const int backFrameHeight = 2000;
const int frontFrameWidth = 960;
const int frontFrameHeight = 1280;

const int backImageWidth = 1458;
const int backImageHeight = 2592;
const int frontImageWidth = 960;
const int frontImageHeight = 1280;

const int widthShake = 30;
const int heightShake = 20;

const int backVfWidth = 720;
const int backVfHeight = 1280;
const int frontVfWidth = 720;
const int frontVfHeight = 960;

FakeData::FakeData()
    : m_frameWidth(backFrameWidth),
      m_frameHeight(backFrameHeight),
      m_offset(widthShake,heightShake),
      m_movecounter(0),
      m_viewfinderSize(backVfWidth,backVfHeight),
      m_imageSize(backImageWidth,backImageHeight),
      m_zoomFactor(1.0)
{
    m_rawData.load(fakeImageBack);

    m_cameraShake.setInterval(67);
    QObject::connect(&m_cameraShake, SIGNAL(timeout()), this, SLOT(updatePosition()));
    startMove();
}

QImage FakeData::viewfinderFrame() const
{
    QImage frame = rawFrame().scaled(m_viewfinderSize);
    QTransform rot;
    rot = rot.rotate(-90);
    return frame.transformed(rot);
}

QImage FakeData::previewFrame() const
{
    return viewfinderFrame();
}

QImage FakeData::capturedImage() const
{
    QImage frame = rawFrame().scaled(m_imageSize);
    QTransform rot;
    rot = rot.rotate(-90);
    return frame.transformed(rot);
}

void FakeData::setViewfinderSize(const QSize& size)
{
    m_viewfinderSize = size;
}

const QSize &FakeData::viewfinderSize() const
{
    return m_viewfinderSize;
}

void FakeData::setZoomFactor(double zoom)
{
    if (zoom < 1.0)
        return;
    m_zoomFactor = zoom;
}

void FakeData::updatePosition()
{
    m_offset.setX(widthShake + widthShake*sin(m_movecounter));
    m_offset.setY(heightShake + heightShake*cos(m_movecounter));

    m_movecounter += 0.05;
    Q_EMIT newViewfinderFrame();
}

void FakeData::startMove()
{
    m_cameraShake.start();
}

void FakeData::stopMove()
{
    m_cameraShake.stop();
}

void FakeData::setSelectedCamera(int idx)
{
    if (idx==1) {
        m_rawData.load(fakeImageFront);
        m_frameWidth = frontFrameWidth;
        m_frameHeight = frontFrameHeight;
        m_viewfinderSize.setWidth(frontVfWidth);
        m_viewfinderSize.setHeight(frontVfHeight);
        m_imageSize.setWidth(frontImageWidth);
        m_imageSize.setHeight(frontImageHeight);
    } else {
        m_rawData.load(fakeImageBack);
        m_frameWidth = backFrameWidth;
        m_frameHeight = backFrameHeight;
        m_viewfinderSize.setWidth(backVfWidth);
        m_viewfinderSize.setHeight(backVfHeight);
        m_imageSize.setWidth(backImageWidth);
        m_imageSize.setHeight(backImageHeight);
    }
}

QImage FakeData::rawFrame() const
{
    QPoint center(500+m_offset.x()+m_frameWidth/2, m_offset.y()+m_frameHeight/2);
    int width = (double)m_frameWidth / m_zoomFactor;
    int height = (double)m_frameHeight / m_zoomFactor;
    int x = center.x() - width/2;
    int y = center.y() - height/2;
    return m_rawData.copy(x, y, width, height);
}
