"""Service units have both a public and private address.
"""
import subprocess

from twisted.internet.defer import inlineCallbacks, returnValue, succeed
from twisted.internet.threads import deferToThread
from twisted.web import client

from juju.errors import JujuError
from juju.state.environment import GlobalSettingsStateManager


@inlineCallbacks
def get_unit_address(client):
    settings = GlobalSettingsStateManager(client)
    provider_type = yield settings.get_provider_type()
    if provider_type == "ec2":
        returnValue(EC2UnitAddress())
    if provider_type in ("openstack", "openstack_s3"):
        returnValue(OpenStackUnitAddress())
    elif provider_type == "local":
        returnValue(LocalUnitAddress())
    elif provider_type == "orchestra":
        returnValue(OrchestraUnitAddress())
    elif provider_type == "dummy":
        returnValue(DummyUnitAddress())
    elif provider_type == "maas":
        returnValue(MAASUnitAddress())
    raise JujuError(
        "Unknown provider type: %r, unit addresses unknown." % provider_type)


class UnitAddress(object):

    def get_private_address(self):
        raise NotImplementedError(self.get_private_address)

    def get_public_address(self):
        raise NotImplementedError(self.get_public_address)


class DummyUnitAddress(UnitAddress):

    def get_private_address(self):
        return succeed("localhost")

    def get_public_address(self):
        return succeed("localhost")


class EC2UnitAddress(UnitAddress):

    @inlineCallbacks
    def get_private_address(self):
        content = yield client.getPage(
            "http://169.254.169.254/latest/meta-data/local-hostname")
        returnValue(content.strip())

    @inlineCallbacks
    def get_public_address(self):
        content = yield client.getPage(
            "http://169.254.169.254/latest/meta-data/public-hostname")
        returnValue(content.strip())


class OpenStackUnitAddress(UnitAddress):
    """Address determination of a service unit on an OpenStack server

    Unlike EC2 there are no promises that an instance will have a resolvable
    hostname, or for that matter a public ip address.
    """

    def _get_metadata_string(self, key):
        return client.getPage("http://169.254.169.254/%s/meta-data/%s" %
            ("2009-04-04", key))

    def get_private_address(self):
        return self._get_metadata_string("local-ipv4")

    @inlineCallbacks
    def get_public_address(self):
        address = yield self._get_metadata_string("public-ipv4")
        if not address:
            address = yield self.get_private_address()
        returnValue(address)


class LocalUnitAddress(UnitAddress):

    def get_private_address(self):
        return deferToThread(self._get_address)

    def get_public_address(self):
        return deferToThread(self._get_address)

    def _get_address(self):
        output = subprocess.check_output(["hostname", "-I"])
        return output.strip().split()[0]


class OrchestraUnitAddress(UnitAddress):

    def get_private_address(self):
        return deferToThread(self._get_address)

    def get_public_address(self):
        return deferToThread(self._get_address)

    def _get_address(self):
        output = subprocess.check_output(["hostname", "-f"])
        return output.strip()


class MAASUnitAddress(UnitAddress):

    def get_private_address(self):
        return deferToThread(self._get_address)

    def get_public_address(self):
        return deferToThread(self._get_address)

    def _get_address(self):
        output = subprocess.check_output(["hostname", "-f"])
        return output.strip()
