import os
import sys
import tempfile
import subprocess

from twisted.internet.defer import inlineCallbacks

from juju.lib.lxc.tests.test_lxc import uses_sudo
from juju.lib.testing import TestCase
from juju.tests.common import get_test_zookeeper_address
from juju.providers.local.agent import ManagedMachineAgent


class ManagedAgentTest(TestCase):

    @inlineCallbacks
    def test_managed_agent_config(self):
        subprocess_calls = []

        def intercept_args(args, **kwargs):
            self.assertEquals(args[0], "sudo")
            if args[2] == "rm":
                return 0
            subprocess_calls.append(args)
            return 0

        self.patch(subprocess, "check_call", intercept_args)

        juju_directory = self.makeDir()
        log_file = self.makeFile()
        agent = ManagedMachineAgent(
            "ns1",
            get_test_zookeeper_address(),
            juju_series="precise",
            juju_directory=juju_directory,
            log_file=log_file,
            juju_origin="lp:juju/trunk")

        try:
            os.remove(agent._service.output_path)
        except OSError:
            pass  # just make sure it's not there, so the .start()
                  # doesn't insert a spurious rm
        yield agent.start()

        start = subprocess_calls[0]
        self.assertEquals(start, [
            "sudo",
            "JUJU_MACHINE_ID=0",
            "JUJU_ORIGIN=lp:juju/trunk",
            "JUJU_ZOOKEEPER=localhost:28181",
            "PYTHONPATH=%s" % ':'.join(sys.path),
            "JUJU_HOME=%s" % juju_directory,
            "PATH=%s" % os.environ['PATH'],
            "JUJU_UNIT_NS=ns1",
            "JUJU_SERIES=precise",
            "/usr/bin/python", "-m", "juju.agents.machine",
            "--logfile", log_file,
            "--zookeeper-servers", get_test_zookeeper_address(),
            "--juju-directory", juju_directory,
            "--machine-id", "0",
            "--session-file",  "/var/run/juju/ns1-machine-agent.zksession",
            "--pidfile", agent._service._pid_path])

    @uses_sudo
    @inlineCallbacks
    def test_managed_agent_root(self):
        juju_directory = self.makeDir()
        log_file = tempfile.mktemp()

        # The pid file and log file get written as root
        def cleanup_root_file(cleanup_file):
            subprocess.check_call(
                ["sudo", "rm", "-f", cleanup_file], stderr=subprocess.STDOUT)
        self.addCleanup(cleanup_root_file, log_file)

        agent = ManagedMachineAgent(
            "test-ns", machine_id="0",
            log_file=log_file,
            juju_series="precise",
            zookeeper_hosts=get_test_zookeeper_address(),
            juju_directory=juju_directory)

        agent.agent_module = "juju.agents.dummy"
        self.assertFalse((yield agent.is_running()))
        yield agent.start()
        # Give a moment for the process to start and write its config
        yield self.sleep(0.1)
        self.assertTrue((yield agent.is_running()))

        # running start again is fine, detects the process is running
        yield agent.start()
        yield agent.stop()
        self.assertFalse((yield agent.is_running()))

        # running stop again is fine, detects the process is stopped.
        yield agent.stop()
