/* 
   message_box.c

   Utility functions for handling Motif message dialog boxes.

   Copyright (C) 2003 Tim Stadelmann

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/



/** HEADERS **/

/* autoconf header */
#if HAVE_CONFIG_H
#  include <config.h>
#endif

/* Motif headers */
#include <Xm/MessageB.h>
#include <Xm/MwmUtil.h>

/* ANSI C headers */
#include <stdio.h>
#include <stdarg.h>
#include <assert.h>

/* program headers */
#include "message_box.h"
#include "gettext.h"


/** MACROS **/

#define _(String) gettext(String)



/** TYPES **/

/* This type stores a yes/no response from the user and allows for a
   third value indicating that no response has been obtained yet.  */
typedef enum {YES, NO, UNDEFINED} Answer;


/** FUNCTION PROTOTYPES **/

static int display_message (Widget parent, unsigned char type,
			    const char *format, va_list argument_list);
static void question_callback (Widget, XtPointer, XtPointer);
static int msgbox_vprintf (Widget msg_box, const char *format,
			   va_list argument_list);
static Widget get_dialog (Widget parent, unsigned char type);



/** FUNCTIONS **/

/* Display a formatted string in a Motif error dialog box.  The dialog
   shell is a popup child of the parent widget and is created the
   first time this function is called.  The format and variable
   arguments are interpreted as by printf and friends.  */

void
display_error (Widget parent, const char *format, ...)
{
  va_list argument_list;	/* the argument list */

  /* Call display_message with the appropriate arguments for an error
     dialog.  */
  va_start (argument_list, format);
  display_message (parent, XmDIALOG_ERROR, format, argument_list);
  va_end (argument_list);
}


/* Display a formatted string in a Motif question dialog box.  The
   dialog shell is a popup child of the parent widget and is created
   the first time this function is called.  The format and variable
   arguments are interpreted as by printf and friends.  Return 1 for
   an affirmative response from the user, 0 otherwise.  */

int
display_question (Widget parent, const char *format, ...)
{
  va_list argument_list;	/* the argument list */
  int answer;			/* the answer from the user */

  /* Call display_message with the appropriate arguments for a
     question dialog.  */
  va_start (argument_list, format);
  answer = display_message (parent, XmDIALOG_QUESTION, format,
			    argument_list);
  va_end (argument_list);
  return answer;
}


/* Display a formatted string in a Motif warning dialog box.  The
   dialog shell is a popup child of the parent widget and is created
   the first time this function is called.  The format and variable
   arguments are interpreted as by printf and friends.  Return 1 for
   an affirmative response from the user, 0 otherwise.  */

int
display_warning (Widget parent, const char *format, ...)
{
  va_list argument_list;	/* the argument list */
  int answer;			/* the answer from the user */
  
  /* Call display_message with the appropriate arguments for a warning
     dialog.  */
  va_start (argument_list, format);
  answer = display_message (parent, XmDIALOG_WARNING, format,
			    argument_list);
  va_end (argument_list);
  return answer;
}


/* Display a formatted string in a Motif message dialog box of the
   indicated type.  The dialog shell is a popup child of the parent
   widget and is created the first time this function is called.  The
   format and variable arguments are interpreted as by printf and
   friends.  If applicable, return 1 for an affirmative response from
   the user, 0 otherwise.  This function implements its own message
   loop and will not return until a response from the user has been
   obtained.  */

static int
display_message (Widget parent, unsigned char type, const char *format,
		 va_list argument_list)
{
  Answer answer;		/* the response from the user */
  XtAppContext app_context;	/* the application context */
  Widget msg_box;		/* the dialog box */
  Arg args[1];
  Cardinal n;

  /* Make sure the type is valid.  */
  assert (type == XmDIALOG_ERROR ||
	  type == XmDIALOG_QUESTION ||
	  type == XmDIALOG_WARNING);

  /* Get the handle to the dialog box.  */
  msg_box = get_dialog (parent, type);
  
  /* Set the message string.  */
  msgbox_vprintf (msg_box, format, argument_list);

  /* Manage the dialog.  */
  XtManageChild (msg_box);

  /* Return immediately in the case of an error dialog.  */
  if (type == XmDIALOG_ERROR)
    return 1;

  /* Set the user data of the dialog box to the address of the local
     variable answer.  This variable can then be accessed and changed
     by the qyestion_callback.  */
  n = 0;
  XtSetArg (args[n], XmNuserData, &answer), n++;
  XtSetValues (msg_box, args, n);

  answer = UNDEFINED;
  app_context = XtWidgetToApplicationContext (msg_box);

  /* Keep processing events until an answer has been obtained from the
     user.  */
  while (answer == UNDEFINED)
    {
      XEvent event;
      XtAppNextEvent (app_context, &event);
      XtDispatchEvent (&event);
    }

  if (answer == YES)
    return 1;
  else
    return 0;
}


/* Check whether a message box with the appropriate name correspondig
   to type exists as a child of the parent widget.  If it doesn't,
   create one and register the relevant callback routine if
   applicable.  Return the handle to the message box widget found this
   way.  */

static Widget
get_dialog (Widget parent, unsigned char type)
{
  /* A pointer to the appropriate widget creation function.  */
  Widget (*create_dialog) (Widget, char *, Arg *, Cardinal);

  char *name;			/* the name of the message box widget */
  Widget msg_box;		/* the dialog box */
  XmString ok_label;		/* for creating compound strings */
  XmString cancel_label;
  XmString title;
  Arg args[5];
  Cardinal n;

  /* Set the widget name according to the requested type.  The
     wildcard is necessary because the message box is not an immediate
     child of the parent.  */
  switch (type)
    {
    case XmDIALOG_ERROR:
      name = "*error_dialog";
      break;
    case XmDIALOG_QUESTION:
      name = "*question_dialog";
      break;
    case XmDIALOG_WARNING:
      name = "*warning_dialog";
      break;
    }
  
  /* Try to find a widget with that name.  */
  msg_box = XtNameToWidget (parent, name);
  if (msg_box)
    {
      assert (XtIsSubclass (msg_box, xmMessageBoxWidgetClass));
      return msg_box;
    }  

  /* Hide the wildcard, so the name variable can later be used as the
     actual widget name.  */
  name++;

  /* Choose the appropriate dialog title and creation function for the
     requested type.  */
  switch (type)
    {
    case XmDIALOG_ERROR:
      title = XmStringCreateLocalized (_("Error"));
      create_dialog = &XmCreateErrorDialog;
      break;
    case XmDIALOG_QUESTION:
      title = XmStringCreateLocalized (_("Question"));
      create_dialog = &XmCreateQuestionDialog;
      break;
    case XmDIALOG_WARNING:
      title = XmStringCreateLocalized (_("Warning"));
      create_dialog = &XmCreateWarningDialog;
      break;
    }

  if (type == XmDIALOG_QUESTION)
    {
      ok_label = XmStringCreateLocalized (_("Yes"));
      cancel_label = XmStringCreateLocalized (_("No"));
    }
  else
    {
      ok_label = XmStringCreateLocalized (_("OK"));
      cancel_label = XmStringCreateLocalized (_("Cancel"));
    }
  
  n = 0;
  XtSetArg (args[n], XmNokLabelString, ok_label), n++;
  XtSetArg (args[n], XmNcancelLabelString, cancel_label), n++;
  XtSetArg (args[n], XmNdialogTitle, title), n++;
  XtSetArg (args[n], XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL), n++;
  XtSetArg (args[n], XmNmwmDecorations,
	    MWM_DECOR_ALL | MWM_DECOR_RESIZEH), n++;
  msg_box = (*create_dialog) (parent, name, args, n);

  XmStringFree (ok_label);
  XmStringFree (cancel_label);
  XmStringFree (title);

  XtUnmanageChild (XtNameToWidget (msg_box, "Help"));

  if (type == XmDIALOG_ERROR)
    {
      XtUnmanageChild (XtNameToWidget (msg_box, "Cancel"));
    }
  else
    {
      XtAddCallback (msg_box, XmNokCallback, question_callback, NULL);    
      XtAddCallback (msg_box, XmNcancelCallback, question_callback, NULL);    
    }
  
  return msg_box;
}


/* The callback routine for question and warning dialogs.  Retrieve
   the address of the answer variable in the relevant instance of
   display_message and set it according to the button pressed by the
   user.  */

static void
question_callback (Widget widget, XtPointer client_data, XtPointer call_data)
{
  Answer *answer;		/* the address of the answer variable */
  Arg args[1];
  Cardinal n;

  XmAnyCallbackStruct *call = (XmAnyCallbackStruct *) call_data;

  /* Get the address of the answer variable.  */
  n = 0;
  XtSetArg (args[n], XmNuserData, &answer), n++;
  XtGetValues (widget, args, n);

  /* Set it according to the callback reason.  */
  switch (call->reason)
    {
    case XmCR_OK:
      *answer = YES;
      break;
    case XmCR_CANCEL:
      *answer = NO;
      break;
    }
}


/* Create a string from the template format according to the
   conventions used by printf.  Use this string as the message string
   of the msg_box.  */

int
msgbox_printf (Widget msg_box, const char *format, ...)
{
  int n_chars;			/* the return value from msg_box_vprintf */
  va_list argument_list;	/* the argument list */

  /* Set the message string.  */
  va_start (argument_list, format);
  n_chars = msgbox_vprintf (msg_box, format, argument_list);
  va_end (argument_list);

  return n_chars;
}


/* Create a string from the template format and the argument_list
   according to the conventions used by printf.  Use this string as
   the message string of the msg_box.  */

int
msgbox_vprintf (Widget msg_box, const char *format, va_list argument_list)
{
  XmString label;		/* for creating the compound string */
  int n_chars;			/* the return value of vsnprintf */
  char *message;		/* the message */
  int message_length;		/* the actual length of the buffer */
  const int blocksize = 256;	/* the allocation block size */
  Arg args[1];
  Cardinal n;

  message = XtMalloc (blocksize);
  message_length = blocksize;

  /* There are two different conventions for the return value of
     snprintf and related functions.  Please refer to the
     documentation.  */

  for (;;)
    {
      n_chars = vsnprintf (message, message_length, format, argument_list);
      if (n_chars == -1)
	{
	  message_length += blocksize;
	}
      else if (n_chars != strlen (message))
	{
	  message_length = n_chars + 1;
	}
      else
	{
	  break;
	}
      message = XtRealloc (message, message_length);
    }

  /* Create the label string.  */
  label = XmStringCreateLocalized (message);
  XtFree (message);

  n = 0;
  XtSetArg (args[n], XmNmessageString, label), n++;
  XtSetValues (msg_box, args, n);

  XmStringFree (label);

  return n_chars;
}
