/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.io.rest.auth;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Priority;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.PreMatching;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.ext.Provider;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.auth.Authentication;
import org.openhab.core.auth.AuthenticationException;
import org.openhab.core.auth.Credentials;
import org.openhab.core.auth.User;
import org.openhab.core.auth.UserApiTokenCredentials;
import org.openhab.core.auth.UserRegistry;
import org.openhab.core.auth.UsernamePasswordCredentials;
import org.openhab.core.common.registry.RegistryChangeListener;
import org.openhab.core.config.core.ConfigParser;
import org.openhab.core.config.core.ConfigurableService;
import org.openhab.core.io.rest.JSONResponse;
import org.openhab.core.io.rest.auth.AnonymousUserSecurityContext;
import org.openhab.core.io.rest.auth.internal.ExpiringUserSecurityContextCache;
import org.openhab.core.io.rest.auth.internal.JwtHelper;
import org.openhab.core.io.rest.auth.internal.JwtSecurityContext;
import org.openhab.core.io.rest.auth.internal.UserSecurityContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsApplicationSelect;
import org.osgi.service.jaxrs.whiteboard.propertytypes.JaxrsExtension;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PreMatching
@Priority(value=1000)
@Provider
@Component(configurationPid={"org.openhab.restauth"}, property={"service.pid=org.openhab.restauth"}, service={ContainerRequestFilter.class, AuthFilter.class})
@ConfigurableService(category="system", label="API Security", description_uri="system:restauth")
@JaxrsExtension
@JaxrsApplicationSelect(value="(osgi.jaxrs.name=openhab)")
@NonNullByDefault
public class AuthFilter
implements ContainerRequestFilter {
    private final Logger logger = LoggerFactory.getLogger(AuthFilter.class);
    static final String ALT_AUTH_HEADER = "X-OPENHAB-TOKEN";
    static final String API_TOKEN_PREFIX = "oh.";
    protected static final String CONFIG_URI = "system:restauth";
    static final String CONFIG_ALLOW_BASIC_AUTH = "allowBasicAuth";
    static final String CONFIG_IMPLICIT_USER_ROLE = "implicitUserRole";
    static final String CONFIG_TRUSTED_NETWORKS = "trustedNetworks";
    static final String CONFIG_CACHE_EXPIRATION = "cacheExpiration";
    private boolean allowBasicAuth = false;
    private boolean implicitUserRole = true;
    private List<CIDR> trustedNetworks = List.of();
    private Long cacheExpiration = 6L;
    private ExpiringUserSecurityContextCache authCache = new ExpiringUserSecurityContextCache(Duration.ofHours(this.cacheExpiration).toMillis());
    private static final byte[] RANDOM_BYTES = new byte[32];
    private final JwtHelper jwtHelper;
    private final UserRegistry userRegistry;
    @Context
    @NonNullByDefault(value={})
    private HttpServletRequest servletRequest;
    private RegistryChangeListener<User> userRegistryListener = new RegistryChangeListener<User>(){

        public void added(User element) {
        }

        public void removed(User element) {
            AuthFilter.this.authCache.clear();
        }

        public void updated(User oldElement, User element) {
            AuthFilter.this.authCache.clear();
        }
    };

    @Activate
    public AuthFilter(@Reference JwtHelper jwtHelper, @Reference UserRegistry userRegistry) {
        this.jwtHelper = jwtHelper;
        this.userRegistry = userRegistry;
        new Random().nextBytes(RANDOM_BYTES);
    }

    @Activate
    protected void activate(Map<String, Object> config) {
        this.modified(config);
        this.userRegistry.addRegistryChangeListener(this.userRegistryListener);
    }

    @Modified
    protected void modified(@Nullable Map<String, Object> properties) {
        if (properties != null) {
            this.allowBasicAuth = (Boolean)ConfigParser.valueAsOrElse((Object)properties.get(CONFIG_ALLOW_BASIC_AUTH), Boolean.class, (Object)false);
            this.implicitUserRole = (Boolean)ConfigParser.valueAsOrElse((Object)properties.get(CONFIG_IMPLICIT_USER_ROLE), Boolean.class, (Object)true);
            this.trustedNetworks = this.parseTrustedNetworks((String)ConfigParser.valueAsOrElse((Object)properties.get(CONFIG_TRUSTED_NETWORKS), String.class, (Object)""));
            try {
                this.cacheExpiration = (Long)ConfigParser.valueAsOrElse((Object)properties.get(CONFIG_CACHE_EXPIRATION), Long.class, (Object)6L);
            }
            catch (NumberFormatException e) {
                this.logger.warn("Ignoring invalid configuration value '{}' for cacheExpiration parameter.", properties.get(CONFIG_CACHE_EXPIRATION));
            }
            this.authCache.clear();
        }
    }

    @Deactivate
    protected void deactivate() {
        this.userRegistry.removeRegistryChangeListener(this.userRegistryListener);
    }

    private @Nullable String getCacheKey(String credentials) {
        if (this.cacheExpiration == 0L) {
            return null;
        }
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            md.update(RANDOM_BYTES);
            return new String(md.digest(credentials.getBytes()));
        }
        catch (NoSuchAlgorithmException e) {
            this.logger.warn("SHA-256 is not available. Cache for basic auth disabled!");
            return null;
        }
    }

    private SecurityContext authenticateBearerToken(String token) throws AuthenticationException {
        if (token.startsWith(API_TOKEN_PREFIX)) {
            UserApiTokenCredentials credentials = new UserApiTokenCredentials(token);
            Authentication auth = this.userRegistry.authenticate((Credentials)credentials);
            User user = (User)this.userRegistry.get((Object)auth.getUsername());
            if (user == null) {
                throw new AuthenticationException("User not found in registry");
            }
            return new UserSecurityContext(user, auth, "ApiToken");
        }
        Authentication auth = this.jwtHelper.verifyAndParseJwtAccessToken(token);
        return new JwtSecurityContext(auth);
    }

    private SecurityContext authenticateBasicAuth(String credentialString) throws AuthenticationException {
        UserSecurityContext cachedValue;
        String cacheKey = this.getCacheKey(credentialString);
        if (cacheKey != null && (cachedValue = this.authCache.get(cacheKey)) != null) {
            return cachedValue;
        }
        String[] decodedCredentials = new String(Base64.getDecoder().decode(credentialString), StandardCharsets.UTF_8).split(":");
        if (decodedCredentials.length != 2) {
            throw new AuthenticationException("Invalid Basic authentication credential format");
        }
        UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(decodedCredentials[0], decodedCredentials[1]);
        Authentication auth = this.userRegistry.authenticate((Credentials)credentials);
        User user = (User)this.userRegistry.get((Object)auth.getUsername());
        if (user == null) {
            throw new AuthenticationException("User not found in registry");
        }
        UserSecurityContext context = new UserSecurityContext(user, auth, "Basic");
        if (cacheKey != null) {
            this.authCache.put(cacheKey, context);
        }
        return context;
    }

    public void filter(@Nullable ContainerRequestContext requestContext) throws IOException {
        if (requestContext != null) {
            try {
                SecurityContext sc = this.getSecurityContext(this.servletRequest, false);
                if (sc != null) {
                    requestContext.setSecurityContext(sc);
                }
            }
            catch (AuthenticationException e) {
                this.logger.warn("Unauthorized API request from {}: {}", (Object)this.getClientIp(this.servletRequest), (Object)e.getMessage());
                requestContext.abortWith(JSONResponse.createErrorResponse((Response.StatusType)Response.Status.UNAUTHORIZED, (String)"Invalid credentials"));
            }
        }
    }

    public @Nullable SecurityContext getSecurityContext(@Nullable String bearerToken) throws AuthenticationException {
        if (bearerToken == null) {
            return null;
        }
        return this.authenticateBearerToken(bearerToken);
    }

    public @Nullable SecurityContext getSecurityContext(HttpServletRequest request, boolean allowQueryToken) throws AuthenticationException, IOException {
        Map parameterMap;
        String[] accessToken;
        String altTokenHeader = request.getHeader(ALT_AUTH_HEADER);
        if (altTokenHeader != null) {
            return this.authenticateBearerToken(altTokenHeader);
        }
        String authHeader = request.getHeader("Authorization");
        String authType = null;
        String authValue = null;
        boolean authFromQuery = false;
        if (authHeader != null) {
            String[] authParts = authHeader.split(" ");
            if (authParts.length == 2) {
                authType = authParts[0];
                authValue = authParts[1];
            }
        } else if (allowQueryToken && (accessToken = (String[])(parameterMap = request.getParameterMap()).get("accessToken")) != null && accessToken.length > 0) {
            authValue = accessToken[0];
            authFromQuery = true;
        }
        if (authValue != null) {
            if (authFromQuery) {
                try {
                    return this.authenticateBearerToken(authValue);
                }
                catch (AuthenticationException e) {
                    if (this.allowBasicAuth) {
                        return this.authenticateBasicAuth(authValue);
                    }
                }
            } else {
                if ("Bearer".equalsIgnoreCase(authType)) {
                    return this.authenticateBearerToken(authValue);
                }
                if ("Basic".equalsIgnoreCase(authType)) {
                    String[] decodedCredentials = new String(Base64.getDecoder().decode(authValue), "UTF-8").split(":");
                    if (decodedCredentials.length > 2) {
                        throw new AuthenticationException("Invalid Basic authentication credential format");
                    }
                    switch (decodedCredentials.length) {
                        case 1: {
                            return this.authenticateBearerToken(decodedCredentials[0]);
                        }
                        case 2: {
                            if (!this.allowBasicAuth) {
                                throw new AuthenticationException("Basic authentication with username/password is not allowed");
                            }
                            return this.authenticateBasicAuth(authValue);
                        }
                    }
                }
            }
        } else if (this.isImplicitUserRole(request)) {
            return new AnonymousUserSecurityContext();
        }
        return null;
    }

    private boolean isImplicitUserRole(HttpServletRequest request) {
        if (this.implicitUserRole) {
            return true;
        }
        try {
            byte[] clientAddress = InetAddress.getByName(this.getClientIp(request)).getAddress();
            return this.trustedNetworks.stream().anyMatch(networkCIDR -> networkCIDR.isInRange(clientAddress));
        }
        catch (IOException e) {
            this.logger.debug("Error validating trusted networks: {}", (Object)e.getMessage());
            return false;
        }
    }

    private List<CIDR> parseTrustedNetworks(String value) {
        ArrayList<CIDR> cidrList = new ArrayList<CIDR>();
        String[] stringArray = value.split(",");
        int n = stringArray.length;
        int n2 = 0;
        while (n2 < n) {
            String cidrString = stringArray[n2];
            try {
                if (!cidrString.isBlank()) {
                    cidrList.add(new CIDR(cidrString.trim()));
                }
            }
            catch (UnknownHostException e) {
                this.logger.warn("Error parsing trusted network cidr: {}", (Object)cidrString);
            }
            ++n2;
        }
        return cidrList;
    }

    private String getClientIp(HttpServletRequest request) throws UnknownHostException {
        String ipForwarded = Objects.requireNonNullElse(request.getHeader("x-forwarded-for"), "");
        String clientIp = ipForwarded.split(",")[0];
        return clientIp.isBlank() ? request.getRemoteAddr() : clientIp;
    }

    private static class CIDR {
        private static final Pattern CIDR_PATTERN = Pattern.compile("(?<networkAddress>.*?)/(?<prefixLength>\\d+)");
        private final byte[] networkBytes;
        private final int prefix;

        public CIDR(String cidr) throws UnknownHostException {
            Matcher m = CIDR_PATTERN.matcher(cidr);
            if (!m.matches()) {
                throw new UnknownHostException();
            }
            this.prefix = Integer.parseInt(m.group("prefixLength"));
            this.networkBytes = InetAddress.getByName(m.group("networkAddress")).getAddress();
        }

        public boolean isInRange(byte[] address) {
            if (this.networkBytes.length != address.length) {
                return false;
            }
            int p = this.prefix;
            int i = 0;
            while (p > 8) {
                if (this.networkBytes[i] != address[i]) {
                    return false;
                }
                ++i;
                p -= 8;
            }
            int m = 65280 >> p & 0xFF;
            return (this.networkBytes[i] & m) == (address[i] & m);
        }
    }
}

