/**
 * This file is a part of VideoCut package.
 * ---------------------------------------------------------------------- 
 * Copyright (C) 2007-2008 troorl
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * ---------------------------------------------------------------------- 
 *
 * @author troorl <troorl@gmail.com>
 */

#include "TPreviewPanel.h"

TPreviewPanel::TPreviewPanel(QWidget* parent, Qt::WindowFlags)
: QGraphicsView(parent)
{
	isOpen = false;
	
	scene = new QGraphicsScene(this);
	
	backgroundRect = new TBackground();

	infoTextHeight = 0;
    countItems = 0;
    offset = 20;
    h = 100;
    this->space = 20;
	style = 0;
	colorBackground = QColor(255, 255, 255, 255);
	this->currentZoom = 1;

	this->menuInit();
    this->setVisible(true);
	
	this->tempSceneBorn();
	this->setScene(tempScene);
	resultPix = new QPixmap();
	
	graphItems = new QList<TGraphItem*>();
	listItems = new QList<frameStruct*>();
	//autoredraw = false;
	this->setDragMode(QGraphicsView::ScrollHandDrag);
	
	//Logo svg
	logo = new TLogo();
	logo_added = false;
}

void TPreviewPanel::tempSceneBorn()
{
	tempScene = new QGraphicsScene(this);
	
	QRadialGradient gradient(QPoint(70, 70), 170, QPoint(70, 70));
	gradient.setColorAt(0.05, Qt::white);
	//gradient.setColorAt(0.8, Qt::green);
	gradient.setColorAt(1, Qt::black);
	
	tempScene->setBackgroundBrush(QBrush(gradient));
	QGraphicsPixmapItem *logoItem = new QGraphicsPixmapItem(QPixmap(":/img/logo.svg"));
	tempScene->addItem(logoItem); 
}

void TPreviewPanel::menuInit()
{
	zoomInAct = new QAction(tr("Zoom in"), this);
	zoomInAct->setIcon(QIcon(":/img/zoomin.png"));
	this->addAction(zoomInAct);
	connect(zoomInAct, SIGNAL(triggered()), this, SLOT(S_ZoomInOut()));
	
	zoomOutAct = new QAction(tr("Zoom out"), this);
	zoomOutAct->setIcon(QIcon(":/img/zoomout.png"));
	this->addAction(zoomOutAct);
	connect(zoomOutAct, SIGNAL(triggered()), this, SLOT(S_ZoomInOut()));
	
	this->setContextMenuPolicy(Qt::ActionsContextMenu);
}

void TPreviewPanel::setAtributes(int offset, qreal h_proc, int space, int hor_count, \
								QFont fontMetaInfo, QColor colorInfoText, QColor colorInfoInfo, \
								QColor colorBackground)
{
    this->countItems = countItems;
    this->offset = offset;
    this->h_proc = h_proc;
    this->space = space;
    this->hor_count = hor_count;
	this->fontMetaInfo = fontMetaInfo;
	this->colorInfoText = colorInfoText;
	this->colorInfoInfo = colorInfoInfo;
	this->colorBackground = colorBackground;
}

void TPreviewPanel::addText(QString infoText, QString infoInfo)
{
	this->infoText = new QGraphicsTextItem(infoText);
	fontMetaInfo.setBold(true);
	this->infoText->setFont(fontMetaInfo);
	this->infoText->setDefaultTextColor(colorInfoText);
	
	infoTextHeight = (int)this->infoText->boundingRect().height();
	scene->addItem(this->infoText);
	this->infoText->setPos(offset, offset);
	this->infoText->setZValue(1);
	
	this->infoInfo = new QGraphicsTextItem(infoInfo);
	fontMetaInfo.setBold(false);
	this->infoInfo->setFont(fontMetaInfo);
	this->infoInfo->setDefaultTextColor(colorInfoInfo);
	
	scene->addItem(this->infoInfo);
	this->infoInfo->setPos(this->infoText->boundingRect().width() + 20, offset);
	this->infoInfo->setZValue(1);
}

void TPreviewPanel::addLogo()
{
	//set logo
	if(!logo_added)
	{
		scene->addItem(logo);
		logo->setPos(0, 0);
		logo_added = true;
	}
}

void TPreviewPanel::removeLogo()
{
	if(logo_added)
	{
		scene->removeItem(logo);
		logo_added = false;
	}
}

void TPreviewPanel::createItems()
{	
	for(int i = 0; i < listItems->count(); i++)
	{
		TGraphItem *item = new TGraphItem(listItems->at(i)->img, style, h_proc, listItems->at(i)->currTime);
		item->setZValue(1);
		*graphItems << item;
		scene->addItem(graphItems->at(i));
	}
	isOpen = true;
}

void TPreviewPanel::setItemsStyle(int style)
{
	for(int i = 0; i < graphItems->count(); i++)
	{
		graphItems->at(i)->setStyle(style);
	}
	this->style = style;
}

void TPreviewPanel::adjustItems()
{
	backgroundRect->setSize(&QRectF(0, 0, 1, 1));
	logo->setSize(QRect(0, 0, 1, 1));
	
	// adjust meta text
	this->infoText->setPos(offset, offset);
	this->infoInfo->setPos(infoText->pos().x() + infoText->boundingRect().width() + 10, offset);
	
	// djust pixmap items
	int inc = offset;
	int vertical = infoTextHeight + offset;
	int j = 0;
	
	for(int i = 0; i < graphItems->count(); i++)
	{	
		if(j == hor_count)
		{
			j = 0;
			inc = offset;
			vertical = (int)(vertical + graphItems->at(i)->boundingRect().height() + space);
		}
		
		graphItems->at(i)->setPos(inc, vertical);
	
		inc = (int)(inc + graphItems->at(i)->boundingRect().width() + space);
		j++;
	}
	
	//adjust background
	scene->setSceneRect(0, 0, scene->itemsBoundingRect().width() + offset, scene->itemsBoundingRect().height() + offset);
	QRectF *bkRect = new QRectF(scene->sceneRect());
	backgroundRect->setSize(bkRect);
	
	//adjust logo
	if(logo_added)
	{
		logo->setSize(QRect(offset, offset, backgroundRect->boundingRect().width() - offset * 2, infoText->boundingRect().height() - offset));
	}
}

void TPreviewPanel::prepare()
{
	QRectF *bkRect = new QRectF(0, 0, scene->itemsBoundingRect().width() + offset * 2, scene->itemsBoundingRect().height() + offset * 2);
	backgroundRect->setSize(bkRect);
	backgroundRect->setColor(colorBackground);
	scene->addItem(backgroundRect);
	backgroundRect->setZValue(-1);
	
	// set background
	QPixmap *tempPixmap = new QPixmap(":img/background.png", "PNG");
	QBrush *tempBrush = new QBrush(QColor(55, 55, 55, 255));
	tempBrush->setTexture(*tempPixmap);
	this->setBackgroundBrush(*tempBrush);
	this->setCacheMode(QGraphicsView::CacheBackground);
	this->setAlignment(Qt::AlignCenter);
}

void TPreviewPanel::setSceneAfter()
{
	// set scene
	this->setScene(scene);
}

void TPreviewPanel::setList(QList<frameStruct*> *list)
{
	listItems = list;
}

int TPreviewPanel::zoomInOut(bool in)
{
	if(in) // then zoom
	{
		this->scale(2, 2);
		currentZoom = currentZoom * 2;
		return (int)(currentZoom * 100);
	}
	else // then unzoom
	{
		this->scale(0.5, 0.5);
		currentZoom = currentZoom * 0.5;
		return (int)(currentZoom * 100);
	}
}

void TPreviewPanel::S_ZoomInOut()
{
	if(sender() == zoomInAct)
	{
		zoomInOut(true);
	}
	if(sender() == zoomOutAct)
	{
		zoomInOut(false);
	}
}

TPreviewPanel::~TPreviewPanel(void)
{

}

bool TPreviewPanel::saveToFile(QString path)
{
	QImage image((int)scene->width(), (int)scene->height(), QImage::Format_ARGB32);
	image.fill(0);
	QPainter painter(&image);
	scene->render(&painter);
	
	if(image.save(path, 0, quality)) // wrote or not
		return true;
	
	else
		return false;
}

void TPreviewPanel::print(QPrinter &printer)
{
	QPainter painter(&printer);
	scene->render(&painter);
}

void TPreviewPanel::removeAll()
{
	qDeleteAll(*graphItems);
	graphItems->clear();
	
	graphItems = NULL;
	listItems = NULL;
	graphItems = new QList<TGraphItem*>();
	listItems = new QList<frameStruct*>();
	
	scene = NULL;
	scene = new QGraphicsScene(this);
	
	isOpen = false;
	this->setScene(tempScene);
}

void TPreviewPanel::deleteItem(int index)
{	
	if(graphItems->count() > 0)
	{
		scene->removeItem(graphItems->at(index));
		graphItems->removeAt(index);
		adjustItems();
	}
}

void TPreviewPanel::setItemTextPos(uint textPos)
{
	for(int i = 0; i < graphItems->count(); i++)
	{
		graphItems->at(i)->setItemTextPos(textPos);
	}
}

void TPreviewPanel::autoRedraw(bool autoRedraw)
{
	this->autoredraw = autoRedraw;
}

void TPreviewPanel::setOffset(uint offset)
{
	this->offset = offset;
	
	if(autoredraw)
		this->adjustItems();
}

void TPreviewPanel::setOffsetValue(uint offset)
{
	this->offset = offset;
}

void TPreviewPanel::setSpace(uint space)
{
	this->space = space;
	
	if(autoredraw)
		this->adjustItems();
}

void TPreviewPanel::setSpaceValue(uint space)
{
	this->space = space;
}

void TPreviewPanel::setScale(qreal scale)
{
	this->h_proc = scale;
	
	for(int i = 0; i < graphItems->count(); i++)
	{
		graphItems->at(i)->scale(scale);
	}
}

void TPreviewPanel::setScaleValue(qreal h_proc)
{
	this->h_proc = h_proc;
}

void TPreviewPanel::setQuality(int quality)
{
	this->quality = quality;
}

void TPreviewPanel::setBackgroundColor(QColor &color)
{
	backgroundRect->setColor(color);
}

void TPreviewPanel::setBackgroundColorValue(QColor colorBackground)
{
	this->colorBackground = colorBackground;
}

void TPreviewPanel::setCountHor(uint hor_count)
{
	this->hor_count = hor_count;
}

void TPreviewPanel::setfontMetaInfo(QFont fontMetaInfo)
{
	this->fontMetaInfo = fontMetaInfo;
}

void TPreviewPanel::setColorInfoText(QColor colorInfoText)
{
	this->colorInfoText = colorInfoText;
}

void TPreviewPanel::setColorInfoInfo(QColor colorInfoInfo)
{
	this->colorInfoInfo = colorInfoInfo;
}

uint TPreviewPanel::getOffset()
{
	return offset;
}

uint TPreviewPanel::getSpace()
{
	return space;
}

qreal TPreviewPanel::getScale()
{
	return h_proc;
}

int TPreviewPanel::getItemStyle()
{
	return style;
}

QColor TPreviewPanel::getBackgroundColor()
{
	return backgroundRect->getColor();
}
