/************************************************************************
 **
 **  @file   vlayoutdetail.h
 **  @author Roman Telezhynskyi <dismine(at)gmail.com>
 **  @date   2 1, 2015
 **
 **  @brief
 **  @copyright
 **  This source code is part of the Valentina project, a pattern making
 **  program, whose allow create and modeling patterns of clothing.
 **  Copyright (C) 2013-2015 Valentina project
 **  <https://gitlab.com/smart-pattern/valentina> All Rights Reserved.
 **
 **  Valentina is free software: you can redistribute it and/or modify
 **  it under the terms of the GNU General Public License as published by
 **  the Free Software Foundation, either version 3 of the License, or
 **  (at your option) any later version.
 **
 **  Valentina is distributed in the hope that it will be useful,
 **  but WITHOUT ANY WARRANTY; without even the implied warranty of
 **  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 **  GNU General Public License for more details.
 **
 **  You should have received a copy of the GNU General Public License
 **  along with Valentina.  If not, see <http://www.gnu.org/licenses/>.
 **
 *************************************************************************/

#ifndef VLAYOUTDETAIL_H
#define VLAYOUTDETAIL_H

#include <qcompilerdetection.h>
#include <QDate>
#include <QLineF>
#include <QMatrix>
#include <QPointF>
#include <QRectF>
#include <QSharedDataPointer>
#include <QString>
#include <QTypeInfo>
#include <QVector>
#include <QtGlobal>
#include <QCoreApplication>

#include "vabstractpiece.h"
#include "../vmisc/typedef.h"
#include "../vpatterndb/floatItemData/floatitemdef.h"

class VLayoutPieceData;
class VLayoutPiecePath;
class QGraphicsItem;
class QGraphicsPathItem;
class VTextManager;
class VPiece;
class VPieceLabelData;
class VAbstractPattern;
class VPatternLabelData;

QT_WARNING_PUSH
QT_WARNING_DISABLE_GCC("-Wsuggest-final-types")
QT_WARNING_DISABLE_GCC("-Wsuggest-final-methods")

class VLayoutPiece :public VAbstractPiece
{
    Q_DECLARE_TR_FUNCTIONS(VLayoutPiece) // NOLINT
public:
    VLayoutPiece();
    VLayoutPiece(const VLayoutPiece &detail);

    virtual ~VLayoutPiece() override;

    VLayoutPiece &operator=(const VLayoutPiece &detail);
#ifdef Q_COMPILER_RVALUE_REFS
    VLayoutPiece(VLayoutPiece &&detail) Q_DECL_NOTHROW;
    VLayoutPiece &operator=(VLayoutPiece &&detail) Q_DECL_NOTHROW;
#endif

    static VLayoutPiece Create(const VPiece &piece, vidtype id, const VContainer *pattern);

    virtual auto GetUniqueID() const -> QString override;

    QVector<QPointF> GetMappedContourPoints() const;
    QVector<QPointF> GetContourPoints() const;
    void SetCountourPoints(const QVector<QPointF> &points, bool hideMainPath = false);

    QVector<QPointF> GetMappedSeamAllowancePoints() const;
    QVector<QPointF> GetSeamAllowancePoints() const;
    void SetSeamAllowancePoints(const QVector<QPointF> &points, bool seamAllowance = true,
                                bool seamAllowanceBuiltIn = false);

    QVector<QPointF> GetMappedLayoutAllowancePoints() const;
    QVector<QPointF> GetLayoutAllowancePoints() const;
    void SetLayoutAllowancePoints();

    QVector<QPointF> GetMappedExternalContourPoints() const;
    QVector<QPointF> GetExternalContourPoints() const;

    QVector<VLayoutPassmark> GetMappedPassmarks() const;
    QVector<VLayoutPassmark> GetPassmarks() const;
    void SetPassmarks(const QVector<VLayoutPassmark> &passmarks);

    QVector<VLayoutPlaceLabel> GetMappedPlaceLabels() const;
    QVector<VLayoutPlaceLabel> GetPlaceLabels() const;
    void SetPlaceLabels(const QVector<VLayoutPlaceLabel> &labels);

    QVector<QVector<QPointF>> MappedInternalPathsForCut(bool cut) const;
    QVector<VLayoutPiecePath> GetInternalPaths() const;
    void SetInternalPaths(const QVector<VLayoutPiecePath> &internalPaths);

    QPointF GetPieceTextPosition() const;
    QStringList GetPieceText() const;
    void SetPieceText(const QString &qsName, const VPieceLabelData& data, const QFont& font, const VContainer *pattern);

    QPointF GetPatternTextPosition() const;
    QStringList GetPatternText() const;
    void SetPatternInfo(VAbstractPattern *pDoc, const VPatternLabelData& geom, const QFont& font,
                        const VContainer *pattern);

    void SetGrainline(const VGrainlineData& geom, const VContainer *pattern);
    QVector<QPointF> GetMappedGrainline() const;
    QVector<QPointF> GetGrainline() const;
    bool  IsGrainlineEnabled() const;
    qreal GrainlineAngle() const;
    GrainlineArrowDirection GrainlineArrowType() const;

    QTransform GetMatrix() const;
    void    SetMatrix(const QTransform &matrix);

    qreal GetLayoutWidth() const;
    void  SetLayoutWidth(qreal value);

    quint16 GetQuantity() const;
    void    SetQuantity(quint16 value);

    vidtype GetId() const;
    void    SetId(vidtype id);

    bool IsMirror() const;
    void SetMirror(bool value);

    void SetGradationId(const QString &id);
    auto GetGradationId() const -> QString;

    auto GetXScale() const -> qreal;
    void SetXScale(qreal xs);

    auto GetYScale() const -> qreal;
    void SetYScale(qreal ys);

    void Translate(const QPointF &p);
    void Translate(qreal dx, qreal dy);
    void Scale(qreal sx, qreal sy);
    void Rotate(const QPointF &originPoint, qreal degrees);
    void Mirror(const QLineF &edge);
    void Mirror();

    int    DetailEdgesCount() const;
    int    LayoutEdgesCount() const;

    QLineF LayoutEdge(int i) const;
    int    LayoutEdgeByPoint(const QPointF &p1) const;

    QRectF MappedDetailBoundingRect() const;
    QRectF DetailBoundingRect() const;
    QRectF MappedLayoutBoundingRect() const;
    qreal  Diagonal() const;

    static QRectF BoundingRect(QVector<QPointF> points);

    bool isNull() const;
    qint64 Square() const;

    QPainterPath MappedContourPath() const;
    QPainterPath ContourPath() const;
    QPainterPath MappedLayoutAllowancePath() const;

    void DrawMiniature(QPainter &painter) const;

    Q_REQUIRED_RESULT QGraphicsItem *GetItem(bool textAsPaths) const;

    bool IsLayoutAllowanceValid() const;

    qreal BiggestEdge() const;

    friend QDataStream& operator<< (QDataStream& dataStream, const VLayoutPiece& piece);
    friend QDataStream& operator>> (QDataStream& dataStream, VLayoutPiece& piece);

protected:
    void SetGrainlineEnabled(bool enabled);
    void SetGrainlineAngle(qreal angle);
    void SetGrainlineArrowType(GrainlineArrowDirection type);
    void SetGrainlinePoints(const QVector<QPointF> &points);

    auto GetPieceLabelRect() const -> QVector<QPointF>;
    void SetPieceLabelRect(const QVector<QPointF> &rect);

    auto GetPieceLabelData() const ->VTextManager;
    void SetPieceLabelData(const VTextManager &data);

    auto GetPatternLabelRect() const -> QVector<QPointF>;
    void SetPatternLabelRect(const QVector<QPointF> &rect);

    auto GetPatternLabelData() const ->VTextManager;
    void SetPatternLabelData(const VTextManager &data);

private:
    QSharedDataPointer<VLayoutPieceData> d;

    QVector<QPointF> DetailPath() const;

    Q_REQUIRED_RESULT QGraphicsPathItem *GetMainItem() const;
    Q_REQUIRED_RESULT QGraphicsPathItem *GetMainPathItem() const;

    void CreateLabelStrings(QGraphicsItem *parent, const QVector<QPointF> &labelShape, const VTextManager &tm,
                            bool textAsPaths) const;
    void CreateGrainlineItem(QGraphicsItem *parent) const;

    template <class T>
    QVector<T> Map(QVector<T> points) const;

    QLineF Edge(const QVector<QPointF> &path, int i) const;
    int    EdgeByPoint(const QVector<QPointF> &path, const QPointF &p1) const;
};

QT_WARNING_POP

Q_DECLARE_TYPEINFO(VLayoutPiece, Q_MOVABLE_TYPE); // NOLINT

#endif // VLAYOUTDETAIL_H
