---
title: Turns
---

# jtcrouter.py
The **J**unction**T**urn**C**ountRouter generates vehicle routes from turn-count data.
It does so by converting the turn counts into into flows and turn-ratio files that are suitable as [jtrrouter](../jtrrouter.md) input.
Then it calls jtrrouter in the background. The turn

```
python tools/jtcrouter.py -n <net-file> -t <turn-file> -o <output-file>
```
There are three basic styles of converting turn-counts to routes:

- Flows start at all turn-count locations in the network but end when reaching the next count location
- Flows start at all turn-count locations in the network and are discounted when reaching the next count location (**--discount-sources**)
- Flows only start on the fringe of the network (**--fringe-flows**)

## Turn count data format
The turn-count data must be provided in the format:

```xml
<data>
  <interval id="generated" begin="0.0" end="99.0">
    <edgeRelation from="-58.121.42" to="64" count="1"/>
    <edgeRelation from="-58.121.42" to="-31" count="3"/>
    <edgeRelation from="45" to="-68" count="3"/>
    <edgeRelation from="-31.80.00" to="31" count="1"/>
    <edgeRelation from="-31.80.00" to="37" count="1"/>
    <edgeRelation from="-31.80.00" to="-23" count="13"/>
    <edgeRelation from="-92.180.00" to="-60" count="1"/>
  </interval>
</data>
```

# routeSampler.py
The script generates routes from *any combination* of turn-count data, edge-count and even origin-destination-count data. It requires a route file as input that defines possible routes. Routes are sampled (heuristically) from the input so that
the resulting traffic fulfills the counting data.

## turn counts
The turn-count data format is the same as as described [above](#turn_count_data_format).
```
python tools/routeSampler.py -r <input-route-file> --turn-files <turn-files> -o <output-file>
```
The attributes for reading the counts from the turn-data file can be set with option **--turn-attribute** (default 'count')

## edge counts
In addition to loading a turn-count file, routeSampler can also load an [edgeData
file](../Simulation/Output/Lane-_or_Edge-based_Traffic_Measures.md) using option **--edgedata-files**.

```
python tools/routeSampler.py -r <input-route-file> --edgedata-files <edgedata-files> -o <output-file>
```

The attributes for reading the counts from edge-data file can be set with option **--edgedata-attribute** (default 'entered')


## Obtaining counting data files

When using routeSampler as a replacement for [dfrouter](../dfrouter.md) or [flowrouter.py](Detector.md#flowrouterpy), the flow input files can be converted to edgeData files with the tool [edgeDataFromFlow.py](Detector.md#edgedatafromflowpy)

For smaller scenarions it is often feasible to define edgeData files with [netedit edgeData mode (in the 'Data' supermode)](../Netedit/elementsData.md). Turn files can also be created with netedit using the edgeRelation mode (also part of the 'Data' supermode).

In other cases it is necessary to write custom code for converting counting data into the required format.

Further source for edgeData files are listed at the [visualizing edge related data](../sumo-gui.md#visualizing_edge-related_data) page.

## Obtaining initial routes
Routes generated by randomTrips.py (**--route-output**) can be a good input source. The following randomTrips.py options may be helpful:

- **--fringe-factor**: setting a high value will generated lots of through-traffic which is plausible for small networks
- **--min-distance**: restricting short routes increases the chance that routes passing multiple counting locations are generated
- **--speed-exponent**, **--lanes**: both options can be used to increase routes starting and ending on "important" roads. This combines well with the **--fringe-factor**-option to generate many routes that enter and leave the network on major roads.

Example:
```
python tools/randomTrips.py -n <input-net-file> -r sampleRoutes.rou.xml
python tools/routeSampler.py -r sampleRoutes.rou.xml --edgedata-files <edgedata-files> -o <output-file>
```

!!! note
    departure times in route files are ignored and only the `<route>`-elements are used. Route with named routes but without vehicles may also be used.

## Generalized route restrictions
By default, the input options --edgedata-files and --turn-files allow restricting counts for single edges and pairs of consecutive edges. 

To define count restrictions for longer consecutive edge sequences the optional 'via' attribute can be used for `<edgeRelation>` elements:

```xml
<edgeRelation from="A" to="D" via="B C" count="42"/>
```

To define count restrictions on non-consecutive edges the option **--turn-max-gap <INT>** can be used. Example:
When setting option **--turn-max-gap 2**, the edgeRelation `<edgeRelation from="A" to="D" .../>` would apply to routes containing "A B", "A X D" or "A X Y D" but not "A X Y Z D".
        
## Origin-Destination restrictions
When loading an edgeRelation file with the option **--od-files**, origin-destination counts will be added.
This can be used to combine (edge-based) OD-relations with other counting data.
        
## Output Styles
By default, routeSampler will generate individual vehicles with embedded routes. This can be changed with the following options (which can also be combined):

- **--write-route-ids**: write named routes and let vehicles reference the route via it's id
- **--wroute-route-distribution STR**: put all routes into a route distribution (with appropriate route probabilities) and let a all vehicles reference this distribution (Simulation counts will then vary due to sampling from the distribution)
- **--write-flows number**: write `<flow>`-definitions instead of vehicles. The exact number of flow vehicles will be spaced evenly between the earliest and latest vehicle departures that would have been generated by default
- **--write-flows probability**: write `<flow>`-definitions instead of vehicles. Flows will be defined with attribute 'probability' so that the expected number of vehicles is equal to the number of vehicles that would have been generated by default but the specific number will vary due to sampling effects
- **--pedestrians**: write persons and personFlows with walks instead of vehicles and flows
- **--prefix**: prefix vehicle- and route-IDs with the given string. This is needed to combine output of multiple routeSampler runs (i.e. for different modes of traffic) into a single simulation. Otherwise there would be errors regarding duplicate IDs.

## Vehicle attributes
With the option **--attributes** {{DT_STR}}, additional parameters can be given to the generated
vehicles (note, usage of the quoting characters).

```
python tools/routeSampler.py -n input_net.net.xml -r candidate.rou.xml -o out.rou.xml -d data.xml
  --attributes="departLane=\"best\" departSpeed=\"max\" departPos=\"random\""
```

The above attriutes would make the vehicles be distributed randomly on their
starting edges and inserted with high speed on a reasonable lane.

!!! note
    Quoting of trip attributes on Linux may also use the style **--attributes 'departLane="best" departSpeed="max" departPos="random"'**

## Multiple vehicle types
  
To distinguish vehicles of different types, routeSampler may be run multiple times with different attributes. Note, that it is also necessary to set the option **--prefix** to prevent duplicate ids. The example below creates traffic consisting of cars and trucks using two edgedata files with distinct count values (stored in the default attribute 'entered').

```
python tools/routeSampler.py --attributes="type=\"car\"" --edgedata-files carcounts.xml --prefix c -o cars.rou.xml -n net.net.xml -r candidate.rou.xml
python tools/routeSampler.py --attributes="type=\"heavy\"" --edgedata-files truckcounts.xml --prefix t -o trucks.rou.xml  -n net.net.xml -r candidate.rou.xml
```
  
Alternatively, the count values might also be stored in different attributes of the same file (i.e. 'count1', 'count2'):
  
```
python tools/routeSampler.py --attributes="type=\"car\"" --edgedata-files counts.xml --edgedata-attribute count1 --prefix c -o cars.rou.xml  -n net.net.xml -r candidate.rou.xml
python tools/routeSampler.py --attributes="type=\"heavy\"" --edgedata-files counts.xml --edgedata-attribute count2 --prefix t -o trucks.rou.xml  -n net.net.xml -r candidate.rou.xml
```
   
When running the simulation, the types 'car' and 'heavy' (previously set as vehicle attributes), must be defined in an additional file which could look like the following example (types.add.xml):
  
```
<additional>
  <vType id="car"/>
  <vType id="heavy" vClass="truck"/>
</additional>
```
  
The simulation could then be called like this:
  
```
  sumo -n net.net.xml -a types.add.xml -r cars.rou.xml,trucks.rou.xml
```

## Sampling
By default, sampling will be performed iteratively by 
1) selecting a random counting location that has not yet reached it's count (and which still has viable routes)
2) selecting a random route that passes this counting location

until all counting locations have reached their measured count or there are no viable routes (routes which have all their passed counting locations below the input count)

By setting the option **--weighted**. The sampling algorithm is changed. For each route a probability value is loaded from the input. The probabilty can either be specified explicitly using route attribute 'probability' or implicitly if a route with the same sequence of edges appears multiple times in the the route input. Sampling will be performed iteratively by
1) selecting a random viable route sampled by probability

until all counting locations have reached their measured count or there are no viable routes (routes which have all their passed counting locations below the input count)

## Optimization
By default, routes will be sampled from the input route set until no further routes can be added without exceeding one of the counts. This may still leave some counts below their target values. At this point an ILP-Solver can be used to swap out routes and get closer to the target values or even reach the exact numbers.
By setting option **--optimize `<INT>`**. The number of times that a route is used can be changed by up to **`<INT>`** times. This defines a trade-off between using routes in the same distribution as found in the input and optimizing the counts.
When setting option **--optimize full**. No constraints on the route distribution are set and any route can be used as often as needed to reach the counts.
        
!!! note
    Optimization requires [scipy](https://www.scipy.org/).

## Further Calibration
It is possible to load the resulting output into routeSampler.py for another round of optimization. By setting the option **--optimize-input** the sampling step is skipped and the optimizer is run directly on the input route set.

By removing specific routes or adding new ones, the user can thus tailor the generating traffic in an iterative manner.
 
# generateTurnRatios.py

This script is used to calculate the turn ratios from a
an edge to its downstream edge with a given route file. The output file
can be directly used as input in [jtrrouter](../jtrrouter.md). The time interval will span the minimum and maximum departure times of the route file.

```
python tools/turn-defs/generateTurnRatios.py -r <route-file>
```

The standard output is the traffic volumes (which jtrrouter normalizes automatically). With the option **-p**,  turning ratios will be written as values from [0,1].


# generateTurnDefs.py

This script allows generation of the turn definitions based on the
number of lanes allowing particular turns. The basic functionality
distributes the traffic uniformly, that is:

1.  distribute the incoming traffic uniformly across the lanes forming
    the road
2.  distribute the amount of traffic assigned to each lane uniformly
    among the destinations that the lane allows turns to.
3.  sum up the values for each of the destinations that the road being
    processed allows turning to.

Example use

```
python tools/turn-defs/generateTurnDefs.py --connections-file connections.con.xml --turn-definitions-file output.turndefs.xml
```

The script allows to be extended with new traffic distribution policies
(for example, based on Gaussian distribution) easily. See the
*DestinationWeightCalculator* class for details.

The script processes the connections given in the provided *\*.con.xml*
file. For usage details, execute the *generateTurnDefs.py* script with
*--help* option.

!!! note
    You can generate a connections file with all the connections in the network using [netconvert](../netconvert.md) - see the *--plain-output-prefix* option.

# turnCount2EdgeCount.py

This script converts turn-count data into [edgeData](../Simulation/Output/Lane-_or_Edge-based_Traffic_Measures.md).

```
python tools/turn-defs/turnCount2EdgeCount.py -t <turn-file> -o <output-file>
```

# turnFile2EdgeRelations.py

This script converts the deprecated turn-file format into edgeRelation format

```
python tools/turn-defs/turnFile2EdgeRelations.py -t <turn-file> -o <output-file>
```
