/*
  Copyright (C) 2008, 2009 Jeroen Frijters

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.

  Jeroen Frijters
  jeroen@frijters.net
  
*/
using System;
using System.Runtime.InteropServices;
#if !NO_SYMBOL_WRITER
using System.Diagnostics.SymbolStore;
#endif
using IKVM.Reflection.Emit;

namespace IKVM.Reflection.Impl
{
	[StructLayout(LayoutKind.Sequential)]
	struct IMAGE_DEBUG_DIRECTORY
	{
		public uint Characteristics;
		public uint TimeDateStamp;
		public ushort MajorVersion;
		public ushort MinorVersion;
		public uint Type;
		public uint SizeOfData;
		public uint AddressOfRawData;
		public uint PointerToRawData;
	}

#if NO_SYMBOL_WRITER
	struct SymbolToken
	{
		internal SymbolToken(int value) { }
	}

	interface ISymbolWriterImpl
	{
		byte[] GetDebugInfo(ref IMAGE_DEBUG_DIRECTORY idd);
		void RemapToken(int oldToken, int newToken);
		void DefineLocalVariable2(string name, FieldAttributes attributes, int signature, int addrKind, int addr1, int addr2, int addr3, int startOffset, int endOffset);
		void OpenMethod(SymbolToken symbolToken, MethodBase mb);
		bool IsDeterministic { get; }
	}
#else
	interface ISymbolWriterImpl : ISymbolWriter
	{
		byte[] GetDebugInfo(ref IMAGE_DEBUG_DIRECTORY idd);
		void RemapToken(int oldToken, int newToken);
		void DefineLocalVariable2(string name, FieldAttributes attributes, int signature, SymAddressKind addrKind, int addr1, int addr2, int addr3, int startOffset, int endOffset);
		void OpenMethod(SymbolToken symbolToken, MethodBase mb);
		bool IsDeterministic { get; }
	}
#endif

	static class SymbolSupport
	{
		internal static ISymbolWriterImpl CreateSymbolWriterFor(ModuleBuilder moduleBuilder)
		{
#if NO_SYMBOL_WRITER
			throw new NotSupportedException("IKVM.Reflection compiled with NO_SYMBOL_WRITER does not support writing debugging symbols.");
#else
			if (Universe.MonoRuntime)
			{
#if MONO
				return new MdbWriter(moduleBuilder);
#else
				throw new NotSupportedException("IKVM.Reflection must be compiled with MONO defined to support writing Mono debugging symbols.");
#endif
			}
			else
			{
				return new PdbWriter(moduleBuilder);
			}
#endif
		}

		internal static byte[] GetDebugInfo(ISymbolWriterImpl writer, ref IMAGE_DEBUG_DIRECTORY idd)
		{
			return writer.GetDebugInfo(ref idd);
		}

		internal static void RemapToken(ISymbolWriterImpl writer, int oldToken, int newToken)
		{
			writer.RemapToken(oldToken, newToken);
		}
	}
}
