/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <QtGui>

#include "backend/ifpinterface.h"
#include "radiofreqdlg.h"
#include "revision.h"
#include "ifp_gui.h"

ifpGUIView *MainIfpGuiWindow;
ifp_progress_info ifpTransferInfo;

ifp_gui::ifp_gui()
        : QMainWindow(0),
        m_view( new ifpGUIView( this ))
{
    QPixmap openIcon, saveIcon, printIcon;

    //resize( 640, 480 );
    setAttribute(Qt::WA_DeleteOnClose);


    createActions();
    createMenus();
    createToolBars();
    createStatusBar();
    readSettings();

    // make m_view the central widget
    MainIfpGuiWindow = m_view;
    setCentralWidget(m_view);

    connect_timer = new QTimer(this);
    connect(connect_timer, SIGNAL(timeout()), this, SLOT(ifpConnect()));
    connect_timer->start(5000); // check every 5 seconds

    progress_timer = new QTimer(this);
    connect(progress_timer, SIGNAL(timeout()), this, SLOT(ifpTransferProgress()));
    progress_timer->start(100); // check every 1/10 seconds

    connect(m_view, SIGNAL(signalSetBatteryStatus(int)),
            this, SLOT(setBatteryStatus(int)));
    connect(m_view, SIGNAL(signalSetCapacity(int, int)),
            this, SLOT(setCapacity(int, int)));

    updateStatus();
}


ifp_gui::~ifp_gui()
{

}

void ifp_gui::enableWindow(bool enable)
{
    ifpToolBar->setEnabled(enable);
    menuBar()->setEnabled(enable);
}

void ifp_gui::ifpConnect()
{
    int ret;

    if (ifp_dh == NULL)
    {
        statusBar()->showMessage(tr("Not Connected - Try unplugging iRiver and reconnecting.  Then, double-click the root ('\\') directory in the iRiver Filesystem panel."), 3000);
    }
    else
    {
        if (m_view->isProcessing() == FALSE)
        {
            ret = updateStatus();
            if (ret >= 0)
                statusBar()->showMessage(tr("iRiver Connected"), 10000);
            connect_timer->start(5000); // check every 5 seconds
        }
        else
        {
            connect_timer->start(500); // check every 1/2 seconds
        }
    }
}

void ifp_gui::ifpTransferProgress()
{
    //if (m_view->isProcessing() &&
    if (ifpTransferInfo.show_progress)
    {
        if (!MainIfpGuiWindow->isTransferProgressVisible())
        {
            MainIfpGuiWindow->showTransferProgress();
        }

        m_view->updateProgress(&ifpTransferInfo);
    }
    else
    {
        MainIfpGuiWindow->hideTransferProgress();
    }
}

void ifp_gui::closeEvent( QCloseEvent* ce )
{
    ce->accept();
    return;
}

int ifp_gui::updateStatus()
{
    int ret;

    ret = m_view->getBatteryStatus();
    setBatteryStatus(ret);
    return m_view->updateStatus();
}

void ifp_gui::setBatteryStatus(int bStatus)
{
    switch (bStatus)
    {
    case 1:
        BatteryStatus->setPixmap(QPixmap(":/images/battery1.png"));
        break;
    case 2:
        BatteryStatus->setPixmap(QPixmap(":/images/battery2.png"));
        break;
    case 3:
        BatteryStatus->setPixmap(QPixmap(":/images/battery3.png"));
        break;
    case 4:
        BatteryStatus->setPixmap(QPixmap(":/images/battery4.png"));
        break;
    default:
        BatteryStatus->setPixmap(QPixmap(":/images/battery0.png"));
    }
}

void ifp_gui::setCapacity(int capSize, int capFree)
{
    int sizeMB;
    CapacityStatus->setMaximum(capSize);
    CapacityStatus->setValue(capSize - capFree);
    sizeMB = capFree / 1000000;
    Capacitylbl->setText(tr("%1 MB\n  free").arg(sizeMB));
}

void ifp_gui::showToolbar()
{
    ifpToolBar->setVisible(TRUE);
}

void ifp_gui::hideToolbar()
{
    ifpToolBar->setVisible(FALSE);
}

void ifp_gui::showToolbarText()
{
    ifpToolBar->setToolButtonStyle(Qt::ToolButtonTextUnderIcon);
}

void ifp_gui::hideToolbarText()
{
    ifpToolBar->setToolButtonStyle(Qt::ToolButtonIconOnly);
}

void ifp_gui::uploadPlayList()
{
    m_view->uploadPlaylist();
}

void ifp_gui::uploadSelectedFile()
{
    m_view->uploadSelectedFile();
}

void ifp_gui::downloadSelectedFile()
{
    m_view->downloadSelectedFile();
}

void ifp_gui::formatIfpDevice()
{
    m_view->formatIfpDevice();
}

void ifp_gui::show_ifp_info()
{
    char info_str[500];
    QString libLoaded, Message;

    if (ifp_dh == NULL)
    {
        showInfoMsgBox(tr("iFP GUI Device Info - Error"), iFPNotConnectedMsg);
        return;
    }
    if (usingLibrary)
        libLoaded = tr("Using ") + ifpLibraryVersion;
    else
    {
        libLoaded  = tr("Using static ifp functions (ifplib not available).\r\n");
        libLoaded += tr("  static ifp functions equivelant to %1").arg(LibRevision);
    }
    ifpgui_ifp_device_info(&ifp_dev, info_str, 500);
    Message = tr(info_str) + "\n\n" + libLoaded;
    showInfoMsgBox(tr("iFP GUI Device Info"), Message);
}

void ifp_gui::show_radiofreq_info()
{
    RadioFreqWindow *rfdialog;

    rfdialog = new RadioFreqWindow();
    rfdialog->setModal(TRUE);
    rfdialog->exec();
    delete rfdialog;
}

void ifp_gui::about()
{
    char info_str[500];
    QString msg;
    QString libLoaded;

    msg  = tr("This application can be used to interface\n");
    msg += tr("with an iRiver iFP device.\n\n");
    msg += tr("Revision: ") + AppRevision + "\n\n";
    msg += tr("please send feedback to ifpgui@gmail.com.\n\n\n");
    if (ifp_dh == NULL)
    {
        msg += tr("iFP Device Not Connected.");
    }
    else
    {
        ifpgui_ifp_device_info(&ifp_dev, info_str, 500);
        msg += QString(info_str);
        if (usingLibrary)
            libLoaded = tr("Using ") + ifpLibraryVersion;
        else
        {
            libLoaded  = tr("Using static ifp functions (ifplib not available).\r\n");
            libLoaded += tr("  static ifp functions equivelant to ") + LibRevision;
        }
        msg += "\n\n";
        msg += libLoaded;
    }
    showInfoMsgBox(tr("iFP GUI Application"), msg);
}


void ifp_gui::aboutQt()
{
    QMessageBox::aboutQt(this, tr("iFP GUI Application"));
}


void ifp_gui::auto_convert()
{
    bool autoConvert;
    autoConvert = auto_convertAct->isChecked();
    autoConvert = !autoConvert;
    auto_convertAct->setChecked(autoConvert);
    m_view->guiSettings->setAutoConvertREC(autoConvert);
}

void ifp_gui::create_directory(void)
{
    m_view->create_directory();
}

void ifp_gui::rename_file(void)
{
    m_view->rename_file();
}

void ifp_gui::delete_file(void)
{
    m_view->delete_file();
}

void ifp_gui::file_properties(void)
{
    m_view->file_properties();
}

void ifp_gui::align_horizontal(void)
{
    m_view->align_horizontal();
    align_horizontalAct->setChecked(true);
    align_verticalAct->setChecked(false);
    m_view->guiSettings->setViewAlignment(Qt::Horizontal);
}

void ifp_gui::align_vertical(void)
{
    m_view->align_vertical();
    align_verticalAct->setChecked(true);
    align_horizontalAct->setChecked(false);
    m_view->guiSettings->setViewAlignment(Qt::Vertical);
}

void ifp_gui::updateFirmware(void)
{
    m_view->UpdateFirmware();
}

void ifp_gui::createActions()
{
    create_directoryAct = new QAction(QIcon(":/images/newfolder.png"), tr("&New Folder"), this);
    create_directoryAct->setShortcut(tr("Ctrl+N"));
    create_directoryAct->setStatusTip(tr("Create a new directory folder"));
    connect(create_directoryAct, SIGNAL(triggered()), this, SLOT(create_directory()));

    delete_fileAct = new QAction(QIcon(":/images/delete.png"), tr("&Delete"), this);
    delete_fileAct->setShortcut(tr("Del"));
    delete_fileAct->setStatusTip(tr("Delete selected file"));
    connect(delete_fileAct, SIGNAL(triggered()), this, SLOT(delete_file()));

    rename_fileAct = new QAction(QIcon(":/images/rename.png"), tr("&Rename"), this);
    rename_fileAct->setShortcut(tr("F2"));
    rename_fileAct->setStatusTip(tr("Delete selected file"));
    connect(rename_fileAct, SIGNAL(triggered()), this, SLOT(rename_file()));

    file_propertiesAct = new QAction(QIcon(":/images/properties.png"), tr("&Properties"), this);
    file_propertiesAct->setShortcut(tr("F3"));
    file_propertiesAct->setStatusTip(tr("Show properties of selected file"));
    connect(file_propertiesAct, SIGNAL(triggered()), this, SLOT(file_properties()));

    closeAllWindowsAct = new QAction(tr("E&xit"), this);
    closeAllWindowsAct->setShortcut(tr("Alt+X"));
    closeAllWindowsAct->setStatusTip(tr("Exit Application"));
    connect(closeAllWindowsAct, SIGNAL(triggered()), qApp, SLOT(closeAllWindows()));

    align_horizontalAct = new QAction(QIcon(":/images/view_left_right.png"), tr("Align &Left-Right"), this);
    align_horizontalAct->setCheckable(true);
    align_horizontalAct->setStatusTip(tr("Show application file trees side to side in a Left-Right layout"));
    connect(align_horizontalAct, SIGNAL(triggered()), this, SLOT(align_horizontal()));

    align_verticalAct = new QAction(QIcon(":/images/view_top_bottom.png"), tr("Align &Top-Bottom"), this);
    align_verticalAct->setStatusTip(tr("Show application file trees one on top and the other on the bottom"));
    align_verticalAct->setCheckable(true);
    connect(align_verticalAct, SIGNAL(triggered()), this, SLOT(align_vertical()));

    uploadPlayListAct = new QAction(QIcon(":/images/playlist.png"),tr("&Upload Playlist"), this);
    uploadPlayListAct->setShortcut(tr("Ctrl+U"));
    uploadPlayListAct->setStatusTip(tr("Upload a playlist to the ifp device"));
    uploadPlayListAct->setWhatsThis(tr("<p>Select a destination on the iFP device.  Then, press this button to upload an m3u playist to the iFP device at the selected destination.</p>"));
    connect(uploadPlayListAct, SIGNAL(triggered()), this, SLOT(uploadPlayList()));

    auto_convertAct = new QAction(tr("&Auto-Convert REC files to MP3"), this);
    auto_convertAct->setCheckable(true);
    auto_convertAct->setShortcut(tr("Ctrl+A"));
    auto_convertAct->setStatusTip(tr("Convert REC files to MP3 as they are downloaded from the ifp device"));
    connect(auto_convertAct, SIGNAL(triggered()), this, SLOT(auto_convert()));

    show_radiofreq_infoAct = new QAction(QIcon(":/images/radio_s.png"), tr("&Tuner Settings"), this);
    show_radiofreq_infoAct->setShortcut(tr("Ctrl+T"));
    show_radiofreq_infoAct->setStatusTip(tr("Open the FM tuner settings options"));
    show_radiofreq_infoAct->setWhatsThis(tr("<p>Press this button to open the dialog that allows you to edit the radio frequency settings for your iFP device.</p>"));
    connect(show_radiofreq_infoAct, SIGNAL(triggered()), this, SLOT(show_radiofreq_info()));

    updateFirmwareAct = new QAction(QIcon(":/images/reload_s.png"), tr("&Upgrade Firmware"), this);
    updateFirmwareAct->setStatusTip(tr("Upgrade the Firmware in the ifp"));
    updateFirmwareAct->setWhatsThis(tr("<p>Use this to update the iFP device firmware.</p> <br>"
                                       "<p>You must first download a firmware file and save it to your hard drive.</p><br>" ));
    connect(updateFirmwareAct, SIGNAL(triggered()), this, SLOT(updateFirmware()));

    show_ifp_infoAct = new QAction(tr("&iFP Info"), this);
    show_ifp_infoAct->setShortcut(tr("Ctrl+F"));
    show_ifp_infoAct->setStatusTip(tr("Show information about this iFP device"));
    connect(show_ifp_infoAct, SIGNAL(triggered()), this, SLOT(show_ifp_info()));

    aboutAct = new QAction(tr("&About"), this);
    aboutAct->setShortcut(tr("F1"));
    aboutAct->setStatusTip(tr("Show information about this program"));
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

    aboutQtAct = new QAction(tr("About &Qt"), this);
    aboutQtAct->setStatusTip(tr("Show the Qt library's About box"));
    connect(aboutQtAct, SIGNAL(triggered()), qApp, SLOT(aboutQt()));


    uploadFileAct = new QAction(QIcon(":/images/upload_s.png"), tr("Upload File"), this);
    uploadFileAct->setToolTip(tr("Upload File to iFP"));
    uploadFileAct->setWhatsThis(tr("<p>Select a file from the left and press this button to upload it to the iFP device.</p>"));
    connect(uploadFileAct, SIGNAL(triggered()), this, SLOT(uploadSelectedFile()));

    downloadFileAct = new QAction(QIcon(":/images/download_s.png"), tr("Download File"), this);
    downloadFileAct->setToolTip(tr("Download File from iFP"));
    downloadFileAct->setWhatsThis(tr("<p>Select a file from the right and press this button to download it to your hard drive.</p>"));
    connect(downloadFileAct, SIGNAL(triggered()), this, SLOT(downloadSelectedFile()));

    formatAct = new QAction(QIcon(":/images/format_s.png"), tr("Format"), this);
    formatAct->setToolTip(tr("Format iFP Filesystem"));
    formatAct->setWhatsThis(tr("<p>Press this button to format the iFP device.</p>"));
    connect(formatAct, SIGNAL(triggered()), this, SLOT(formatIfpDevice()));

}

void ifp_gui::createMenus()
{
    fileMenu = menuBar()->addMenu(tr("&File"));
    fileMenu->addAction(create_directoryAct);
    fileMenu->addAction(delete_fileAct);
    fileMenu->addAction(rename_fileAct);
    fileMenu->addAction(file_propertiesAct);
    fileMenu->addSeparator();
    fileMenu->addAction(closeAllWindowsAct);

    viewMenu = menuBar()->addMenu(tr("&View"));
    viewMenu->addAction(align_horizontalAct);
    viewMenu->addAction(align_verticalAct);

    optionsMenu = menuBar()->addMenu(tr("&Tools"));
    optionsMenu->addAction(uploadPlayListAct);
    optionsMenu->addAction(auto_convertAct);
    optionsMenu->addAction(show_radiofreq_infoAct);
    optionsMenu->addSeparator();
    optionsMenu->addAction(updateFirmwareAct);

    menuBar()->addSeparator();

    helpMenu = menuBar()->addMenu(tr("&Help"));
    helpMenu->addAction(show_ifp_infoAct);
    helpMenu->addSeparator();
    helpMenu->addAction(aboutAct);
    helpMenu->addAction(aboutQtAct);
}

void ifp_gui::createToolBars()
{
    //ifpToolBar = new QToolBar(tr("File"), parent);
    ifpToolBar = addToolBar(tr("Toolbar"));
    ifpToolBar->setIconSize(QSize(30,30));
    ifpToolBar->addAction(uploadPlayListAct);
    ifpToolBar->addAction(show_radiofreq_infoAct);
    ifpToolBar->addAction(updateFirmwareAct);


    BatteryStatus = new QLabel(tr("BatteryStatus"));
    BatteryStatus->setPixmap(QPixmap(":/images/battery0.png"));
    BatteryStatus->setToolTip(tr("Battery Status"));
    QString batteryStatusText = tr("<p>This status bar shows the Battery Status.</p>");
    BatteryStatus->setWhatsThis(batteryStatusText);
    ifpToolBar->addWidget(BatteryStatus);

    ifpToolBar->addWidget(new QLabel("      "));

    CapacityStatus = new QProgressBar();
    CapacityStatus->setMaximumWidth(250);
    CapacityStatus->setMinimum(0);
    CapacityStatus->setMaximum(100000);
    CapacityStatus->setToolTip(tr("Used Storage"));
    QString CapacityStatusText = tr("<p>This shows the ammount of space used by uploaded songs and files.</p>");
    CapacityStatus->setWhatsThis(CapacityStatusText);
    ifpToolBar->addWidget(CapacityStatus);

    Capacitylbl      = new QLabel(tr("   MB"));
    Capacitylbl->setAlignment(Qt::AlignVCenter | Qt::AlignLeft);
    Capacitylbl->setToolTip(tr("Free Storage"));
    QString CapacityText = tr("<p>This shows the ammount of space used by uploaded songs and files.</p>");
    Capacitylbl->setWhatsThis(CapacityText);
    ifpToolBar->addWidget(Capacitylbl);

    ifpToolBar->addWidget(new QLabel("      "));

    ifpToolBar->addAction(uploadPlayListAct);
    ifpToolBar->addAction(uploadFileAct);
    ifpToolBar->addAction(downloadFileAct);

    ifpToolBar->addWidget(new QLabel("      "));

    ifpToolBar->addAction(show_radiofreq_infoAct);

    ifpToolBar->addWidget(new QLabel("      "));

    ifpToolBar->addAction(formatAct);
    ifpToolBar->addAction(updateFirmwareAct);
}

void ifp_gui::createStatusBar()
{
    statusBar()->showMessage( tr("Ready"), 2000 );
}

void ifp_gui::readSettings()
{
    bool autoConvert;
    int  viewAlignment;

    viewAlignment    = m_view->guiSettings->getViewAlignment();
    if (viewAlignment == Qt::Vertical)
    {
        align_horizontalAct->setChecked(false);
        align_verticalAct->setChecked(true);
        align_vertical();
    }
    else
    {
        align_horizontalAct->setChecked(true);
        align_verticalAct->setChecked(false);
        align_horizontal();
    }

    autoConvert = m_view->guiSettings->getAutoConvertREC();
    auto_convertAct->setChecked(autoConvert);

}
