//*****************************************************************************
//                              PnlGnuCapAC.cpp                               *
//                             -----------------                              *
// Started     : 18/08/2003                                                   *
// Last Update : 16/05/2011                                                   *
// Copyright   : (C) 2003 by MSWaters                                         *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/panels/PnlGnuCapAC.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( PnlGnuCapAC, PnlAnaBase )

  EVT_RADIOBOX( ID_RBX_SCALE,   PnlGnuCapAC::OnScale   )
  EVT_CHOICE  ( ID_CHO_SRCNAME, PnlGnuCapAC::OnSrcName )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.

PnlGnuCapAC::PnlGnuCapAC( wxWindow * poWin ) : PnlAnaBase( poWin )
{
  bSetSimrEng( eSIMR_GNUCAP );
  bSetAnaType( eCMD_AC );

  Create( );  // Create the analysis panel
  bClear( );  // Clear all object attributes
}

//*****************************************************************************
// Destructor.

PnlGnuCapAC::~PnlGnuCapAC( )
{
}

//*****************************************************************************
// Create the display objects.

void  PnlGnuCapAC::Create( void )
{
#ifndef LAYOUT_MNGR
  int  ix, iy;

  // Set the frequency sweep parameter labels
  m_oSbxSwpPars.GetSize( &ix, &iy );
  m_oSbxSwpPars.SetSize(  ix, 167 );
  m_oSbxSwpPars.SetLabel( wxT(" AC Sweep ") );
#endif // LAYOUT_MNGR
  m_oPnlStart  .bSetName( wxT("Start Frequency") );
  m_oPnlStop   .bSetName( wxT("Stop Frequency") );

  // Set sweep parameter units
  m_oPnlStart.bSetUnitsType( eUNITS_FREQ );
  m_oPnlStop .bSetUnitsType( eUNITS_FREQ );
  m_oPnlStep .bSetUnitsType( eUNITS_FREQ );

  PnlAnaBase::CreateScale( );  // Create the scale controls
  PnlAnaBase::CreateCpxPrt( ); // Create the parameter complex part check boxes
  PnlAnaBase::CreateSrc( );    // Create the input source controls
  PnlAnaBase::CreateTemp( );   // Create the analysis temperature controls

  PnlAnaBase::DoLayout( );     // Layout the panel's GUI objects

#ifdef LAYOUT_MNGR
  SetSwpPnlLbl( wxT(" AC Sweep ") );
#endif // LAYOUT_MNGR
}

//*****************************************************************************
// Initialize the step scale.

void  PnlGnuCapAC::InitScale( void )
{
  switch( m_oRbxScale.GetSelection( ) )
  {
    case eSCALE_LIN :
      m_oPnlStep.bSetName( wxT("Step Increment") );
      m_oPnlStep.bSetVarType( eVAR_FLT );
      m_oPnlStep.bSetUnitsType( eUNITS_FREQ );
      m_oPnlStep.bShowUnits( TRUE );
      break;

    case eSCALE_LOG :
      m_oPnlStep.bSetName( wxT("Step Multiplier") );
      m_oPnlStep.bSetVarType( eVAR_FLT );
      m_oPnlStep.bSetParms( 1.1, 0.01, 10.0, 1.0, 10.0 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    case eSCALE_DEC :
      m_oPnlStep.bSetName( wxT("Steps / Decade") );
      m_oPnlStep.bSetVarType( eVAR_INT );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    case eSCALE_OCT :
      m_oPnlStep.bSetName( wxT("Steps / Octave") );
      m_oPnlStep.bSetVarType( eVAR_INT );
      m_oPnlStep.bSetParms( 10, 1, 10000, 1, 100 );
      m_oPnlStep.bShowUnits( FALSE );
      break;

    default :
      break;
  }
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapAC::bClear( void )
{
  bool  bRtn=TRUE;

  // Clear the base class
  if( ! PnlAnaBase::bClear( ) )   bRtn = FALSE;

  // Set default step scale type and sweep values
  m_oPnlStart.bSetValue( (float)   1.0 );
  m_oPnlStop .bSetValue( (float) 100.0 );
  m_oPnlStep .bSetValue( (float)  10.0 );
  m_oPnlStep .bShowUnits( TRUE );
  m_oPnlStart.bSetUnits( wxT("kHz") );
  m_oPnlStop .bSetUnits( wxT("kHz") );
  m_oPnlStep .bSetUnits( wxT("kHz") );

  // Set default scale value
  if( ! bSetScale( eSCALE_DEC ) ) bRtn = FALSE;

  // Set the complex part check box default values
  m_oCbxMag  .SetValue( TRUE );
  m_oCbxPhase.SetValue( FALSE );
  m_oCbxReal .SetValue( FALSE );
  m_oCbxImag .SetValue( FALSE );
  m_oCbxMagDb.SetValue( TRUE );

  return( bRtn );
}

//*****************************************************************************
// Load information from a simulation object.
//
// Argument List :
//   roSimn - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapAC::bLoad( SimnGnuCap & roSimn )
{
  bool  bRtn=TRUE;

  // Load the components into the signal source choice box
  PnlAnaBase::LoadSrcNames( roSimn.m_oaCpnts, wxT("VI") );

  // Go no further if the AC command isn't valid
  if( ! roSimn.m_oCmdAC.bIsValid( ) )                        return( FALSE );

  // Set the source component
  if( ! PnlAnaBase::bSetSrcCpnt( roSimn.m_oCpntSwpSrc ) )    bRtn = FALSE;

  // Set the step scale (do this before setting the sweep step)
  if( roSimn.m_oCmdAC.m_eScale != eSCALE_NONE )
  {
    m_oRbxScale.SetSelection( roSimn.m_oCmdAC.m_eScale );
    InitScale( );
  }

  // Set the sweep values
  if( ! m_oPnlStart.bSetValue( roSimn.m_oCmdAC.m_osStart ) ) bRtn = FALSE;
  if( ! m_oPnlStop .bSetValue( roSimn.m_oCmdAC.m_osStop  ) ) bRtn = FALSE;
  if( ! m_oPnlStep .bSetValue( roSimn.m_oCmdAC.m_osStep  ) ) bRtn = FALSE;
  if( m_oPnlStart.dfGetValue( ) == 0.0 )
        m_oPnlStart.bSetUnits( m_oPnlStop.rosGetUnits( ) );

  if( roSimn.eGetAnaType( ) == eCMD_AC )
  {
    // Set the parameters to derive
    m_oCbxVoltage.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] );
    m_oCbxCurrent.SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] );
    m_oCbxPower  .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] );
    m_oCbxResist .SetValue( roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] );

    // Set the complex parts to derive
    m_oCbxMag  .SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAG   ] );
    m_oCbxPhase.SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_PHASE ] );
    m_oCbxReal .SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_REAL  ] );
    m_oCbxImag .SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_IMAG  ] );
    m_oCbxMagDb.SetValue( roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAGDB ] );
  }

  // Set the analysis temperature
  if( ! m_oPnlTemp.bSetValue( roSimn.m_oCmdAC.m_osTempC ) )  bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Save information to a simulation object.
//
// Argument List :
//   roSimn - A simulation object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PnlGnuCapAC::bSave( SimnGnuCap & roSimn )
{
  wxString  os1;
  size_t    sz1;
  bool      b1;

  m_osErrMsg.Empty( );

  // Set the sweep values
  roSimn.m_oCmdAC.m_osStart = m_oPnlStart.rosGetValue( );
  roSimn.m_oCmdAC.m_osStop  = m_oPnlStop .rosGetValue( );
  roSimn.m_oCmdAC.m_osStep  = m_oPnlStep .rosGetValue( );

  // Set the step scale
  roSimn.m_oCmdAC.m_eScale = (eScaleType) m_oRbxScale.GetSelection( );

  // Set the component to be used as the sweep source
  if( m_oChoSrcName.GetStringSelection( ) == wxT("None") )
    SetErrMsg( wxT("No sweep source component has been selected.") );
  else if ( m_oPnlSrcLvl.dfGetValue( ) == 0.0 )
    SetErrMsg( wxT("Sweep source component value of zero is not permitted.") );
  else
  {
    os1 = m_oChoSrcName.GetStringSelection( );
    roSimn.m_oCpntSwpSrc = roSimn.NetList::roGetCpnt( os1 );
    roSimn.m_oCpntSwpSrc.m_osValue = wxT("GENERATOR(1) AC ")
                                                 + m_oPnlSrcLvl.rosGetValue( );
  }

  // Set the analysis type
  roSimn.m_oCmdPR.m_eAnaType = eCMD_AC;

  // Store the parameters to derive
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_VLT ] = m_oCbxVoltage.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_CUR ] = m_oCbxCurrent.GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_PWR ] = m_oCbxPower  .GetValue( );
  roSimn.m_oCmdPR.m_bParmtrs[ ePAR_RES ] = m_oCbxResist .GetValue( );

  // Store the complex parts of the parameters to derive
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAG   ] = m_oCbxMag  .GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_PHASE ] = m_oCbxPhase.GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_REAL  ] = m_oCbxReal .GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_IMAG  ] = m_oCbxImag .GetValue( );
  roSimn.m_oCmdPR.m_bCpxPrts[ eCPX_MAGDB ] = m_oCbxMagDb.GetValue( );

  // Set the analysis temperature
  roSimn.m_oCmdAC.m_osTempC = m_oPnlTemp.rosGetValue( );

  // Create the command strings
  roSimn.m_oCmdAC.bFormat( );
  roSimn.m_oCmdPR.bFormat( );

  // Check for errors
  if( ! roSimn.m_oCmdAC.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdAC.rosGetErrMsg( ) );
  if( ! roSimn.m_oCmdPR.bIsValid( ) )
    SetErrMsg( roSimn.m_oCmdPR.rosGetErrMsg( ) );
  for( sz1=eCPX_MAG, b1=FALSE; sz1<=eCPX_IMAG; sz1++ )
    if( roSimn.m_oCmdPR.m_bCpxPrts[ sz1 ] ) b1 = TRUE;
  if( ! b1 ) SetErrMsg( wxT("No complex parts have been selected.") );

  return( bIsOk( ) );
}

//*****************************************************************************
//                                                                            *
//                             Event Handlers                                 *
//                                                                            *
//*****************************************************************************
// Step scale radio box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapAC::OnScale( wxCommandEvent & roEvtCmd )
{
  InitScale( );
}

//*****************************************************************************
// Source component choice box event handler.
//
// Argument List :
//   roEvtCmd - An object holding information about the event

void  PnlGnuCapAC::OnSrcName( wxCommandEvent & roEvtCmd )
{
  wxString  os1;

  // Set the units type
  os1 = m_oChoSrcName.GetStringSelection( );
  m_oPnlSrcLvl.bSetUnitsType( Component::eGetUnitsType( os1 ) );

  // Set the source value
  if( m_oChoSrcName.GetStringSelection( ) == wxT("None") )
    m_oPnlSrcLvl.bSetValue( (double) 0.0 );
  else if( m_oPnlSrcLvl.dfGetValue( ) == 0.0 )
    m_oPnlSrcLvl.bSetValue( (double) 1.0 );

  // Permit the base class to process this event as well
  roEvtCmd.Skip( );
}

//*****************************************************************************
