/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#ifndef _GTKBOOKLIST_H
#define _GTKBOOKLIST_H

#include <gtkmm.h>
#include <iostream>
#include <libintl.h>
#include <vector>
#include <map>
#include "Book.h"

#define _(String) gettext (String)
#define gettext_noop(String) (String)
#define N_(String) gettext_noop (String)

enum GTKBOOKLIST_COLUMNS {
  GTKBOOKLIST_COLUMN_ICON,
  GTKBOOKLIST_COLUMN_AUTHORANDTITLE,
  GTKBOOKLIST_COLUMN_TITLE,
  GTKBOOKLIST_COLUMN_CATEGORY,
  GTKBOOKLIST_COLUMN_READDATE,
  GTKBOOKLIST_COLUMN_READDATE_TIMET,
  GTKBOOKLIST_COLUMN_RATING,
  GTKBOOKLIST_COLUMN_RATING_INT,
  GTKBOOKLIST_COLUMN_BOOK_PTR
};

class GtkBookList : public Gtk::TreeView {
public:
  GtkBookList();
  ~GtkBookList();
  
  /* Triggered whenever a book has been selected. */
  SigC::Signal1<void, Book*> signal_book_selected;
  /* Triggered whenever a book has been activated. */
  SigC::Signal1<void, Book*> signal_book_activated;
  /* Triggered whenever a book has been added or removed. */
  SigC::Signal0<void>        signal_changed;
  /* Triggered whenever the list sorting was changed. */
  SigC::Signal0<void>        signal_sorting_changed;
  
  /* Inserts the given book. */
  void insert_book(Book* book);
  
  /* Removes the given book from the list. */
  void remove_book(Book* book);
  
  /* Removes the selected book. */
  void remove_selected(void);
  
  /* Walks through all books that are currently in the list, updating the
   * row text in case anything changed. */
  void update_soft(void);
  
  /* Returns the first selected book. */
  Book* get_first_selected(void);
  
  /* Returns the number books in the list. */
  int get_numitems(void);
  
  /* Defines the sort column. */
  void set_sorting(int col);
  
  /* Returns the sort column number. */
  int get_sorting(void);
  
private:
  /* Fills the given tree row with the data from the given book. */
  void row_fill(Gtk::TreeModel::Row &row, Book* book);
  
  void on_selection_activated(Gtk::TreePath path,
                              Gtk::TreeViewColumn* column);
  void on_selection_changed(void);
  
  // List model columns.
  class ModelColumns : public Gtk::TreeModel::ColumnRecord {
  public:
    Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > icon;
    Gtk::TreeModelColumn<Glib::ustring>              author_and_title;
    Gtk::TreeModelColumn<Glib::ustring>              title;
    Gtk::TreeModelColumn<Glib::ustring>              category;
    Gtk::TreeModelColumn<Glib::ustring>              readdate;
    Gtk::TreeModelColumn<time_t>                     readdate_time_t;
    Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > rating;
    Gtk::TreeModelColumn<int>                        rating_integer;
    Gtk::TreeModelColumn<Book*>                      book;
    ModelColumns() {
      add(icon);
      add(author_and_title);
      add(title);
      add(category);
      add(readdate);
      add(readdate_time_t);
      add(rating);
      add(rating_integer);
      add(book);
    }
  };
  Glib::RefPtr<Gtk::ListStore>            store;
  Glib::RefPtr<Gdk::Pixbuf>               pixbuf_book;
  std::vector<Glib::RefPtr<Gdk::Pixbuf> > pixbuf_rating;
  ModelColumns                            columns;
  std::map<Book*, Gtk::TreeIter>          booklist;
  int                                     numitems;
};

#endif /* _GTKBOOKLIST_H */
