/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * ranonymize - argus data anonymization function.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */

#include <argus_client.h>
#include <ranonymize.h>


#define MAX_OBJ_SIZE            1024

unsigned int RaMapHash = 0;
unsigned int RaHashSize  = 0;

unsigned int RaMapOffsetValue  = 0;
unsigned int RaMapTransOffsetValue  = 0;
unsigned int RaMapSequenceOffsetValue = 0;
unsigned int RaMapTimeSecOffsetValue  = 0;
unsigned int RaMapTimeuSecOffsetValue = 0;
unsigned int RaPortMappingOffsetValue  = 0;
unsigned short RaMapIpIdValue = 0;

struct RaMapHashTableStruct {
   int size;
   struct RaMapHashTableHeader **array;
};

struct RaMapHashTableHeader {
   struct RaMapHashTableHeader *nxt, *prv;
   unsigned int hash;
   int type, len, value, mask;
   struct RaClassNets *net;
   void *obj, *sub;
};

struct RaMapHashTableStruct RaMapHashTable;
struct RaMapHashTableStruct RaMapNetTable;

struct RaClassNets {
   struct RaClassNets *supernet;
   unsigned int net;
   unsigned int index;
};

struct RaClassNets *RaClassANets[256];

struct RaPortStruct {
   struct ArgusQueueHeader qhdr;
   unsigned short value;
};

#define RA_MAX_PORT		65535


unsigned short RaPortMapping [65536];
unsigned short RaPortRandom [65536];
unsigned short RaPortTemp [65536];
int RaMapConvert = 0;

int RaNonParseResourceFile (char *);

int RaPreserveBroadcastAddress = 1;
int RaPreserveWellKnownPorts = 1;
int RaPreserveRegisteredPorts = 0;
int RaPreservePrivatePorts = 0;
int RaPreserveIpId = 0;

char *RaPortMappingOffset = "offset:random";
char *RaNonTransRefNumOffset = "random";
char *RaNonSeqNumOffset = "random";
char *RaNonTimeSecOffset = "random";
char *RaNonTimeuSecOffset = "random";

#define RANON_NONE		0
#define RANON_SUBNET		1
#define RANON_CLASS		2
#define RANON_CIDR		3

#define RANON_RANDOM		1
#define RANON_SHIFT		2


char *RaNonPreserveNetAddrHierarchy = "class";
unsigned int RaMapNetAddrHierarchy = RANON_CLASS;

#include <rastrip.h>
struct ArgusRecord * RaConstructArgusRecord (struct ArgusRecord *);

int ArgusStripDown = 0;
int RaFirstMOptionField = 1;

#define RASTRIP_DSR_TYPES                19

#define RASTRIP_DSR_FAR                  0
#define RASTRIP_DSR_MAC                  1
#define RASTRIP_DSR_TCP                  2
#define RASTRIP_DSR_ICMP                 3
#define RASTRIP_DSR_RTP                  4
#define RASTRIP_DSR_IGMP                 5
#define RASTRIP_DSR_ARP                  6
#define RASTRIP_DSR_FRG                  7
#define RASTRIP_DSR_ESP                  8
#define RASTRIP_DSR_MPLS                 9
#define RASTRIP_DSR_VLAN                10
#define RASTRIP_DSR_PPPOE               11
#define RASTRIP_DSR_AGR                 12
#define RASTRIP_DSR_TIME                13
#define RASTRIP_DSR_SRCTIME             14
#define RASTRIP_DSR_DSTTIME             15 
#define RASTRIP_DSR_USRDATA		16      
#define RASTRIP_DSR_SRCUSRDATA		17      
#define RASTRIP_DSR_DSTUSRDATA		18      

int RaDSRFields[RASTRIP_DSR_TYPES];

char *RaDSRKeyWords[RASTRIP_DSR_TYPES] = {
   "far",
   "mac",
   "tcp",
   "icmp",
   "rtp",
   "igmp",
   "arp",
   "frag",
   "esp",
   "mpls",
   "vlan",
   "pppoe",
   "agr",
   "jitter",
   "stime",
   "dtime",
   "user",
   "srcuser",
   "dstuser",
};

void RaProcessMOptions(struct ArgusModeStruct *);

#define RA_ADD_OPTION		1
#define RA_SUB_OPTION		2

void
RaProcessMOptions(struct ArgusModeStruct *mode)
{
   int x, RaOptionOperation, setValue = 0;
   char *ptr = NULL;
   char *endptr;

   if (mode != NULL) {
      while (mode) {
         if (isdigit(*mode->mode)) {
             ArgusStripDown = strtol(mode->mode, &endptr, 10);
             if (mode->mode == endptr)
                usage();

         } else {
            if (*mode->mode == '-') {
               if (RaFirstMOptionField) {
                  for (x = 0; x < RASTRIP_DSR_TYPES; x++)
                     RaDSRFields[x] = 1;
                  RaFirstMOptionField = 0;
               }
               ptr = mode->mode + 1;
               RaOptionOperation = RA_SUB_OPTION;
            } else 
            if (*mode->mode == '+') {
               ptr = mode->mode + 1;
               RaOptionOperation = RA_ADD_OPTION;
            } else {
               if (RaFirstMOptionField) {
                  bzero ((char *) RaDSRFields, sizeof(RaDSRFields));
                  RaFirstMOptionField = 0;
               }
               ptr = mode->mode;
               RaOptionOperation = RA_ADD_OPTION;
            }

            setValue = (RaOptionOperation == RA_ADD_OPTION) ? 1 : 0;

            for (x = 0; x < RASTRIP_DSR_TYPES; x++) {
               if (!strncmp (RaDSRKeyWords[x], ptr, strlen(RaPrintKeyWords[x]))) {
                  switch (x) {
                     case RASTRIP_DSR_USRDATA: {
                        RaDSRFields[RASTRIP_DSR_SRCUSRDATA] = setValue;
                        RaDSRFields[RASTRIP_DSR_DSTUSRDATA] = setValue;
                        break;
                     }

                     default:
                        RaDSRFields[x] = setValue;
                        break;
                  }
               }
            }
            mode = mode->nxt;
         }
      }
   }
}

void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   int retn, cnt, ind = 0, x, y, count = 0, shiftcount = 0, start;
   unsigned short i, value, startport = 0;
   char *ptr = NULL;

   if (!(RaInitialized)) {
      RaInitialized++;
      RaWriteOut = 0;
      RaCumulativeMerge = 1;
      RaMapConvert = 1;
      RaHashSize = 0x10000;

      bzero ((char *)RaDSRFields, sizeof(RaDSRFields));
      RaDSRFields[RASTRIP_DSR_FAR] = 1;
      RaDSRFields[RASTRIP_DSR_TCP] = 1;

      if ((mode = ArgusModeList) != NULL)
         RaProcessMOptions(mode);

      if (!(ArgusStripDown))
         ArgusStripDown++;

      if ((RaMapHashTable.array = (struct RaMapHashTableHeader **)
                  ArgusCalloc (RaHashSize, sizeof (struct RaMapHashTableHeader))) == NULL)
         ArgusLog (LOG_ERR, "RaMapInit: ArgusCalloc error %s\n", strerror(errno));

      RaMapHashTable.size = RaHashSize;

      if ((RaMapNetTable.array = (struct RaMapHashTableHeader **)
                  ArgusCalloc (RaHashSize, sizeof (struct RaMapHashTableHeader))) == NULL)
         ArgusLog (LOG_ERR, "RaMapInit: ArgusCalloc error %s\n", strerror(errno));

      RaMapNetTable.size = RaHashSize;

      if (ArgusFlowModelFile)
         if ((RaNonParseResourceFile (ArgusFlowModelFile)) < 0)
            exit(0);

      bzero ((char *)RaClassANets, sizeof(RaClassANets));
      gettimeofday(tvp, 0L);
      srandom(tvp->tv_usec);

      if (!(strncmp(RaNonTransRefNumOffset, "random", 6)))
         RaMapTransOffsetValue    = random() % 100000;
      else {
         if (!(strncmp(RaNonTransRefNumOffset, "fixed", 5))) {
            if ((ptr = strchr(RaNonTransRefNumOffset, ':')) != NULL)
               RaMapTransOffsetValue = atoi(++ptr);
            else
               ArgusLog(LOG_ERR, "RANON_TIME_SEC_OFFSET syntax error\n");
         } else
            if (!(strncmp(RaNonTransRefNumOffset, "no", 2)))
               RaMapTransOffsetValue = 0;
            else
               ArgusLog(LOG_ERR, "RANON_TRANSREFNUM_OFFSET syntax error\n");
      }

      if (!(strncmp(RaNonSeqNumOffset, "random", 6)))
         RaMapSequenceOffsetValue = random() % 1000000;
      else {
         if (!(strncmp(RaNonSeqNumOffset, "fixed", 5))) {
            if ((ptr = strchr(RaNonSeqNumOffset, ':')) != NULL)
               RaMapSequenceOffsetValue = atoi(++ptr);
            else
               ArgusLog(LOG_ERR, "RANON_TIME_SEC_OFFSET syntax error\n");
         } else
            if (!(strncmp(RaNonSeqNumOffset, "no", 2)))
               RaMapSequenceOffsetValue = 0;
            else
               ArgusLog(LOG_ERR, "RANON_SEQNUM_OFFSET syntax error\n");
      }

      if (!(strncmp(RaNonTimeSecOffset, "random", 6)))
         RaMapTimeSecOffsetValue  = random() % (tvp->tv_sec / 2);
      else {
         if (!(strncmp(RaNonTimeSecOffset, "fixed", 5))) {
            if ((ptr = strchr(RaNonTimeSecOffset, ':')) != NULL)
               RaMapTimeSecOffsetValue = atoi(++ptr);
            else
               ArgusLog(LOG_ERR, "RANON_TIME_SEC_OFFSET syntax error\n");
         } else
            if (!(strncmp(RaNonTimeSecOffset, "no", 2)))
               RaMapTimeSecOffsetValue  = 0;
            else
               ArgusLog(LOG_ERR, "RANON_TIME_SEC_OFFSET syntax error\n");
      }

      if (!(strncmp(RaNonTimeuSecOffset, "random", 6)))
         RaMapTimeuSecOffsetValue = random() % 500000;
      else {
         if (!(strncmp(RaNonTimeuSecOffset, "fixed", 5))) {
            if ((ptr = strchr(RaNonTimeuSecOffset, ':')) != NULL)
               RaMapTimeuSecOffsetValue = atoi(++ptr);
            else
               ArgusLog(LOG_ERR, "RANON_TIME_USEC_OFFSET syntax error\n");
         } else
            if (!(strncmp(RaNonTimeuSecOffset, "no", 2)))
               RaMapTimeuSecOffsetValue = 0;
            else
               ArgusLog(LOG_ERR, "RANON_TIME_USEC_OFFSET syntax error\n");
      }

      if (!(strncmp(RaNonPreserveNetAddrHierarchy, "cidr", 4))) {
         RaMapNetAddrHierarchy = RANON_CIDR;
      } else {
         if (!(strncmp(RaNonPreserveNetAddrHierarchy, "class", 5))) {
               RaMapNetAddrHierarchy = RANON_CLASS;
         } else {
            if (!(strncmp(RaNonPreserveNetAddrHierarchy, "subnet", 6)))
               RaMapNetAddrHierarchy = RANON_SUBNET;
            else {
               if (!(strncmp(RaNonPreserveNetAddrHierarchy, "no", 2)))
                  RaMapNetAddrHierarchy = RANON_NONE;
               else
                  ArgusLog(LOG_ERR, "RANON_PRESERVE_NET_ADDRESS_HIERARCHY syntax error\n");
            }
         }
      }

      if (!(RaPreserveIpId))
         RaMapIpIdValue = (unsigned short) (random() % 0x10000);

      bzero ((char *) RaPortMapping, sizeof(RaPortMapping));
      bzero ((char *) RaPortRandom, sizeof(RaPortRandom));

      for (i = 0; i < RA_MAX_PORT; i++) {
         RaPortMapping[i] = i;
         RaPortRandom[i] = i;
      }

      RaPortMapping[RA_MAX_PORT] = RA_MAX_PORT;
      RaPortMapping[RA_MAX_PORT] = RA_MAX_PORT;

      if (RaPreserveWellKnownPorts) {
         if (RaPreserveRegisteredPorts) {
            if (RaPreservePrivatePorts) {
                  startport = RA_MAX_PORT - 1;
            } else
               startport = 49152;
         } else
            startport = 1024;
      } else
         startport = 1;

      if (!(strncmp(RaPortMappingOffset, "random", 6))) {
         for (i = startport, start = startport; i < RA_MAX_PORT; i++) {
               if (shiftcount > 5) {
                  cnt = 0;
                  shiftcount = 0;
                  y = start;
                  bzero ((char *) RaPortTemp, sizeof(RaPortTemp));
                  for (x = i; x < RA_MAX_PORT; x++) {
                     while ((RaPortRandom[y] == 0) && (y < RA_MAX_PORT)) y++;
                     if (RaPortRandom[y]) {
                        RaPortTemp[x] = RaPortRandom[y];
                        RaPortRandom[y] = 0;
                        cnt++;
                     } else
                        break;
                  }
                  bcopy ((char *)RaPortTemp, (char *) RaPortRandom, sizeof(RaPortRandom));
                  start = i;
               }

               retn = i + (random() % (RA_MAX_PORT - i));
               ind = retn;
               while (RaPortRandom[ind] == 0) {
                  if (retn & 0x01) {
                     if (ind >= start) {
                        ind--;
                     } else {
                        ind = retn++;
                        shiftcount++;
                     }
                  } else {
                     if (ind <= RA_MAX_PORT) {
                        ind++;
                     } else {
                        ind = retn--;
                        shiftcount++;
                     }
                  }
               }

               if ((value = RaPortRandom[ind])) {
                  RaPortMapping[i] = value;
                  RaPortRandom[ind] = 0;
                  count++;
               }
         }
      } else {
         if (!(strncmp(RaPortMappingOffset, "offset:random", 13))) {
            RaPortMappingOffsetValue = (random() % RA_MAX_PORT);
         } else 
         if (!(strncmp(RaPortMappingOffset, "offset:fixed", 12))) {
            if ((ptr = strchr(RaPortMappingOffset, ':')) != NULL)
               RaPortMappingOffsetValue = atoi(++ptr);
            else
               ArgusLog(LOG_ERR, "RANON_PORT_METHOD syntax error\n");
         } else 
         if (!(strncmp(RaPortMappingOffset, "no", 2)))
            RaPortMappingOffsetValue = 0;
         else
            ArgusLog(LOG_ERR, "RANON_PORT_METHOD syntax error\n");

         if (RaPortMappingOffsetValue > 0) {
            for (i = startport, start = startport; i < RA_MAX_PORT; i++) {
               RaPortMapping[i] += RaPortMappingOffsetValue;
               if (RaPortMapping[i] < startport)
                  RaPortMapping[i] += startport;
            }
         }
      }

      setuid(getuid());
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusClientInit: returning\n");
#endif
}

int RaParseCompleting = 0;
int RaMapMacCounter = 0;
int RaMapMacMultiCastCounter = 0;
int RaMapClassACounter = 0;
int RaMapClassBCounter = 0;
int RaMapClassCCounter = 0;
int RaMapClassANetCounter = 0;
int RaMapClassBNetCounter = 0;
int RaMapClassCNetCounter = 0;
int RaMapMultiCastCounter = 0;

struct ArgusListStruct *RaMapEtherAddrList = NULL;
struct ArgusListStruct *RaMapIPAddrList = NULL;


void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;
/*
      printf ("Total Records        %9d\n", totalrecords);
      printf ("Total EtherNet               Addrs  %8d\n", RaMapMacCounter);
      printf ("Total EtherNet Multicast     Addrs  %8d\n", RaMapMacMultiCastCounter);
      printf ("Total Class A Nets   %5d   Addrs  %8d\n", RaMapClassANetCounter, RaMapClassACounter);
      printf ("Total Class B Nets   %5d   Addrs  %8d\n", RaMapClassBNetCounter, RaMapClassBCounter);
      printf ("Total Class C Nets   %5d   Addrs  %8d\n", RaMapClassCNetCounter, RaMapClassCCounter);
      printf ("Total Multicast              Addrs  %8d\n\n", RaMapMultiCastCounter);
      printf ("Distributing %8d Hosts  into %6d Nets\n",
                     (RaMapClassACounter    + RaMapClassBCounter    + RaMapClassCCounter),
                     (RaMapClassANetCounter + RaMapClassBNetCounter + RaMapClassCNetCounter));
   
      if (RaMapEtherAddrList) {
         struct RaMapHashTableHeader *retn = NULL;

         while (!ArgusListEmpty (RaMapEtherAddrList)) {
            if ((retn = ArgusFrontList(RaMapEtherAddrList)) != NULL) {
               fprintf (stdout, " Address %17s  ",      etheraddr_string (retn->obj));
               fprintf (stdout, " Converted to %17s\n", etheraddr_string (retn->sub));
               ArgusPopFrontList(RaMapEtherAddrList);
            }
         }
      }

      if (RaMapIPAddrList) {
         struct RaMapHashTableHeader *retn = NULL;

         while (!ArgusListEmpty (RaMapIPAddrList)) {
            if ((retn = ArgusFrontList(RaMapIPAddrList)) != NULL) {
               fprintf (stdout, " Address %17s  ",      ipaddr_string (retn->obj));
               fprintf (stdout, " Converted to %17s\n", ipaddr_string (retn->sub));
               ArgusPopFrontList(RaMapIPAddrList);
            }
         }
      }
*/
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "RaParseComplete: returning\n");
#endif
}

void
ArgusClientTimeout ()
{
#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusClientTimeout: returning\n");
#endif
}

void
parse_arg (int argc, char**argv)
{ 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "parse_arg: returning\n");
#endif
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Ratemplate Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -a                print record summaries on termination.\n");
   fprintf (stderr, "         -A                print application bytes.\n");
   fprintf (stderr, "         -b                dump packet-matching code.\n");
   fprintf (stderr, "         -C                treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -D <level>        specify debug level\n");
   fprintf (stderr, "         -E <file>         write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -f <flowfile>     read flow model from <flowfile>.\n");
   fprintf (stderr, "         -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                print help.\n");
   fprintf (stderr, "         -n                don't convert numbers to names.\n");
   fprintf (stderr, "         -p <digits>       print fractional time with <digits> precision.\n");
   fprintf (stderr, "         -q                quiet mode. don't print record outputs.\n");
   fprintf (stderr, "         -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -R                print out response data when availabile.\n");
   fprintf (stderr, "         -S <host[:port]>  specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "                                               -%%d{yMhdms}\n");
   fprintf (stderr, "         -T <secs>         attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u                print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>    specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>         write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "         -z                print Argus TCP state changes.\n");
   fprintf (stderr, "         -Z <s|d|b>        print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


#define RAMAP_ETHER_MAC_ADDR		0x1
#define RAMAP_IP_ADDR      		0x10
int RaFirstRecord = 1;

void
RaProcessRecord (struct ArgusRecord *argus)
{
   struct ArgusRecordStore *store = NULL;

   if (RaFirstRecord) {
      ArgusNtoH(&ArgusInput->ArgusInitCon);
      RaMapInventory(&ArgusInput->ArgusInitCon.argus_mar.localnet, RAMAP_IP_ADDR, 4);
      RaMapInventory(&ArgusInput->ArgusInitCon.argus_mar.argusid, RAMAP_IP_ADDR, 4);

      ArgusInput->ArgusInitCon.argus_mar.startime.tv_sec  -= RaMapTimeSecOffsetValue;
      ArgusInput->ArgusInitCon.argus_mar.startime.tv_usec -= RaMapTimeuSecOffsetValue;
      if (ArgusInput->ArgusInitCon.argus_mar.startime.tv_usec < 0) {
         ArgusInput->ArgusInitCon.argus_mar.startime.tv_sec--;
         ArgusInput->ArgusInitCon.argus_mar.startime.tv_usec += 1000000;
      }

      ArgusInput->ArgusInitCon.argus_mar.now.tv_sec  -= RaMapTimeSecOffsetValue;
      ArgusInput->ArgusInitCon.argus_mar.now.tv_usec -= RaMapTimeuSecOffsetValue;
      if (ArgusInput->ArgusInitCon.argus_mar.now.tv_usec < 0) {
         ArgusInput->ArgusInitCon.argus_mar.now.tv_sec--;
         ArgusInput->ArgusInitCon.argus_mar.now.tv_usec += 1000000;
      }

      ArgusHtoN(&ArgusInput->ArgusInitCon);
      RaFirstRecord = 0;
   }

   argus->ahdr.status |= ARGUS_ANON;
   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {
      struct ArgusRecord *targus = NULL;

      if ((targus = RaConstructArgusRecord(argus)) != NULL)
         argus = targus;
      else
         ArgusLog (LOG_ERR, "RaProcessRecord: RaConstructArgusRecord(0x%x) %s",
                       argus, strerror(errno));

      ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

      RaMapInventory (&argus->ahdr.argusid, RAMAP_IP_ADDR, 4);
      argus->ahdr.seqNumber += RaMapSequenceOffsetValue;

      argus->argus_far.ArgusTransRefNum += RaMapTransOffsetValue;
      argus->argus_far.time.start.tv_sec  -= RaMapTimeSecOffsetValue;
      argus->argus_far.time.start.tv_usec -= RaMapTimeuSecOffsetValue;
      if (argus->argus_far.time.start.tv_usec < 0) {
         argus->argus_far.time.start.tv_sec--;
         argus->argus_far.time.start.tv_usec += 1000000;
      }

      argus->argus_far.time.last.tv_sec  -= RaMapTimeSecOffsetValue;
      argus->argus_far.time.last.tv_usec -= RaMapTimeuSecOffsetValue;

      if (argus->argus_far.time.last.tv_usec < 0) {
         argus->argus_far.time.last.tv_sec--;
         argus->argus_far.time.last.tv_usec += 1000000;
      }

      if (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS) {
         struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];

         RaMapInventory (&mac->phys_union.ether.ethersrc, RAMAP_ETHER_MAC_ADDR, 6);
         RaMapInventory (&mac->phys_union.ether.etherdst, RAMAP_ETHER_MAC_ADDR, 6);
      }

      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:

            RaMapInventory (&argus->argus_far.flow.ip_flow.ip_src, RAMAP_IP_ADDR, 4);
            RaMapInventory (&argus->argus_far.flow.ip_flow.ip_dst, RAMAP_IP_ADDR, 4);

            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:

            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }

   if (argus != NULL) {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         if ((store->data[0] = RaNewArgusData(argus)) != NULL) {
            store->data[0]->status |= RA_MODIFIED;
            RaSendArgusRecord (store);
            RaDeleteArgusStore (store);

         } else
            ArgusLog (LOG_ERR, "RaProcessRecord: RaNewArgusData(0x%x) %s\n",
                               argus, strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaProcessRecord: RaNewArgusStore(0x%x) %s\n",
                               argus, strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaProcessRecord (0x%x) returning\n", argus);
#endif
}


struct RaMapHashTableHeader *RaMapAllocateEtherAddr (struct ether_header *, int, int);
struct RaMapHashTableHeader *RaMapAllocateIPAddr (unsigned int *, int, int);
struct RaMapHashTableHeader *RaMapAllocateNet (unsigned int, unsigned int);

void
RaProcessManRecord (struct ArgusRecord *argus)
{
   RaMapInventory (&argus->argus_mar.localnet, RAMAP_IP_ADDR, 4);
   RaMapInventory (&argus->argus_mar.argusid, RAMAP_IP_ADDR, 4);

   argus->argus_mar.startime.tv_sec  -= RaMapTimeSecOffsetValue;
   argus->argus_mar.startime.tv_usec -= RaMapTimeuSecOffsetValue;
   if (argus->argus_mar.startime.tv_usec < 0) {
      argus->argus_mar.startime.tv_sec--;
      argus->argus_mar.startime.tv_usec += 1000000;
   }

   argus->argus_mar.now.tv_sec  -= RaMapTimeSecOffsetValue;
   argus->argus_mar.now.tv_usec -= RaMapTimeuSecOffsetValue;
   if (argus->argus_mar.now.tv_usec < 0) {
      argus->argus_mar.now.tv_sec--;
      argus->argus_mar.now.tv_usec += 1000000;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessManRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   if (RaMapConvert) {
      struct ArgusTCPObject *tcp = NULL;

      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            tcp->src.seqbase += RaMapSequenceOffsetValue;
            tcp->dst.seqbase += RaMapSequenceOffsetValue;
         }
      }

      argus->argus_far.flow.ip_flow.sport = RaPortMapping[argus->argus_far.flow.ip_flow.sport];
      argus->argus_far.flow.ip_flow.dport = RaPortMapping[argus->argus_far.flow.ip_flow.dport];
      argus->argus_far.flow.ip_flow.ip_id += RaMapIpIdValue;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessTCPRecord (0x%x) returning\n", argus);
#endif
}

#if defined(__OpenBSD__)
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#endif

#include <netinet/ip_icmp.h>

void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   struct ArgusICMPFlow *icmpFlow = &argus->argus_far.flow.icmp_flow;
   struct ArgusICMPObject *icmp = NULL;
   unsigned char ra_icmp_type = 0, ra_icmp_code = 0;
   
   if (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS) {
      icmp = (struct ArgusICMPObject *)ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];
      RaMapInventory (&icmp->isrcaddr, RAMAP_IP_ADDR, 4);
      RaMapInventory (&icmp->idstaddr, RAMAP_IP_ADDR, 4);
      RaMapInventory (&icmp->igwaddr, RAMAP_IP_ADDR, 4);
      icmp->iseq += (unsigned short) RaMapSequenceOffsetValue;
      ra_icmp_type = icmp->icmp_type;
      ra_icmp_code = icmpFlow->code;

   } else {
      ra_icmp_type = icmpFlow->type;
      ra_icmp_code = icmpFlow->code;
   }

   if ((ra_icmp_type == ICMP_UNREACH) && (ra_icmp_code == ICMP_UNREACH_PORT))
      icmpFlow->id = RaPortMapping[icmpFlow->id];

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessICMPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessUDPRecord (struct ArgusRecord *argus)
{
   argus->argus_far.flow.ip_flow.sport = RaPortMapping[argus->argus_far.flow.ip_flow.sport];
   argus->argus_far.flow.ip_flow.dport = RaPortMapping[argus->argus_far.flow.ip_flow.dport];

   argus->argus_far.flow.ip_flow.ip_id += RaMapIpIdValue;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessUDPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   argus->argus_far.flow.ip_flow.ip_id += RaMapIpIdValue;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessIPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessARPRecord (struct ArgusRecord *argus)
{
   RaMapInventory (&argus->argus_far.flow.arp_flow.arp_spa, RAMAP_IP_ADDR, 4);
   RaMapInventory (&argus->argus_far.flow.arp_flow.arp_tpa, RAMAP_IP_ADDR, 4);
   RaMapInventory (&argus->argus_far.flow.arp_flow.etheraddr, RAMAP_ETHER_MAC_ADDR, 6);
   RaMapInventory (&argus->argus_far.attr_arp.response, RAMAP_ETHER_MAC_ADDR, 6);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessARPRecord (0x%x) returning\n", argus);
#endif
}

void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessNonIPRecord (0x%x) returning\n", argus);
#endif
}

#include <math.h>
int RaLabelCounter = 0;

int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   int retn = 0;

   if ((data = store->data[0]) != NULL) {
      if ((argus = data->argus) == NULL)
         return (retn);

      if ((Lflag > 0) && !(qflag)) {
         if (RaLabel == NULL)
            RaLabel = RaGenerateLabel(argus);

         if (!(RaLabelCounter++ % Lflag)) {
            if (!(wfile))
               printf ("%s\n", RaLabel);
            fflush (stdout);
         }

         if (Lflag < 0)
            Lflag = 0;
      }

      if ((ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusWfileList)))) {
         struct ArgusWfileStruct *wfile = NULL, *start = NULL;
 
         if ((wfile = ArgusFrontList(ArgusWfileList)) != NULL) { 
            start = wfile; 
            do { 
               if ((exceptfile == NULL) || strcmp(wfile->filename, exceptfile)) { 
#ifdef _LITTLE_ENDIAN
                  ArgusHtoN(argus); 
#endif
                  ArgusWriteNewLogfile (wfile, argus); 
 
#ifdef _LITTLE_ENDIAN
                  ArgusNtoH(argus); 
#endif
               }   
               ArgusPopFrontList(ArgusWfileList); 
               ArgusPushBackList(ArgusWfileList, wfile); 
               wfile = ArgusFrontList(ArgusWfileList); 

            } while (wfile != start); 
         }   

      } else {

         if (!(argus->ahdr.type & ARGUS_MAR))
            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

         printf ("%s\n", get_argus_string (argus));
         fflush (stdout);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif

   return (retn);
}


#ifndef RaMap
#define RaMap
#endif

#ifndef RaMapUtil
#define RaMapUtil
#endif

#include <unistd.h>
#include <stdlib.h>

#include <arpa/inet.h>


#define MAXSTRSIZE 1024
#define MAX_OBJ_SIZE		1024

 
struct RaMapHashTableStruct RaMapHashTable;

#ifndef IN_CLASSD
#define IN_CLASSD(i) (((int32_t)(i) & 0xf0000000) == 0xe0000000)
#endif

int RaPreserveEtherVendor = 0;
int RaPreserveIPHierarchy = 1;
int RaPreserveNoHierarchy = 0;

unsigned int RaMapEtherAddrs = 2;
unsigned int RaMapIPAddrs = 2;


#define ETHER_INTERNET_BROADCAST	0x01005E00
#define RA_ETHER_VENDOR			0x00000001

struct RaMapHashTableHeader *RaMapFindHashObject (struct RaMapHashTableStruct *, void *, int, int);
struct RaMapHashTableHeader *RaMapAddHashEntry (struct RaMapHashTableStruct *, void *, int, int);
void RaMapRemoveHashEntry (struct RaMapHashTableStruct *, struct RaMapHashTableHeader *);
unsigned int RaMapCalcHash (void *, int, int);


struct RaMapHashTableHeader *
RaMapAllocateEtherAddr (struct ether_header *src, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL;
   struct ether_header dstbuf, *dst = &dstbuf;
   unsigned int RaMapEtherVendors = htonl(RA_ETHER_VENDOR);
   unsigned int tmp = htonl(ETHER_INTERNET_BROADCAST);
   unsigned char *ptr = (unsigned char *) src;
   int i = 0, broadcast = 1;
   
   if (src) {
      for (i = 0; i < 6; i++)
         if (ptr[i] != (unsigned char) 0xff)
            broadcast = 0;

      if (broadcast) {
         bcopy ((char *) src, (char *) dst, 6);

      } else {
         if (RaPreserveEtherVendor || (!(bcmp (((char *)src), (char *)&tmp, 3)))) {
            bcopy ((char *) src, (char *) dst, 3);
         } else {
            for (i = 0; i < (sizeof (int) - 1); i++)
               ((char *)dst)[i] = ((char *)&RaMapEtherVendors)[i + 1];
         }

         ptr = &((unsigned char *) dst)[3];

         RaMapEtherAddrs = ntohl(RaMapEtherAddrs);

         for (i = 0; i < (sizeof (int) - 1); i++)
            ptr[i] = ((char *)&RaMapEtherAddrs)[i + 1];

         ptr = (unsigned char *) src;

         if (ptr[5] & 0x01)
            ((unsigned char *) dst)[5] |= 0x01;

         RaMapEtherAddrs = htonl(RaMapEtherAddrs);

         RaMapEtherAddrs += 2;
      }

      if (!(retn = RaMapAddHashEntry (&RaMapHashTable, src, type, len))) {
         ArgusLog (LOG_ERR, "RaMapInventory: RaMapAddHashEntry error %s\n", strerror(errno));
      } else {
         if (((char *)src)[0] & 0x01) {
            RaMapMacMultiCastCounter++;
#ifdef ARGUSDEBUG
            ArgusDebug (2, "RaMapInventory () adding multicast addr %s\n", etheraddr_string ((char *)src));
#endif
         } else {
            RaMapMacCounter++;
#ifdef ARGUSDEBUG
            ArgusDebug (2, "RaMapInventory () adding %s\n", etheraddr_string ((char *)src));
#endif
         }

         if ((retn->sub = (void *) ArgusCalloc (1, len)) != NULL)
            bcopy ((char *) dst, (char *) retn->sub, len);
      }

#ifdef ARGUSDEBUG
   {
      char sbuf1[32], sbuf2[32];
      sprintf (sbuf1, "%s", etheraddr_string ((void *) dst));
      sprintf (sbuf2, "%s", etheraddr_string ((void *) src));
      ArgusDebug (2, "RaMapAllocateEtherAddr allocating %s for %s", sbuf1, sbuf2);
   }
#endif
   }

   return (retn);
}

/*
   So available class A addresses for our algorithm are
   [0 1 2 5 10 23 27 31 36 37 39 41 42 58-60 69-79 82-127 197 220-223 240-254]
   we will start with 1.0.0 and move up to 254.0.0
*/

#define RA_MAX_CLASS_VALUE	65535
#define RA_MAX_CLASS		109
int RaClassIndex = 0;

int RaClassAddress[RA_MAX_CLASS] =
{  1,2,5,10,23,27,31,36,37,39,
   41,42,58,59,60,69,70,71,72,73,
   74,75,76,77,78,79,82,83,84,85,
   86,87,88,89,90,91,92,93,94,95,
   96,97,98,99,
   100,101,102,103,104,105,106,107,108,109,
   110,111,112,113,114,115,116,117,118,119,
   120,121,122,123,124,125,126,127, 197,
   220,221,222,223,224,225,226,227,228,229,
   230,231,232,233,234,235,236,237,238,239,
   240,241,242,243,244,245,246,247,248,249,
   250,251,252,253,254,255,
};

#define RA_MAX_CLASS_A		44
int RaClassAIndex = 0;

int RaClassAAddress[RA_MAX_CLASS_A] =
{  1,2,5,10,23,27,31,36,37,39,
   41,42,58,59,60,69,70,71,72,73,
   74,75,76,77,78,79,82,83,84,85,
   86,87,88,89,90,91,92,93,94,95,
   96,97,98,99,
};

#define RA_MAX_CLASS_B		28
int RaClassBIndex = 0;

int RaClassBAddress[RA_MAX_CLASS_B] = 
{  100,101,102,103,104,105,106,107,108,109,
   110,111,112,113,114,115,116,117,118,119,
   120,121,122,123,124,125,126,127,
};

#define RA_MAX_CLASS_C		21
int RaClassCIndex = 0;

int RaClassCAddress[RA_MAX_CLASS_C] = 
{  197,220,221,222,223,240,241,242,243,244,
   245,246,247,248,249,250,251,252,253,254,
   255,
};

#define RA_MAX_CLASS_M		16
int RaClassMIndex = 0;

int RaClassMAddress[RA_MAX_CLASS_M] = 
{  
   224,225,226,227,228,229,230,231,232,233,
   234,235,236,237,238,239,
};


/* in comes a request for a network alias */

#ifndef IN_CLASSD_NET
#define IN_CLASSD_NET	0xffffffff
#endif

struct RaMapHashTableHeader *RaMapNewNetwork (unsigned int, unsigned int);

unsigned int RaMapLocalNetwork = 0;

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


#define RAMAP_CLASSA	1
#define RAMAP_CLASSB	2
#define RAMAP_CLASSC	3
#define RAMAP_CLASSM	4

struct RaMapHashTableHeader *
RaMapNewNetwork (unsigned int host, unsigned int mask)
{
   struct RaMapHashTableHeader *retn = NULL;
   struct RaClassNets *thisNet = NULL;
   char buf[MAXSTRLEN];
   struct in_addr addrbuf, *addr = &addrbuf;
   struct hostent *hp;
   unsigned int **p;
   int hostclass;

   if (IN_CLASSA(host)) hostclass = RAMAP_CLASSA; else
   if (IN_CLASSB(host)) hostclass = RAMAP_CLASSB; else
   if (IN_CLASSC(host)) hostclass = RAMAP_CLASSC; else
   if (IN_CLASSD(host)) hostclass = RAMAP_CLASSM; else
   hostclass = RAMAP_CLASSC;

   bzero (buf, MAXSTRLEN);
   do {
      if (RaMapNetAddrHierarchy > RANON_SUBNET) {
         switch (hostclass) {
            case RAMAP_CLASSA:
               sprintf (buf, "%d.0.0.0", RaClassAAddress[RaClassAIndex]);
               break;
            case RAMAP_CLASSB:
               sprintf (buf, "%d.0.0.0", RaClassBAddress[RaClassBIndex]);
               break;
            default:
            case RAMAP_CLASSC:
               sprintf (buf, "%d.0.0.0", RaClassCAddress[RaClassCIndex]);
               break;
            case RAMAP_CLASSM:
               sprintf (buf, "%d.0.0.0", RaClassMAddress[RaClassMIndex]);
               break;
         }
      } else
         sprintf (buf, "%d.0.0.0", RaClassAddress[RaClassIndex]);
   
      if ((hp = gethostbyname(buf)) != NULL) {
         for (p = (unsigned int **)hp->h_addr_list; *p; ++p)
            **p = ntohl(**p);
      } else
         ArgusLog (LOG_ERR, "RaMapNewNetwork: gethostbyname(%s) error %s", optarg, strerror(errno));
   
      addr->s_addr = **(unsigned int **)hp->h_addr_list;
      if ((retn = RaMapFindHashObject (&RaMapNetTable, &addr->s_addr, RAMAP_IP_ADDR, 4))) {
         if (retn->net->index >= RA_MAX_CLASS_VALUE) {
            if (RaMapNetAddrHierarchy > RANON_SUBNET) {
               switch (hostclass) {
                  case RAMAP_CLASSA:
                     if (RaClassAIndex < RA_MAX_CLASS_A) {
                        RaClassAIndex++;
                        break;
                     } else
                        hostclass = RAMAP_CLASSB;
                  case RAMAP_CLASSB:
                     if (RaClassBIndex < RA_MAX_CLASS_B) {
                        RaClassBIndex++;
                        break;
                     } else
                        hostclass = RAMAP_CLASSC;
                  case RAMAP_CLASSC:
                     if (RaClassCIndex < RA_MAX_CLASS_C) {
                        RaClassCIndex++;
                        break;
                     } else
                        ArgusLog (LOG_ERR, "RaMapNewNetwork: no addresses");

                  case RAMAP_CLASSM:
                     if (RaClassMIndex < RA_MAX_CLASS_M) {
                        RaClassMIndex++;
                     } else
                        ArgusLog (LOG_ERR, "RaMapNewNetwork: no multicast addresses");
               }
            }
         }
      }

   } while (retn && (retn->net->index >= RA_MAX_CLASS_VALUE));

   if (!(retn)) {
      if ((retn = RaMapAddHashEntry (&RaMapNetTable, &addr->s_addr, RAMAP_IP_ADDR, 4))) {
         if ((thisNet = (void *) ArgusCalloc (1, sizeof(struct RaClassNets))) != NULL) {
            thisNet->net = addr->s_addr;
            thisNet->index = 1;
            retn->net = thisNet;

            if ((retn->sub = (void *) ArgusCalloc (1, sizeof(addr->s_addr))) != NULL)
               bcopy ((char *)&addr->s_addr, (char *) retn->sub, sizeof(addr->s_addr));
            else
               ArgusLog (LOG_ERR, "RaMapNewNetwork: ArgusCalloc() error %s", strerror(errno));

         } else
            ArgusLog (LOG_ERR, "RaMapNewNetwork: ArgusCalloc() error %s", strerror(errno));
      } else
         ArgusLog (LOG_ERR, "RaMapNewNetwork: RaMapAddHashEntry() error %s", strerror(errno));
   }

#ifdef ARGUSDEBUG
   {
      char sbuf1[32], sbuf2[32];
      sprintf (sbuf1, "%s", ipaddr_string(&host));
      sprintf (sbuf2, "%s", ipaddr_string(&mask));
      ArgusDebug (2, "RaMapNewNetwork (%s, %s) returns 0x%x\n", sbuf1, sbuf2, retn);
   }
#endif
   return (retn);
}


struct RaMapHashTableHeader *
RaMapAllocateNet (unsigned int addr, unsigned int mask)
{
   struct RaMapHashTableHeader *retn = NULL;
   struct RaMapHashTableHeader *net = NULL;
   unsigned int newaddr = 0;
   char buf[MAXSTRLEN];

   bzero (buf, MAXSTRLEN);
   if (!(retn = RaMapFindHashObject (&RaMapNetTable, &addr, RAMAP_IP_ADDR, 4))) {
      switch (mask) {
         case IN_CLASSA_NET:
            break;

         case IN_CLASSB_NET:
            if (RaMapNetAddrHierarchy == RANON_CIDR) {
               return (RaMapAllocateNet (addr & IN_CLASSA_NET, IN_CLASSA_NET));
               break;
            }
            break;

         case IN_CLASSC_NET:
            if (RaMapNetAddrHierarchy == RANON_CIDR) {
               return (RaMapAllocateNet (addr & IN_CLASSB_NET, IN_CLASSB_NET));
               break;
            }
            break;

         case IN_CLASSD_NET:
            return (RaMapAllocateNet (addr & IN_CLASSC_NET, IN_CLASSC_NET));
            break;
      }

      if (!(RaMapNetAddrHierarchy)) {
         
      } 

      if (!(net) || (net->net->index >= RA_MAX_CLASS_VALUE))
         net = RaMapNewNetwork(addr, mask);

      if ((retn = RaMapAddHashEntry (&RaMapNetTable, (void *)&addr, RAMAP_IP_ADDR, 4))) {
         if ((retn->net = (void *) ArgusCalloc (1, sizeof(struct RaClassNets))) != NULL) {
            retn->net->supernet = net->net;
            newaddr = net->net->net | (net->net->index++ << 8);
            retn->net->net = newaddr;
            retn->net->index = 1;
         }


         if (!(retn->sub)) {
            if ((retn->sub = (void *) ArgusCalloc (1, 4)) != NULL) {
               bcopy ((char *)&newaddr, (char *) retn->sub, 4);
            } else {
               ArgusLog (LOG_ERR, "RaMapAllocateNet: ArgusCalloc error %s", strerror(errno));
            }
#ifdef ARGUSDEBUG
            {
               char sbuf1[32], sbuf2[32], sbuf3[32];
               sprintf (sbuf1, "%s", ipaddr_string(&addr));
               sprintf (sbuf2, "%s", ipaddr_string(&mask));
               sprintf (sbuf3, "%s", ipaddr_string(&newaddr));
               ArgusDebug (2, "RaMapAllocateNet (%s, %s) maps to %s\n", sbuf1, sbuf2, sbuf3);
            }
#endif
         }
         
      } else
         ArgusLog (LOG_ERR, "RaMapAllocateNet () RaMapAddHashEntry error %s\n", strerror(errno));
   }

#ifdef ARGUSDEBUG
   {
      char sbuf1[32], sbuf2[32];
      sprintf (sbuf1, "%s", ipaddr_string(&addr));
      sprintf (sbuf2, "%s", ipaddr_string(&mask));
      ArgusDebug (2, "RaMapAllocateNet (%s, %s) return 0x%x\n", sbuf1, sbuf2, retn);
   }
#endif

   return (retn);
}


struct RaMapHashTableHeader *
RaMapAllocateIPAddr (unsigned int *addr, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL, *net = NULL;
   unsigned int newaddr = 0;

   if (!(retn = RaMapFindHashObject (&RaMapHashTable, addr, type, len))) {
      if ((net = RaMapAllocateNet (*addr, IN_CLASSD_NET))) {
         if (((*addr & 0xff) == 0xff) && RaPreserveBroadcastAddress)
            newaddr = *((unsigned int *)net->sub) | 0xFF;
         else
         if (((*addr & 0xff) == 0x00) && RaPreserveBroadcastAddress)
            newaddr = *((unsigned int *)net->sub);
         else
            newaddr = *((unsigned int *)net->sub) + ((struct RaClassNets *)net->net)->index++;

         if ((retn = RaMapAddHashEntry (&RaMapHashTable, (void *)addr, type, len))) {
            retn->net = net->net;
            if ((retn->sub = (void *) ArgusCalloc (1, 4)) != NULL) {
               bcopy ((char *)&newaddr, (char *) retn->sub, 4);
            } else {
               ArgusLog (LOG_ERR, "RaMapAllocateNet: ArgusCalloc error %s", strerror(errno));
            }
         } else
            ArgusLog (LOG_ERR, "RaAllocateIPAddr () RaMapAddHashEntry error %s\n", strerror(errno));

      } else
         ArgusLog (LOG_ERR, "RaAllocateIPAddr () RaMapAllocateNet failed.");
   } else
      newaddr = *(unsigned int *)retn->sub;

#ifdef ARGUSDEBUG
   {
      char sbuf1[32], sbuf2[32];
      sprintf (sbuf1, "%s", ipaddr_string(addr));
      sprintf (sbuf2, "%s", ipaddr_string(&newaddr));
      ArgusDebug (2, "RaAllocateIPAddr (%s, %d, %d) converts to %s\n", sbuf1, type, len, sbuf2);
   }
#endif

   return (retn);
}


void
RaMapInventory (void *oid, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL;

   switch (type) {
      case RAMAP_ETHER_MAC_ADDR: {
         if (!(retn = RaMapFindHashObject (&RaMapHashTable, oid, type, len))) {
            if (!(retn = RaMapAllocateEtherAddr (oid, type, len))) 
               ArgusLog (LOG_ERR, "RaMapInventory: RaMapAllocateEtherAddr error %s\n", strerror(errno));

            if (!(RaMapEtherAddrList))
               RaMapEtherAddrList = ArgusNewList();

            ArgusPushBackList(RaMapEtherAddrList, retn);
         }

         if (RaMapConvert)
            if (retn->sub)
               bcopy ((char *)retn->sub, (char *) oid, len);
      
         break;     
      }

      case RAMAP_IP_ADDR: {
         if (!(retn = RaMapFindHashObject (&RaMapHashTable, oid, type, len))) {
            if (!(retn = RaMapAllocateIPAddr (oid, type, len)))
               ArgusLog (LOG_ERR, "RaMapInventory: RaMapAllocateIPAddr error %s\n", strerror(errno));
            
            if (!(RaMapIPAddrList))
               RaMapIPAddrList = ArgusNewList();

            ArgusPushBackList(RaMapIPAddrList, retn);
         }

         if (RaMapConvert)
            if (retn->sub)
               bcopy ((char *)retn->sub, (char *) oid, len);

         break;     
      }
   }
 
#ifdef ARGUSDEBUG
   ArgusDebug (3, "RaMapInventory (0x%x, %d, %d) returning\n", oid, type, len);
#endif
}


unsigned int
RaMapCalcHash (void *obj, int type, int len)
{
   u_char buf[MAX_OBJ_SIZE];
   unsigned int retn = 0;

   switch (type) {
      case RAMAP_ETHER_MAC_ADDR:
          len = 6;
          break;

      case RAMAP_IP_ADDR:
          len = 4;
          break;

      default:
          break;
   }

   bzero (buf, sizeof buf);

   if (RaHashSize <= 0x100) {
      unsigned char hash = 0, *ptr = (unsigned char *) buf;
      int i, nitems = len;

      bcopy ((char *) obj, (char *) &buf, len);

      for (i = 0; i < nitems; i++)
         hash += *ptr++;

      retn = hash;

   } else
   if (RaHashSize <= 0x10000) {
      unsigned short hash = 0, *ptr = (unsigned short *) buf;
      int i, nitems = (len / sizeof(unsigned short)) + 2;

      bcopy ((char *) obj, &buf[1], len);

      for (i = 0; i < nitems; i++)
         hash += *ptr++;

      retn = hash;

   } else {
      unsigned int hash = 0, *ptr = (unsigned int *) buf;
      int i, nitems = (len /sizeof(unsigned int)) + 2;

      bcopy ((char *) obj, &buf[3], len);

      for (i = 0; i < nitems; i++)
         hash += *ptr++;

      retn = hash;
   }

   return (retn);
}



struct RaMapHashTableHeader *
RaMapFindHashObject (struct RaMapHashTableStruct *table, void *obj, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL, *head = NULL, *target;
   int RaMapHash = 0;

   RaMapHash = RaMapCalcHash (obj, type, len);

   if ((target = table->array[RaMapHash % table->size]) != NULL) {
      head = target;
      do {
         if ((type == target->type) && (len == target->len)) {
            if (!(bcmp ((char *) obj, (char *) target->obj, len))) {
               retn = target;
               break;
            }
         }

         target = target->nxt;
      } while (target != head);
   }

#ifdef TCPCLEANDEBUG
   RaMapDebug (6, "RaMapFindHashEntry () returning 0x%x RaMapHash %d\n", retn, RaMapHash);
#endif
 
   return (retn);
}


struct RaMapHashTableHeader *
RaMapAddHashEntry (struct RaMapHashTableStruct *table, void *oid, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL, *start = NULL;

   if ((retn = (struct RaMapHashTableHeader *) ArgusCalloc (1, sizeof (struct RaMapHashTableHeader))) != NULL) {
      RaMapHash = RaMapCalcHash (oid, type, len);

      retn->hash = RaMapHash;
      retn->type = type;
      retn->len  = len;

      if ((retn->obj = (void *) ArgusCalloc (1, len)) == NULL)
         ArgusLog (LOG_ERR, "RaMapAddHashEntry: ArgusCalloc error %s\n", strerror(errno));
      else
         bcopy ((char *) oid, (char *)retn->obj, len);
      
      if ((start = table->array[RaMapHash % table->size]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      table->array[RaMapHash % table->size] = retn;
   }

#ifdef TCPCLEANDEBUG
   RaMapDebug (3, "RaMapAddHashEntry (0x%x, %d, %d) returning 0x%x\n", oid, type, len, retn);
#endif

   return (retn);
}

 
void
RaMapRemoveHashEntry (struct RaMapHashTableStruct *table, struct RaMapHashTableHeader *htblhdr)
{
   unsigned short hash = htblhdr->hash;

   htblhdr->prv->nxt = htblhdr->nxt;
   htblhdr->nxt->prv = htblhdr->prv;

   if (htblhdr == table->array[hash % table->size]) {
      if (htblhdr == htblhdr->nxt)
         table->array[hash % table->size] = NULL;
      else
         table->array[hash % table->size] = htblhdr->nxt;
   }

   ArgusFree (htblhdr);

#ifdef TCPCLEANDEBUG
   RaMapDebug (6, "RaMapRemoveHashEntry (0x%x) returning\n", htblhdr);
#endif
}


#define RANON_RCITEMS				23

#define RANON_TRANSREFNUM_OFFSET 		0
#define RANON_SEQNUM_OFFSET 			1
#define RANON_TIME_SEC_OFFSET 			2
#define RANON_TIME_USEC_OFFSET 			3
#define RANON_NET_ANONYMIZATION			4
#define RANON_HOST_ANONYMIZATION		5
#define RANON_PRESERVE_NET_ADDRESS_HIERARCHY	6
#define RANON_PRESERVE_BROADCAST_ADDRESS	7
#define RANON_SPECIFY_NET_TRANSLATION		8
#define RANON_SPECIFY_HOST_TRANSLATION		9
#define RANON_PRESERVE_WELLKNOWN_PORT_NUMS	10
#define RANON_PRESERVE_REGISTERED_PORT_NUMS	11
#define RANON_PRESERVE_PRIVATE_PORT_NUMS	12
#define RANON_PRESERVE_PORT_NUMS		13
#define RANON_PRESERVE_PORT_NUM			14
#define RANON_PRESERVE_IP_ID			15
#define RANON_CLASSA_NET_ADDRESS_LIST		16
#define RANON_CLASSB_NET_ADDRESS_LIST		17
#define RANON_CLASSC_NET_ADDRESS_LIST		18
#define RANON_CLASSM_NET_ADDRESS_LIST		19
#define RANON_MAP_DUMPFILE			20
#define RANON_PRESERVE_ETHERNET_VENDOR		21
#define RANON_PORT_METHOD			22


char *RaNonResourceFileStr [] = {
   "RANON_TRANSREFNUM_OFFSET=",
   "RANON_SEQNUM_OFFSET=",
   "RANON_TIME_SEC_OFFSET=",
   "RANON_TIME_USEC_OFFSET=",
   "RANON_NET_ANONYMIZATION=",
   "RANON_HOST_ANONYMIZATION=",
   "RANON_PRESERVE_NET_ADDRESS_HIERARCHY=",
   "RANON_PRESERVE_BROADCAST_ADDRESS=",
   "RANON_SPECIFY_NET_TRANSLATION=",
   "RANON_SPECIFY_HOST_TRANSLATION=",
   "RANON_PRESERVE_WELLKNOWN_PORT_NUMS=",
   "RANON_PRESERVE_REGISTERED_PORT_NUMS=",
   "RANON_PRESERVE_PRIVATE_PORT_NUMS=",
   "RANON_PRESERVE_PORT_NUMS=",
   "RANON_PRESERVE_PORT_NUM=",
   "RANON_PRESERVE_IP_ID=",
   "RANON_CLASSA_NET_ADDRESS_LIST=",
   "RANON_CLASSB_NET_ADDRESS_LIST=",
   "RANON_CLASSC_NET_ADDRESS_LIST=",
   "RANON_CLASSM_NET_ADDRESS_LIST=",
   "RANON_MAP_DUMPFILE=",
   "RANON_PRESERVE_ETHERNET_VENDOR=",
   "RANON_PORT_METHOD=",
};

#include <ctype.h>

int
RaNonParseResourceFile (char *file)
{
   int retn = 0, i, len, found = 0, lines = 0;
   char strbuf[MAXSTRLEN], *str = strbuf, *optarg = NULL;
   FILE *fd;

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         retn = 1;
         while ((fgets(str, MAXSTRLEN, fd)) != NULL)  {
            lines++;
            while (*str && isspace((int)*str))
                str++;

            if (*str && (*str != '#') && (*str != '\n') && (*str != '!')) {
               found = 0;
               for (i = 0; i < RANON_RCITEMS; i++) {
                  len = strlen(RaNonResourceFileStr[i]);
                  if (!(strncmp (str, RaNonResourceFileStr[i], len))) {

                     optarg = &str[len];

                     if (optarg[strlen(optarg) - 1] == '\n')
                        optarg[strlen(optarg) - 1] = '\0';

                     if (*optarg == '\"')
                        optarg++;

                     if (optarg[strlen(optarg) - 1] == '\"')
                        optarg[strlen(optarg) - 1] = '\0';
                        
                     if (*optarg == '\0')
                        optarg = NULL;

                     if (optarg) {
                        switch (i) {
                           case RANON_TRANSREFNUM_OFFSET:
                              RaNonTransRefNumOffset = strdup(optarg);
                              break;

                           case RANON_SEQNUM_OFFSET:
                              RaNonSeqNumOffset = strdup(optarg);
                              break;

                           case RANON_TIME_SEC_OFFSET:
                              RaNonTimeSecOffset = strdup(optarg);
                              break;

                           case RANON_TIME_USEC_OFFSET:
                              RaNonTimeuSecOffset = strdup(optarg);
                              break;

                           case RANON_NET_ANONYMIZATION:
                              break;

                           case RANON_HOST_ANONYMIZATION:
                              break;

                           case RANON_PRESERVE_NET_ADDRESS_HIERARCHY:
                              RaNonPreserveNetAddrHierarchy = strdup(optarg); 
                              break;

                           case RANON_PRESERVE_BROADCAST_ADDRESS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPreserveBroadcastAddress++;
                              else
                                 RaPreserveBroadcastAddress = 0;
                              break;

                           case RANON_SPECIFY_NET_TRANSLATION: {
                              struct RaClassNets *thisNet = NULL;
                              struct RaMapHashTableHeader *mhdr = NULL;
                              struct in_addr oaddrbuf, *oaddr = &oaddrbuf;
                              struct in_addr naddrbuf, *naddr = &naddrbuf;
                              struct hostent *hp;
                              unsigned int **p;
                              char *ptr;

                              if ((ptr = strstr(optarg, "::")) == NULL)
                                 ArgusLog (LOG_ERR, "%s: syntax error line %d RANON_SPECIFY_HOST_TRANSLATION needs '::'\n",
                                                 file, lines);
                              *ptr = '\0';
                              ptr += 2;

                              if ((hp = gethostbyname(optarg)) != NULL) {
                                 for (p = (unsigned int **)hp->h_addr_list; *p; ++p)
                                    **p = ntohl(**p);
                                 oaddr->s_addr = **(unsigned int **)hp->h_addr_list;
                              } else
                                 ArgusLog (LOG_ERR, "RaNonParseResourceFile: gethostbyname(%s) error %s",
                                                optarg, strerror(errno));

                              if ((hp = gethostbyname(ptr)) != NULL) {
                                 for (p = (unsigned int **)hp->h_addr_list; *p; ++p)
                                    **p = ntohl(**p);
                              } else
                                 ArgusLog (LOG_ERR, "RaNonParseResourceFile: gethostbyname(%s) error %s",
                                                ptr, strerror(errno));

                              naddr->s_addr = **(unsigned int **)hp->h_addr_list;

                              if ((mhdr = RaMapAddHashEntry (&RaMapNetTable, &oaddr->s_addr, RAMAP_IP_ADDR, 4))) {
                                 if ((thisNet = (void *) ArgusCalloc (1, sizeof(struct RaClassNets))) != NULL) {
                                    thisNet->net = oaddr->s_addr;
                                    thisNet->index = 1;
                                    mhdr->net = thisNet;

                                    if ((mhdr->sub = (void *) ArgusCalloc (1, sizeof(naddr->s_addr))) != NULL)
                                       bcopy ((char *)&naddr->s_addr, (char *) mhdr->sub, sizeof(naddr->s_addr));
                                    else
                                       ArgusLog (LOG_ERR, "RaMapNewNetwork: ArgusCalloc() error %s", strerror(errno));

                                 } else
                                    ArgusLog (LOG_ERR, "RaMapNewNetwork: ArgusCalloc() error %s", strerror(errno));
                              } else
                                 ArgusLog (LOG_ERR, "RaMapNewNetwork: RaMapAddHashEntry() error %s", strerror(errno));

                              break;
                           }

                           case RANON_SPECIFY_HOST_TRANSLATION: {
                              struct RaMapHashTableHeader *mhdr = NULL;
                              struct in_addr oaddrbuf, *oaddr = &oaddrbuf;
                              struct in_addr naddrbuf, *naddr = &naddrbuf;
                              struct hostent *hp;
                              unsigned int **p;
                              char *ptr;

                              if ((ptr = strstr(optarg, "::")) == NULL)
                                 ArgusLog (LOG_ERR, "%s: syntax error line %d RANON_SPECIFY_HOST_TRANSLATION needs '::'\n",
                                                 file, lines);

                              *ptr = '\0';
                              ptr += 2;

                              if ((hp = gethostbyname(optarg)) != NULL) {
                                 for (p = (unsigned int **)hp->h_addr_list; *p; ++p)
                                    **p = ntohl(**p);
                                 oaddr->s_addr = **(unsigned int **)hp->h_addr_list;
                              } else
                                 ArgusLog (LOG_ERR, "RaNonParseResourceFile: gethostbyname(%s) error %s",
                                                optarg, strerror(errno));

                              if ((hp = gethostbyname(ptr)) != NULL) {
                                 for (p = (unsigned int **)hp->h_addr_list; *p; ++p)
                                    **p = ntohl(**p);
                                 naddr->s_addr = **(unsigned int **)hp->h_addr_list;
                              } else
                                 ArgusLog (LOG_ERR, "RaNonParseResourceFile: gethostbyname(%s) error %s",
                                                ptr, strerror(errno));
#ifdef ARGUSDEBUG
                              ArgusDebug (1, "RaNonParseResourceFile: RANON_SPECIFY_HOST_TRANSLATION=%s::%s\n",
                                                optarg, ptr);
#endif
                              if (!(mhdr = RaMapFindHashObject (&RaMapHashTable, &oaddr->s_addr, RAMAP_IP_ADDR, 4))) {
                                 if ((mhdr = RaMapAddHashEntry (&RaMapHashTable, (void *)&oaddr->s_addr, RAMAP_IP_ADDR, 4))) {
                                    if ((mhdr->sub = (void *) ArgusCalloc (1, 4)) != NULL) {
                                       bcopy ((char *)&naddr->s_addr, (char *) mhdr->sub, 4);
                                    } else {
                                       ArgusLog (LOG_ERR, "RaNonParseResourceFile: ArgusCalloc error %s",
                                                      strerror(errno));
                                    }

                                    if (!(RaMapIPAddrList))
                                       RaMapIPAddrList = ArgusNewList();

                                    ArgusPushBackList(RaMapIPAddrList, mhdr);

                                 } else
                                    ArgusLog (LOG_ERR, "RaAllocateIPAddr () RaMapAddHashEntry error %s\n", strerror(errno));

                              } else
                                 ArgusLog (LOG_ERR, "RaNonParseResourceFile: line %d address %s already allocated\n",
                                                lines, optarg);
                              break;
                           }

                           case RANON_PRESERVE_WELLKNOWN_PORT_NUMS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPreserveWellKnownPorts++;
                              else
                                 RaPreserveWellKnownPorts = 0;
                              break;

                           case RANON_PRESERVE_REGISTERED_PORT_NUMS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPreserveRegisteredPorts++;
                              else
                                 RaPreserveRegisteredPorts = 0;
                              break;

                           case RANON_PRESERVE_PRIVATE_PORT_NUMS:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPreservePrivatePorts++;
                              else
                                 RaPreservePrivatePorts = 0;
                              break;

                           case RANON_PORT_METHOD:
                              RaPortMappingOffset = strdup(optarg);
                              break;

                           case RANON_PRESERVE_PORT_NUMS:
                              break;

                           case RANON_PRESERVE_PORT_NUM:
                              break;

                           case RANON_PRESERVE_IP_ID:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPreserveIpId++;
                              else
                                 RaPreserveIpId = 0;
                              break;

                           case RANON_CLASSA_NET_ADDRESS_LIST:
                              break;

                           case RANON_CLASSB_NET_ADDRESS_LIST:
                              break;

                           case RANON_CLASSC_NET_ADDRESS_LIST:
                              break;

                           case RANON_MAP_DUMPFILE:
                              break;

                           case RANON_PRESERVE_ETHERNET_VENDOR:
                              if (!(strncasecmp(optarg, "yes", 3)))
                                 RaPreserveEtherVendor++;
                              else
                                 RaPreserveEtherVendor = 0;
                              break;
                        }
                     }
                     found++;
                     break;
                  }
               }
               if (!found) {
                  ArgusLog (LOG_ERR, "%s: syntax error line %d\n", file, lines);
               }
            }
         }

      } else {
#ifdef ARGUSDEBUG
         ArgusDebug (2, "%s: %s\n", file, strerror(errno));
#endif
      }

      if (RaPrintStartTime && RaPrintLastTime)
         Gflag++;
      else
      if (RaPrintLastTime)
         lflag++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusParseResourceFile (%s) returning %d\n", file, retn);
#endif

   return (retn);
}


#include <argus_util.h>

unsigned char argusBuf[2048];
extern struct ArgusCanonicalRecord *RaThisCanon;


struct ArgusRecord *
RaConstructArgusRecord (struct ArgusRecord *argus)
{
   struct ArgusRecord *newarg = NULL;
   struct ArgusFarHeaderStruct **hdrs = NULL, *hdr;
   
   if (argus != NULL) {
      ArgusThisFarStatus = ArgusIndexRecord (argus, ArgusThisFarHdrs);
      hdrs = ArgusThisFarHdrs;

      bzero (argusBuf, sizeof(argusBuf));
      newarg = (struct ArgusRecord *) argusBuf;

      bcopy ((char *)&argus->ahdr, (char *)&newarg->ahdr, sizeof (struct ArgusRecordHeader));
      newarg->ahdr.length = sizeof (struct ArgusRecordHeader);

      if (argus->ahdr.type & ARGUS_MAR) {
         bcopy ((char *)&argus->argus_mar, (char *)&newarg->argus_mar, sizeof(struct ArgusMarStruct));
         newarg->ahdr.length += sizeof(struct ArgusMarStruct);

      } else {
         if (((hdr = hdrs[ARGUS_FAR_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_FAR])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_MAC_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_MAC])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_TCP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_TCP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_VLAN_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_VLAN])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_MPLS_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_MPLS])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_AGR_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_AGR])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_TIME_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_TIME])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_ICMP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_ICMP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_RTP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_RTP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_IGMP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_IGMP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_ARP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_ARP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_FRG_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_FRG])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_ESP_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_ESP])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_SRCUSRDATA_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_SRCUSRDATA])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }

         if (((hdr = hdrs[ARGUS_DSTUSRDATA_DSR_INDEX]) != NULL) && (RaDSRFields[RASTRIP_DSR_DSTUSRDATA])) {
            bcopy((char *) hdr, (char *)&((char *)newarg)[newarg->ahdr.length], hdr->length);
            newarg->ahdr.length += hdr->length;
         }
      }
   }

   return (newarg);
}
