/*
 * Copyright (c) 2018, Alliance for Open Media. All rights reserved
 *
 * This source code is subject to the terms of the BSD 2 Clause License and
 * the Alliance for Open Media Patent License 1.0. If the BSD 2 Clause License
 * was not distributed with this source code in the LICENSE file, you can
 * obtain it at www.aomedia.org/license/software. If the Alliance for Open
 * Media Patent License 1.0 was not distributed with this source code in the
 * PATENTS file, you can obtain it at www.aomedia.org/license/patent.
 */

#ifndef AOM_AV1_ENCODER_PARTITION_MODEL_WEIGHTS_H_
#define AOM_AV1_ENCODER_PARTITION_MODEL_WEIGHTS_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "av1/encoder/ml.h"

// TODO(chiyotsai@google.com): The performance of these models are getting worse
// due the changes in the encoder. We should retrain the models here to get
// better performance once we have the time.

#define FEATURE_SIZE 10
#define LABEL_SIZE 16
// nn model for ab partition pruning, 128x128.
static const float av1_ab_partition_nn_weights_128_layer0[FEATURE_SIZE * 64] = {
  -0.715251f, -0.015767f, -0.667353f, -0.345255f, 0.177887f,  -0.469759f,
  0.426152f,  0.489798f,  0.469865f,  0.773821f,  0.088517f,  0.074585f,
  0.838754f,  0.048449f,  -0.007584f, 0.638968f,  0.233305f,  -0.319236f,
  -0.257124f, -0.170869f, 0.137180f,  0.114852f,  -0.721241f, -0.947962f,
  -0.411298f, 0.494306f,  -0.060435f, -0.648421f, -0.126624f, 0.072686f,
  -0.143904f, -0.115839f, -0.175527f, -0.117728f, 0.040686f,  -0.189925f,
  0.134361f,  -0.258070f, -0.177558f, 0.158049f,  0.168668f,  -0.062919f,
  0.341986f,  0.038100f,  -0.435577f, -0.321255f, 0.203213f,  0.213061f,
  0.533304f,  0.359296f,  -0.079558f, 0.004637f,  0.663904f,  0.043779f,
  0.383018f,  1.136559f,  -0.084155f, 0.333057f,  -0.199011f, 0.152059f,
  -0.078419f, -0.167752f, -0.093651f, 0.083171f,  -0.190143f, 0.086195f,
  -0.280632f, -0.160663f, -0.017298f, 0.122628f,  -0.138116f, 0.062927f,
  0.222462f,  0.626979f,  0.426928f,  0.117170f,  -0.240457f, 0.053750f,
  0.038017f,  0.007359f,  -0.017595f, 0.101407f,  0.332891f,  0.074933f,
  0.306498f,  0.219380f,  -0.151638f, -0.247976f, 0.343405f,  0.121256f,
  0.049173f,  0.171474f,  -0.139608f, -1.016599f, -0.345553f, -0.901138f,
  0.243401f,  0.059928f,  -0.089396f, -0.195565f, 0.364705f,  -0.020400f,
  -1.383672f, 0.413018f,  0.536950f,  -0.020904f, -1.335306f, -0.732290f,
  0.102885f,  0.315290f,  -0.208521f, -0.081811f, 0.182300f,  0.125712f,
  -0.593833f, -0.220639f, -0.314155f, 0.188327f,  0.118503f,  0.524427f,
  -1.083859f, -1.130640f, 0.390352f,  -0.045591f, 0.113160f,  -0.009149f,
  -0.096183f, 0.115829f,  0.377752f,  0.318396f,  -0.591983f, 0.004797f,
  -0.497377f, -0.342248f, 0.079546f,  -0.025249f, -0.295972f, 0.615501f,
  -0.464372f, 0.418315f,  -0.173556f, 0.105217f,  0.298073f,  0.082478f,
  0.033223f,  0.977341f,  -0.372982f, -0.052337f, 0.154124f,  0.396787f,
  0.536654f,  -0.139061f, -0.223702f, 0.229666f,  -0.846766f, 0.107723f,
  0.563839f,  -0.483141f, 0.304813f,  -0.765283f, 0.070964f,  0.151101f,
  0.275188f,  0.490303f,  1.175892f,  0.085377f,  -0.191200f, 0.544532f,
  -0.365075f, 0.167546f,  0.052183f,  -0.220529f, -0.212227f, -0.144988f,
  -0.273356f, -0.062023f, 0.103993f,  -0.238493f, -0.161204f, -0.054611f,
  -0.166672f, 0.128327f,  0.461751f,  -0.545822f, 0.739798f,  0.594386f,
  -0.163192f, -0.332501f, 0.363834f,  -0.065043f, 0.474812f,  -0.138811f,
  0.170924f,  -0.778142f, -0.316474f, -0.508065f, -0.039986f, -0.478001f,
  0.340591f,  0.041783f,  0.055419f,  0.015155f,  -0.981830f, -1.355237f,
  0.347516f,  1.155327f,  0.081319f,  0.274163f,  -0.327230f, -0.113478f,
  0.556552f,  -0.055986f, 0.217318f,  -0.445351f, 0.325759f,  0.526547f,
  -0.657434f, -0.572214f, -0.037087f, 0.081384f,  0.064518f,  0.014892f,
  0.215279f,  1.834504f,  -0.242107f, 0.079810f,  0.129558f,  0.079588f,
  -0.035189f, -0.221745f, -0.163414f, 0.043978f,  -1.028662f, -0.623609f,
  1.130336f,  0.664661f,  -0.063975f, -0.415863f, 0.018581f,  0.157758f,
  0.200570f,  0.063420f,  0.901039f,  -0.746286f, 0.196230f,  -0.290592f,
  0.042373f,  -0.502500f, 0.183638f,  0.103394f,  -0.298858f, 0.145436f,
  0.196916f,  0.108319f,  -0.448572f, -0.881385f, 0.302497f,  0.121679f,
  -0.021327f, 0.025150f,  0.481306f,  -0.359634f, 0.350257f,  -0.228647f,
  -0.669860f, 0.260025f,  -0.034182f, 0.619247f,  -0.158826f, -0.405864f,
  0.674112f,  -0.027885f, -0.325274f, -0.241492f, 0.036024f,  -0.437685f,
  -0.091458f, -0.109295f, -0.350676f, 0.044706f,  0.297059f,  0.016290f,
  1.121203f,  1.289062f,  -1.299476f, -1.129221f, 0.103752f,  0.131302f,
  -0.263265f, 0.222155f,  -0.229908f, 0.013922f,  -0.226001f, -0.248383f,
  -0.004415f, -0.020958f, 0.055634f,  0.086200f,  0.114556f,  -0.184061f,
  -0.096210f, -0.146466f, -0.249618f, -0.195998f, 0.088758f,  0.023781f,
  -0.264460f, 0.157026f,  -0.235228f, -0.102564f, 0.043463f,  -0.187823f,
  -0.257500f, -0.199049f, -0.242210f, 0.030448f,  0.221604f,  0.151804f,
  -0.100404f, -0.073931f, 0.144749f,  -0.001572f, -1.438079f, -0.233716f,
  0.733422f,  1.727080f,  -0.036397f, 0.027551f,  0.425321f,  0.085703f,
  0.031186f,  0.032333f,  -0.675130f, 1.437733f,  -0.202392f, -0.525003f,
  0.087048f,  0.328194f,  -0.079989f, -0.391088f, -0.238732f, -0.120660f,
  -0.139600f, 0.154665f,  0.026202f,  -0.233501f, -0.009046f, -0.149187f,
  -0.199646f, 0.115375f,  0.209762f,  -0.014875f, 0.124038f,  -0.119985f,
  1.079625f,  -0.461513f, 0.614114f,  0.021003f,  0.439449f,  -0.824834f,
  -0.299701f, 0.193817f,  -0.870551f, -1.262313f, -0.079517f, 0.341570f,
  0.305310f,  -0.089721f, -0.317314f, -0.075631f, 0.127172f,  -0.208635f,
  1.191922f,  0.163141f,  0.564285f,  0.286352f,  0.480865f,  0.173094f,
  -0.094034f, -0.071339f, -0.328992f, -0.006382f, 0.314705f,  0.090258f,
  -0.016099f, 0.193230f,  0.188061f,  0.398144f,  0.722781f,  0.769949f,
  0.025442f,  -0.162016f, 0.070192f,  -0.056946f, -0.100957f, -0.219934f,
  -0.203492f, -0.015454f, -0.013272f, -0.098008f, 0.051707f,  -0.017493f,
  0.527446f,  0.083605f,  0.588318f,  0.878215f,  0.028747f,  -0.146479f,
  -0.345170f, -0.136059f, -0.152005f, -0.203634f, 0.232702f,  -0.101340f,
  -0.027733f, -0.282611f, 0.265366f,  0.082362f,  -0.265420f, -0.131124f,
  0.166303f,  0.040194f,  -0.100710f, 0.579151f,  -0.530136f, 0.163422f,
  -0.998821f, -1.565311f, -1.774785f, -2.493372f, 0.116970f,  -0.090302f,
  1.723272f,  0.552370f,  -0.295954f, -0.439095f, -0.266730f, 0.027936f,
  0.539616f,  -0.234902f, -0.167601f, -0.149877f, -0.242983f, 0.122353f,
  -0.121620f, -0.205517f, -0.180144f, -0.264208f, 0.151500f,  -0.159378f,
  0.029145f,  -0.050892f, -0.223407f, -0.246239f, 0.043152f,  -0.018460f,
  0.169972f,  -0.187769f, -0.034670f, -0.238330f, 0.288070f,  -0.093243f,
  -0.437105f, -0.573376f, 0.660073f,  0.285727f,  0.408470f,  0.158475f,
  0.032699f,  0.056280f,  -0.237176f, -0.083003f, 0.105598f,  -0.169522f,
  -0.260420f, -0.121100f, -0.173983f, -0.195693f, -0.232028f, 0.224940f,
  0.029124f,  0.009580f,  -0.252034f, 0.103087f,  1.156561f,  0.603848f,
  -0.562805f, -1.652742f, -0.568288f, -1.829395f, 0.046169f,  0.076095f,
  1.490819f,  0.415893f,  -0.277788f, -0.115787f, 0.093750f,  0.270726f,
  -0.395983f, -0.353742f, 0.034605f,  0.005342f,  0.184537f,  0.086445f,
  0.156417f,  1.476367f,  0.122587f,  0.002145f,  0.431057f,  -0.381184f,
  -1.646457f, -0.014009f, -0.671224f, 0.193726f,  -0.019247f, -0.031267f,
  -0.046208f, 0.298733f,  0.064734f,  0.616984f,  0.039381f,  0.182722f,
  -0.116670f, 0.233093f,  -1.214374f, -0.817970f, -0.064394f, -0.584783f,
  0.077697f,  -0.266720f, 0.130875f,  -0.235295f, -0.265754f, -0.159999f,
  -0.250114f, -0.183017f, 0.194403f,  -0.105808f, -0.169215f, -0.240866f,
  -0.026662f, -0.045123f, -0.036175f, -0.167471f, -0.192908f, -0.232602f,
  -0.267036f, -0.112500f, -0.257944f, -0.111909f, -0.802226f, -0.008800f,
  0.881460f,  -0.678603f, 0.008666f,  -0.252053f, -0.341035f, -0.175290f,
  0.183012f,  0.385991f,  0.079888f,  -0.014039f, -0.148653f, 0.671778f,
  -0.130219f, 1.086467f,  0.129267f,  -0.040400f, -0.201221f, -0.077005f,
  0.015890f,  0.000781f,  0.137764f,  1.389546f,  0.172152f,  0.047279f,
  -0.042783f, 0.127740f,  0.141467f,  -0.335738f, -1.396392f, 0.031496f,
  0.357385f,  0.343602f,  -0.714553f, 0.311014f,  0.132845f,  0.061149f,
  0.006796f,  0.568106f,  -0.255949f, 0.104134f,  -0.993447f, 0.298135f,
  -0.406590f, -0.049228f, -0.578570f, -0.188561f, -0.107046f, 0.374095f,
  0.068481f,  0.036240f,  -0.495801f, 0.180574f,  -0.766129f, 0.886967f,
  -0.568868f, -0.936062f, -0.418886f, -0.058735f, -0.511964f, -0.438596f,
  0.019016f,  -0.015837f, 0.600197f,  0.429773f,  0.315026f,  0.319667f,
  0.214617f,  -0.017316f, 0.270257f,  -0.040524f, 0.695803f,  -0.015223f,
  -1.554965f, 0.356997f,  -1.472428f, 0.024637f,  -0.562958f, 0.870351f,
  0.193635f,  0.036063f,  0.328638f,  0.200274f,  -1.634707f, 0.110534f,
  0.420104f,  -0.072042f, -0.006404f, 0.171680f,
};

static const float av1_ab_partition_nn_bias_128_layer0[64] = {
  0.643147f,  -1.348826f, 0.431627f,  0.000000f,  0.102717f,  -0.772628f,
  -0.034351f, -0.761977f, -0.638397f, 0.541969f,  -0.391311f, 0.563076f,
  0.148553f,  0.267217f,  -0.788092f, 0.544573f,  -0.546280f, 0.000000f,
  -0.446945f, 0.127732f,  0.270624f,  -0.219435f, -1.220203f, 0.324584f,
  0.110885f,  0.276547f,  0.179726f,  -0.375160f, 0.026401f,  -0.032595f,
  0.000000f,  -0.047932f, -0.648602f, -0.512637f, -0.031661f, -0.236761f,
  0.476453f,  -0.028021f, -0.013673f, -0.015578f, -0.920077f, 0.000000f,
  0.915351f,  -0.209962f, 0.000000f,  -0.025731f, 0.218288f,  0.000000f,
  0.047726f,  -0.813077f, -1.263281f, 0.239087f,  0.278614f,  -0.030753f,
  0.000000f,  0.346744f,  -0.948543f, -1.174211f, 0.216377f,  0.498913f,
  0.853918f,  0.002504f,  -0.190403f, 0.452050f,
};

static const float av1_ab_partition_nn_weights_128_layer1[64 * LABEL_SIZE] = {
  0.179769f,  1.499417f,  -0.445135f, -0.142278f, -0.337661f, 0.682064f,
  -0.203213f, 0.302171f,  0.226877f,  -0.422169f, 1.687586f,  0.783773f,
  0.220995f,  0.253482f,  0.370435f,  -1.342775f, 0.337229f,  -0.271473f,
  0.291796f,  1.362227f,  -1.751397f, -0.086178f, 0.725496f,  -0.118597f,
  0.227963f,  -0.501577f, 0.223849f,  -0.122421f, -0.123437f, -0.051045f,
  -0.020115f, 0.212711f,  0.246025f,  0.088120f,  -0.168995f, 1.740190f,
  -0.195098f, 0.680339f,  -0.589572f, -0.075244f, 0.878766f,  0.064092f,
  -3.548527f, 0.001660f,  0.107926f,  -0.169501f, -0.455212f, 0.123045f,
  -1.836998f, 0.330365f,  1.301475f,  0.454761f,  -0.576552f, -0.190761f,
  0.208459f,  0.618483f,  1.383364f,  0.970718f,  0.390174f,  0.406252f,
  -0.564519f, -0.312062f, 1.345712f,  -0.151873f, 0.109290f,  0.408847f,
  0.391243f,  0.152024f,  0.181764f,  -0.036263f, -0.160466f, 0.153595f,
  0.049163f,  -0.753012f, -1.804062f, 0.347475f,  -2.746580f, 0.575618f,
  0.261799f,  0.210505f,  -0.302054f, -0.109872f, 0.199506f,  -1.182971f,
  0.723668f,  0.177758f,  -0.338202f, 0.254396f,  -0.220023f, 0.043504f,
  0.669866f,  -0.040816f, -0.402730f, 0.017990f,  0.215523f,  -0.216816f,
  0.454826f,  -0.726067f, -0.018750f, -0.928679f, 0.154315f,  -0.465641f,
  0.144566f,  -0.030064f, -0.054667f, -0.154055f, 0.625384f,  1.323795f,
  -0.159496f, 0.097072f,  -0.463197f, -0.057938f, 0.750290f,  -0.233061f,
  0.412631f,  -0.535223f, -0.151423f, -0.154583f, 0.024721f,  -0.494448f,
  0.230594f,  -0.980138f, -0.653968f, 0.126079f,  0.051814f,  -0.053219f,
  -0.421708f, -0.228853f, 0.237885f,  0.888157f,  0.059655f,  0.241295f,
  0.210443f,  0.228238f,  0.119127f,  -0.051989f, -0.355408f, 0.182215f,
  0.244277f,  -0.104577f, -0.558035f, -0.023270f, 0.054571f,  0.700646f,
  -0.223006f, 0.115523f,  0.023391f,  0.437264f,  0.709477f,  -0.531212f,
  -0.094731f, 0.328161f,  -0.105418f, -0.133511f, 0.497168f,  -0.030948f,
  -0.407132f, -0.043943f, 0.155505f,  0.251945f,  0.205010f,  0.167160f,
  0.083654f,  -0.636810f, 0.401315f,  -0.398414f, 0.290046f,  0.206846f,
  0.042218f,  0.168150f,  0.843181f,  -0.671242f, -0.202392f, -0.073301f,
  0.142895f,  0.237466f,  0.212145f,  -0.091828f, 0.187038f,  -0.720841f,
  -0.616069f, -0.238021f, 0.065365f,  0.434119f,  0.179023f,  -0.040107f,
  -0.430734f, -0.297368f, 0.575954f,  0.382619f,  -0.709787f, -0.320810f,
  0.242342f,  -0.047614f, 0.705216f,  0.098077f,  0.357179f,  0.046017f,
  0.115074f,  -0.412305f, -0.272304f, 0.048096f,  -0.803811f, 0.275000f,
  0.642198f,  0.180286f,  -0.087178f, -0.112707f, -0.394443f, 0.201989f,
  0.241759f,  -1.038870f, 0.728124f,  0.800559f,  -1.296268f, 0.198612f,
  -0.053478f, 0.414344f,  -0.510529f, 0.124179f,  -2.219115f, -0.074583f,
  -0.143055f, 0.001697f,  0.810811f,  -0.657140f, 0.186818f,  -0.936414f,
  0.539578f,  -0.308244f, -0.126624f, -0.204767f, 0.091145f,  -0.049340f,
  0.252014f,  0.394582f,  0.018764f,  -0.060377f, -0.019133f, 0.064083f,
  0.069211f,  -0.526693f, 0.209850f,  -0.481466f, -0.468302f, -0.100407f,
  0.241018f,  -1.037781f, 0.038539f,  -2.113840f, -0.974895f, 0.163187f,
  0.425132f,  -0.772546f, -1.261254f, -0.217488f, -0.971748f, -0.805640f,
  -0.745175f, -0.177077f, 0.217658f,  0.381431f,  -0.052338f, 0.087176f,
  -0.165972f, 0.085937f,  0.472564f,  -0.796627f, -2.453307f, 0.569664f,
  -0.233010f, -0.192134f, 0.064339f,  -0.111411f, -0.262469f, -0.410022f,
  0.519993f,  -0.684620f, 0.393460f,  -0.277753f, -0.153624f, 0.528984f,
  -0.415558f, -0.445863f, 0.588512f,  -0.142439f, -0.132127f, 0.199776f,
  -0.579284f, 0.119488f,  -0.033590f, -0.503846f, -0.674979f, 0.335125f,
  0.020519f,  0.233973f,  -0.297998f, -0.051511f, 0.518626f,  -0.412782f,
  -0.074045f, 0.130523f,  0.465751f,  -0.117795f, 2.535813f,  0.352108f,
  -0.499228f, 0.379784f,  0.056699f,  0.173142f,  -0.076519f, -0.026666f,
  0.017834f,  0.492333f,  0.093364f,  0.037867f,  -0.165420f, -0.356429f,
  -0.562334f, 0.057656f,  -0.307544f, 0.085857f,  -0.559851f, 0.107230f,
  -0.398633f, 0.152618f,  -0.216835f, -0.024539f, 0.026044f,  -0.249519f,
  -0.563594f, -0.746025f, 0.025265f,  -0.298888f, -0.185243f, 0.058794f,
  0.233696f,  -0.115223f, 0.144617f,  -0.864390f, 0.619944f,  -0.023980f,
  0.019481f,  0.225252f,  0.416552f,  -0.115993f, 0.935387f,  0.744386f,
  0.053353f,  -0.052582f, -0.065650f, 0.228488f,  -0.032042f, -0.371252f,
  -0.003638f, -0.736984f, -0.203776f, 0.030922f,  -0.065577f, -0.031643f,
  -0.049253f, -0.054640f, 0.787134f,  0.545414f,  -0.140297f, -0.124274f,
  -0.110011f, -0.029552f, 0.657005f,  0.214973f,  -0.374300f, 0.251642f,
  0.276591f,  0.030566f,  -0.145470f, 0.350579f,  -0.356436f, -0.052694f,
  -0.063966f, -0.751008f, -1.042392f, 0.328892f,  -0.425058f, -0.421571f,
  -0.571889f, -1.141472f, -0.125216f, 0.212713f,  -0.485170f, -0.088791f,
  0.124589f,  0.023237f,  0.077635f,  0.020901f,  -0.271402f, -0.321424f,
  -0.513946f, -0.867872f, -0.284593f, 0.106276f,  0.220192f,  -0.143532f,
  -0.014648f, 0.073402f,  0.327256f,  -0.139803f, 0.168763f,  0.048199f,
  -0.122526f, 0.111713f,  -0.134257f, 0.810364f,  -0.085222f, -0.259221f,
  -0.239349f, 0.044448f,  0.205031f,  0.413113f,  -0.107720f, -0.018816f,
  -0.247741f, -0.004963f, 0.041170f,  -0.158019f, 0.134839f,  0.129502f,
  0.800488f,  -1.041584f, -0.129336f, 0.170834f,  0.566586f,  -0.230443f,
  0.437937f,  -0.149922f, -0.046665f, -0.094646f, 0.200070f,  0.072943f,
  -0.076943f, -0.084971f, -0.515843f, -0.146720f, 0.472869f,  -0.444731f,
  -0.100877f, 0.545196f,  -1.786626f, -0.482946f, 0.500509f,  -0.843257f,
  0.200374f,  0.045103f,  -0.575718f, -0.164335f, -0.232522f, -0.021825f,
  -0.139490f, 0.356058f,  -0.352075f, 0.061751f,  -0.200616f, -1.180921f,
  -0.181355f, -0.137459f, 0.247574f,  0.181541f,  0.184314f,  -0.961482f,
  0.493615f,  0.910261f,  -2.279238f, 0.648631f,  -0.055526f, -0.037137f,
  0.038643f,  0.136609f,  -0.819373f, -0.040840f, -0.265989f, 0.006877f,
  0.454651f,  -0.595323f, -0.099500f, -0.263717f, 0.150456f,  0.245077f,
  -0.268666f, 0.162232f,  -0.516451f, -0.024501f, 0.188046f,  -0.002262f,
  0.261319f,  0.004173f,  0.746982f,  0.174761f,  0.470447f,  -0.159558f,
  -0.385240f, 0.023084f,  -0.133520f, -0.220607f, -0.018731f, -0.373558f,
  -0.707763f, -1.850150f, -0.807404f, -0.168063f, -0.071435f, -0.160740f,
  -0.478789f, -1.070674f, -0.489740f, -0.255796f, 0.100486f,  -0.153361f,
  0.334394f,  -0.569472f, -0.198118f, 0.255922f,  0.104717f,  -0.065179f,
  0.111879f,  -0.447237f, 1.373623f,  -0.190191f, -0.063311f, 0.337529f,
  -0.138800f, 0.057009f,  -0.137006f, 0.641378f,  0.883147f,  -0.679655f,
  0.267717f,  -0.351602f, -0.135225f, 0.229398f,  -0.513225f, -1.120345f,
  0.528786f,  -0.051081f, 0.086653f,  0.140141f,  -0.563969f, 0.333402f,
  -0.174745f, 0.321093f,  -0.438641f, -0.005131f, 0.247415f,  0.110120f,
  -0.076308f, -0.083244f, 0.838944f,  -0.113043f, -0.013258f, -0.175028f,
  -0.179941f, 0.272676f,  -0.047946f, -0.088076f, -0.450031f, 0.053929f,
  -0.083549f, -0.089952f, -0.186253f, 0.257483f,  0.011019f,  0.586435f,
  0.060580f,  -0.052078f, 0.090277f,  -0.780869f, 0.969811f,  -0.025349f,
  -0.281917f, 0.014857f,  0.231863f,  -0.228601f, -0.003861f, 0.226550f,
  0.141825f,  -0.102171f, -0.010387f, 0.220378f,  -2.561975f, -0.497071f,
  -0.315117f, 0.371981f,  0.138247f,  0.625031f,  -0.308133f, -0.217876f,
  0.005615f,  -0.860179f, 0.747491f,  0.006356f,  -0.057024f, -0.483189f,
  0.055592f,  -0.316834f, 0.069858f,  0.218788f,  -0.200044f, 0.227588f,
  0.215496f,  -0.055324f, -0.393147f, -0.394062f, -0.253264f, -0.075619f,
  -0.152512f, -0.332995f, 0.129053f,  0.178668f,  -0.302694f, 0.030678f,
  0.925896f,  0.964375f,  0.169021f,  -0.218657f, -0.627204f, 0.206437f,
  -0.521336f, 0.176206f,  0.142733f,  0.139248f,  0.411682f,  0.181544f,
  0.224850f,  -0.935547f, -0.558208f, 0.348096f,  0.342129f,  -0.389340f,
  -0.236308f, -0.132099f, 0.073642f,  0.089391f,  -0.306901f, -0.397842f,
  0.444282f,  0.074623f,  -0.051075f, -0.106617f, -0.184037f, -0.239046f,
  -0.138761f, 0.120794f,  -0.647577f, -0.336471f, 0.527899f,  -0.164234f,
  -0.028354f, 1.083678f,  -0.251534f, -0.145903f, -0.182783f, 0.070976f,
  -0.199590f, -0.400306f, -0.029763f, -0.548042f, -0.266270f, -0.118084f,
  -1.152632f, 0.383685f,  -0.105895f, -0.096829f, 0.118382f,  0.047447f,
  -0.019051f, 0.310180f,  -0.162793f, -0.029574f, 0.058054f,  -0.636017f,
  0.490639f,  0.158347f,  -0.385701f, -0.147057f, 1.285825f,  -1.276083f,
  -0.021795f, -0.101600f, 0.163254f,  0.267160f,  -2.317864f, -0.098598f,
  -0.296337f, -0.309017f, 0.164127f,  -0.270012f, -0.071187f, -0.262270f,
  0.075415f,  -0.368328f, 0.186728f,  -0.158031f, 0.481663f,  0.515950f,
  -0.162551f, 0.497981f,  0.262196f,  0.168479f,  0.726066f,  -0.243856f,
  -0.058998f, 0.140168f,  0.053242f,  -0.624623f, -0.249480f, 0.055197f,
  -1.376804f, 0.417571f,  0.203784f,  0.174370f,  -0.155531f, -0.029400f,
  -0.491473f, 0.079811f,  -0.080123f, 1.345900f,  0.637077f,  0.434862f,
  -1.787438f, 0.005756f,  -0.362706f, 0.179458f,  -0.288263f, 0.516788f,
  -0.921248f, 0.043794f,  -0.137729f, -0.196171f, -0.046295f, -0.793781f,
  -0.156532f, -0.132566f, 0.517989f,  -0.154321f, -0.054174f, -0.077900f,
  -0.373316f, -0.117718f, 0.188986f,  -0.476188f, -0.245312f, 0.181439f,
  -0.161024f, -0.229059f, -3.079907f, -0.225452f, -0.594355f, -0.558027f,
  -0.135429f, 0.125766f,  -0.081314f, -0.350894f, -0.163165f, -1.936507f,
  -0.205966f, 0.031472f,  0.744446f,  -0.006680f, -0.837551f, 0.605862f,
  -0.854929f, -1.543750f, -0.307704f, -0.240517f, 0.178240f,  -0.183586f,
  -0.010307f, 0.099373f,  -0.228278f, 0.175236f,  -0.000133f, 0.104491f,
  -1.540545f, -0.570971f, -0.252885f, 0.483036f,  0.052531f,  0.260214f,
  -0.515016f, -0.602081f, -0.485690f, -0.730710f, 0.163719f,  -1.775975f,
  -0.298634f, 0.323626f,  -0.373579f, -0.872977f, 0.619574f,  0.026862f,
  -0.122531f, -0.084698f, -2.436297f, 0.483996f,  -0.203640f, -0.302157f,
  -0.150666f, -0.238320f, 0.089250f,  0.236485f,  -0.668654f, -0.122863f,
  0.491152f,  -0.226444f, -0.181248f, 0.120158f,  0.294027f,  0.250056f,
  0.307601f,  0.357875f,  -1.746455f, -0.175670f, 0.385447f,  -0.108808f,
  -0.090235f, -0.642504f, -0.486004f, -0.055160f, -0.068692f, 0.009736f,
  0.607555f,  -0.489426f, 0.150624f,  0.598114f,  -0.128816f, -0.445793f,
  -0.066524f, -0.254380f, 0.227106f,  -0.406495f, -0.121632f, -0.275960f,
  -0.136494f, 0.339457f,  -1.318132f, -0.417572f, -2.614077f, 0.324603f,
  -0.001211f, 0.375192f,  -0.473448f, -0.162510f, 0.099329f,  -0.277965f,
  0.101221f,  -0.060263f, 0.121867f,  -1.042140f, 0.440851f,  0.078898f,
  -0.209007f, -0.243699f, 0.715197f,  -0.093997f, 0.086022f,  -0.178203f,
  -2.275496f, -0.098413f, 0.199352f,  -0.526791f, -0.162086f, -0.197806f,
  -0.231657f, -0.269202f, -0.794294f, -0.223461f, 0.503584f,  0.416236f,
  0.064082f,  0.197655f,  0.340871f,  -0.186645f, -0.291498f, 0.433938f,
  -1.110063f, 0.003751f,  0.392738f,  0.069360f,  0.102088f,  -0.302128f,
  -1.518457f, 0.106939f,  0.404527f,  -0.306868f, -0.286928f, 0.729276f,
  -0.531710f, 0.745048f,  -0.168837f, -1.953886f, -0.258828f, -0.190252f,
  0.241877f,  -0.916744f, -0.030326f, -0.070541f, -0.271037f, 0.211303f,
  -0.489957f, 0.100850f,  0.323999f,  -0.802837f, -0.462408f, -0.079350f,
  -0.029374f, 0.131213f,  -0.825032f, 0.040202f,  0.351821f,  0.002869f,
  -0.132516f, -0.471264f, -0.297002f, 0.263913f,  0.033478f,  0.146161f,
  0.533229f,  -0.228608f, -0.200639f, -0.170955f, -0.915037f, 0.724491f,
  0.005151f,  0.018584f,  -0.029771f, -0.396038f, -0.159236f, 0.038691f,
  -1.197056f, 0.146302f,  0.226840f,  -0.852126f, 0.031214f,  0.108880f,
  0.562000f,  -0.134633f, -0.713343f, -0.342252f, -1.764521f, -0.114653f,
  0.515073f,  -0.080515f, -0.121155f, -0.865139f, -0.833694f, -0.368553f,
  0.347673f,  0.623379f,  0.722067f,  -0.492458f, -0.513263f, 0.585167f,
  0.721518f,  -0.693499f, 0.343725f,  -0.273861f, -0.040230f, -0.785664f,
  -0.157500f, -0.308445f, 0.054062f,  0.600131f,  -0.860887f, 0.434470f,
  -0.191382f, -0.306150f, -0.243965f, 0.705444f,  0.007789f,  -0.146154f,
  -0.054499f, -0.073500f, -1.067364f, 0.404936f,  -2.864590f, 0.182323f,
  0.326126f,  0.102405f,  -0.135800f, 1.128095f,  -0.012267f, -0.023996f,
  -0.264834f, -0.108967f, -1.176746f, -0.926666f, 0.082999f,  -0.498361f,
  0.083560f,  -0.210074f, 0.019225f,  -0.201614f, -0.904760f, 0.181421f,
  0.586384f,  -0.177706f, 0.065471f,  0.168552f,  0.054705f,  0.045241f,
  0.048057f,  -0.410957f, -2.188854f, -0.169812f, 0.015521f,  0.176856f,
  -0.179331f, -0.352640f, -0.491735f, -1.743206f, 0.044227f,  0.010454f,
  0.823643f,  -0.119781f, -0.098359f, 0.093119f,
};

static const float av1_ab_partition_nn_bias_128_layer1[LABEL_SIZE] = {
  -0.433195f, -0.120488f, -0.116721f, 0.112134f,  0.118170f, -0.259769f,
  -0.077530f, 0.394044f,  0.279167f,  -0.317988f, 0.189538f, 0.314776f,
  0.325655f,  -0.107123f, 0.591049f,  0.358744f,
};

static const NN_CONFIG av1_ab_partition_nnconfig_128 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      64,  // num_hidden_nodes
  },
  {
      av1_ab_partition_nn_weights_128_layer0,
      av1_ab_partition_nn_weights_128_layer1,
  },
  {
      av1_ab_partition_nn_bias_128_layer0,
      av1_ab_partition_nn_bias_128_layer1,
  },
};

// nn model for ab partition pruning, 64x64.
static const float av1_ab_partition_nn_weights_64_layer0[FEATURE_SIZE * 64] = {
  -0.495347f, -0.049498f, -0.026804f, 0.030474f,  -0.289308f, -0.264193f,
  -0.141121f, -0.072562f, -0.391665f, -0.051491f, -0.234761f, 0.027155f,
  -0.038217f, 0.014872f,  -0.289728f, -0.233577f, -0.415875f, -0.343615f,
  -0.442543f, -0.482492f, 0.073510f,  0.007503f,  2.162329f,  -0.362849f,
  2.145915f,  -0.883135f, 0.185636f,  -0.062859f, -0.465574f, -0.486205f,
  -0.056710f, -0.330642f, -0.321860f, 0.042321f,  -0.348965f, 0.003542f,
  -0.291365f, -0.078164f, -0.345093f, -0.220272f, -0.471270f, -0.763853f,
  0.246622f,  0.199651f,  -0.663420f, -0.154152f, -1.220383f, 0.047138f,
  0.816811f,  0.083247f,  -0.218839f, 0.038143f,  -0.063436f, 0.015517f,
  -0.307320f, -0.166956f, -0.169499f, -0.399005f, -0.234638f, -0.162266f,
  0.050425f,  -0.221723f, -0.256942f, -0.287285f, 0.144011f,  -0.033245f,
  0.083649f,  0.119428f,  -0.056706f, -0.117805f, 0.021866f,  -0.257300f,
  -0.201378f, -0.217484f, -0.413780f, -0.145793f, 0.082792f,  -0.347247f,
  0.042539f,  -0.302697f, 1.652316f,  0.000701f,  -0.482843f, -0.160332f,
  -0.450099f, 0.212399f,  -4.715360f, -5.336774f, -5.375758f, -6.048339f,
  0.085956f,  -0.037767f, 1.052409f,  -0.931924f, -2.221907f, 0.268946f,
  0.015512f,  1.237094f,  -1.092185f, 0.418247f,  -0.082143f, -0.076914f,
  -0.060749f, -0.325440f, -0.296960f, -0.066815f, -0.158477f, -0.373945f,
  -0.122322f, -0.113495f, -0.097978f, -0.192816f, -0.270418f, 0.035840f,
  -0.015458f, -0.121071f, -0.279582f, -0.067683f, 0.097855f,  0.019839f,
  0.451127f,  0.004376f,  1.410392f,  3.255835f,  -0.344815f, 0.145202f,
  0.204132f,  0.171948f,  -0.527736f, -0.110353f, 0.901448f,  0.003238f,
  -3.822090f, 0.235462f,  1.024823f,  -0.821244f, 0.876056f,  2.553762f,
  -3.478597f, -2.076582f, -0.265515f, -0.055923f, -0.156980f, -0.164097f,
  -0.246040f, 0.039430f,  -0.071769f, -0.118847f, -0.304053f, -0.281541f,
  -0.226021f, -0.263091f, -0.127359f, -0.249410f, -0.051023f, 0.083911f,
  0.084721f,  0.168089f,  -0.272169f, -0.204998f, -0.008303f, -0.173998f,
  0.079376f,  -0.197426f, -0.199052f, -0.118794f, -0.063753f, -0.094769f,
  0.066176f,  -0.175832f, -0.238752f, -0.287960f, -0.134307f, -0.185953f,
  -0.385845f, 0.119769f,  -0.006567f, -0.382126f, -0.214221f, 0.038449f,
  -0.253484f, -0.282766f, -0.020249f, -0.193929f, 0.016281f,  -0.114423f,
  -0.145940f, -0.281621f, -0.007588f, -0.131470f, -0.189012f, -0.185699f,
  -0.279011f, -0.008132f, 0.208463f,  0.020569f,  -0.206803f, -0.213408f,
  -0.206131f, -0.290245f, 0.069701f,  -0.000371f, -0.307572f, -0.451785f,
  -0.300838f, -0.453186f, -0.301691f, 0.046327f,  -0.312668f, 0.058272f,
  -0.303131f, -0.376252f, 0.108384f,  -0.086623f, -0.100630f, -0.027330f,
  -0.003969f, 0.089502f,  -0.200722f, -0.107889f, 0.061843f,  -0.008478f,
  -0.265057f, -0.271132f, -0.073562f, 0.129337f,  -0.283698f, -0.353414f,
  0.076420f,  -0.244280f, -0.119537f, -0.105366f, -0.184692f, -0.038817f,
  -0.478507f, -0.118808f, -0.472979f, -0.305884f, -0.462813f, -0.189581f,
  -0.011932f, -0.585700f, 0.253212f,  -1.061900f, -0.205116f, -0.336407f,
  -0.762199f, 0.577737f,  0.230832f,  0.434440f,  -0.096713f, 0.038552f,
  -0.147800f, -0.213553f, 0.041740f,  -0.281907f, -0.026154f, -0.082356f,
  -0.331871f, -0.408247f, -0.129022f, -0.037550f, -0.310233f, -0.320883f,
  -0.391963f, -0.467392f, 0.027453f,  -0.394761f, -0.045544f, 0.076052f,
  0.483985f,  0.067093f,  0.141361f,  0.576772f,  0.859718f,  2.566515f,
  -0.025476f, 0.769738f,  -0.680235f, -1.683309f, -2.394131f, -0.000714f,
  -0.615021f, -0.195856f, -0.434035f, -0.295010f, -0.668659f, -0.245959f,
  0.551148f,  1.777227f,  -0.461630f, 0.043093f,  0.012293f,  -0.255841f,
  -0.097070f, -0.371156f, -0.146323f, -0.015508f, -0.103873f, -0.087476f,
  -0.297266f, -0.128699f, -0.149555f, 0.016534f,  -0.375498f, -0.346759f,
  -0.455156f, -0.147509f, -0.427076f, -0.354431f, -0.158025f, -0.164604f,
  -0.237038f, -0.010314f, -0.092884f, -0.397084f, -0.217980f, -0.127184f,
  -0.048421f, -0.144133f, 0.889073f,  0.012606f,  3.007608f,  -0.602584f,
  -1.849480f, -0.373159f, -1.890695f, -3.609938f, 0.811923f,  -1.867208f,
  -0.244326f, -0.018012f, -0.211192f, -0.220196f, 0.169363f,  0.119141f,
  -0.230715f, 0.083247f,  0.020367f,  -0.128629f, -0.217455f, -0.159640f,
  1.815952f,  -0.369238f, -1.186447f, -0.658753f, -0.511026f, -0.096934f,
  0.662971f,  0.486475f,  0.159746f,  -0.018932f, 3.692397f,  1.384353f,
  -0.401984f, -0.248380f, -0.140861f, 0.215248f,  -0.023711f, 0.059679f,
  -0.072260f, 0.004271f,  0.039545f,  -0.347971f, -0.081851f, -0.474896f,
  -0.181572f, 0.066736f,  -0.157822f, -0.163760f, -0.171113f, -0.089935f,
  -0.338281f, -0.421444f, -0.306687f, -0.085283f, -0.377953f, -0.138750f,
  -0.102701f, -0.312336f, 0.149831f,  0.007229f,  -0.155700f, -0.173611f,
  4.074261f,  1.342306f,  -1.272712f, 1.570899f,  -0.545093f, -0.317605f,
  -0.189440f, -0.133910f, -0.273190f, -0.108020f, -0.166107f, 0.021413f,
  -0.239130f, -0.067211f, 0.041957f,  -0.039234f, -1.003587f, -0.094412f,
  0.532512f,  -0.870538f, -1.118023f, -1.160983f, -0.736307f, -0.418752f,
  0.419466f,  0.492122f,  -0.004368f, -0.022096f, -1.115132f, 0.150886f,
  2.396852f,  2.660000f,  -0.376537f, 0.468628f,  0.149413f,  -0.074898f,
  -0.067154f, 0.021245f,  0.127857f,  0.294189f,  0.508056f,  0.390232f,
  -3.899177f, -3.414681f, -3.929195f, -4.160545f, -0.274323f, -0.052583f,
  -0.003545f, -0.433084f, -0.404891f, -0.145051f, -0.312367f, 0.004579f,
  -0.398724f, -0.372068f, -0.234279f, 0.017799f,  -0.424760f, -0.646717f,
  -0.047568f, 2.924664f,  -0.644165f, 0.359349f,  -0.294800f, 0.591746f,
  -0.404710f, -0.092358f, -0.250729f, 0.030829f,  -0.147149f, -0.476023f,
  -0.071803f, -0.482516f, -0.293117f, -0.215923f, -0.373122f, -0.085315f,
  -0.377052f, -0.449899f, -0.056452f, 0.138081f,  -0.085350f, -0.308391f,
  0.106661f,  0.176234f,  0.258869f,  -0.230172f, -0.233029f, -0.241208f,
  -0.067509f, -0.223172f, -0.118353f, -0.302478f, -0.579632f, -0.561326f,
  -0.158114f, -0.223167f, -0.026689f, 0.051863f,  0.212834f,  -0.304714f,
  -0.169071f, -0.193695f, -0.075682f, -0.170860f, -0.241008f, -0.044648f,
  0.280815f,  -0.002585f, -0.283552f, -0.037701f, -0.681169f, -0.274535f,
  -0.380595f, 0.109504f,  -0.111141f, -0.437685f, -0.094459f, 0.144206f,
  -0.106139f, -0.211832f, -0.054742f, -0.172813f, -0.295905f, -0.071907f,
  -0.418429f, -0.183240f, 0.031319f,  -0.095785f, -0.315447f, 0.069404f,
  -0.422910f, -0.029867f, -0.357321f, -0.199976f, -0.337707f, -0.070188f,
  -0.178198f, 0.177208f,  0.134688f,  -0.081933f, -0.229452f, -0.208872f,
  0.026287f,  -0.364040f, -0.063696f, -0.227443f, -0.234401f, -0.205699f,
  -0.267238f, -0.494125f, -0.056255f, 0.053715f,  -0.487754f, 0.014818f,
  0.087383f,  -0.077556f, -0.168085f, -0.436851f, -0.276286f, -0.137845f,
  -0.107606f, -0.103653f, -0.233766f, -0.419083f, 0.169185f,  0.010186f,
  -0.001587f, 0.086735f,  -2.465718f, 1.482185f,  1.621193f,  -2.081680f,
  1.386553f,  -3.204335f, -0.267111f, -0.004508f, 0.164712f,  0.274147f,
  1.724306f,  -2.273659f, 0.749574f,  -0.891905f, 0.105965f,  -0.030428f,
  -0.416018f, -0.300762f, 0.122911f,  -0.316908f, -0.292504f, 0.138666f,
  -0.161327f, -0.042143f, -0.249128f, 0.149210f,  -0.088987f, -0.654101f,
  -1.501843f, 0.216777f,  0.955914f,  0.524158f,  -1.642561f, -1.643626f,
  0.864797f,  -0.425451f, -2.115764f, -0.012502f, 0.065172f,  1.297270f,
  0.018845f,  1.167276f,  -0.470970f, -0.244995f, 0.374782f,  -1.811056f,
  -0.055430f, -0.024102f, -0.376519f, -0.339640f, -0.119177f, -0.277995f,
  -0.290095f, -0.081362f, -0.144139f, -0.118037f, -0.180357f, -0.217559f,
  -0.370683f, 0.172816f,  -0.265069f, 0.194321f,  -0.273478f, 0.037442f,
  -0.235552f, -0.078625f, -0.447541f, 0.016836f,  -0.271123f, -0.171481f,
  -0.321477f, -0.184826f, -0.442981f, -0.227273f, -0.370666f, -0.237232f,
  -0.257493f, -0.225714f, -0.153716f, -0.283487f, -0.155399f, 0.067697f,
  0.230343f,  -0.034318f, -0.022687f, -0.047090f,
};

static const float av1_ab_partition_nn_bias_64_layer0[64] = {
  -0.212182f, -0.233725f, -0.758846f, -0.158162f, 0.614743f,  -0.150944f,
  -0.075727f, -0.208414f, 1.054996f,  0.713758f,  -0.300051f, -0.151482f,
  -2.443570f, 0.430590f,  -0.129001f, -0.160733f, -0.230547f, -0.143228f,
  -0.140577f, -0.086812f, -0.212298f, -0.159557f, -0.055647f, -0.211423f,
  0.578161f,  -0.220318f, -0.210107f, -3.111584f, 0.604419f,  -0.232622f,
  -0.209924f, -0.130794f, -0.084097f, -0.036005f, 0.294594f,  -2.535531f,
  -0.209783f, -0.211189f, -2.766337f, 0.000000f,  0.450177f,  -1.754884f,
  3.262664f,  -0.209691f, -0.614886f, -0.211257f, -0.109096f, -0.190492f,
  -0.109007f, -0.026910f, -0.136035f, -0.212321f, -0.139320f, -0.212233f,
  -0.305430f, 0.739171f,  0.991277f,  -0.088150f, 0.086313f,  -0.023379f,
  -0.125366f, -0.063576f, -0.212169f, -0.047463f,
};

static const float av1_ab_partition_nn_weights_64_layer1[64 * LABEL_SIZE] = {
  -0.036800f, 0.528721f,  0.490767f,   0.144409f,  1.103640f,  0.361910f,
  -0.180069f, 0.068033f,  -14.868382f, 0.359013f,  0.322567f,  -0.199212f,
  0.906164f,  -0.488254f, 0.149653f,   -0.216394f, -0.099347f, 0.004936f,
  -0.111391f, 0.074848f,  -0.041709f,  0.147627f,  -0.018905f, 0.096116f,
  0.184817f,  -0.016241f, 0.115739f,   2.376754f,  0.637097f,  0.052954f,
  0.136428f,  0.225267f,  -0.181873f,  -0.142876f, 0.684048f,  0.658791f,
  0.105795f,  0.241705f,  1.381114f,   -0.209379f, 1.145949f,  0.795293f,
  -9.361877f, 0.198302f,  0.539600f,   0.092317f,  -0.081695f, 0.200777f,
  0.102334f,  0.081583f,  0.060948f,   -0.025110f, 0.160951f,  -0.020170f,
  0.234006f,  -0.029369f, 0.375036f,   0.270209f,  -0.556529f, 1.402949f,
  0.101777f,  -0.027331f, 0.004502f,   -0.153166f, -0.116651f, 0.151573f,
  -0.022187f, 0.144044f,  -0.108719f,  -0.129942f, -0.270321f, 0.227363f,
  1.892330f,  -0.661052f, -0.219398f,  -0.229417f, -0.856438f, -1.196988f,
  -0.081774f, 0.078847f,  -0.207057f,  -0.048947f, 0.152073f,  -0.243056f,
  -0.233329f, -0.288689f, -0.158333f,  -0.141177f, -0.715436f, 0.016947f,
  -0.093752f, 0.204984f,  -1.209782f,  0.155683f,  0.092239f,  0.146495f,
  0.813146f,  -0.027757f, 0.330982f,   2.173948f,  -0.028867f, -0.141815f,
  0.292708f,  -0.204794f, 0.014496f,   1.032799f,  1.312155f,  0.107020f,
  0.824752f,  -0.013945f, 0.184829f,   -0.041633f, 0.215300f,  -0.476088f,
  -0.053213f, 0.126862f,  -0.020777f,  0.082893f,  -0.223727f, -0.923063f,
  0.466529f,  0.082140f,  -0.845758f,  -1.140791f, -0.262033f, 0.138491f,
  0.151717f,  -0.182479f, -0.131128f,  0.055411f,  0.106771f,  0.125552f,
  0.297184f,  -0.257403f, -0.059884f,  -0.274903f, 2.694357f,  -0.108244f,
  0.025377f,  0.043092f,  -0.558317f,  3.517159f,  -0.270833f, -0.240676f,
  0.205100f,  -0.057068f, -0.140445f,  -0.193449f, -0.030061f, -0.286762f,
  -0.467523f, -0.012647f, 0.190564f,   0.022394f,  -0.101479f, 0.339684f,
  -0.902743f, -0.169578f, -0.178029f,  -0.041836f, -3.952108f, -0.028298f,
  -0.221137f, -0.733895f, -0.223895f,  0.039012f,  0.687867f,  0.021423f,
  0.113063f,  0.676087f,  -0.961000f,  -0.064847f, 0.712856f,  -0.192765f,
  -0.001132f, 0.016689f,  -0.236020f,  -0.766186f, -0.175729f, 0.012879f,
  -0.251064f, -0.105523f, -0.039212f,  -0.347584f, 0.304352f,  -0.034174f,
  -0.364258f, -0.685252f, -0.266115f,  -0.247345f, -0.155905f, 0.152283f,
  -0.156315f, 0.174082f,  -0.757654f,  0.102303f,  -2.192316f, -0.245815f,
  0.119882f,  -0.086542f, 1.987246f,   -1.353163f, -0.374813f, -0.233504f,
  -1.980895f, 0.692093f,  -0.168351f,  0.172700f,  -0.009052f, -0.015734f,
  0.106679f,  -0.060472f, -0.256813f,  -0.074874f, -0.207488f, -0.329515f,
  -0.418268f, -0.017940f, -0.036081f,  0.064719f,  -1.488016f, 0.020591f,
  -0.176325f, -0.141074f, 0.944494f,   0.150237f,  -0.249805f, -0.277280f,
  0.012686f,  0.132483f,  0.116123f,   0.013737f,  -0.116091f, 0.750340f,
  3.251343f,  -0.188864f, 1.096992f,   0.058467f,  -0.041433f, -0.037937f,
  -0.133294f, -0.137908f, -0.171132f,  0.106362f,  0.069383f,  -0.052662f,
  -0.177883f, -0.408049f, 0.680221f,   -0.117035f, -0.904240f, -1.395228f,
  0.154527f,  0.134427f,  0.022767f,   -0.158886f, -0.230316f, 0.161096f,
  0.362213f,  -0.235060f, -0.941620f,  0.055912f,  -0.049458f, -0.166632f,
  0.481418f,  0.930146f,  0.041108f,   0.033674f,  1.372066f,  -1.847709f,
  0.003324f,  0.259534f,  0.177014f,   -0.202761f, -0.262017f, -0.190852f,
  -0.102839f, 0.028338f,  0.187193f,   -0.041684f, 0.123973f,  -0.198576f,
  -0.110369f, -1.431400f, 0.208369f,   -0.302370f, -0.248549f, 0.062985f,
  0.673409f,  0.036662f,  -0.711340f,  -0.120584f, -0.189789f, 0.098812f,
  2.947819f,  0.216567f,  -0.414472f,  -0.181742f, 1.873779f,  -0.222726f,
  -0.782870f, 0.007889f,  0.015062f,   -0.554328f, 0.182928f,  -0.191430f,
  0.123636f,  -0.215460f, -0.225245f,  0.251516f,  -0.013025f, -1.359595f,
  -0.750602f, 0.342667f,  -0.141899f,  -0.687493f, -0.072639f, 0.048018f,
  -0.242107f, -0.031917f, -0.287472f,  -0.046088f, 0.832197f,  -0.016576f,
  -1.553349f, -0.216341f, 0.023077f,   -0.410867f, 4.243743f,  -0.514878f,
  -0.066007f, -0.160696f, -0.262678f,  -0.648790f, -0.430586f, 0.199940f,
  -0.202496f, -0.222241f, -0.016406f,  -0.121473f, 0.000828f,  -0.081584f,
  -0.152641f, -0.190166f, 0.644400f,   0.040196f,  -0.302104f, -1.143654f,
  -0.160327f, -0.320780f, -0.187006f,  0.037311f,  0.440618f,  -0.070733f,
  -0.117785f, 1.527539f,  -0.419310f,  0.001300f,  1.389956f,  -0.036366f,
  -0.269203f, 0.612265f,  2.721897f,   -0.086836f, -0.446999f, 0.012525f,
  -0.078317f, -0.287052f, -0.111188f,  -0.085181f, -0.164667f, -0.010466f,
  -0.569722f, -0.018888f, -0.101663f,  -1.147130f, -0.465204f, 0.114524f,
  -2.192402f, -0.221325f, 0.375748f,   0.206284f,  -0.261548f, -0.246257f,
  -0.143004f, -0.069981f, -0.057306f,  -0.116481f, -0.435903f, -0.314970f,
  0.013210f,  -0.010175f, 4.630571f,   -0.473226f, -0.197199f, -0.028204f,
  0.122907f,  2.475548f,  0.025011f,   -0.092603f, -0.127561f, -0.151330f,
  -0.077295f, 0.245016f,  -0.045005f,  0.183396f,  -0.330556f, -0.384887f,
  0.356374f,  -0.016618f, -0.463353f,  -1.291546f, -0.071986f, -0.311599f,
  0.072385f,  -0.430786f, -2.094788f,  0.202733f,  -0.910109f, -1.336543f,
  -0.086800f, -0.096413f, 1.544383f,   0.031860f,  -0.796211f, 0.762786f,
  3.250022f,  -0.441798f, -0.698537f,  0.062839f,  0.033525f,  -0.362996f,
  0.027022f,  -1.131264f, -0.228926f,  0.053885f,  -0.338628f, 0.155037f,
  -0.046844f, -0.888172f, -0.241767f,  0.084965f,  -0.617743f, -0.049896f,
  -0.036894f, -0.304783f, -0.002639f,  0.137957f,  0.052121f,  -0.131161f,
  -0.117200f, -0.253380f, -0.205561f,  -0.302450f, -0.047397f, -0.330518f,
  3.613420f,  -1.525951f, -0.026738f,  0.209150f,  -2.103534f, 2.019689f,
  -0.366199f, -0.095260f, 0.027417f,   -0.242512f, 0.162579f,  0.052113f,
  -0.293851f, -0.068138f, -0.005799f,  -0.344696f, -0.114824f, -0.431107f,
  -0.120058f, -1.139926f, -1.048379f,  0.036446f,  -0.323020f, -0.432945f,
  0.454151f,  -0.140058f, 0.050649f,   -0.094900f, -0.017278f, -0.238719f,
  1.193153f,  0.120447f,  -0.496061f,  0.917431f,  2.936126f,  -0.115521f,
  -0.347397f, -0.435325f, -0.004383f,  -0.211864f, 0.162383f,  -1.040726f,
  0.089537f,  -0.128579f, -0.133505f,  0.107129f,  -0.435657f, -0.180388f,
  0.043650f,  0.018709f,  -0.773242f,  -0.687192f, -0.120633f, -0.063626f,
  0.029912f,  0.113972f,  -0.403502f,  -0.127640f, -0.269625f, 0.129794f,
  -0.188539f, 0.041641f,  0.029769f,   -0.198374f, 1.401407f,  0.353887f,
  -0.219925f, 0.260515f,  1.157034f,   -2.992044f, -0.097618f, -0.064417f,
  -0.203626f, -0.008217f, -0.112339f,  -0.227407f, -0.155118f, 0.247705f,
  -0.012304f, -0.248447f, -0.913463f,  -0.064788f, -0.214619f, -0.251761f,
  -0.386861f, -0.040574f, -0.163219f,  -0.100700f, 1.488274f,  -0.071684f,
  -0.033626f, -0.006497f, -0.246945f,  -0.145221f, -3.747390f, 0.149609f,
  -0.263326f, -0.297385f, -1.039896f,  -0.083174f, -0.025473f, -0.235586f,
  -0.001087f, 0.254286f,  0.265106f,   0.007325f,  0.199239f,  0.134103f,
  -0.578211f, -0.259801f, -0.062373f,  2.368348f,  0.560556f,  -0.252260f,
  0.889997f,  -0.447872f, -0.059218f,  -0.095315f, -0.061667f, 0.183580f,
  -0.157479f, 0.055387f,  -0.831734f,  0.007606f,  -1.104906f, 0.301180f,
  -0.117115f, 0.212959f,  4.727223f,   -0.243833f, -0.397495f, -0.025021f,
  -0.367587f, -2.082058f, -0.217699f,  0.148111f,  0.252430f,  0.111088f,
  -0.260692f, 0.095124f,  -0.407774f,  -0.322169f, 0.002927f,  0.126169f,
  -1.272325f, -0.279772f, -0.373680f,  -0.485177f, -0.605458f, 0.021225f,
  -0.092031f, -0.226585f, 1.895162f,   0.037866f,  -0.275475f, 1.614360f,
  -0.014972f, -0.277679f, -3.449082f,  -0.092060f, -0.747873f, 0.020716f,
  2.776178f,  -0.049963f, 0.183999f,   -0.295259f, -0.028868f, 0.221895f,
  0.001265f,  0.336823f,  0.219372f,   0.112824f,  0.408132f,  -0.017940f,
  -0.311666f, 1.489606f,  -0.058093f,  -0.305659f, -0.491933f, -0.143847f,
  0.166115f,  0.042867f,  -0.123447f,  -0.087099f, -0.305395f, -0.365079f,
  -0.755801f, -0.160649f, 0.736260f,   -0.008611f, 0.095836f,  -0.017345f,
  5.697515f,  -0.498971f, -0.125280f,  0.199907f,  0.300053f,  0.605026f,
  -0.228225f, -0.259523f, 0.016384f,   0.146973f,  0.210258f,  0.226766f,
  -0.075178f, -0.050924f, 0.188496f,   -0.415266f, -0.484880f, -0.236384f,
  0.071931f,  -0.331863f, -0.601243f,  -0.232479f, -0.285272f, 0.123789f,
  -1.341333f, 0.037082f,  -0.315202f,  -1.587215f, -0.271576f, 0.003216f,
  -4.437186f, -0.256205f, -0.576589f,  -0.114147f, 2.153916f,  -0.369618f,
  0.271415f,  0.145036f,  -0.158731f,  -0.240938f, -0.187369f, 0.036325f,
  0.254771f,  0.211488f,  -0.240297f,  0.098417f,  -0.415011f, 2.334793f,
  -0.127252f, 0.020069f,  -0.168755f,  -0.448922f, -0.219207f, 0.016232f,
  -0.221935f, -0.269500f, -0.100636f,  0.102545f,  -0.809376f, -0.054979f,
  0.360713f,  -0.326541f, 0.112933f,   0.138073f,  4.229404f,  -0.763801f,
  -0.305429f, 0.199955f,  -1.787713f,  0.272866f,  0.109895f,  0.138466f,
  -0.250259f, -0.167162f, -0.212588f,  -0.217589f, -0.067125f, -0.077490f,
  -0.208970f, -0.006863f, -0.671146f,  -0.298320f, -0.165509f, 0.044597f,
  -1.408624f, -0.213957f, -0.220947f,  0.129718f,  1.316777f,  -0.098928f,
  -0.008121f, -0.558293f, -0.297290f,  -0.218873f, -4.346638f, -0.228174f,
  -0.204710f, -0.388864f, 2.697919f,   0.025260f,  0.857020f,  0.009921f,
  0.036915f,  -0.320275f, -0.087937f,  0.022636f,  0.236667f,  0.135496f,
  -0.059616f, -0.192955f, 0.009470f,   2.139589f,  -0.200449f, 0.129818f,
  1.017444f,  -0.608299f, 0.257914f,   -0.134306f, -0.033327f, 0.002855f,
  -0.338598f, 0.015559f,  0.117362f,   -0.166760f, 0.086903f,  -0.167666f,
  0.193523f,  0.033852f,  -1.147686f,  0.489468f,  -0.006969f, 0.125630f,
  1.557907f,  -1.604449f, -0.071114f,  0.096178f,  0.007065f,  0.200013f,
  0.213393f,  0.168466f,  -0.100568f,  -0.117861f, -0.161542f, -0.072561f,
  -1.069871f, -0.470138f, -0.352578f,  -1.503513f, -0.001394f, -0.380109f,
  0.065089f,  -0.281668f, 0.988953f,   -0.002778f, -0.659026f, -0.470692f,
  -0.407292f, 0.011710f,  -1.362085f,  0.184738f,  -0.135786f, -1.374241f,
  4.487930f,  -0.067274f, -0.956404f,  -0.233995f, 0.224527f,  -0.454556f,
  0.037900f,  -0.281658f, 0.208224f,   -0.254753f, 0.045740f,  0.051444f,
  -0.388281f, 0.257112f,  -0.485030f,  -0.082659f, 0.148103f,  -1.007456f,
  -0.022295f, 0.036984f,  -0.369401f,  -0.076943f, -0.007636f, -0.293022f,
  0.470466f,  0.199012f,  -2.158182f,  0.036577f,  -0.014725f, -0.229516f,
  2.236929f,  0.030945f,  -0.400045f,  0.109348f,  0.214691f,  -0.891516f,
  -0.251379f, -0.217358f, 0.013733f,   0.205573f,  -0.151725f, -0.191782f,
  -0.339630f, -0.163905f, -0.119191f,  -0.032516f, 0.503015f,  0.025772f,
  0.029094f,  -1.146153f, 0.216723f,   -0.330023f, 0.064695f,  -0.262521f,
  0.425612f,  -0.093080f, -0.489648f,  1.051293f,  -0.092332f, 0.095557f,
  -0.874132f, 0.218483f,  -0.127648f,  -1.605802f, 2.763617f,  -0.186734f,
  -1.243166f, -0.193514f, -0.173748f,  0.337822f,  0.183873f,  -0.251594f,
  -0.211582f, 0.144081f,  0.029620f,   -0.024853f, -0.385140f, 0.467341f,
  -0.928316f, -0.195442f, 0.917783f,   0.357084f,  0.174445f,  -0.073659f,
  -0.012811f, -0.115420f, -0.181147f,  -0.364449f, -0.567395f, -0.012969f,
  -1.680714f, 0.065323f,  0.198063f,   -0.244201f, 1.428545f,  -0.432539f,
  -0.208931f, -0.091205f, 0.957125f,   0.813519f,  -0.262677f, 0.246852f,
  0.015536f,  0.055026f,  0.067054f,   0.262103f,  -0.358115f, -0.095206f,
  -0.267522f, -0.402710f, -0.680397f,  -0.123627f, -0.385590f, -1.504680f,
  -0.169513f, -0.215338f, 0.043633f,   -0.079052f, -0.464410f, 0.122894f,
  -0.278231f, -2.456445f, -0.159917f,  -0.015597f, -0.735449f, -0.078854f,
  -0.400290f, -1.153870f, 3.657228f,   -0.287093f, -1.174355f, -0.102001f,
  -0.288281f, 0.185209f,  -0.145228f,  -0.200449f, -0.099914f, -0.138354f,
  0.254428f,  -0.161751f, -0.118206f,  0.296043f,  -0.482613f, 0.080932f,
  1.097605f,  -0.010190f, 0.232439f,   0.447617f,  -0.133508f, 0.115763f,
  -0.388589f, 0.174695f,  -0.236014f,  0.006284f,  -1.374129f, 0.092015f,
  -0.241419f, -0.231667f, 2.763950f,   -0.922932f, -0.061605f, 0.208740f,
  -1.597190f, 1.353325f,  -0.198528f,  0.250498f,  -0.013950f, -0.203861f,
  -0.254563f, 0.081931f,  -0.413369f,  0.011844f,  0.080961f,  -0.231161f,
  -1.234909f, -0.440843f, -0.174980f,  -0.315283f, -0.337474f, -0.123243f,
  -0.310001f, -0.271028f, 0.364179f,   0.022845f,  -0.535517f, -0.772936f,
  -0.188435f, 0.039667f,  -0.807463f,  0.266550f,  -0.288857f, -1.630789f,
  1.280155f,  0.065712f,  -0.279960f,  -0.300056f, 0.258440f,  -0.073781f,
  0.213878f,  0.042196f,  0.021360f,   0.211698f,  -0.003751f, -0.192673f,
  -0.137008f, 0.247878f,  -0.470604f,  0.073164f,  1.523241f,  0.734755f,
  -0.114126f, -0.193834f, -0.025759f,  0.263183f,
};

static const float av1_ab_partition_nn_bias_64_layer1[LABEL_SIZE] = {
  -0.343508f, -0.706936f, -0.160676f, -0.877101f, -0.517567f, -0.253254f,
  -0.148074f, 0.923430f,  -0.364770f, 0.203550f,  0.401216f,  0.938246f,
  -0.872737f, 0.718723f,  0.703398f,  2.560015f,
};

static const NN_CONFIG av1_ab_partition_nnconfig_64 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      64,  // num_hidden_nodes
  },
  {
      av1_ab_partition_nn_weights_64_layer0,
      av1_ab_partition_nn_weights_64_layer1,
  },
  {
      av1_ab_partition_nn_bias_64_layer0,
      av1_ab_partition_nn_bias_64_layer1,
  },
};

// nn model for ab partition pruning, 32x32.
static const float av1_ab_partition_nn_weights_32_layer0[FEATURE_SIZE * 64] = {
  -0.323723f, -0.214013f, -0.007772f, -0.458851f, -0.125542f, -0.123860f,
  -0.410973f, -0.209389f, -0.087580f, -0.272881f, -0.168500f, -1.130845f,
  0.344916f,  -0.475017f, -0.362262f, -0.195662f, -0.566124f, 0.782163f,
  0.411575f,  -0.013378f, -0.318650f, -0.124678f, -0.612909f, -0.315788f,
  -0.263990f, -0.508783f, -0.048938f, -0.416407f, -0.402648f, -0.156644f,
  0.225887f,  -0.000493f, 2.682241f,  0.871204f,  0.059014f,  0.803542f,
  -1.407028f, -1.154669f, 1.388148f,  -0.293348f, -0.003669f, -0.009607f,
  1.330030f,  -0.337841f, 2.118617f,  1.033059f,  -0.084788f, 0.212904f,
  0.082405f,  -0.070579f, -0.494005f, -0.173392f, 0.039546f,  -0.463865f,
  0.077163f,  -0.434066f, 0.030835f,  -0.427139f, -0.560520f, -0.031606f,
  -0.368541f, -0.027458f, 0.370574f,  0.461418f,  1.087682f,  -0.572137f,
  -1.509596f, -0.765697f, -0.499383f, -0.277998f, -0.106492f, -0.129564f,
  -0.169133f, -0.269834f, -0.114270f, -0.275431f, 0.016339f,  -0.156744f,
  -0.267922f, 0.171216f,  0.110556f,  0.002954f,  -0.200327f, -0.187663f,
  3.691601f,  1.234152f,  0.186315f,  -0.125370f, -0.211235f, -0.554432f,
  -0.131072f, -0.124982f, -0.130339f, -0.235350f, 0.018903f,  0.012896f,
  -0.159372f, -0.269571f, -0.025709f, -0.221251f, 0.061919f,  0.016307f,
  0.384673f,  -0.134525f, -1.599126f, -0.416459f, -0.743052f, 0.670249f,
  -0.169709f, 0.421681f,  -0.033360f, -0.072817f, 0.003647f,  -0.110632f,
  -0.158651f, -0.095136f, 0.223759f,  0.165767f,  -0.269129f, -0.196075f,
  -0.023183f, -0.293420f, 0.014875f,  0.018688f,  -0.153407f, -0.172009f,
  -0.259947f, -0.124015f, 0.173653f,  -0.089103f, -0.021001f, -0.334230f,
  0.027177f,  0.103371f,  -0.183860f, -0.204051f, -0.023721f, -0.192297f,
  -0.143771f, -0.247106f, 0.218116f,  -0.013240f, 2.831783f,  1.483928f,
  -0.877025f, -0.313462f, -0.411320f, -0.447825f, 0.605977f,  0.234684f,
  -0.119150f, -0.075182f, -0.330463f, 0.071503f,  -0.254924f, -0.360071f,
  -0.037022f, 0.063261f,  -0.148759f, -0.238254f, -0.462018f, -0.027166f,
  0.065318f,  -0.235743f, -0.257194f, -0.094784f, 0.022423f,  0.055925f,
  0.086672f,  -0.021010f, 0.009965f,  -0.001648f, -0.104917f, -0.387443f,
  -0.102673f, -0.281706f, 0.145923f,  -0.233391f, -0.378365f, -0.145584f,
  -0.077751f, -0.121166f, 1.134565f,  -0.097500f, -0.749202f, -0.544566f,
  -1.361374f, -0.102494f, 1.089275f,  0.375299f,  -0.105091f, 0.037641f,
  -0.054248f, -0.282691f, -0.377797f, -0.066427f, -0.253815f, -0.329677f,
  -0.339326f, -0.128217f, -0.282905f, 0.014937f,  1.067185f,  -0.171764f,
  0.484458f,  0.396706f,  -0.557055f, -0.891596f, -0.257839f, -0.720879f,
  -0.218449f, -0.004755f, 1.572857f,  0.006229f,  1.962895f,  -0.029746f,
  -4.137691f, -2.185991f, -2.763477f, -0.520437f, -0.208708f, 0.006444f,
  -1.263078f, -0.304560f, 1.072374f,  2.556429f,  0.312850f,  0.257488f,
  -0.634264f, 0.156769f,  -0.188943f, 0.040295f,  -0.389915f, 0.085250f,
  -0.248525f, 0.045667f,  -0.776115f, -0.274680f, -0.448145f, -0.566161f,
  -1.285316f, 0.079060f,  0.389124f,  -0.510401f, -0.015299f, -0.664661f,
  0.099901f,  -0.470694f, -0.051593f, -1.076381f, -0.442104f, -0.197867f,
  -0.330011f, -0.448523f, -0.301018f, -0.442093f, -0.491953f, -0.582091f,
  -0.064569f, -0.156516f, 0.543522f,  -0.005924f, 0.161432f,  0.974793f,
  0.273712f,  1.104850f,  -0.290312f, 0.313417f,  -0.125370f, 0.136234f,
  -0.191227f, -0.165054f, 0.011872f,  -0.298871f, 0.095740f,  0.142760f,
  -0.215771f, -0.031437f, 0.101041f,  -0.085620f, 0.435387f,  0.002786f,
  1.971375f,  0.018392f,  -1.771940f, -0.401433f, 0.808263f,  -3.350013f,
  2.296952f,  -1.024403f, -0.041645f, -0.034799f, -0.024078f, -0.347301f,
  -0.276088f, -0.455907f, 0.266021f,  0.087348f,  -0.146566f, 0.040492f,
  -0.539866f, -0.206851f, -0.387874f, -0.125508f, -0.496676f, -0.373845f,
  -0.472356f, -0.357082f, -0.081254f, -0.456466f, 0.554713f,  0.002185f,
  -4.225019f, 0.344025f,  0.728796f,  -0.262936f, 1.383924f,  1.577300f,
  -2.653320f, -2.516156f, -0.301604f, -0.204105f, -0.138252f, -0.587536f,
  -0.097889f, -0.352414f, -0.288276f, -0.184340f, -0.122741f, -0.243376f,
  0.031970f,  -0.373402f, -0.396079f, 0.045566f,  0.072595f,  -0.222681f,
  -0.243802f, -0.340129f, -0.258494f, -0.192041f, -0.386112f, -0.240940f,
  -0.047268f, -0.555802f, -0.032514f, -0.241341f, -0.167463f, -0.478308f,
  -0.205936f, -0.316275f, 0.103729f,  -0.197893f, -0.128029f, -0.218796f,
  -0.167362f, -0.111814f, -0.126062f, -0.394260f, -0.025357f, -0.402697f,
  -0.587395f, -0.400385f, -0.259664f, -0.415588f, -0.338503f, -0.399166f,
  -0.270504f, 0.234505f,  0.272144f,  0.266938f,  -0.392395f, -0.011717f,
  -0.384221f, -0.473446f, -0.038420f, -0.241101f, -0.234402f, -0.275567f,
  -0.410454f, -0.377599f, -0.179099f, -0.138432f, -0.248083f, -0.543026f,
  -0.428043f, -0.239895f, -0.333193f, -0.103346f, -0.039038f, -0.171109f,
  -0.119432f, -0.222351f, 0.000450f,  0.208724f,  -0.510526f, -0.144656f,
  -0.316721f, -0.344846f, -0.244794f, -0.129134f, -0.045634f, -0.400183f,
  0.043714f,  -0.235414f, 0.115594f,  -0.195616f, -0.106693f, -0.124242f,
  0.083990f,  0.049110f,  -0.196130f, -0.059860f, -0.464235f, -0.516443f,
  -0.101521f, -0.422379f, -0.413955f, -0.042991f, -0.345263f, -0.129264f,
  -0.106911f, -0.140156f, -0.457841f, -0.199848f, -0.218954f, -0.329850f,
  -0.364097f, -0.335262f, -0.312254f, -0.299331f, -0.052710f, -0.251019f,
  -0.023459f, -0.222538f, 0.028849f,  -0.088038f, -0.301550f, -0.273566f,
  0.067295f,  -0.174608f, -0.445784f, -0.158366f, -0.567275f, -0.557652f,
  -0.353503f, -0.302092f, -0.302049f, -0.551793f, -0.034535f, -0.225190f,
  -0.210733f, -0.219377f, -0.057197f, -0.430933f, -0.025185f, -0.388150f,
  -0.086147f, -0.430088f, 0.058466f,  -0.152129f, -0.058411f, -0.236392f,
  -0.547669f, -0.613849f, -0.893774f, -0.351715f, -0.399227f, -0.454909f,
  -0.324501f, 0.000490f,  -0.282167f, -0.073163f, -0.281452f, 0.047932f,
  -0.175500f, 0.165220f,  -0.276212f, 0.062153f,  -0.217054f, -0.255487f,
  -0.146416f, -0.097718f, -0.173809f, -0.559328f, -0.055695f, -0.391193f,
  -0.132020f, -0.561184f, -0.308666f, -0.474053f, -0.219149f, -0.246558f,
  -0.158325f, 0.151907f,  -0.266835f, -0.144697f, -0.193960f, -0.046587f,
  -0.220028f, -0.247355f, 0.135584f,  0.016511f,  0.367705f,  -1.855877f,
  0.435622f,  0.444710f,  -3.372301f, -3.030489f, 1.013267f,  0.380951f,
  -0.170011f, -0.111415f, -0.456146f, -0.107254f, -0.095220f, -0.053078f,
  -0.135864f, -0.591949f, -0.252810f, -0.324799f, -0.094796f, -0.260969f,
  -0.391981f, -0.063170f, -0.336130f, -0.470127f, -0.405168f, -0.433219f,
  -0.309563f, -0.295462f, -0.552270f, -0.012300f, -0.057793f, -0.034494f,
  -0.446843f, -0.640160f, -1.188681f, -0.791361f, 0.543271f,  1.189112f,
  1.458468f,  -0.005876f, -0.927475f, 0.062038f,  -1.170818f, 0.338227f,
  -3.007096f, -4.559296f, -4.045457f, -5.953635f, -0.228386f, -0.266890f,
  -0.092595f, -0.377440f, -0.044534f, -0.053565f, -0.349268f, -0.415030f,
  -0.310094f, 0.062721f,  0.251422f,  -0.014350f, -1.282910f, 1.619560f,
  1.180566f,  -0.032163f, -1.322951f, -0.603601f, 1.443710f,  0.654650f,
  -0.393227f, 0.003536f,  0.029725f,  -0.108925f, -0.053911f, 0.133977f,
  -0.036145f, -0.168438f, 0.046989f,  -0.331463f, -0.176983f, -0.311922f,
  -0.272389f, -0.379592f, -0.399993f, -0.297873f, -0.193425f, -0.177524f,
  -0.258309f, -0.567312f, -0.260217f, -0.241869f, 0.024010f,  -0.032867f,
  -0.039424f, -0.063670f, 0.193808f,  -0.303514f, -0.013376f, -0.057761f,
  0.187922f,  0.006938f,  0.031810f,  0.180594f,  -1.198427f, 2.820662f,
  0.154986f,  -0.375518f, 0.116925f,  -0.795782f, -0.085139f, -0.079365f,
  -0.197936f, -0.321468f, -0.205271f, -0.558203f, -0.296235f, -0.151193f,
  -0.158282f, -0.245402f, -0.208504f, -0.042335f, -0.087426f, -0.557129f,
  -0.381427f, -0.441551f, -0.541011f, -0.060567f, -0.469305f, -0.032326f,
  -2.453587f, -0.045568f, -0.296932f, 0.613061f,  -0.320284f, 0.191620f,
  -0.827145f, -0.225277f, 0.275800f,  1.696635f,
};

static const float av1_ab_partition_nn_bias_32_layer0[64] = {
  -0.176206f, 0.660189f,  -0.186156f, -2.481963f, -1.564218f, -0.280424f,
  0.732684f,  -0.135581f, -2.193132f, -0.172771f, 0.605001f,  -0.060392f,
  -0.067190f, -0.132969f, -1.410812f, -0.298701f, -0.105963f, -0.086173f,
  0.632779f,  0.005585f,  1.310169f,  1.392136f,  -0.563860f, -0.051053f,
  0.660998f,  -0.214726f, -1.894342f, -0.128288f, -0.330721f, -0.053988f,
  -0.177726f, 1.200859f,  -0.178902f, -0.172620f, -0.184476f, -0.175559f,
  0.538503f,  -0.322158f, -0.219080f, -0.058208f, -0.171347f, -0.216060f,
  -0.174950f, -0.295740f, -0.184820f, -0.213896f, 1.317728f,  -0.020116f,
  -0.208096f, 0.000000f,  1.246166f,  -0.225421f, -0.181555f, 0.861761f,
  1.172429f,  -0.172892f, -0.737092f, -0.189904f, -0.179385f, -0.114618f,
  -1.384604f, -0.201713f, -0.271948f, 0.372351f,
};

static const float av1_ab_partition_nn_weights_32_layer1[64 * 16] = {
  -0.037828f,  1.529029f,  0.004927f,  1.475763f,  0.627172f,  0.325872f,
  -0.990757f,  0.129476f,  0.889958f,  -0.082031f, 0.332133f,  0.074422f,
  -0.176212f,  -0.074355f, 0.774378f,  0.110987f,  -0.155469f, 0.253310f,
  0.882538f,   0.253605f,  0.332436f,  -5.389474f, 0.278470f,  0.168644f,
  0.914611f,   0.154165f,  0.809262f,  -0.174734f, 0.923673f,  0.064716f,
  -0.070228f,  -0.228735f, 0.002312f,  0.112222f,  -0.045502f, -0.046004f,
  0.514101f,   0.306480f,  0.021232f,  -0.015955f, -0.288260f, 0.189177f,
  -0.104158f,  0.103273f,  0.096910f,  -0.086328f, 1.327289f,  -0.154247f,
  0.056676f,   -0.243327f, -0.646676f, 0.177221f,  -0.086761f, 0.729729f,
  -14.710893f, -0.044881f, 0.339003f,  -0.134737f, 0.073621f,  -0.162913f,
  1.215237f,   0.140723f,  0.138630f,  1.241719f,  0.204092f,  -0.463080f,
  -0.176086f,  1.125868f,  1.034814f,  0.225455f,  -0.203421f, -0.078787f,
  -0.527498f,  0.012491f,  -0.563307f, -0.170792f, 0.002679f,  0.116153f,
  0.211348f,   -0.191900f, -0.212505f, 0.263445f,  -0.074679f, -0.081441f,
  -0.815405f,  2.448215f,  0.781299f,  0.149542f,  -1.045162f, 0.043014f,
  0.217381f,   -0.094500f, -0.090427f, 0.025784f,  -0.228906f, -2.741798f,
  0.230475f,   -0.256112f, -0.103297f, 0.159121f,  -0.229793f, -0.014883f,
  -0.104131f,  -0.123816f, 0.164148f,  -0.052279f, -0.071845f, -0.041197f,
  0.208527f,   -0.234197f, -0.542336f, 0.020053f,  0.088870f,  0.014346f,
  2.502164f,   -0.010244f, -0.267792f, 0.844394f,  2.711486f,  -0.015262f,
  -0.868053f,  -0.295704f, 0.222289f,  -0.000286f, -0.352098f, -0.079000f,
  0.021267f,   -0.721739f, -0.240558f, -0.384775f, 0.065974f,  -2.161058f,
  0.195889f,   0.268966f,  -0.009329f, 0.014949f,  0.314943f,  0.235885f,
  0.072591f,   -0.127120f, 0.150784f,  0.105697f,  -1.297403f, -0.207509f,
  -0.217688f,  -0.076752f, 0.170952f,  -0.294235f, 0.449973f,  -1.712690f,
  0.860989f,   0.054757f,  -0.812627f, -0.105316f, -0.736230f, -0.133192f,
  -3.741608f,  0.495660f,  -0.288936f, 4.654852f,  -0.021305f, -0.308916f,
  0.049205f,   -0.259996f, 0.114248f,  -0.252647f, -0.253180f, -0.449314f,
  0.022979f,   0.063281f,  -0.196154f, 0.078295f,  -0.322317f, -0.145142f,
  0.300573f,   0.048385f,  -0.254787f, 0.123939f,  -1.263088f, -0.228565f,
  -0.389061f,  0.391084f,  2.322438f,  0.075009f,  0.225743f,  -0.198808f,
  -0.280538f,  -0.173939f, -0.120543f, -0.070792f, -0.417187f, -0.781056f,
  -0.102756f,  -1.760965f, 0.019149f,  -0.867342f, 0.347141f,  0.031588f,
  0.302572f,   -0.203573f, -0.357320f, -0.096078f, -0.527528f, 0.046699f,
  -0.108561f,  -0.167077f, -2.851509f, -0.307116f, 0.202720f,  -0.160280f,
  -0.215525f,  0.064355f,  -0.427220f, 1.516230f,  0.634453f,  0.099400f,
  -1.013887f,  -0.029740f, -0.093426f, -0.044272f, -1.297636f, -0.237614f,
  -0.160953f,  0.399036f,  -0.030685f, -0.113619f, -0.184704f, 0.040519f,
  -0.588252f,  -0.210235f, -0.067623f, -0.031841f, -0.107261f, -0.192582f,
  -0.253959f,  -0.430821f, -0.103184f, -0.280185f, -0.357723f, 0.197761f,
  -0.175087f,  -0.055171f, 1.642014f,  -0.192559f, -0.288147f, 0.610311f,
  4.688195f,   -0.128728f, -0.914869f, -0.108286f, 0.013789f,  0.092125f,
  0.019770f,   -0.178386f, 0.074164f,  -1.152658f, -0.216738f, -0.277286f,
  0.012381f,   0.418259f,  -0.680727f, -0.221917f, -0.485946f, 0.101672f,
  2.009457f,   0.054302f,  1.019838f,  -0.116170f, 0.165134f,  -0.112567f,
  0.852632f,   -0.385796f, -0.108666f, 0.053181f,  -0.311797f, -0.372875f,
  -0.675717f,  2.409268f,  -0.514720f, -0.214245f, -0.646596f, 0.009756f,
  0.203993f,   0.093617f,  -0.301290f, 0.253551f,  -0.128909f, -1.448442f,
  -0.186823f,  -0.278001f, -0.294993f, -0.176928f, -0.473605f, 0.062049f,
  -0.212084f,  -0.137326f, 0.012505f,  0.087850f,  -0.200413f, -0.394119f,
  -0.132224f,  0.146917f,  0.155746f,  0.198725f,  -0.322541f, 0.196391f,
  -0.945500f,  0.036736f,  -0.155646f, -0.677341f, 1.130545f,  -0.339554f,
  0.411628f,   -0.355813f, -0.249843f, 0.213694f,  -2.035607f, 0.055694f,
  -0.111669f,  0.408696f,  -0.067043f, -0.048182f, 0.398110f,  -0.067542f,
  1.459801f,   0.236833f,  -0.178806f, 0.168758f,  0.492387f,  0.099691f,
  -0.776680f,  -0.172865f, 0.204225f,  0.193982f,  0.575685f,  -0.062248f,
  0.011486f,   0.058571f,  -0.493391f, 0.026893f,  -0.900467f, 3.793129f,
  -0.634613f,  -0.064660f, -0.048262f, 0.361905f,  0.033641f,  0.245171f,
  -0.064671f,  0.034954f,  0.204358f,  -0.904023f, -0.052714f, -0.250134f,
  0.136700f,   0.000734f,  -0.371720f, 0.226483f,  0.217958f,  0.060559f,
  0.180111f,   0.000970f,  0.079556f,  -0.096775f, 0.093855f,  -0.026224f,
  -0.243664f,  0.004290f,  0.123281f,  -0.239476f, 1.230374f,  -0.107826f,
  -0.101982f,  -0.153917f, 5.464427f,  0.304375f,  -0.809957f, 0.090564f,
  -0.278416f,  -0.245555f, -2.078421f, 0.243093f,  -0.127666f, 0.052451f,
  -0.126662f,  -0.783505f, 0.025149f,  -1.422675f, -0.207769f, -0.362547f,
  0.115310f,   0.133390f,  1.264754f,  -0.027055f, -0.485312f, -0.240717f,
  -0.239722f,  0.146818f,  -1.265043f, -0.235553f, 0.267104f,  -0.021357f,
  -0.435949f,  -0.309371f, 0.049920f,  1.302721f,  -0.233978f, -0.097551f,
  -0.240631f,  -0.287821f, -0.378380f, -0.273131f, -3.075169f, 0.226404f,
  -0.029361f,  2.703590f,  -0.430659f, 0.067927f,  -0.387520f, -0.370630f,
  -0.229236f,  0.085653f,  -0.370956f, -0.065556f, -0.187859f, 0.068309f,
  -0.109299f,  -0.259898f, -0.103644f, -0.271199f, -0.209350f, 0.140993f,
  -0.196713f,  -0.135508f, -1.423209f, -0.406385f, -0.019956f, -0.864694f,
  5.963707f,   -0.201157f, 0.726377f,  -0.011076f, 0.010553f,  -0.102918f,
  -2.230088f,  -0.258098f, -0.039547f, -0.029262f, -0.082324f, -0.860222f,
  -0.094735f,  -1.381839f, 0.587298f,  -0.173048f, 0.721360f,  0.241900f,
  0.764302f,   -0.023609f, -1.173755f, 0.103912f,  -0.185363f, 0.078435f,
  -2.245062f,  -0.127269f, 0.202234f,  0.158975f,  -0.260909f, 0.098608f,
  -0.348247f,  1.732502f,  -0.412298f, -0.269602f, -0.425771f, -0.146243f,
  -0.530730f,  0.125716f,  -1.004419f, 0.145109f,  -0.059289f, 1.096304f,
  0.012891f,   0.045033f,  -0.306875f, 0.003514f,  -0.176110f, 0.037544f,
  -0.441537f,  -0.518921f, -0.262149f, -0.060407f, -0.379419f, -0.141245f,
  -0.128894f,  -0.176537f, -1.161318f, -0.249100f, -0.118330f, 0.042816f,
  1.173404f,   0.088312f,  -0.393568f, -0.175134f, 6.529819f,  -0.326652f,
  -0.631917f,  -0.393476f, 0.057781f,  -0.217748f, -1.781139f, -0.012614f,
  -0.212621f,  -0.720322f, -0.218498f, -0.388556f, -0.254796f, -0.248399f,
  -0.608744f,  -0.265146f, 0.238517f,  0.066882f,  -2.916806f, 0.054642f,
  0.282590f,   0.075248f,  0.010188f,  -0.133486f, 0.985945f,  -0.045849f,
  -0.347564f,  0.057320f,  -0.417920f, 0.063664f,  0.387062f,  -2.692059f,
  -0.535549f,  0.263736f,  0.327889f,  -0.070273f, -0.775254f, 0.147250f,
  3.309425f,   -0.212191f, -0.067204f, -2.912663f, -0.061496f, 0.084233f,
  0.022907f,   0.138421f,  -0.112159f, -0.288447f, -0.010799f, 0.056049f,
  -0.036527f,  0.021525f,  0.106649f,  -0.291883f, 0.088424f,  -0.057773f,
  -0.086031f,  0.015277f,  -0.318505f, -0.269049f, -1.008913f, -0.224785f,
  -0.025820f,  -0.649037f, 0.706381f,  0.096410f,  0.643776f,  -0.046743f,
  -0.009654f,  -0.024246f, 1.469255f,  -0.183536f, -0.370046f, -0.048442f,
  -0.376527f,  -0.431264f, -0.245109f, -0.093951f, 0.203683f,  -0.099872f,
  0.087210f,   0.160692f,  -3.527694f, -0.068891f, -0.228994f, -0.231817f,
  -0.241949f,  0.193613f,  0.979597f,  -0.091259f, 0.414424f,  -0.047341f,
  -0.209582f,  -0.295134f, -0.016824f, 0.460327f,  -0.072671f, 0.246234f,
  0.235896f,   0.127238f,  -1.068683f, 0.035648f,  2.254888f,  0.180105f,
  -0.260098f,  -2.322120f, -0.184249f, -0.314801f, -0.099969f, -0.272117f,
  -0.237916f,  0.031103f,  -0.274063f, -0.049384f, -0.044917f, 0.102477f,
  -0.342148f,  -0.257558f, -0.346300f, 0.115333f,  -0.115456f, 0.208354f,
  -0.359301f,  -0.167395f, 1.146514f,  -0.177861f, -0.098658f, -0.444570f,
  6.759993f,   -0.369772f, -0.831118f, 0.001866f,  -0.073298f, -0.072095f,
  0.811902f,   -0.431997f, -0.286587f, -0.269500f, 0.111492f,  -0.525364f,
  -0.351785f,  -2.463474f, -1.852659f, 0.135325f,  0.138267f,  0.100643f,
  -2.373278f,  -0.285514f, -0.395388f, -0.185016f, -0.030249f, -0.005767f,
  -0.716424f,  -0.031674f, 0.011147f,  0.057405f,  -0.215873f, -0.094401f,
  0.573528f,   -1.223820f, 0.414852f,  -0.059053f, -0.076488f, -0.287168f,
  -0.842640f,  0.174084f,  -0.567186f, 0.336629f,  -0.062514f, 2.075448f,
  -0.061680f,  -0.131529f, -0.098994f, -0.204111f, -0.347865f, 0.108516f,
  -0.049616f,  -0.069212f, -0.273935f, -0.096545f, -0.210784f, -0.284698f,
  0.141501f,   -0.176924f, -0.361341f, -0.251197f, -0.286694f, 0.245569f,
  -1.521661f,  -0.122639f, -0.015760f, -0.718912f, 5.877828f,  0.146916f,
  0.151767f,   0.220785f,  -0.032298f, 0.230902f,  0.663943f,  -0.252613f,
  0.057718f,   -0.436038f, -0.323994f, -1.139787f, -0.042489f, -1.326298f,
  -1.031206f,  -0.104136f, 0.389897f,  0.127602f,  -2.667789f, -0.212366f,
  -0.506262f,  -0.009115f, -0.213202f, 0.076167f,  -1.629405f, 0.055129f,
  0.375393f,   -0.150272f, -0.241515f, -0.326497f, 0.100069f,  0.410703f,
  0.340622f,   0.042437f,  -0.349945f, 0.041176f,  -1.178950f, 0.030992f,
  0.933908f,   -0.035844f, -0.098660f, 1.030584f,  -0.092043f, -0.355739f,
  -0.305562f,  0.036161f,  -0.049558f, -0.033225f, -0.403856f, -0.088276f,
  0.215493f,   -0.149105f, -0.013363f, 0.025886f,  -0.101306f, -0.205781f,
  -1.072487f,  -0.076019f, 0.077555f,  0.131003f,  1.267763f,  -0.008954f,
  -0.327617f,  -0.246539f, 6.664081f,  -0.404403f, -1.442489f, 0.191301f,
  -0.336361f,  0.181156f,  0.833108f,  0.007879f,  -0.194464f, -1.029408f,
  -0.036268f,  -0.927110f, -0.379190f, -0.293443f, -1.848579f, -0.242548f,
  -0.065990f,  0.203160f,  -0.291788f, 0.000680f,  0.587011f,  -0.241289f,
  0.037034f,   0.000552f,  1.072308f,  -0.387230f, -0.230050f, 0.292322f,
  -0.720001f,  0.034109f,  -0.467260f, 2.211644f,  -1.839191f, -0.048797f,
  -0.083469f,  -0.334686f, -0.269056f, 0.051295f,  1.319904f,  -0.035603f,
  -0.018457f,  -0.824915f, -0.212285f, -0.230516f, -0.035093f, -0.400843f,
  -0.305469f,  -0.099011f, 0.014225f,  -0.452772f, 0.170331f,  -0.389312f,
  -0.115084f,  -0.014770f, -0.429387f, -0.155961f, -0.568200f, -0.037853f,
  -0.125137f,  0.067228f,  -1.329271f, -0.117874f, -0.132499f, -0.218376f,
  -0.588325f,  -0.320024f, 0.085695f,  -0.235047f, -0.217790f, 0.103015f,
  -0.698644f,  0.017766f,  -0.058299f, 0.199411f,  -0.122485f, -0.563949f,
  -0.349011f,  -0.557045f, -0.131165f, 0.002281f,  0.118559f,  -0.210302f,
  -1.153815f,  0.116738f,  -0.236007f, -0.003487f, -0.006885f, -0.244816f,
  0.953222f,   0.093748f,  0.266869f,  0.241869f,  -0.860832f, -0.387012f,
  -0.338986f,  2.097515f,  -1.942512f, -0.298021f, 0.543911f,  -0.043214f,
  0.082125f,   -0.120242f, 0.712231f,  0.213327f,  -0.301687f, -0.544011f,
  -0.392131f,  0.004302f,  0.004825f,  -0.317440f, -0.107518f, -0.293407f,
  -0.159111f,  -0.080367f, 0.132663f,  -0.017726f, -0.237521f, -0.190297f,
  -0.361633f,  0.200518f,  -0.538296f, -0.027975f, -0.381704f, -0.016963f,
  0.630105f,   -0.190997f, -0.287840f, -0.603488f, 3.605598f,  -0.276614f,
  -1.346383f,  0.186912f,  -0.047575f, -0.189232f, -1.519072f, 0.097816f,
  -0.223722f,  0.304924f,  -0.213022f, -1.052433f, -0.322283f, -1.706734f,
  -2.458027f,  0.237976f,  0.171050f,  -0.103139f, -0.278689f, 0.329824f,
  -0.262448f,  -0.122916f, -0.236398f, -0.013848f, -0.969160f, -0.374907f,
  0.091018f,   -0.386471f, -0.723940f, 0.064956f,  -0.057652f, 1.321024f,
  -1.397418f,  -0.143136f, 0.272468f,  -0.030749f, 0.037324f,  0.069316f,
  -0.904925f,  -0.333693f, -0.117709f, 2.279598f,  -0.428065f, -0.131157f,
  -0.014288f,  -0.402862f, -0.666090f, 0.017070f,  -0.028333f, 0.002481f,
  0.197156f,   -0.038120f, -0.271062f, -0.188275f, -0.021370f, -0.070849f,
  -0.905007f,  -0.095886f, -0.093055f, -0.121821f, -1.239812f, -0.411799f,
  -0.089948f,  -0.936827f, 1.437569f,  -0.388908f, 0.126170f,  0.186162f,
  -0.018819f,  -0.138364f, -1.066412f, -0.138222f, -0.022186f, 0.107331f,
  -0.230436f,  -1.352605f, -0.161323f, -1.081810f, -0.933825f, -0.136675f,
  0.378157f,   0.113377f,  -0.850610f, 0.080245f,  -0.087305f, -0.002852f,
  0.044408f,   -0.188172f, -1.891998f, 0.092189f,  0.125325f,  -0.105090f,
  -0.848510f,  -0.396308f, -0.384130f, 2.007509f,  -1.480787f, -0.126946f,
  0.314767f,   0.000195f,  -0.285628f, -0.110442f, -0.293948f, 0.258559f,
  -0.417603f,  1.570705f,  0.092459f,  -0.340974f, -0.284754f, -0.007801f,
  -0.324610f,  -0.004734f, -0.207716f, -0.057175f, 0.055467f,  -0.210830f,
  -0.113005f,  -0.299177f, 0.068074f,  0.017929f,  -2.897598f, -0.260074f,
  -0.014422f,  -0.206467f, 1.246997f,  -0.372863f, -0.214160f, -0.114035f,
  5.805862f,   0.003611f,  -1.340990f, -0.021085f, -0.260431f, -0.002720f,
  -1.251640f,  -0.353531f, -0.304009f, -0.153376f,
};

static const float av1_ab_partition_nn_bias_32_layer1[LABEL_SIZE] = {
  -0.521497f, -1.061572f, -0.078756f, -0.660662f, -0.403741f, -0.960163f,
  0.001427f,  0.523607f,  0.225068f,  -0.055273f, 1.019519f,  1.181880f,
  -0.010198f, 0.130597f,  1.276752f,  2.028188f,
};

static const NN_CONFIG av1_ab_partition_nnconfig_32 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      64,  // num_hidden_nodes
  },
  {
      av1_ab_partition_nn_weights_32_layer0,
      av1_ab_partition_nn_weights_32_layer1,
  },
  {
      av1_ab_partition_nn_bias_32_layer0,
      av1_ab_partition_nn_bias_32_layer1,
  },
};

// nn model for ab partition pruning, 16x16.
static const float av1_ab_partition_nn_weights_16_layer0[FEATURE_SIZE * 64] = {
  0.151902f,  0.007947f,  -1.788454f, 0.431869f,  -2.971387f, 0.923566f,
  1.632542f,  -1.665136f, -0.338632f, -5.075884f, 0.398267f,  0.030467f,
  2.263534f,  -0.045532f, -1.066128f, 0.915139f,  -0.560500f, -3.293125f,
  2.072793f,  -1.011414f, 0.122716f,  -0.060169f, -0.388860f, 0.031019f,
  -0.381861f, 0.001551f,  -0.328472f, 0.038296f,  -0.060398f, -0.375556f,
  0.209226f,  0.014764f,  -1.443469f, -0.345486f, 2.409269f,  1.524846f,
  -0.640666f, 1.322139f,  -2.074771f, -0.580944f, -0.203960f, -0.072893f,
  0.329701f,  0.115339f,  -1.339542f, 0.249024f,  -0.421545f, -0.409151f,
  -0.258293f, 0.836288f,  -0.073685f, -0.009624f, 0.895712f,  0.320639f,
  0.451002f,  -1.544558f, 0.193709f,  -1.389012f, 1.305451f,  0.089795f,
  0.050338f,  -0.017433f, -0.304667f, 0.500729f,  0.504346f,  0.073757f,
  0.582649f,  -0.993623f, 1.766766f,  -3.067265f, -0.415774f, -0.006036f,
  -1.245281f, 0.253205f,  -0.591245f, -0.626238f, 0.551852f,  0.593755f,
  0.491023f,  1.099384f,  -0.348448f, 0.054564f,  -0.451422f, -0.375781f,
  -0.248390f, -0.052548f, -0.380069f, -0.165391f, -0.297968f, -0.052142f,
  -0.316381f, -0.045246f, -0.243905f, -0.034169f, -0.247523f, -0.180773f,
  0.068066f,  -0.374920f, 0.057536f,  -0.189748f, 0.058375f,  -0.267749f,
  -0.147286f, -0.246153f, 0.006183f,  -0.202029f, -0.059128f, 0.116852f,
  0.134719f,  -0.126900f, -0.064646f, -0.196458f, -0.182331f, 0.108029f,
  -0.264499f, 0.155816f,  -0.107255f, -0.056983f, -0.209771f, -0.099070f,
  0.007313f,  -0.254124f, -0.231964f, -0.275972f, 0.032098f,  -0.264564f,
  -0.208743f, 0.155599f,  -0.121511f, -0.156145f, -0.162315f, -0.059788f,
  -0.257073f, -0.076654f, -0.110616f, -0.321675f, -0.051952f, 0.006301f,
  -0.154114f, 0.017032f,  -0.017364f, -0.233247f, 0.009918f,  -0.179289f,
  -0.190722f, 0.147106f,  -0.063910f, -0.396872f, -0.263123f, -0.003850f,
  -0.040718f, -0.324699f, 0.118660f,  -0.170727f, -0.316788f, 0.100886f,
  -0.202842f, 0.045371f,  0.150561f,  -0.057054f, -0.308150f, 0.028346f,
  -0.381473f, -0.195365f, 0.026221f,  -0.281795f, 0.087204f,  0.047689f,
  -0.027643f, -0.104724f, -0.089030f, -0.117661f, -0.349160f, 0.056982f,
  -0.340273f, 0.048086f,  0.046103f,  -0.121527f, 0.021697f,  0.054109f,
  -0.002768f, -0.008461f, -2.297240f, 0.124651f,  3.621661f,  -0.057120f,
  -1.151656f, 2.296894f,  -3.678720f, -0.290240f, 0.087683f,  -0.186389f,
  0.007656f,  -0.090236f, -0.245217f, 0.110389f,  -0.251719f, -0.029084f,
  -0.128203f, -0.100005f, -0.032779f, 0.007281f,  -0.366596f, -0.267870f,
  -0.215620f, 0.047687f,  0.010303f,  0.097980f,  -0.191569f, -0.341162f,
  0.119249f,  0.026279f,  -2.161546f, 0.459591f,  1.290566f,  1.791797f,
  -0.409835f, 0.127081f,  -1.156367f, 0.198286f,  0.099561f,  -0.067445f,
  -0.034352f, 0.017966f,  -0.277380f, -0.057220f, -0.174198f, -0.014164f,
  0.146090f,  -0.357530f, 0.097644f,  -0.000932f, 0.446603f,  -0.066793f,
  2.448620f,  0.937617f,  -1.232922f, 0.313183f,  0.816827f,  -0.275115f,
  -0.245205f, -0.126895f, 0.156668f,  -0.186977f, -0.273505f, 0.013315f,
  0.168629f,  -0.089084f, 0.006166f,  -0.116107f, -0.199316f, -0.024010f,
  -0.242303f, 0.011612f,  -0.218485f, -0.229661f, -0.123922f, 0.136699f,
  0.006732f,  -0.148718f, -0.164225f, 0.116063f,  1.587898f,  0.690519f,
  0.360566f,  0.009739f,  -0.678702f, -0.046003f, 0.126984f,  0.605212f,
  1.240663f,  -0.000228f, -1.119369f, -0.415589f, -0.721003f, 0.097936f,
  -1.410586f, -2.358833f, -2.773129f, -3.983361f, -0.087144f, -0.050029f,
  -0.242255f, 0.137424f,  -0.307490f, -0.084637f, -0.023812f, -0.196582f,
  -0.078695f, 0.038257f,  -0.012110f, -0.263521f, 0.009839f,  -0.109125f,
  -0.226036f, 0.060712f,  0.093671f,  0.153143f,  0.039116f,  -0.290891f,
  0.227057f,  -0.204633f, -0.207539f, -0.148242f, 0.046204f,  -0.231268f,
  -0.209315f, -0.307579f, -0.436556f, 0.023475f,  0.131793f,  -0.038301f,
  1.650584f,  0.392570f,  1.446576f,  1.254380f,  -0.516867f, -0.057116f,
  0.149320f,  0.414424f,  -0.246309f, 0.003877f,  -0.480238f, -1.037035f,
  -0.830779f, -1.122244f, -0.408267f, -0.253956f, 0.382005f,  0.940609f,
  -1.113370f, -0.018554f, 0.141064f,  -0.182504f, 1.270707f,  0.414904f,
  -0.216036f, 0.203831f,  0.450716f,  -0.452909f, 0.139358f,  -0.027143f,
  1.956892f,  1.643732f,  -0.867839f, -0.620520f, -0.334607f, -0.519982f,
  0.205023f,  0.661159f,  -0.000809f, 0.049033f,  -0.348579f, -0.200338f,
  -0.362144f, -0.346590f, -0.230096f, 0.180746f,  -0.149954f, -0.253429f,
  -0.378170f, -0.040724f, -0.041597f, 0.243659f,  -0.472181f, 0.015401f,
  -0.180376f, 0.153139f,  -0.247738f, -0.010485f, -0.157158f, 0.016825f,
  -0.238925f, -0.265798f, -0.318374f, 0.142352f,  -0.210520f, 0.051928f,
  -0.352190f, -0.179052f, -0.185498f, 0.025540f,  -0.111667f, -0.235187f,
  -0.215454f, 0.010931f,  -0.238372f, -0.126659f, 0.075691f,  -0.091167f,
  -2.462379f, -0.007950f, -0.637990f, 0.285554f,  -0.051275f, 0.282279f,
  -0.744083f, -0.570646f, 0.592198f,  1.421332f,  -0.256027f, -0.140315f,
  0.160247f,  -0.063185f, -0.055895f, -0.199864f, -0.287353f, -0.074561f,
  -0.071228f, 0.055864f,  -1.084764f, -0.263409f, 0.779266f,  0.228187f,
  0.375013f,  0.121204f,  -0.656948f, 0.533561f,  0.272671f,  -0.015423f,
  -0.124180f, -0.009127f, 2.934838f,  -0.150998f, 1.163152f,  0.081997f,
  -4.715939f, -3.676595f, -1.524886f, -0.167593f, 0.281186f,  0.024046f,
  -1.451709f, 0.332558f,  0.990504f,  0.376290f,  -1.466773f, -0.448439f,
  -2.929108f, -4.255188f, 0.065238f,  0.019950f,  1.372393f,  0.444052f,
  -2.538772f, 1.579767f,  -0.464911f, -1.866114f, 1.053958f,  0.434467f,
  -0.125964f, 0.034671f,  0.077116f,  -0.138466f, -0.413395f, -0.223453f,
  -0.172127f, -0.251265f, -0.048239f, -0.395519f, 0.023141f,  0.037459f,
  -0.249593f, -0.062215f, -0.047209f, -0.435189f, -0.164155f, -0.077590f,
  -0.241164f, -0.126128f, -0.038243f, -0.180888f, 0.198840f,  -0.328036f,
  -0.169790f, 0.036506f,  0.052572f,  -0.183570f, -0.073617f, -0.244959f,
  0.266498f,  0.032846f,  -1.902106f, 0.486078f,  2.414993f,  0.975182f,
  -0.382875f, 1.647810f,  -2.197017f, -0.890107f, 0.221287f,  0.010889f,
  3.817042f,  0.572728f,  0.092466f,  0.473337f,  -1.634659f, -1.069455f,
  1.486776f,  -1.023850f, 0.088184f,  0.008842f,  0.518202f,  0.270259f,
  1.757191f,  -0.121839f, -2.912229f, -1.250866f, -2.381808f, 0.335309f,
  -0.120079f, -0.061294f, -0.058725f, -0.315169f, -0.262443f, 0.072434f,
  -0.267836f, -0.319354f, -0.274975f, 0.068970f,  -0.406467f, 0.044074f,
  -0.152311f, -0.333656f, -0.228355f, -0.185613f, 0.017346f,  -0.177674f,
  -0.090675f, -0.102047f, -0.011768f, -0.025280f, -0.271661f, 0.098099f,
  -0.312272f, -0.222217f, -0.100548f, 0.106260f,  -0.034655f, 0.135109f,
  -0.021276f, 0.018177f,  -0.353097f, -0.011128f, 0.061136f,  -0.511662f,
  -0.223236f, -0.308841f, 0.118789f,  -0.154628f, -0.053178f, -0.055973f,
  0.013175f,  -0.368337f, -0.090863f, -0.116920f, 0.178990f,  -0.025278f,
  -0.190553f, -0.238092f, 0.303943f,  -0.024944f, 0.719373f,  0.384332f,
  -0.378480f, -0.423316f, 0.709922f,  0.758514f,  -1.559023f, -2.503173f,
  0.068652f,  -0.234741f, -0.182932f, 0.037878f,  0.020684f,  -0.174142f,
  -0.182300f, -0.052796f, -0.219145f, 0.113028f,  -1.041826f, 0.035317f,
  0.919904f,  -0.676011f, 0.652297f,  1.456447f,  -0.166904f, -0.861823f,
  0.895827f,  0.429821f,  -0.180376f, -0.076587f, -0.273945f, -0.288990f,
  -0.206692f, -0.080745f, -0.085444f, 0.186953f,  -0.050135f, 0.044243f,
  -0.391706f, -0.160498f, -0.292268f, 0.164060f,  0.412649f,  0.211611f,
  -0.327294f, -0.919399f, 0.320297f,  0.385284f,  -0.088848f, -0.072556f,
  -0.384813f, -0.176267f, -0.065918f, 0.134724f,  -0.231104f, -0.337707f,
  -0.195442f, -0.263569f, 0.098090f,  -0.341411f, -0.189211f, -0.439276f,
  -0.404046f, 0.262491f,  -0.311093f, -0.086454f, -0.013400f, -0.061447f,
  -0.026945f, -0.112036f, -0.322985f, 0.078500f,  -0.230205f, -0.344535f,
  -0.021087f, 0.110220f,  -0.128671f, 0.044219f,
};

static const float av1_ab_partition_nn_bias_16_layer0[64] = {
  2.936406f,  -0.396539f, -0.110456f, -1.254954f, 0.785350f,  0.516290f,
  -0.172341f, 0.254386f,  -0.192465f, -0.106751f, -0.055518f, -0.094994f,
  0.000000f,  -0.065018f, -0.004908f, -0.130483f, -0.119580f, -0.142072f,
  0.457446f,  -0.125051f, -0.107712f, 0.714607f,  -0.140809f, -1.788650f,
  -0.087199f, 0.000000f,  -1.290050f, 0.443930f,  -0.110634f, -0.109380f,
  -0.188213f, -1.414179f, 1.193579f,  0.388775f,  -0.873193f, -0.110050f,
  -0.072565f, -0.117050f, -0.119132f, 0.456959f,  -0.132069f, 0.131974f,
  1.160474f,  1.746465f,  0.442628f,  -0.188849f, -0.207794f, -0.108364f,
  -0.856655f, -2.141620f, 0.335476f,  -0.105508f, -0.212162f, -0.109319f,
  -0.237213f, -0.109980f, -0.291044f, -0.137877f, 0.470191f,  -0.023908f,
  0.123809f,  -0.109797f, 0.200510f,  -0.147542f,
};

static const float av1_ab_partition_nn_weights_16_layer1[64 * LABEL_SIZE] = {
  -6.823716f, 1.406568f,  -0.144009f, 2.228765f,  0.838336f,  0.738107f,
  -0.319014f, -0.148756f, 0.240862f,  -0.111089f, -0.004241f, 0.025758f,
  -0.193820f, -0.246362f, -0.181363f, -0.201556f, 0.024268f,  0.252994f,
  -0.289443f, 0.194932f,  0.057467f,  0.724735f,  0.014063f,  1.361352f,
  0.025191f,  0.024274f,  0.231462f,  -7.227959f, -0.094515f, 0.039946f,
  0.412719f,  0.812318f,  3.038903f,  -0.286289f, 0.647482f,  -0.115114f,
  0.053590f,  0.066069f,  0.153134f,  0.996250f,  -0.125700f, 0.951365f,
  -6.243494f, -4.827697f, 0.566320f,  0.239515f,  -0.099702f, 0.054546f,
  1.847330f,  3.680076f,  -3.049829f, -0.127709f, 0.068469f,  -0.017794f,
  0.223864f,  -0.106778f, -0.020425f, -0.040226f, -0.251890f, -0.168673f,
  -0.552073f, 0.043311f,  0.218668f,  0.033209f,  -3.199210f, 0.193079f,
  0.321406f,  0.718307f,  -0.181418f, -0.459612f, -1.981170f, 0.968496f,
  -0.029757f, -0.130065f, 0.043782f,  0.072394f,  -0.088686f, 0.025322f,
  0.129882f,  0.101324f,  0.335707f,  0.072714f,  -2.079774f, 0.203997f,
  0.239321f,  -0.301757f, 0.257845f,  1.288382f,  -0.031275f, -0.234194f,
  0.310722f,  2.045469f,  0.034716f,  0.135638f,  -0.251388f, 0.320071f,
  -1.065301f, -0.322731f, -0.545028f, 0.226276f,  0.090799f,  0.019289f,
  0.048950f,  -1.079300f, 0.231938f,  0.083683f,  4.762127f,  0.145037f,
  -0.145549f, 0.075592f,  0.172336f,  0.108175f,  0.333751f,  1.090501f,
  1.056114f,  0.047073f,  0.182052f,  -0.081587f, 0.089900f,  0.339286f,
  2.049988f,  0.073585f,  0.537355f,  -0.243322f, -0.010179f, -0.052601f,
  -0.174915f, 0.117793f,  2.222990f,  -2.520837f, -0.092699f, 1.199887f,
  0.138720f,  0.679918f,  -0.463155f, -0.659496f, -0.109913f, -0.003398f,
  0.114633f,  -0.128377f, 0.092970f,  -0.107489f, -0.191078f, 0.185182f,
  0.216980f,  -0.019343f, 3.443133f,  0.287953f,  0.099314f,  0.985958f,
  0.157268f,  -0.606516f, 0.049418f,  -0.221809f, -0.453081f, -0.344796f,
  -0.003735f, -0.107269f, -0.128541f, -0.259543f, -0.934806f, -0.542456f,
  -1.011192f, 0.022795f,  0.186363f,  -0.076356f, -0.050932f, -0.165098f,
  0.168177f,  -0.101596f, -5.270886f, 2.553943f,  -0.440870f, -0.017494f,
  0.215208f,  -0.017032f, 1.495915f,  -4.304677f, 0.762211f,  0.182937f,
  0.254406f,  -0.029433f, -0.088364f, -0.110160f, -0.108257f, -0.036538f,
  0.737697f,  -0.234989f, 0.168095f,  0.245118f,  -0.077262f, 0.195718f,
  0.753302f,  -1.637869f, 0.126227f,  0.982129f,  -0.121444f, -0.295570f,
  -1.215799f, 0.147867f,  -0.068496f, 0.132726f,  -0.005772f, -0.181774f,
  0.126513f,  0.204723f,  -0.366123f, 0.103906f,  -0.148053f, -0.075272f,
  0.243884f,  -0.104828f, 0.198988f,  0.501034f,  -0.112671f, 0.111421f,
  0.167508f,  -0.117803f, -0.738624f, 2.046292f,  0.124011f,  0.057983f,
  -0.359154f, -0.648883f, -0.259462f, -0.459041f, -2.501223f, -0.065138f,
  0.122417f,  0.060291f,  -0.129033f, -0.843086f, 0.268241f,  -0.399927f,
  1.585888f,  1.816393f,  -0.631427f, 0.127826f,  0.088105f,  0.073488f,
  0.717694f,  -1.497362f, 2.608528f,  0.066896f,  -0.079230f, 0.223436f,
  -0.010530f, 0.175310f,  1.120365f,  0.034391f,  0.835312f,  0.071652f,
  -0.080615f, 0.111395f,  0.162742f,  0.079927f,  -3.859582f, -0.638431f,
  -0.167880f, -0.992659f, -0.885355f, -1.276197f, 1.334344f,  0.931940f,
  -0.078244f, -0.149030f, -0.070974f, -0.133566f, 0.200034f,  0.102793f,
  -0.048546f, 0.063545f,  0.023864f,  -0.190863f, 1.934257f,  -0.136286f,
  -0.107916f, -0.637468f, 0.066449f,  1.089693f,  -0.214047f, -0.265780f,
  0.899660f,  -0.130333f, 0.288311f,  -0.049024f, 0.090202f,  0.487969f,
  0.339704f,  0.858479f,  0.841253f,  -0.184100f, -0.637070f, -0.125071f,
  -0.077650f, -0.087877f, 0.202268f,  -0.027300f, 2.842862f,  -0.100698f,
  -0.259080f, 0.260556f,  0.157912f,  -0.070364f, 0.467190f,  1.200037f,
  1.419317f,  -0.033588f, -0.227824f, 0.292617f,  0.228574f,  0.213839f,
  -1.091099f, -0.022258f, -1.294681f, 0.136118f,  0.081652f,  -0.185359f,
  -0.039706f, 0.191407f,  -2.053219f, -0.261934f, 0.047812f,  -0.029536f,
  -0.823869f, -1.090534f, -0.755890f, 0.441035f,  -0.167945f, 0.231441f,
  -0.135013f, -0.260762f, 0.256872f,  0.130339f,  -0.243751f, 0.189760f,
  -0.288454f, 0.145363f,  0.338490f,  0.403898f,  -0.022814f, -1.263598f,
  -0.101315f, 0.860135f,  0.136511f,  0.028942f,  0.574047f,  2.656370f,
  0.037587f,  -0.188690f, -0.125312f, 1.100435f,  -1.080402f, 0.380905f,
  0.004635f,  0.097144f,  -0.214309f, 0.085552f,  -0.285066f, -0.705134f,
  -0.054704f, -0.319951f, 5.486626f,  0.958158f,  -1.380585f, 0.223340f,
  -0.169167f, -0.170697f, -0.216748f, 0.324232f,  2.684204f,  -0.008490f,
  -0.211052f, -0.201190f, 0.123466f,  -0.000234f, 0.579907f,  0.096938f,
  -0.042745f, 0.201855f,  0.157195f,  -0.261440f, 0.029699f,  -0.046599f,
  1.618216f,  -2.596280f, -0.377420f, -0.526725f, -0.493592f, -0.579615f,
  0.579699f,  -0.100392f, 0.150694f,  0.061794f,  0.200425f,  -0.062515f,
  -0.179122f, 0.250112f,  -0.344675f, -0.118359f, -0.095670f, 0.152311f,
  3.662276f,  -0.154921f, -0.312991f, 0.972008f,  -0.308596f, -0.190426f,
  0.133889f,  -0.238673f, -0.094726f, 1.683835f,  -0.215629f, -0.198890f,
  -0.035278f, -0.367973f, -0.822435f, 0.240848f,  -0.194656f, 0.034655f,
  -0.079424f, 0.146670f,  0.026646f,  -0.034507f, 0.059467f,  -0.153109f,
  -0.431033f, 2.552991f,  -1.894091f, -0.180462f, -0.306839f, -0.025648f,
  1.026326f,  -3.096230f, 1.346935f,  0.033633f,  -0.181827f, 0.094376f,
  0.001696f,  -0.379264f, -1.069503f, -0.140972f, -0.208769f, -0.195239f,
  0.281795f,  -0.127251f, 0.180776f,  0.067763f,  0.697124f,  -1.040779f,
  0.111280f,  0.188351f,  -0.340234f, -0.207790f, -0.720075f, -0.137409f,
  -0.070310f, -0.032918f, -0.060787f, 0.131484f,  -0.077845f, -0.258652f,
  0.056911f,  -0.062034f, 0.007663f,  -0.185100f, 1.340361f,  0.014096f,
  -0.124602f, 0.194241f,  0.128383f,  0.360465f,  0.082979f,  -0.050475f,
  -0.519294f, 3.323262f,  0.067014f,  0.221203f,  -0.085082f, -0.228606f,
  -0.916668f, -0.022643f, -1.386737f, -0.131902f, -0.349952f, -0.032874f,
  -0.189190f, -0.898790f, -0.102394f, -1.017387f, 2.214050f,  1.790253f,
  -1.913561f, -0.043716f, -0.214924f, -0.194598f, -0.064723f, -1.671793f,
  2.251166f,  -0.146007f, 0.138527f,  -0.003134f, 0.103665f,  0.006928f,
  -0.240253f, -0.227464f, 0.578437f,  -0.214724f, 0.503085f,  0.158093f,
  0.033091f,  0.008061f,  4.815371f,  2.132264f,  0.281850f,  -2.288560f,
  -0.145012f, 1.296832f,  -0.362401f, -0.403252f, 0.109873f,  0.185746f,
  0.244764f,  0.172367f,  -0.185588f, 0.139801f,  -0.178254f, 0.068629f,
  0.358488f,  -0.153969f, -6.433524f, 0.225983f,  -0.138123f, -0.095971f,
  -0.036089f, -1.400083f, 0.265908f,  0.257787f,  0.181144f,  -1.647228f,
  -0.136289f, -0.074206f, 0.122988f,  -0.088895f, -1.266717f, 0.006010f,
  0.536681f,  0.263061f,  -0.032207f, -0.155136f, 0.086431f,  0.441950f,
  -0.060755f, -0.280683f, -0.783475f, -2.567033f, 1.093221f,  0.117667f,
  -0.000408f, 0.225719f,  -2.199698f, 0.141447f,  -1.459051f, 0.051315f,
  0.203228f,  0.354432f,  -0.005775f, -0.028073f, -0.965817f, 0.231083f,
  -0.666884f, 0.026283f,  -0.317486f, 0.210754f,  0.123897f,  0.223827f,
  4.214405f,  1.457334f,  -0.253945f, -1.306733f, -0.391235f, 0.451154f,
  -1.553888f, -0.353429f, 0.069533f,  0.159278f,  -0.173836f, -0.004952f,
  -0.137033f, 0.127012f,  0.143600f,  0.051587f,  -0.070549f, 0.066509f,
  -5.776547f, 0.180021f,  -0.189183f, -1.288504f, -0.233575f, -1.473873f,
  0.140940f,  0.144451f,  -0.104534f, 2.089873f,  -0.168168f, 0.110726f,
  0.132134f,  -0.215223f, -1.682754f, 0.157757f,  -0.146163f, 0.064882f,
  0.117313f,  -0.038780f, -0.124720f, -0.501697f, 0.092047f,  -0.233992f,
  3.324976f,  0.516601f,  1.294202f,  0.119989f,  0.061055f,  0.043420f,
  -2.750727f, -0.382812f, -0.648496f, -0.115353f, -0.334205f, 0.024354f,
  -0.282998f, -0.282705f, 0.073798f,  0.169851f,  0.135651f,  0.182677f,
  -0.040220f, 0.132462f,  -0.303120f, -0.230113f, 6.165739f,  -0.258596f,
  0.024127f,  -1.388283f, -0.006042f, 0.572600f,  0.348411f,  -0.387376f,
  -0.075845f, 0.122319f,  -0.029616f, 0.077873f,  0.154763f,  0.049073f,
  0.018597f,  0.102688f,  -0.204165f, 0.020734f,  -1.389133f, -0.032854f,
  -0.147561f, 0.853944f,  0.132100f,  -3.259659f, 0.243745f,  0.181529f,
  -0.738414f, 1.509994f,  0.023470f,  -0.005329f, 0.066115f,  -1.345081f,
  -1.455402f, -0.172023f, -0.194625f, 0.071885f,  -0.201742f, -0.262402f,
  0.077601f,  -0.048938f, 0.257993f,  -0.504029f, -2.032415f, 1.158880f,
  0.448647f,  -0.025633f, 0.117586f,  -0.072275f, -0.673744f, -3.854342f,
  -0.983843f, 0.047766f,  -0.017193f, -0.215775f, -0.158743f, -0.232042f,
  -0.509112f, 0.148812f,  0.130122f,  0.006486f,  -0.099016f, 0.022514f,
  -0.486850f, -0.059623f, 4.012731f,  0.025454f,  0.029059f,  -0.783546f,
  -0.295260f, 0.322521f,  -0.473201f, -0.172100f, -0.100087f, -0.076516f,
  -0.258367f, -0.112897f, 0.269364f,  -0.065912f, 0.169022f,  -0.178783f,
  -0.095114f, 0.122089f,  -2.790099f, -0.100431f, -0.087963f, -0.009431f,
  -0.087819f, -2.774399f, -0.100757f, 0.013005f,  -0.964533f, 3.236665f,
  -0.354903f, -0.144169f, -0.166869f, -1.396513f, -0.931271f, -0.046261f,
  -1.799262f, -0.365269f, 0.108611f,  0.037994f,  0.024747f,  -1.073639f,
  -0.203158f, -0.935006f, 1.880891f,  1.578385f,  0.726272f,  -0.024546f,
  -0.011626f, -0.151363f, -1.121716f, -1.787484f, 0.232806f,  0.075451f,
  0.182899f,  0.092215f,  -0.207347f, -0.030111f, 0.054316f,  0.192481f,
  0.594639f,  -0.247694f, 0.547471f,  -0.032094f, -0.065000f, 0.007198f,
  1.605377f,  -0.155945f, -0.066200f, -2.343716f, -1.016283f, -0.079321f,
  0.919365f,  0.599980f,  0.125545f,  0.265813f,  0.246884f,  0.095385f,
  -0.260374f, -0.202916f, -0.042770f, 0.234967f,  -0.233139f, -0.326994f,
  -1.375256f, 0.121766f,  0.077433f,  -1.103569f, 0.019497f,  -1.029185f,
  0.253905f,  0.206569f,  0.187334f,  -0.237089f, -0.294351f, 0.164137f,
  0.149696f,  -0.749787f, -0.413433f, 0.976587f,  1.027976f,  -0.285264f,
  0.209273f,  -0.124762f, 0.050884f,  0.250764f,  -0.082031f, -0.646520f,
  4.116680f,  0.437336f,  0.671684f,  0.129509f,  -0.078462f, 0.014072f,
  -0.678232f, 0.094831f,  1.125624f,  0.207070f,  -0.154750f, -0.025780f,
  -0.103030f, 0.118019f,  -0.908186f, -0.263546f, -1.555324f, -0.236887f,
  -0.217854f, -0.051790f, 0.017915f,  0.171001f,  1.355562f,  0.094603f,
  -0.233929f, -1.282169f, -0.773183f, -0.161682f, -0.834565f, -0.286776f,
  -0.298901f, 0.038162f,  0.251899f,  0.039612f,  -0.022935f, -0.232308f,
  -0.043855f, -0.192892f, -0.279009f, -0.182234f, -1.272808f, -0.070344f,
  -0.092432f, -1.915946f, -0.134373f, -1.405496f, -0.067071f, -0.131922f,
  0.185269f,  1.465082f,  0.040240f,  0.112665f,  0.144329f,  -0.286112f,
  -0.617649f, 0.916177f,  0.221044f,  -0.079867f, 0.170251f,  -0.093638f,
  -0.212620f, -0.305945f, -0.234356f, -0.482501f, 3.928472f,  1.241179f,
  0.355922f,  -0.170848f, -0.189168f, 0.080225f,  -1.357793f, 0.190890f,
  0.976800f,  -0.068070f, -0.016295f, -0.088623f, -0.129560f, -0.212267f,
  -0.071537f, -0.219501f, -0.655198f, -0.225188f, -0.116024f, 0.224174f,
  -0.049715f, -0.178005f, 3.029985f,  -1.141546f, 0.080066f,  -1.932316f,
  -0.641137f, -0.189564f, 0.935080f,  0.136119f,  0.015558f,  -0.179331f,
  0.204571f,  0.020350f,  0.009362f,  0.108478f,  0.037076f,  -0.049009f,
  0.081090f,  -0.180202f, 1.455561f,  -0.081559f, 0.059361f,  0.484971f,
  0.160923f,  -2.170744f, -0.013204f, 0.126561f,  -0.407122f, 1.223661f,
  0.044262f,  0.118044f,  0.058274f,  -1.747100f, -0.171318f, 0.971374f,
  0.306995f,  -0.103268f, -0.319443f, -0.333176f, -0.038608f, 0.119674f,
  -0.106479f, -0.907933f, 1.121231f,  1.673840f,  -0.421458f, -0.021146f,
  -0.254838f, 0.097632f,  0.235109f,  -2.901782f, 0.289518f,  -0.355459f,
  -0.068264f, -0.179121f, 0.068560f,  -0.047570f, -0.522523f, -0.228963f,
  -1.037158f, -0.163723f, 0.280563f,  -0.000868f, -0.197220f, -0.239329f,
  1.985274f,  -0.256181f, -0.064341f, -0.822417f, -0.465140f, -0.010942f,
  -0.792024f, -0.114290f, 0.060969f,  0.104106f,  -0.252123f, -0.150400f,
  -0.133277f, 0.267147f,  0.274413f,  0.223744f,  -0.180223f, -0.345415f,
  -0.104883f, 0.119210f,  -0.095041f, -0.301635f, 0.013175f,  -2.128121f,
  -0.147208f, -0.151509f, -0.692013f, 3.418555f,  -0.016541f, 0.171511f,
  0.107159f,  -1.516672f, 0.127408f,  0.687035f,  -0.906486f, -0.145463f,
  -0.169382f, -0.143906f, 0.125091f,  -0.960645f, -0.180869f, -0.716908f,
  2.840951f,  1.904919f,  -0.416268f, -0.425181f, -0.194697f, -0.075932f,
  -0.950604f, -1.599800f, 0.943671f,  -0.022744f, -0.270492f, 0.080843f,
  -0.372916f, 0.047838f,  -0.100300f, -0.026600f, 0.011733f,  -0.226051f,
  0.172790f,  -0.172982f, 0.041258f,  -0.299379f,
};

static const float av1_ab_partition_nn_bias_16_layer1[LABEL_SIZE] = {
  -0.053805f, -1.248639f, 0.520965f, -0.904962f, -0.126425f, -0.118798f,
  0.748430f,  0.203096f,  0.059317f, 0.418219f,  0.841294f,  0.402693f,
  -0.658522f, 0.723479f,  0.544264f, 1.035225f,
};

static const NN_CONFIG av1_ab_partition_nnconfig_16 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      64,  // num_hidden_nodes
  },
  {
      av1_ab_partition_nn_weights_16_layer0,
      av1_ab_partition_nn_weights_16_layer1,
  },
  {
      av1_ab_partition_nn_bias_16_layer0,
      av1_ab_partition_nn_bias_16_layer1,
  },
};

#undef FEATURE_SIZE
#undef LABEL_SIZE

#define FEATURE_SIZE 18
#define LABEL_SIZE 4

static const float av1_4_partition_nn_weights_16_layer0[FEATURE_SIZE * 24] = {
  -2.032866f, 0.056691f,  0.495960f,  0.778785f,  0.548153f,  -0.806942f,
  0.481155f,  0.282298f,  0.584980f,  0.504688f,  0.209648f,  0.234616f,
  0.213484f,  0.221969f,  0.205862f,  0.235054f,  0.317863f,  0.257139f,
  0.529478f,  0.098122f,  -0.657532f, 0.036296f,  0.327728f,  1.323180f,
  -0.813082f, 0.160216f,  -0.702030f, 0.722733f,  -0.270576f, -0.347416f,
  -0.264700f, -0.254248f, 0.159820f,  0.087995f,  -0.184163f, 0.117357f,
  0.074194f,  -0.667369f, 0.498246f,  0.420506f,  0.072409f,  -0.121581f,
  0.315788f,  0.000525f,  0.414986f,  0.678166f,  -0.011230f, 0.188131f,
  -0.227749f, 0.009564f,  0.108672f,  0.106923f,  -0.080695f, -0.279382f,
  -0.061339f, -0.297835f, -0.134707f, 0.145865f,  -0.009655f, -0.000842f,
  -0.047436f, -0.159149f, -0.320353f, -0.089646f, -0.344765f, 0.313416f,
  -0.143413f, 0.279668f,  0.000885f,  -0.022380f, -0.140194f, -0.310473f,
  0.252699f,  0.066204f,  0.477568f,  0.994609f,  -0.276000f, 1.213182f,
  0.277028f,  -0.411570f, -0.211559f, 0.377815f,  0.121488f,  -0.100559f,
  -0.317082f, -0.251039f, -0.335181f, -0.154114f, -0.052726f, -0.332558f,
  -0.143196f, -0.334035f, 0.162305f,  0.142279f,  -0.001210f, -0.135252f,
  -0.033562f, 0.204307f,  -0.039757f, -0.394174f, 0.126617f,  -0.128648f,
  -0.410979f, 0.107641f,  -0.117573f, -0.326512f, 0.235166f,  0.084959f,
  0.290063f,  -0.005838f, 0.459894f,  1.023709f,  -0.196145f, 1.100137f,
  -0.319815f, -0.308526f, -0.443389f, -0.272769f, -0.035259f, -0.026932f,
  -0.029743f, 0.125113f,  -0.131024f, -0.321458f, -0.143996f, 0.008714f,
  -0.101234f, 0.079706f,  -1.128615f, -0.467381f, 0.220563f,  -0.409900f,
  -0.435353f, 0.759499f,  -0.465799f, -0.394309f, 0.176282f,  -0.086275f,
  -0.161225f, -0.354814f, 0.562871f,  0.418253f,  0.414361f,  0.445480f,
  -0.995903f, -0.086632f, -0.230645f, 0.354656f,  -0.317576f, 0.079926f,
  0.424369f,  0.997232f,  -0.304388f, 1.071667f,  -0.023540f, 0.029677f,
  0.108564f,  0.183581f,  -0.201395f, -0.054854f, -0.193039f, -0.049899f,
  -0.271949f, -0.358483f, 0.304930f,  0.023823f,  -0.009319f, -0.214247f,
  0.100712f,  -0.050162f, 0.327103f,  -0.212999f, -0.030496f, 0.316380f,
  -0.439589f, -0.249959f, 0.229777f,  -0.353664f, -0.384559f, 0.114236f,
  0.023119f,  0.007927f,  0.618368f,  0.957759f,  -0.019780f, -1.002389f,
  0.564277f,  -0.839531f, 1.040445f,  0.054340f,  0.031908f,  -0.032893f,
  -0.019170f, -0.042011f, 0.568928f,  0.362567f,  -0.559999f, -0.605344f,
  -0.586146f, -0.290778f, 0.195943f,  -0.109580f, -0.088898f, -0.113054f,
  0.293282f,  0.429019f,  0.306136f,  0.863025f,  0.021234f,  0.125770f,
  -0.097108f, -0.072659f, -0.137053f, -0.191631f, 0.106281f,  0.064151f,
  0.029883f,  0.076287f,  0.757543f,  0.276713f,  -2.529775f, -0.351727f,
  -1.832316f, 0.544780f,  -0.944529f, 0.509705f,  -0.010236f, -0.016181f,
  0.021520f,  0.086417f,  0.041312f,  0.296853f,  -0.372378f, 0.354446f,
  -1.366762f, 0.048875f,  0.464918f,  -0.007450f, 0.750013f,  -0.360261f,
  0.518532f,  0.753776f,  0.641448f,  0.710746f,  0.250866f,  0.257063f,
  0.283421f,  0.253585f,  0.170303f,  0.210426f,  0.208842f,  0.158000f,
  -0.033144f, 0.130748f,  0.907147f,  0.409248f,  -0.854301f, -0.981307f,
  0.294427f,  -0.507137f, 1.079967f,  0.203203f,  0.383890f,  0.368278f,
  0.305122f,  0.449288f,  -0.044507f, -0.547263f, -0.298245f, -0.497834f,
  0.007016f,  -0.101982f, -0.073488f, -0.096111f, -0.479418f, -0.045497f,
  0.033502f,  -0.018578f, -0.231531f, 0.177949f,  0.099564f,  -0.010233f,
  -0.333055f, -0.078586f, -0.417867f, 0.171271f,  0.013662f,  -0.143599f,
  -0.117296f, 0.135382f,  0.048321f,  0.000924f,  -0.055024f, -0.405595f,
  -0.068260f, -0.271011f, -0.436425f, 0.206751f,  -0.899890f, 0.605510f,
  0.535649f,  -0.238919f, -0.037619f, -0.213734f, -0.391360f, -0.132344f,
  0.004660f,  0.176644f,  -1.008475f, -0.038895f, 0.155429f,  -0.095229f,
  -0.680124f, -0.258063f, -0.261901f, 0.110380f,  -0.337649f, -0.505870f,
  -1.428536f, 0.610629f,  0.254905f,  0.045098f,  0.044109f,  0.172329f,
  0.060001f,  -0.234009f, -0.184855f, -0.153028f, -0.140897f, -0.152006f,
  -0.312134f, 0.081261f,  0.160166f,  0.112690f,  0.266081f,  0.030175f,
  -0.242746f, 0.000754f,  -0.341811f, -0.149774f, -0.017484f, -0.301342f,
  -0.121466f, 0.067300f,  0.342176f,  0.474538f,  0.085441f,  -0.263935f,
  0.479235f,  -0.003713f, -0.784840f, 0.119480f,  0.456632f,  -0.640082f,
  -0.080575f, -0.744403f, 0.259970f,  0.034667f,  -0.274641f, -0.257594f,
  -1.121124f, -0.003745f, -0.420693f, 0.300441f,  -0.100976f, -1.049016f,
  0.201960f,  0.113054f,  0.187010f,  1.237427f,  0.054803f,  -0.028673f,
  0.003596f,  -0.034724f, 0.117246f,  0.190977f,  0.278915f,  0.224307f,
  0.017852f,  -0.336233f, -0.372311f, -0.182284f, -0.143510f, 0.331466f,
  0.045698f,  -0.301095f, 0.184447f,  0.348240f,  -0.017021f, -0.145064f,
  -0.000221f, -0.382256f, -0.302683f, -0.083927f, -0.008070f, 0.217907f,
  0.647597f,  -0.050490f, -0.572736f, -0.985748f, -0.289943f, 0.041391f,
  -0.795464f, -0.186680f, -0.354062f, -0.617400f, -0.282783f, -0.170450f,
  -0.197197f, -0.146496f, -0.173692f, -0.106277f, -0.071004f, -0.124405f,
  -0.971412f, 0.038542f,  0.705204f,  0.887113f,  0.150430f,  -0.243676f,
  0.638410f,  0.320953f,  0.776676f,  0.527584f,  0.070389f,  0.051554f,
  0.177519f,  0.140451f,  0.128892f,  0.087771f,  0.197660f,  0.194764f,
};

static const float av1_4_partition_nn_bias_16_layer0[24] = {
  0.614063f,  -0.384872f, 0.084884f,  -0.023980f, -0.378765f, -0.082312f,
  -0.458271f, 0.189578f,  -0.046169f, -0.073308f, -0.372322f, 0.162793f,
  0.148803f,  0.829214f,  -0.221162f, -0.111157f, -0.017484f, -0.280596f,
  -0.031905f, -0.143459f, 0.078823f,  -0.021940f, 0.026834f,  0.257472f,
};

static const float av1_4_partition_nn_weights_16_layer1[24 * LABEL_SIZE] = {
  -0.985391f, 0.587616f,  0.740683f,  0.192066f,  0.447080f,  -0.016585f,
  0.680449f,  0.028983f,  0.643111f,  0.234338f,  0.107148f,  0.328456f,
  -0.216394f, 1.106838f,  -0.179062f, -0.129108f, -0.121655f, -0.151340f,
  -0.306017f, -0.350989f, 0.859284f,  -0.372831f, -0.954419f, 0.250495f,
  1.046732f,  0.287923f,  -0.421088f, 0.326613f,  -0.314396f, -0.084757f,
  -0.474228f, 0.687999f,  0.052334f,  0.441708f,  -0.630698f, -0.350348f,
  -0.602067f, -0.434161f, -0.489824f, -0.313193f, 0.315568f,  0.603119f,
  0.120245f,  0.182920f,  -1.117797f, -0.239594f, -0.296296f, -0.718093f,
  0.489497f,  -0.527019f, 0.102453f,  0.426731f,  0.034606f,  0.311461f,
  -0.012723f, -0.229877f, -0.284290f, 0.383227f,  0.065696f,  -0.222400f,
  1.279248f,  -0.862190f, 0.629766f,  -0.250011f, -0.325060f, -0.360115f,
  -0.159540f, -0.291856f, -0.038348f, 0.224639f,  0.600934f,  0.030205f,
  1.337615f,  -0.286409f, -0.473710f, -0.418995f, -1.035249f, 0.004359f,
  -0.481860f, 0.563625f,  -0.154709f, -0.101198f, -0.758796f, -0.507616f,
  -0.095253f, -0.711135f, 0.207759f,  0.076313f,  -0.056087f, -0.162719f,
  -0.232918f, -0.128402f, -0.444620f, -0.447344f, 1.126012f,  -1.504446f,
};

static const float av1_4_partition_nn_bias_16_layer1[LABEL_SIZE] = {
  -0.462133f,
  0.465060f,
  0.062211f,
  0.401786f,
};

static const NN_CONFIG av1_4_partition_nnconfig_16 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      24,  // num_hidden_nodes
  },
  {
      av1_4_partition_nn_weights_16_layer0,
      av1_4_partition_nn_weights_16_layer1,
  },
  {
      av1_4_partition_nn_bias_16_layer0,
      av1_4_partition_nn_bias_16_layer1,
  },
};

static const float av1_4_partition_nn_weights_32_layer0[FEATURE_SIZE * 32] = {
  -0.219494f, -0.428273f, 0.471006f,  0.448210f,  -0.152935f, 0.440435f,
  0.922857f,  -0.074436f, 1.002195f,  0.414176f,  -0.327202f, -0.380066f,
  -0.212346f, 0.061868f,  -0.056620f, 0.594134f,  0.617995f,  0.308358f,
  0.232484f,  0.129849f,  1.483593f,  -0.071460f, 1.984515f,  1.116422f,
  -1.141762f, -0.306220f, 0.089075f,  -0.271845f, 0.187524f,  0.050396f,
  -0.061025f, 0.030809f,  0.172799f,  -0.458151f, -0.318357f, 0.122052f,
  -0.414329f, 0.089366f,  0.118898f,  -0.376213f, -0.206151f, -0.519946f,
  -0.463252f, -0.206694f, -0.254383f, -0.379487f, 0.093059f,  -0.245280f,
  -0.205044f, -0.280060f, -0.171229f, -0.045389f, -0.179481f, -0.306245f,
  -0.500856f, 0.003388f,  -0.527397f, -0.449330f, -0.174272f, 0.123769f,
  0.023005f,  0.157273f,  0.073400f,  0.019099f,  -0.113848f, -0.098601f,
  -0.290946f, -0.046770f, -0.314592f, -0.179914f, -0.391411f, -0.235631f,
  -1.282604f, 0.048505f,  -0.746382f, 0.093740f,  -0.706583f, -0.085729f,
  0.947382f,  -0.002961f, 1.175362f,  1.007309f,  0.141638f,  -0.037608f,
  -0.118807f, -0.021474f, -0.146763f, 0.069363f,  -0.074372f, -0.215713f,
  -0.004134f, -0.114110f, -0.330438f, -0.031136f, 0.111821f,  -0.534598f,
  -0.357759f, -0.455950f, 0.139469f,  0.036582f,  -0.384743f, -0.168828f,
  -0.239250f, 0.003520f,  -0.049003f, 0.075702f,  -0.025809f, -0.225972f,
  -0.228905f, -0.412489f, 0.060570f,  -0.328819f, -0.206446f, -0.080231f,
  -0.372008f, -0.218118f, -0.011954f, 0.024155f,  0.156014f,  0.020679f,
  0.194398f,  -0.283491f, -0.024463f, -0.275099f, 0.028031f,  0.026340f,
  -0.254668f, 0.103637f,  2.178693f,  0.552284f,  0.109366f,  -0.474806f,
  -0.379286f, -0.026315f, 2.487924f,  -0.089466f, 0.206428f,  0.114578f,
  0.152248f,  0.184050f,  -0.631948f, -0.014793f, -0.283782f, -0.830353f,
  0.009343f,  -0.021029f, -0.060534f, -0.025164f, 1.841311f,  1.842748f,
  -1.979708f, 0.450985f,  -1.606357f, -0.785454f, -0.212679f, -0.344342f,
  0.198991f,  -0.258070f, 0.055974f,  0.224069f,  0.453051f,  0.408053f,
  0.027873f,  -0.180538f, 0.056609f,  0.207654f,  0.104086f,  -0.194426f,
  -0.359789f, -0.381143f, -0.331212f, -0.203973f, -0.324313f, -0.160825f,
  -0.160439f, -0.044856f, -0.346647f, 0.044859f,  0.231398f,  -0.023643f,
  -0.140316f, -0.260177f, 0.206965f,  -0.425386f, -0.420268f, -0.409748f,
  0.006971f,  0.066186f,  -0.034950f, -0.345518f, 0.018633f,  -0.122489f,
  -0.038506f, -0.330942f, 0.161236f,  -0.314119f, -0.050202f, -0.179597f,
  0.731897f,  -0.184481f, 0.153598f,  -0.539501f, -0.301493f, -0.184967f,
  -0.883754f, -0.586959f, -0.136292f, -1.772065f, -0.196276f, -0.053272f,
  -0.101083f, -0.064142f, 0.161190f,  0.430826f,  0.355647f,  0.138266f,
  0.051114f,  -0.028893f, -0.477673f, -0.238663f, -0.354117f, -0.056747f,
  -0.334273f, -0.497688f, -0.486004f, -0.092033f, -0.241304f, -0.373250f,
  0.120193f,  0.011360f,  -0.010475f, -0.092739f, -0.159650f, -0.033129f,
  -0.259893f, -0.073217f, 0.200128f,  0.103407f,  -0.229233f, 0.128831f,
  -0.063450f, -0.241732f, -0.408428f, -0.342239f, -0.264326f, -0.105403f,
  -0.442879f, -0.310456f, -0.112881f, 0.263696f,  -0.205014f, -0.497936f,
  -0.261734f, -0.382312f, -0.426807f, -0.021995f, -0.152794f, -0.301494f,
  0.117232f,  -0.577809f, 0.154596f,  -0.409522f, -0.413113f, -0.359199f,
  0.307294f,  -0.008746f, -0.310522f, 0.347620f,  -0.384845f, -0.451398f,
  -0.226199f, 0.054154f,  -0.167608f, 0.046836f,  -0.013285f, -0.408119f,
  -0.177973f, -0.248293f, -0.465830f, 0.035827f,  -0.222208f, -0.221717f,
  0.066392f,  -0.349769f, -0.428029f, -0.516692f, 0.022398f,  -0.251682f,
  0.134746f,  0.011167f,  -2.078787f, 0.173592f,  -1.948348f, 0.330060f,
  1.993785f,  -0.052859f, -0.004795f, -3.703177f, 0.013450f,  -0.011687f,
  0.073079f,  0.034803f,  0.025515f,  0.005994f,  0.101731f,  0.074303f,
  -0.109962f, -0.270825f, -0.068273f, -0.163268f, -0.252826f, 0.137190f,
  0.007667f,  -0.358453f, 0.027412f,  0.033492f,  0.021197f,  -0.049991f,
  0.104468f,  -0.012157f, -0.056252f, -0.380756f, -0.338483f, 0.233235f,
  -0.048631f, -0.441209f, -0.158482f, -0.148108f, -0.263453f, 0.138847f,
  -0.304073f, -0.336312f, -0.017941f, -0.135563f, 0.075137f,  -0.246475f,
  -0.229144f, -0.087744f, -0.346909f, 0.172611f,  0.004377f,  -0.009386f,
  -0.023104f, 0.008000f,  -0.029390f, -0.317842f, 0.549674f,  -0.195337f,
  -0.863979f, 0.160889f,  -0.269014f, -0.442104f, -1.799191f, 1.396533f,
  -0.112837f, 0.881303f,  0.000764f,  -0.035415f, -0.141877f, 0.184831f,
  -0.363566f, -0.178569f, 0.254134f,  -0.326893f, 0.127325f,  0.310620f,
  -0.384621f, 0.146058f,  -0.287682f, -0.373447f, 0.026930f,  0.251650f,
  0.053817f,  0.227509f,  0.121396f,  0.396514f,  -0.278381f, -0.038969f,
  -1.538756f, -0.002856f, -0.892900f, 0.363426f,  -1.257922f, 0.743795f,
  0.941177f,  0.219345f,  0.684189f,  1.396858f,  0.026299f,  -0.093433f,
  -0.066182f, 0.057868f,  -0.089278f, -0.159680f, -0.262035f, -0.236656f,
  0.005349f,  -0.031314f, 0.027917f,  -0.182113f, -0.212086f, -0.160774f,
  0.051468f,  0.036787f,  0.183881f,  -0.288205f, -0.349691f, 0.162511f,
  0.117878f,  -0.294534f, -0.365037f, -0.246313f, 0.073977f,  -0.072378f,
  -0.173579f, -0.584560f, 0.547194f,  0.259853f,  -0.405287f, -0.421146f,
  0.165788f,  -0.146964f, 0.257415f,  0.772394f,  -0.475302f, -0.310906f,
  0.058723f,  0.276833f,  0.586842f,  0.248998f,  -0.061135f, 0.255779f,
  0.152158f,  -0.024781f, 2.821834f,  1.365141f,  0.914744f,  0.165752f,
  -1.048304f, -0.333891f, 1.804087f,  -0.437028f, -0.120211f, -0.020443f,
  0.040077f,  0.258600f,  -0.598893f, -0.494579f, -0.281054f, -0.517041f,
  0.005258f,  0.053986f,  0.322755f,  0.429495f,  -1.992364f, -0.717192f,
  -1.774802f, 2.047362f,  -0.016194f, 0.312606f,  0.019331f,  0.060950f,
  0.116428f,  0.168458f,  -0.307001f, -0.420734f, 0.475843f,  0.425346f,
  -0.107119f, 0.049892f,  -1.168619f, 0.010878f,  0.354872f,  0.902717f,
  -0.391407f, 0.332772f,  -1.335037f, -0.447100f, 0.481719f,  -0.101069f,
  -1.806565f, 0.925280f,  0.346999f,  0.093809f,  0.006275f,  0.270814f,
  -0.691123f, 0.230748f,  0.137033f,  0.068228f,  1.555975f,  -0.271637f,
  -0.370403f, 0.236131f,  0.367464f,  -0.136562f, 0.428838f,  0.181750f,
  0.338762f,  0.292449f,  -0.748204f, -0.922731f, -0.959445f, -0.806418f,
  -0.140501f, 0.070525f,  1.248748f,  0.637990f,  -1.307246f, -0.514055f,
  0.393858f,  -1.858727f, 0.713591f,  -0.141044f, 0.080723f,  0.120220f,
  -0.031175f, 0.224488f,  0.753818f,  -0.833351f, -1.099132f, 0.651100f,
  -0.135061f, -0.043820f, 0.026983f,  -0.059259f, 0.001345f,  -0.281775f,
  0.006958f,  0.046103f,  -0.246539f, 0.057630f,  -0.360778f, -0.160681f,
  -0.414870f, -0.301979f, 0.000683f,  0.132957f,  -0.477609f, 0.106110f,
  -0.637769f, -0.078374f, -0.229494f, 0.583108f,  -0.822973f, -0.107540f,
  1.063426f,  -0.268346f, 1.105787f,  2.587550f,  -0.020314f, -0.002161f,
  -0.063836f, -0.099990f, -0.103975f, -0.114078f, -0.094199f, -0.065181f,
  -0.019870f, -0.018920f, -0.219732f, 0.035608f,  -1.789450f, 0.483032f,
  -0.464729f, 1.563277f,  -1.054195f, 0.359991f,  0.065204f,  0.135623f,
  0.158380f,  -0.103815f, -1.398726f, -1.436666f, -0.356311f, 0.507752f,
};

static const float av1_4_partition_nn_bias_32_layer0[32] = {
  0.421645f,  -0.620548f, -0.187819f, -0.189414f, -0.204975f, -0.189600f,
  -0.174917f, -0.651928f, -0.799655f, -0.086105f, -0.163449f, -0.089212f,
  -0.214495f, -0.108500f, -0.065777f, -0.127704f, 1.544948f,  -0.032831f,
  -0.165621f, 0.145844f,  -0.032104f, -0.453246f, -0.113444f, 0.321589f,
  -0.862375f, -0.108826f, -0.486259f, 0.685325f,  0.072569f,  -0.187961f,
  0.109579f,  -0.082685f,
};

static const float av1_4_partition_nn_weights_32_layer1[32 * LABEL_SIZE] = {
  0.255012f,  0.658860f,  0.216907f,  0.165947f,  0.241182f,  0.340854f,
  0.409445f,  0.165220f,  0.553373f,  -0.242385f, -0.209571f, 0.255515f,
  0.222500f,  0.037032f,  0.238590f,  0.061624f,  -2.038693f, 0.264167f,
  -0.230144f, 0.129952f,  -0.027979f, 0.847761f,  0.438922f,  0.462323f,
  0.555345f,  0.030689f,  0.336357f,  -0.357326f, -0.113137f, 0.272631f,
  0.421022f,  0.367776f,  -0.197094f, 0.157117f,  -0.015008f, -0.056123f,
  -0.283913f, 0.186417f,  0.178561f,  -0.763041f, 0.602038f,  0.341092f,
  0.320453f,  -0.312776f, -0.371240f, -0.356279f, 0.220117f,  -0.131871f,
  1.517429f,  0.162223f,  -0.255069f, 0.451861f,  0.045071f,  -0.223257f,
  0.003257f,  0.015734f,  -0.630447f, -0.672588f, 0.670164f,  0.571031f,
  -0.657948f, 0.034506f,  -0.249076f, 0.790293f,  0.066491f,  -0.131245f,
  0.355173f,  0.564622f,  0.374048f,  0.033974f,  0.253970f,  0.495498f,
  -0.556321f, -0.104651f, 0.276947f,  0.057148f,  -0.039126f, -0.170050f,
  -0.141542f, 0.158541f,  0.582763f,  -0.100992f, 0.096705f,  -0.209029f,
  0.008449f,  0.255865f,  0.103565f,  0.317719f,  0.479499f,  0.599126f,
  -0.065613f, -0.268614f, 0.508736f,  0.180813f,  -0.815868f, 0.051238f,
  0.001223f,  -0.305423f, -0.270079f, 0.036180f,  0.304342f,  0.202634f,
  0.218348f,  -0.304304f, -0.438297f, 0.241123f,  0.200230f,  0.151804f,
  0.051944f,  0.160422f,  -0.262981f, -0.417412f, 1.845729f,  -0.086183f,
  0.403517f,  0.059667f,  0.564543f,  -0.081752f, 0.114907f,  -0.284489f,
  -0.673943f, 0.056965f,  0.362221f,  0.403224f,  -0.000233f, -0.209552f,
  -0.800926f, -0.134132f,
};

static const float av1_4_partition_nn_bias_32_layer1[LABEL_SIZE] = {
  -0.019518f,
  0.198546f,
  0.339015f,
  -0.261961f,
};

static const NN_CONFIG av1_4_partition_nnconfig_32 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      32,  // num_hidden_nodes
  },
  {
      av1_4_partition_nn_weights_32_layer0,
      av1_4_partition_nn_weights_32_layer1,
  },
  {
      av1_4_partition_nn_bias_32_layer0,
      av1_4_partition_nn_bias_32_layer1,
  },
};

static const float av1_4_partition_nn_weights_64_layer0[FEATURE_SIZE * 24] = {
  -0.152649f, 0.074509f,  1.000136f,  0.601661f,  -1.416694f, -1.932396f,
  -1.163850f, 0.640931f,  -0.888625f, -0.345711f, 0.161799f,  0.103165f,
  0.147513f,  0.089956f,  0.204329f,  0.196922f,  0.014927f,  0.283714f,
  -0.110422f, 0.062005f,  -0.531870f, -0.075287f, -0.448349f, -0.218881f,
  -0.005592f, -0.130490f, -0.015779f, 0.093521f,  -0.158487f, 0.072241f,
  0.066879f,  -0.418566f, -0.206281f, 0.025634f,  0.048334f,  -0.534750f,
  0.302081f,  0.028707f,  -1.543248f, 0.103799f,  -1.214052f, 0.395870f,
  0.394754f,  -0.272170f, -0.702953f, -4.057464f, -0.033497f, -0.042142f,
  0.014742f,  0.065263f,  0.000879f,  -0.019768f, 0.101275f,  0.163059f,
  -0.371392f, -0.283484f, 0.241915f,  0.012684f,  -0.210101f, -0.166534f,
  -0.024894f, 0.274696f,  0.098993f,  0.104086f,  0.055044f,  -0.289378f,
  0.146571f,  -0.147441f, 0.004056f,  0.112244f,  -0.416162f, -0.033176f,
  -0.214836f, -0.213787f, 0.023197f,  -0.339043f, 0.301109f,  -0.408551f,
  0.284922f,  -0.344418f, -0.039255f, 0.158748f,  -0.344169f, 0.078286f,
  -0.043957f, -0.302162f, -0.310826f, 0.063425f,  0.198166f,  -0.285324f,
  -0.108252f, 0.038992f,  -1.053110f, -1.663290f, -0.417185f, 1.504443f,
  0.643206f,  -0.850240f, 0.889641f,  -0.733214f, 0.147302f,  0.060291f,
  -0.052954f, 0.167453f,  0.111870f,  0.085471f,  0.035107f,  0.064361f,
  0.176053f,  0.184373f,  0.676576f,  0.066164f,  1.455569f,  0.925111f,
  -0.640845f, 0.803795f,  -0.653782f, -0.201038f, 0.060033f,  0.016964f,
  -0.047590f, 0.045908f,  0.354162f,  0.014812f,  0.156978f,  0.058792f,
  -0.238119f, 0.002450f,  -0.094388f, -0.155229f, 0.194858f,  -0.355429f,
  -0.187098f, -0.119264f, -0.088694f, -0.102845f, 0.184905f,  -0.425339f,
  -0.157808f, -0.104599f, -0.393248f, -0.379842f, 0.027741f,  -0.185816f,
  -0.317294f, 0.002453f,  -0.498241f, -0.204302f, -0.079093f, 0.020646f,
  -0.412850f, -0.426039f, -0.177050f, -0.419304f, -0.064478f, -0.191802f,
  -0.146812f, 0.171111f,  0.090261f,  -0.367033f, -0.299051f, -0.322132f,
  0.428192f,  -0.252613f, 0.488498f,  -0.559682f, 0.486720f,  -0.511084f,
  0.992506f,  0.346765f,  -0.118697f, -0.065127f, -0.376612f, -0.345137f,
  -0.426517f, -0.516836f, 0.307083f,  0.609362f,  0.369555f,  0.093775f,
  -0.375664f, -0.221595f, -0.025465f, 0.134374f,  -0.387031f, 0.096236f,
  0.337465f,  -0.124029f, -0.157340f, -0.368790f, -0.104490f, -0.279507f,
  -0.247705f, 0.146559f,  -0.236206f, -0.036073f, 0.064206f,  -0.330919f,
  0.516591f,  -0.013492f, 1.269568f,  1.182530f,  -0.455390f, -1.328091f,
  -0.200950f, -0.380513f, -0.195532f, -0.341479f, 0.016064f,  0.021176f,
  0.169119f,  0.103707f,  -0.174504f, -0.462719f, -0.079445f, -0.247128f,
  0.459111f,  0.036129f,  0.769570f,  -0.080405f, 1.667107f,  0.355567f,
  -2.433896f, 0.627572f,  -0.600090f, -0.651872f, -0.059769f, -0.041945f,
  -0.009933f, 0.014864f,  -0.049378f, -0.041561f, 0.075180f,  0.138307f,
  0.122366f,  -0.160756f, 0.215327f,  0.013572f,  0.198194f,  -0.762650f,
  0.054466f,  1.110332f,  1.692853f,  0.658654f,  -0.409549f, 0.506085f,
  0.330962f,  -0.223008f, 0.007448f,  -0.289062f, -0.476231f, -0.228359f,
  0.013977f,  -0.000609f, -0.673604f, 0.275996f,  0.405291f,  1.693561f,
  -1.079768f, 1.122516f,  -0.203227f, 0.099265f,  -0.165207f, -0.323899f,
  -0.269973f, -0.080122f, 0.127700f,  0.190201f,  0.219527f,  0.306194f,
  0.026049f,  -0.003779f, 1.107357f,  1.720315f,  1.017908f,  0.078664f,
  -1.599813f, -0.482636f, -0.117450f, 0.122249f,  0.030220f,  0.039794f,
  0.176350f,  0.129715f,  -0.305755f, -0.274044f, -0.299640f, -0.187335f,
  -0.073616f, -0.564507f, -0.127758f, 0.044855f,  -0.191090f, 0.039095f,
  0.115378f,  0.969352f,  -0.088360f, 0.301443f,  0.065726f,  -0.019740f,
  -0.102350f, -0.084913f, -0.194615f, 0.118582f,  0.920789f,  -0.171615f,
  -1.436553f, -0.026419f, -0.730864f, 0.615697f,  -0.795079f, 0.119701f,
  0.601782f,  0.792902f,  0.184920f,  1.635090f,  -0.085860f, -0.033187f,
  -0.166883f, 0.008487f,  -0.128300f, -0.089923f, -0.108781f, -0.133719f,
  -0.011988f, -0.239816f, -0.092563f, -0.238471f, -0.339722f, 0.177432f,
  -0.063101f, -0.121002f, 0.058072f,  -0.031166f, 0.086413f,  -0.016203f,
  -0.305075f, -0.005420f, -0.168796f, 0.148745f,  -0.116737f, -0.050222f,
  -0.287952f, -0.290982f, -0.090449f, 0.076098f,  -0.345632f, -0.061309f,
  0.142218f,  0.035692f,  0.304517f,  -0.228031f, 0.119608f,  -0.120350f,
  0.163404f,  -0.105605f, -0.305462f, -0.176657f, 0.210070f,  -0.227600f,
  -0.081965f, -0.464027f, -0.053782f, -0.018367f, 0.119159f,  0.017162f,
  -0.069792f, 0.305768f,  -0.421095f, 0.187740f,  -0.032059f, 0.575115f,
  -0.064283f, -0.091828f, 0.772648f,  -0.393189f, -0.297098f, 0.141420f,
  0.826389f,  -0.071586f, -0.893968f, -0.346793f, -1.151655f, 0.039393f,
  1.546000f,  -0.094029f, -0.005786f, -0.195764f, -0.169724f, -0.133167f,
  -0.129312f, -0.418860f, -0.026553f, -0.053667f, -0.091976f, -0.106275f,
  -0.492625f, 0.025350f,  -0.332075f, -0.475638f, -0.076667f, -0.065779f,
  0.108957f,  0.246298f,  -0.289007f, -0.442552f, -0.206692f, -0.257453f,
  0.073806f,  -0.458606f, -0.410390f, -0.312674f, -0.144813f, 0.170128f,
  0.018810f,  -0.098241f, 1.027369f,  0.479328f,  1.129707f,  0.484813f,
  -0.085207f, 0.621873f,  -0.520981f, 0.236175f,  0.273487f,  0.061426f,
  0.306085f,  0.161487f,  0.220991f,  0.223783f,  -0.091826f, 0.391031f,
};

static const float av1_4_partition_nn_bias_64_layer0[24] = {
  0.580225f,  -0.191304f, 1.091767f,  -0.134522f, -0.089361f, 0.398750f,
  -0.882708f, -0.213102f, -0.119981f, 0.378296f,  -0.075719f, 0.426598f,
  -2.015505f, 0.202534f,  -1.044792f, -0.841519f, 0.266421f,  -0.047115f,
  -0.131147f, -0.075066f, -0.009441f, 0.853007f,  -0.175606f, -0.868306f,
};

static const float av1_4_partition_nn_weights_64_layer1[24 * LABEL_SIZE] = {
  -0.851937f, -0.211148f, -2.289513f, -0.275071f, 0.251340f,  -0.340847f,
  0.498032f,  0.308652f,  -0.051574f, 0.323146f,  -0.097547f, -0.040269f,
  1.909655f,  0.098348f,  0.588136f,  0.568112f,  0.313297f,  0.920848f,
  -0.014486f, 0.386014f,  0.029199f,  -0.537330f, -0.021502f, 0.349073f,
  -0.524715f, -0.351848f, 1.565454f,  -0.297148f, 0.020177f,  0.648369f,
  0.027321f,  -0.096052f, -0.363163f, -0.132642f, 0.024292f,  -0.734176f,
  -0.782700f, 0.408299f,  0.476945f,  -0.489512f, -0.728318f, -0.632042f,
  0.405417f,  0.184086f,  -0.400730f, 0.359032f,  0.019710f,  -0.217409f,
  0.519159f,  -0.136316f, 0.993592f,  -0.147128f, 0.097495f,  0.426189f,
  -0.295233f, 0.278799f,  0.080667f,  -0.025052f, -0.307757f, 0.418716f,
  -0.853388f, -0.374878f, -0.322725f, 0.696335f,  -0.380649f, -0.160356f,
  -0.140060f, 0.502455f,  0.656728f,  -0.095023f, -0.184198f, -0.347069f,
  0.456372f,  -0.029754f, 0.907923f,  0.265710f,  -0.065505f, 0.226763f,
  -0.277798f, 0.413292f,  -0.593899f, -0.060740f, -0.313358f, -0.249944f,
  -0.627329f, -0.327151f, -0.853788f, -1.163807f, -0.388944f, -0.228788f,
  -0.057382f, 0.334741f,  -0.283083f, 0.368280f,  -0.407197f, -0.441849f,
};

static const float av1_4_partition_nn_bias_64_layer1[LABEL_SIZE] = {
  -0.478735f,
  0.292948f,
  0.293172f,
  0.040013f,
};

static const NN_CONFIG av1_4_partition_nnconfig_64 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      24,  // num_hidden_nodes
  },
  {
      av1_4_partition_nn_weights_64_layer0,
      av1_4_partition_nn_weights_64_layer1,
  },
  {
      av1_4_partition_nn_bias_64_layer0,
      av1_4_partition_nn_bias_64_layer1,
  },
};

#undef FEATURE_SIZE
#undef LABEL_SIZE

#define FEATURE_SIZE 4
static const float
    av1_partition_breakout_nn_weights_128_layer0[FEATURE_SIZE * 32] = {
      -0.331785f,  0.068675f,  -0.323814f,  0.033714f,  -0.237835f, 0.166316f,
      -0.498766f,  -0.545634f, -0.266173f,  -0.476957f, -0.120409f, -0.021042f,
      0.124056f,   -0.278750f, -0.110120f,  -0.372812f, 4.547939f,  0.097618f,
      -0.002710f,  -0.064169f, -1.841173f,  -0.403833f, 0.005536f,  0.067188f,
      -0.434935f,  -0.227421f, -0.000011f,  -0.139961f, -0.174056f, -0.652384f,
      -0.000015f,  -0.262847f, -3.319706f,  -0.947693f, 0.002981f,  0.016717f,
      -10.408850f, -0.014568f, -0.000018f,  0.019084f,  1.523383f,  0.074525f,
      -0.002076f,  -0.020734f, 4.881495f,   0.002799f,  0.000342f,  -0.019623f,
      1.786154f,   0.037462f,  -0.019037f,  0.052833f,  11.408153f, -0.044602f,
      0.026155f,   -0.518627f, -0.474499f,  -0.427430f, -0.442733f, -0.011116f,
      -22.379410f, -0.000549f, -0.001418f,  0.008090f,  -0.295090f, -0.230268f,
      -0.337278f,  -0.001127f, -0.644282f,  -0.598783f, -0.539417f, -0.003303f,
      9.189824f,   0.038066f,  -0.004097f,  -0.460045f, -0.308858f, -0.242691f,
      -0.230835f,  -0.273057f, 0.152226f,   0.179239f,  -0.146382f, -0.004655f,
      -0.242940f,  -0.718862f, -0.001685f,  -0.214736f, 3.263186f,  0.079463f,
      -0.003854f,  -0.187461f, -0.599144f,  -0.419808f, -0.000597f, -0.136980f,
      0.184813f,   -0.319525f, -0.007246f,  0.079709f,  -0.883229f, -0.343748f,
      -0.000077f,  -0.172214f, -0.548759f,  -0.194674f, -0.144786f, 0.043896f,
      -0.176364f,  -0.248394f, -0.090215f,  -0.294743f, -0.280980f, -0.181436f,
      -0.115681f,  -0.071915f, -13.035494f, -0.075623f, 0.017052f,  -0.171152f,
      5.910803f,   0.128344f,  0.010256f,   -1.073301f, 2.387826f,  0.166183f,
      -0.007193f,  -0.257836f,
    };

static const float av1_partition_breakout_nn_bias_128_layer0[32] = {
  0.115591f,  -0.100178f, -0.165523f, -0.122997f, 11.045759f,  1.034761f,
  -0.323672f, -0.189087f, 2.850950f,  7.010029f,  -21.447067f, 1.877031f,
  0.437442f,  5.929414f,  -0.117274f, 4.462253f,  -0.135198f,  -0.145927f,
  8.727211f,  0.000000f,  -3.532987f, -0.405898f, 11.364439f,  -0.141728f,
  -5.994947f, -0.362574f, 1.857687f,  -0.100400f, -0.130312f,  0.006080f,
  0.429660f,  -8.439470f,
};

static const float av1_partition_breakout_nn_weights_128_layer1[32] = {
  -0.013738f, 0.022052f,  -0.074437f, -0.211377f, -0.080433f, 0.015543f,
  0.002091f,  0.014252f,  0.134834f,  0.190263f,  0.244175f,  -0.031747f,
  0.020068f,  -0.068326f, 0.185471f,  0.660268f,  -0.134898f, -0.010376f,
  -0.276023f, -0.282921f, -0.022769f, 0.007070f,  -0.186235f, 0.024407f,
  -0.024837f, 0.005764f,  0.016599f,  -0.040077f, 0.020990f,  0.095054f,
  -0.039662f, 0.131499f,
};

static const float av1_partition_breakout_nn_bias_128_layer1[1] = {
  0.86678213f,
};

static const NN_CONFIG av1_partition_breakout_nnconfig_128 = {
  FEATURE_SIZE,  // num_inputs
  1,             // num_outputs
  1,             // num_hidden_layers
  {
      32,  // num_hidden_nodes
  },
  {
      av1_partition_breakout_nn_weights_128_layer0,
      av1_partition_breakout_nn_weights_128_layer1,
  },
  {
      av1_partition_breakout_nn_bias_128_layer0,
      av1_partition_breakout_nn_bias_128_layer1,
  },
};

static const float
    av1_partition_breakout_nn_weights_64_layer0[FEATURE_SIZE * 16] = {
      0.872892f,  -0.235539f, -0.412159f, -0.142533f, -2.251479f, -0.057073f,
      -0.001373f, 0.112147f,  5.281734f,  0.060704f,  0.000838f,  -0.961554f,
      0.244995f,  0.154515f,  -0.292654f, -0.167177f, -3.759112f, -0.486347f,
      0.003208f,  -0.418226f, 2.618152f,  0.026832f,  0.003988f,  -0.404406f,
      -0.405434f, 0.102791f,  -0.033406f, -0.029820f, -4.492342f, -0.154291f,
      0.012947f,  -0.195075f, 0.009311f,  -0.411410f, -0.010986f, -0.554822f,
      0.160576f,  0.020796f,  -0.457230f, -0.191111f, -7.759542f, -0.065039f,
      -0.001322f, 0.055691f,  0.291924f,  -0.053076f, -0.148379f, -0.298383f,
      1.022023f,  -0.033668f, -0.000804f, -0.825778f, -3.902254f, -0.085812f,
      -0.052520f, -0.035012f, -0.465468f, -0.319231f, -0.497529f, -0.183068f,
      -2.407131f, -0.062304f, 0.000874f,  0.108786f,
    };

static const float av1_partition_breakout_nn_bias_64_layer0[16] = {
  0.081425f,  -14.404084f, 11.511393f, -0.930053f, 1.841889f,  15.020920f,
  -1.872288f, 5.392535f,   -0.329335f, -0.005358f, 12.600776f, 0.000000f,
  -0.337413f, 4.492778f,   0.000000f,  17.043072f,
};

static const float av1_partition_breakout_nn_weights_64_layer1[16] = {
  -0.465338f, -0.103023f, -0.174808f, -0.005156f, -0.016366f, -0.172494f,
  0.014185f,  0.067030f,  -0.001939f, -0.175049f, 0.245992f,  -0.181660f,
  -0.038572f, 0.307899f,  -0.294283f, 0.118323f,
};

static const float av1_partition_breakout_nn_bias_64_layer1[1] = {
  -1.33438122f,
};

static const NN_CONFIG av1_partition_breakout_nnconfig_64 = {
  FEATURE_SIZE,  // num_inputs
  1,             // num_outputs
  1,             // num_hidden_layers
  {
      16,  // num_hidden_nodes
  },
  {
      av1_partition_breakout_nn_weights_64_layer0,
      av1_partition_breakout_nn_weights_64_layer1,
  },
  {
      av1_partition_breakout_nn_bias_64_layer0,
      av1_partition_breakout_nn_bias_64_layer1,
  },
};

static const float
    av1_partition_breakout_nn_weights_32_layer0[FEATURE_SIZE * 16] = {
      -4.825528f, -0.145737f, 0.001907f,  0.145415f,  -1.858153f, -0.080744f,
      0.000601f,  0.211991f,  0.384265f,  -0.043945f, -0.521332f, -0.170622f,
      -0.046866f, -0.600506f, -0.001216f, -0.332760f, -0.447677f, -0.605844f,
      -0.121008f, -0.119936f, -0.215739f, -0.269665f, -0.668587f, 0.071318f,
      -1.202551f, -0.729727f, -0.370084f, 0.088215f,  -1.926800f, -0.086519f,
      0.000359f,  0.215120f,  0.718749f,  0.022942f,  0.003840f,  -0.176518f,
      1.213451f,  0.080786f,  0.001557f,  -1.053430f, 0.202698f,  -0.583919f,
      -0.535512f, -0.239927f, -0.110151f, -0.128832f, -0.441087f, -0.145575f,
      -0.178518f, -0.585784f, 0.000029f,  -0.833014f, -0.331358f, -0.520297f,
      -0.088676f, -0.178487f, -1.430755f, 0.022981f,  -0.106931f, 0.015573f,
      -0.520814f, -0.045386f, -0.443123f, -0.484209f,
    };

static const float av1_partition_breakout_nn_bias_32_layer0[16] = {
  11.747026f, -9.337718f, 0.341648f, -0.155847f, -0.104005f, 4.666283f,
  6.669584f,  16.625504f, 9.885626f, 15.439183f, -0.346080f, 0.000000f,
  -0.423808f, 0.000000f,  6.352258f, -0.155787f,
};

static const float av1_partition_breakout_nn_weights_32_layer1[16] = {
  0.168561f,  -0.122519f, 0.524667f,  0.032474f,  0.059097f,  0.011900f,
  0.166445f,  0.127256f,  -0.034838f, -0.212586f, -0.317973f, 0.348419f,
  -0.004171f, 0.157694f,  0.117845f,  0.272115f,
};

static const float av1_partition_breakout_nn_bias_32_layer1[1] = {
  0.09049262f,
};

static const NN_CONFIG av1_partition_breakout_nnconfig_32 = {
  FEATURE_SIZE,  // num_inputs
  1,             // num_outputs
  1,             // num_hidden_layers
  {
      16,  // num_hidden_nodes
  },
  {
      av1_partition_breakout_nn_weights_32_layer0,
      av1_partition_breakout_nn_weights_32_layer1,
  },
  {
      av1_partition_breakout_nn_bias_32_layer0,
      av1_partition_breakout_nn_bias_32_layer1,
  },
};

static const float
    av1_partition_breakout_nn_weights_16_layer0[FEATURE_SIZE * 16] = {
      0.209371f,  0.028758f,  0.005764f,  -0.384401f, -0.625777f, -0.005647f,
      -0.316867f, 0.042985f,  0.127344f,  0.025461f,  0.011465f,  -0.071043f,
      -0.295977f, -0.076093f, -0.209681f, -0.311653f, -0.147538f, 0.009910f,
      -0.130997f, -0.012326f, 0.024124f,  -0.323578f, -0.005790f, -0.085664f,
      -1.575066f, -0.119221f, 0.015018f,  0.187204f,  0.238117f,  0.084924f,
      -0.004444f, -1.271538f, -0.709860f, -0.006226f, -0.903111f, 0.090573f,
      -0.278642f, -0.011114f, 0.021162f,  0.081290f,  -0.467486f, -0.040771f,
      -0.224069f, -0.714390f, -0.281905f, -0.001336f, -0.761212f, -0.060385f,
      -0.814479f, -0.050450f, -0.003666f, 0.085668f,  -0.272589f, 0.057330f,
      -0.206540f, -0.303418f, 0.075335f,  -0.180468f, -0.064872f, -0.755948f,
      -0.509287f, -0.048877f, -0.001512f, 0.077086f,
    };

static const float av1_partition_breakout_nn_bias_16_layer0[16] = {
  16.421495f, 4.012273f,  -1.828571f, 0.000000f,  -0.263564f, -0.201972f,
  6.564987f,  14.651000f, -3.227779f, 2.241833f,  -0.137116f, 0.762876f,
  5.625762f,  0.615822f,  0.040057f,  16.668884f,
};

static const float av1_partition_breakout_nn_weights_16_layer1[16] = {
  -0.096440f, 0.184316f,  -0.021148f, 0.424974f, 0.003743f,  0.006310f,
  0.046266f,  -0.219224f, -0.087004f, 0.024623f, -0.275798f, 0.120164f,
  0.269773f,  -0.021105f, -0.146698f, 0.188764f,
};

static const float av1_partition_breakout_nn_bias_16_layer1[1] = {
  1.60751927f,
};

static const NN_CONFIG av1_partition_breakout_nnconfig_16 = {
  FEATURE_SIZE,  // num_inputs
  1,             // num_outputs
  1,             // num_hidden_layers
  {
      16,  // num_hidden_nodes
  },
  {
      av1_partition_breakout_nn_weights_16_layer0,
      av1_partition_breakout_nn_weights_16_layer1,
  },
  {
      av1_partition_breakout_nn_bias_16_layer0,
      av1_partition_breakout_nn_bias_16_layer1,
  },
};

static const float
    av1_partition_breakout_nn_weights_8_layer0[FEATURE_SIZE * 16] = {
      -0.255885f, 0.109548f,  -0.111054f, -0.476119f, -1.083031f, -0.342003f,
      0.048241f,  -0.356013f, -0.085054f, 0.124908f,  0.000084f,  -0.149906f,
      -0.729829f, 0.133535f,  -0.002125f, 0.207516f,  -0.210163f, -0.567365f,
      -0.590103f, 0.045308f,  -0.539406f, 0.130550f,  -0.663879f, -0.170549f,
      0.017587f,  -0.054187f, 0.000550f,  0.038297f,  -0.112891f, -0.012751f,
      -0.048067f, 0.095564f,  0.079892f,  0.077285f,  -0.749708f, -0.286312f,
      -0.054334f, 0.132242f,  -0.004152f, -0.209758f, -0.073407f, 0.082306f,
      -0.001034f, -0.090990f, 0.122823f,  -0.109794f, -0.230066f, -0.391155f,
      -0.262245f, -0.004744f, -0.232246f, 0.099290f,  -0.637484f, 0.111937f,
      -0.548556f, -0.598344f, 0.123265f,  -0.281395f, -0.399711f, -0.525671f,
      -0.596269f, 0.098494f,  -0.005765f, 0.173652f,
    };

static const float av1_partition_breakout_nn_bias_8_layer0[16] = {
  0.194141f, -0.111223f, 2.503733f, -7.155602f, -0.695068f, 0.114874f,
  2.056990f, 5.284306f,  0.639643f, -2.792049f, -2.232339f, -0.232209f,
  2.336705f, -0.278834f, 0.231905f, 7.954366f,
};

static const float av1_partition_breakout_nn_weights_8_layer1[16] = {
  -0.014439f, 0.010171f, 0.048116f,  -0.090659f, -0.081235f, -0.021840f,
  -0.017360f, 0.031063f, -0.031737f, -0.023439f, -0.037725f, 0.021954f,
  0.055858f,  0.230970f, -0.056466f, 0.119780f,
};

static const float av1_partition_breakout_nn_bias_8_layer1[1] = {
  1.27784479f,
};

static const NN_CONFIG av1_partition_breakout_nnconfig_8 = {
  FEATURE_SIZE,  // num_inputs
  1,             // num_outputs
  1,             // num_hidden_layers
  {
      16,  // num_hidden_nodes
  },
  {
      av1_partition_breakout_nn_weights_8_layer0,
      av1_partition_breakout_nn_weights_8_layer1,
  },
  {
      av1_partition_breakout_nn_bias_8_layer0,
      av1_partition_breakout_nn_bias_8_layer1,
  },
};
#undef FEATURE_SIZE

#define FEATURE_SIZE 9  // Input layer size
#define NUM_NODES 32    // Hidden layer size
#define LABEL_SIZE 3    // Output layer size

static const float av1_rect_partition_nn_weights_8_layer0[FEATURE_SIZE *
                                                          NUM_NODES] = {
  0.22151f,  0.99424f,  0.23415f,  -1.13841f, -0.11277f, 0.09530f,  0.14769f,
  -1.18895f, -0.96640f, -0.21421f, -0.13974f, 0.03236f,  0.15777f,  -0.03176f,
  0.02729f,  -0.37344f, -0.01727f, -0.05469f, 0.19402f,  -3.45508f, 0.90106f,
  -2.91557f, 0.19379f,  0.14356f,  -0.13291f, 0.05734f,  -0.03032f, -0.13060f,
  0.35744f,  1.31630f,  -1.54493f, -0.20749f, -0.24413f, -0.04524f, -0.12400f,
  1.08305f,  -0.21596f, 0.76244f,  1.10616f,  -1.71706f, 0.05768f,  0.10966f,
  0.00949f,  -0.12680f, 0.00699f,  -0.11522f, -0.38566f, 0.34283f,  -0.35266f,
  -0.40643f, -0.22462f, 0.32300f,  -0.39737f, -0.20587f, -0.16096f, 1.07543f,
  0.30314f,  -1.35659f, -0.38212f, 0.45857f,  0.76615f,  0.16819f,  -1.24459f,
  0.39677f,  0.87436f,  -2.33757f, 1.27471f,  0.27488f,  0.01019f,  -0.01221f,
  -0.07461f, -0.14577f, -0.01231f, -0.64426f, -1.02733f, -1.96242f, 0.95143f,
  -0.06777f, -1.13868f, 0.01354f,  -0.75590f, -0.78222f, -0.07453f, 0.61788f,
  0.56899f,  1.17144f,  0.70899f,  0.48568f,  0.11266f,  0.81579f,  -0.03929f,
  0.01088f,  0.33599f,  -0.22401f, -0.49654f, -0.02598f, 0.04509f,  -0.08217f,
  -0.30687f, 0.19851f,  -2.96860f, -2.30698f, 0.01848f,  0.11801f,  0.06614f,
  0.01673f,  -0.11002f, -0.08168f, 0.09204f,  -0.06379f, 0.27972f,  -0.31716f,
  -0.00566f, -0.13651f, -0.37276f, 0.01511f,  -0.23697f, 0.21696f,  -0.19480f,
  0.60758f,  -0.43506f, -0.02247f, -1.45073f, 0.84442f,  -0.94018f, 0.32550f,
  0.03985f,  -0.06581f, 0.21665f,  0.79472f,  -2.41080f, 0.04788f,  -0.09492f,
  -0.10677f, 0.07250f,  0.14329f,  -0.37319f, 0.53043f,  -0.49108f, 0.25792f,
  -0.36569f, -0.28669f, -0.18416f, -0.52385f, -1.17081f, -1.32153f, -1.13403f,
  -0.26196f, 0.93379f,  0.72115f,  0.54464f,  0.27642f,  0.04757f,  2.01629f,
  1.55787f,  -0.11665f, 1.00722f,  -0.24352f, 0.53308f,  0.57719f,  0.39344f,
  0.19174f,  0.06339f,  -0.02530f, 0.07724f,  -0.32416f, -0.26992f, -0.35887f,
  -0.35285f, -0.33379f, -0.37475f, -0.77335f, 1.70027f,  -1.52153f, -0.26503f,
  0.97552f,  -2.96705f, -0.91220f, -0.11827f, 0.00406f,  -0.14514f, 0.18417f,
  -0.20874f, 0.27293f,  -0.34072f, -0.34838f, -0.19054f, -0.29806f, -0.27960f,
  -0.19293f, -0.18275f, -0.05902f, 0.58625f,  -0.05470f, -0.48814f, -0.45382f,
  -0.05959f, 2.01250f,  -0.30014f, 0.69546f,  -1.24180f, 1.34923f,  0.20337f,
  0.16850f,  0.07187f,  0.72630f,  -0.15380f, -2.40973f, -2.73561f, -1.71375f,
  -1.61695f, 0.50052f,  0.09730f,  0.00579f,  0.06133f,  -0.06512f, -0.61439f,
  -1.16173f, -0.58716f, 1.60438f,  0.23242f,  0.91847f,  0.49041f,  -0.16277f,
  -0.02574f, -0.64593f, 1.17028f,  0.46852f,  0.14926f,  0.73853f,  -0.78521f,
  0.05959f,  -0.35590f, 0.02039f,  0.10812f,  -0.28650f, 1.34038f,  -0.72188f,
  0.62385f,  -0.35271f, -0.39599f, 0.41543f,  0.53124f,  -0.23510f, -0.15480f,
  -0.05066f, -0.33529f, 0.05238f,  -0.35311f, -0.26983f, -0.39764f, 0.01085f,
  0.26593f,  -0.18411f, -0.29945f, 0.50090f,  -0.03397f, 0.78562f,  -0.33068f,
  1.21308f,  -2.23273f, -0.33366f, -0.15164f, -1.13270f, 0.17394f,  0.65567f,
  0.76496f,  0.44325f,  0.01368f,  -0.33619f, -0.64256f, 0.64478f,  0.84553f,
  1.74183f,  0.22563f,  -0.14550f, -0.16258f, 0.03010f,  0.49922f,  0.64575f,
  -0.29187f, -0.10348f, -1.43619f, -0.56540f, -0.14779f, 0.04616f,  0.87411f,
  -1.08228f,
};

static const float av1_rect_partition_nn_bias_8_layer0[NUM_NODES] = {
  0.33919f,  -0.03003f, 0.79073f,  -0.18508f, 0.00668f,  -0.12017f, 0.35362f,
  -0.51642f, 0.06536f,  0.41668f,  -0.06509f, 0.94606f,  -0.15385f, 0.14936f,
  1.46274f,  -0.06961f, 2.82537f,  -1.95576f, -0.09457f, 0.02042f,  -0.07480f,
  -0.55083f, 0.26170f,  4.39883f,  0.33999f,  -0.10502f, 0.70884f,  -0.06992f,
  -0.22638f, 1.40940f,  -0.09309f, 0.05828f,
};

static const float av1_rect_partition_nn_weights_8_layer1[NUM_NODES *
                                                          LABEL_SIZE] = {
  0.09209f,  0.26236f,  0.62136f,  0.76324f,  -1.14678f, 0.42289f,  -0.08895f,
  -0.97267f, 2.05958f,  0.00843f,  0.35335f,  1.12096f,  -0.11679f, 0.07350f,
  -1.23231f, -0.61990f, 1.51379f,  -1.99450f, 0.22441f,  2.41974f,  -0.30488f,
  -0.37869f, 0.47168f,  -3.70132f, 0.00061f,  0.19432f,  0.11512f,  0.26200f,
  -0.35285f, 0.37985f,  0.90571f,  0.27344f,  0.74840f,  -0.17965f, -2.51433f,
  0.59235f,  1.16670f,  -0.53446f, 0.67897f,  0.04505f,  -0.86874f, 0.45361f,
  -0.35033f, 1.21283f,  0.31426f,  -0.20841f, 0.56757f,  0.45909f,  -1.23683f,
  0.09835f,  -0.17214f, -0.96323f, 0.01138f,  -0.50233f, 0.30104f,  2.01814f,
  1.15821f,  -0.11947f, 0.74574f,  -0.30714f, -0.39646f, -1.30086f, -0.88541f,
  -0.12259f, -0.54977f, 0.30069f,  1.84299f,  -0.95141f, -0.65887f, -0.25888f,
  -0.63265f, 1.29531f,  -0.56672f, 0.10837f,  -0.21297f, -2.19131f, 0.01156f,
  0.51912f,  0.46704f,  0.42810f,  -0.59271f, 0.98469f,  -0.17914f, -1.91163f,
  -0.32807f, 0.48199f,  -0.99525f, 1.67108f,  -0.87631f, -0.60258f, -0.78731f,
  -0.32877f, 0.44237f,  0.01087f,  0.07489f,  -0.28224f,
};

static const float av1_rect_partition_nn_bias_8_layer1[LABEL_SIZE] = {
  1.70665f,
  -0.77954f,
  -0.92709f,
};

static const NN_CONFIG av1_rect_partition_nnconfig_8 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      NUM_NODES,
  },  // num_hidden_nodes
  { av1_rect_partition_nn_weights_8_layer0,
    av1_rect_partition_nn_weights_8_layer1 },
  { av1_rect_partition_nn_bias_8_layer0, av1_rect_partition_nn_bias_8_layer1 }
};

static const float av1_rect_partition_nn_weights_16_layer0[FEATURE_SIZE *
                                                           NUM_NODES] = {
  -0.18480f, -0.05410f, -0.18957f, 0.15451f,  -0.38649f, -0.26162f, -0.22727f,
  -0.38555f, -0.36738f, 0.74384f,  -1.85999f, 0.98491f,  -0.72119f, 1.77321f,
  0.39983f,  0.96314f,  0.23695f,  0.30200f,  0.30629f,  -0.47617f, -1.43320f,
  -1.81730f, 0.36554f,  -0.07142f, -1.27242f, -1.27697f, 0.00110f,  -0.32179f,
  0.27460f,  0.45428f,  0.15308f,  -0.73906f, -0.28577f, -0.01238f, -0.16958f,
  -0.85390f, 1.05484f,  -1.62812f, 0.77632f,  -0.27327f, -0.32527f, 0.32726f,
  1.73255f,  0.53763f,  0.59121f,  -0.39068f, -0.32451f, -0.31869f, 0.17777f,
  0.07519f,  -0.18066f, -0.11250f, -0.14616f, -0.16882f, -0.04099f, -0.67959f,
  0.39674f,  -0.08596f, 0.18587f,  -2.04097f, -1.73993f, 1.57212f,  1.42410f,
  -1.36762f, -0.41485f, -1.12103f, 0.56959f,  0.11500f,  0.48945f,  -0.13585f,
  1.22125f,  0.67071f,  -1.11812f, -0.20660f, -0.52856f, 0.70663f,  0.74382f,
  0.61114f,  -0.11454f, 1.14687f,  0.80322f,  -0.45965f, -0.44466f, -0.05830f,
  0.13206f,  -0.53750f, -0.11324f, -0.37971f, -0.13491f, -0.21268f, 1.93407f,
  1.34433f,  2.49427f,  2.91955f,  1.71730f,  0.03295f,  0.03587f,  -0.14550f,
  0.08189f,  -0.38655f, -0.35432f, -0.62706f, -0.01849f, -0.57882f, -0.60438f,
  -1.01334f, -0.57302f, 0.22592f,  0.05916f,  -0.05305f, -0.89824f, -0.52969f,
  -0.24542f, 0.27029f,  -0.40924f, -0.82452f, -0.60665f, -5.03025f, 0.83302f,
  1.83695f,  2.19716f,  2.31001f,  0.03657f,  0.00063f,  -0.04379f, 0.05835f,
  -0.08623f, 0.20557f,  -0.17791f, 0.07874f,  -0.25456f, -0.19513f, -0.27753f,
  -0.31982f, 0.00245f,  -0.33183f, 0.26059f,  -0.22165f, 0.37582f,  -0.30411f,
  -0.22639f, -0.14739f, -0.20201f, -0.37507f, -1.30653f, 0.49570f,  1.03673f,
  0.66139f,  0.44941f,  -0.44461f, -0.50376f, -0.49664f, 0.18608f,  -0.26175f,
  0.14844f,  0.78715f,  -0.70344f, -0.87624f, -0.98535f, -0.35346f, 0.37094f,
  -0.43135f, -0.22571f, 3.46263f,  3.13580f,  -1.33203f, -0.15247f, -0.15866f,
  -0.11214f, 0.12211f,  0.03964f,  -1.87597f, -4.81597f, -4.80195f, -4.98096f,
  -5.62336f, -0.05337f, -0.00943f, 0.00792f,  0.02742f,  1.05679f,  2.41455f,
  0.85382f,  1.42504f,  0.58096f,  0.21443f,  1.02694f,  1.06746f,  1.20242f,
  0.60767f,  1.98667f,  -0.80879f, -0.63495f, 1.95508f,  0.23952f,  -0.15019f,
  -0.16097f, 0.30155f,  -3.42407f, -1.34998f, 9.07689f,  -2.22559f, 2.22562f,
  -0.03348f, -0.05229f, 0.05931f,  0.03042f,  -0.18068f, -0.05732f, -0.33010f,
  -0.32279f, -0.26607f, -0.02723f, -0.04067f, 0.08700f,  -0.16366f, -0.24935f,
  -0.69124f, 0.58508f,  0.50654f,  0.04492f,  1.38340f,  -1.51487f, 1.72889f,
  -1.95618f, -3.65013f, -1.38525f, -3.05516f, -2.40448f, 2.47467f,  0.03784f,
  0.08052f,  -0.01971f, -0.08918f, -0.84997f, -0.55302f, -1.07861f, -0.62626f,
  0.61751f,  -0.11012f, -0.24185f, -0.39201f, -1.85390f, -0.31261f, -0.11927f,
  0.15671f,  -0.23450f, -0.14916f, -0.31715f, -0.19350f, 0.01795f,  -0.11533f,
  -0.05799f, -0.03142f, 0.20218f,  -0.39499f, -0.33859f, -0.13201f, -0.19527f,
  -0.28459f, -0.20346f, 0.89457f,  -2.22103f, -2.37455f, -2.00221f, 2.44553f,
  0.33915f,  0.50047f,  -0.34625f, -0.19667f, -0.56333f, -0.84328f, 1.25767f,
  -1.70297f, 1.00482f,  -0.00103f, -1.40813f, 0.21311f,  0.39230f,  -0.07302f,
  -3.49100f, 1.60675f,  -2.90692f, 0.11022f,  0.13507f,  -0.13308f, 0.15201f,
  -0.05573f,
};

static const float av1_rect_partition_nn_bias_16_layer0[NUM_NODES] = {
  -0.16783f, -0.16023f, 0.52215f,  -0.04109f, 2.00122f,  -0.11633f, 0.25535f,
  1.80638f,  1.69273f,  -0.25998f, -6.83550f, -0.79682f, -1.03466f, 1.42721f,
  0.00000f,  -0.00000f, -0.11665f, -0.12047f, -1.01497f, 7.27181f,  -0.78548f,
  -1.39335f, -5.42248f, -0.10388f, 0.07634f,  2.81012f,  -0.57429f, -0.15629f,
  -0.12044f, 1.65478f,  -0.75153f, 1.18441f,
};

static const float av1_rect_partition_nn_weights_16_layer1[NUM_NODES *
                                                           LABEL_SIZE] = {
  -0.26407f, 0.06322f,  0.87932f,  0.17772f,  0.71686f,  -0.12283f, 0.08454f,
  0.20098f,  -0.31763f, -0.33178f, -4.59535f, -0.04367f, 0.17099f,  3.80486f,
  0.16750f,  0.29218f,  0.57234f,  -0.96550f, -0.10599f, -4.91130f, -0.14658f,
  0.95803f,  -4.13925f, 0.24567f,  0.25708f,  1.60547f,  -1.03251f, -0.31053f,
  -0.05659f, -0.94121f, -0.68926f, -0.24738f, -0.38019f, 0.98950f,  0.13689f,
  0.24504f,  0.49623f,  0.19980f,  0.38349f,  0.37481f,  0.54540f,  -0.02198f,
  3.43385f,  1.02543f,  -0.40921f, -3.07235f, 0.02996f,  0.00323f,  -0.35414f,
  0.71099f,  1.39334f,  2.43741f,  -1.11007f, -0.22739f, -4.21757f, 0.11905f,
  0.00353f,  -1.69637f, 0.45944f,  -0.19884f, 0.03624f,  0.25729f,  0.23659f,
  -2.08405f, 0.08573f,  -0.53393f, -1.28103f, -0.53970f, -0.65465f, 0.31821f,
  -0.09884f, -0.69026f, -0.37284f, 0.04622f,  1.32973f,  -0.15414f, 0.19138f,
  -0.67927f, -0.17658f, 0.36008f,  -0.51832f, 0.09887f,  -1.94414f, 2.95227f,
  1.76937f,  -0.26687f, 8.50976f,  0.26247f,  0.60262f,  -0.27910f, 0.30061f,
  -0.05117f, 0.16018f,  0.71195f,  0.57871f,  1.57794f,
};

static const float av1_rect_partition_nn_bias_16_layer1[3] = {
  2.68750f,
  -1.31894f,
  -1.36768f,
};

static const NN_CONFIG av1_rect_partition_nnconfig_16 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      NUM_NODES,
  },  // num_hidden_nodes
  { av1_rect_partition_nn_weights_16_layer0,
    av1_rect_partition_nn_weights_16_layer1 },
  { av1_rect_partition_nn_bias_16_layer0, av1_rect_partition_nn_bias_16_layer1 }
};

static const float av1_rect_partition_nn_weights_32_layer0[FEATURE_SIZE *
                                                           NUM_NODES] = {
  -0.54654f, -0.43537f, -0.10620f, -0.48051f, -0.43543f, -0.22737f, -0.15429f,
  -0.09858f, -0.09438f, 0.37306f,  0.23934f,  -1.86375f, -1.18307f, -0.32995f,
  -0.09745f, 0.05431f,  -0.13799f, 0.14734f,  -0.33219f, 0.18057f,  -0.23792f,
  -0.28126f, 0.02977f,  -0.07431f, 0.07860f,  0.00067f,  -0.01927f, 1.01841f,
  -0.57739f, 0.08412f,  -1.33843f, -1.05563f, -0.28693f, -0.39425f, -0.69572f,
  -0.16703f, 0.02808f,  0.11994f,  -0.26267f, 0.19706f,  -0.29707f, -0.25305f,
  -0.07050f, -0.02704f, -0.31528f, -0.42301f, 0.22496f,  -0.37001f, -0.23319f,
  -0.11139f, -0.30513f, 0.04213f,  -0.12550f, 0.02504f,  0.33245f,  0.01102f,
  -0.35950f, -0.05949f, -0.19590f, -0.27457f, -0.28339f, -0.15676f, -0.21538f,
  0.65066f,  0.28443f,  -1.24943f, -3.00246f, -1.01897f, 0.09304f,  0.70052f,
  -0.12877f, 0.21120f,  -0.37476f, 0.23261f,  -0.28401f, 0.09837f,  0.00020f,
  -0.12106f, -0.32354f, -0.02472f, -0.19772f, 1.01886f,  0.16596f,  -0.06532f,
  1.72938f,  1.57754f,  0.55963f,  0.33246f,  -0.20023f, 0.30715f,  0.08629f,
  0.18945f,  -0.45988f, -1.22610f, -0.05152f, -0.48859f, -1.02104f, -0.27315f,
  -0.57698f, 0.04157f,  -0.92428f, -1.31268f, 1.78210f,  0.10291f,  1.55042f,
  -1.26793f, 1.39042f,  -1.43729f, 0.25600f,  5.21263f,  5.31955f,  5.19316f,
  5.43430f,  0.00294f,  -0.00970f, -0.02333f, 0.00250f,  1.17672f,  6.27544f,
  4.95973f,  3.54009f,  4.51269f,  0.30750f,  0.78780f,  -0.44741f, -0.76442f,
  0.75050f,  0.58799f,  0.03400f,  -2.09859f, 1.67313f,  0.12503f,  0.28609f,
  1.15809f,  2.46530f,  -0.04898f, 0.23072f,  -0.12635f, -0.82097f, -0.63827f,
  2.16779f,  1.77132f,  0.15434f,  -1.06427f, 0.06206f,  -0.87732f, -0.61897f,
  -0.44593f, -0.77131f, -0.15979f, -0.02282f, -0.74381f, 0.66052f,  -0.22992f,
  1.74638f,  1.29199f,  -0.55464f, 0.98316f,  0.06665f,  0.50254f,  -0.66292f,
  0.17113f,  -0.32633f, -1.85803f, -0.92759f, 4.44965f,  1.33057f,  0.02135f,
  -0.27446f, -0.26018f, -0.12613f, -0.14470f, -0.23355f, -0.09717f, -0.24123f,
  -0.05535f, -0.19146f, -0.36222f, -0.30458f, -0.40323f, 0.21779f,  0.14248f,
  -0.48630f, 0.18840f,  0.11040f,  0.17287f,  -0.51880f, 1.12466f,  -0.38888f,
  -0.16421f, -0.31784f, -0.36112f, -0.25386f, -0.01636f, 0.10029f,  -0.26881f,
  -0.17051f, -0.30903f, -0.08573f, -0.28774f, -0.01173f, -0.09706f, -0.23089f,
  -0.12922f, -0.17463f, -0.12433f, -0.23074f, 0.15220f,  1.29826f,  0.23788f,
  0.04189f,  2.66416f,  0.48815f,  -0.06803f, 0.96742f,  1.27165f,  -0.70348f,
  -0.09941f, -0.42948f, -0.20243f, -0.02364f, -0.26689f, -0.40629f, -0.68217f,
  -0.48073f, 2.43657f,  -2.60191f, -1.82837f, 0.50440f,  0.71829f,  0.76491f,
  0.28293f,  0.20568f,  0.92642f,  -0.02496f, 1.43637f,  -0.24474f, -1.21030f,
  0.54084f,  1.05130f,  1.29572f,  0.03750f,  -0.36894f, 0.74548f,  -1.33857f,
  -0.84858f, 1.35230f,  0.80175f,  0.66136f,  1.06473f,  0.18701f,  1.42413f,
  0.04661f,  -0.07820f, 0.64990f,  -0.43595f, 1.18304f,  -0.11437f, -0.06365f,
  0.03558f,  0.78260f,  -1.74890f, 1.56217f,  -1.23424f, 4.59193f,  -3.35072f,
  0.01180f,  -0.18296f, -0.20870f, 0.04510f,  1.52595f,  -1.37402f, -0.33123f,
  -0.85957f, 0.80598f,  0.03743f,  0.02354f,  0.37707f,  1.62095f,  -0.29627f,
  -0.31778f, -0.45789f, -0.14906f, 0.25315f,  -0.10817f, -0.32610f, -0.40890f,
  0.33984f,
};

static const float av1_rect_partition_nn_bias_32_layer0[NUM_NODES] = {
  -0.17482f, 0.39042f,  0.00000f,  1.69677f,  0.08792f,  -0.09301f, 0.13809f,
  4.84061f,  0.00000f,  0.40515f,  0.46246f,  0.20644f,  -5.77478f, -1.54510f,
  0.05660f,  -0.32013f, 0.23649f,  0.03778f,  -2.53710f, -0.27869f, 0.45623f,
  -0.04155f, -0.18445f, -0.73405f, -0.50243f, 2.23191f,  1.93272f,  -1.07032f,
  -0.27602f, -1.98063f, 0.20816f,  -0.01315f,
};

static const float av1_rect_partition_nn_weights_32_layer1[NUM_NODES *
                                                           LABEL_SIZE] = {
  0.02827f,  1.02560f,  -0.07137f, -0.31911f, 0.11365f,  0.13684f,  -0.07816f,
  -5.23036f, -0.34340f, 0.84526f,  -1.51845f, 0.07017f,  -8.12570f, 6.24061f,
  0.35739f,  -0.09937f, -0.30978f, 0.22032f,  0.74968f,  -0.34557f, 0.45547f,
  -0.16512f, 0.07118f,  1.66415f,  0.41320f,  -1.81533f, -1.96004f, 1.04666f,
  0.84049f,  4.31009f,  0.68850f,  0.26322f,  -0.24634f, -1.25889f, 0.31952f,
  0.63632f,  0.05801f,  -0.10664f, -0.21992f, 2.44386f,  0.19526f,  -0.09838f,
  1.53049f,  -0.26630f, 3.54126f,  -3.40574f, 0.72730f,  0.04557f,  0.92652f,
  0.15522f,  2.35895f,  -0.13347f, 0.56907f,  0.15352f,  0.01823f,  -0.73939f,
  0.43104f,  1.90321f,  0.31267f,  -0.51972f, 0.50094f,  -3.98372f, -3.41518f,
  -0.48183f, 0.26661f,  0.64146f,  0.14500f,  -0.01695f, 0.16653f,  -0.37846f,
  0.08412f,  2.69714f,  -0.20258f, -0.75786f, 0.11201f,  0.61878f,  4.22231f,
  -3.55330f, -1.14137f, -0.37722f, -0.28000f, -0.72581f, -2.62827f, -0.19448f,
  -0.59398f, -0.30136f, -0.17725f, -0.69630f, -0.41132f, 0.12208f,  2.11441f,
  -1.08794f, -1.41694f, 0.02620f,  2.18792f,  0.04271f,
};

static const float av1_rect_partition_nn_bias_32_layer1[3] = {
  2.47332f,
  -1.65756f,
  -0.81573f,
};

static const NN_CONFIG av1_rect_partition_nnconfig_32 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      NUM_NODES,
  },  // num_hidden_nodes
  { av1_rect_partition_nn_weights_32_layer0,
    av1_rect_partition_nn_weights_32_layer1 },
  { av1_rect_partition_nn_bias_32_layer0, av1_rect_partition_nn_bias_32_layer1 }
};

static const float av1_rect_partition_nn_weights_64_layer0[FEATURE_SIZE *
                                                           NUM_NODES] = {
  0.08972f,  4.09095f,  -0.31398f, -2.43631f, -0.74767f, 1.42471f,  1.60926f,
  1.44721f,  1.88259f,  2.35375f,  1.88299f,  2.01109f,  0.98679f,  2.24131f,
  0.06279f,  -0.08315f, 0.32107f,  0.91334f,  -0.36569f, 5.55049f,  5.44943f,
  5.20471f,  5.39099f,  -0.01943f, -0.00284f, 0.02203f,  -0.01309f, 1.41917f,
  6.68460f,  -6.15986f, 6.41341f,  -3.20630f, -0.00567f, -0.00038f, 0.05960f,
  0.04308f,  0.95366f,  3.48535f,  2.98266f,  4.11784f,  3.44255f,  0.61630f,
  0.71405f,  0.63945f,  -0.00713f, 0.39193f,  1.91621f,  3.32755f,  0.71674f,
  -0.11647f, 2.07090f,  2.64191f,  0.07949f,  -0.05023f, 0.99935f,  0.83145f,
  0.75898f,  -0.98764f, -0.58731f, 1.21734f,  -0.08076f, -3.26780f, 1.66278f,
  0.04189f,  -0.33177f, -1.58648f, 1.00883f,  -0.56132f, -2.34877f, 0.67056f,
  -2.32297f, -0.91641f, -1.02909f, 4.19781f,  3.87484f,  4.32778f,  -1.97171f,
  -0.24734f, 0.00822f,  0.05892f,  0.12697f,  -3.62915f, -2.93127f, 7.94856f,
  -3.29311f, 3.26001f,  -0.02231f, 0.02741f,  0.05919f,  0.08190f,  -1.49344f,
  -0.64475f, -0.24627f, 4.03324f,  -1.14799f, -0.18465f, -0.17829f, 0.10394f,
  0.08580f,  -5.74721f, 4.42467f,  3.63964f,  3.00258f,  -1.22744f, -0.29408f,
  0.00767f,  0.12305f,  0.05249f,  -0.17166f, -0.20120f, -0.32941f, -0.31901f,
  0.04628f,  -0.35249f, -0.18272f, 0.03956f,  -0.19329f, -0.33564f, 0.09856f,
  -0.00173f, -0.31751f, -0.05702f, -0.20558f, -0.31464f, -0.02488f, -0.00729f,
  -0.35854f, -0.14762f, -0.34897f, -0.12746f, 0.04011f,  -0.24918f, -0.53516f,
  -0.28440f, -0.36789f, -1.34889f, -9.10044f, -9.19238f, 4.48042f,  6.54429f,
  -0.00226f, 0.00430f,  0.00321f,  0.00442f,  0.87551f,  -0.16224f, -0.22832f,
  -0.60640f, -0.28738f, 0.18062f,  0.22008f,  -0.47406f, 0.80302f,  0.12149f,
  1.49530f,  1.05069f,  -2.02985f, -0.92833f, 0.25616f,  0.12852f,  3.51840f,
  0.25226f,  -2.63283f, -4.04386f, 8.46300f,  -2.93408f, 0.44069f,  0.08276f,
  0.34482f,  -0.22615f, 0.28666f,  3.02962f,  -1.20055f, -1.04832f, -0.97632f,
  -0.99530f, 1.44196f,  1.68550f,  0.49360f,  1.08155f,  -0.26059f, -0.02876f,
  -0.27492f, -0.06205f, -0.09496f, -0.12314f, -0.30228f, -0.07453f, -0.38857f,
  1.17443f,  2.41497f,  1.90537f,  2.37716f,  2.91495f,  -0.44455f, -0.51176f,
  0.48195f,  0.53032f,  0.23696f,  -1.06211f, 1.47459f,  -0.89029f, 0.29521f,
  0.66291f,  -0.42653f, 1.82308f,  -1.30372f, -0.36192f, -3.40388f, -1.61476f,
  -2.29745f, -0.66886f, -2.08252f, -0.54552f, -4.06849f, 0.02948f,  0.27297f,
  -4.81472f, 4.60404f,  -0.11053f, 0.14765f,  0.02826f,  -0.14688f, -0.07066f,
  -0.01224f, 1.20377f,  7.02725f,  -6.02627f, 6.87255f,  -3.14257f, 0.01074f,
  0.02397f,  -0.02359f, 0.01901f,  0.14956f,  -1.67671f, 2.26714f,  2.57043f,
  -0.45888f, -1.60265f, -2.11475f, -2.74029f, -2.74658f, -0.35630f, -2.63013f,
  -2.14814f, -0.67266f, -1.56850f, 0.57137f,  -1.14428f, -0.34265f, -0.12521f,
  0.01220f,  -0.74906f, -0.19270f, 0.68110f,  -0.24737f, -0.70568f, -1.64826f,
  -0.35847f, -0.15984f, -1.17932f, -8.72306f, -8.72834f, 3.93701f,  6.17812f,
  -0.03191f, -0.00104f, 0.01402f,  -0.00046f, -0.94517f, 1.51266f,  -0.56318f,
  0.72260f,  -0.09253f, -0.09069f, -2.16695f, -0.23653f, 0.24418f,  2.21148f,
  -1.47954f, -1.01439f, 0.31536f,  0.77238f,  -0.85083f, -0.15758f, -0.50886f,
  0.09101f,
};

static const float av1_rect_partition_nn_bias_64_layer0[NUM_NODES] = {
  0.91706f,  -1.31328f, -5.16196f, 1.13191f,  -0.98044f, -1.61122f, 1.03039f,
  -0.98537f, -4.45568f, -4.34802f, -0.92116f, 0.66836f,  -0.10752f, -0.13065f,
  -0.35567f, -0.35693f, 1.74941f,  1.17379f,  -3.45555f, 5.66321f,  -0.24917f,
  -1.11940f, -0.73656f, -0.19299f, -0.04181f, 1.11010f,  -2.97859f, -0.16774f,
  0.59835f,  -0.31269f, -0.30585f, -1.66212f,
};

static const float av1_rect_partition_nn_weights_64_layer1[NUM_NODES *
                                                           LABEL_SIZE] = {
  0.58963f,  4.20320f,  -8.62465f, -6.54014f, 5.41108f,  2.33581f,   -0.10354f,
  -1.17753f, -3.45909f, -2.24722f, 2.20881f,  3.21971f,  -0.09087f,  -0.21624f,
  0.16529f,  -8.40985f, -1.60205f, -1.41538f, 4.41826f,  -4.63069f,  -0.27742f,
  4.08710f,  0.26439f,  -1.46028f, 0.51234f,  6.25212f,  -3.35650f,  -1.21348f,
  1.37201f,  8.89151f,  0.28859f,  -0.97328f, -0.36196f, -2.71701f,  4.54196f,
  -0.62476f, -2.43814f, -1.34209f, 0.12850f,  1.73859f,  3.09809f,   -4.42434f,
  -1.82552f, -3.66420f, -0.31535f, 0.00968f,  -0.02019f, 9.66824f,   0.58835f,
  1.50425f,  2.84487f,  2.55522f,  0.01409f,  -2.27594f, -0.31800f,  0.91076f,
  -0.66808f, 0.33120f,  -0.12460f, 0.64457f,  -0.36416f, -10.30843f, 1.51013f,
  2.06861f,  -0.20989f, -0.87119f, 3.68642f,  7.33662f,  -2.88037f,  -0.52414f,
  -0.35036f, -0.45947f, -0.07406f, 6.46346f,  -0.16031f, 0.27071f,   0.38845f,
  -0.21940f, 0.08583f,  -1.39526f, 0.50554f,  0.45279f,  -6.61856f,  1.84069f,
  -0.19149f, -1.77235f, 0.75136f,  1.11797f,  0.32677f,  -7.10427f,  3.82908f,
  1.04238f,  -0.91435f, 1.93317f,  -1.84946f, -0.48909f,
};

static const float av1_rect_partition_nn_bias_64_layer1[3] = {
  0.32215f,
  -0.57522f,
  0.25314f,
};

static const NN_CONFIG av1_rect_partition_nnconfig_64 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      NUM_NODES,
  },  // num_hidden_nodes
  { av1_rect_partition_nn_weights_64_layer0,
    av1_rect_partition_nn_weights_64_layer1 },
  { av1_rect_partition_nn_bias_64_layer0, av1_rect_partition_nn_bias_64_layer1 }
};

static const float av1_rect_partition_nn_weights_128_layer0[FEATURE_SIZE *
                                                            NUM_NODES] = {
  -0.70901f, -3.03481f, 3.30604f,  -1.28803f,  -0.08610f, -0.33320f, -0.30716f,
  0.25100f,  0.14323f,  -0.98422f, -0.89084f,  -0.24508f, -1.10785f, -0.82524f,
  0.11766f,  -0.42777f, 1.08965f,  4.35125f,   -1.19388f, 4.22042f,  4.96306f,
  6.32406f,  3.29899f,  -0.90768f, 0.05203f,   0.38467f,  1.74257f,  -0.19918f,
  -0.11335f, 0.00140f,  -0.42303f, -0.04419f,  0.03583f,  -0.05441f, -0.19586f,
  0.01484f,  -1.19964f, 0.25497f,  3.04502f,   0.05446f,  -0.23253f, 0.00266f,
  0.07117f,  -2.78986f, -4.62953f, 1.45331f,   0.43923f,  0.92298f,  -0.47736f,
  1.49165f,  0.45942f,  -1.99787f, 3.33510f,   0.17234f,  0.04024f,  -1.42780f,
  0.23566f,  -0.90970f, 1.18041f,  -1.45865f,  2.30878f,  -1.28507f, 1.87290f,
  1.91186f,  4.74826f,  -3.70735f, 4.49808f,   -4.72275f, -0.02696f, -0.02642f,
  -0.06093f, -0.01121f, -0.70683f, 2.69737f,   -1.88563f, 2.48637f,  1.10922f,
  0.74624f,  0.40308f,  2.06396f,  1.39289f,   0.00909f,  -2.05271f, -1.53539f,
  -1.38323f, 0.83303f,  -0.32250f, 0.51172f,   3.91249f,  1.66373f,  1.13184f,
  -2.22874f, -1.13448f, -0.11185f, 0.19387f,   0.36770f,  -0.58933f, 0.22789f,
  1.17307f,  0.77461f,  0.20817f,  0.33417f,   0.54037f,  0.32961f,  -0.18456f,
  -9.78171f, -0.17216f, -3.44703f, -2.42158f,  0.51946f,  4.35949f,  -0.73335f,
  -1.61515f, -0.29622f, -0.37617f, -0.42316f,  0.74922f,  1.44386f,  3.92704f,
  -3.76274f, 4.19775f,  -3.86958f, 0.00074f,   -0.02418f, -0.12944f, 0.05857f,
  -0.85507f, 5.42546f,  5.40338f,  5.54347f,   5.59791f,  -0.01611f, 0.01618f,
  -0.01654f, -0.00270f, -0.39608f, -0.40410f,  -0.24551f, 0.09124f,  -0.34413f,
  -0.11504f, 0.12793f,  -0.31523f, 0.09148f,   -0.08567f, -0.05140f, -0.13310f,
  -0.81200f, 0.06882f,  -0.52537f, -12.74048f, -0.45395f, -4.04775f, -1.84887f,
  -1.02573f, 0.32788f,  1.06828f,  -1.25503f,  -0.42693f, 2.01413f,  -2.29103f,
  0.62271f,  1.11764f,  -1.83113f, -1.32325f,  -1.65651f, -2.87826f, 1.46910f,
  0.60885f,  0.16079f,  0.00171f,  -0.25658f,  -0.25465f, -0.14149f, 0.19497f,
  -0.07866f, -0.37080f, -0.05778f, -0.08870f,  -0.20491f, 0.84521f,  -0.18214f,
  -1.38441f, -1.08932f, -1.76627f, 0.73172f,   0.05967f,  1.28057f,  3.42722f,
  1.69287f,  0.77169f,  0.44528f,  1.85513f,   0.07840f,  1.31252f,  2.89948f,
  1.49489f,  0.15281f,  0.54708f,  -1.14185f,  -2.51063f, 0.36618f,  -0.55322f,
  0.96671f,  1.59470f,  1.38252f,  1.99697f,   0.03266f,  -0.23200f, -0.01127f,
  -0.18918f, -0.37598f, -0.03119f, -0.36039f,  -0.21192f, -0.11565f, -4.22635f,
  1.41252f,  0.56608f,  -0.08867f, 3.11924f,   -0.54597f, -0.12504f, -0.05289f,
  -0.28665f, -0.58297f, -1.18362f, -0.76201f,  -1.22011f, -0.58756f, 0.14740f,
  1.43971f,  0.98381f,  -0.02998f, -0.40678f,  -0.23047f, -0.12979f, 0.04003f,
  -0.22081f, -0.09294f, -0.15955f, -0.10379f,  -0.10192f, -1.51316f, 2.39482f,
  -1.69975f, 3.58976f,  -0.91032f, -0.03498f,  0.48982f,  -0.13418f, 0.76256f,
  1.61003f,  -2.01676f, -1.24430f, -3.25763f,  1.12314f,  2.00740f,  0.04613f,
  -0.14746f, -0.57374f, 3.44511f,  -0.56767f,  -4.08432f, -2.04894f, 2.35951f,
  -0.00458f, 0.18512f,  0.09916f,  -0.04084f,  -1.56207f, 1.38034f,  4.17302f,
  -1.47326f, -2.03530f, -0.00210f, 0.27469f,   -0.17423f, 0.86860f,  2.76195f,
  2.43269f,  -3.57331f, 2.08715f,  -1.44171f,  -0.17389f, 2.26157f,  -0.07852f,
  2.02519f,
};

static const float av1_rect_partition_nn_bias_128_layer0[NUM_NODES] = {
  2.53427f,  1.66678f,  -0.84914f, -0.15070f, -1.74769f, 0.45218f,  -0.26067f,
  2.05916f,  0.08978f,  5.30984f,  2.66243f,  -1.62740f, 0.70018f,  1.96403f,
  -4.97152f, -0.05425f, -3.84474f, -1.28006f, 3.47490f,  -0.08373f, 0.00225f,
  -1.40692f, -0.27569f, -0.30253f, 0.77377f,  -0.67636f, -0.26379f, 1.82348f,
  0.66120f,  0.61119f,  -1.42293f, 0.32676f,
};

static const float av1_rect_partition_nn_weights_128_layer1[NUM_NODES *
                                                            LABEL_SIZE] = {
  1.53453f,  -0.23707f, 7.88368f,  0.33340f,  0.97523f,  1.38538f,  -0.16746f,
  4.42070f,  3.18678f,  -5.03545f, -2.27029f, -3.75719f, -0.26850f, -4.93432f,
  -8.75673f, 0.27398f,  -5.77882f, -0.91616f, -2.62725f, -0.23961f, 0.31249f,
  3.32134f,  0.25375f,  -0.00394f, 2.30213f,  -0.14183f, 0.14544f,  -1.42830f,
  1.31101f,  3.99389f,  -0.00017f, -2.90184f, -2.11444f, 2.16734f,  -3.05133f,
  0.39206f,  4.61489f,  -2.88181f, -0.47745f, 2.86649f,  -1.20621f, 3.70550f,
  1.58029f,  -4.58731f, -2.29350f, -0.76930f, 5.19135f,  -0.22521f, -5.08782f,
  2.17316f,  1.30563f,  0.16777f,  -2.17767f, -2.09904f, 1.37001f,  0.25091f,
  -1.76743f, 1.57940f,  0.30544f,  -2.39895f, -0.08532f, -1.77122f, 1.84010f,
  -0.88449f, 0.79299f,  -1.35368f, -4.54110f, 0.02244f,  -5.11580f, 1.60883f,
  0.29352f,  -6.47042f, -1.81426f, 1.24013f,  0.90980f,  7.93977f,  2.12555f,
  5.24720f,  4.19508f,  0.21499f,  11.06045f, -0.74752f, 0.89396f,  0.26422f,
  1.72332f,  -1.25113f, -1.71136f, 0.13676f,  -0.07867f, -0.96929f, 0.19911f,
  3.58233f,  -0.76470f, -2.24162f, -2.87465f, 3.18736f,
};

static const float av1_rect_partition_nn_bias_128_layer1[3] = {
  1.09014f,
  -0.53317f,
  -0.55668f,
};

static const NN_CONFIG av1_rect_partition_nnconfig_128 = {
  FEATURE_SIZE,  // num_inputs
  LABEL_SIZE,    // num_outputs
  1,             // num_hidden_layers
  {
      NUM_NODES,
  },  // num_hidden_nodes
  { av1_rect_partition_nn_weights_128_layer0,
    av1_rect_partition_nn_weights_128_layer1 },
  { av1_rect_partition_nn_bias_128_layer0,
    av1_rect_partition_nn_bias_128_layer1 }
};
#undef FEATURE_SIZE
#undef NUM_NODES
#undef LABEL_SIZE

// Below are the models used for simple_motion_search_based_split
// Thresholds
// The first index level is for aggresiveness, and the second is frame
// resolution, third is bsize
static const float av1_simple_motion_search_split_thresh[4][3][5] = {
  // Aggressiveness = 0
  {
      // lowres
      {
          1.40402595879f,  // p = 0.8028197
          4.72845183649f,  // p = 0.99123732
          1.86517797783f,  // p = 0.86589934
          1.58715223005f,  // p = 0.83021506
          7.22695596987f,  // p = 0.9992738
      },
      // midres
      {
          5.839480f,  // p = 0.997098
          1.877167f,  // p = 0.867285
          3.073499f,  // p = 0.955783
          1.405601f,  // p = 0.803071
          2.555636f,  // p = 0.927951
      },
      // hdres
      {
          5.839480f,  // p = 0.997098
          1.877167f,  // p = 0.867285
          3.073499f,  // p = 0.955783
          1.405601f,  // p = 0.803071
          2.555636f,  // p = 0.927951
      },
  },
  // Aggressiveness = 1
  {
      // Lowres
      {
          100.0000f,  // p = 1.000000
          4.952535f,  // p = 0.992984
          1.720880f,  // p = 0.848242
          1.426233f,  // p = 0.806314
          1.491905f,  // p = 0.816364
      },
      // Midres
      {
          100.0000f,  // p = 100.0000
          3.137263f,  // p = 0.958404
          2.703262f,  // p = 0.937219
          1.877166f,  // p = 0.867285
          2.221149f,  // p = 0.902133
      },
      // Hdres
      {
          4.417680f,  // p = 0.988082
          3.086898f,  // p = 0.956349
          3.966704f,  // p = 0.981416
          1.532565f,  // p = 0.822381
          3.449975f,  // p = 0.969230
      },
  },
  // Aggressiveness = 2
  {
      // lowres
      {
          100.000000f,  // p = 0.998048
          1.484020f,    // p = 0.815179
          1.866781f,    // p = 0.866085
          1.706711f,    // p = 0.846409
          2.080369f,    // p = 0.888980
      },
      // midres
      {
          100.000000f,  // p = 0.0
          3.265763f,    // p = 0.963235428881
          2.024598f,    // p = 0.883355591569
          1.846446f,    // p = 0.863709256976
          2.240962f,    // p = 0.903868036126
      },
      // hdres
      {
          3.133026f,  // p = 0.958234684141
          2.940954f,  // p = 0.949834204693
          2.484544f,  // p = 0.923051170045
          1.702972f,  // p = 0.845922460525
          1.655562f,  // p = 0.839641385729
      },
  },
  // Aggressiveness = 3
  {
      // lowres
      { 100.000000f, 1.41409519484f, 0.606066095487f, 0.0993410805635f,
        0.762099214988f },
      // midres
      { 100.000000f, 0.702207995397f, 0.503550081119f, 0.0403228785199f,
        0.557298794638f },
      // hdres
      { 1.21895384144f, 1.26798450469f, 0.872537808115f, 0.975869438148f,
        1.86572095242f },
  },
};

static const float av1_simple_motion_search_no_split_thresh[4][3][5] = {
  // Aggressiveness = 0
  {
      // lowres
      {
          -100.0f,  // p = 0.0
          -100.0f,  // p = 0.0
          -100.0f,  // p = 0.0
          -100.0f,  // p = 0.0
          -100.0f,  // p = 0.0
      },
      // midres
      {
          -3.38168078f,  // p = 0.032872917
          -4.08610739f,  // p = 0.016526795
          -1.78302370f,  // p = 0.15270848
          -100.000000f,  // p = 0.0
          -100.000000f,  // p = 0.0
      },
      // hdres
      {
          -100.000000f,  // p = 0.0
          -100.000000f,  // p = 0.0
          -2.98718897f,  // p = 0.048008
          -100.000000f,  // p = 0.0
          -3.33229488f,  // p = 0.03447975
      },
  },
  // Aggressiveness = 1
  {
      // Lowres
      {
          -100.0000f,  // p = 0.0
          -4.893793f,  // p = 0.007437
          -3.387766f,  // p = 0.032680
          -2.982806f,  // p = 0.048209
          -2.330372f,  // p = 0.088639
      },
      // Midres
      {
          -100.0000f,  // p = 0.000000
          -6.131853f,  // p = 0.002168
          -2.346579f,  // p = 0.087338
          -2.712849f,  // p = 0.062219
          -3.195430f,  // p = 0.039338
      },
      // Hdres
      {
          -3.491416f,  // p = 0.029557
          -2.192853f,  // p = 0.100394
          -3.620180f,  // p = 0.026079
          -2.030855f,  // p = 0.116001
          -2.797586f,  // p = 0.057455
      },
  },
  // Aggressiveness = 2
  {
      // lowres
      {
          -100.0000f,  // p = 0.0
          -3.617350f,  // p = 0.026151
          -5.902503f,  // p = 0.002725
          -4.677840f,  // p = 0.009213
          -2.168378f,  // p = 0.102626
      },
      // midres
      {
          -100.0000f,  // p = 0.0
          -3.204195f,  // p = 0.0390081679555
          -2.354128f,  // p = 0.0867382128969
          -2.523326f,  // p = 0.0742390077132
          -3.112328f,  // p = 0.0426016085803
      },
      // hdres
      {
          -5.047760f,  // p = 0.00638270448225
          -3.414994f,  // p = 0.0318301469487
          -5.628090f,  // p = 0.00358255438917
          -2.122691f,  // p = 0.10691083145
          -1.972387f,  // p = 0.122132728355
      },
  },
  // Aggressiveness = 3
  {
      // lowres
      { -100.000000f, -2.04766486133f, -1.00442099188f, -1.15077982642f,
        -1.0830321897f },
      // midres
      { -100.000000f, -0.985686808303f, -0.757739584866f, -0.890120107569f,
        -0.228236297886f },
      // hdres
      { -1.03535679263f, -1.57431743203f, -0.564851540156f, -0.35442301663f,
        -1.36741555171f },
  },
};

static const float av1_simple_motion_search_split_mean_128[17] = {
  14.119120f, 14.087010f, 12.016185f, 11.966075f, 12.042454f, 11.994805f,
  12.152105f, 12.100394f, 12.178377f, 12.128937f, 4.779944f,  0.714786f,
  3.535450f,  3.566207f,  0.835913f,  3.315452f,  3.302908f,
};

static const float av1_simple_motion_search_split_std_128[17] = {
  1.832420f, 1.835338f, 2.019207f, 2.020793f, 2.008731f, 2.008403f,
  1.900999f, 1.907081f, 1.908915f, 1.913122f, 2.109345f, 0.451517f,
  1.407097f, 1.372501f, 0.370355f, 1.321495f, 1.319665f,
};

static const float av1_simple_motion_search_split_mean_64[17] = {
  12.363721f, 12.314348f, 10.404341f, 10.333541f, 10.405775f, 10.336996f,
  10.402246f, 10.330084f, 10.405584f, 10.334330f, 4.554232f,  0.896393f,
  2.819613f,  2.855845f,  0.926296f,  2.808782f,  2.798229f,
};

static const float av1_simple_motion_search_split_std_64[17] = {
  1.878920f, 1.882255f, 1.950167f, 1.953289f, 1.913869f, 1.914781f,
  1.920096f, 1.924454f, 1.880200f, 1.882499f, 2.050922f, 0.304750f,
  1.144391f, 1.125088f, 0.261289f, 1.145059f, 1.131215f,
};

static const float av1_simple_motion_search_split_mean_32[17] = {
  10.750278f, 10.679627f, 8.745625f, 8.644149f, 8.757436f, 8.656657f,
  8.759780f,  8.656299f,  8.772563f, 8.669839f, 4.208026f, 0.958573f,
  2.308769f,  2.347375f,  0.961685f, 2.323464f, 2.296322f,
};

static const float av1_simple_motion_search_split_std_32[17] = {
  1.879269f, 1.883531f, 1.935828f, 1.935677f, 1.915823f, 1.914773f,
  1.909733f, 1.910315f, 1.890451f, 1.890032f, 1.913318f, 0.199276f,
  0.988825f, 0.972115f, 0.191956f, 0.977131f, 0.951418f,
};

static const float av1_simple_motion_search_split_mean_16[17] = {
  9.076768f, 8.974986f, 7.078364f, 6.926072f, 7.088739f, 6.936111f,
  7.096697f, 6.942841f, 7.114978f, 6.961046f, 3.865480f, 0.982632f,
  1.886023f, 1.912892f, 0.981492f, 1.926059f, 1.891233f,
};

static const float av1_simple_motion_search_split_std_16[17] = {
  1.922965f, 1.925609f, 1.851980f, 1.847558f, 1.848410f, 1.843990f,
  1.843931f, 1.839582f, 1.840304f, 1.836144f, 1.760042f, 0.130639f,
  0.841086f, 0.833523f, 0.134780f, 0.840790f, 0.831309f,
};

static const float av1_simple_motion_search_split_mean_8[17] = {
  7.120238f, 6.957731f, 5.176309f, 4.889594f, 5.178396f, 4.886607f,
  5.195322f, 4.905566f, 5.198845f, 4.904745f, 3.648933f, 0.993198f,
  1.496831f, 1.520804f, 0.991864f, 1.489763f, 1.460761f,
};

static const float av1_simple_motion_search_split_std_8[17] = {
  1.698498f, 1.696000f, 1.629605f, 1.614641f, 1.632476f, 1.618831f,
  1.618352f, 1.603742f, 1.623089f, 1.609674f, 1.668587f, 0.082193f,
  0.759407f, 0.759684f, 0.089830f, 0.742797f, 0.730632f,
};

static const float *const av1_simple_motion_search_split_mean[5] = {
  av1_simple_motion_search_split_mean_128,
  av1_simple_motion_search_split_mean_64,
  av1_simple_motion_search_split_mean_32,
  av1_simple_motion_search_split_mean_16,
  av1_simple_motion_search_split_mean_8,
};

static const float *const av1_simple_motion_search_split_std[5] = {
  av1_simple_motion_search_split_std_128, av1_simple_motion_search_split_std_64,
  av1_simple_motion_search_split_std_32,  av1_simple_motion_search_split_std_16,
  av1_simple_motion_search_split_std_8,
};

#define NUM_HIDDEN_LAYERS_128 1
#define NUM_FEATURES_128 17
#define NUM_LAYER_0_UNITS_128 20
#define NUM_LOGITS_128 1

static const float av1_simple_motion_search_split_hiddenlayer_0_kernel_128[] = {
  0.24095f,    -0.397761f,  -0.388619f,  -0.0629548f, -0.44577f,   0.688212f,
  -0.20889f,   -1.08227f,   -0.0313894f, -0.615505f,  -0.401839f,  0.40233f,
  -0.171305f,  0.439803f,   1.58527f,    -0.968535f,  -1.29255f,   1.14846f,
  0.885777f,   0.116412f,   -0.225704f,  0.316506f,   0.793951f,   -0.63591f,
  0.097789f,   -0.327027f,  -0.778396f,  -0.231667f,  -0.9622f,    1.0044f,
  0.32594f,    0.179768f,   -0.115529f,  -0.499395f,  -1.14727f,   -1.26111f,
  0.269818f,   -0.0882028f, -0.349107f,  0.100901f,   0.0249506f,  0.528929f,
  0.113961f,   0.929794f,   0.242494f,   -0.122828f,  -0.0477379f, 0.170659f,
  0.0500187f,  0.28859f,    0.78783f,    0.482412f,   0.795298f,   0.179517f,
  0.453911f,   -0.298029f,  -0.903332f,  0.510615f,   0.691994f,   0.433383f,
  -0.140802f,  -1.11635f,   -0.547326f,  1.11318f,    0.71905f,    0.978538f,
  0.097444f,   -0.0386012f, 0.713599f,   0.465164f,   0.391278f,   -0.472864f,
  0.230224f,   -0.279508f,  0.558192f,   -0.468625f,  0.55995f,    -0.57507f,
  -1.39947f,   -0.755819f,  -1.04512f,   -0.411552f,  -0.830444f,  -0.106571f,
  -0.0972184f, 0.251842f,   0.269955f,   0.230492f,   -0.290581f,  -0.484799f,
  0.0151041f,  0.171047f,   0.829999f,   -0.384581f,  0.220301f,   -0.121687f,
  1.88848f,    -0.482809f,  -0.48185f,   1.34482f,    -0.716438f,  -0.284482f,
  -1.78592f,   -1.29333f,   0.886867f,   0.80106f,    0.456415f,   0.649095f,
  0.231093f,   0.361562f,   0.290018f,   0.128009f,   -0.196343f,  0.0607802f,
  0.576761f,   -0.0413836f, 0.0300984f,  -0.318998f,  0.204434f,   -0.712524f,
  0.833394f,   -0.81168f,   0.765488f,   -0.720973f,  1.12866f,    -0.838694f,
  1.295f,      -0.159127f,  1.05404f,    0.736519f,   0.248662f,   0.229233f,
  0.0434302f,  0.0551856f,  0.197862f,   0.354823f,   -0.32429f,   -0.227353f,
  -0.132198f,  -0.438118f,  -0.210401f,  -0.81046f,   0.653555f,   0.826737f,
  0.154235f,   0.228945f,   0.123089f,   0.614964f,   -0.0940471f, -0.00676807f,
  0.24996f,    0.949233f,   0.746526f,   -0.044474f,  0.386414f,   0.503221f,
  0.155133f,   -0.698848f,  -0.735356f,  -0.255091f,  0.413235f,   -0.335295f,
  -0.145757f,  0.326299f,   -0.602629f,  -0.844474f,  -0.346722f,  -0.42598f,
  -0.491016f,  -0.447732f,  -0.965366f,  -0.0242841f, 0.836606f,   -0.104877f,
  1.23236f,    0.683986f,   0.787005f,   -0.0253437f, 1.2145f,     1.29554f,
  -1.24302f,   -0.229495f,  0.439415f,   0.885087f,   -0.408704f,  -0.119299f,
  -0.0960972f, 0.60148f,    0.683271f,   -0.057129f,  -0.180295f,  -0.264815f,
  -0.363184f,  0.638271f,   0.631083f,   -0.252899f,  -0.164364f,  -1.31274f,
  0.354408f,   0.0429172f,  0.371154f,   -1.0978f,    0.0433642f,  -0.467394f,
  -0.706572f,  1.57198f,    -0.0701271f, 1.93149f,    -0.446267f,  1.4519f,
  -1.29567f,   0.309978f,   -0.878062f,  0.891494f,   0.364005f,   -0.209611f,
  -0.125927f,  0.184097f,   0.0629695f,  -0.43375f,   -0.0980562f, 1.08547f,
  0.578312f,   0.16566f,    -0.198852f,  -0.241854f,  -0.523934f,  -0.206037f,
  -0.867721f,  1.00041f,    1.09848f,    -2.12562f,   -0.19992f,   -0.186128f,
  -0.03507f,   0.0484884f,  0.160856f,   0.10802f,    -0.805141f,  -1.06902f,
  0.290363f,   0.0222096f,  -0.849266f,  0.112932f,   0.148682f,   -0.0457585f,
  1.139f,      1.79141f,    0.194122f,   -0.342508f,  -0.403572f,  0.133678f,
  0.217553f,   -0.263759f,  0.18441f,    0.254529f,   0.0471115f,  0.733178f,
  -0.416205f,  0.441447f,   -0.443335f,  0.725005f,   -0.78946f,   0.71301f,
  -0.644969f,  1.5445f,     0.365277f,   -0.455775f,  -0.365066f,  0.4742f,
  -0.381714f,  -0.545794f,  -0.0464861f, -0.222768f,  -0.0106466f, -0.069743f,
  0.0335566f,  0.378348f,   -0.249663f,  0.922286f,   0.125711f,   -0.894619f,
  0.444682f,   0.447893f,   -1.98936f,   -1.41978f,   0.0406667f,  -0.199928f,
  -0.199786f,  0.463481f,   0.334931f,   -0.396222f,  -0.0732259f, 0.796684f,
  -0.140817f,  -0.26878f,   0.194642f,   0.895784f,   -0.369976f,  -2.26981f,
  -0.0791776f, -0.0492268f, 0.6715f,     0.281805f,   0.0156664f,  -0.779785f,
  0.17743f,    0.188786f,   -0.588077f,  -0.359153f,  0.258319f,   0.881688f,
  0.846894f,   1.00292f,    0.838134f,   0.680632f,   0.273098f,   -0.329261f,
  0.217757f,   -0.506726f,  -0.336523f,  -0.695875f,  -0.252006f,  0.751216f,
  0.334409f,   -0.0151467f, 0.0885474f,  0.0973114f,  -0.248754f,  -0.263716f,
  0.369906f,   -0.213749f,  -0.0355395f, -0.137799f,  2.43233f,    -0.944233f,
  -0.745167f,  0.318558f,   0.316608f,   0.568678f
};

static const float av1_simple_motion_search_split_hiddenlayer_0_bias_128[] = {
  0.821344f,  1.11542f,   -1.24172f,  1.03642f,  1.13511f,
  1.16414f,   -0.278655f, -1.35558f,  -1.26788f, -1.63189f,
  -0.323271f, 1.21319f,   -0.888415f, 0.987145f, -1.16767f,
  0.255833f,  -0.1392f,   1.43265f,   -1.54952f, 1.65159f
};

static const float av1_simple_motion_search_split_logits_kernel_128[] = {
  0.3565753f, 0.5490161f, -1.015597f, 0.565366f,   0.751604f,
  0.922747f,  -1.931846f, 1.759353f,  -0.7362949f, 0.5707034f,
  -1.092127f, 0.936767f,  2.034499f,  2.08148f,    0.9509507f,
  -1.342504f, -0.834566f, 0.618184f,  0.844113f,   1.182693f
};

static const float av1_simple_motion_search_split_logits_bias_128[] = {
  1.819351f
};

static const NN_CONFIG av1_simple_motion_search_split_nn_config_128 = {
  NUM_FEATURES_128,
  NUM_LOGITS_128,
  NUM_HIDDEN_LAYERS_128,
  {
      NUM_LAYER_0_UNITS_128,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_kernel_128,
      av1_simple_motion_search_split_logits_kernel_128,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_bias_128,
      av1_simple_motion_search_split_logits_bias_128,
  },
};

#undef NUM_HIDDEN_LAYERS_128
#undef NUM_FEATURES_128
#undef NUM_LAYER_0_UNITS_128
#undef NUM_LOGITS_128

#define NUM_HIDDEN_LAYERS_64 1
#define NUM_FEATURES_64 17
#define NUM_LAYER_0_UNITS_64 24
#define NUM_LOGITS_64 1

static const float av1_simple_motion_search_split_hiddenlayer_0_kernel_64[] = {
  -1.40663f,    -0.851503f,   -0.0613111f,  0.741591f,    0.302754f,
  0.184001f,    0.0474853f,   0.371096f,    0.0541624f,   0.381508f,
  0.355427f,    0.0428822f,   0.154916f,    -0.00490099f, 0.025484f,
  0.0208921f,   0.140596f,    -0.292525f,   -0.459067f,   -0.081393f,
  0.109824f,    -0.290183f,   0.720236f,    0.385835f,    -0.150643f,
  -0.078518f,   0.0979819f,   -0.102135f,   0.137152f,    -0.0786457f,
  0.0171441f,   0.991338f,    -0.546583f,   -1.0714f,     -0.0842851f,
  0.244072f,    0.427379f,    0.146775f,    -0.921613f,   -0.912093f,
  0.393566f,    -0.232375f,   0.19963f,     0.312355f,    0.55659f,
  -0.104714f,   -0.137563f,   0.0985237f,   0.0788307f,   -0.225514f,
  0.0228832f,   -0.288733f,   -0.00737685f, -0.711657f,   -0.256796f,
  0.0869605f,   0.583977f,    0.384306f,    1.46692f,     -0.741126f,
  -0.21105f,    -0.276604f,   -0.0151463f,  -0.0227997f,  -0.0403232f,
  0.044122f,    0.0185784f,   -0.0451951f,  0.00489513f,  -0.387131f,
  0.0966724f,   -0.599174f,   -0.00243351f, -0.21439f,    0.302043f,
  0.130334f,    -0.191251f,   0.863261f,    -1.50112f,    0.00901057f,
  0.000324294f, -0.0572545f,  0.0117685f,   -0.0734682f,  -0.0570435f,
  -0.126253f,   1.2313f,      -0.328267f,   0.211788f,    -0.175438f,
  -0.0419298f,  0.166447f,    -0.178739f,   -0.326221f,   -0.0439188f,
  1.01182f,     -0.390678f,   -0.426343f,   0.0944665f,   -0.225042f,
  -0.183344f,   0.0500763f,   -0.377393f,   -0.673401f,   -0.436907f,
  -0.00366876f, -0.363412f,   0.195194f,    0.250248f,    -0.397193f,
  -0.0917222f,  -0.0221579f,  1.7693f,      -0.0694484f,  -0.0410764f,
  -0.134571f,   -0.159992f,   -0.170359f,   -0.249333f,   -0.128056f,
  -0.617054f,   -0.808701f,   -0.540642f,   0.396391f,    0.147787f,
  0.346916f,    0.709852f,    0.116064f,    0.0509731f,   0.073713f,
  -0.365082f,   -1.09287f,    -0.618214f,   0.20545f,     0.126161f,
  -0.140012f,   0.62592f,     0.316326f,    -0.392765f,   -0.15934f,
  0.337617f,    -0.41669f,    -0.295225f,   0.0602025f,   -0.0150657f,
  -0.319629f,   0.783729f,    -0.0661199f,  -0.362657f,   0.390042f,
  -0.043614f,   -0.0414596f,  0.121155f,    -0.309775f,   -0.284761f,
  -0.243932f,   0.279855f,    -0.266823f,   0.734824f,    -0.164028f,
  0.261776f,    -0.105585f,   0.10733f,     -0.180469f,   1.18875f,
  -1.12836f,    -0.173008f,   0.150221f,    0.111598f,    0.148306f,
  -1.2833f,     -1.06346f,    0.233546f,    0.16432f,     0.00142378f,
  0.340574f,    -0.0140885f,  0.634761f,    -0.122096f,   0.821487f,
  0.421424f,    -0.0256687f,  -0.035503f,   -0.0453547f,  -0.0215179f,
  -0.0671277f,  -0.0486862f,  -0.962761f,   -0.208383f,   0.109573f,
  -0.210668f,   -0.176485f,   0.421279f,    0.41605f,     0.342084f,
  0.619364f,    0.103718f,    -0.00341643f, 0.00266677f,  0.249089f,
  -0.22848f,    -0.0368968f,  1.12092f,     -0.64912f,    -0.456579f,
  0.477823f,    0.418345f,    1.41515f,     0.0936279f,   0.886155f,
  -0.785656f,   -0.217109f,   -0.561829f,   -0.286435f,   -0.884068f,
  -0.148839f,   -0.282848f,   0.0683745f,   0.0962815f,   -0.111975f,
  0.0509158f,   -0.211274f,   0.744909f,    -0.8982f,     0.315232f,
  -0.78624f,    0.598387f,    -0.530952f,   0.677357f,    0.0371339f,
  0.99209f,     -0.681899f,   -0.291416f,   -0.224822f,   -0.26049f,
  -0.0436525f,  -0.380004f,   -0.27187f,    0.534779f,    0.717939f,
  0.418197f,    -0.152539f,   -0.0684039f,  -0.186308f,   -0.0653121f,
  0.194145f,    -0.196367f,   0.256997f,    -0.726269f,   -0.307672f,
  -0.153362f,   0.450827f,    0.708842f,    -0.0667079f,  0.555564f,
  0.0486892f,   0.0715072f,   -0.7211f,     -0.849797f,   0.0650271f,
  1.2747f,      -0.646738f,   -0.53042f,    0.182197f,    0.928203f,
  0.180621f,    -0.00640791f, -0.171416f,   0.092688f,    -0.391275f,
  -0.0650657f,  0.0843773f,   0.170824f,    0.378085f,    0.0596657f,
  0.844398f,    -1.3083f,     -1.27828f,    -0.199179f,   0.557855f,
  0.241479f,    0.385804f,    0.169533f,    -0.0028072f,  0.0538041f,
  0.00136234f,  0.0130481f,   0.0349449f,   -0.0366494f,  -0.000474055f,
  0.437956f,    0.286724f,    -0.298187f,   0.461967f,    0.43065f,
  -0.0877194f,  -0.19133f,    0.379121f,    -0.687751f,   -1.64077f,
  -0.375191f,   -0.336836f,   -0.323904f,   -0.101859f,   0.0126672f,
  -0.346332f,   0.112303f,    -0.863336f,   0.155538f,    0.366509f,
  -0.0976829f,  0.635278f,    -0.681967f,   -0.527729f,   0.591839f,
  0.366678f,    0.189981f,    0.0208007f,   -0.565809f,   0.70183f,
  -0.282844f,   -0.327485f,   0.347243f,    -1.13014f,    -0.373378f,
  -0.514978f,   0.662994f,    -0.144931f,   0.1402f,      -0.820049f,
  0.711498f,    0.681156f,    1.06515f,     -0.423409f,   -0.0392664f,
  0.0675396f,   -0.0508602f,  0.0431443f,   0.0212639f,   -0.0279887f,
  -0.62611f,    -0.202064f,   0.701934f,    1.28452f,     -0.00858481f,
  -0.517249f,   0.0615832f,   -0.260215f,   0.0949119f,   -0.28423f,
  -0.39573f,    -0.0574246f,  -0.318658f,   0.0601775f,   -0.0629386f,
  -0.134208f,   0.111686f,    -0.23355f,    0.078667f,    0.741023f,
  0.828523f,    -0.345067f,   -0.315135f,   -0.0957154f,  0.522825f,
  -0.190057f,   -0.473789f,   -0.390489f,   0.200677f,    -0.0271802f,
  0.110336f,    0.493302f,    0.663126f,    0.570148f,    -0.380042f,
  -0.437349f,   -0.660884f,   0.301908f,    0.0644179f,   0.172494f,
  0.461917f,    0.330938f,    -0.140041f,   -0.0430205f,  -1.51003f,
  -0.410984f,   -0.182161f,   0.0235313f,   -0.364849f,   0.154183f,
  -0.592465f,   0.272701f,    0.192389f,    -0.0497777f,  -0.924467f,
  -0.179513f,   -0.592217f,   0.436363f,    -0.0716164f,  0.189094f,
  -0.574697f,   -0.304303f,   0.326441f,    -0.0865553f,  0.735948f,
  0.266912f,    0.435824f,    -0.123322f
};

static const float av1_simple_motion_search_split_hiddenlayer_0_bias_64[] = {
  -1.19333f,  1.01834f,   -1.10844f,  0.0454873f, -1.45506f,   0.580864f,
  -0.040979f, -0.505681f, -1.15072f,  0.692697f,  -0.520812f,  -0.479384f,
  0.529652f,  0.507252f,  -1.08619f,  0.0586375f, 0.0929614f,  -0.46753f,
  -0.701857f, -0.362933f, -0.291983f, -0.133933f, -0.0131351f, -0.267582f
};

static const float av1_simple_motion_search_split_logits_kernel_64[] = {
  -3.32501f,  0.43082f,   -1.060692f, 1.328908f,  0.8892894f,  0.6488833f,
  -1.096516f, -0.664786f, -1.301339f, 0.508805f,  -2.128406f,  -0.757304f,
  0.383839f,  0.694763f,  -0.591725f, 0.770385f,  1.021594f,   0.589181f,
  -0.76238f,  1.488826f,  0.709135f,  -0.575738f, 0.26421759f, -0.2484219f
};

static const float av1_simple_motion_search_split_logits_bias_64[] = {
  0.699037f
};

static const NN_CONFIG av1_simple_motion_search_split_nn_config_64 = {
  NUM_FEATURES_64,
  NUM_LOGITS_64,
  NUM_HIDDEN_LAYERS_64,
  {
      NUM_LAYER_0_UNITS_64,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_kernel_64,
      av1_simple_motion_search_split_logits_kernel_64,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_bias_64,
      av1_simple_motion_search_split_logits_bias_64,
  },
};

#undef NUM_HIDDEN_LAYERS_64
#undef NUM_FEATURES_64
#undef NUM_LAYER_0_UNITS_64
#undef NUM_LOGITS_64

#define NUM_HIDDEN_LAYERS_32 1
#define NUM_FEATURES_32 17
#define NUM_LAYER_0_UNITS_32 20
#define NUM_LOGITS_32 1

static const float av1_simple_motion_search_split_hiddenlayer_0_kernel_32[] = {
  -0.980626f,   -0.946611f,    0.103761f,    0.408899f,    0.498149f,
  0.0490161f,   0.253279f,     0.332029f,    0.00367441f,  0.364401f,
  -0.236433f,   0.0592119f,    -0.0978848f,  0.159733f,    -0.018052f,
  -1.10726f,    1.16167f,      -0.244982f,   -0.147819f,   -0.147095f,
  0.111404f,    -0.349502f,    0.441178f,    0.0984191f,   -0.135537f,
  -0.0423312f,  0.0123079f,    0.358012f,    -0.266796f,   0.0125811f,
  0.196563f,    0.337093f,     -1.07266f,    -1.25134f,    0.57337f,
  -0.521717f,   0.259824f,     0.537383f,    -0.463688f,   -0.336128f,
  0.373385f,    0.483443f,     -0.229293f,   -0.33373f,    -0.656021f,
  0.768647f,    0.179279f,     0.315415f,    0.187749f,    1.07839f,
  0.0626629f,   -0.230299f,    0.662606f,    -0.414154f,   0.459334f,
  -0.6312f,     0.427704f,     -0.249849f,   0.701056f,    -0.707969f,
  0.057401f,    0.620434f,     0.665748f,    -0.501356f,   -0.230685f,
  0.0722371f,   -0.0988625f,   -0.114035f,   -0.653799f,   0.571353f,
  0.268276f,    1.13251f,      -1.0695f,     -0.225607f,   -0.984355f,
  -0.42213f,    0.300422f,     1.21492f,     -0.139931f,   -0.000726004f,
  0.045964f,    -0.0817352f,   -0.0278813f,  -0.0102341f,  -0.0144087f,
  -0.475882f,   1.20682f,      -0.359919f,   0.277189f,    -0.166401f,
  0.599211f,    -0.129872f,    0.574211f,    -0.247573f,   0.824405f,
  -1.53329f,    -0.202151f,    -0.328698f,   -0.516322f,   -0.281416f,
  -0.383651f,   -0.252862f,    -0.43185f,    0.456802f,    -0.430055f,
  -0.55245f,    -0.6884f,      -0.541456f,   -0.281376f,   1.10425f,
  -0.140706f,   1.59816f,      -0.0343895f,  -0.00920039f, -0.0307667f,
  0.0560132f,   -0.0340302f,   -0.10848f,    0.0593314f,   -0.951795f,
  0.876831f,    -1.00548f,     -0.566244f,   0.430061f,    1.10109f,
  -0.634212f,   -0.0755369f,   -0.108953f,   1.03191f,     0.109036f,
  -0.0415309f,  0.0681162f,    -0.0611775f,  -0.0231938f,  0.0973158f,
  -0.0558169f,  -0.823484f,    -0.918509f,   0.16756f,     0.27087f,
  0.286074f,    0.174069f,     0.1304f,      0.386074f,    0.433953f,
  0.0291467f,   -1.74087f,     0.0296094f,   -0.00793714f, -0.13041f,
  0.00990992f,  -0.0137848f,   -0.0742606f,  -0.251029f,   -0.645316f,
  0.640029f,    0.550607f,     0.470097f,    0.549451f,    -0.285723f,
  -0.164759f,   -0.128166f,    -0.391496f,   -0.80287f,    0.0769472f,
  1.34391f,     0.0215005f,    0.0669497f,   0.131919f,    0.291674f,
  0.0952889f,   -0.677953f,    -0.364054f,   0.144823f,    0.246198f,
  -0.12393f,    0.363661f,     0.215091f,    -0.239658f,   0.18491f,
  0.118703f,    0.0064156f,    1.38619f,     -1.3845f,     0.0567323f,
  1.20812f,     -0.720374f,    -1.92158f,    -1.48657f,    0.335601f,
  0.409379f,    0.373618f,     0.231274f,    0.292194f,    0.368619f,
  0.2398f,      0.473579f,     0.83402f,     -0.0133751f,  -0.00344358f,
  2.20688e-05f, 0.00836757f,   0.00405377f,  0.0110539f,   -0.260154f,
  0.192112f,    -0.666986f,    0.302875f,    -0.113302f,   0.17882f,
  -0.221493f,   0.146161f,     -0.448697f,   0.584187f,    0.122109f,
  0.989981f,    -1.14706f,     -0.734042f,   0.0638213f,   0.213357f,
  0.068543f,    -0.808558f,    0.404741f,    0.808313f,    1.57523f,
  -0.113448f,   0.254102f,     -0.350065f,   -0.615f,      0.0753549f,
  -0.540936f,   -0.0250732f,   -0.225681f,   -0.161384f,   0.0128342f,
  -0.0933368f,  -0.286904f,    0.130133f,    -0.874747f,   0.392585f,
  -0.493135f,   0.169708f,     0.0909804f,   1.89921f,     -0.469954f,
  0.65165f,     -0.953401f,    -0.21595f,    -0.37479f,    0.0451146f,
  0.0234621f,   -0.0596903f,   -0.0682308f,  -0.0830426f,  0.130011f,
  -0.409141f,   0.0627038f,    -0.581148f,   -0.513922f,   0.631676f,
  0.0637034f,   0.0539081f,    0.0638872f,   0.515863f,    -0.0123463f,
  0.177238f,    0.279506f,     -0.930345f,   1.23726f,     0.202851f,
  0.708792f,    -0.445086f,    -0.0267075f,  -0.913822f,   -0.0714978f,
  -0.281107f,   -0.0770565f,   -0.23086f,    -0.165893f,   -0.319683f,
  0.216235f,    -0.490999f,    2.04841f,     -0.0524071f,  -0.239043f,
  -0.0526375f,  0.023002f,     -0.132685f,   -0.155354f,   -0.186503f,
  -0.904296f,   0.166478f,     0.063268f,    -0.302842f,   -0.27179f,
  -0.428299f,   0.50193f,      0.480717f,    -0.864275f,   0.317096f,
  0.40698f,     0.0286107f,    0.189432f,    -0.0374374f,  0.0671728f,
  0.203681f,    -0.457959f,    -0.155776f,   0.340948f,    0.542841f,
  0.342675f,    -0.000952399f, 0.470957f,    0.744418f,    -1.11763f,
  -0.658812f,   -0.044832f,    0.0688237f,   -0.357766f,   0.428662f,
  -0.087152f,   -0.291903f,    0.373244f,    -0.587853f,   0.415895f,
  -0.535694f,   0.621785f,     -0.143648f,   0.0451373f,   0.00068827f,
  1.84432f,     -1.26239f,     -0.432087f,   -0.152307f,   0.0293551f,
  0.184744f,    -0.0173156f,   -0.00572154f, -0.0305062f,  -0.0900071f
};

static const float av1_simple_motion_search_split_hiddenlayer_0_bias_32[] = {
  0.160011f,  0.903856f,   -0.13738f,  0.358221f, -0.0906044f,
  -0.606558f, -0.0215651f, -0.03377f,  -1.67017f, -0.144554f,
  -0.201482f, -0.87719f,   0.639815f,  -0.51976f, -0.309922f,
  -1.33421f,  0.721328f,   -0.889354f, -1.7158f,  -0.285963f
};

static const float av1_simple_motion_search_split_logits_kernel_32[] = {
  -0.2745374f,  0.333548f,  -0.2437388f, 0.288009f,   0.55635f,
  0.4560176f,   0.2970518f, 0.391192f,   1.311854f,   -0.231219f,
  -0.2968651f,  -1.819984f, 0.2775824f,  0.28929857f, 0.419126f,
  -0.32868411f, -0.916399f, -0.1921077f, -0.617489f,  0.637953f
};

static const float av1_simple_motion_search_split_logits_bias_32[] = {
  0.208473f
};

static const NN_CONFIG av1_simple_motion_search_split_nn_config_32 = {
  NUM_FEATURES_32,
  NUM_LOGITS_32,
  NUM_HIDDEN_LAYERS_32,
  {
      NUM_LAYER_0_UNITS_32,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_kernel_32,
      av1_simple_motion_search_split_logits_kernel_32,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_bias_32,
      av1_simple_motion_search_split_logits_bias_32,
  },
};

#undef NUM_HIDDEN_LAYERS_32
#undef NUM_FEATURES_32
#undef NUM_LAYER_0_UNITS_32
#undef NUM_LOGITS_32

#define NUM_HIDDEN_LAYERS_16 1
#define NUM_FEATURES_16 17
#define NUM_LAYER_0_UNITS_16 20
#define NUM_LOGITS_16 1

static const float av1_simple_motion_search_split_hiddenlayer_0_kernel_16[] = {
  0.0136957f,   0.182135f,    -0.583394f,    0.0556956f,   0.211152f,
  0.168234f,    -0.694203f,   -0.678216f,    0.289943f,    1.00014f,
  -0.0427784f,  -0.0427538f,  -0.0276009f,   -0.00133608f, 0.0901944f,
  0.0674892f,   0.104068f,    -0.308582f,    -0.43596f,    0.855997f,
  -0.223414f,   0.0390026f,   0.366492f,     0.216065f,    -0.386863f,
  -0.148823f,   -0.297022f,   0.0529546f,    -0.202885f,   1.26471f,
  -0.861163f,   -0.0949431f,  0.573627f,     -0.00277083f, -0.616063f,
  -0.626927f,   0.371583f,    -0.411743f,    0.173387f,    -0.209734f,
  0.293697f,    -0.260714f,   0.442728f,     -0.594486f,   1.38987f,
  0.208025f,    -0.0433776f,  0.01173f,      0.921766f,    -0.168379f,
  0.000697326f, 0.209967f,    -0.304577f,    0.149551f,    -0.196658f,
  0.389251f,    -0.449106f,   -0.456329f,    0.669073f,    -0.163806f,
  0.083348f,    -0.0783998f,  0.0678355f,    0.0510435f,   0.103964f,
  0.104537f,    -0.778093f,   -1.0641f,      -0.626102f,   -2.02131f,
  0.159591f,    0.254161f,    -0.000362642f, 0.289859f,    0.192713f,
  0.139801f,    -0.0251327f,  0.164002f,     1.22892f,     -0.0852193f,
  0.0769487f,   0.0296408f,   -0.0418688f,   0.0936023f,   0.0448523f,
  0.674015f,    -0.0732944f,  0.313575f,     -0.593432f,   0.642067f,
  -1.06063f,    0.468223f,    -0.769085f,    -0.173798f,   -0.175663f,
  0.692808f,    0.00753295f,  -0.123327f,    -0.0234937f,  -0.0923153f,
  0.0216917f,   -0.0690157f,  -0.397488f,    0.426628f,    0.264475f,
  0.342074f,    -0.139817f,   0.215915f,     0.422544f,    -0.321102f,
  0.0355587f,   0.460193f,    0.0315326f,    0.080556f,    -0.0256533f,
  -0.0857874f,  -0.488283f,   -0.299653f,    -0.245987f,   0.104383f,
  0.203731f,    0.328734f,    0.668104f,     -0.586909f,   -0.501335f,
  -0.661292f,   -0.359811f,   0.00951363f,   0.816315f,    -0.0124104f,
  0.0545827f,   0.089863f,    0.0125486f,    0.043609f,    -0.0259544f,
  0.0123911f,   0.12557f,     -0.539875f,    -0.0556721f,  0.16532f,
  0.265834f,    -0.384171f,   0.646496f,     0.366147f,    -0.111272f,
  0.262096f,    -0.0845724f,  0.382724f,     0.165783f,    0.1025f,
  0.392988f,    0.290525f,    0.038659f,     0.540269f,    -0.485586f,
  -0.273065f,   -0.154052f,   -0.0896895f,   -0.35394f,    0.193214f,
  -0.423728f,   0.654576f,    -0.373321f,    0.814914f,    0.026278f,
  -0.0328304f,  -0.220913f,   -0.0442121f,   0.487545f,    -0.509537f,
  -0.777581f,   -1.23886f,    0.223482f,     0.206009f,    0.20391f,
  0.194628f,    0.226762f,    0.171609f,     -0.219037f,   0.557892f,
  -0.312011f,   1.27709f,     0.064013f,     0.105384f,    0.0493933f,
  0.074059f,    -0.0100078f,  -0.0176888f,   -0.440005f,   0.302922f,
  -0.197456f,   0.296128f,    -0.326647f,    0.305323f,    -0.30696f,
  0.201951f,    -0.15874f,    -0.793042f,    0.0197254f,   0.0569867f,
  -0.0295468f,  -0.0215012f,  0.025855f,     -0.0196102f,  0.215558f,
  -0.253069f,   0.298469f,    0.261269f,     0.435305f,    0.0120354f,
  -0.384789f,   -0.2772f,     0.0366613f,    -0.494994f,   0.149072f,
  1.32981f,     -0.427717f,   0.43938f,      -0.16375f,    -0.444342f,
  0.548214f,    0.127955f,    -1.24387f,     0.0863676f,   0.175071f,
  0.172673f,    -0.0906204f,  0.444454f,     -0.546669f,   0.215857f,
  -0.100621f,   0.200699f,    -0.0985915f,   0.134706f,    -0.256396f,
  0.393427f,    0.119606f,    -0.214278f,    -0.0183637f,  0.194266f,
  -0.238025f,   0.182203f,    0.599718f,     0.846933f,    0.0607852f,
  -0.183434f,   -0.723743f,   -0.72414f,     -0.124701f,   0.0227527f,
  -0.0664636f,  -0.0385867f,  -0.0257377f,   -0.149054f,   0.12077f,
  0.678029f,    -0.624456f,   0.189644f,     -0.518604f,   0.134397f,
  -0.189777f,   -0.309376f,   -0.00377086f,  0.701132f,    -0.170915f,
  0.00736111f,  -0.121906f,   0.329136f,     0.165514f,    0.0328356f,
  0.171275f,    0.248619f,    0.247704f,     -0.449933f,   0.0841684f,
  0.136982f,    0.122703f,    -0.0169439f,   -0.0726496f,  0.302648f,
  -0.128556f,   0.0667425f,   -0.289717f,    -0.207532f,   -1.20269f,
  -0.68892f,    0.045259f,    0.0973945f,    0.0988314f,   -0.944748f,
  -0.180401f,   0.134331f,    0.033834f,     0.109023f,    0.265723f,
  0.38063f,     -0.106518f,   -0.0686953f,   0.3744f,      -1.0957f,
  0.0302782f,   0.0515164f,   0.00188222f,   0.0014413f,   -0.0404425f,
  0.0124618f,   -0.0828645f,  0.506166f,     -0.776352f,   -0.405138f,
  -0.123887f,   0.0732116f,   0.379928f,     0.604524f,    -0.492317f,
  0.439191f,    0.0744193f,   0.389101f,     0.0604518f,   0.0943165f,
  0.0339942f,   0.0917975f,   0.0161988f,    0.512227f,    0.538021f,
  -0.411495f,   0.307281f,    0.33746f,      -0.218639f,   0.265742f,
  0.39738f,     -0.12442f,    0.125236f,     -0.0845223f,  -0.150396f,
  0.0334878f,   -0.00391915f, 0.0406864f,    -0.0487059f,  0.0377073f
};

static const float av1_simple_motion_search_split_hiddenlayer_0_bias_16[] = {
  0.0535976f, -0.0130279f, 0.150146f,   -0.511132f, -0.357698f,
  0.6719f,    -1.27877f,   -0.0208048f, 0.0961914f, 0.263603f,
  0.704574f,  -1.48998f,   0.728063f,   0.941829f,  -0.199981f,
  0.797802f,  -0.29816f,   -0.60894f,   -0.116624f, -1.16723f
};

static const float av1_simple_motion_search_split_logits_kernel_16[] = {
  0.343153f,   -0.2110482f, -0.487199f,   0.3274144f, -2.1975f,
  -0.6051438f, 0.1901127f,  0.4741924f,   -0.24029f,  -0.185018f,
  -0.652635f,  2.57714f,    -0.31033031f, -0.307222f, 0.329035f,
  -0.430181f,  0.3429f,     0.742292f,    0.3269808f, 0.4142165f
};

static const float av1_simple_motion_search_split_logits_bias_16[] = {
  -0.783658f
};

static const NN_CONFIG av1_simple_motion_search_split_nn_config_16 = {
  NUM_FEATURES_16,
  NUM_LOGITS_16,
  NUM_HIDDEN_LAYERS_16,
  {
      NUM_LAYER_0_UNITS_16,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_kernel_16,
      av1_simple_motion_search_split_logits_kernel_16,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_bias_16,
      av1_simple_motion_search_split_logits_bias_16,
  },
};

#undef NUM_HIDDEN_LAYERS_16
#undef NUM_FEATURES_16
#undef NUM_LAYER_0_UNITS_16
#undef NUM_LOGITS_16

#define NUM_HIDDEN_LAYERS_8 1
#define NUM_FEATURES_8 17
#define NUM_LAYER_0_UNITS_8 20
#define NUM_LOGITS_8 1

static const float av1_simple_motion_search_split_hiddenlayer_0_kernel_8[] = {
  0.079443f,   -1.04068f,   0.336819f,    -0.20901f,   0.796251f,
  0.181066f,   0.0118876f,  -0.207145f,   0.250671f,   -0.402119f,
  -0.0847227f, 1.88683f,    0.303469f,    0.0718458f,  0.0338589f,
  0.158896f,   0.0540238f,  -0.385426f,   0.955925f,   0.424506f,
  0.492584f,   -0.795058f,  -0.248667f,   -0.905349f,  -0.316989f,
  0.545471f,   0.63762f,    -0.232613f,   -0.238947f,  -0.395338f,
  -0.322673f,  -0.0761563f, -0.125357f,   0.0694415f,  -0.371599f,
  0.358387f,   -0.486841f,  0.403863f,    -0.0295666f, 0.283074f,
  -0.424396f,  0.156318f,   -0.685355f,   0.6663f,     0.337949f,
  0.273198f,   0.517448f,   0.458911f,    0.157252f,   0.692096f,
  0.64965f,    -0.23987f,   -1.08431f,    -0.252475f,  -0.332614f,
  -0.712291f,  -0.380973f,  0.460545f,    0.48936f,    0.337601f,
  0.489223f,   1.65336f,    -0.223585f,   0.17367f,    -0.235057f,
  -0.456773f,  0.327877f,   -0.221192f,   -0.940151f,  -1.06616f,
  0.687084f,   -0.109973f,  0.106636f,    0.445895f,   0.163432f,
  0.378306f,   0.201902f,   0.176811f,    0.693082f,   1.62156f,
  -0.178346f,  0.455175f,   1.61943f,     0.231376f,   0.0890932f,
  -0.889693f,  -1.03298f,   0.778196f,    -0.0289539f, 0.137848f,
  0.18707f,    0.171889f,   0.119157f,    0.24893f,    -0.313628f,
  0.00250735f, -0.0758209f, 0.272974f,    -0.229825f,  2.47926f,
  -0.0354665f, 0.175366f,   0.0411555f,   -1.52149f,   -0.0258663f,
  0.253027f,   -0.0520839f, -0.0189782f,  0.362387f,   -0.371154f,
  0.622929f,   0.0447056f,  0.242529f,    -0.168391f,  0.308935f,
  -0.117294f,  2.16307f,    0.0673638f,   0.080771f,   -0.460779f,
  -0.940176f,  0.473266f,   -0.0125302f,  0.475145f,   -0.218187f,
  0.43258f,    -0.0380196f, 0.413607f,    -0.110856f,  -1.52076f,
  0.0896812f,  0.246636f,   -0.0612008f,  0.189583f,   0.0106902f,
  -0.158403f,  -0.629377f,  -0.0634279f,  -0.0864584f, -0.226568f,
  -0.286234f,  -0.0721132f, -0.43702f,    0.113702f,   0.433372f,
  0.743396f,   0.14312f,    0.29914f,     0.801188f,   0.7609f,
  0.385046f,   0.480314f,   0.171119f,    -1.59058f,   -1.18853f,
  0.150676f,   0.408123f,   -0.00677924f, 0.398145f,   0.0914611f,
  0.176945f,   0.0677457f,  0.316478f,    0.998219f,   -0.22618f,
  0.0756793f,  -0.0156674f, 0.105716f,    0.0496245f,  -0.0827133f,
  -0.423119f,  -0.161033f,  0.212962f,    -0.234453f,  0.743366f,
  1.04108f,    0.0597604f,  -0.285993f,   -0.114829f,  -0.557364f,
  -0.840051f,  0.326509f,   -0.192508f,   -0.141769f,  0.370626f,
  -0.126353f,  0.00672923f, 0.493623f,    -0.852076f,  0.466798f,
  -0.226436f,  0.259268f,   -0.452662f,   0.0721126f,  0.0198245f,
  0.2048f,     0.02506f,    0.316194f,    0.814651f,   1.01288f,
  -0.569607f,  -0.0838994f, 1.37146f,     -0.613135f,  0.441761f,
  -0.643901f,  0.364269f,   -0.147177f,   0.338001f,   -0.332376f,
  0.518875f,   -0.628964f,  -0.291889f,   -0.050736f,  0.108047f,
  1.05673f,    0.0479492f,  0.466756f,    -0.0867334f, -0.0355575f,
  0.57626f,    -0.227583f,  -0.146421f,   0.0990489f,  0.117351f,
  -0.103858f,  -0.0336936f, 0.0201903f,   -0.0766383f, -0.010211f,
  0.0400779f,  0.0725462f,  0.137142f,    0.478261f,   0.287869f,
  0.0882359f,  -0.739754f,  -0.853521f,   -0.43703f,   0.316856f,
  0.27593f,    0.312149f,   0.175575f,    0.441839f,   0.264325f,
  0.0148051f,  -0.005559f,  0.373176f,    0.933701f,   -0.0197615f,
  0.0219723f,  -0.0559883f, -0.103456f,   -0.0323009f, 0.0773202f,
  -0.390838f,  0.855488f,   -0.596525f,   -0.249093f,  0.124262f,
  0.220172f,   0.0552478f,  1.04041f,     -0.960992f,  -0.495255f,
  -0.211612f,  0.350007f,   -0.238998f,   -0.0265068f, 0.384686f,
  -0.0815808f, -0.0570019f, 0.123903f,    -0.485114f,  -0.00282573f,
  -0.0649603f, 0.163719f,   -0.469479f,   -0.439713f,  0.0602562f,
  -0.527993f,  -0.111458f,  2.48686f,     -0.180723f,  0.0553895f,
  0.0560679f,  -0.0978928f, -0.216063f,   0.089457f,   -1.5602f,
  -1.62332f,   -0.147388f,  0.736155f,    0.440409f,   0.243519f,
  0.0622638f,  0.522932f,   0.109686f,    0.422849f,   0.510589f,
  1.01116f,    0.174019f,   0.0191171f,   -0.0717751f, -0.0068308f,
  0.172932f,   -0.834888f,  -0.635788f,   0.32012f,    0.298656f,
  0.274309f,   -0.155456f,  0.1755f,      -0.175171f,  0.343498f,
  -0.122832f,  -0.107696f,  0.279924f,    -0.797633f,  -0.344658f,
  0.162669f,   0.389092f,   0.644479f,    -0.635216f,  -0.181868f,
  0.0579244f,  -0.0568976f, 0.433003f,    -0.591067f,  0.71013f,
  -0.165515f,  0.225725f,   -0.358156f,   0.0541944f,  1.95485f,
  -0.315223f,  0.61537f,    -0.0401568f,  0.22811f,    0.271147f
};

static const float av1_simple_motion_search_split_hiddenlayer_0_bias_8[] = {
  1.63441f,  -0.616459f, -0.437775f, -0.71669f,  1.56616f,  2.28109f, 1.64054f,
  -1.51476f, 0.0274108f, 0.935156f,  -0.966329f, 0.906069f, 1.19954f, -1.25867f,
  -1.7376f,  -0.594211f, 0.322242f,  0.438631f,  -1.01682f, 1.30032f
};

static const float av1_simple_motion_search_split_logits_kernel_8[] = {
  -0.463187f, 0.2936127f, 0.16762f,    -0.1663271f, -0.292418f,
  -0.421457f, -0.378265f, 1.053049f,   0.32432879f, -0.49775575f,
  0.427357f,  -0.239251f, -0.1631546f, 0.335468f,   0.255371f,
  0.276901f,  -0.665683f, -0.7021493f, 0.381513f,   -0.1339761f
};

static const float av1_simple_motion_search_split_logits_bias_8[] = {
  -1.739754f
};

static const NN_CONFIG av1_simple_motion_search_split_nn_config_8 = {
  NUM_FEATURES_8,
  NUM_LOGITS_8,
  NUM_HIDDEN_LAYERS_8,
  {
      NUM_LAYER_0_UNITS_8,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_kernel_8,
      av1_simple_motion_search_split_logits_kernel_8,
  },
  {
      av1_simple_motion_search_split_hiddenlayer_0_bias_8,
      av1_simple_motion_search_split_logits_bias_8,
  },
};

#undef NUM_HIDDEN_LAYERS_8
#undef NUM_FEATURES_8
#undef NUM_LAYER_0_UNITS_8
#undef NUM_LOGITS_8

static const NN_CONFIG *const av1_simple_motion_search_split_nn_config[5] = {
  &av1_simple_motion_search_split_nn_config_128,
  &av1_simple_motion_search_split_nn_config_64,
  &av1_simple_motion_search_split_nn_config_32,
  &av1_simple_motion_search_split_nn_config_16,
  &av1_simple_motion_search_split_nn_config_8,
};

// Model based on simple_motion_search for pruning rect
// Thresholds. The first idx level is aggresiveness, second is frame resolution,
// third is bsize
static const float av1_simple_motion_search_prune_rect_thresh[4][3][5] = {
  // Aggressivness = 0
  {
      // Lowres
      { 0.0288721601835f, 0.0281573780991f, 0.0225501403434f,
        0.000961189195907f, 0.0f },
      // Midres
      { 0.0288721601835f, 0.0281573780991f, 0.0225501403434f,
        0.000961189195907f, 0.0f },
      // Hdres
      { 0.0288721601835f, 0.0281573780991f, 0.0225501403434f,
        0.000961189195907f, 0.0f },
  },
  // Aggressivness = 1
  {
      // Lowres
      {
          0.000000f,
          0.116076f,
          0.049759f,
          0.057747f,
          0.006001f,
      },
      // Midres
      {
          0.000000f,
          0.017380f,
          0.026077f,
          0.078111f,
          0.064477f,
      },
      // Hdres
      {
          0.002994f,
          0.103093f,
          0.076408f,
          0.010456f,
          0.187211f,
      },
  },
  // Aggressiveness = 2
  {
      // Lowres
      {
          0.000000f,
          0.003111f,
          0.144294f,
          0.144884f,
          0.069924f,
      },
      // Midres
      {
          0.000000f,
          0.013696f,
          0.055203f,
          0.152271f,
          0.078886f,
      },
      // Hdres
      {
          0.030577f,
          0.082486f,
          0.040690f,
          0.140924f,
          0.067608f,
      },
  },
  // Aggressiveness = 3
  {
      // Lowres
      { 0.0f, 0.352338114654f, 0.171190796972f, 0.322629318068f,
        0.287219697095f },
      // Midres
      { 0.0f, 0.30938393361f, 0.271772875141f, 0.240627957104f,
        0.178833795641f },
      // Hdres
      { 0.285731215187f, 0.37521798723f, 0.142380566244f, 0.338288917819f,
        0.21329309279f },
  },
};

// Mean and std
static const float av1_simple_motion_search_prune_rect_mean_128[25] = {
  13.292176f, 13.231236f, 11.098058f, 11.049944f, 10.481336f,
  10.431587f, 10.789337f, 10.732787f, 10.233817f, 10.173738f,
  12.214045f, 12.157505f, 11.863353f, 11.802220f, 12.204053f,
  12.152315f, 11.517566f, 11.465651f, 5.383040f,  0.757934f,
  4.012611f,  4.052191f,  0.853365f,  3.954503f,  3.944135f,
};

static const float av1_simple_motion_search_prune_rect_std_128[25] = {
  2.589217f, 2.559396f, 2.268402f, 2.282274f, 3.341234f, 3.341994f, 3.033007f,
  3.041550f, 3.786247f, 3.784053f, 2.523459f, 2.511275f, 3.349364f, 3.340481f,
  2.390149f, 2.384226f, 3.599467f, 3.587460f, 2.319911f, 0.428335f, 1.241087f,
  1.208679f, 0.353742f, 1.228122f, 1.211777f,
};

static const float av1_simple_motion_search_prune_rect_mean_64[25] = {
  11.439831f, 11.382639f, 9.647134f, 9.578121f, 9.146770f,
  9.084122f,  8.559063f,  8.499496f, 8.095865f, 8.041795f,
  10.547537f, 10.486240f, 9.362147f, 9.308391f, 10.548071f,
  10.484358f, 10.002225f, 9.944480f, 4.964504f, 0.897164f,
  3.306144f,  3.351039f,  0.928582f, 3.319739f, 3.287726f,
};

static const float av1_simple_motion_search_prune_rect_std_64[25] = {
  2.033404f, 2.050657f, 2.064671f, 2.081519f, 2.916312f, 2.914649f, 3.628949f,
  3.618760f, 4.011421f, 3.996068f, 2.087127f, 2.103106f, 3.885277f, 3.876166f,
  2.035599f, 2.052976f, 3.052501f, 3.050985f, 2.232998f, 0.303745f, 1.111161f,
  1.081292f, 0.257521f, 1.112510f, 1.089404f,
};

static const float av1_simple_motion_search_prune_rect_mean_32[25] = {
  9.862349f, 9.793658f, 8.043962f, 7.954083f, 8.058867f, 7.966165f, 8.046844f,
  7.956817f, 8.061414f, 7.967906f, 8.966450f, 8.890165f, 8.968315f, 8.891513f,
  8.953573f, 8.877070f, 8.974275f, 8.895363f, 4.387239f, 0.954143f, 2.701000f,
  2.751266f, 0.963302f, 2.716584f, 2.709725f,
};

static const float av1_simple_motion_search_prune_rect_std_32[25] = {
  1.971555f, 1.985517f, 1.935986f, 1.944743f, 1.924122f, 1.932169f, 1.943151f,
  1.950612f, 1.931156f, 1.938242f, 1.987803f, 1.997670f, 2.000859f, 2.009913f,
  1.938270f, 1.949277f, 1.922999f, 1.933145f, 1.991504f, 0.209175f, 0.973824f,
  0.952221f, 0.188018f, 0.985295f, 0.946228f,
};

static const float av1_simple_motion_search_prune_rect_mean_16[25] = {
  8.391692f, 8.303431f, 6.590342f, 6.459725f, 6.460719f, 6.333274f, 6.592615f,
  6.461661f, 6.464787f, 6.337191f, 7.499753f, 7.395166f, 7.503220f, 7.398344f,
  7.498312f, 7.395039f, 7.353743f, 7.253139f, 3.874267f, 0.979701f, 2.087404f,
  2.131698f, 0.981005f, 2.110868f, 2.106539f,
};

static const float av1_simple_motion_search_prune_rect_std_16[25] = {
  1.865867f, 1.870012f, 1.773885f, 1.770447f, 1.972922f, 1.961361f, 1.777224f,
  1.772864f, 1.974519f, 1.962281f, 1.831632f, 1.831837f, 1.837595f, 1.837008f,
  1.822791f, 1.822053f, 2.074991f, 2.067200f, 1.676261f, 0.141022f, 0.840297f,
  0.829935f, 0.136507f, 0.828972f, 0.808563f,
};

static const float av1_simple_motion_search_prune_rect_mean_8[25] = {
  6.997798f, 6.867032f, 5.134819f, 4.883330f, 5.134804f, 4.879707f, 5.140518f,
  4.886751f, 5.142186f, 4.885262f, 6.069946f, 5.896944f, 6.080442f, 5.906130f,
  6.077539f, 5.905929f, 6.083087f, 5.909298f, 3.552709f, 0.990654f, 1.497349f,
  1.531762f, 0.989606f, 1.496581f, 1.484139f,
};

static const float av1_simple_motion_search_prune_rect_std_8[25] = {
  1.727562f, 1.725050f, 1.633396f, 1.618773f, 1.633586f, 1.620657f, 1.620798f,
  1.604892f, 1.621570f, 1.607439f, 1.691024f, 1.684225f, 1.676065f, 1.668442f,
  1.680016f, 1.672452f, 1.677775f, 1.671586f, 1.451902f, 0.096223f, 0.751190f,
  0.754040f, 0.101419f, 0.738239f, 0.729455f,
};

static const float *const av1_simple_motion_search_prune_rect_mean[5] = {
  av1_simple_motion_search_prune_rect_mean_128,
  av1_simple_motion_search_prune_rect_mean_64,
  av1_simple_motion_search_prune_rect_mean_32,
  av1_simple_motion_search_prune_rect_mean_16,
  av1_simple_motion_search_prune_rect_mean_8,
};

static const float *const av1_simple_motion_search_prune_rect_std[5] = {
  av1_simple_motion_search_prune_rect_std_128,
  av1_simple_motion_search_prune_rect_std_64,
  av1_simple_motion_search_prune_rect_std_32,
  av1_simple_motion_search_prune_rect_std_16,
  av1_simple_motion_search_prune_rect_std_8,
};

#define NUM_HIDDEN_LAYERS_128 1
#define NUM_FEATURES_128 25
#define NUM_LAYER_0_UNITS_128 8
#define NUM_LOGITS_128 4

static const float av1_simple_motion_search_prune_rect_logits_kernel_128[] = {
  -0.129103f, 0.457758f,  -0.489986f, 0.65462f,   -0.184312f, 3.81202f,
  -0.444407f, -0.64198f,  -0.575008f, 0.0311711f, 0.525243f,  -20.892f,
  1.08811f,   -65.0976f,  -12.3973f,  -1.38278f,  -0.264233f, 0.241636f,
  -10.6925f,  -0.725414f, -18.8987f,  -40.2284f,  -16.08f,    0.995331f,
  1.47614f,   -0.964864f, 0.405506f,  0.140449f,  0.459534f,  -1.9093f,
  0.398452f,  0.696949f
};

static const float av1_simple_motion_search_prune_rect_layer_0_bias_128[] = {
  1.22789f, -1.34527f, 0.759048f,  0.315086f,
  1.0834f,  -1.58019f, -0.465158f, 1.20716f
};

static const float av1_simple_motion_search_prune_rect_layer_0_kernel_128[] = {
  -0.668677f,  0.58694f,    -0.417094f,   0.754735f,   -0.7859f,
  0.377479f,   -0.0415929f, -0.0140585f,  -0.730001f,  0.747528f,
  -0.135247f,  0.406505f,   -0.234184f,   0.956362f,   -0.637555f,
  0.791884f,   0.0303722f,  1.04424f,     -0.727859f,  -0.274321f,
  -0.122986f,  0.066312f,   -0.00559175f, -0.239643f,  -0.0188767f,
  -0.102787f,  -0.262967f,  0.071882f,    -0.283398f,  0.111607f,
  -0.425826f,  0.02699f,    0.108873f,    -0.180558f,  -0.0794057f,
  0.29665f,    -0.0252969f, -0.0266213f,  -0.277462f,  -0.361973f,
  0.512552f,   0.395011f,   -0.225876f,   0.301924f,   0.136954f,
  0.507259f,   1.23425f,    0.0137135f,   0.662572f,   0.591583f,
  0.101564f,   0.416805f,   -0.645081f,   -0.179086f,  -0.36747f,
  -0.332213f,  0.095177f,   0.220739f,    -0.153256f,  0.706155f,
  0.161701f,   0.696815f,   -1.21531f,    -0.115059f,  0.486764f,
  -0.396093f,  0.784883f,   0.535357f,    -0.278021f,  0.143496f,
  -0.44931f,   -0.144543f,  0.319326f,    0.0190167f,  -0.206295f,
  0.373995f,   -0.247897f,  -0.608095f,   -0.41796f,   -0.137129f,
  -0.709562f,  0.678273f,   0.537607f,    0.557474f,   0.453308f,
  0.21405f,    -0.0466495f, 0.519139f,    -0.168832f,  0.902911f,
  0.681131f,   -0.139876f,  -0.2052f,     -0.393271f,  0.262222f,
  -0.246246f,  -0.213993f,  0.646619f,    0.0496181f,  -0.00354157f,
  0.822927f,   0.0939522f,  0.180738f,    0.118355f,   0.120456f,
  -0.0472214f, -0.144958f,  0.173405f,    -0.886644f,  -0.0949769f,
  -0.813518f,  -0.3947f,    -0.128021f,   0.356196f,   0.469169f,
  -0.413702f,  1.04242f,    0.428853f,    -0.387293f,  0.0850877f,
  0.279409f,   -0.142276f,  0.0579376f,   0.211112f,   0.0703013f,
  -1.9274f,    -0.729147f,  0.534193f,    0.773586f,   0.922864f,
  0.642881f,   1.15127f,    0.621032f,    0.933942f,   1.01837f,
  -0.660282f,  -0.40059f,   -1.11279f,    -0.77088f,   -0.43349f,
  0.202361f,   -0.0840912f, 0.0935707f,   0.056333f,   -0.0779369f,
  0.0173447f,  -0.0104756f, 0.0115005f,   -0.0195593f, 0.03592f,
  -0.343454f,  -0.618048f,  0.258172f,    -0.412322f,  -0.0463746f,
  -0.0413654f, -0.0400194f, 0.615981f,    -0.452094f,  0.644555f,
  0.0822476f,  -0.359791f,  -0.0904274f,  0.209427f,   0.0116338f,
  -0.190978f,  0.890233f,   0.737769f,    -1.66663f,   -0.392605f,
  0.0785728f,  -0.224553f,  -0.128258f,   -0.227227f,  -0.0777773f,
  0.685976f,   0.347042f,   -0.555325f,   -0.249221f,  0.0919837f,
  -0.0660016f, -0.272316f,  0.0390632f,   -0.619624f,  -0.0565801f,
  0.585026f,   0.597375f,   0.54114f,     0.593389f,   0.604391f,
  0.0820294f,  -0.85339f,   -1.40741f,    -0.391675f,  0.0579205f,
  -0.197626f,  0.130044f,   -0.234488f,   -0.0373991f, -0.0717973f
};

static const float av1_simple_motion_search_prune_rect_logits_bias_128[] = {
  1.58571f, -4.6314f, -2.00273f, 0.543699f
};

static const NN_CONFIG av1_simple_motion_search_prune_rect_nn_config_128 = {
  NUM_FEATURES_128,
  NUM_LOGITS_128,
  NUM_HIDDEN_LAYERS_128,
  {
      NUM_LAYER_0_UNITS_128,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_kernel_128,
      av1_simple_motion_search_prune_rect_logits_kernel_128,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_bias_128,
      av1_simple_motion_search_prune_rect_logits_bias_128,
  },
};

#undef NUM_HIDDEN_LAYERS_128
#undef NUM_FEATURES_128
#undef NUM_LAYER_0_UNITS_128
#undef NUM_LOGITS_128

#define NUM_HIDDEN_LAYERS_64 1
#define NUM_FEATURES_64 25
#define NUM_LAYER_0_UNITS_64 32
#define NUM_LOGITS_64 10

static const float av1_simple_motion_search_prune_rect_logits_kernel_64[] = {
  0.10424f,    -0.346025f,  0.534547f,   -0.385925f,  2.58341f,    -0.256414f,
  -0.232498f,  0.329823f,   -0.0777376f, -0.590939f,  0.062657f,   -0.628252f,
  0.0934588f,  2.04029f,    -0.224448f,  0.371168f,   -0.385348f,  -0.589883f,
  -3.73627f,   -0.943144f,  0.346409f,   -0.211215f,  -0.351008f,  0.418807f,
  0.943663f,   0.173267f,   1.16585f,    -0.0840888f, 0.227464f,   0.374412f,
  0.0422597f,  -0.338868f,  0.222576f,   0.431713f,   1.12366f,    0.00753411f,
  0.248412f,   -0.0902425f, 0.542455f,   -0.665629f,  -0.311245f,  -0.205639f,
  -0.447149f,  -0.0502733f, -0.290186f,  -0.794384f,  0.0940881f,  -0.0686117f,
  -0.0199961f, -0.587965f,  0.777096f,   -0.083381f,  -1.21282f,   0.652959f,
  -1.18238f,   0.539991f,   0.352497f,   -0.540076f,  -0.26222f,   -0.568556f,
  0.409102f,   -0.131146f,  -0.407161f,  -0.188287f,  -0.478657f,  0.000401932f,
  -0.689324f,  0.351064f,   -1.43704f,   -0.315185f,  -0.868726f,  0.376341f,
  -0.0566277f, 0.364831f,   0.611298f,   -0.495253f,  -0.0193132f, 0.617978f,
  0.189586f,   -0.236758f,  -0.608246f,  -0.149017f,  -1.78303f,   0.143023f,
  0.698386f,   -0.994086f,  -0.673327f,  0.233868f,   0.360425f,   0.0294123f,
  -0.248683f,  -0.148392f,  0.0861829f,  -0.190843f,  -0.414906f,  0.607378f,
  -0.756715f,  -0.511713f,  -0.321556f,  1.0078f,     -1.18141f,   0.519751f,
  0.834629f,   -0.359343f,  0.612262f,   -0.0730553f, 0.262935f,   0.488276f,
  0.387071f,   -1.44123f,   1.08269f,    0.554402f,   -0.069f,     0.14113f,
  0.323817f,   0.824314f,   -0.431417f,  -0.349448f,  0.950728f,   -0.587836f,
  -0.83914f,   -0.10844f,   0.26602f,    0.831933f,   -0.271315f,  0.231563f,
  0.417049f,   0.190627f,   -0.0940667f, 0.255363f,   -0.0741022f, -0.0987662f,
  -0.847522f,  0.00287554f, 0.0615741f,  -0.0832218f, 0.0847148f,  -0.392843f,
  -0.938068f,  -0.10621f,   -0.260859f,  -0.825175f,  -0.401039f,  0.315213f,
  -0.108269f,  0.288036f,   -8.66166f,   -0.970752f,  -0.66678f,   -0.593405f,
  -0.518294f,  -0.138722f,  -0.454698f,  -0.22969f,   -0.553006f,  -0.440111f,
  0.462661f,   -0.536854f,  0.0108295f,  -0.522888f,  0.00111157f, 0.229999f,
  0.0267768f,  0.176266f,   -1.57043f,   0.0318106f,  0.257534f,   -0.198583f,
  0.175564f,   -0.251465f,  -0.262441f,  -1.65283f,   -0.319603f,  -0.875282f,
  -0.301303f,  0.0170948f,  -0.227075f,  0.0299545f,  -4.98346f,   0.470046f,
  -1.28051f,   -0.213809f,  -0.486585f,  -0.906463f,  -0.169984f,  -0.333153f,
  -0.376733f,  0.108016f,   0.486744f,   -0.186936f,  -0.429259f,  0.056501f,
  -0.266545f,  0.265447f,   -0.137718f,  -0.490687f,  -0.935668f,  -0.16229f,
  -0.696932f,  0.173157f,   0.434959f,   -0.140595f,  0.345845f,   -1.08013f,
  -0.0205929f, -0.815874f,  -0.179812f,  0.02767f,    -0.141727f,  0.471936f,
  -7.29453f,   -1.04362f,   -0.745482f,  -0.28725f,   -0.214997f,  -0.0850651f,
  -0.748471f,  0.161325f,   -1.04387f,   -0.705305f,  0.489427f,   -0.765373f,
  -0.301576f,  0.0742467f,  -0.331282f,  0.0372328f,  -0.90298f,   -0.0608646f,
  -2.18756f,   0.170384f,   -0.258357f,  0.106287f,   -0.161684f,  -0.103799f,
  -0.127774f,  -0.156313f,  0.0705286f,  -0.977908f,  -0.281191f,  -0.056757f,
  -0.309474f,  0.050476f,   -9.78198f,   -2.42795f,   -0.289626f,  -1.07579f,
  -0.439256f,  -1.09948f,   -0.564671f,  0.0913182f,  -0.417216f,  -1.19909f,
  0.287063f,   0.402315f,   -0.17646f,   0.540488f,   0.00840239f, 0.397492f,
  0.702393f,   -0.10566f,   0.655296f,   -0.0443876f, 0.154918f,   -0.760479f,
  -0.0523153f, -0.366199f,  -1.08212f,   -0.398556f,  -0.415203f,  -1.10488f,
  0.208349f,   0.27079f,    0.101546f,   -0.205752f,  -13.7923f,   -0.218637f,
  -1.10077f,   0.355735f,   -0.306196f,  0.627434f,   -0.473101f,  -0.308027f,
  -1.12724f,   0.301597f,   0.660785f,   0.0576217f,  -0.155925f,  -0.56107f,
  -0.223537f,  0.114299f,   -0.53803f,   -0.252674f,  -2.66103f,   -0.185245f,
  -0.314673f,  0.403337f,   0.679821f,   -0.69231f,   0.506264f,   -0.999705f,
  -0.549097f,  0.353745f,   0.188249f,   0.414484f,   -0.615853f,  0.525681f,
  -5.23065f,   -3.05174f,   1.02074f,    -0.965499f,  -0.158947f,  0.0436088f,
  -0.485824f,  0.0375094f,  -1.39985f,   -0.481392f,  0.485785f,   -0.24874f,
  -0.359633f,  0.668108f
};

static const float av1_simple_motion_search_prune_rect_layer_0_bias_64[] = {
  0.0735592f, -0.045064f, -0.0114103f, 1.39246f,    -0.683467f,  0.155765f,
  -0.667652f, -0.202425f, -0.585433f,  -0.146752f,  -0.0812931f, 0.580642f,
  0.578542f,  -0.831916f, 0.610063f,   0.0101856f,  -0.235863f,  0.538141f,
  -2.91334f,  -1.71887f,  0.126616f,   0.582497f,   -0.438879f,  0.221833f,
  0.850773f,  -0.280886f, 0.443233f,   -0.0964873f, -0.216161f,  0.34413f,
  0.656818f,  0.0169274f
};

static const float av1_simple_motion_search_prune_rect_layer_0_kernel_64[] = {
  -0.310947f,   -0.232675f,    0.0171092f,    0.0834474f,   0.373977f,
  0.300429f,    0.215072f,     -0.454074f,    0.187565f,    0.282742f,
  0.562562f,    -0.0419322f,   0.000978486f,  -0.298267f,   0.216934f,
  -0.388722f,   -0.146866f,    -0.275946f,    0.202361f,    0.225847f,
  1.42868f,     0.473127f,     -0.145747f,    -0.104986f,   0.153459f,
  0.69382f,     0.162266f,     0.0207715f,    -0.45095f,    -0.412071f,
  -0.235109f,   -0.130199f,    0.231741f,     0.460193f,    0.0378202f,
  0.429516f,    0.387691f,     -0.272479f,    0.0723884f,   -0.453914f,
  -0.150618f,   -0.10745f,     -0.258615f,    0.0838312f,   -0.00554958f,
  0.105377f,    -0.0415479f,   0.13228f,      1.09044f,     -0.73053f,
  -0.422553f,   -0.435842f,    0.211416f,     0.420332f,    0.0181353f,
  -0.030891f,   0.522788f,     0.613526f,     0.374032f,    0.287986f,
  -0.403118f,   -0.287362f,    -1.11523f,     -0.577713f,   -0.020228f,
  0.86465f,     -0.0590579f,   0.341274f,     -0.0115644f,  -0.260236f,
  0.192123f,    -0.0849825f,   0.0501709f,    0.444382f,    0.0762727f,
  0.0926596f,   -0.101157f,    -0.142787f,    0.40861f,     0.555805f,
  -0.00614654f, -0.122846f,    0.203163f,     0.234266f,    0.409795f,
  -0.0206245f,  -0.224679f,    0.025081f,     0.518044f,    -0.287186f,
  0.016494f,    -0.0886331f,   0.236438f,     -1.01032f,    0.118332f,
  0.364217f,    0.061438f,     0.0381303f,    0.128418f,    0.0257077f,
  -0.975751f,   -0.694894f,    0.00351914f,   0.278179f,    0.29363f,
  0.525576f,    0.0604849f,    0.531734f,     0.406643f,    0.812497f,
  -0.403196f,   -0.16664f,     -0.620887f,    -0.428194f,   0.275401f,
  0.432063f,    -0.00378342f,  0.295758f,     0.105615f,    -0.00683626f,
  0.00396146f,  0.00598654f,   -0.0131701f,   -0.0115787f,  0.00386643f,
  -0.69686f,    -0.139623f,    -0.440817f,    0.0542873f,   0.217962f,
  0.527035f,    -0.0201046f,   0.0471354f,    0.0271858f,   -0.0775197f,
  -0.309797f,   0.184879f,     -0.232854f,    -0.407081f,   0.706227f,
  -0.0877534f,  0.306843f,     0.455075f,     -0.333961f,   0.0759148f,
  0.0444791f,   -0.0693626f,   -0.0850289f,   -0.513063f,   -0.643971f,
  -0.630279f,   -0.153889f,    0.123315f,     0.00548238f,  0.170707f,
  0.734339f,    -0.176988f,    0.322519f,     0.178365f,    0.183519f,
  -0.698683f,   -0.12043f,     -0.349914f,    -0.0696762f,  -0.53986f,
  -0.104738f,   1.05264f,      0.983568f,     -0.109035f,   0.0113748f,
  0.0815189f,   -0.0628812f,   0.0769389f,    0.010261f,    0.146573f,
  -0.433194f,   -0.211572f,    -0.000397392f, 0.445325f,    0.145091f,
  -0.0625902f,  0.29394f,      0.302315f,     0.0892226f,   -0.209504f,
  -0.0150374f,  0.242608f,     0.216223f,     0.366857f,    0.209829f,
  -0.540035f,   0.117599f,     -0.329315f,    0.0471133f,   -0.0115449f,
  -0.0638235f,  0.0527461f,    0.348149f,     0.360802f,    1.06624f,
  -0.615991f,   -0.341396f,    0.18972f,      0.0709888f,   -0.0414466f,
  -0.0193809f,  0.0938933f,    0.209058f,     0.575042f,    0.483608f,
  -0.285875f,   -0.115905f,    -0.363637f,    0.375425f,    0.336217f,
  0.0336358f,   -0.00265618f,  -0.406854f,    -0.792959f,   -0.219354f,
  0.0331615f,   0.0298859f,    -0.211446f,    -0.00280773f, -0.194011f,
  0.262109f,    0.548076f,     0.120183f,     -0.661603f,   0.241855f,
  -0.501428f,   0.00102718f,   -0.347331f,    -0.58306f,    0.0977254f,
  0.117491f,    0.0840667f,    0.00693675f,   0.000600294f, 0.649569f,
  -0.0553811f,  -0.197198f,    0.397236f,     -0.523737f,   -0.564192f,
  -0.374679f,   -0.249344f,    0.00861428f,   0.00393439f,  -0.0834608f,
  0.124389f,    -0.0393049f,   0.0425391f,    -0.153383f,   -0.182346f,
  0.420953f,    0.464221f,     0.288984f,     0.570921f,    -0.239965f,
  0.247239f,    -0.083434f,    0.714418f,     0.986323f,    -0.460244f,
  -0.260993f,   -0.947743f,    -1.0789f,      -0.0391231f,  0.612407f,
  -0.0306767f,  0.281419f,     0.0072426f,    -0.37623f,    0.188744f,
  0.221666f,    -0.424914f,    0.29703f,      0.261715f,    0.277809f,
  -0.0617616f,  -0.000611999f, -0.0547053f,   -0.0901018f,  -0.347669f,
  0.856072f,    0.596675f,     -0.467639f,    -1.09324f,    -0.184224f,
  -0.56051f,    -0.0144704f,   0.102894f,     -0.122982f,   -0.0020749f,
  -0.0423487f,  0.0328702f,    -0.0154263f,   0.0349021f,   -0.00315595f,
  0.0254802f,   -0.729191f,    0.207296f,     -0.0212349f,  -0.207078f,
  0.20636f,     -0.156883f,    0.429765f,     -0.42672f,    0.138775f,
  -0.0267343f,  0.631528f,     0.300646f,     -0.4793f,     -0.273833f,
  -0.0135367f,  -0.530819f,    -0.534881f,    0.830896f,    0.0266992f,
  0.473744f,    0.210334f,     0.0234739f,    0.255394f,    0.123531f,
  -0.489341f,   -0.796627f,    0.372617f,     0.190136f,    0.275342f,
  0.739505f,    0.402354f,     0.782806f,     0.437374f,    1.04948f,
  -0.55963f,    0.382704f,     -0.698321f,    0.0817868f,   -0.440108f,
  -0.0635004f,  -0.277851f,    -0.524194f,    0.286157f,    -0.01097f,
  -0.0293145f,  -0.0405071f,   -0.035662f,    -0.012871f,   -0.0516409f,
  -0.406671f,   0.709259f,     -0.525177f,    0.521123f,    -0.44813f,
  0.48412f,     -0.0546513f,   0.305253f,     -0.468328f,   0.316453f,
  -0.36307f,    0.497515f,     -0.0606276f,   0.315764f,    -0.422066f,
  0.554025f,    -0.679183f,    0.616914f,     0.00283324f,  -0.000643824f,
  0.0639999f,   0.0488285f,    -0.141031f,    0.068003f,    -0.0792678f,
  -0.425307f,   -0.152235f,    0.269917f,     -0.352327f,   0.44792f,
  -0.116514f,   -0.465868f,    0.154287f,     0.0161028f,   -0.16848f,
  -0.255487f,   0.189832f,     0.254883f,     0.0240822f,   0.432638f,
  -0.136564f,   0.137036f,     0.0375734f,    0.989246f,    -0.126287f,
  0.111416f,    -0.0271002f,   0.718755f,     -0.0412969f,  0.00645681f,
  0.253811f,    -0.0186998f,   0.691971f,     -0.282042f,   -0.0783915f,
  0.274592f,    -0.358449f,    0.34155f,      -0.186374f,   -0.136907f,
  -0.192334f,   -0.251168f,    -0.100874f,    -0.166578f,   -0.336507f,
  0.402373f,    0.173695f,     0.108788f,     0.00885581f,  -0.310063f,
  1.05545f,     0.0295867f,    0.180785f,     -0.173469f,   -0.469924f,
  -0.224155f,   0.665862f,     -0.126546f,    0.240691f,    -0.0415301f,
  -0.598534f,   0.0012723f,    -0.122297f,    -0.558947f,   0.268844f,
  0.241193f,    0.0524422f,    -0.1683f,      0.575588f,    -0.139012f,
  0.0636691f,   -0.446709f,    -0.094532f,    0.883809f,    -0.112981f,
  -0.224047f,   0.0811193f,    -0.140571f,    -0.09683f,    -0.0796143f,
  -0.102246f,   -0.863392f,    -0.0755124f,   0.23125f,     -0.0301361f,
  -0.153029f,   -0.172238f,    -0.0286382f,   -0.338495f,   -0.317216f,
  -0.146629f,   -0.242264f,    -0.702306f,    -0.285052f,   0.0623479f,
  0.265735f,    0.00674475f,   0.666196f,     0.883586f,    0.278416f,
  -0.341692f,   -0.509931f,    -0.156263f,    0.635885f,    -0.544143f,
  -0.572632f,   -0.213285f,    0.443396f,     -0.268329f,   0.0638439f,
  -0.185397f,   0.071126f,     0.386503f,     -0.402212f,   -0.140784f,
  -0.411661f,   0.049398f,     -0.0672907f,   -0.267034f,   -0.0560875f,
  0.0607937f,   0.0445484f,    -0.547651f,    0.574718f,    0.417189f,
  -0.0610166f,  0.0632293f,    0.391619f,     -0.00671215f, -0.136883f,
  -0.339346f,   0.0356183f,    0.511993f,     0.178676f,    0.286998f,
  0.136511f,    -0.00796929f,  0.203985f,     0.0423532f,   -0.175196f,
  0.378534f,    0.770417f,     0.593778f,     0.0256067f,   -0.82394f,
  -0.500691f,   -0.425725f,    -0.623708f,    -0.0406241f,  -0.00226464f,
  0.0207836f,   0.30732f,      -0.00784268f,  0.0065445f,   -0.0991039f,
  -0.20871f,    -0.206835f,    0.281219f,     0.119361f,    0.259346f,
  -0.102713f,   0.186488f,     -0.034455f,    -0.00198392f, -0.279107f,
  -0.638993f,   -0.374404f,    -0.48601f,     -0.262345f,   0.624532f,
  0.620632f,    -0.227014f,    0.433579f,     -0.0455096f,  1.22123f,
  -0.429156f,   0.12396f,      0.0815152f,    -0.0837355f,  0.0282623f,
  -0.407475f,   0.787321f,     -0.434974f,    0.312904f,    -0.230805f,
  0.213042f,    -0.250929f,    0.302997f,     -0.354709f,   0.0504905f,
  -0.561706f,   0.595558f,     0.374951f,     0.802969f,    -0.674902f,
  0.33136f,     0.156606f,     0.0218968f,    -0.694188f,   -0.0221949f,
  -0.00639123f, 0.0146536f,    0.0104145f,    0.021635f,    -0.0499428f,
  -0.575116f,   -0.239035f,    -0.0588276f,   0.599722f,    0.541932f,
  0.437433f,    0.716268f,     0.193207f,     0.548351f,    0.326951f,
  -0.197124f,   0.0355353f,    -0.0952009f,   -0.217265f,   -0.389789f,
  0.0528124f,   -0.21334f,     -0.190296f,    -1.17367f,    0.108905f,
  0.109397f,    -0.0192577f,   0.0343813f,    0.085004f,    -0.0556737f,
  -0.0411158f,  -0.534989f,    0.0361896f,    0.124415f,    0.291603f,
  -0.0311974f,  -0.326726f,    0.343131f,     0.0276456f,   -0.231827f,
  -0.373894f,   -0.208898f,    -0.273011f,    0.061323f,    -0.0910538f,
  -0.30746f,    -0.108644f,    -0.190736f,    1.58048f,     -0.0739711f,
  -0.0623489f,  -0.137967f,    -0.0601359f,   -0.133004f,   -0.0857153f,
  0.00955987f,  -0.365561f,    -0.0329051f,   0.463463f,    0.14758f,
  -0.512256f,   -0.227463f,    -0.26008f,     -0.567777f,   0.0646234f,
  1.02161f,     0.66157f,      -0.16733f,     0.264921f,    -0.242036f,
  0.214622f,    0.0712054f,    -0.260377f,    0.0849665f,   0.735094f,
  0.11001f,     0.297301f,     -0.333342f,    0.066978f,    -0.123625f,
  1.07596f,     0.401263f,     0.0800875f,    -0.340862f,   -0.115587f,
  -0.32692f,    -0.300842f,    0.0277397f,    0.0630788f,   -0.261198f,
  0.428695f,    -0.0544757f,   -0.124511f,    0.036992f,    0.126322f,
  0.0317603f,   0.0820762f,    0.117277f,     -1.14594f,    -0.108076f,
  -0.0258198f,  -0.00337525f,  -0.00512531f,  0.1274f,      -0.0660535f,
  -0.640733f,   0.197142f,     0.147278f,     0.489271f,    0.226507f,
  -0.0668414f,  0.0946318f,    0.0994164f,    -0.820516f,   0.512939f,
  -0.305172f,   -0.715187f,    -0.195125f,    0.279346f,    0.462144f,
  0.913882f,    -0.453879f,    0.0582033f,    -0.462866f,   0.0538736f,
  0.0115737f,   0.00626993f,   -0.0185185f,   0.0114601f,   -0.0181164f,
  0.41588f,     -0.0447331f,   0.611756f,     0.43385f,     0.834465f,
  0.122019f,    -0.352983f,    0.340429f,     -0.245425f,   -0.365328f,
  -0.521825f,   0.0371057f,    0.172188f,     -0.387949f,   0.221054f,
  0.0126359f,   0.422958f,     0.584198f,     -0.581498f,   -0.019466f,
  -0.0271737f,  -0.0740885f,   0.00540879f,   0.186086f,    -0.0324402f,
  -0.563462f,   -0.458759f,    -0.425296f,    -0.0118862f,  -0.641508f,
  0.0132084f,   0.0581128f,    0.0231444f,    0.468587f,    0.258838f,
  0.0296665f,   0.0562801f,    0.630014f,     0.381816f,    -0.269761f,
  -0.135515f,   0.046186f,     1.07632f,      -0.050616f,   0.104987f,
  0.29991f,     0.119316f,     0.117248f,     0.0795009f,   0.242573f,
  0.0416634f,   -0.0577639f,   -0.0974078f,   0.106255f,    -0.13098f,
  0.0141486f,   -0.00418257f,  0.144848f,     -0.463934f,   0.0452591f,
  0.252617f,    0.205222f,     -0.189843f,    0.0652245f,   -0.135386f,
  0.0500646f,   -0.200368f,    -0.0142312f,   -0.0286832f,  -0.254355f,
  -1.02752f,    -0.73549f,     0.0364518f,    0.0416227f,   -0.13185f,
  -0.0886515f,  -0.502314f,    -0.102916f,    0.410911f,    -0.355655f,
  0.400416f,    -0.340217f,    0.208829f,     0.245972f,    0.149739f,
  -0.49458f,    0.589482f,     0.550827f,     0.912709f,    -0.351275f,
  -0.128076f,   -0.285172f,    -0.672752f,    0.090583f,    -0.245286f,
  -0.737297f,   -0.201515f,    -0.025122f,    -0.109854f,   0.36738f
};

static const float av1_simple_motion_search_prune_rect_logits_bias_64[] = {
  0.346819f,  0.442965f,  -0.0216032f,  0.0229235f, -0.402797f,
  -0.666074f, -0.455388f, -0.00353411f, -0.595511f, -0.845667f
};

static const NN_CONFIG av1_simple_motion_search_prune_rect_nn_config_64 = {
  NUM_FEATURES_64,
  NUM_LOGITS_64,
  NUM_HIDDEN_LAYERS_64,
  {
      NUM_LAYER_0_UNITS_64,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_kernel_64,
      av1_simple_motion_search_prune_rect_logits_kernel_64,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_bias_64,
      av1_simple_motion_search_prune_rect_logits_bias_64,
  },
};

#undef NUM_HIDDEN_LAYERS_64
#undef NUM_FEATURES_64
#undef NUM_LAYER_0_UNITS_64
#undef NUM_LOGITS_64

#define NUM_HIDDEN_LAYERS_32 1
#define NUM_FEATURES_32 25
#define NUM_LAYER_0_UNITS_32 28
#define NUM_LOGITS_32 10

static const float av1_simple_motion_search_prune_rect_logits_kernel_32[] = {
  0.486581f,    0.340847f,   -0.109226f,   0.467224f,   -0.541561f,
  0.0943619f,   -0.429442f,  -0.207442f,   0.959963f,   0.618666f,
  -0.0636751f,  0.144508f,   -0.0278289f,  0.332293f,   -0.751493f,
  0.245438f,    -0.917758f,  0.612128f,    -0.32648f,   0.534618f,
  -0.615239f,   2.71641f,    0.233759f,    0.820558f,   -0.249758f,
  -0.427783f,   -0.359361f,  0.0375732f,   0.806973f,   0.352512f,
  -0.0532192f,  0.0576861f,  -0.464178f,   -0.334877f,  -0.697042f,
  0.0538218f,   0.0919659f,  -0.00765812f, 0.0603847f,  -0.460315f,
  0.37979f,     -0.0867612f, -0.670683f,   -0.188619f,  -0.570586f,
  0.233418f,    0.153581f,   0.290905f,    -0.624885f,  -0.557842f,
  -0.555567f,   0.463773f,   -0.123909f,   -0.277731f,  0.0374468f,
  0.409903f,    0.287638f,   -0.593066f,   -0.223434f,  0.154263f,
  -0.250464f,   -0.077696f,  0.229652f,    -0.304174f,  0.308053f,
  0.33155f,     -0.502825f,  0.361216f,    -0.499294f,  0.00595444f,
  -0.307201f,   0.5766f,     -0.438384f,   -0.093701f,  -0.118586f,
  0.202337f,    -0.486623f,  0.261552f,    0.139756f,   -0.655642f,
  -0.0627001f,  -0.213053f,  -0.243037f,   0.205918f,   0.0718368f,
  0.188041f,    0.141529f,   -0.132239f,   0.425827f,   -0.218353f,
  0.153114f,    0.33268f,    0.0226116f,   0.167394f,   0.269854f,
  -0.457001f,   0.1973f,     -0.526087f,   0.467528f,   0.290934f,
  1.16267f,     0.0823663f,  -0.754389f,   -0.83716f,   0.270157f,
  -1.41229f,    0.148511f,   -0.286832f,   0.664796f,   0.492254f,
  0.360567f,    -0.533993f,  0.0435672f,   -0.103001f,  0.220668f,
  0.594621f,    -0.0213356f, -0.347638f,   -0.694457f,  0.0759505f,
  0.161358f,    -0.389384f,  -0.0455192f,  -0.61252f,   -0.174173f,
  -0.00788878f, -1.22487f,   0.332233f,    -0.0457021f, -0.225918f,
  -0.197657f,   -0.115408f,  -0.240589f,   -2.05681f,   0.00914629f,
  -1.92213f,    0.0268578f,  -0.49076f,    -0.0120123f, 0.291157f,
  0.267116f,    -0.0775724f, 0.181115f,    -0.392441f,  -0.488114f,
  -0.28842f,    -0.115465f,  0.128974f,    -0.0829899f, -0.14096f,
  -0.140145f,   -0.700281f,  0.0368945f,   -0.437598f,  0.243485f,
  -1.00301f,    0.332324f,   0.125014f,    -0.0604481f, -0.0652028f,
  -0.207295f,   -1.0209f,    -0.341525f,   0.191326f,   -0.147578f,
  0.0878327f,   0.129827f,   -0.0848319f,  0.187381f,   -1.28663f,
  0.00537885f,  -0.134277f,  -0.0411126f,  -0.3434f,    -0.0456494f,
  0.37861f,     0.409095f,   0.237177f,    -0.396855f,  -0.205418f,
  -1.31701f,    -0.319032f,  -0.123404f,   -0.240005f,  -0.305206f,
  -0.0258176f,  -0.26367f,   -0.142396f,   0.191672f,   -1.44061f,
  0.0554776f,   -0.571839f,  -0.284789f,   -0.425677f,  -0.0307376f,
  0.20275f,     -0.223146f,  0.144612f,    0.0212636f,  0.0238303f,
  -0.253802f,   -0.188922f,  -0.0637066f,  -0.340836f,  0.124774f,
  0.130474f,    -0.154099f,  -0.0292733f,  0.158148f,   -0.246989f,
  -0.259059f,   0.220224f,   0.228449f,    -0.41956f,   -0.321848f,
  -0.2396f,     -0.316449f,  -1.3363f,     0.0264099f,  -1.46865f,
  0.113073f,    0.0722885f,  -0.166986f,   -0.164877f,  0.0360911f,
  0.534472f,    -0.551152f,  -0.328501f,   0.0781121f,  -0.378112f,
  -0.459502f,   0.28015f,    -0.212302f,   -0.521641f,  0.618993f,
  -0.347709f,   0.266253f,   -0.0280894f,  0.348511f,   -0.0155031f,
  -0.100693f,   0.0447673f,  0.277519f,    -0.233998f,  -0.0796738f,
  -1.73644f,    -0.160776f,  0.53092f,     -0.180406f,  0.056447f,
  0.385356f,    -0.262337f,  -0.241479f,   -0.271426f,  -0.457354f,
  -0.266788f,   0.367371f,   -0.103065f,   0.47783f,    -0.188327f,
  -0.159636f,   0.00142907f, -0.409756f,   0.454889f,   -0.24566f,
  -0.0760084f,  0.286355f,   0.462102f,    0.0431695f,  -0.127395f,
  -0.200476f,   -0.350557f,  0.217275f,    -0.23975f,   0.255148f,
  -0.280626f,   0.42476f,    0.157411f,    0.0358675f,  -0.192591f
};

static const float av1_simple_motion_search_prune_rect_layer_0_bias_32[] = {
  0.940498f,  0.15602f,   -0.234831f, 0.0268585f, 0.144769f,  0.243081f,
  0.611406f,  0.366093f,  0.361868f,  0.39668f,   0.401479f,  0.369467f,
  0.0909503f, 0.710595f,  0.032786f,  0.525891f,  -1.0232f,   0.732557f,
  -0.064425f, 0.865222f,  -0.042917f, -0.237191f, -0.527006f, -0.0172101f,
  0.59681f,   -0.472405f, 0.0969218f, -0.250624f
};

static const float av1_simple_motion_search_prune_rect_layer_0_kernel_32[] = {
  0.355607f,    0.126701f,    -0.0825159f,  0.200675f,     -0.011308f,
  -0.280057f,   0.559816f,    0.142689f,    0.0422419f,    -0.151692f,
  -0.0275637f,  -0.283101f,   -0.20822f,    -0.200394f,    0.465427f,
  0.344491f,    -0.525319f,   -0.358813f,   -0.39767f,     0.0974486f,
  0.00559058f,  -0.00546089f, 0.0506486f,   0.114475f,     -0.0436463f,
  -0.574152f,   -0.376294f,   0.16563f,     -0.0967032f,   0.00579838f,
  0.0639909f,   -0.037129f,   0.407574f,    -0.231428f,    0.489326f,
  -0.221566f,   -0.270382f,   -0.784628f,   -0.155502f,    0.481698f,
  -0.0296057f,  0.431855f,    0.840807f,    0.112291f,     0.773874f,
  -0.0610936f,  -0.012892f,   0.365154f,    0.0267687f,    -0.0751114f,
  0.25043f,     0.516472f,    -0.186133f,   -0.12762f,     -0.168804f,
  -0.146309f,   0.139314f,    -0.367113f,   -0.601079f,    0.0559856f,
  0.176081f,    0.22397f,     0.434113f,    0.0363256f,    0.313051f,
  0.0143976f,   0.190076f,    0.474607f,    -0.681134f,    -0.0709097f,
  -0.253289f,   -0.216277f,   -0.0593789f,  -0.107795f,    -0.194842f,
  0.513945f,    0.239171f,    -0.720561f,   0.0136723f,    -0.391147f,
  -0.272043f,   -0.164766f,   0.124248f,    0.147178f,     -0.35497f,
  0.397725f,    -0.117603f,   0.262937f,    -0.331964f,    0.182418f,
  0.315671f,    -0.0385649f,  0.488769f,    -0.334568f,    0.00596018f,
  0.0661557f,   -0.0446985f,  -0.0928255f,  -0.0221032f,   -0.019045f,
  -0.20881f,    0.197907f,    -0.381881f,   0.0598071f,    -0.0434551f,
  0.159283f,    -0.110631f,   0.266996f,    -0.0265494f,   0.135199f,
  -0.00833162f, 0.804482f,    -0.114698f,   -0.15066f,     -0.479553f,
  0.448407f,    -0.344069f,   -0.0280952f,  -0.208211f,    -0.102269f,
  -0.679066f,   -0.37476f,    -0.0228875f,  0.0535049f,    0.111015f,
  -0.18125f,    -0.167584f,   0.0110497f,   0.262723f,     -0.413839f,
  -0.0611238f,  0.358499f,    0.0807514f,   0.208254f,     0.214499f,
  0.11137f,     -0.14262f,    -0.0513973f,  0.243718f,     -0.373716f,
  -0.00413366f, 0.216501f,    -0.164149f,   -0.064935f,    -0.0840282f,
  0.0566148f,   0.0377686f,   0.289835f,    0.769388f,     0.891198f,
  -0.592739f,   0.40744f,     -0.153095f,   0.657311f,     0.140737f,
  0.28209f,     0.158344f,    0.353546f,    0.0868246f,    0.116887f,
  0.402004f,    0.437184f,    0.589219f,    0.760594f,     -0.575419f,
  -0.754308f,   -0.709219f,   -0.297814f,   -0.418609f,    -0.0262104f,
  0.0411959f,   0.0597708f,   -0.143728f,   -0.136642f,    0.099614f,
  -0.257601f,   -0.2404f,     0.305893f,    0.254009f,     -0.0301398f,
  -0.0653091f,  -0.459002f,   -0.163404f,   0.123152f,     -0.0284252f,
  -0.457272f,   0.00788622f,  -0.828399f,   -0.0534199f,   0.586877f,
  0.982728f,    0.424581f,    0.0891856f,   0.383182f,     -0.122053f,
  0.0808408f,   -0.00384914f, -0.0560201f,  -0.0524772f,   -0.263444f,
  -0.239287f,   -0.882777f,   0.0180592f,   -0.0948711f,   -0.177946f,
  0.0296473f,   0.096082f,    0.0455604f,   -0.108608f,    0.00777951f,
  -0.140896f,   0.117187f,    -0.342467f,   -0.0691604f,   0.0761611f,
  -0.0892053f,  0.111386f,    -0.167456f,   1.40616f,      -0.00478793f,
  0.00547665f,  -0.0441829f,  0.0151323f,   -0.0674099f,   -0.0380578f,
  0.16072f,     0.31882f,     0.245486f,    -0.424318f,    0.101845f,
  -0.203343f,   -0.197402f,   -0.163025f,   -0.0771961f,   -0.264435f,
  0.319429f,    0.250076f,    0.782726f,    0.386003f,     0.00700673f,
  -0.375715f,   0.151453f,    -0.296265f,   -0.560183f,    -0.00767249f,
  -0.109593f,   -0.119419f,   -0.0161516f,  0.0380283f,    -0.156417f,
  0.131708f,    0.396268f,    -0.221796f,   0.232099f,     0.128852f,
  0.0567268f,   0.297297f,    0.173269f,    0.213411f,     0.0384426f,
  -0.290985f,   -0.0426841f,  -0.488292f,   -0.087101f,    -0.311582f,
  0.83009f,     -0.153163f,   0.903335f,    -1.15644f,     -0.0378635f,
  -0.0552129f,  -0.126362f,   -0.176945f,   0.0653115f,    0.0989368f,
  -0.333543f,   -0.330586f,   0.29775f,     -0.103535f,    0.210824f,
  -0.00300509f, 0.317105f,    0.216852f,    0.479718f,     0.0485808f,
  -0.15662f,    0.718199f,    0.327513f,    0.115169f,     -0.423598f,
  -0.456633f,   -0.575814f,   -0.494454f,   0.304411f,     0.0493055f,
  -0.381171f,   0.467251f,    -0.122872f,   -0.167441f,    0.017253f,
  -0.0583646f,  -0.1586f,     0.214046f,    -0.0284424f,   -0.217112f,
  0.606567f,    -0.107533f,   0.36615f,     -0.0709227f,   0.604761f,
  -0.244657f,   -0.296651f,   -0.595611f,   -0.156629f,    -0.693468f,
  -0.310603f,   0.499272f,    0.282941f,    0.295043f,     -0.178704f,
  0.281186f,    0.014329f,    -0.120819f,   0.154234f,     0.0131325f,
  -0.472231f,   -0.631281f,   0.422955f,    0.711432f,     -0.118025f,
  0.0864996f,   0.343971f,    -0.301477f,   -0.246638f,    0.165068f,
  0.218044f,    0.224236f,    -0.0848522f,  0.00671216f,   0.401141f,
  -0.218857f,   -0.0298495f,  -0.135725f,   -0.377618f,    0.022473f,
  0.106955f,    -0.0582005f,  0.0468484f,   -0.0217442f,   0.130911f,
  -0.0926905f,  0.383007f,    -0.159353f,   -0.222711f,    -0.0286419f,
  0.372315f,    -0.469095f,   0.797571f,    -0.301315f,    0.239327f,
  -0.997507f,   -0.363409f,   0.353717f,    0.676686f,     -0.0500028f,
  0.0638539f,   -0.431927f,   0.243852f,    0.000884826f,  -0.00166585f,
  0.0613292f,   -0.029558f,   -0.0248432f,  -0.0125607f,   -0.0309674f,
  -0.743308f,   0.0409806f,   0.0921015f,   0.167816f,     0.406849f,
  0.095677f,    0.0308913f,   0.139956f,    -0.400472f,    0.396617f,
  0.936517f,    0.355057f,    -0.423816f,   -0.232472f,    -0.220188f,
  -0.399746f,   -0.409623f,   -0.158797f,   0.361153f,     0.0327019f,
  0.0690844f,   -0.032197f,   0.0248558f,   0.00438518f,   0.0222724f,
  -0.326832f,   -0.314295f,   0.156563f,    0.0562703f,    0.332694f,
  0.299424f,    0.228206f,    0.322038f,    0.0136098f,    0.0060297f,
  -0.165851f,   -0.306512f,   0.0796508f,   -0.37158f,     0.239395f,
  -0.349442f,   0.198515f,    -0.253854f,   -1.13694f,     0.0202873f,
  -0.0504009f,  -0.130528f,   -0.017126f,   -0.0370001f,   -0.087458f,
  -0.119952f,   -0.130404f,   0.0333733f,   -0.184736f,    0.182162f,
  0.227776f,    -0.166563f,   -0.156162f,   0.118215f,     -0.220183f,
  0.00474779f,  -0.107792f,   0.260493f,    0.11884f,      0.156587f,
  0.303936f,    -0.131788f,   -0.314774f,   0.310606f,     0.0935523f,
  0.790767f,    0.26461f,     0.0236426f,   0.0629469f,    0.0344072f,
  -0.151513f,   0.211498f,    0.0245435f,   0.0629973f,    0.052019f,
  -0.03308f,    0.123487f,    0.0885027f,   0.159172f,     -0.0510615f,
  0.0298033f,   -0.130515f,   -0.121799f,   -0.104915f,    0.208822f,
  -0.310496f,   -0.314106f,   0.303307f,    -0.0196736f,   0.0420045f,
  0.461777f,    -0.433699f,   0.00345407f,  0.703139f,     -0.655637f,
  -0.210767f,   -0.201278f,   0.163694f,    -0.236534f,    0.300877f,
  0.0769982f,   -0.282453f,   0.149721f,    -0.0303466f,   -0.191473f,
  -0.406056f,   -0.213472f,   0.1619f,      -0.245953f,    0.00544399f,
  -0.121434f,   0.193012f,    -0.307165f,   1.45431f,      -0.161468f,
  -0.12444f,    -0.146129f,   -0.0528212f,  -0.0925165f,   -0.134528f,
  -0.479475f,   0.315525f,    0.133845f,    0.382158f,     -0.0799693f,
  -0.151041f,   0.255772f,    0.409536f,    -0.240663f,    -0.323741f,
  -0.205876f,   0.03699f,     -0.217541f,   0.108511f,     0.640628f,
  0.705993f,    -0.423899f,   -0.78314f,    -0.100733f,    -0.00859087f,
  0.0251879f,   0.0458335f,   0.00210128f,  -0.047576f,    -0.0560518f,
  -1.23869f,    -0.829914f,   0.0346551f,   0.350505f,     0.193688f,
  0.459154f,    0.137898f,    0.503818f,    0.260867f,     0.649539f,
  0.0150802f,   0.0239274f,   -0.276069f,   -0.0621478f,   -0.193106f,
  -0.0375665f,  -0.654529f,   0.189493f,    0.446625f,     -0.0208265f,
  0.019838f,    -0.0201955f,  0.00180428f,  -0.0110678f,   -0.0172414f,
  0.0276489f,   -0.252882f,   -0.0351807f,  -0.0518874f,   0.279098f,
  -0.245122f,   0.101287f,    -0.114202f,   -0.0812187f,   0.572429f,
  -0.0821731f,  0.564183f,    0.0222552f,   0.190111f,     -0.0417497f,
  -0.00385925f, -0.182995f,   -0.240482f,   -0.291572f,    -0.0450444f,
  0.0962974f,   -0.165973f,   -0.0954637f,  -0.163841f,    -0.833405f,
  -1.31541f,    -0.336473f,   -0.0920702f,  0.816105f,     0.393377f,
  0.0340241f,   -0.0844545f,  0.61729f,     -0.17596f,     0.241149f,
  -0.42825f,    -0.59091f,    -0.290702f,   0.0796465f,    0.0982819f,
  0.466934f,    0.261666f,    0.0373333f,   0.332509f,     -0.0266694f,
  -0.0476951f,  -0.00642167f, -0.0132542f,  -0.000320841f, 0.00475532f,
  0.000502778f, 0.296534f,    -0.13297f,    -0.113082f,    -0.327923f,
  0.35901f,     -0.302246f,   0.189799f,    -0.37994f,     0.16107f,
  -0.20414f,    0.548575f,    -0.460821f,   0.591878f,     -0.213113f,
  -0.169373f,   -0.07332f,    0.228841f,    0.682302f,     -0.0665316f,
  -0.142456f,   -0.0873117f,  0.00607451f,  0.0376443f,    0.0536673f,
  -0.0109536f,  -0.400279f,   0.550058f,    0.820871f,     -0.666373f,
  -0.471962f,   -0.315925f,   -0.313142f,   0.952742f,     0.473928f,
  -0.119006f,   0.153241f,    -0.0383078f,  0.631869f,     -0.343423f,
  -0.233473f,   -0.218195f,   -0.077688f,   -0.728291f,    0.0382408f,
  -0.00662886f, -0.0419666f,  0.0309776f,   -0.0281592f,   0.0154229f,
  -0.198534f,   0.0206324f,   0.0152272f,   -0.235067f,    0.0330486f,
  0.139198f,    -0.0612118f,  0.133154f,    -0.258675f,    0.0900275f,
  -0.127771f,   0.157322f,    -0.00767807f, -0.329258f,    0.327458f,
  0.0528581f,   -0.181125f,   0.409995f,    -0.162979f,    -0.0193475f,
  0.186009f,    0.0519501f,   0.651877f,    -0.37821f,     -1.10341f,
  -0.189776f,   -0.0922788f,  0.460256f,    0.168011f,     0.440295f,
  0.478135f,    0.374573f,    0.384048f,    0.116953f,     0.68886f,
  -0.427727f,   -0.36676f,    -0.500013f,   -0.228685f,    -0.218859f,
  0.208396f,    -0.0173765f,  -0.0680241f,  -0.00538013f,  -0.0674409f,
  -0.092764f,   0.0295707f,   -0.0462887f,  -0.00636006f,  0.0334169f
};

static const float av1_simple_motion_search_prune_rect_logits_bias_32[] = {
  0.176459f,  0.154405f, 0.281821f,  0.375264f,  -0.882863f,
  -0.240261f, -1.17075f, -0.280216f, -0.743836f, -0.317511f
};

static const NN_CONFIG av1_simple_motion_search_prune_rect_nn_config_32 = {
  NUM_FEATURES_32,
  NUM_LOGITS_32,
  NUM_HIDDEN_LAYERS_32,
  {
      NUM_LAYER_0_UNITS_32,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_kernel_32,
      av1_simple_motion_search_prune_rect_logits_kernel_32,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_bias_32,
      av1_simple_motion_search_prune_rect_logits_bias_32,
  },
};

#undef NUM_HIDDEN_LAYERS_32
#undef NUM_FEATURES_32
#undef NUM_LAYER_0_UNITS_32
#undef NUM_LOGITS_32

#define NUM_HIDDEN_LAYERS_16 1
#define NUM_FEATURES_16 25
#define NUM_LAYER_0_UNITS_16 32
#define NUM_LOGITS_16 10

static const float av1_simple_motion_search_prune_rect_logits_kernel_16[] = {
  -0.520913f,   0.395611f,    0.0369091f,   -0.318591f,  -0.463252f,
  0.134992f,    -0.43154f,    -0.0739112f,  -0.118817f,  0.476373f,
  -0.281406f,   0.3413f,      0.456255f,    0.33307f,    0.2942f,
  0.1317f,      0.498113f,    1.95406f,     -0.165726f,  -0.219306f,
  -0.302656f,   -1.31157f,    -0.433662f,   0.151716f,   -0.214817f,
  0.504523f,    -0.710049f,   0.359616f,    -0.412695f,  -0.103193f,
  0.341912f,    0.351378f,    -0.181486f,   0.573862f,   -0.0396254f,
  -0.17855f,    -0.276163f,   0.0367465f,   -0.353905f,  -0.204689f,
  0.309581f,    -0.0439686f,  -0.147855f,   0.152745f,   0.290871f,
  0.131049f,    -0.27808f,    -0.142997f,   0.207843f,   -1.23074f,
  -0.267714f,   -0.336923f,   0.313781f,    -0.61488f,   -0.161984f,
  0.238059f,    -0.0879942f,  -0.085543f,   -0.260156f,  -0.13614f,
  -0.242196f,   0.201216f,    -0.248691f,   0.0936671f,  -0.350522f,
  -0.35002f,    -0.156583f,   -0.00579001f, 0.300578f,   -0.341269f,
  -0.290712f,   0.354802f,    -0.31629f,    0.509107f,   -0.236953f,
  -0.0923519f,  0.544509f,    -0.280991f,   -0.017437f,  -0.202721f,
  -0.116388f,   -0.7191f,     0.324586f,    0.254249f,   0.125505f,
  0.00658697f,  -0.333322f,   -0.126537f,   -0.140004f,  -0.0241202f,
  -0.172466f,   0.210035f,    -0.270833f,   0.0579044f,  0.0950352f,
  -0.120382f,   0.063292f,    -0.394925f,   0.482165f,   0.147753f,
  0.331465f,    -0.187444f,   0.1083f,      0.414028f,   0.279238f,
  -0.486889f,   -0.674349f,   -0.313656f,   -0.131186f,  -0.100662f,
  0.238191f,    -1.19083f,    -0.30667f,    -2.4324f,    0.235311f,
  0.108605f,    1.67197f,     0.476157f,    0.30055f,    0.0839538f,
  0.408469f,    -0.473517f,   0.560283f,    -0.0188136f, 0.273824f,
  -0.43707f,    -0.0346978f,  -0.438315f,   -0.0196275f, -0.0567921f,
  -0.220166f,   0.216175f,    -0.0180461f,  0.0116429f,  -0.0096949f,
  -0.32613f,    0.176829f,    -0.243563f,   -0.240972f,  -0.621819f,
  -0.00619648f, -0.145525f,   0.124324f,    -0.0306925f, 0.172208f,
  -2.04631f,    -0.200087f,   -0.594135f,   -0.352303f,  -0.309826f,
  0.0922786f,   -0.698371f,   -0.0366823f,  0.0244036f,  0.338775f,
  -0.115947f,   0.144971f,    -0.0607037f,  -0.762412f,  0.0125584f,
  -0.262427f,   -0.0830273f,  -0.291252f,   -0.176059f,  -0.203983f,
  0.0871455f,   -0.0894925f,  0.0426263f,   -0.060001f,  -0.542355f,
  -0.407837f,   -0.0419273f,  0.226608f,    -0.114844f,  0.158733f,
  -0.187237f,   0.113163f,    -1.86337f,    -0.367544f,  -0.547048f,
  -0.24192f,    -0.226764f,   0.090912f,    0.819604f,   0.433766f,
  -0.841657f,   0.446987f,    -0.622761f,   -0.0296385f, -0.130176f,
  -0.0518136f,  -0.640326f,   -0.330107f,   -0.137832f,  -0.0119033f,
  0.39401f,     0.111331f,    -0.141367f,   -0.230289f,  0.171054f,
  -0.924059f,   -0.107317f,   -0.347983f,   0.0261109f,  0.423002f,
  -0.305817f,   0.247696f,    0.0436002f,   0.0305862f,  -1.52448f,
  -0.595587f,   -0.155552f,   -1.11949f,    -0.513937f,  0.138347f,
  -0.301487f,   0.352144f,    -0.615801f,   0.0326701f,  -0.215322f,
  -0.0608176f,  -0.416557f,   -0.306073f,   -0.441512f,  -0.0569277f,
  -0.709768f,   -0.602527f,   -0.311134f,   0.152471f,   -0.255299f,
  0.354505f,    0.194464f,    0.0144251f,   0.110732f,   -0.4452f,
  -0.804814f,   0.205325f,    -0.0957486f,  0.502684f,   0.09112f,
  -0.533087f,   -1.77979f,    0.556992f,    -0.176157f,  -0.642633f,
  0.11553f,     -0.232561f,   0.161277f,    -0.0631125f, -0.20759f,
  0.489253f,    -0.067533f,   0.0231024f,   -0.179831f,  -0.272985f,
  -0.390059f,   0.3089f,      0.185733f,    -0.257065f,  -0.508838f,
  -0.550028f,   0.0665621f,   -0.138288f,   -0.413188f,  0.191193f,
  -1.32969f,    -0.431025f,   0.270242f,    -0.340062f,  0.0817257f,
  0.0376051f,   -0.18633f,    0.0828274f,   0.00670051f, -0.431295f,
  -0.450316f,   -0.173042f,   -0.322248f,   0.370628f,   0.10019f,
  0.317293f,    -0.266613f,   0.0752441f,   -0.425656f,  -0.112223f,
  0.557991f,    -0.324368f,   -0.195261f,   -0.0526129f, -0.807472f,
  -0.387466f,   0.192186f,    0.353213f,    -0.120238f,  0.107686f,
  0.200678f,    -0.75363f,    0.466857f,    -0.282345f,  -0.0849236f,
  -0.0490695f,  -0.00643182f, 0.123047f,    -0.207805f,  -0.130456f,
  -1.09455f,    0.340973f,    0.334784f,    0.0706643f,  -1.65681f,
  -0.319952f,   -0.198514f,   -0.0787972f,  0.089524f,   0.0531034f,
  -0.202705f,   -0.0852339f,  -0.62572f,    -0.0734234f, -0.838088f
};

static const float av1_simple_motion_search_prune_rect_layer_0_bias_16[] = {
  -0.0616197f, 0.939947f, 0.521161f,  0.213886f,  0.130324f,  -0.127443f,
  -0.0538715f, 0.708746f, 0.445031f,  0.418781f,  -0.114539f, 0.521941f,
  1.13719f,    0.606545f, -0.32193f,  -0.150788f, 0.158487f,  -0.224005f,
  0.654715f,   0.115729f, -0.286506f, -2.06223f,  0.0117697f, 0.503905f,
  -0.102339f,  0.653256f, -0.813561f, 0.905235f,  -0.417269f, -0.206265f,
  0.661496f,   0.95533f
};

static const float av1_simple_motion_search_prune_rect_layer_0_kernel_16[] = {
  -0.203489f,   0.00686229f,  -0.161414f,   0.0637276f,   0.27516f,
  0.512219f,    0.164205f,    0.00326062f,  -0.41914f,    -0.400334f,
  0.554419f,    0.715772f,    -0.295569f,   -0.703503f,   0.0137744f,
  -0.0934259f,  0.174234f,    -0.148618f,   -0.0360558f,  -0.0986598f,
  -0.138502f,   -0.0770713f,  0.122922f,    -0.00784415f, 0.0953234f,
  -0.255754f,   -0.310967f,   0.185306f,    0.464554f,    0.147338f,
  -0.0612304f,  0.164783f,    0.301097f,    0.161364f,    -0.12723f,
  -0.0265984f,  -0.471361f,   0.0578776f,   -0.362865f,   0.425789f,
  0.402758f,    -0.190235f,   0.00549738f,  -0.570908f,   1.27206f,
  0.048868f,    -0.0097675f,  0.0708324f,   0.0456103f,   0.0149062f,
  -0.563032f,   -0.420573f,   0.107278f,    0.0938258f,   0.142712f,
  -0.00251036f, -0.250583f,   0.522272f,    0.0113175f,   0.126751f,
  -0.433028f,   -0.035542f,   -0.536686f,   -0.0668722f,  0.253094f,
  0.254007f,    -0.435505f,   0.343001f,    0.0531542f,   -0.361914f,
  -0.102664f,   0.0404874f,   0.132686f,    0.0762298f,   0.0236971f,
  -0.419454f,   0.230877f,    -0.223714f,   0.037813f,    0.0818604f,
  0.383705f,    -0.235028f,   -0.0554801f,  0.429851f,    0.0845829f,
  0.166295f,    0.355111f,    -0.421197f,   0.298949f,    0.0218224f,
  0.445705f,    -0.392217f,   -0.429578f,   -0.076276f,   -0.0963531f,
  -0.631425f,   -0.225977f,   8.06349e-06f, 0.0676679f,   0.0779651f,
  0.0706891f,   0.101377f,    0.517103f,    0.0945502f,   -0.52522f,
  -0.312022f,   0.0358089f,   0.616509f,    -0.0507444f,  -0.465814f,
  -0.0326024f,  0.591298f,    0.188544f,    -0.0633316f,  -0.199987f,
  0.403118f,    -0.511281f,   -0.696263f,   0.112996f,    0.103875f,
  0.0495595f,   -0.0107449f,  0.521539f,    -0.0123823f,  -0.0642751f,
  0.08548f,     -0.0679207f,  0.526558f,    0.0651114f,   -0.342643f,
  -0.349934f,   0.307437f,    0.368763f,    -0.194851f,   -0.134117f,
  0.102448f,    -0.0520666f,  0.0415824f,   -0.175085f,   0.272685f,
  0.0675856f,   0.120627f,    0.391408f,    -0.135249f,   -0.357024f,
  0.019666f,    -0.0622677f,  0.407427f,    0.22655f,     -0.129432f,
  -0.165327f,   0.004893f,    0.5479f,      0.0613981f,   -0.479682f,
  -0.144228f,   -0.130106f,   0.206458f,    -0.342086f,   0.12691f,
  -0.113554f,   0.231164f,    -0.051419f,   0.0401286f,   -0.560429f,
  -0.070609f,   0.420232f,    0.442465f,    -0.237501f,   -0.000293732f,
  -1.017f,      -0.210222f,   0.0157063f,   0.0488178f,   0.0734721f,
  -0.52626f,    -0.276441f,   -0.521579f,   0.443532f,    -0.0819051f,
  -0.0732633f,  -0.17999f,    0.258525f,    -0.0374872f,  0.150115f,
  0.0510939f,   0.168116f,    0.473372f,    0.824489f,    0.302195f,
  -0.348613f,   0.238569f,    0.176444f,    -0.633945f,   -0.0567195f,
  -0.0305827f,  -0.0551851f,  0.85822f,     -0.0628099f,  0.0364294f,
  -0.234823f,   0.179067f,    0.143208f,    -0.0511014f,  -0.404191f,
  0.428035f,    0.0235506f,   0.371991f,    -0.312909f,   0.550933f,
  -0.389265f,   -0.271813f,   -0.293461f,   -0.583752f,   0.179991f,
  0.191698f,    0.659094f,    1.07941f,     -0.509555f,   -0.100638f,
  0.079988f,    -0.0519107f,  -0.112723f,   -0.0663326f,  0.0353569f,
  -0.795055f,   -0.465999f,   0.283579f,    0.340913f,    0.152738f,
  0.294664f,    0.527839f,    0.187735f,    0.359461f,    0.164629f,
  0.107512f,    0.390402f,    0.236702f,    0.114674f,    -0.525655f,
  -0.555476f,   -0.6589f,     -0.266601f,   -0.0946547f,  0.6306f,
  0.0248513f,   0.038497f,    0.432706f,    -0.0715465f,  0.0410172f,
  -0.115313f,   -0.428684f,   0.136283f,    0.0913185f,   0.11277f,
  0.0968689f,   -0.00437052f, 0.0888981f,   0.10304f,     0.02442f,
  -0.211315f,   0.00981596f,  -0.0974827f,  0.208611f,    0.140644f,
  0.0315567f,   0.350332f,    -0.291049f,   -0.0715449f,  -0.352992f,
  -0.858004f,   0.828658f,    0.439092f,    0.0151291f,   0.0503828f,
  0.0656112f,   -0.710749f,   -0.0951757f,  0.193908f,    0.00908018f,
  0.141486f,    -0.0657711f,  0.099791f,    0.153729f,    -0.419576f,
  -0.892636f,   -0.0449268f,  -0.170786f,   -0.156564f,   0.384511f,
  0.296565f,    0.0569815f,   -0.103938f,   1.27479f,     -0.0406475f,
  0.154083f,    -0.186442f,   0.0282588f,   0.0312102f,   -0.188994f,
  0.284243f,    -0.564693f,   0.425525f,    -0.00924596f, 0.810003f,
  0.233812f,    -0.0180273f,  0.121082f,    -0.209096f,   0.151437f,
  0.286921f,    -0.348095f,   0.174813f,    -0.413798f,   0.108994f,
  -0.34266f,    -0.0337981f,  -0.459f,      -0.409812f,   -0.0890104f,
  0.0834802f,   -0.00259191f, -0.105914f,   -0.164207f,   0.0697689f,
  -0.312098f,   -0.00650536f, -0.486758f,   -0.248486f,   0.24314f,
  -0.0857144f,  0.0884781f,   -0.65615f,    -0.121744f,   0.0709335f,
  -0.0237193f,  0.10764f,     -0.0409452f,  -0.0824305f,  0.42329f,
  0.138258f,    0.502607f,    0.228545f,    0.0687789f,   0.0361586f,
  0.39074f,     0.0722654f,   -0.0133148f,  0.283278f,    0.0743384f,
  0.310292f,    -0.297675f,   -0.359935f,   0.521021f,    -0.10082f,
  -0.272333f,   0.0120283f,   0.138118f,    -0.123711f,   -0.0711386f,
  0.0170747f,   0.831039f,    0.0509626f,   0.790608f,    -0.0863406f,
  -0.31962f,    0.0631013f,   0.0873453f,   -0.472331f,   -0.0826027f,
  -0.241722f,   0.148835f,    -0.131611f,   0.000195347f, -0.0615804f,
  -0.838663f,   -0.586979f,   0.247713f,    0.362254f,    0.492727f,
  -0.132163f,   0.0516545f,   0.477838f,    -0.0395182f,  0.0124993f,
  -0.771514f,   0.0386912f,   -0.118525f,   -0.346172f,   -0.265905f,
  -0.175257f,   -0.406287f,   0.393837f,    0.409096f,    -0.408501f,
  -0.0207146f,  0.0487809f,   0.0636982f,   0.0276368f,   0.0878249f,
  0.0425889f,   0.0868633f,   0.17423f,     -0.128217f,   -0.477068f,
  -0.321294f,   0.0393771f,   0.00812823f,  -0.350529f,   -0.129012f,
  0.439953f,    0.396662f,    0.410475f,    -0.123129f,   -0.565966f,
  0.0298635f,   -0.614611f,   -0.477514f,   0.453651f,    0.0617068f,
  0.0530563f,   0.0479074f,   0.213551f,    0.039034f,    0.0449095f,
  -1.06868f,    -1.2654f,     -0.175482f,   0.595068f,    -0.230095f,
  0.719838f,    -0.272148f,   0.696564f,    0.0485396f,   0.468584f,
  0.0695439f,   -0.0842122f,  -0.228978f,   0.161397f,    -0.000441421f,
  -0.0297514f,  -0.250599f,   0.196656f,    0.608423f,    -0.0112096f,
  0.0236881f,   -0.00167311f, 0.0040709f,   0.015495f,    0.00757698f,
  -0.165886f,   0.359767f,    -0.0214696f,  0.377208f,    0.0303547f,
  0.0657094f,   0.140775f,    0.21867f,     -0.203922f,   0.263878f,
  -0.0529099f,  0.202438f,    -0.243226f,   0.156659f,    -0.627056f,
  -0.845036f,   -0.500873f,   0.172588f,    0.402972f,    -0.147734f,
  0.151792f,    -0.075579f,   0.443519f,    0.0311335f,   -0.0328222f,
  -0.0299781f,  0.435956f,    -0.0987376f,  0.288402f,    0.135902f,
  -0.173584f,   -0.186255f,   0.224524f,    -0.249645f,   0.123702f,
  -0.0846244f,  0.491317f,    0.544846f,    0.338677f,    -0.258885f,
  -0.617434f,   -0.629003f,   -0.347233f,   0.181262f,    -0.0606015f,
  -0.537766f,   0.215089f,    -0.334527f,   0.0488534f,   0.0577997f,
  -1.12431f,    -0.932292f,   -0.11559f,    0.573715f,    0.151128f,
  0.693818f,    -0.16956f,    0.802591f,    -0.231531f,   1.04318f,
  -0.476417f,   0.293452f,    -0.610136f,   0.27506f,     -0.384012f,
  0.305366f,    -0.0540464f,  -0.337583f,   -0.174285f,   0.157248f,
  0.0477345f,   -0.0229535f,  0.0475766f,   -0.00603319f, 0.00856119f,
  -0.702893f,   -0.0579673f,  0.183024f,    -0.166222f,   0.109763f,
  -0.148019f,   -0.258873f,   -0.0820157f,  -0.186716f,   -0.449265f,
  -0.0534138f,  0.15732f,     0.46357f,     0.00502591f,  -0.0282085f,
  0.152277f,    -0.855199f,   -0.357115f,   0.0366159f,   0.0131101f,
  -0.0407758f,  0.0462835f,   0.146309f,    -0.00276278f, -0.0591814f,
  -0.109437f,   0.506764f,    -0.044421f,   0.465907f,    0.114444f,
  -0.241053f,   -0.362649f,   -0.432615f,   0.199989f,    -0.00635866f,
  -0.521886f,   0.0958924f,   -0.485725f,   0.0430527f,   0.069746f,
  0.681091f,    -0.288144f,   0.505671f,    0.0489065f,   -0.0373836f,
  0.266079f,    0.145173f,    -0.011481f,   -0.225074f,   -0.754501f,
  -0.122939f,   -0.294213f,   0.334738f,    0.281561f,    0.558977f,
  -0.21551f,    -0.346507f,   -0.0625635f,  0.0782034f,   -0.236999f,
  -0.803783f,   -0.601117f,   0.091192f,    0.636122f,    -0.250626f,
  0.0354961f,   0.103915f,    0.508571f,    0.329911f,    -0.0425999f,
  -0.0867587f,  -0.0385824f,  1.13914f,     -0.0261992f,  0.00484478f,
  0.124603f,    -0.012173f,   -0.377358f,   -0.243563f,   0.236094f,
  0.145663f,    -0.132752f,   0.347497f,    -0.529315f,   0.271632f,
  -0.372805f,   0.0261836f,   0.126169f,    0.0941008f,   0.283773f,
  0.765701f,    -0.226477f,   -0.181549f,   -0.306896f,   0.110165f,
  -0.0784234f,  -0.0827892f,  -0.0374252f,  -0.0950872f,  -0.451015f,
  -0.995793f,   -0.452663f,   0.293338f,    -0.380865f,   0.032683f,
  0.0178248f,   0.0699194f,   -0.0811722f,  -0.0866096f,  0.139289f,
  0.296604f,    0.192293f,    -0.0589607f,  -0.179878f,   0.00360266f,
  -0.0905794f,  0.136744f,    -0.191555f,   1.31877f,     -0.0592033f,
  -0.158766f,   0.0214746f,   -0.190113f,   -0.116671f,   0.0449292f,
  -0.109533f,   -0.709307f,   0.386424f,    0.40201f,     0.262211f,
  -0.155244f,   0.233988f,    -0.0166317f,  0.462665f,    0.0484462f,
  0.210902f,    -0.352798f,   0.38698f,     -0.228261f,   -0.084309f,
  -0.220751f,   -0.170879f,   -0.352617f,   -1.24277f,    0.266004f,
  -0.0125749f,  -0.0380073f,  0.101838f,    -0.0483024f,  -0.0629178f,
  -0.0695577f,  -0.103439f,   0.242131f,    -0.0796858f,  0.349718f,
  -0.332045f,   0.0138352f,   -0.380235f,   -0.28717f,    -0.176276f,
  0.865903f,    0.36593f,     0.243925f,    -0.422289f,   -0.117327f,
  0.21876f,     0.245393f,    -0.426134f,   -0.186077f,   0.0352515f,
  -0.123742f,   0.249376f,    1.3281f,      0.0707771f,   0.071415f,
  -0.286827f,   -0.131691f,   -0.270881f,   -0.434378f,   0.376064f,
  0.35966f,     0.513374f,    0.439378f,    -0.222716f,   -0.5874f,
  0.487997f,    -0.293271f,   -0.184245f,   -0.037256f,   0.17723f,
  -0.438651f,   0.428184f,    0.112983f,    -0.449287f,   -0.0451963f,
  0.0854929f,   0.0735442f,   -0.0148642f,  -0.0586782f,  -0.176455f,
  -0.438979f,   -0.127109f,   0.211478f,    0.388035f,    -0.0372021f,
  0.220575f,    0.382144f,    0.302121f,    0.0857121f,   0.193445f,
  -0.488858f,   -0.195288f,   -0.316184f,   -0.314026f,   -0.111956f,
  0.0744768f,   0.292709f,    0.30187f,     -0.285506f,   -0.105006f,
  0.0851402f,   -0.082318f,   0.277518f,    0.725294f,    -0.756304f,
  0.0155309f,   -0.378542f,   0.293377f,    -0.347252f,   -0.338458f,
  0.221449f,    -0.176443f,   -0.131972f,   0.0129163f,   -0.290649f,
  0.198596f,    -0.0721333f,  0.620591f,    0.568736f,    0.174001f,
  -0.205186f,   -0.265606f,   -0.249155f,   0.299163f,    1.11842f,
  0.17423f,     0.196417f,    -0.014484f,   0.0735422f,   0.26329f,
  0.12284f,     -0.750305f,   -0.351337f,   0.121994f,    -0.00542878f,
  -0.295707f,   -0.094124f,   0.300993f,    0.412408f,    -0.170761f,
  -0.0676329f,  -0.106638f,   -0.419785f,   -0.43878f,    0.22421f,
  0.0339903f,   0.619851f,    0.0615381f,   0.514631f,    1.35424f,
  -0.0679228f,  -0.203457f,   0.131948f,    -0.0041251f,  -0.209054f
};

static const float av1_simple_motion_search_prune_rect_logits_bias_16[] = {
  0.304025f,  0.131887f, 0.259279f,  -0.561564f, -0.161729f,
  -0.208036f, 0.102206f, -0.162937f, -1.42311f,  -0.708305f
};

static const NN_CONFIG av1_simple_motion_search_prune_rect_nn_config_16 = {
  NUM_FEATURES_16,
  NUM_LOGITS_16,
  NUM_HIDDEN_LAYERS_16,
  {
      NUM_LAYER_0_UNITS_16,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_kernel_16,
      av1_simple_motion_search_prune_rect_logits_kernel_16,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_bias_16,
      av1_simple_motion_search_prune_rect_logits_bias_16,
  },
};

#undef NUM_HIDDEN_LAYERS_16
#undef NUM_FEATURES_16
#undef NUM_LAYER_0_UNITS_16
#undef NUM_LOGITS_16

#define NUM_HIDDEN_LAYERS_8 1
#define NUM_FEATURES_8 25
#define NUM_LAYER_0_UNITS_8 32
#define NUM_LOGITS_8 4

static const float av1_simple_motion_search_prune_rect_logits_kernel_8[] = {
  -0.266303f,  -0.387676f,  0.204501f,   -0.120842f,  -0.0752326f, 0.0337739f,
  0.0243477f,  -0.356748f,  0.0143051f,  -0.16403f,   -0.139013f,  0.175003f,
  -0.206754f,  0.349059f,   0.181763f,   0.212768f,   -0.313783f,  0.182829f,
  0.00205376f, -0.939525f,  -0.0992424f, 0.306254f,   0.083329f,   -0.133137f,
  -0.179022f,  -0.0237902f, 0.0601026f,  -0.216698f,  -0.551149f,  0.081711f,
  -0.442191f,  0.0680832f,  -0.0353678f, 0.237704f,   0.23155f,    -0.36097f,
  0.123389f,   -0.288927f,  0.178133f,   -0.152222f,  -0.235648f,  -0.0495293f,
  -0.316522f,  0.034207f,   0.0463139f,  -0.817825f,  0.417443f,   -0.110984f,
  -0.402371f,  0.0341694f,  -0.37383f,   0.414532f,   0.093993f,   0.0039505f,
  0.0803175f,  -0.511859f,  -0.0154802f, 0.0979595f,  0.0909049f,  -0.120938f,
  -0.577382f,  -0.155041f,  -0.404295f,  0.122223f,   -0.084703f,  0.00415336f,
  0.149135f,   0.113219f,   0.124236f,   -0.240905f,  0.163909f,   -0.154202f,
  -0.208917f,  0.00200158f, -0.71796f,   0.105984f,   -0.131996f,  -0.539603f,
  0.223768f,   -0.0710733f, -0.346679f,  -0.0745909f, 0.171032f,   0.215701f,
  0.218519f,   0.105981f,   -0.096209f,  -0.166453f,  -0.468894f,  -0.401578f,
  -0.239222f,  0.111382f,   0.38747f,    -0.164734f,  -0.175955f,  0.336621f,
  -0.0305501f, -0.0576765f, 0.0672671f,  -0.183692f,  0.412082f,   -0.262951f,
  -0.153429f,  -0.128589f,  -0.530472f,  0.0936412f,  -1.08296f,   -0.45147f,
  0.0714904f,  -3.96842f,   0.438125f,   -0.313945f,  0.231104f,   -0.00183851f,
  -0.0192768f, -0.637531f,  -0.109296f,  0.0531702f,  0.00262162f, -0.615951f,
  -0.546241f,  -0.635305f,  -0.0762367f, 0.0122019f,  0.423693f,   -0.129142f,
  -0.112242f,  0.295184f
};

static const float av1_simple_motion_search_prune_rect_layer_0_bias_8[] = {
  -2.16023f,  -3.12831f, -0.213206f,  -2.97875f, -1.83791f, -2.84713f,
  -0.909636f, -2.05893f, 0.00525274f, -1.51672f, -3.95017f, 1.82847f,
  -0.853224f, -3.29503f, -0.537517f,  0.923106f, -3.18665f, -1.29905f,
  1.64506f,   -1.99848f, -2.24315f,   0.408613f, 0.503671f, -3.83393f,
  -2.88388f,  -3.52337f, 1.46818f,    -1.67169f, -3.83253f, 1.52644f,
  -0.490783f, -0.415782f
};

static const float av1_simple_motion_search_prune_rect_layer_0_kernel_8[] = {
  -0.702198f,  -0.102148f,   0.0564545f,   -0.0555548f,  0.16184f,
  0.0950792f,  0.136974f,    -0.00824146f, 0.05746f,     0.0447542f,
  0.145978f,   0.0855769f,   -0.041449f,   0.301347f,    -0.0206691f,
  -0.0662514f, -0.0525079f,  -0.0998387f,  -0.0891438f,  0.110545f,
  -0.863098f,  -1.83798f,    0.238818f,    0.127797f,    0.116872f,
  -0.270655f,  -0.21057f,    0.197013f,    -0.123332f,   0.137104f,
  -0.174766f,  -0.00803025f, 0.0234369f,   -0.0894175f,  -0.0380927f,
  0.00827928f, -0.134148f,   0.110575f,    -0.250173f,   0.116273f,
  0.0197749f,  0.270391f,    0.108437f,    0.173197f,    -0.0650348f,
  0.0884626f,  0.262792f,    0.0649228f,   0.5573f,      -2.81315f,
  -0.479801f,  -1.15825f,    0.0807932f,   -0.19144f,    0.404016f,
  -0.211521f,  0.233269f,    -0.391414f,   0.160381f,    -0.277233f,
  0.426354f,   0.156839f,    0.494315f,    -0.214259f,   -0.0132062f,
  0.148628f,   -0.0899568f,  0.161845f,    0.467689f,    0.229474f,
  0.590634f,   -0.705793f,   -0.0486113f,  -0.439088f,   0.994566f,
  0.679065f,   0.777869f,    -0.225291f,   -0.0303006f,  -0.638782f,
  -0.0824632f, -0.128561f,   -0.327603f,   0.105624f,    0.567581f,
  -0.396135f,  -0.471028f,   0.181286f,    0.274604f,    0.180169f,
  0.0612144f,  -0.865004f,   0.0306804f,   0.142985f,    -0.0914358f,
  -0.243284f,  0.358359f,    -0.443847f,   -0.371978f,   0.606933f,
  -0.900408f,  -0.52076f,    0.472118f,    0.0610973f,   0.152526f,
  -0.550379f,  0.309331f,    -0.141573f,   0.203046f,    -0.231485f,
  0.505156f,   0.393224f,    0.435487f,    -0.218681f,   0.123707f,
  -0.270383f,  -0.033565f,   0.210373f,    -2.33967f,    0.367434f,
  0.0308118f,  -0.205771f,   0.546141f,    0.19837f,     0.035648f,
  -0.467007f,  -1.50995f,    -0.0314176f,  0.11762f,     -0.15307f,
  0.618257f,   -0.139502f,   0.303386f,    -0.00758681f, 0.228107f,
  -0.594499f,  -0.201984f,   -0.239666f,   0.114878f,    -0.922174f,
  -0.530137f,  -0.379366f,   -0.319582f,   0.0889624f,   -0.00544663f,
  0.316264f,   -0.204262f,   -0.0959358f,  0.23552f,     0.141369f,
  -0.207129f,  -1.04067f,    -0.0780501f,  0.226768f,    -0.246752f,
  0.0823105f,  0.114783f,    0.49315f,     0.0197732f,   0.705433f,
  0.158076f,   -0.250584f,   -0.157326f,   -0.0439547f,  -0.139047f,
  0.090531f,   -0.38833f,    0.743143f,    -1.47418f,    -0.155009f,
  0.511466f,   -0.726716f,   -0.181075f,   0.450133f,    -0.390204f,
  0.292725f,   0.00811462f,  -0.347738f,   0.613381f,    -0.237124f,
  0.750748f,   -0.383123f,   0.410309f,    -0.204166f,   0.667199f,
  -0.313197f,  0.436059f,    -0.607571f,   0.193681f,    0.409399f,
  0.631747f,   -0.0454149f,  0.198232f,    0.345591f,    -0.0137374f,
  -0.307014f,  -0.535515f,   0.764678f,    -0.225686f,   -0.451621f,
  -2.75564f,   -1.52877f,    0.0511933f,   0.905979f,    0.145029f,
  0.759615f,   0.130166f,    0.83827f,     0.0655081f,   1.07555f,
  -0.529777f,  0.682967f,    -0.412052f,   0.611947f,    -0.83676f,
  0.940695f,   -0.465681f,   0.51505f,     -0.883659f,   -0.105524f,
  -0.0344173f, -0.0683618f,  -0.00698688f, -0.139349f,   0.135741f,
  -0.294455f,  -0.377834f,   -0.602084f,   -1.00128f,    0.483291f,
  1.25327f,    0.178987f,    0.75068f,     -0.520731f,   -0.325517f,
  0.272032f,   0.144144f,    -0.279453f,   0.564907f,    0.144036f,
  0.297448f,   -0.504243f,   -0.250508f,   -1.26395f,    0.4816f,
  0.392771f,   -0.389961f,   -0.261585f,   -0.127124f,   -0.202945f,
  -0.709716f,  -0.174719f,   0.113613f,    0.477753f,    -0.226659f,
  0.0697828f,  -0.177994f,   0.300726f,    -0.185504f,   0.339424f,
  -0.316746f,  0.369693f,    -0.339723f,   -0.143886f,   -0.0326589f,
  -0.268761f,  -0.241094f,   0.284876f,    -0.0270867f,  -0.207397f,
  -1.42738f,   0.495612f,    -0.0277732f,  0.199675f,    1.48638f,
  -0.659257f,  -1.28199f,    0.498702f,    0.140695f,    0.571152f,
  0.416368f,   0.14153f,     0.126876f,    0.521114f,    -0.00150571f,
  0.375581f,   0.00537624f,  0.1286f,      -0.332227f,   0.417663f,
  -0.539023f,  0.217124f,    -0.787111f,   -0.0335266f,  1.56751f,
  0.0640563f,  -0.158791f,   0.118195f,    0.000970493f, -0.0403852f,
  -0.0572557f, -0.0201181f,  -0.10255f,    0.63237f,     0.156662f,
  0.418696f,   -0.274802f,   -0.663923f,   -0.375232f,   -0.40846f,
  0.462092f,   1.2176f,      -0.301532f,   -0.779704f,   -0.112876f,
  0.0806591f,  -0.0141923f,  0.00960801f,  -0.663557f,   0.0979948f,
  -0.0575999f, -0.012847f,   0.0403853f,   -0.133666f,   -0.00330217f,
  -0.931518f,  -0.774599f,   -0.21391f,    0.377601f,    -0.183365f,
  0.299094f,   0.0238552f,   0.206716f,    -0.18959f,    0.346013f,
  -0.150991f,  -0.192817f,   -0.293962f,   -0.0537604f,  -0.0648171f,
  -0.275941f,  -0.144854f,   -0.224092f,   2.43113f,     0.0422494f,
  -0.047236f,  -0.0262028f,  0.0282119f,   -0.175553f,   0.0888502f,
  0.580682f,   0.951055f,    -0.284441f,   -0.120133f,   -0.268058f,
  -0.312083f,  -0.411556f,   0.21431f,     -0.28033f,    0.324851f,
  -1.02787f,   -0.936816f,   -0.577628f,   0.544743f,    0.295807f,
  0.406157f,   0.447927f,    0.25369f,     -0.811421f,   -0.0424979f,
  -0.189867f,  0.00778673f,  -0.113587f,   -0.116175f,   -0.0542222f,
  -1.80089f,   -1.44175f,    -0.35332f,    0.191314f,    -0.236691f,
  -0.0261926f, -0.502363f,   0.252278f,    -0.485478f,   0.296495f,
  0.455612f,   -0.0489631f,  0.227255f,    0.170975f,    0.473487f,
  0.257812f,   0.178048f,    0.2506f,      2.04637f,     -0.173857f,
  0.0583379f,  0.00765589f,  -0.025772f,   -0.162666f,   -0.016214f,
  -0.607486f,  -0.0808025f,  0.0551611f,   -0.0772291f,  0.126421f,
  0.10869f,    -0.0877463f,  -0.111527f,   -0.0775766f,  0.503886f,
  -0.002757f,  -0.0421354f,  -0.247857f,   0.140827f,    0.383576f,
  0.228232f,   -0.157877f,   -0.0927911f,  0.344687f,    0.191181f,
  0.236533f,   0.00102869f,  -0.0184502f,  -1.4509f,     -1.15945f,
  -0.521978f,  -0.643225f,   0.133139f,    0.0660321f,   0.0851957f,
  0.0303648f,  0.0296239f,   0.0455713f,   0.175647f,    0.080532f,
  0.0445691f,  -0.257356f,   -0.125602f,   -0.138829f,   -0.167057f,
  -0.0992552f, -0.13944f,    0.507531f,    0.444997f,    0.221452f,
  -0.308384f,  -0.327554f,   0.13235f,     2.1487f,      -1.15453f,
  -0.280239f,  -0.363582f,   -0.00358745f, 0.012866f,    0.251088f,
  0.0676416f,  0.178492f,    -0.136631f,   0.197938f,    -0.078198f,
  0.812439f,   1.1173f,      0.712113f,    1.10124f,     -0.836503f,
  -1.22433f,   -1.07894f,    -1.29215f,    0.56057f,     2.23928f,
  -0.419029f,  0.282178f,    -0.0719266f,  -0.172192f,   0.28034f,
  -2.99124f,   -2.01481f,    0.0688982f,   0.697466f,    0.00635555f,
  0.566069f,   0.047534f,    0.507755f,    -0.00690707f, 0.712594f,
  -0.191467f,  0.355733f,    -0.480016f,   0.664669f,    -0.390619f,
  0.351199f,   -0.482342f,   0.325005f,    1.9089f,      0.155987f,
  0.17032f,    0.132729f,    0.0402649f,   0.146991f,    0.0314905f,
  -0.775316f,  -0.208892f,   -0.105993f,   0.0181653f,   -0.12735f,
  0.0897852f,  0.0470231f,   0.25807f,     0.127406f,    -0.0893252f,
  -0.279776f,  0.190844f,    0.110384f,    -0.148833f,   0.025293f,
  0.239838f,   0.00932245f,  0.35103f,     -0.128268f,   -0.0536754f,
  0.506899f,   -0.16793f,    0.0955582f,   -2.01108f,    0.721433f,
  -2.31413f,   -2.08646f,    0.033315f,    0.689828f,    -0.271213f,
  0.790425f,   -0.114234f,   0.755325f,    -0.211533f,   0.774544f,
  -0.263268f,  0.795762f,    -0.551455f,   0.953602f,    -0.168454f,
  0.529055f,   -0.768991f,   0.882371f,    0.29763f,     -0.155017f,
  0.00464101f, 0.121093f,    0.948271f,    0.113138f,    -0.110332f,
  -2.0492f,    -1.31322f,    -0.129212f,   0.464778f,    -0.181465f,
  0.618403f,   0.0627984f,   0.465228f,    0.165729f,    0.278277f,
  -0.563276f,  -0.358358f,   -0.590638f,   0.0104993f,   0.731206f,
  0.752569f,   0.631615f,    0.811822f,    0.129804f,    -0.0558327f,
  0.570081f,   -0.417922f,   -0.168275f,   0.0703671f,   0.269127f,
  0.240457f,   -0.197159f,   -0.00179261f, 0.220065f,    0.463511f,
  0.0714626f,  -0.716477f,   -0.441865f,   -0.717028f,   -0.149176f,
  0.452182f,   0.662699f,    -0.906534f,   -0.817133f,   0.237747f,
  0.26024f,    -7.7441e-05f, 0.0934616f,   0.824641f,    -0.0404494f,
  -0.088297f,  -0.157899f,   0.037408f,    0.132435f,    -0.316155f,
  -0.276785f,  0.0117868f,   0.185008f,    0.32369f,     -0.465855f,
  -0.302127f,  0.303289f,    0.338597f,    -0.665408f,   -0.507594f,
  0.526979f,   0.532091f,    0.234395f,    0.754063f,    0.116769f,
  0.0800309f,  -0.939344f,   -1.51269f,    1.4583f,      0.178444f,
  0.0106756f,  -0.213468f,   -0.00369439f, 0.071015f,    -0.192798f,
  -0.0933147f, -0.129901f,   -0.368279f,   -0.246564f,   0.126966f,
  0.478565f,   -0.476246f,   -0.762863f,   0.168883f,    0.536136f,
  -0.272969f,  0.2573f,      -0.161577f,   0.311428f,    -0.777994f,
  -1.29752f,   0.216046f,    0.329016f,    1.57265f,     0.168075f,
  -0.192518f,  0.0829308f,   -0.073533f,   -0.0202034f,  0.114716f,
  -0.34888f,   -0.519215f,   0.190809f,    0.0138507f,   0.133635f,
  0.14194f,    0.410618f,    -0.165106f,   0.214438f,    0.0438265f,
  -0.8481f,    -1.19182f,    -1.07878f,    -0.882217f,   0.45616f,
  0.977385f,   0.74929f,     0.918466f,    0.904704f,    0.041938f,
  0.0362776f,  0.0757255f,   1.14007f,     0.0516825f,   -0.160068f,
  0.219535f,   0.638634f,    -0.0284544f,  -0.222849f,   -0.0344915f,
  -0.0350256f, -0.0504452f,  -0.0458416f,  0.146099f,    0.0783083f,
  0.206579f,   0.241264f,    0.28401f,     0.0425312f,   -0.802049f,
  -0.746271f,  -0.578969f,   -0.078218f,   0.436176f,    -0.281465f,
  -2.5539f,    0.237868f,    -0.121796f,   0.0715619f,   0.106992f,
  -0.621862f,  -0.167142f,   0.153716f,    0.0570912f,   -0.06525f,
  -0.923773f,  0.130759f,    0.0517066f,   0.0729862f,   -0.873064f,
  0.0403328f,  -0.186499f,   -0.0831918f,  -0.223723f,   0.144697f,
  0.212845f,   0.416876f,    0.361598f,    0.138229f,    0.0728777f,
  -1.95419f,   -0.00382816f, -0.0440387f,  0.433627f,    0.44781f,
  -1.05229f,   -1.54506f,    0.564827f,    -0.263456f,   0.296105f,
  -0.158055f,  0.388274f,    -0.366639f,   0.212006f,    -0.245619f,
  0.593064f,   0.088727f,    0.410632f,    -0.263462f,   0.507075f,
  -0.0974155f, 0.275268f,    -0.1293f,     0.136679f,    1.98276f,
  0.411766f,   0.391987f,    0.34283f,     -0.114077f,   0.258462f,
  -0.302443f,  0.301138f,    -0.00726621f, 0.276441f,    -0.291582f,
  0.66498f,    -0.321451f,   -0.332805f,   0.0943272f,   0.572253f,
  -0.45818f,   -0.0219593f,  -0.151679f,   0.402033f,    -1.15502f,
  -0.882955f,  0.772904f,    0.88126f,     -0.149555f,   0.709525f,
  0.350116f,   -0.21531f,    0.797893f,    0.0230234f,   0.0203034f,
  0.2744f,     1.08273f,     0.039349f,    0.503909f,    -0.45892f,
  -0.579516f,  -0.344058f,   0.390628f,    -0.386941f,   -0.430317f,
  -0.0807066f, 0.435906f,    0.522996f,    0.724476f,    -0.74371f,
  -0.05376f,   -0.340898f,   -0.962646f,   -0.0278005f,  0.0981149f,
  -0.0811161f, 0.00237994f,  0.850042f,    0.0665473f,   0.134413f
};

static const float av1_simple_motion_search_prune_rect_logits_bias_8[] = {
  1.63404f, -0.715866f, -1.0132f, -2.08745f
};

static const NN_CONFIG av1_simple_motion_search_prune_rect_nn_config_8 = {
  NUM_FEATURES_8,
  NUM_LOGITS_8,
  NUM_HIDDEN_LAYERS_8,
  {
      NUM_LAYER_0_UNITS_8,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_kernel_8,
      av1_simple_motion_search_prune_rect_logits_kernel_8,
  },
  {
      av1_simple_motion_search_prune_rect_layer_0_bias_8,
      av1_simple_motion_search_prune_rect_logits_bias_8,
  },
};

#undef NUM_HIDDEN_LAYERS_8
#undef NUM_FEATURES_8
#undef NUM_LAYER_0_UNITS_8
#undef NUM_LOGITS_8

static const NN_CONFIG
    *const av1_simple_motion_search_prune_rect_nn_config[5] = {
      &av1_simple_motion_search_prune_rect_nn_config_128,
      &av1_simple_motion_search_prune_rect_nn_config_64,
      &av1_simple_motion_search_prune_rect_nn_config_32,
      &av1_simple_motion_search_prune_rect_nn_config_16,
      &av1_simple_motion_search_prune_rect_nn_config_8,
    };

// nn model for predicting max square partition level of a superblock
#define NUM_HIDDEN_LAYERS 1
#define NUM_FEATURES 13
#define NUM_LAYER_0_UNITS 48
#define NUM_LOGITS 4

static const float av1_max_part_pred_logits_kernel[] = {
  -0.304561f,   0.0885596f,   -0.988539f,   1.08147f,    0.215213f,
  0.202965f,    -0.828457f,   -0.233945f,   -0.0866977f, -0.115521f,
  0.02079f,     0.196491f,    -0.0285075f,  0.05067f,    -0.00872862f,
  0.00281844f,  -0.238954f,   0.0253801f,   0.0257775f,  0.339269f,
  0.176174f,    -0.152545f,   -0.0588704f,  -1.62275f,   -0.189329f,
  0.0808033f,   0.233844f,    -4.53798f,    0.674968f,   -0.0361688f,
  -0.0754075f,  1.16129f,     -0.0188879f,  0.113255f,   -3.04378f,
  0.814728f,    -0.568517f,   -0.00179383f, -3.61223f,   -1.67535f,
  -2.20417f,    -0.197196f,   0.0507745f,   -0.0909394f, -0.0507879f,
  -1.27999f,    -0.055623f,   0.0318497f,   0.192867f,   0.138726f,
  0.0443392f,   -0.595075f,   -0.166774f,   0.0882958f,  -0.348161f,
  0.0214428f,   -0.0599275f,  -0.0995385f,  -0.82358f,   0.141205f,
  -0.053232f,   0.00508296f,  -1.90872f,    1.15004f,    -0.194219f,
  0.0229019f,   -0.00354318f, 0.22016f,     0.154101f,   -0.159231f,
  -0.0446647f,  -0.197503f,   0.0408453f,   0.197659f,   0.797858f,
  -0.189722f,   0.343653f,    0.124666f,    -1.03083f,   0.603059f,
  0.101565f,    0.0932993f,   0.462484f,    0.295984f,   1.11198f,
  0.143709f,    -0.846232f,   -0.464392f,   -1.06058f,   -0.124889f,
  0.0727475f,   1.18446f,     -0.100302f,   0.0641918f,  -0.101622f,
  0.10219f,     0.130189f,    0.0915623f,   -0.166904f,  -1.10606f,
  -0.16726f,    -0.146152f,   0.145443f,    -0.177091f,  -0.0215214f,
  0.0158506f,   -0.553294f,   0.0784749f,   -0.0416628f, -0.027785f,
  0.280027f,    0.484898f,    -0.164225f,   0.0238317f,  -0.0345254f,
  0.0410244f,   0.131529f,    0.0239622f,   -0.0749436f, -0.0224914f,
  0.128926f,    0.224539f,    0.413297f,    0.0638572f,  0.103308f,
  0.0913242f,   -0.119274f,   0.0163103f,   0.113828f,   0.119809f,
  0.297057f,    -0.124889f,   -0.533108f,   -0.181408f,  -0.129896f,
  0.0221064f,   -0.0773281f,  -0.0386467f,  0.0342961f,  0.126575f,
  -0.24114f,    0.0735576f,   0.0524791f,   0.246896f,   -0.130674f,
  -0.03979f,    0.173639f,    1.95193f,     -0.113029f,  -0.0305852f,
  -0.00671737f, 0.157159f,    -0.00102858f, -0.543688f,  0.566772f,
  0.124124f,    -0.0294064f,  -0.0699021f,  -0.0704103f, -0.766097f,
  -0.0625802f,  -0.0906173f,  -0.0520414f,  -0.0272724f, 0.283064f,
  0.236213f,    -0.127319f,   0.019392f,    0.170042f,   -0.0214542f,
  0.0740938f,   0.356578f,    -0.236257f,   0.269021f,   0.114759f,
  -0.641166f,   0.136308f,    -0.0386959f,  -0.112024f,  -0.361209f,
  0.686095f,    0.183906f,    0.288656f,    0.182007f,   0.337458f,
  0.058974f,    -0.305512f,   -0.841708f,   -0.243779f,  -0.0614058f,
  0.208747f,    0.448697f
};

static const float av1_max_part_pred_layer_0_bias[] = {
  -0.776544f, -2.0022f,    -0.330294f, 2.47665f,  1.90206f,   -1.61571f,
  0.536246f,  1.00455f,    5.24561f,   1.55111f,  -0.816399f, -4.88703f,
  -1.06417f,  -1.15359f,   -0.145289f, 1.91831f,  0.630915f,  -1.94256f,
  -3.35239f,  -1.05007f,   -1.05186f,  1.36824f,  -5.2878f,   1.10482f,
  -5.00077f,  -0.0445198f, 3.41427f,   2.3439f,   -0.413306f, -1.88152f,
  -2.28638f,  8.24783f,    -1.91961f,  -1.49324f, 1.96599f,   -6.32309f,
  -0.332426f, -0.425506f,  4.06511f,   5.84386f,  4.15747f,   1.22402f,
  2.8512f,    2.53027f,    0.0170272f, -1.43966f, -0.997785f, 5.43064f
};

static const float av1_max_part_pred_logits_bias[] = { -4.25432f, 0.144758f,
                                                       1.96217f, 0.728905f };

static const float av1_max_part_pred_layer_0_kernel[] = {
  0.992471f,    0.533006f,    0.143743f,     -2.51788f,    -0.468337f,
  -0.201376f,   -0.151834f,   0.479883f,     1.16061f,     -0.278878f,
  -0.814954f,   -0.152405f,   -0.0521608f,   0.797104f,    -2.08912f,
  0.385839f,    -2.22889f,    -0.106858f,    -0.239766f,   -0.951128f,
  -0.698753f,   0.0831051f,   1.1702f,       0.342834f,    -0.0352795f,
  -0.0847639f,  -0.802086f,   0.258982f,     1.14174f,     0.645885f,
  -1.19226f,    -0.592888f,   -0.343659f,    1.1912f,      1.45411f,
  -1.22927f,    0.152858f,    0.00373585f,   -1.60637f,    0.592611f,
  0.0857475f,   -0.346147f,   -0.150784f,    -0.0817408f,  -0.189918f,
  -0.804952f,   -1.33036f,    -1.03307f,     0.0248769f,   0.16607f,
  -2.896f,      -2.1293f,     0.12293f,      -0.173179f,   -0.212128f,
  -6.76221f,    0.033188f,    0.0231787f,    0.905957f,    0.0551327f,
  -0.356276f,   0.0181795f,   0.0977523f,    -0.0352873f,  -0.0396386f,
  2.3241f,      0.0632874f,   -0.11804f,     -6.32521f,    0.0224659f,
  -0.00188896f, 0.267992f,    0.272337f,     0.00936963f,  0.659969f,
  -2.25707f,    -0.0278229f,  -0.0185089f,   -1.14466f,    0.104827f,
  0.0435885f,   0.558586f,    -0.00697004f,  0.0312611f,   0.540574f,
  -0.568625f,   0.218608f,    0.378911f,     -0.0289192f,  -0.0734742f,
  -1.08782f,    -2.42069f,    -0.0127239f,   0.0493651f,   -1.15837f,
  0.261831f,    0.401824f,    -1.04545f,     0.284173f,    0.784972f,
  -0.511243f,   -0.982599f,   -0.106134f,    -0.325964f,   -1.44107f,
  -1.42434f,    -1.02402f,    -1.52034f,     0.0737116f,   0.0462242f,
  0.628722f,    -1.0405f,     -0.113718f,    2.20573f,     -4.33951f,
  -0.0192695f,  -0.0229314f,  -1.89156f,     0.645942f,    0.375708f,
  -1.97447f,    -0.267014f,   0.0989443f,    -0.450534f,   -1.01737f,
  -0.642416f,   -0.0897288f,  -2.08724f,     -0.190965f,   -0.279135f,
  -0.830178f,   0.808754f,    -0.139091f,    1.11004f,     -0.454439f,
  -0.479238f,   -1.44001f,    0.0888059f,    0.885689f,    -0.642505f,
  -0.00773651f, -0.0265721f,  -0.906346f,    1.68504f,     0.084257f,
  -0.951101f,   -8.06495f,    0.19231f,      0.16389f,     -0.193678f,
  0.729837f,    -1.98392f,    -5.98513f,     3.32638f,     -0.0658378f,
  -0.0910426f,  -0.666567f,   -0.315339f,    0.123124f,    -2.66375f,
  -0.714852f,   -0.136176f,   -0.460166f,    -0.567551f,   -1.06193f,
  -1.21389f,    -0.83865f,    0.00280695f,   -0.199519f,   -0.534704f,
  0.419311f,    -0.149008f,   -3.68707f,     0.00285113f,  -0.0718198f,
  -1.41026f,    -1.34155f,    -0.538687f,    -0.623666f,   -2.56462f,
  -0.0183333f,  -0.323532f,   -1.27141f,     -0.0212039f,  0.198633f,
  0.459554f,    -4.65103f,    -1.01293f,     -1.39512f,    -0.289026f,
  0.208724f,    -0.665226f,   1.13369f,      -1.96734f,    -1.45442f,
  -3.46172f,    0.810681f,    -0.603973f,    0.842764f,    -3.90371f,
  -0.394561f,   -3.61363f,    -2.88085f,     0.031645f,    -0.23125f,
  -2.63898f,    -1.35314f,    -0.46726f,     1.33145f,     1.20269f,
  1.38682f,     -0.331637f,   0.069021f,     0.149523f,    -1.24957f,
  -0.878857f,   -0.200368f,   0.465744f,     1.01365f,     -0.0122221f,
  -0.550586f,   -1.12581f,    -0.422132f,    -0.0744868f,  -2.4804f,
  -1.07072f,    -0.479006f,   0.101817f,     -0.118947f,   0.341576f,
  -1.0538f,     -0.812346f,   -1.13727f,     -0.00939806f, 10.1571f,
  -0.0441302f,  0.00280407f,  -21.5044f,     0.0181152f,   -0.0143246f,
  3.23462f,     -1.38624f,    -1.80416f,     4.89763f,     -2.67364f,
  2.31771e-05f, 0.000393989f, 0.352204f,     -0.193455f,   0.531455f,
  0.488757f,    -0.442555f,   -0.518528f,    0.431482f,    -2.67727f,
  -2.00626f,    -0.39729f,    -0.221494f,    -0.0188888f,  -0.0377649f,
  -1.80169f,    0.0810332f,   -0.0408335f,   -1.28675f,    -0.0353824f,
  -0.666723f,   -1.07281f,    0.252912f,     -1.24547f,    -1.7831f,
  -1.14354f,    -0.137662f,   0.00230182f,   0.736862f,    0.175872f,
  -0.187556f,   0.43963f,     -0.796524f,    0.056219f,    -0.387874f,
  0.0710224f,   -0.16548f,    -0.100993f,    0.931481f,    -3.20738f,
  -0.0197576f,  0.266148f,    -0.173909f,    -0.337795f,   -0.0682381f,
  0.176844f,    0.140286f,    1.12033f,      0.429064f,    -2.24192f,
  -1.54682f,    2.23646f,     -0.0371138f,   -0.0475339f,  -3.21766f,
  0.0412858f,   0.387811f,    6.6711f,       0.140649f,    0.0559547f,
  -0.802839f,   0.599977f,    0.64552f,      -2.08103f,    -0.503401f,
  -0.0407036f,  -0.0299199f,  0.0849445f,    -0.111657f,   -1.63462f,
  3.33762f,     0.0441394f,   0.0466889f,    -0.951806f,   0.0723954f,
  0.00348661f,  -1.36903f,    2.24625f,      -0.0348915f,  -0.0508893f,
  -0.240891f,   -0.120143f,   -0.17991f,     -2.09137f,    0.0150871f,
  0.0480333f,   1.72012f,     0.0309551f,    -0.0370507f,  -0.377075f,
  0.103916f,    -0.0169255f,  -0.0145395f,   -4.02144f,    0.83193f,
  -0.316502f,   6.3832f,      -1.70038f,     -1.97215f,    -1.94501f,
  1.45479f,     0.711725f,    -0.348496f,    -0.279056f,   -1.13396f,
  -1.51744f,    -0.853307f,   1.53131f,      -0.0032358f,  1.41808f,
  -1.32989f,    -0.245221f,   -0.161614f,    -0.500845f,   -0.449252f,
  0.0724151f,   -0.116333f,   -0.0946182f,   -2.0945f,     0.0564572f,
  0.393261f,    -1.06861f,    -0.111458f,    -0.839943f,   -0.0880348f,
  0.0365742f,   0.415339f,    -1.57494f,     -0.713697f,   1.02349f,
  -0.221371f,   -0.0446281f,  1.89223f,      -0.0811754f,  -0.402773f,
  -0.930987f,   0.0243194f,   0.0678332f,    -0.0233014f,  0.165372f,
  -0.44083f,    -1.2404f,     0.35675f,      -0.040916f,   -0.0512548f,
  -2.9071f,     0.861174f,    -0.778133f,    2.14436f,     -0.688427f,
  -0.480371f,   -1.69032f,    0.706687f,     -0.281982f,   -2.30451f,
  1.61541f,     -0.0213638f,  -0.740509f,    -0.266677f,   0.0268434f,
  -0.0116908f,  -3.17595f,    0.0114825f,    0.0196997f,   -0.144005f,
  0.0550181f,   -0.851459f,   -0.000285073f, -0.538441f,   -0.0254868f,
  -0.0104454f,  -0.0661998f,  -0.196469f,    -0.346372f,   -5.52892f,
  -0.643683f,   -0.622224f,   -0.31463f,     -0.555956f,   -0.520132f,
  -0.843166f,   -2.59479f,    -0.750195f,    0.00635995f,  -0.338615f,
  -0.216676f,   -0.391544f,   -1.62185f,     -0.718471f,   -0.475406f,
  -0.782041f,   -0.608824f,   -1.09633f,     -1.27308f,    -0.560719f,
  -0.207539f,   -0.0196445f,  -1.05519f,     -0.575249f,   -1.0642f,
  1.01615f,     -0.873633f,   -0.417953f,    -0.428051f,   0.350259f,
  -2.53833f,    -2.72203f,    0.672846f,     -0.503094f,   -1.1374f,
  0.214291f,    0.013305f,    0.0112064f,    1.10532f,     0.030455f,
  0.0239614f,   0.628072f,    0.0539135f,    -0.472441f,   -0.688439f,
  -0.32044f,    -0.0234867f,  -0.0158436f,   -0.949314f,   -0.0453161f,
  -1.18306f,    0.626845f,    -0.426925f,    -0.688371f,   0.415062f,
  0.0640985f,   -0.638387f,   -2.01399f,     -0.209744f,   -0.762892f,
  -0.0753296f,  -0.879315f,   -0.520433f,    -0.111375f,   0.389742f,
  -0.398862f,   -0.643227f,   -0.246396f,    0.0317051f,   1.06973f,
  0.413617f,    0.180506f,    -0.0507897f,   -0.00650435f, 0.620892f,
  0.046312f,    0.475032f,    0.906993f,     -0.0388061f,  -0.256271f,
  -1.03323f,    0.0125266f,   -0.31116f,     -0.377611f,   -0.0386407f,
  -0.0232745f,  -0.353644f,   -2.27289f,     0.0571779f,   -0.00865006f,
  1.65101f,     0.0175711f,   0.0184585f,    0.558458f,    0.2213f,
  -0.285089f,   0.433445f,    -0.427177f,    -0.0103682f,  -0.0101273f,
  0.214085f,    -0.0459885f,  0.00761981f,   0.836381f,    0.0175293f,
  0.02508f,     -1.51778f,    0.0143956f,    -0.162589f,   0.595418f,
  0.21445f,     -0.0335848f,  -0.0136684f,   -0.16686f,    -0.14612f,
  0.0816238f,   0.499636f,    0.12458f,      -2.41673f,    -0.261721f,
  -0.676805f,   -1.88366f,    0.730462f,     0.69196f,     -0.0288489f,
  -2.38272f,    0.329876f,    0.014517f,     -0.115145f,   -3.48151f,
  -0.00209072f, -0.0732377f,  0.820443f,     -0.0118701f,  0.112145f,
  0.272315f,    0.137531f,    -0.0200997f,   -0.0397883f,  -2.19458f,
  0.183554f,    -0.639716f,   0.481605f,     -0.621639f,   -0.0980299f,
  -0.710534f,   -0.143105f,   -6.77626f,     -1.65139f,    -2.37718f,
  -0.533127f,   -1.12574f,    3.34182f,      -0.0758663f,  0.0334238f,
  -9.48647f,    0.0674974f,   0.0507665f,    0.523007f,    -0.0668f,
  0.5736f,      -0.589761f,   -1.1692f,      -0.0236497f,  -0.00828928f,
  -0.265823f,   1.15284f,     0.307927f,     -0.695308f,   0.13725f,
  -0.20394f,    -0.363965f,   -0.331159f,    -1.50927f,    -1.20051f,
  -0.0205825f,  -0.0381859f,  -0.0579876f,   -1.6913f,     -1.94626f,
  3.4214f,      3.3922f,      -2.13798f,     -0.679848f,   -0.890735f,
  0.235017f,    -0.253202f,   -1.0571f,      1.40354f,     0.00719052f,
  -1.54365f,    -0.7289f,     -1.05492f,     0.0238169f,   -0.00543592f,
  -0.0510353f,  -0.175386f,   -0.724207f,    -0.788936f,   0.039976f,
  1.36966f,     0.869475f,    -0.0302774f,   -0.0537556f
};

static const NN_CONFIG av1_max_part_pred_nn_config = {
  NUM_FEATURES,
  NUM_LOGITS,
  NUM_HIDDEN_LAYERS,
  {
      NUM_LAYER_0_UNITS,
  },
  {
      av1_max_part_pred_layer_0_kernel,
      av1_max_part_pred_logits_kernel,
  },
  {
      av1_max_part_pred_layer_0_bias,
      av1_max_part_pred_logits_bias,
  },
};

#undef NUM_HIDDEN_LAYERS
#undef NUM_FEATURES
#undef NUM_LAYER_0_UNITS
#undef NUM_LOGITS

// Early termination in second pass
static const float av1_simple_motion_search_term_none_mean_128[28] = {
  12.661922f, 12.638062f, 10.896497f, 10.865719f, 10.978963f, 10.940105f,
  11.012235f, 10.972760f, 11.069924f, 11.018533f, 11.773865f, 11.747426f,
  11.891315f, 11.858107f, 11.793916f, 11.766356f, 11.874997f, 11.840164f,
  5.940535f,  0.770746f,  4.292692f,  4.309581f,  0.848423f,  4.292334f,
  4.298179f,  8.514713f,  14.911736f, 19.825352f,
};

static const float av1_simple_motion_search_term_none_std_128[28] = {
  1.796731f, 1.797056f, 1.898383f, 1.900753f, 1.846624f, 1.846953f, 1.906632f,
  1.908089f, 1.836533f, 1.835967f, 1.840262f, 1.840671f, 1.816836f, 1.817103f,
  1.879846f, 1.881333f, 1.803102f, 1.802654f, 2.263402f, 0.420354f, 1.117165f,
  1.083779f, 0.358611f, 1.101183f, 1.084938f, 2.462638f, 1.577009f, 1.574711f,
};

static const float av1_simple_motion_search_term_none_mean_64[28] = {
  10.904455f, 10.853546f, 9.247903f,  9.184479f,  9.251985f,  9.186686f,
  9.253490f,  9.190190f,  9.270079f,  9.204357f,  10.086511f, 10.031060f,
  10.100875f, 10.045429f, 10.069688f, 10.013173f, 10.082980f, 10.024640f,
  4.888378f,  0.878113f,  3.598450f,  3.628491f,  0.925833f,  3.560971f,
  3.573322f,  8.807137f,  13.348477f, 18.269117f,
};

static const float av1_simple_motion_search_term_none_std_64[28] = {
  1.789300f, 1.787061f, 1.823519f, 1.820226f, 1.794643f, 1.788620f, 1.797194f,
  1.795135f, 1.777795f, 1.773634f, 1.794000f, 1.790377f, 1.772197f, 1.769692f,
  1.819050f, 1.817139f, 1.793577f, 1.789333f, 1.998251f, 0.327156f, 0.885748f,
  0.853767f, 0.262043f, 0.902435f, 0.860033f, 1.224865f, 1.603411f, 1.589296f,
};

static const float av1_simple_motion_search_term_none_mean_32[28] = {
  9.818970f, 9.751199f, 8.015079f, 7.927318f, 8.029113f, 7.938330f,  8.012570f,
  7.923719f, 8.033508f, 7.941911f, 8.933057f, 8.857422f, 8.935639f,  8.859187f,
  8.905495f, 8.829741f, 8.929428f, 8.851351f, 4.114069f, 0.954752f,  2.645082f,
  2.709703f, 0.964678f, 2.652077f, 2.673393f, 9.430499f, 11.922798f, 16.942251f,
};

static const float av1_simple_motion_search_term_none_std_32[28] = {
  1.737107f, 1.734327f, 1.727923f, 1.720244f, 1.721570f, 1.712775f, 1.718028f,
  1.710370f, 1.711612f, 1.702596f, 1.754856f, 1.748855f, 1.741871f, 1.736304f,
  1.722428f, 1.717380f, 1.713563f, 1.707582f, 1.761170f, 0.207847f, 0.900058f,
  0.862356f, 0.184593f, 0.903822f, 0.856120f, 1.529199f, 1.412085f, 1.453153f,
};

static const float av1_simple_motion_search_term_none_mean_16[28] = {
  8.998877f, 8.912468f, 7.085255f, 6.953476f, 7.086386f, 6.954091f,  7.088727f,
  6.955747f, 7.093955f, 6.960635f, 8.065050f, 7.961432f, 8.071631f,  7.967233f,
  8.041699f, 7.937715f, 8.046791f, 7.942183f, 3.833521f, 0.978421f,  1.901347f,
  1.950124f, 0.979418f, 1.928000f, 1.936727f, 9.773951f, 10.735227f, 15.949769f,
};

static const float av1_simple_motion_search_term_none_std_16[28] = {
  1.641193f, 1.640172f, 1.614794f, 1.608906f, 1.609571f, 1.603580f, 1.606928f,
  1.601246f, 1.599230f, 1.593529f, 1.633747f, 1.630219f, 1.625695f, 1.622547f,
  1.633827f, 1.630182f, 1.626607f, 1.622777f, 1.548838f, 0.145303f, 0.744550f,
  0.736552f, 0.141980f, 0.742979f, 0.736977f, 1.366255f, 1.258794f, 1.294309f,
};

static const float av1_simple_motion_search_term_none_model_128[] = {
  -0.6106842357f, -1.0402954455f, 0.6054417656f,  -0.2116623578f,
  0.2447714930f,  0.3782256209f,  0.5095592479f,  -0.3275620904f,
  0.3886188013f,  0.2629499420f,  -0.1979599415f, -0.5389565605f,
  0.1209207902f,  -0.4913347466f, 0.3798542731f,  -0.2812861709f,
  -0.1049824167f, -0.1088672020f, 0.4059596517f,  -0.1347896613f,
  0.2276868621f,  0.0506386970f,  0.0071088411f,  0.0467952100f,
  0.2091247458f,  -0.7371964736f, 0.1368935545f,  0.3175247786f,
  -0.5493146094f,
};

static const float av1_simple_motion_search_term_none_model_64[] = {
  -0.4150046575f, -0.3954358561f, 0.1997997444f,  0.3395826831f,
  0.2827215753f,  0.3395683652f,  0.2483140395f,  0.2722216476f,
  0.2610308009f,  0.3724974359f,  -0.0551479654f, -0.1721616359f,
  -0.3459358629f, -0.0952524186f, -0.1428993840f, -0.0415654914f,
  -0.3169539902f, -0.0269429900f, 0.9891530919f,  -0.0125084982f,
  0.0972182377f,  0.0008889801f,  0.0205418050f,  0.0057237854f,
  0.1005222691f,  -0.2851321920f, -1.5150336445f, 0.1893942436f,
  -0.4337360901f,
};

static const float av1_simple_motion_search_term_none_model_32[] = {
  -0.4667392852f, -0.3893302767f, 0.1603498635f,  0.2304974726f,
  0.1404975592f,  0.2505516225f,  0.1423053884f,  0.2189318406f,
  0.1379765409f,  0.2638241296f,  -0.1342865463f, -0.0549054345f,
  -0.1925223436f, -0.1142702769f, 0.0127811659f,  0.0868639997f,
  -0.0643197251f, 0.0279496470f,  0.9904395769f,  -0.0095178685f,
  0.1179410649f,  -0.0013411972f, 0.0095060660f,  0.0195730400f,
  0.0779717771f,  -0.2498860763f, -0.8168817125f, -0.4798397348f,
  -0.6609679881f,
};

static const float av1_simple_motion_search_term_none_model_16[] = {
  -0.3021081992f, -0.4620153673f, 0.0448577479f,  0.1738455035f,
  0.0663209177f,  0.1629614573f,  0.0555168744f,  0.1631870212f,
  0.0425805150f,  0.1688564954f,  0.0434083772f,  -0.0046603915f,
  -0.0271580056f, -0.0183879127f, 0.1073730471f,  0.0314201476f,
  0.0576891756f,  0.0119723753f,  0.9084332022f,  -0.0188429077f,
  0.0755089811f,  -0.0172550234f, 0.0037663075f,  0.0022094472f,
  0.0500247894f,  -0.2944572004f, -0.8908521199f, -0.2555515792f,
  -0.5396254205f,
};

#define FEATURES 31
#define HIDDEN_NODES 32
static const float av1_early_term_after_split_nn_weights_64_layer0[] = {
  -0.306296f, -0.691664f, 0.335148f,  -0.298465f, -0.509241f, -0.632796f,
  -0.527979f, -0.009904f, -0.503646f, -0.494002f, -0.575101f, 0.239911f,
  -0.413312f, -0.622825f, -0.405448f, -0.419103f, -0.505903f, -0.392550f,
  -0.240293f, 0.121749f,  -0.489777f, -0.756647f, 0.001047f,  -0.016528f,
  0.145714f,  0.172910f,  0.086197f,  0.162882f,  -0.070588f, -0.077104f,
  0.502730f,  -0.244954f, 0.265605f,  -0.323994f, 0.223397f,  -1.086453f,
  0.391886f,  0.200343f,  0.253878f,  0.018925f,  0.201819f,  -0.205136f,
  0.427314f,  0.041155f,  0.070484f,  0.159925f,  -0.057095f, -0.146544f,
  -0.073792f, 0.152628f,  0.003986f,  -0.515965f, -0.209754f, 0.037457f,
  0.070622f,  -0.143571f, -0.059602f, 0.111734f,  0.319674f,  0.149894f,
  -0.219883f, 0.206678f,  0.015809f,  -0.210549f, 0.130156f,  -0.189502f,
  -0.850392f, -0.156363f, -0.060354f, 0.189044f,  0.266495f,  0.151305f,
  -0.563677f, -0.354896f, 0.300637f,  0.257568f,  -0.008359f, -0.535497f,
  -0.003127f, 0.293054f,  -0.020212f, -0.157278f, 0.229972f,  -0.309799f,
  -0.329927f, -0.077140f, 0.001177f,  -0.024415f, 0.134044f,  -0.181587f,
  -0.135380f, 0.230989f,  -0.281451f, 0.912282f,  0.511562f,  -3.900779f,
  -0.039917f, 1.956406f,  -0.357589f, 0.292998f,  -0.950158f, 0.422041f,
  0.526572f,  0.605746f,  -0.147110f, 0.256576f,  0.090010f,  0.221641f,
  0.029763f,  0.351592f,  0.458324f,  -0.005888f, 0.010521f,  -0.389326f,
  -0.094006f, -0.171489f, -0.013153f, 0.026333f,  -0.454571f, -1.932891f,
  -0.168211f, 0.051298f,  -0.258061f, -0.028936f, -0.555937f, -0.475566f,
  -0.304046f, -0.318113f, 0.099697f,  -0.217145f, 0.139433f,  -0.203986f,
  -0.164012f, 0.051527f,  0.138603f,  -0.085100f, -0.082887f, -0.242955f,
  -0.663410f, -0.535772f, -0.181665f, -0.197883f, 0.071319f,  0.135086f,
  0.146200f,  0.184827f,  -0.199041f, 0.162570f,  -0.300167f, 0.017748f,
  -0.140111f, 0.103553f,  0.206929f,  0.193446f,  0.123141f,  -1.201898f,
  -0.052254f, -0.750121f, 0.111741f,  0.204092f,  -0.166266f, 0.124008f,
  -0.455496f, 0.306035f,  0.275903f,  0.193599f,  -0.730011f, 0.126808f,
  0.051059f,  0.103634f,  -0.044334f, 0.048889f,  0.405228f,  0.574099f,
  0.061167f,  0.260576f,  0.070032f,  -0.038040f, 0.229183f,  -0.243269f,
  -0.130116f, -0.538563f, -0.070199f, -0.129249f, -0.205153f, -0.268530f,
  -0.290828f, -0.233006f, 0.068712f,  0.618085f,  -0.407008f, 0.686868f,
  0.172247f,  0.826287f,  -0.002672f, 0.239825f,  -0.051548f, 0.420773f,
  0.218747f,  0.041057f,  -0.071189f, 0.286987f,  -0.113915f, 0.122561f,
  0.013979f,  -0.049046f, 0.148175f,  0.031313f,  -0.248601f, 0.209488f,
  0.069008f,  0.072763f,  0.332475f,  0.079986f,  -0.151042f, -0.205110f,
  -0.155550f, -0.510408f, 0.330429f,  0.577729f,  0.266524f,  -0.378489f,
  0.228204f,  0.055318f,  0.117583f,  -0.588557f, -0.778201f, 0.434622f,
  -0.227820f, 0.611642f,  0.170548f,  0.817761f,  0.006642f,  -1.005794f,
  -0.911490f, 1.633684f,  -0.290664f, 0.308128f,  0.295986f,  0.243377f,
  -0.001275f, -0.131156f, 0.275205f,  -0.041865f, -0.201951f, -0.016380f,
  0.336604f,  -0.258118f, 0.890810f,  0.441065f,  -0.968006f, 0.135989f,
  -1.447191f, 0.353426f,  -0.343235f, 0.376837f,  -0.071602f, -0.319639f,
  -0.072347f, 0.547450f,  -0.215380f, 0.182141f,  -0.066186f, 0.033787f,
  0.257482f,  0.217428f,  -0.130249f, 0.057525f,  0.263991f,  0.230664f,
  -0.245113f, 0.048610f,  -0.079955f, 0.251737f,  -0.070368f, -0.017968f,
  -0.151815f, 0.025945f,  -0.257769f, 0.299735f,  0.077263f,  -0.565526f,
  0.326263f,  0.096429f,  0.113414f,  0.092754f,  -0.141908f, 0.172060f,
  0.393117f,  -0.216755f, 0.331051f,  -0.363369f, -0.113363f, -0.095164f,
  -0.072784f, 0.214572f,  0.010993f,  0.209456f,  0.260381f,  -0.314747f,
  -0.422173f, -0.189963f, -0.225130f, 0.339448f,  0.153814f,  0.265616f,
  -0.103575f, -0.123841f, -0.106236f, 0.155894f,  -0.156264f, -1.361406f,
  -0.040736f, -0.614998f, -0.468200f, -0.266505f, -0.342786f, -0.908088f,
  0.105758f,  0.040788f,  -0.313589f, -1.359318f, 0.071329f,  0.176404f,
  -0.476141f, 0.010108f,  -0.201440f, -0.221167f, -0.197448f, -0.013927f,
  -0.610270f, -0.607285f, 0.178070f,  0.174320f,  0.313115f,  0.026191f,
  -0.112330f, 0.122338f,  -0.367751f, 0.196794f,  0.153709f,  -0.205454f,
  -0.397471f, -1.879336f, -0.030129f, 0.143429f,  -0.079832f, 0.435259f,
  -1.729539f, 0.518301f,  -0.141393f, 0.199399f,  -1.914601f, 0.142865f,
  -0.219899f, 0.508458f,  0.086365f,  -0.220740f, -0.012507f, 1.263320f,
  0.042136f,  0.050922f,  -0.329644f, -0.188198f, 0.251522f,  0.394731f,
  -0.047866f, -0.260853f, -0.267207f, -0.248489f, 0.146474f,  0.359257f,
  -0.427732f, -0.100652f, 0.192129f,  0.075572f,  0.916708f,  0.255747f,
  0.486384f,  0.127989f,  -0.556449f, -0.484913f, 0.392298f,  0.045401f,
  -0.839551f, -0.703619f, 0.069263f,  -0.040720f, 0.542265f,  0.443739f,
  0.862552f,  -0.021726f, 0.230858f,  -0.261004f, -0.125697f, -0.106435f,
  0.002341f,  0.013904f,  0.011034f,  0.542296f,  -0.284325f, 0.135736f,
  0.113882f,  0.040610f,  -0.255485f, 0.224061f,  -0.087140f, 0.127872f,
  -0.002638f, 0.164889f,  -0.335958f, -0.031166f, -0.393581f, 0.075455f,
  0.055995f,  0.087934f,  -0.133859f, -0.342187f, 0.002492f,  -0.340722f,
  0.058304f,  0.104165f,  -0.142136f, -0.351111f, -0.158037f, -0.079924f,
  -0.253209f, -0.092840f, -0.174646f, -0.202772f, -0.353438f, -0.031111f,
  0.076088f,  -0.232091f, -0.070052f, 0.097595f,  0.063173f,  -0.211195f,
  0.126478f,  -0.178828f, 0.278723f,  -0.070807f, -0.179783f, 0.034123f,
  0.035721f,  -0.200431f, 0.170640f,  0.107933f,  0.226594f,  -0.301499f,
  -0.291096f, 0.228076f,  -0.272951f, 0.002490f,  -0.210707f, -0.128033f,
  -0.194009f, -0.011347f, -0.256694f, -0.011841f, -0.005167f, -0.163203f,
  -0.253796f, -0.198877f, -0.055827f, -0.882685f, -0.443471f, 0.349601f,
  0.749334f,  -1.161845f, 0.505480f,  0.221733f,  0.210490f,  -0.234984f,
  0.014183f,  -0.510401f, 0.238692f,  -0.134111f, 0.083844f,  -0.478751f,
  -0.088434f, 0.304063f,  0.150336f,  -0.749682f, -0.081999f, 0.729739f,
  0.412508f,  0.132571f,  0.058306f,  -0.047451f, -0.117435f, -0.445395f,
  -0.005182f, -0.025757f, 0.175051f,  -0.258194f, -0.150311f, -0.196533f,
  -1.314316f, -0.428627f, 0.512451f,  0.045138f,  -0.200925f, 0.081538f,
  -0.346151f, -0.358197f, -0.422258f, -0.028542f, -0.383534f, -0.026163f,
  -0.419858f, -0.154321f, 0.376970f,  0.094017f,  0.783520f,  0.110641f,
  0.077966f,  -0.093064f, 0.160522f,  -0.863041f, 0.086210f,  0.560764f,
  0.057032f,  0.159224f,  0.323068f,  -0.173109f, 0.014042f,  -0.126856f,
  -0.128237f, -0.245273f, -0.317312f, -0.257597f, -0.181977f, 0.259485f,
  -0.215834f, 0.062076f,  -0.270596f, 0.271581f,  -0.153486f, -0.247165f,
  0.079737f,  -0.157049f, -0.027459f, -0.299397f, 0.136729f,  -0.334192f,
  -0.191722f, 0.145865f,  -0.031324f, -0.307165f, -0.244923f, -0.228027f,
  0.063807f,  0.054965f,  -0.005709f, -0.041977f, -0.276245f, 0.020003f,
  0.133323f,  -0.145992f, -0.951030f, 0.414083f,  -1.063323f, 0.137872f,
  0.104732f,  -0.123728f, 0.542532f,  0.213654f,  0.542954f,  0.155619f,
  0.543072f,  0.399067f,  0.191402f,  -0.102552f, -0.176734f, -0.136776f,
  -0.012814f, -0.021298f, -0.802467f, -0.957481f, -0.238787f, -0.138482f,
  0.058331f,  0.126601f,  0.104420f,  -0.148684f, 0.343218f,  0.093604f,
  -0.055642f, -0.383918f, -0.045250f, -0.090480f, -0.155464f, 0.278299f,
  0.042791f,  -0.029084f, -0.373861f, -0.073233f, -0.085172f, 0.186841f,
  -0.070898f, -0.156415f, 0.112831f,  -0.065931f, -0.353007f, 0.058453f,
  -0.136982f, 0.233393f,  0.017240f,  -0.018428f, 0.229104f,  -0.371440f,
  -0.262212f, 0.203075f,  -0.263293f, 0.034413f,  -0.299354f, 0.227269f,
  0.204977f,  -0.118107f, -0.359832f, -0.068252f, 0.480105f,  -0.214711f,
  -0.614381f, 0.209048f,  -0.456014f, -0.188819f, -0.220995f, -0.322104f,
  -0.191457f, 0.420874f,  -0.454919f, 0.023119f,  0.291700f,  -0.532885f,
  -0.032642f, 0.043271f,  0.133974f,  0.002399f,  -0.179899f, -0.044158f,
  -0.027078f, -0.350075f, 0.236766f,  0.346771f,  -0.118534f, -0.421221f,
  0.019544f,  0.109349f,  0.141517f,  0.403561f,  0.409102f,  0.054555f,
  -0.561751f, 0.577183f,  -0.705156f, -0.231188f, -1.969772f, 0.172289f,
  -0.048122f, 0.205671f,  -0.667130f, -0.066870f, 0.202838f,  -0.095538f,
  -0.842651f, 0.254170f,  0.046256f,  -0.271891f, -0.369254f, 0.492101f,
  0.001189f,  -0.186525f, 0.188470f,  -0.207072f, 0.030086f,  -0.132904f,
  0.127001f,  0.116662f,  -0.079246f, 0.227241f,  -0.462178f, 0.446304f,
  -1.660753f, 0.241832f,  -0.288040f, 0.054663f,  -0.435804f, 0.296782f,
  -0.026421f, -0.115618f, 0.163416f,  0.834001f,  0.008019f,  -0.014243f,
  0.524658f,  0.067894f,  -0.253936f, -0.100657f, 1.285389f,  -0.005952f,
  0.087134f,  -0.088375f, -0.121866f, -0.171172f, 0.279463f,  -0.598593f,
  -0.727761f, 0.189831f,  -0.822575f, -0.291141f, -0.012410f, -0.069999f,
  0.098842f,  -0.218513f, 0.009494f,  0.100106f,  -0.402884f, -0.299236f,
  -0.345668f, -0.057739f, -0.213248f, -0.426661f, -0.360268f, -0.349860f,
  -0.382177f, -0.357802f, -0.032030f, -0.110597f, -0.155442f, -0.418794f,
  -0.012113f, -0.032962f, -0.450648f, 0.129060f,  -0.135227f, -0.298593f,
  0.001435f,  0.278790f,  -0.272945f, 0.162759f,  -0.290208f, 0.058481f,
  -0.490971f, 0.019630f,  -0.210347f, 0.000520f,  -0.340413f, 0.641562f,
  0.023104f,  0.194832f,  -0.441894f, -0.253538f, -0.228332f, 0.423264f,
  -1.094073f, -0.475657f, -0.238752f, 0.033910f,  0.440425f,  0.036320f,
  0.566989f,  -0.065326f, -0.297939f, 0.406098f,  0.529561f,  -0.113084f,
  0.141472f,  -0.024462f, -0.179212f, 0.187801f,  -0.235787f, -0.229624f,
  0.357791f,  0.061110f,  -0.607788f, -1.713694f, -0.651041f, 1.734283f,
  -0.334701f, 0.161687f,  0.010215f,  0.320708f,  0.169447f,  0.513558f,
  0.488340f,  -0.619036f, -0.525441f, -1.144352f, -0.546154f, 0.669973f,
  0.327028f,  -0.100539f, 0.012048f,  -0.223013f, -0.239680f, 0.323035f,
  0.165950f,  -0.155110f, 0.128664f,  -0.157378f, -0.124490f, 0.291553f,
  0.055849f,  -0.221664f, 0.077770f,  -0.350658f, -0.181939f, 0.110230f,
  -0.078219f, 0.007472f,  -0.031620f, 0.007708f,  -0.201794f, 0.017594f,
  -0.027480f, 0.058884f,  -0.369166f, -0.369770f, 0.181635f,  -0.183318f,
  -0.389184f, -0.256661f, 0.160107f,  0.037127f,  -0.082573f, -0.095815f,
  -0.322782f, 0.072528f,  -0.348875f, 0.216247f,  -0.161757f, -0.385502f,
  -0.315738f, 0.020123f,  -0.155609f, 0.114403f,  -0.383232f, 0.629529f,
  0.066142f,  0.448392f,  -0.389557f, -0.083315f, 0.829535f,  -0.015531f,
  -0.050728f, -0.325127f, 0.812992f,  -0.196780f, 0.021060f,  -0.952647f,
  0.006687f,  -0.512715f, -0.066778f, 0.410067f,  -0.116945f, -0.288283f,
  0.189334f,  -0.083153f, 0.159980f,  -0.068208f, 0.107358f,  -0.154411f,
  -0.068914f, 0.186816f,  0.032251f,  0.109242f,  0.134825f,  0.035101f,
  -0.253175f, 0.157309f,  -0.363597f, -0.138176f, -0.334141f, -0.172697f,
  0.045800f,  -0.286057f, 0.173403f,  -0.172444f, -0.117996f, -0.383848f,
  -0.173303f, -0.258482f, -0.021404f, -0.017898f, -0.001970f, 0.003273f,
  0.056121f,  0.155046f,  0.044708f,  -0.295609f, -0.211688f, -0.233229f,
  -0.264980f, 0.145549f,  0.045323f,  -0.027112f, 0.175638f,  -0.207251f,
  -0.055274f, 0.092706f,  0.086200f,  -0.241340f, -0.147416f, 0.024510f,
  -0.357194f, -0.181944f, -0.050104f, -0.079024f, -0.290473f, -0.169790f,
  -0.277982f, -0.017781f, -0.004854f, -0.094132f, -0.348555f, 0.199291f,
  -0.343989f, -0.319299f, -0.268935f, -0.021208f, 0.020938f,  -0.090609f,
  0.006595f,  -0.200790f, 0.171856f,  -0.027766f, -0.032017f, -0.006745f,
  0.566426f,  -0.096850f, 0.727633f,  -0.408065f, -0.012436f, 0.005646f,
  -0.305148f, -0.095075f, -0.391549f, -0.020378f, -0.236498f, -0.252773f,
  -0.231385f, -0.203175f, 0.041903f,  -0.373694f, 0.058239f,  -0.101116f,
  0.183772f,  0.164523f,  -0.099046f, -0.201272f, -0.394523f, -0.157517f,
  0.032079f,  -0.381173f, -0.238496f, -0.037990f, -0.294553f, 0.141473f,
  0.100268f,  -0.023806f, 0.004978f,  0.184916f,  0.142699f,  -0.113240f,
  -0.213364f, -0.160059f, -0.216263f, -0.406387f, -0.301140f, -0.406355f,
  -0.113085f, -0.279699f, -0.267434f, 0.126263f,  -0.260527f, -0.153904f,
  -0.494653f, -0.355144f, 0.030549f,  -0.216400f, -0.123363f, 0.189090f,
  0.219122f,  0.096677f,  -0.202037f, -0.014489f, -0.137859f, -0.114184f,
  -0.279423f, -0.270683f,
};

static const float av1_early_term_after_split_nn_bias_64_layer0[] = {
  -0.491455f, 0.464538f,  -0.005742f, -0.219951f, -0.073682f, 0.102027f,
  0.567071f,  0.441402f,  0.277521f,  0.314498f,  -0.448199f, -0.065032f,
  0.488139f,  -0.079632f, 0.000000f,  0.521555f,  -0.151950f, -0.034616f,
  0.393438f,  -0.072242f, -0.087343f, -0.571308f, 0.017372f,  -0.126144f,
  0.372261f,  -0.451537f, -0.140238f, -0.092377f, -0.074475f, -0.068879f,
  -0.109614f, -0.164492f,
};

static const float av1_early_term_after_split_nn_weights_64_layer1[] = {
  -0.373195f, -0.283141f, 0.416113f,  0.483659f,  0.230583f,  0.349197f,
  -0.168582f, -0.813338f, -0.472369f, -0.173872f, 1.297845f,  0.339355f,
  -0.828033f, 0.019617f,  0.118757f,  -0.619360f, 0.282295f,  -0.054116f,
  -0.730596f, 0.068567f,  -0.248707f, 0.461225f,  0.330224f,  -0.287080f,
  -0.458103f, 0.591852f,  -0.008491f, 0.632119f,  -0.007872f, 0.007869f,
  -0.230698f, -0.011437f,
};

static const float av1_early_term_after_split_nn_bias_64_layer1[] = {
  -0.55403697f,
};

static const NN_CONFIG av1_early_term_after_split_nnconfig_64 = {
  FEATURES,
  1,
  1,
  {
      HIDDEN_NODES,
  },
  {
      av1_early_term_after_split_nn_weights_64_layer0,
      av1_early_term_after_split_nn_weights_64_layer1,
  },
  {
      av1_early_term_after_split_nn_bias_64_layer0,
      av1_early_term_after_split_nn_bias_64_layer1,
  },
};

static const float av1_early_term_after_split_nn_weights_32_layer0[] = {
  0.026050f,  -0.226531f, 0.308107f,  -0.083744f, 0.201785f,  0.098562f,
  0.147595f,  -0.495771f, -0.245741f, 0.201616f,  -0.272070f, -0.579545f,
  -0.127261f, -0.229588f, 0.250831f,  -0.176929f, -0.031689f, 0.284718f,
  0.085845f,  -0.285027f, 0.012304f,  0.382402f,  -0.204591f, 0.272514f,
  -0.065854f, -0.054228f, -0.231174f, -0.174504f, 0.258287f,  0.195689f,
  0.242530f,  0.023528f,  -0.294242f, -0.272132f, 0.460180f,  -0.731281f,
  -0.208103f, 0.208204f,  0.348250f,  0.016328f,  0.043707f,  -0.169551f,
  0.108521f,  0.226895f,  -0.020471f, 0.102443f,  0.429640f,  -0.252555f,
  -0.218434f, -0.163665f, 0.175531f,  0.101588f,  -0.135798f, -0.158102f,
  0.142565f,  0.128277f,  0.174985f,  -0.100073f, 0.113967f,  0.223682f,
  -0.145576f, -0.008443f, 0.112748f,  -0.037845f, 0.076954f,  -0.287137f,
  -0.518185f, -0.106833f, 0.175359f,  0.031408f,  0.219069f,  -0.294440f,
  0.007766f,  0.067754f,  -0.049168f, -0.212368f, -0.261708f, 0.309252f,
  0.220859f,  -0.274852f, -0.653157f, 0.083438f,  -0.265386f, 0.174429f,
  -0.116931f, -0.091594f, -0.244897f, -0.089015f, 0.274453f,  0.212890f,
  0.272053f,  -0.425315f, -0.107726f, 0.294444f,  -0.354629f, 0.104402f,
  -0.307663f, 0.558430f,  0.140334f,  -0.054831f, -0.449456f, 0.058274f,
  -0.033768f, -0.354117f, -0.331618f, -0.411772f, 0.232064f,  -0.079297f,
  -0.638571f, 0.181823f,  -0.039611f, 0.206310f,  -0.659157f, -0.102930f,
  -0.067303f, -0.176881f, -0.001038f, 0.091835f,  0.079739f,  -0.121923f,
  0.211070f,  0.362719f,  -0.154915f, -0.151876f, -0.165460f, 0.023469f,
  -0.251036f, 0.210014f,  -0.537125f, 0.156832f,  -0.216987f, 0.062975f,
  -0.198462f, 0.329123f,  0.125870f,  0.225830f,  0.086377f,  -0.128773f,
  -0.179673f, -0.074612f, 0.456645f,  0.021905f,  -0.243140f, 0.059145f,
  -0.273942f, -0.277822f, 0.154556f,  -0.025459f, 0.227614f,  -0.313076f,
  0.044705f,  -0.019017f, 0.108999f,  -0.020243f, -0.016373f, 0.560270f,
  -0.064818f, 0.050880f,  -0.218458f, 0.825699f,  -0.534056f, -0.258253f,
  0.222073f,  0.013295f,  0.477870f,  -0.386727f, 0.388509f,  0.004128f,
  0.451388f,  -0.175788f, 0.264093f,  -0.109812f, 0.358132f,  0.500992f,
  -0.446933f, -0.222397f, 0.345834f,  0.370943f,  -0.233115f, -0.047005f,
  -0.111335f, -0.111586f, 0.026975f,  -0.052191f, -0.111800f, -0.129782f,
  0.225132f,  0.102524f,  0.544557f,  -0.111674f, -0.857884f, 0.133258f,
  0.310001f,  0.043829f,  0.104143f,  0.256493f,  0.242520f,  -0.342082f,
  0.421447f,  0.124227f,  0.061542f,  -0.090206f, 0.316681f,  0.353452f,
  -0.918408f, -0.001903f, -0.052303f, -0.004816f, -0.446393f, -0.053038f,
  0.255725f,  -0.126346f, 0.034095f,  -0.240276f, -0.135918f, 0.095682f,
  -0.147457f, -0.338216f, -0.200426f, 0.010265f,  -0.243915f, -0.231375f,
  -0.323924f, -0.014353f, 0.150252f,  -0.264346f, 0.205303f,  -0.194610f,
  -0.282527f, 0.180555f,  -0.000087f, 0.027240f,  -0.000903f, -0.345877f,
  -0.353274f, -0.311829f, 0.172985f,  -0.111748f, -0.309380f, 0.108110f,
  -0.260914f, -0.164990f, 0.183625f,  -0.319692f, -0.096988f, 0.094147f,
  -0.047062f, -0.080978f, 0.227387f,  -0.000450f, -0.220159f, -0.211448f,
  -0.020885f, -0.139646f, -0.086721f, 0.067928f,  -0.033084f, -0.251996f,
  0.090317f,  0.086313f,  -0.228420f, -0.111356f, -0.314304f, -0.223664f,
  0.188176f,  -0.002360f, -0.029491f, -0.006000f, -0.075343f, 0.173699f,
  -0.272800f, -0.238507f, -0.272071f, -0.015000f, -0.215305f, -0.192943f,
  -0.038595f, 0.119537f,  0.260477f,  -0.168014f, -0.172751f, 0.532861f,
  -0.753250f, -0.017485f, -0.115541f, -0.109291f, -1.098943f, 0.418559f,
  -0.532110f, 0.359323f,  -0.254786f, 0.471316f,  -0.545024f, 0.291912f,
  -0.836939f, 0.443427f,  -0.441709f, 0.168866f,  -0.140372f, 0.546607f,
  -0.315465f, 0.023328f,  0.137709f,  -0.083492f, -0.049986f, -0.071302f,
  -0.293680f, -0.105049f, 0.315317f,  0.279569f,  0.220762f,  0.088161f,
  -0.756456f, -0.074512f, 0.958318f,  -0.332924f, -0.004906f, -0.629271f,
  0.212050f,  0.279123f,  0.311523f,  -0.599580f, 0.516150f,  0.456952f,
  0.020255f,  0.247290f,  -0.182670f, -0.335554f, 0.021203f,  0.131081f,
  -0.208584f, 0.112530f,  -0.198980f, 0.211583f,  -0.101271f, -0.206453f,
  -0.502688f, -0.294976f, -0.187019f, -0.114473f, 0.282050f,  -0.165483f,
  0.094953f,  -0.182578f, 0.055068f,  0.135605f,  -0.266941f, -0.297556f,
  0.199181f,  0.015979f,  -0.158659f, -0.226841f, 0.171306f,  0.013438f,
  -0.286309f, -0.071753f, -0.170300f, -0.238188f, 0.093572f,  -0.026230f,
  -0.254502f, -0.297786f, -0.063480f, -0.300799f, -0.065644f, 0.074710f,
  0.248576f,  -0.144425f, -0.113948f, -0.247297f, 0.276682f,  0.010963f,
  -0.737786f, 0.026347f,  0.007830f,  0.753543f,  0.371904f,  0.305614f,
  0.105028f,  0.073530f,  -0.119137f, 0.102352f,  -0.080523f, 0.176366f,
  -0.159457f, -0.339948f, 0.360131f,  -0.007051f, -0.388378f, -0.101695f,
  0.663041f,  -0.234486f, -0.142536f, -0.099931f, 0.041478f,  0.230425f,
  0.005743f,  0.154060f,  0.056233f,  -0.080668f, -0.009754f, -0.194356f,
  0.185474f,  -0.296474f, 0.192700f,  0.257767f,  0.348529f,  0.458265f,
  0.060276f,  -0.130473f, 0.139889f,  0.310073f,  -0.306869f, -0.272922f,
  -0.259862f, 0.409207f,  0.431991f,  -0.100357f, -0.050415f, -0.071830f,
  -0.239665f, 0.153399f,  0.177192f,  -0.611644f, -0.176114f, -0.022694f,
  -0.033701f, -0.345842f, 0.015660f,  0.158931f,  -0.097586f, 0.222001f,
  0.257887f,  -0.171307f, -0.222607f, -0.245508f, -0.145742f, -0.096461f,
  -0.010895f, 0.052815f,  -0.265306f, -0.081059f, 0.219162f,  -0.256084f,
  -0.372676f, 0.148977f,  0.174831f,  0.086980f,  0.108518f,  0.074011f,
  0.038032f,  -0.070856f, -0.109407f, 0.126174f,  0.022341f,  -0.249786f,
  -0.356164f, -0.202841f, -0.087437f, -0.133740f, 0.090956f,  -0.017953f,
  -0.028353f, 0.233621f,  0.109426f,  0.232798f,  -0.104950f, -0.241798f,
  -0.018995f, -0.167954f, 0.002473f,  0.060418f,  -0.232717f, -0.195980f,
  -0.283971f, -0.371881f, 0.219728f,  0.018072f,  -0.166694f, -0.083301f,
  -0.000616f, -0.212641f, -0.173158f, 0.222739f,  -0.235302f, 0.237624f,
  0.222232f,  -0.041235f, -0.342411f, 0.121194f,  0.211291f,  -0.032237f,
  -0.249401f, -0.291668f, 0.206055f,  -0.148200f, 0.011824f,  -0.272728f,
  -0.194854f, 0.367175f,  -0.257243f, 0.103433f,  -0.231077f, 0.236734f,
  0.135733f,  -0.362845f, 0.197147f,  0.242782f,  -0.135289f, 0.123311f,
  0.259420f,  -0.116278f, 0.127287f,  0.236789f,  -0.097438f, 0.118073f,
  0.112796f,  -0.035949f, 0.184408f,  0.200948f,  -0.008859f, 0.195989f,
  0.161970f,  -0.295320f, -0.330389f, 0.141034f,  0.066081f,  -0.707857f,
  0.357037f,  0.149633f,  0.679877f,  0.548674f,  0.469076f,  0.194123f,
  -0.209872f, -0.071764f, -0.126960f, 0.199420f,  0.327116f,  -0.169053f,
  -0.429156f, 0.443429f,  -0.225530f, -0.130738f, -0.028351f, 0.644393f,
  0.049606f,  -0.243602f, -0.409920f, 0.117028f,  -0.258557f, 0.073865f,
  -0.200454f, -0.139957f, -0.031314f, 0.162325f,  0.247221f,  0.071909f,
  -0.336276f, 0.079922f,  0.192780f,  -0.148882f, 0.133192f,  -0.143177f,
  -0.121327f, 0.126221f,  -0.089521f, -0.181826f, 0.149923f,  -0.280682f,
  0.391572f,  0.108990f,  -0.445494f, -0.170787f, 0.225182f,  0.223313f,
  -0.234828f, -0.071072f, -0.072673f, -0.093686f, 0.223892f,  -0.049377f,
  0.057976f,  0.033558f,  0.068733f,  -0.283353f, 0.217877f,  0.158093f,
  -0.276761f, -0.097049f, -0.351913f, -0.383604f, 0.002863f,  -0.474510f,
  -0.096738f, 0.256940f,  0.234203f,  -0.226667f, -0.260576f, -0.183403f,
  -0.035578f, 0.141570f,  0.078764f,  -0.028086f, 0.155800f,  -0.251115f,
  -0.286703f, -0.014739f, -0.072621f, -0.311506f, -0.048639f, 0.081621f,
  0.043057f,  0.068136f,  -0.179903f, 0.143699f,  -0.002571f, 0.239012f,
  0.197456f,  0.035745f,  -0.311927f, 0.220320f,  0.102687f,  -0.294105f,
  0.426740f,  0.209050f,  0.211907f,  0.083453f,  0.006578f,  -0.143338f,
  0.003157f,  0.040295f,  0.234497f,  0.035344f,  -0.163909f, 0.411115f,
  0.289453f,  -0.075357f, -0.008884f, 0.469798f,  -0.033304f, -0.153293f,
  -0.229322f, -0.004162f, 0.113363f,  0.395381f,  0.067414f,  -0.188966f,
  -0.117424f, -0.166423f, 0.066839f,  0.595641f,  -0.204782f, -0.451727f,
  0.198509f,  -0.921583f, -0.246765f, -0.153411f, 0.046491f,  0.365906f,
  0.376710f,  -0.017355f, -0.035232f, 0.138785f,  -0.163918f, -0.283449f,
  -0.094340f, 0.192127f,  0.154815f,  0.035787f,  -0.029087f, 0.115649f,
  -0.220133f, -0.452741f, 0.311667f,  0.157666f,  0.091401f,  0.236040f,
  -0.168523f, 0.122176f,  -0.219016f, -0.214856f, 0.172824f,  -0.091810f,
  0.031520f,  -0.857420f, 0.643446f,  -0.017471f, 0.206082f,  -0.933517f,
  -0.020070f, -0.065091f, -0.117680f, -1.271870f, -0.069177f, -0.149409f,
  0.289970f,  -0.889775f, -0.044741f, 0.232647f,  -0.319416f, 0.073030f,
  0.278549f,  0.238782f,  -0.202206f, 0.272540f,  0.201412f,  0.175574f,
  -0.127971f, -0.253164f, -0.086352f, -0.005381f, 0.114714f,  0.505169f,
  -0.175049f, -1.534280f, -0.320666f, -2.119298f, -0.023075f, -0.021259f,
  -0.161019f, 0.344837f,  0.361958f,  -0.097050f, 0.014375f,  0.267110f,
  0.341442f,  -0.016688f, 0.073393f,  0.131500f,  0.246331f,  0.011059f,
  0.033597f,  0.014779f,  -0.269366f, -0.504788f, 0.048651f,  0.295682f,
  0.237363f,  0.227484f,  -0.235814f, -0.160530f, 0.182682f,  -0.172999f,
  -0.126630f, 0.168357f,  -0.078729f, 0.052805f,  0.377021f,  -0.004727f,
  0.230415f,  -0.876673f, 0.458457f,  0.099401f,  -0.019616f, 0.611982f,
  -0.231508f, -0.070894f, -0.056142f, 0.548969f,  -0.376599f, -0.600428f,
  0.241930f,  -0.592893f, 0.189371f,  0.488651f,  -0.092446f, -0.272569f,
  0.251643f,  0.315945f,  -0.301468f, 0.112961f,  0.052119f,  -0.066076f,
  -0.082249f, 0.252805f,  -0.195539f, 0.150386f,  -0.865534f, 0.673447f,
  0.030177f,  -0.438528f, -1.006174f, 0.575176f,  -0.271656f, 0.035835f,
  -1.056916f, 0.495267f,  -0.092428f, -0.109511f, -0.192359f, 0.166669f,
  -0.624326f, -0.000354f, -0.089075f, 0.176279f,  -0.289347f, 0.021346f,
  0.020375f,  0.255282f,  -0.045588f, 0.173675f,  0.100957f,  -0.294373f,
  0.049303f,  -0.134132f, -0.255731f, -0.025559f, -0.307463f, -0.205100f,
  0.079024f,  0.101113f,  0.135742f,  -0.348869f, -0.026759f, -0.134155f,
  -0.179275f, -0.054297f, -0.054948f, 0.029351f,  0.190560f,  0.102476f,
  -0.025785f, 0.169442f,  -0.271303f, 0.200667f,  0.099063f,  0.074767f,
  -0.326533f, 0.044426f,  -0.290251f, -0.082443f, -0.164482f, -0.349412f,
  0.045109f,  -0.157330f, 0.165935f,  0.012672f,  -0.059818f, 0.399140f,
  -0.316620f, 0.386638f,  -0.285399f, -0.296777f, -0.200473f, -0.144232f,
  0.251851f,  -0.203768f, 0.001071f,  -0.179063f, 0.248952f,  -0.143029f,
  0.010423f,  -0.030293f, -0.046786f, -0.196195f, -0.016845f, 0.295023f,
  0.322825f,  0.133683f,  0.017388f,  0.142467f,  0.221320f,  0.004059f,
  -0.115770f, 0.143363f,  0.137972f,  -0.272584f, 0.489366f,  -0.091828f,
  -0.014703f, 0.082332f,  -0.476226f, -0.202859f, 0.356094f,  -0.283049f,
  0.218086f,  0.202015f,  0.201724f,  0.012617f,  0.050720f,  0.255695f,
  0.244653f,  0.111296f,  -0.151450f, -0.056210f, -0.757348f, 0.441724f,
  -0.022455f, -0.244662f, 0.296205f,  -0.421883f, -0.217386f, -0.254301f,
  0.409105f,  -0.031309f, 0.050147f,  -0.337170f, -0.106620f, -0.606455f,
  0.308024f,  0.298144f,  0.363993f,  0.704870f,  -0.047292f, 0.166901f,
  0.105991f,  -0.536757f, -0.424031f, -0.226034f, 0.213635f,  -0.526754f,
  0.310990f,  -0.116038f, 0.007775f,  0.538330f,  -0.177912f, 0.445357f,
  -0.290365f, 0.451169f,  0.030931f,  0.033388f,  0.209905f,  -0.244492f,
  -0.097792f, -0.246042f, 0.132047f,  0.032576f,  0.115516f,  0.022890f,
  0.093508f,  -0.071840f, 0.362948f,  -0.135245f, 0.659911f,  -0.321413f,
  0.193118f,  -0.795001f, -0.218311f, 0.024862f,  0.206172f,  -0.832878f,
  -0.255670f, 0.343402f,  -0.275211f, -0.898363f, -0.025172f, 0.158565f,
  0.171347f,  -0.127518f, -0.215156f, -0.159198f, 0.250355f,  -0.132452f,
  0.061254f,  -0.097544f, -0.223246f, 0.013183f,  0.239468f,  0.259017f,
  -0.217739f, -0.032263f, 0.123755f,  -0.701777f, 0.150049f,  -0.555293f,
  0.062430f,  -0.260304f, 0.494894f,  -0.168702f, -0.134829f, -0.113989f,
  0.150092f,  -0.060248f, 0.115711f,  -0.277202f, 0.499811f,  0.417116f,
  0.191081f,  -0.376432f, -0.321092f, 0.033992f,  0.057193f,  0.127077f,
  -0.009042f, 0.014443f,  0.142808f,  -0.124349f, 0.213087f,  -0.381686f,
  0.129726f,  -0.038396f,
};

static const float av1_early_term_after_split_nn_bias_32_layer0[] = {
  -0.107171f, 0.060848f,  -0.069480f, -0.121982f, 0.037637f,  -0.291839f,
  0.102257f,  -0.065889f, -0.032452f, 0.034171f,  -0.073984f, -0.005236f,
  0.218820f,  0.132123f,  -0.089621f, -0.067679f, 0.049368f,  0.329444f,
  -0.184729f, 0.031702f,  0.009735f,  -0.039964f, -0.018024f, -0.073031f,
  -0.030166f, -0.191037f, -0.074862f, -0.076548f, 0.076537f,  0.216609f,
  -0.078358f, -0.007740f,
};

static const float av1_early_term_after_split_nn_weights_32_layer1[] = {
  0.047869f,  -0.231773f, -0.185663f, 0.460676f,  -0.208182f, 0.590555f,
  -0.622627f, 0.279377f,  0.351681f,  0.633504f,  1.069884f,  0.332449f,
  -0.457703f, -0.435817f, -0.028853f, 0.327490f,  -0.282469f, -0.975792f,
  -0.062975f, -0.147187f, 0.348340f,  -1.207116f, 0.516159f,  -1.509626f,
  -0.805072f, 0.522999f,  0.143671f,  0.304246f,  -0.360720f, -0.612472f,
  0.260045f,  -0.223243f,
};

static const float av1_early_term_after_split_nn_bias_32_layer1[] = {
  -0.07571174f,
};

static const NN_CONFIG av1_early_term_after_split_nnconfig_32 = {
  FEATURES,
  1,
  1,
  {
      HIDDEN_NODES,
  },
  {
      av1_early_term_after_split_nn_weights_32_layer0,
      av1_early_term_after_split_nn_weights_32_layer1,
  },
  {
      av1_early_term_after_split_nn_bias_32_layer0,
      av1_early_term_after_split_nn_bias_32_layer1,
  },
};

static const float av1_early_term_after_split_nn_weights_16_layer0[] = {
  -0.113798f, 0.053357f,  -0.037947f, -0.477171f, 0.276517f,  -0.349252f,
  -0.177284f, 0.189597f,  0.141744f,  0.230207f,  -0.328104f, 0.074328f,
  0.247717f,  0.233533f,  0.145167f,  0.018029f,  -0.398725f, -0.226199f,
  -0.309724f, 0.125279f,  0.194759f,  0.025531f,  0.349714f,  -0.273944f,
  0.186871f,  0.181735f,  -0.520614f, -0.264076f, 0.308207f,  0.157438f,
  -0.137791f, -0.054582f, 0.125879f,  0.796218f,  -0.897562f, 0.885439f,
  0.381640f,  0.106625f,  -2.027456f, 0.000874f,  0.179581f,  0.013287f,
  -2.329439f, -0.163169f, -0.136191f, 0.320108f,  -2.318779f, -0.196722f,
  -0.295721f, 0.203658f,  -0.182275f, 0.615941f,  0.015762f,  0.257181f,
  -0.115297f, 0.295774f,  -0.026144f, -0.022686f, -0.219423f, -0.042861f,
  0.207647f,  -0.057791f, 0.201671f,  -0.169569f, 0.291492f,  -0.994991f,
  0.137473f,  0.230948f,  0.505626f,  -1.065860f, 0.275225f,  -0.250861f,
  0.519466f,  -1.217242f, -0.087384f, 0.053441f,  0.030729f,  -1.702304f,
  -0.034635f, 0.010177f,  -0.035422f, -0.749979f, 0.355499f,  0.408166f,
  -0.086883f, 0.017203f,  0.195706f,  -0.218056f, -0.029153f, 0.367335f,
  -0.061732f, -0.241068f, 0.078496f,  -0.370346f, -0.124223f, -0.172708f,
  0.037971f,  0.038875f,  -0.282489f, -0.266323f, -0.210864f, 0.214714f,
  0.234695f,  -0.045625f, 0.015357f,  -0.007464f, -0.362003f, -0.113465f,
  0.145141f,  0.238470f,  -0.202664f, -0.286587f, -0.347112f, 0.054501f,
  -0.190290f, -0.283256f, 0.062179f,  0.041165f,  -0.006935f, -0.220351f,
  -0.088800f, 0.220924f,  -0.200982f, 0.058493f,  -0.225175f, 0.057175f,
  -0.618187f, 0.761023f,  -0.743774f, -0.500599f, -0.584999f, 1.545211f,
  0.123055f,  -0.106848f, -0.353057f, 1.552187f,  0.174104f,  0.068060f,
  -0.449859f, 1.254299f,  -0.161716f, -0.060630f, -0.230721f, 0.165976f,
  -0.101582f, -0.422415f, 0.110384f,  -0.130098f, 0.104428f,  0.083518f,
  0.031626f,  0.083048f,  0.158877f,  0.173340f,  0.063962f,  0.427845f,
  0.663268f,  0.376996f,  0.146435f,  -0.091329f, 0.443447f,  0.518432f,
  -0.182777f, -0.091313f, 0.331229f,  0.532604f,  -0.187001f, 0.054774f,
  0.298068f,  0.502295f,  -0.362378f, 0.054283f,  0.292806f,  0.168901f,
  -0.214787f, 0.025637f,  0.458009f,  -0.322714f, -0.264059f, 0.140313f,
  -0.102696f, -0.431208f, -0.134450f, -0.545415f, 0.253851f,  -0.009061f,
  -0.050681f, 0.108681f,  0.043272f,  -1.073133f, 0.206410f,  0.469576f,
  0.291494f,  -2.021244f, -0.001183f, -0.067542f, 0.364907f,  -2.470543f,
  0.049147f,  -0.018868f, 0.658500f,  -2.531048f, 0.275433f,  -0.034224f,
  -0.171386f, 0.096369f,  0.728069f,  0.272332f,  0.222255f,  -0.030426f,
  0.026994f,  0.208928f,  -0.173943f, -0.227581f, -0.214798f, 0.079341f,
  0.032344f,  -0.253575f, -0.044353f, -0.239265f, -0.055852f, -0.162582f,
  -0.086592f, 0.066487f,  0.337353f,  -0.168704f, 0.015702f,  0.022607f,
  0.286647f,  0.218106f,  0.193319f,  -0.358714f, 0.030796f,  0.007646f,
  -0.045617f, 0.165007f,  -0.284641f, -0.291812f, 0.207544f,  0.082823f,
  -0.141907f, -0.331336f, -0.052908f, 0.120716f,  0.202521f,  0.232782f,
  -0.348141f, -0.017332f, 1.191126f,  -0.391987f, -0.154537f, -0.206551f,
  -2.378690f, 0.057918f,  -0.328183f, 2.151556f,  0.238803f,  0.164880f,
  -0.480039f, 1.616200f,  0.260243f,  0.083704f,  -0.174461f, 1.804634f,
  0.194810f,  0.223837f,  0.550107f,  -0.068171f, -0.293435f, -0.186770f,
  -0.364846f, 0.127181f,  0.105556f,  -0.016202f, 0.278403f,  -0.344995f,
  -0.009761f, -0.082555f, 0.046731f,  -0.301452f, 0.604259f,  0.055895f,
  0.049862f,  0.314249f,  -0.305811f, -0.112937f, 0.658787f,  -0.549288f,
  -0.307567f, -0.460650f, -0.840643f, 0.082576f,  0.373711f,  0.138318f,
  0.336901f,  0.284984f,  -0.281400f, 0.408210f,  -0.449858f, 0.461054f,
  0.227629f,  -0.131705f, 0.301769f,  -0.278540f, 0.189290f,  -0.269041f,
  0.111350f,  -0.300257f, 0.436858f,  -0.265920f, -0.211938f, 0.272631f,
  0.206291f,  0.253273f,  -0.229776f, -0.031112f, -0.171183f, -0.109676f,
  -0.202390f, -0.068857f, 0.182125f,  -0.140523f, -0.308742f, -0.045840f,
  0.256545f,  -0.262405f, 0.225951f,  -0.287463f, -0.189203f, -0.055552f,
  -0.052448f, -0.242839f, -0.278877f, 0.140920f,  -0.175755f, 0.215402f,
  -0.248841f, -0.264080f, -0.178303f, 0.147777f,  0.049460f,  -0.279877f,
  -0.539725f, -0.004622f, 0.182874f,  0.338814f,  0.265974f,  0.249851f,
  -0.141154f, 0.157228f,  -0.090972f, 0.179444f,  0.305255f,  0.127788f,
  0.123270f,  0.355320f,  0.076797f,  0.263495f,  0.235965f,  -0.133816f,
  0.243624f,  0.227062f,  -0.213629f, 0.002075f,  0.061203f,  -0.077820f,
  -0.008807f, -0.247324f, -0.051464f, -0.191894f, -0.238713f, -0.389526f,
  -0.274248f, 0.053950f,  -0.225750f, -0.367097f, -0.122391f, 0.181212f,
  -0.411824f, -0.084241f, -0.302288f, 0.077860f,  -0.187443f, -0.300262f,
  0.083156f,  -0.392461f, -0.332320f, -0.346474f, 0.140658f,  -0.283656f,
  0.120714f,  -0.056577f, -0.280968f, 0.017795f,  -0.024686f, 0.073113f,
  -0.346637f, 0.082567f,  -0.036556f, -0.369730f, 0.081225f,  -0.005211f,
  0.144886f,  -0.003544f, 0.178307f,  -0.366035f, -0.063887f, -0.191767f,
  0.105835f,  -0.273978f, -0.266532f, -0.023984f, 0.039166f,  0.065848f,
  -0.026802f, -0.268923f, 0.189659f,  0.086300f,  0.030718f,  0.216565f,
  -0.130025f, -0.215687f, 0.146341f,  -0.286438f, -0.394226f, -0.181509f,
  -0.005612f, 0.186040f,  0.133491f,  0.032096f,  -0.261609f, 0.074007f,
  -0.042929f, -0.234479f, 0.189704f,  0.088395f,  -0.003671f, -0.125055f,
  -0.252418f, -0.086387f, 0.111197f,  -0.297071f, -0.018793f, -0.031902f,
  -0.333191f, -0.186279f, 0.039868f,  0.091419f,  -0.264438f, -0.216150f,
  -0.212550f, 0.203412f,  -0.113028f, -0.197169f, -0.346771f, 0.086066f,
  0.091443f,  -0.128507f, -0.007281f, -0.118389f, 0.003370f,  -0.338661f,
  0.026739f,  -0.063571f, -0.281567f, -0.166824f, 0.167455f,  0.216173f,
  0.199163f,  0.256314f,  -0.222679f, 0.040282f,  -0.154808f, -0.133943f,
  -0.270163f, -0.357398f, 0.260373f,  0.176950f,  -0.125162f, -0.085050f,
  0.226376f,  -0.124585f, -0.324804f, 0.035536f,  -0.133600f, 0.173450f,
  0.068107f,  -0.337442f, 0.169629f,  0.047223f,  0.057878f,  0.055555f,
  -0.317449f, -0.103768f, 0.080899f,  -0.194759f, -1.137593f, 0.508999f,
  0.045372f,  1.746454f,  1.250347f,  -0.342930f, -0.127821f, -0.220175f,
  -0.417649f, -0.480595f, 0.071902f,  0.050231f,  -0.562554f, -0.677866f,
  -0.121416f, -0.247558f, -0.483876f, -0.504157f, 1.731953f,  0.572936f,
  0.047325f,  0.050619f,  0.112611f,  -0.035393f, 0.052585f,  -0.071076f,
  -0.015798f, -0.050228f, -0.142875f, 0.189329f,  0.048833f,  0.503633f,
  0.249588f,  0.175492f,  -0.137664f, -0.018533f, 0.288453f,  -0.025644f,
  0.079131f,  0.195096f,  -0.154039f, -0.104220f, -0.224072f, 0.095946f,
  -0.208424f, 0.214745f,  0.056468f,  0.182603f,  0.341784f,  -0.134664f,
  -0.194050f, 0.058532f,  -0.107336f, -0.087783f, -0.238795f, -0.387212f,
  0.049055f,  -0.127417f, -0.299919f, -0.094371f, -0.011735f, -0.264753f,
  0.407375f,  -0.462654f, -0.609488f, 0.027742f,  -0.985512f, -0.109154f,
  -0.423276f, 2.347960f,  0.129240f,  0.187610f,  -0.057081f, 2.424892f,
  0.087666f,  0.106716f,  -0.039379f, 2.764866f,  0.113309f,  0.028196f,
  -0.582789f, 0.335385f,  -0.538029f, -0.477337f, -0.114207f, 0.178829f,
  0.006276f,  0.123179f,  0.095101f,  0.139898f,  -0.372074f, -0.111010f,
  0.136330f,  0.272900f,  0.126737f,  -0.097808f, -0.363697f, 0.108665f,
  -0.227749f, -0.083421f, 1.714677f,  0.451943f,  0.107931f,  -0.392281f,
  1.615846f,  0.022307f,  -0.247011f, 0.257703f,  1.039134f,  0.537789f,
  0.022177f,  -0.271532f, 0.351350f,  -0.399205f, -0.240534f, -0.315399f,
  0.026928f,  -0.005618f, 0.053179f,  -0.010277f, 0.000501f,  0.040896f,
  -0.109160f, 0.018282f,  0.003887f,  0.199599f,  0.095349f,  -0.337284f,
  0.169929f,  -0.109409f, -0.166983f, 0.059908f,  -0.226574f, -0.120114f,
  0.077329f,  -0.333133f, -0.220936f, 0.114309f,  -0.233965f, -0.281551f,
  0.042948f,  0.100940f,  0.116037f,  -0.313122f, 0.215149f,  -0.309057f,
  -0.341052f, -0.294417f, -0.179722f, 0.010795f,  0.192053f,  -0.275261f,
  -0.033077f, 0.117348f,  0.090206f,  0.781573f,  0.602456f,  -0.220296f,
  0.172159f,  0.758513f,  0.157910f,  -0.217897f, -0.372659f, 0.031935f,
  0.791463f,  0.267195f,  0.931593f,  -0.057349f, 0.405512f,  -0.058512f,
  -0.641663f, -0.076592f, 0.550227f,  -0.024094f, 0.048218f,  -0.289971f,
  0.180940f,  0.167533f,  0.052711f,  -0.360726f, 0.019210f,  -0.488879f,
  0.380498f,  0.151608f,  -0.276895f, -0.596554f, 0.106076f,  -0.245833f,
  -0.048783f, 0.073823f,  0.098780f,  0.000211f,  0.113958f,  -0.068964f,
  -0.265533f, -0.185457f, 0.175586f,  -0.163621f, -0.204919f, 0.145802f,
  -0.163421f, 0.129576f,  -0.153486f, -0.105573f, 0.067289f,  -0.213120f,
  -0.286103f, 0.249543f,  -0.044970f, -0.170464f, -0.105501f, -0.094765f,
  -0.050734f, -0.369468f, 0.180020f,  -0.363328f, -0.151654f, -0.262550f,
  -0.424503f, 0.829032f,  -0.559452f, 0.506837f,  0.143823f,  0.276660f,
  -1.808608f, -0.259517f, -0.053945f, 0.035676f,  -1.842195f, -0.065960f,
  -0.069285f, 0.462022f,  -2.319453f, -0.370299f, 0.183329f,  -0.146412f,
  -0.563875f, 0.305068f,  0.480904f,  0.044319f,  -0.016098f, 0.168516f,
  0.114874f,  -0.097621f, -0.030373f, 0.177700f,  0.181591f,  -0.146003f,
  -0.330853f, -0.259200f, 0.779319f,  -1.517524f, 0.178781f,  0.135451f,
  0.088784f,  -2.076089f, 0.628717f,  -0.048685f, 0.281327f,  -2.341596f,
  0.422171f,  0.006135f,  0.367096f,  -1.663118f, 0.365253f,  -0.072884f,
  -0.197620f, -0.688634f, 0.477354f,  0.395841f,  -0.098505f, 0.208709f,
  -0.027523f, 0.127119f,  0.106274f,  0.114424f,  -0.122877f, -0.087245f,
  0.086923f,  -0.527398f, -0.342062f, -0.764662f, 0.713094f,  -0.626453f,
  -0.081454f, -0.087683f, 0.885047f,  0.323440f,  -0.018579f, -0.217166f,
  1.617984f,  -0.159038f, 0.265991f,  -0.390313f, 1.933182f,  -0.032431f,
  -0.057513f, -0.300841f, 0.461248f,  -0.072147f, -0.287052f, -0.078056f,
  0.011734f,  0.044013f,  0.177174f,  0.093400f,  0.028819f,  0.193686f,
  -0.224853f, 0.268321f,  -0.075059f, 0.074526f,  -0.015618f, 0.165615f,
  -0.276780f, -0.063908f, -0.369264f, -0.171497f, -0.173624f, -0.130743f,
  -0.224625f, -0.124980f, -0.104482f, 0.076864f,  -0.009631f, -0.164682f,
  0.150480f,  -0.111880f, -0.260425f, 0.086234f,  -0.176936f, -0.136771f,
  -0.168867f, -0.405626f, -0.288716f, -0.128950f, -0.207327f, 0.015581f,
  -0.109061f, -0.098970f, 0.090792f,  -0.109623f, 0.349851f,  0.266341f,
  -0.088602f, -0.108071f, 0.082519f,  0.472650f,  -1.838758f, 0.456694f,
  0.119927f,  0.461077f,  -2.860022f, 0.231495f,  0.235771f,  0.256424f,
  -1.938516f, -0.188202f, -0.000832f, -0.518206f, 0.194644f,  0.505510f,
  0.615657f,  0.193760f,  0.224600f,  0.265732f,  -0.121553f, -0.354597f,
  -0.242414f, -0.276639f, -0.057591f, 0.026369f,  -0.261148f, -0.356155f,
  -0.149178f, -0.353566f, -0.340835f, -0.141776f, 0.076535f,  0.221299f,
  -0.108857f, -0.156514f, 0.050901f,  0.058541f,  -0.077141f, 0.071515f,
  -0.333283f, -0.181489f, -0.212900f, -0.224698f, -0.174693f, -0.178665f,
  -0.143374f, -0.091811f, 0.165161f,  0.060156f,  -0.086103f, -0.039031f,
  -0.377759f, -0.370533f, 0.074431f,  0.064192f,  0.186576f,  0.447858f,
  -0.082260f, -0.020268f, -0.123089f, -0.402017f, 0.080500f,  0.176286f,
  2.850013f,  0.019385f,  -0.225361f, -0.235315f, 1.654694f,  -0.073978f,
  -0.341412f, -1.187575f, 2.815900f,  -0.228063f, -0.174547f, 0.623825f,
  -0.010676f, 0.157189f,  0.111879f,  -0.198965f, 0.051851f,  0.158396f,
  0.045194f,  0.293531f,  -0.246714f, -0.351493f, 0.026954f,  0.076233f,
  0.420367f,  0.168154f,  -0.131450f, 0.134487f,  -0.288851f, -0.134553f,
  0.014902f,  0.756381f,  0.277713f,  0.190080f,  -0.020869f, 1.446672f,
  0.029792f,  -0.025927f, 0.060640f,  0.559864f,  0.422229f,  0.198459f,
  0.036167f,  0.029432f,  0.001882f,  0.038480f,  -0.160528f, -0.288855f,
  -0.310886f, 0.291296f,  0.190558f,  -0.182816f, -0.002252f, 0.073101f,
  -0.172245f, -0.305980f, 0.112492f,  -0.422839f, -0.295999f, -0.078160f,
  -0.173405f, -0.032819f, 0.373774f,  -0.715223f, 0.018911f,  0.131753f,
  -0.237364f, -0.128499f, -0.228406f, 0.341619f,  0.343552f,  -0.521581f,
  -0.263790f, 0.362502f,  -0.018450f, 0.054233f,  0.183068f,  0.382772f,
  0.188811f,  -0.627287f, 0.040399f,  -0.487338f, -0.192591f, 0.247426f,
  0.154372f,  -0.483994f,
};

static const float av1_early_term_after_split_nn_bias_16_layer0[] = {
  -0.173976f, 0.305495f,  0.250981f,  -0.067127f, -0.313100f, 0.242464f,
  0.315196f,  -0.056052f, -0.241227f, -0.253308f, -0.002697f, 0.003687f,
  -0.124421f, -0.090383f, -0.070366f, -0.064074f, -0.056115f, 0.123313f,
  -0.239698f, -0.182082f, -0.065296f, 0.021503f,  -0.036787f, 0.311861f,
  0.118135f,  -0.320456f, -0.110719f, 0.220692f,  -0.071727f, -0.088226f,
  -0.110874f, -0.111671f,
};

static const float av1_early_term_after_split_nn_weights_16_layer1[] = {
  -0.338573f, 0.398159f,  0.314774f,  -0.037448f, -0.271950f, -0.774991f,
  0.950901f,  -0.225380f, -1.841906f, -0.350379f, -0.079350f, 0.383148f,
  -0.183676f, -0.313132f, -0.340820f, -0.309401f, -1.050540f, -0.432267f,
  -0.657195f, 0.927632f,  -0.040150f, 0.578920f,  0.212301f,  0.292495f,
  0.563590f,  -0.205735f, 0.195877f,  0.582122f,  -0.217860f, 1.613379f,
  0.313278f,  -0.555802f,
};

static const float av1_early_term_after_split_nn_bias_16_layer1[] = {
  0.16553f,
};

static const NN_CONFIG av1_early_term_after_split_nnconfig_16 = {
  FEATURES,
  1,
  1,
  {
      HIDDEN_NODES,
  },
  {
      av1_early_term_after_split_nn_weights_16_layer0,
      av1_early_term_after_split_nn_weights_16_layer1,
  },
  {
      av1_early_term_after_split_nn_bias_16_layer0,
      av1_early_term_after_split_nn_bias_16_layer1,
  },
};

static const float av1_early_term_after_split_nn_weights_8_layer0[] = {
  -0.719472f, 0.305806f,  0.855829f,  0.100094f,  0.412517f,  1.254673f,
  1.552105f,  -5.890773f, -0.089957f, -0.016736f, 1.418074f,  -5.393506f,
  -0.028214f, 0.117758f,  1.479209f,  -5.299794f, 0.171585f,  -0.084182f,
  -0.162105f, 0.388577f,  -0.044319f, -0.025861f, 0.251782f,  -0.181462f,
  -0.101545f, -0.079999f, -0.033014f, -0.191627f, -0.032802f, -0.053404f,
  0.038038f,  -0.119492f, 0.049104f,  -0.344384f, -0.354513f, 0.036977f,
  0.017513f,  -0.004025f, -0.163212f, -0.261999f, 0.146575f,  0.207541f,
  0.130365f,  -0.252127f, 0.097419f,  -0.231057f, -0.309421f, 0.347866f,
  -0.064670f, -0.283171f, -0.244193f, -0.193323f, -0.226954f, -0.276194f,
  -0.233553f, 0.156354f,  -0.184009f, 0.344289f,  -0.308058f, -0.205202f,
  -0.325068f, 0.183820f,  -0.361667f, -0.069559f, -0.121834f, -0.038357f,
  -0.210043f, -0.266129f, 0.003188f,  0.074902f,  -0.328843f, 0.293679f,
  -0.234698f, -0.428268f, -0.308772f, -0.136538f, -0.008384f, -0.078227f,
  0.166074f,  -0.262899f, 0.102114f,  -0.323420f, 0.057064f,  -0.203318f,
  -0.397413f, -0.317324f, -0.307093f, 0.020574f,  -0.188627f, 0.132529f,
  0.118992f,  -0.487387f, -0.282975f, 0.573231f,  -0.266071f, 0.125140f,
  -0.970034f, 1.424008f,  -0.487366f, -0.196415f, 3.680273f,  -0.008407f,
  0.081109f,  -0.187479f, 3.876021f,  0.159168f,  0.111721f,  -0.337423f,
  3.901760f,  0.261268f,  -0.245555f, -0.187632f, -0.324298f, 0.167234f,
  0.170986f,  -0.473055f, 0.087016f,  -0.003469f, 0.051035f,  0.251794f,
  0.153549f,  0.217609f,  -0.326870f, -0.175511f, 0.637341f,  -0.694837f,
  -0.873487f, -0.186614f, -1.089884f, -0.607316f, -0.523519f, 5.256331f,
  0.071414f,  0.215265f,  -0.835999f, 5.735746f,  0.300101f,  0.089626f,
  -0.450261f, 5.608051f,  0.190491f,  0.110220f,  -0.595360f, -0.446324f,
  0.311380f,  0.268812f,  -0.339656f, -0.008708f, 0.011111f,  -0.027557f,
  0.171534f,  0.000676f,  0.227232f,  0.033993f,  0.146684f,  0.094817f,
  -0.175381f, -0.211927f, -0.362471f, 0.168834f,  0.264149f,  -0.350538f,
  -0.463249f, -0.288105f, 0.347155f,  0.183231f,  -0.229732f, -0.252202f,
  -0.218074f, -0.008769f, -0.156103f, 0.181233f,  -0.354736f, 0.263270f,
  -0.106636f, 0.081057f,  0.060634f,  -0.046887f, 0.050468f,  0.071259f,
  0.221287f,  0.199071f,  -0.180185f, -0.406902f, -0.239351f, -0.034957f,
  0.369140f,  0.864600f,  0.233798f,  0.423612f,  -0.468918f, 0.976987f,
  0.691198f,  -1.597908f, 0.102926f,  0.305546f,  0.391196f,  -3.909059f,
  0.333635f,  0.311561f,  0.738886f,  -4.002001f, 0.236394f,  -0.233141f,
  0.263342f,  0.679898f,  0.136233f,  0.254743f,  -0.367571f, 0.066412f,
  0.001606f,  -0.059542f, 0.051726f,  -0.347145f, -0.045501f, -0.313847f,
  -0.021952f, 1.386316f,  -0.579139f, -1.275844f, -0.003493f, -1.716577f,
  0.250209f,  0.192086f,  4.177055f,  0.351835f,  0.338177f,  0.140163f,
  4.099592f,  0.321866f,  -0.128153f, -0.360414f, 4.350767f,  0.025943f,
  -0.116740f, -0.664107f, -0.064558f, -0.039553f, -0.208186f, -0.678774f,
  0.149441f,  -0.019823f, 0.012759f,  0.404442f,  -0.108881f, 0.067974f,
  -0.188278f, 0.136327f,  0.109927f,  -0.179270f, -0.272342f, 0.018064f,
  -0.304216f, -0.469470f, 0.109310f,  -0.326214f, 0.061909f,  -0.278997f,
  -0.352329f, -0.333770f, -0.186522f, -0.328567f, -0.206211f, -0.008804f,
  0.042441f,  -0.126699f, -0.420399f, -0.033842f, 0.016773f,  -0.273789f,
  0.081928f,  -0.191552f, -0.179533f, -0.263070f, -0.471807f, 0.062601f,
  -0.232576f, 0.082955f,  -0.490080f, 0.073820f,  -0.090384f, 0.035781f,
  -0.158880f, -0.506793f, -0.069132f, 0.047602f,  -0.349640f, -0.058389f,
  -0.017387f, -0.194636f, -0.457227f, -0.143105f, 0.222045f,  -0.548909f,
  -0.131561f, 0.247196f,  -0.207923f, 0.133056f,  -0.509854f, -0.193685f,
  -0.181327f, -0.242442f, 0.091821f,  0.114430f,  -0.375233f, -0.015254f,
  -0.336632f, -0.060279f, -0.169169f, -0.429914f, -0.036563f, -0.400560f,
  -0.076332f, -0.186232f, -0.268491f, 0.075561f,  -0.389082f, -0.077435f,
  0.352562f,  -0.020086f, -0.338181f, -0.404629f, 0.254983f,  0.150477f,
  -0.265903f, 0.003341f,  0.099969f,  -0.211964f, -0.129372f, -0.166366f,
  0.327712f,  -0.276234f, 0.140675f,  -0.433677f, -0.163050f, -0.143578f,
  -0.397840f, -0.422130f, -0.293835f, -0.075362f, -0.468375f, 1.021238f,
  1.394155f,  -0.922486f, -1.350222f, 2.030201f,  0.057717f,  0.227650f,
  -0.193179f, 0.037224f,  0.065555f,  0.020558f,  -0.059205f, -0.023690f,
  -0.008718f, 0.095976f,  -0.549587f, -0.321164f, -0.243728f, 1.344381f,
  -1.254107f, 0.294244f,  -0.154737f, -0.152597f, 0.342419f,  0.301883f,
  0.069866f,  -0.327766f, 0.209323f,  -0.364913f, -0.005530f, -0.558972f,
  0.057684f,  -0.309357f, -0.283325f, -0.278445f, -0.420115f, -0.418457f,
  -0.391481f, -0.418460f, -0.003897f, -0.023744f, -0.312330f, -0.366213f,
  0.269628f,  -0.274877f, -0.189988f, -0.419555f, -0.034033f, 0.192874f,
  -0.135487f, -0.326108f, -0.039019f, 0.185029f,  -0.264883f, -0.563447f,
  -0.163532f, -0.447652f, -0.141851f, 0.001714f,  -0.193184f, 0.032609f,
  -0.112883f, 0.074599f,  0.490665f,  0.434764f,  0.021652f,  -0.219618f,
  0.743267f,  0.147195f,  -0.303479f, -0.097674f, 0.195813f,  0.704007f,
  -1.290851f, 0.119701f,  0.224065f,  0.260246f,  -0.580657f, -0.096201f,
  -0.333214f, -0.586689f, 0.567178f,  0.157340f,  -0.043184f, 0.194358f,
  -0.026506f, -0.339894f, -0.571803f, -0.234828f, 0.147054f,  -0.564178f,
  -0.156933f, -0.366055f, -0.691687f, -0.187501f, 0.215834f,  -0.346106f,
  -0.256892f, 0.110915f,  -0.337464f, -0.341474f, -0.216113f, 0.249445f,
  -0.070175f, -0.412141f, 0.153458f,  -0.081280f, 0.164669f,  -0.356396f,
  -0.294971f, -0.165121f, -0.133585f, -0.071467f, 0.295147f,  -0.253233f,
  -0.213833f, -0.343416f, -0.474344f, -0.304000f, -0.341379f, -0.331456f,
  -0.393952f, -0.508004f, -0.569518f, -0.509864f, 0.121961f,  0.011957f,
  0.000498f,  -0.201969f, -0.407195f, -0.414375f, -0.295846f, 0.247492f,
  0.124249f,  -0.550804f, -0.420397f, -0.123462f, 0.333292f,  -0.240230f,
  -0.025604f, 0.337536f,  -0.295006f, -0.272614f, -0.496850f, -0.278521f,
  0.234591f,  -0.052775f, -0.014052f, -0.260078f, -0.279128f, -0.036385f,
  0.008714f,  -0.064018f, -0.124873f, -0.334014f,
};

static const float av1_early_term_after_split_nn_bias_8_layer0[] = {
  1.202379f,  -0.117005f, -0.135527f, -0.262255f, -0.443658f, -0.078981f,
  0.615653f,  -0.124482f, -0.227768f, -0.227014f, -0.135898f, 0.143216f,
  -0.225995f, 0.370877f,  -0.214821f, -0.227752f,
};

static const float av1_early_term_after_split_nn_weights_8_layer1[] = {
  0.376594f,  0.266703f,  -0.039847f, 1.680142f,  -0.879939f, 0.286806f,
  -0.378223f, -0.405295f, -0.021107f, 0.039188f,  0.259308f,  0.193091f,
  0.077994f,  -0.269141f, 0.011180f,  -0.019262f,
};

static const float av1_early_term_after_split_nn_bias_8_layer1[] = {
  -1.29585564f,
};

static const NN_CONFIG av1_early_term_after_split_nnconfig_8 = {
  FEATURES,
  1,
  1,
  {
      16,
  },
  {
      av1_early_term_after_split_nn_weights_8_layer0,
      av1_early_term_after_split_nn_weights_8_layer1,
  },
  {
      av1_early_term_after_split_nn_bias_8_layer0,
      av1_early_term_after_split_nn_bias_8_layer1,
  },
};
#undef FEATURES
#undef HIDDEN_NODES

#ifdef __cplusplus
}  // extern "C"
#endif

#endif  // AOM_AV1_ENCODER_PARTITION_MODEL_WEIGHTS_H_
