/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include "color-test.h"
#include "core/gcs-color.h"
#include <stdexcept>
#include <cppunit/extensions/HelperMacros.h>

using namespace gcs::Test;

/*******************************
 * CONSTRUCTOR TESTS
 ******************************/
void Color::ConstructionSuite::testDefaultConstructor(void)
{
    ColorPtr c = gcs::Color::create();
    CPPUNIT_ASSERT_EQUAL(0, c->get_red());
    CPPUNIT_ASSERT_EQUAL(0, c->get_green());
    CPPUNIT_ASSERT_EQUAL(0, c->get_blue());
    CPPUNIT_ASSERT_EQUAL((Glib::ustring) "#000000", c->get_hexstring());
    CPPUNIT_ASSERT_EQUAL(0, c->get_hue());
    CPPUNIT_ASSERT_EQUAL(0, c->get_saturation());
    CPPUNIT_ASSERT_EQUAL(0, c->get_value());
}

void Color::ConstructionSuite::testHexConstructor(void)
{
    tHexString hex = "#347a9b";
    tHexString noHash = "347a9b";
    ColorPtr c = gcs::Color::create(hex);
    tColorHsv expectedHsv(199, 66, 61);
    tColorRgb expectedRgb(52, 122, 155);

    CPPUNIT_ASSERT_EQUAL(hex, c->get_hexstring());
    CPPUNIT_ASSERT_EQUAL(expectedRgb, c->get_rgb());
    CPPUNIT_ASSERT_EQUAL(expectedHsv, c->get_hsv());

    ColorPtr d = gcs::Color::create(noHash);
    CPPUNIT_ASSERT_EQUAL(hex, d->get_hexstring());
    CPPUNIT_ASSERT_EQUAL(expectedRgb, d->get_rgb());
    CPPUNIT_ASSERT_EQUAL(expectedHsv, d->get_hsv());

    tHexString shortHex = "#2b5";
    tHexString notShort = "#22bb55";
    ColorPtr e = gcs::Color::create(shortHex);
    CPPUNIT_ASSERT_EQUAL(notShort, e->get_hexstring());

    tHexString shortNoHash = "2b5";
    ColorPtr f = gcs::Color::create(shortNoHash);
    CPPUNIT_ASSERT_EQUAL(notShort, f->get_hexstring());
}

void Color::ConstructionSuite::testRgbConstructor(void)
{
    gint r = 56, g = 95, b = 145;
    ColorPtr c = gcs::Color::create(r, g, b);

    tColorRgb expectedRgb(r, g, b);
    tHexString expectedHex("#385f91");
    tColorHsv expectedHsv(214, 61, 57);

    CPPUNIT_ASSERT_EQUAL(expectedRgb, c->get_rgb());
    CPPUNIT_ASSERT_EQUAL(expectedHex, c->get_hexstring());
    CPPUNIT_ASSERT_EQUAL(expectedHsv, c->get_hsv());
}

void Color::ConstructionSuite::testGdkConstructor(void)
{
    tHexString hex("#76ab51");
    Gdk::Color gdkcolor(hex);
    tColorRgb expectedRgb(118, 171, 81);
    tColorHsv expectedHsv(95, 53, 67);
    ColorPtr c = gcs::Color::create(gdkcolor);

    CPPUNIT_ASSERT_EQUAL(expectedRgb, c->get_rgb());
    CPPUNIT_ASSERT_EQUAL(expectedHsv, c->get_hsv());
    CPPUNIT_ASSERT_EQUAL(hex, c->get_hexstring());
}

void Color::ConstructionSuite::testCopyConstructor(void)
{
    tHexString hex = "#347a9b";
    ColorPtr c = gcs::Color::create(hex);
    tColorHsv expectedHsv(199, 66, 61);
    tColorRgb expectedRgb(52, 122, 155);

    ColorPtr d(c);

    CPPUNIT_ASSERT_EQUAL(hex, d->get_hexstring());
    CPPUNIT_ASSERT_EQUAL(expectedRgb, d->get_rgb());
    CPPUNIT_ASSERT_EQUAL(expectedHsv, d->get_hsv());
}

void Color::ConstructionSuite::testDestructor(void)
{
    // nothing to do for now.
}


/*******************************
 * STATIC FUNCTION TESTS
 ******************************/

void Color::StaticFunctions::testRgbToHex(void)
{
    tColorRgb rgb(123, 54, 211);
    tHexString hex = gcs::Color::rgb_to_hex(rgb);
    tHexString expected("#7b36d3");
    CPPUNIT_ASSERT_EQUAL(expected, hex);
}

void Color::StaticFunctions::testRgbToHsv(void)
{
    tColorRgb rgb(123, 54, 211);
    tColorHsv expectedHsv(266, 74, 83);
    tColorHsv actualHsv = gcs::Color::rgb_to_hsv(rgb);

    CPPUNIT_ASSERT_EQUAL(expectedHsv, actualHsv);
}

void Color::StaticFunctions::testRgbAsP(void)
{
    gint r, g, b, x, y, z, zz;
    gdouble r_expected, g_expected, b_expected, x_expected, y_expected,
            z_expected, zz_expected;
    r = 219;
    g = 76;
    b = 198;
    x = 0;
    y = 255;
    z = 1234;
    zz = -23;

    r_expected = 0.85882;
    g_expected = 0.29803;
    b_expected = 0.77647;
    x_expected = 0.0;
    y_expected = 1.0;
    z_expected = 4.83921;
    zz_expected = -0.09019;

    CPPUNIT_ASSERT_DOUBLES_EQUAL(r_expected, gcs::Color::rgb_as_p(r), 0.001);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(g_expected, gcs::Color::rgb_as_p(g), 0.001);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(b_expected, gcs::Color::rgb_as_p(b), 0.001);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(x_expected, gcs::Color::rgb_as_p(x), 0.001);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(y_expected, gcs::Color::rgb_as_p(y), 0.001);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(z_expected, gcs::Color::rgb_as_p(z), 0.001);
    CPPUNIT_ASSERT_DOUBLES_EQUAL(zz_expected, gcs::Color::rgb_as_p(zz), 0.001);
}


/*******************************
 * GetsAndSets
 ******************************/

void Color::GetsAndSets::testSetRgb(void)
{
    gint r, g, b;
    r = 219;
    g = 76;
    b = 198;
    tHexString expected_hex = "#db4cc6";
    tColorHsv expected_hsv(309, 65, 86);

    ColorPtr c = gcs::Color::create();
    c->set(r, g, b);

    CPPUNIT_ASSERT_EQUAL(r, c->get_red());
    CPPUNIT_ASSERT_EQUAL(g, c->get_green());
    CPPUNIT_ASSERT_EQUAL(b, c->get_blue());
    CPPUNIT_ASSERT_EQUAL(expected_hex, c->get_hexstring());
    CPPUNIT_ASSERT_EQUAL(expected_hsv, c->get_hsv());
}


void Color::GetsAndSets::testSetHex(void)
{
    tHexString hex = "#db4cc6";
    ColorPtr c = gcs::Color::create();
    c->set(hex);
    CPPUNIT_ASSERT_EQUAL(hex, c->get_hexstring());

    tHexString withHash = "#4cc6db";
    tHexString noHash = "4cc6db";
    ColorPtr d = gcs::Color::create();
    d->set(noHash);
    CPPUNIT_ASSERT_EQUAL(withHash, d->get_hexstring());

    tHexString shortHex = "#4c9";
    tHexString notShort = "#44cc99";
    ColorPtr e = gcs::Color::create();
    e->set(shortHex);
    CPPUNIT_ASSERT_EQUAL(notShort, e->get_hexstring());

    tHexString shortNoHash = "#4c9";
    ColorPtr f = gcs::Color::create();
    f->set(shortNoHash);
    CPPUNIT_ASSERT_EQUAL(notShort, f->get_hexstring());
}


void Color::GetsAndSets::testSetRed(void)
{
    gint r1 = 0;
    gint r2 = 255;
    gint r3 = 127;
    gint r4 = 301;
    gint r5 = -23;

    ColorPtr c = gcs::Color::create();
    c->set_red(r1);
    CPPUNIT_ASSERT_EQUAL(r1, c->get_red());
    c->set_red(r2);
    CPPUNIT_ASSERT_EQUAL(r2, c->get_red());
    c->set_red(r3);
    CPPUNIT_ASSERT_EQUAL(r3, c->get_red());
    c->set_red(r4);
    CPPUNIT_ASSERT_EQUAL(r4 % maxRgbValue, c->get_red());
    c->set_red(r5);
    CPPUNIT_ASSERT_EQUAL(r5 + maxRgbValue, c->get_red());
}


void Color::GetsAndSets::testSetGreen(void)
{
    gint g1 = 0;
    gint g2 = 255;
    gint g3 = 127;
    gint g4 = 301;
    gint g5 = -23;

    ColorPtr c = gcs::Color::create();
    c->set_green(g1);
    CPPUNIT_ASSERT_EQUAL(g1, c->get_green());
    c->set_green(g2);
    CPPUNIT_ASSERT_EQUAL(g2, c->get_green());
    c->set_green(g3);
    CPPUNIT_ASSERT_EQUAL(g3, c->get_green());
    c->set_green(g4);
    CPPUNIT_ASSERT_EQUAL(g4 % maxRgbValue, c->get_green());
    c->set_green(g5);
    CPPUNIT_ASSERT_EQUAL(g5 + maxRgbValue, c->get_green());
}


void Color::GetsAndSets::testSetBlue(void)
{
    gint b1 = 0;
    gint b2 = 255;
    gint b3 = 127;
    gint b4 = 301;
    gint b5 = -23;

    ColorPtr c = gcs::Color::create();
    c->set_blue(b1);
    CPPUNIT_ASSERT_EQUAL(b1, c->get_blue());
    c->set_blue(b2);
    CPPUNIT_ASSERT_EQUAL(b2, c->get_blue());
    c->set_blue(b3);
    CPPUNIT_ASSERT_EQUAL(b3, c->get_blue());
    c->set_blue(b4);
    CPPUNIT_ASSERT_EQUAL(b4 % maxRgbValue, c->get_blue());
    c->set_blue(b5);
    CPPUNIT_ASSERT_EQUAL(b5 + maxRgbValue, c->get_blue());
}


void Color::GetsAndSets::testSetHue(void)
{
    gint h1 = 0;
    gint h2 = maxHueValue;
    gint h3 = 127;
    gint h4 = 401;
    gint h5 = -23;

    // hue only makes sense if there is an s and v first
    ColorPtr c = gcs::Color::create("#FF0000");
    c->set_hue(h1);
    CPPUNIT_ASSERT_EQUAL(h1 % maxHueValue, c->get_hue());
    c->set_hue(h2);
    CPPUNIT_ASSERT_EQUAL(h2 % maxHueValue, c->get_hue());
    c->set_hue(h3);
    CPPUNIT_ASSERT_EQUAL(h3 % maxHueValue, c->get_hue());
    c->set_hue(h4);
    CPPUNIT_ASSERT_EQUAL(h4 % maxHueValue, c->get_hue());
    c->set_hue(h5);
    CPPUNIT_ASSERT_EQUAL(h5 + maxHueValue, c->get_hue());
}


void Color::GetsAndSets::testSetSaturation(void)
{
    gint s1 = 0;
    gint s2 = maxSvValue;
    gint s3 = 66;
    gint s4 = 301;
    gint s5 = -23;

    ColorPtr c = gcs::Color::create("#00FF00");
    c->set_saturation(s1);
    CPPUNIT_ASSERT_EQUAL(s1 > maxSvValue ? maxSvValue : s1, c->get_saturation());
    c->set_saturation(s2);
    CPPUNIT_ASSERT_EQUAL(s2 > maxSvValue ? maxSvValue : s2, c->get_saturation());
    c->set_saturation(s3);
    CPPUNIT_ASSERT_EQUAL(s3 > maxSvValue ? maxSvValue : s3, c->get_saturation());
    c->set_saturation(s4);
    CPPUNIT_ASSERT_EQUAL(s4 > maxSvValue ? maxSvValue : s4, c->get_saturation());
    c->set_saturation(s5);
    CPPUNIT_ASSERT_EQUAL(s5 < 0 ? 0 : s5, c->get_saturation());
}


void Color::GetsAndSets::testSetValue(void)
{
    gint v1 = 0;
    gint v2 = maxSvValue;
    gint v3 = 66;
    gint v4 = 301;
    gint v5 = -23;

    ColorPtr c = gcs::Color::create("#0000FF");
    c->set_value(v1);
    CPPUNIT_ASSERT_EQUAL(v1 > maxSvValue ? maxSvValue : v1, c->get_value());
    c->set_value(v2);
    CPPUNIT_ASSERT_EQUAL(v2 > maxSvValue ? maxSvValue : v2, c->get_value());
    c->set_value(v3);
    CPPUNIT_ASSERT_EQUAL(v3 > maxSvValue ? maxSvValue : v3, c->get_value());
    c->set_value(v4);
    CPPUNIT_ASSERT_EQUAL(v4 > maxSvValue ? maxSvValue : v4, c->get_value());
    c->set_value(v5);
    CPPUNIT_ASSERT_EQUAL(v5 < 0 ? 0 : v5, c->get_value());
}

