/*
    -- MAGMA (version 2.7.2) --
       Univ. of Tennessee, Knoxville
       Univ. of California, Berkeley
       Univ. of Colorado, Denver
       @date August 2023

       @author Hartwig Anzt

       @generated from sparse_hip/blas/zparilu.cpp, normal z -> c, Fri Aug 25 13:17:52 2023
*/
#include "magmasparse_internal.h"

#include "magma_trisolve.h"

#define PRECISION_c

// This file is deprecated and will be removed in future.
// The ParILU/ParIC functionality is provided by 
// src/cparilu_gpu.cpp and src/cparic_gpu.cpp
// 

/**
    Purpose
    -------

    Prepares the ILU preconditioner via the iterative ILU iteration.

    Arguments
    ---------

    @param[in]
    A           magma_c_matrix
                input matrix A
                
    @param[in]
    b           magma_c_matrix
                input RHS b

    @param[in,out]
    precond     magma_c_preconditioner*
                preconditioner parameters
                
    @param[in]
    queue       magma_queue_t
                Queue to execute in.

    @ingroup magmasparse_cgepr
    ********************************************************************/
extern "C"
magma_int_t
magma_cparilusetup(
    magma_c_matrix A,
    magma_c_matrix b,
    magma_c_preconditioner *precond,
    magma_queue_t queue )
{
    magma_int_t info = 0;

    magma_c_matrix hAh={Magma_CSR}, hA={Magma_CSR}, hL={Magma_CSR}, hU={Magma_CSR},
    hAcopy={Magma_CSR}, hAL={Magma_CSR}, hAU={Magma_CSR}, hAUt={Magma_CSR},
    hUT={Magma_CSR}, hAtmp={Magma_CSR}, hACSRCOO={Magma_CSR}, dAinitguess={Magma_CSR},
    dL={Magma_CSR}, dU={Magma_CSR};

    // copy original matrix as CSRCOO to device
    CHECK( magma_cmtransfer(A, &hAh, A.memory_location, Magma_CPU, queue ));
    CHECK( magma_cmconvert( hAh, &hA, hAh.storage_type, Magma_CSR , queue ));
    magma_cmfree(&hAh, queue );

    CHECK( magma_cmtransfer( hA, &hAcopy, Magma_CPU, Magma_CPU , queue ));

    // in case using fill-in
    CHECK( magma_csymbilu( &hAcopy, precond->levels, &hAL, &hAUt,  queue ));
    // add a unit diagonal to L for the algorithm
    CHECK( magma_cmLdiagadd( &hAL , queue ));
    // transpose U for the algorithm
    CHECK( magma_c_cucsrtranspose(  hAUt, &hAU , queue ));
    magma_cmfree( &hAUt , queue );

    // ---------------- initial guess ------------------- //
    CHECK( magma_cmconvert( hAcopy, &hACSRCOO, Magma_CSR, Magma_CSRCOO , queue ));
    CHECK( magma_cmtransfer( hACSRCOO, &dAinitguess, Magma_CPU, Magma_DEV , queue ));
    magma_cmfree(&hACSRCOO, queue );
    magma_cmfree(&hAcopy, queue );

    // transfer the factor L and U
    CHECK( magma_cmtransfer( hAL, &dL, Magma_CPU, Magma_DEV , queue ));
    CHECK( magma_cmtransfer( hAU, &dU, Magma_CPU, Magma_DEV , queue ));
    magma_cmfree(&hAL, queue );
    magma_cmfree(&hAU, queue );

    for(int i=0; i<precond->sweeps; i++){
        CHECK( magma_cparilu_csr( dAinitguess, dL, dU , queue ));
    }

    CHECK( magma_cmtransfer( dL, &hL, Magma_DEV, Magma_CPU , queue ));
    CHECK( magma_cmtransfer( dU, &hU, Magma_DEV, Magma_CPU , queue ));
    CHECK( magma_c_cucsrtranspose(  hU, &hUT , queue ));

    magma_cmfree(&dL, queue );
    magma_cmfree(&dU, queue );
    magma_cmfree(&hU, queue );
    CHECK( magma_cmlumerge( hL, hUT, &hAtmp, queue ));

    magma_cmfree(&hL, queue );
    magma_cmfree(&hUT, queue );

    CHECK( magma_cmtransfer( hAtmp, &precond->M, Magma_CPU, Magma_DEV , queue ));

    hAL.diagorder_type = Magma_UNITY;
    CHECK( magma_cmconvert(hAtmp, &hAL, Magma_CSR, Magma_CSRL, queue ));
    hAL.storage_type = Magma_CSR;
    CHECK( magma_cmconvert(hAtmp, &hAU, Magma_CSR, Magma_CSRU, queue ));
    hAU.storage_type = Magma_CSR;

    magma_cmfree(&hAtmp, queue );

    // CHECK( magma_ccsrsplit( 0, 256, hAL, &DL, &RL , queue ));
    // CHECK( magma_ccsrsplit( 0, 256, hAU, &DU, &RU , queue ));
    // 
    // CHECK( magma_cmtransfer( DL, &precond->LD, Magma_CPU, Magma_DEV , queue ));
    // CHECK( magma_cmtransfer( DU, &precond->UD, Magma_CPU, Magma_DEV , queue ));

    // for hipsparse uncomment this
    CHECK( magma_cmtransfer( hAL, &precond->L, Magma_CPU, Magma_DEV , queue ));
    CHECK( magma_cmtransfer( hAU, &precond->U, Magma_CPU, Magma_DEV , queue ));
    
/*

    //-- for ba-solve uncomment this

    if( RL.nnz != 0 )
        CHECK( magma_cmtransfer( RL, &precond->L, Magma_CPU, Magma_DEV , queue ));
    else {
        precond->L.nnz = 0;
        precond->L.val = NULL;
        precond->L.col = NULL;
        precond->L.row = NULL;
        precond->L.blockinfo = NULL;
    }

    if( RU.nnz != 0 )
        CHECK( magma_cmtransfer( RU, &precond->U, Magma_CPU, Magma_DEV , queue ));
    else {
        precond->U.nnz = 0;
        precond->L.val = NULL;
        precond->L.col = NULL;
        precond->L.row = NULL;
        precond->L.blockinfo = NULL;
    }

    //-- for ba-solve uncomment this
*/

        // extract the diagonal of L into precond->d
    CHECK( magma_cjacobisetup_diagscal( precond->L, &precond->d, queue ));
    CHECK( magma_cvinit( &precond->work1, Magma_DEV, hA.num_rows, 1, MAGMA_C_ZERO, queue ));
    
    // extract the diagonal of U into precond->d2
    CHECK( magma_cjacobisetup_diagscal( precond->U, &precond->d2, queue ));
    CHECK( magma_cvinit( &precond->work2, Magma_DEV, hA.num_rows, 1, MAGMA_C_ZERO, queue ));

    magma_cmfree(&hAL, queue );
    magma_cmfree(&hAU, queue );
    // magma_cmfree(&DL, queue );
    // magma_cmfree(&RL, queue );
    // magma_cmfree(&DU, queue );
    // magma_cmfree(&RU, queue );

    CHECK(magma_ctrisolve_analysis(precond->L, &precond->cuinfoL, false, false, false, queue));
    CHECK(magma_ctrisolve_analysis(precond->U, &precond->cuinfoU, true, false, false, queue));
    
    
    
cleanup:
    magma_cmfree( &hAh, queue );
    magma_cmfree( &hA, queue );
    magma_cmfree( &hL, queue );
    magma_cmfree( &hU, queue );
    magma_cmfree( &hAcopy, queue );
    magma_cmfree( &hAL, queue );
    magma_cmfree( &hAU, queue );
    magma_cmfree( &hAUt, queue );
    magma_cmfree( &hUT, queue );
    magma_cmfree( &hAtmp, queue );
    magma_cmfree( &hACSRCOO, queue );
    magma_cmfree( &dAinitguess, queue );
    magma_cmfree( &dL, queue );
    magma_cmfree( &dU, queue );
    // magma_cmfree( &DL, queue );
    // magma_cmfree( &DU, queue );
    // magma_cmfree( &RL, queue );
    // magma_cmfree( &RU, queue );

    return info;
}



/**
    Purpose
    -------

    Updates an existing preconditioner via additional iterative ILU sweeps for
    previous factorization initial guess (PFIG).
    See  Anzt et al., Parallel Computing, 2015.

    Arguments
    ---------
    
    @param[in]
    A           magma_c_matrix
                input matrix A, current target system

    @param[in]
    precond     magma_c_preconditioner*
                preconditioner parameters

    @param[in]
    updates     magma_int_t 
                number of updates
    
    @param[in]
    queue       magma_queue_t
                Queue to execute in.
                
    @ingroup magmasparse_chepr
    ********************************************************************/
extern "C"
magma_int_t
magma_cpariluupdate(
    magma_c_matrix A,
    magma_c_preconditioner *precond,
    magma_int_t updates,
    magma_queue_t queue )
{
    magma_int_t info = 0;

    magma_c_matrix hALt={Magma_CSR};
    magma_c_matrix d_h={Magma_CSR};
    
    magma_c_matrix hL={Magma_CSR}, hU={Magma_CSR},
    hAcopy={Magma_CSR}, hAL={Magma_CSR}, hAU={Magma_CSR}, hAUt={Magma_CSR},
    hUT={Magma_CSR}, hAtmp={Magma_CSR},
    dL={Magma_CSR}, dU={Magma_CSR};

    if ( updates > 0 ){
        CHECK( magma_cmtransfer( precond->M, &hAcopy, Magma_DEV, Magma_CPU , queue ));
        // in case using fill-in
        CHECK( magma_csymbilu( &hAcopy, precond->levels, &hAL, &hAUt,  queue ));
        // add a unit diagonal to L for the algorithm
        CHECK( magma_cmLdiagadd( &hAL , queue ));
        // transpose U for the algorithm
        CHECK( magma_c_cucsrtranspose(  hAUt, &hAU , queue ));
        // transfer the factor L and U
        CHECK( magma_cmtransfer( hAL, &dL, Magma_CPU, Magma_DEV , queue ));
        CHECK( magma_cmtransfer( hAU, &dU, Magma_CPU, Magma_DEV , queue ));
        magma_cmfree(&hAL, queue );
        magma_cmfree(&hAU, queue );
        magma_cmfree(&hAUt, queue );
        magma_cmfree(&precond->M, queue );
        magma_cmfree(&hAcopy, queue );
        
        // copy original matrix as CSRCOO to device
        for(int i=0; i<updates; i++){
            CHECK( magma_cparilu_csr( A, dL, dU, queue ));
        }
        CHECK( magma_cmtransfer( dL, &hL, Magma_DEV, Magma_CPU , queue ));
        CHECK( magma_cmtransfer( dU, &hU, Magma_DEV, Magma_CPU , queue ));
        CHECK( magma_c_cucsrtranspose(  hU, &hUT , queue ));
        magma_cmfree(&dL, queue );
        magma_cmfree(&dU, queue );
        magma_cmfree(&hU, queue );
        CHECK( magma_cmlumerge( hL, hUT, &hAtmp, queue ));
        // for CUSPARSE
        CHECK( magma_cmtransfer( hAtmp, &precond->M, Magma_CPU, Magma_DEV , queue ));
        
        magma_cmfree(&hL, queue );
        magma_cmfree(&hUT, queue );
        hAL.diagorder_type = Magma_UNITY;
        CHECK( magma_cmconvert(hAtmp, &hAL, Magma_CSR, Magma_CSRL, queue ));
        hAL.storage_type = Magma_CSR;
        CHECK( magma_cmconvert(hAtmp, &hAU, Magma_CSR, Magma_CSRU, queue ));
        hAU.storage_type = Magma_CSR;
        
        magma_cmfree(&hAtmp, queue );
        CHECK( magma_cmtransfer( hAL, &precond->L, Magma_CPU, Magma_DEV , queue ));
        CHECK( magma_cmtransfer( hAU, &precond->U, Magma_CPU, Magma_DEV , queue ));
        magma_cmfree(&hAL, queue );
        magma_cmfree(&hAU, queue );
    
        magma_cmfree( &precond->d , queue );
        magma_cmfree( &precond->d2 , queue );
        
        CHECK( magma_cjacobisetup_diagscal( precond->L, &precond->d, queue ));
        CHECK( magma_cjacobisetup_diagscal( precond->U, &precond->d2, queue ));
    }

cleanup:
    magma_cmfree(&d_h, queue );
    magma_cmfree(&hALt, queue );
    
    return info;
}


/**
    Purpose
    -------

    Prepares the IC preconditioner via the iterative IC iteration.

    Arguments
    ---------

    @param[in]
    A           magma_c_matrix
                input matrix A
                
    @param[in]
    b           magma_c_matrix
                input RHS b

    @param[in,out]
    precond     magma_c_preconditioner*
                preconditioner parameters
                
    @param[in]
    queue       magma_queue_t
                Queue to execute in.

    @ingroup magmasparse_chepr
    ********************************************************************/
extern "C"
magma_int_t
magma_cparicsetup(
    magma_c_matrix A,
    magma_c_matrix b,
    magma_c_preconditioner *precond,
    magma_queue_t queue )
{
    magma_int_t info = 0;
    
    hipsparseHandle_t hipsparseHandle=NULL;
    hipsparseMatDescr_t descrL=NULL;
    hipsparseMatDescr_t descrU=NULL;

    magma_c_matrix hAh={Magma_CSR}, hA={Magma_CSR}, hAtmp={Magma_CSR},
    hAL={Magma_CSR}, hAUt={Magma_CSR}, hALt={Magma_CSR}, hM={Magma_CSR},
    hACSRCOO={Magma_CSR}, dAinitguess={Magma_CSR}, dL={Magma_CSR};
    magma_c_matrix d_h={Magma_CSR};


    // copy original matrix as CSRCOO to device
    CHECK( magma_cmtransfer(A, &hAh, A.memory_location, Magma_CPU, queue ));
    CHECK( magma_cmconvert( hAh, &hA, hAh.storage_type, Magma_CSR , queue ));
    magma_cmfree(&hAh, queue );

    // in case using fill-in
    CHECK( magma_csymbilu( &hA, precond->levels, &hAL, &hAUt , queue ));

    // need only lower triangular
    magma_cmfree(&hAUt, queue );
    magma_cmfree(&hAL, queue );
    CHECK( magma_cmconvert( hA, &hAtmp, Magma_CSR, Magma_CSRL , queue ));
    magma_cmfree(&hA, queue );

    // ---------------- initial guess ------------------- //
    CHECK( magma_cmconvert( hAtmp, &hACSRCOO, Magma_CSR, Magma_CSRCOO , queue ));
    //int blocksize = 1;
    //magma_cmreorder( hACSRCOO, n, blocksize, blocksize, blocksize, &hAinitguess , queue );
    CHECK( magma_cmtransfer( hACSRCOO, &dAinitguess, Magma_CPU, Magma_DEV , queue ));
    magma_cmfree(&hACSRCOO, queue );
    CHECK( magma_cmtransfer( hAtmp, &dL, Magma_CPU, Magma_DEV , queue ));
    magma_cmfree(&hAtmp, queue );

    for(int i=0; i<precond->sweeps; i++){
        CHECK( magma_cparic_csr( dAinitguess, dL , queue ));
    }
    CHECK( magma_cmtransfer( dL, &hAL, Magma_DEV, Magma_CPU , queue ));
    magma_cmfree(&dL, queue );
    magma_cmfree(&dAinitguess, queue );


    // for CUSPARSE
    CHECK( magma_cmtransfer( hAL, &precond->M, Magma_CPU, Magma_DEV , queue ));

    // Jacobi setup
    CHECK( magma_cjacobisetup_matrix( precond->M, &precond->L, &precond->d , queue ));

    // for Jacobi, we also need U
    CHECK( magma_c_cucsrtranspose(   hAL, &hALt , queue ));
    CHECK( magma_cjacobisetup_matrix( hALt, &hM, &d_h , queue ));

    CHECK( magma_cmtransfer( hM, &precond->U, Magma_CPU, Magma_DEV , queue ));

    magma_cmfree(&hM, queue );

    magma_cmfree(&d_h, queue );


        // copy the matrix to precond->L and (transposed) to precond->U
    CHECK( magma_cmtransfer(precond->M, &(precond->L), Magma_DEV, Magma_DEV, queue ));
    CHECK( magma_cmtranspose( precond->L, &(precond->U), queue ));

    // extract the diagonal of L into precond->d
    CHECK( magma_cjacobisetup_diagscal( precond->L, &precond->d, queue ));
    CHECK( magma_cvinit( &precond->work1, Magma_DEV, hAL.num_rows, 1, MAGMA_C_ZERO, queue ));

    // extract the diagonal of U into precond->d2
    CHECK( magma_cjacobisetup_diagscal( precond->U, &precond->d2, queue ));
    CHECK( magma_cvinit( &precond->work2, Magma_DEV, hAL.num_rows, 1, MAGMA_C_ZERO, queue ));


    magma_cmfree(&hAL, queue );
    magma_cmfree(&hALt, queue );

    CHECK(magma_ctrisolve_analysis(precond->M, &precond->cuinfoL, false, false, false, queue));
    CHECK(magma_ctrisolve_analysis(precond->M, &precond->cuinfoU, false, false, true, queue));
    
    cleanup:
    hipsparseDestroy( hipsparseHandle );
    hipsparseDestroyMatDescr( descrL );
    hipsparseDestroyMatDescr( descrU );
    hipsparseHandle=NULL;
    descrL=NULL;
    descrU=NULL;    
    magma_cmfree( &hAh, queue );
    magma_cmfree( &hA, queue );
    magma_cmfree( &hAtmp, queue );
    magma_cmfree( &hAL, queue );
    magma_cmfree( &hAUt, queue );
    magma_cmfree( &hALt, queue );
    magma_cmfree( &hM, queue );
    magma_cmfree( &hACSRCOO, queue );
    magma_cmfree( &dAinitguess, queue );
    magma_cmfree( &dL, queue );
    magma_cmfree( &d_h, queue );
    
    return info;
}


/**
    Purpose
    -------

    Updates an existing preconditioner via additional iterative IC sweeps for
    previous factorization initial guess (PFIG).
    See  Anzt et al., Parallel Computing, 2015.

    Arguments
    ---------
    
    @param[in]
    A           magma_c_matrix
                input matrix A, current target system

    @param[in]
    precond     magma_c_preconditioner*
                preconditioner parameters

    @param[in]
    updates     magma_int_t 
                number of updates
    
    @param[in]
    queue       magma_queue_t
                Queue to execute in.
                
    @ingroup magmasparse_chepr
    ********************************************************************/
extern "C"
magma_int_t
magma_cparicupdate(
    magma_c_matrix A,
    magma_c_preconditioner *precond,
    magma_int_t updates,
    magma_queue_t queue )
{
    magma_int_t info = 0;

    magma_c_matrix hALt={Magma_CSR};
    magma_c_matrix d_h={Magma_CSR};
        
    if( updates > 0 ){
        // copy original matrix as CSRCOO to device
        for(int i=0; i<updates; i++){
            CHECK( magma_cparic_csr( A, precond->M , queue ));
        }
        //magma_cmtransfer( precond->M, &hALt, Magma_DEV, Magma_CPU , queue );
        magma_cmfree(&precond->L, queue );
        magma_cmfree(&precond->U, queue );
        magma_cmfree( &precond->d , queue );
        magma_cmfree( &precond->d2 , queue );
        
        // copy the matrix to precond->L and (transposed) to precond->U
        CHECK( magma_cmtransfer(precond->M, &(precond->L), Magma_DEV, Magma_DEV, queue ));
        CHECK( magma_cmtranspose( precond->L, &(precond->U), queue ));

        CHECK( magma_cjacobisetup_diagscal( precond->L, &precond->d, queue ));
        CHECK( magma_cjacobisetup_diagscal( precond->U, &precond->d2, queue ));
    }
    
cleanup:
    magma_cmfree(&d_h, queue );
    magma_cmfree(&hALt, queue );
    
    return info;
}

