/*
 * board_mon.c
 *
 *  Created on: Oct 3, 2016
 *      Author: elleryc
 */

#include <stdlib.h>
#include <stdio.h>
#include "xil_printf.h"
#include "xil_io.h"
#include "xiic.h"
#include "xgpio.h"
#include "sleep.h"
#include "xuartlite_l.h"

//FIXED_BOARD should follow vbnv format: xilinx:vcu1525:4ddr-xpr:4.2
#ifdef FIXED_BOARD
#define USE_FIXED_BOARD 1
#else
#define USE_FIXED_BOARD 0
#define FIXED_BOARD ""
#endif

#ifndef ERT_VERBOSE
#define xil_printf(c, ...) fake_printf(c)
#endif

void fake_printf(char* somestring, ...) {
	return; //Dont actually print out
}

// Board info constants
#define MAX_BOARD_INFO_LENGTH 64 //Gotten from xclfeatures.h
#define VBNV_OFFSET 86

//Supply info struct
typedef struct {
	s32 sum_iout; //format is mamps
	s32 max_iout; //format is mamps
	s32 avg_iout; //format is mamps
	s32 cur_iout; //format is mamps
	u32 BRAM_ADDR;
	const char *supply_name;
	const u8 IIC_ADDR;
	const u8 chipid;
	const u8 isPresent;
} SupplyStats;

//Board info struct
typedef struct  {
	char vbnv_info[MAX_BOARD_INFO_LENGTH];
	u32 iic_mux_addr;
	SupplyStats *supplies;
	u32 num_supplies;
} BoardInfo;

BoardInfo _board_info;

// function headers
void iic_mux_select (u32 IIC, u8 iic_mux_chan);
int reset_stats(u32 IIC, SupplyStats *supply);
int pmbus_read(u32 IIC, u8 dev_addr, u8 command, u8 *rxBuf);
void pmbus_reset(u32 IIC);
void pmbus_print_voltage(u32 IIC, SupplyStats *supply);
void pmbus_print_current(u32 IIC, SupplyStats *supply);
int store_pmbus_current(u32 IIC, SupplyStats *supply);
void bram_clear(SupplyStats *supply);
void bram_display(SupplyStats *supply);
void bram_write_checksum(u32 bram);
void bram_clear_checksum(u32 bram);
int init_board_info(u32 feature_rom);


#define DRIVER_TIMEOUT_US 1000000 //1s timeout

//GPIO Constants
#define GPIO_CHAN_INDEX 1
#define MB_GPIO_DIR_MASK 0xFFFFFFF0 //bottom nibble are outputs
//GPIO States
#define GPIO_INACTIVE 0x0
#define GPIO_CLEAR_BRAM 0x1
#define GPIO_DISPLAY_BRAM 0x2
#define GPIO_DRIVER_REQ 0x3
#define GPIO_DRIVER_ACK 0x4
#define GPIO_DRIVER_DONE 0x5
#define GPIO_STOP_MB 0x6
#define GPIO_STOP_ACK 0x7

// I2C MUX Selection and slave addresses
#define DISABLE_MUX 				0x00
#define PMBUS_SEL 					0x01

// Status from store commands
#define STORE_SUCCESS 0
#define STORE_FAILED 1
#define STORE_OVERFLOW 2

// COMMAND_NAMES FOR MAX15301 / MAX15303 / MAX20571
#define READ_VOUT				0x8B
#define	READ_IOUT				0x8C

// I2C MUX Selection and slave addresses
#define DISABLE_MUX 				0x00
#define PMBUS_SEL 					0x01

// Status from store commands
#define STORE_SUCCESS 0
#define STORE_FAILED 1
#define STORE_OVERFLOW 2

// Regulators
#define MAX15301 1
#define MAX20751 2
#define LTC3884 3

//Supply BRAM offsets
u32 num_samps = 0; //Number of samples taken used for averaging
#define NUM_BYTES_PER_SUPPLY 12
#define MAX_IOUT_BRAM_OFFSET 0x00
#define AVG_IOUT_BRAM_OFFSET 0x04
#define CUR_IOUT_BRAM_OFFSET 0x08

//VU9P_HP Definitions
#define VU9P_HP_IIC_MUX_ADDR 	0x74 //This is the true 7bit address
#define VU9P_HP_NUM_SUPPLIES 	6
static SupplyStats VU9P_HP_SUPPLIES[VU9P_HP_NUM_SUPPLIES] = {
		{.supply_name="VCCINT",  	.IIC_ADDR=0x50, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V8",  	.IIC_ADDR=0x14, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V2",  	.IIC_ADDR=0x12, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCCBRAM", 	.IIC_ADDR=0x0D, .chipid=MAX15301, .isPresent=1},
		{.supply_name="MGTAVCC",  	.IIC_ADDR=0x72, .chipid=MAX20751, .isPresent=1},
		{.supply_name="MGTAVTT",  	.IIC_ADDR=0x73, .chipid=MAX20751, .isPresent=1}
};

//VU9P Definitions
#define VU9P_IIC_MUX_ADDR 	0x74 //This is the true 7bit address
#define VU9P_NUM_SUPPLIES 	6
static SupplyStats VU9P_SUPPLIES[VU9P_NUM_SUPPLIES] = {
		{.supply_name="VCCINT",  	.IIC_ADDR=0x0A, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V8",  	.IIC_ADDR=0x14, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V2",  	.IIC_ADDR=0x12, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCCBRAM", 	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="MGTAVCC",  	.IIC_ADDR=0x72, .chipid=MAX20751, .isPresent=1},
		{.supply_name="MGTAVTT",  	.IIC_ADDR=0x73, .chipid=MAX20751, .isPresent=1}
};

//KU115 Definitions
#define KU115_IIC_MUX_ADDR 	0x74 //This is the true 7bit address
#define KU115_NUM_SUPPLIES 	6
static SupplyStats KU115_SUPPLIES[KU115_NUM_SUPPLIES] = {
		{.supply_name="VCCINT",  	.IIC_ADDR=0x0A, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V8",  	.IIC_ADDR=0x14, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V2",  	.IIC_ADDR=0x12, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCCBRAM", 	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="MGTAVCC",  	.IIC_ADDR=0x72, .chipid=MAX20751, .isPresent=1},
		{.supply_name="MGTAVTT",  	.IIC_ADDR=0x73, .chipid=MAX20751, .isPresent=1}
};

//KCU1500 Definitions
#define KCU1500_IIC_MUX_ADDR 	0x74 //This is the true 7bit address
#define KCU1500_NUM_SUPPLIES 	6
static SupplyStats KCU1500_SUPPLIES[KCU1500_NUM_SUPPLIES] = {
		{.supply_name="VCCINT",  	.IIC_ADDR=0x0A, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V8",  	.IIC_ADDR=0x14, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCC1V2",  	.IIC_ADDR=0x12, .chipid=MAX15301, .isPresent=1},
		{.supply_name="VCCBRAM", 	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="MGTAVCC",  	.IIC_ADDR=0x72, .chipid=MAX20751, .isPresent=1},
		{.supply_name="MGTAVTT",  	.IIC_ADDR=0x73, .chipid=MAX20751, .isPresent=1}
};

//VCU1525 Definitions
#define VCU1525_IIC_MUX_ADDR 	0x74 //This is the true 7bit address
#define VCU1525_NUM_SUPPLIES 	6
static SupplyStats VCU1525_SUPPLIES[VCU1525_NUM_SUPPLIES] = {
		{.supply_name="VCCINT",  	.IIC_ADDR=0x44, .chipid=LTC3884,  .isPresent=1},
		{.supply_name="VCC1V8",  	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="VCC1V2",  	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="VCCBRAM", 	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="MGTAVCC",  	.IIC_ADDR=0x00, .chipid=MAX20751, .isPresent=0},
		{.supply_name="MGTAVTT",  	.IIC_ADDR=0x00, .chipid=MAX20751, .isPresent=0}
};

//VCU1526 Definitions
#define VCU1526_IIC_MUX_ADDR 	0x74 //This is the true 7bit address
#define VCU1526_NUM_SUPPLIES 	6
static SupplyStats VCU1526_SUPPLIES[VCU1526_NUM_SUPPLIES] = {
		{.supply_name="VCCINT",  	.IIC_ADDR=0x44, .chipid=LTC3884,  .isPresent=1},
		{.supply_name="VCC1V8",  	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="VCC1V2",  	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="VCCBRAM", 	.IIC_ADDR=0x00, .chipid=MAX15301, .isPresent=0},
		{.supply_name="MGTAVCC",  	.IIC_ADDR=0x00, .chipid=MAX20751, .isPresent=0},
		{.supply_name="MGTAVTT",  	.IIC_ADDR=0x00, .chipid=MAX20751, .isPresent=0}
};

int init_board_info(u32 feature_rom) {
	u8 byte_offset = VBNV_OFFSET%4;
	char temp[MAX_BOARD_INFO_LENGTH];
	char *vendor, *board, *name, *version;

	if(USE_FIXED_BOARD == 1) {
		xil_printf("USING FIXED BOARD INFO FOR: %s\n", FIXED_BOARD);
		memcpy(temp, (void *)FIXED_BOARD, MAX_BOARD_INFO_LENGTH);
		vendor = temp;
		board = strstr(vendor, ":"); board[0] = '\0'; board += 1;
		name = strstr(board, ":"); name[0] = '\0'; name += 1;
		version = strstr(name, ":"); version[0] = '\0'; version += 1;
	} else {
		memcpy(temp, (void *)(feature_rom+VBNV_OFFSET), MAX_BOARD_INFO_LENGTH);
		memcpy(_board_info.vbnv_info, &temp[byte_offset],MAX_BOARD_INFO_LENGTH-byte_offset); //Store for later use
		xil_printf("Board VBNV: %s\n", _board_info.vbnv_info);

		//Seperate fields which allows for different configurations based on each field
		vendor = &temp[byte_offset];
		board = strstr(vendor, ":"); board[0] = '\0'; board += 1;
		name = strstr(board, ":"); name[0] = '\0'; name += 1;
		version = strstr(name, ":"); version[0] = '\0'; version += 1;

		xil_printf("Board vendor: %s\n", 		vendor);
		xil_printf("Board board id: %s\n", 	board);
		xil_printf("Board name: %s\n", 		name);
		xil_printf("Board version: %s\n", 	version);
	}

	if(strcmp(vendor,"xilinx") == 0 && strcmp(board,"xil-accel-rd-vu9p-hp") == 0) {
		xil_printf("VU9P-HP\n");
		_board_info.supplies = VU9P_HP_SUPPLIES;
		_board_info.num_supplies = VU9P_HP_NUM_SUPPLIES;
		_board_info.iic_mux_addr = VU9P_HP_IIC_MUX_ADDR;
	}
	else if(strcmp(vendor,"xilinx") == 0 && strcmp(board,"xil-accel-rd-vu9p") == 0) {
		xil_printf("VU9P\n");
		_board_info.supplies = VU9P_SUPPLIES;
		_board_info.num_supplies = VU9P_NUM_SUPPLIES;
		_board_info.iic_mux_addr = VU9P_IIC_MUX_ADDR;
	} else if(strcmp(vendor,"xilinx") == 0 && strcmp(board,"xil-accel-rd-ku115") == 0) {
		xil_printf("KU115\n");
		_board_info.supplies = KU115_SUPPLIES;
		_board_info.num_supplies = KU115_NUM_SUPPLIES;
		_board_info.iic_mux_addr = KU115_IIC_MUX_ADDR;
	} else if(strcmp(vendor,"xilinx") == 0 && strcmp(board,"kcu1500") == 0) {
		xil_printf("KCU1500\n");
		_board_info.supplies = KCU1500_SUPPLIES;
		_board_info.num_supplies = KCU1500_NUM_SUPPLIES;
		_board_info.iic_mux_addr = KCU1500_IIC_MUX_ADDR;
	} else if(strcmp(vendor,"xilinx") == 0 && strcmp(board,"vcu1525") == 0) {
		xil_printf("VCU1525\n");
		_board_info.supplies = VCU1525_SUPPLIES;
		_board_info.num_supplies = VCU1525_NUM_SUPPLIES;
		_board_info.iic_mux_addr = VCU1525_IIC_MUX_ADDR;
	} else if(strcmp(vendor,"xilinx") == 0 && strcmp(board,"vcu1526") == 0) {
		xil_printf("VCU1526\n");
		_board_info.supplies = VCU1526_SUPPLIES;
		_board_info.num_supplies = VCU1526_NUM_SUPPLIES;
		_board_info.iic_mux_addr = VCU1526_IIC_MUX_ADDR;
	} else {
		xil_printf("ERROR: Unrecognized vbnv! %s:%s:%s:%s\n", vendor, board, name, version);
		return -1;
	}
	return 0;
}

void bram_clear(SupplyStats *supply){
	for(size_t i=0; i<NUM_BYTES_PER_SUPPLY; i+=4)
		*((u32 *)(supply->BRAM_ADDR+i)) = 0;
}

void bram_display(SupplyStats *supply) {
#ifdef ERT_VERBOSE
	xil_printf("BRAM Contents for supply %s\n", supply->supply_name);
	s32 bram_read_max_iout = (s32) (*((u32 *)(supply->BRAM_ADDR+MAX_IOUT_BRAM_OFFSET)));
	s32 bram_read_avg_iout = (s32) (*((u32 *)(supply->BRAM_ADDR+AVG_IOUT_BRAM_OFFSET)));
	s32 bram_read_cur_iout = (s32) (*((u32 *)(supply->BRAM_ADDR+CUR_IOUT_BRAM_OFFSET)));
	xil_printf("%s (0x%08X):\t\t Max amps %dmA\n", supply->supply_name, supply->BRAM_ADDR+MAX_IOUT_BRAM_OFFSET, bram_read_max_iout);
	xil_printf("%s (0x%08X):\t\t Avg amps %dmA\n", supply->supply_name, supply->BRAM_ADDR+AVG_IOUT_BRAM_OFFSET, bram_read_avg_iout);
	xil_printf("%s (0x%08X):\t\t Cur amps %dmA\n", supply->supply_name, supply->BRAM_ADDR+CUR_IOUT_BRAM_OFFSET, bram_read_cur_iout);
#endif
}

void bram_write_checksum(u32 bram) {
	u32 bram_checksum_loc = bram + NUM_BYTES_PER_SUPPLY*_board_info.num_supplies;
	u32 bram_checksum = 0;
	for(size_t i=0; i<NUM_BYTES_PER_SUPPLY*_board_info.num_supplies; i+=4)
		bram_checksum += *((u32 *)(bram+i));

	*((u32 *)bram_checksum_loc) = bram_checksum;
}

void bram_clear_checksum(u32 bram) {
	*((u32 *)(bram + NUM_BYTES_PER_SUPPLY*_board_info.num_supplies)) = 0xFFFFFFFF;
}

void iic_mux_select (u32 IIC, u8 iic_mux_chan){
	XIic_Send(IIC, _board_info.iic_mux_addr, &iic_mux_chan, sizeof(iic_mux_chan), XIIC_STOP);
	xil_printf("IIC Mux Channel PMBUS Selected\n");
}

u32 convert_bits_to_mv(u16 volts_bits, u8 chip_id) {
	if(chip_id == MAX15301) //V*2^-12
		return (volts_bits * 1000) >> 12;
	else if(chip_id == MAX20751)
		return (volts_bits - 1) * 5 + 250;
	else if(chip_id == LTC3884)
		return (volts_bits * 1000) >> 12;

	return 0;
}

s32 convert_bits_to_mamps(s16 amps_bits) {
	s8 exp = amps_bits >> 11;
	s8 exp_pos = ~(exp) + 1;
	s32 mantissa = amps_bits << 21; mantissa >>= 21; //remove extra bits
	mantissa *= 1000; //convert to milliamps

	return (exp < 0) ? mantissa >> exp_pos : mantissa << exp;
}

u8 get_pmbus_rx_bytes(u8 command) {
	switch(command) {
	case READ_VOUT:
	case READ_IOUT:
		return 2;
	}
	return 0;
}

void pmbus_reset(u32 IIC) {
	xil_printf("Resetting IIC bus.\n");
	XIic_DynInit(IIC);
	usleep(10000); //sleep for 10ms
}

int pmbus_read(u32 IIC, u8 dev_addr, u8 command, u8 *rxBuf) {
	if(sizeof(command) != XIic_Send(IIC, dev_addr, &command, sizeof(command), XIIC_REPEATED_START)) {
		xil_printf("IIC write failed!\n");
		pmbus_reset(IIC);
		return XST_FAILURE;
	}
	if(get_pmbus_rx_bytes(command) != XIic_Recv(IIC, dev_addr, rxBuf, get_pmbus_rx_bytes(command), XIIC_STOP)) {
		xil_printf("IIC read failed!\n");
		pmbus_reset(IIC);
		return XST_FAILURE;
	}

	return XST_SUCCESS;
}

void pmbus_print_voltage(u32 IIC, SupplyStats *supply) {
	u8 rxBuf[32];
	u16 vout_bits;
	u32 vout_mv;
	u32 vout_v;

	if(pmbus_read(IIC, supply->IIC_ADDR, READ_VOUT, rxBuf) != XST_SUCCESS) {
		xil_printf("Failed to read %s!\n", supply->supply_name);
	}else {
		vout_bits = (rxBuf[1] << 8 | rxBuf[0]);
		vout_mv = convert_bits_to_mv(vout_bits, supply->chipid);
		vout_v = vout_mv/1000;
		vout_mv -= vout_v*1000;
		//xil_printf("%s (0x%02X):\t\t 0x%04X\n", supply->supply_name, supply->IIC_ADDR, vout_bits);
		xil_printf("%s (0x%02X):\t\t %d.%dV\n", supply->supply_name, supply->IIC_ADDR, vout_v, vout_mv);
	}
}

void pmbus_print_current(u32 IIC, SupplyStats *supply) {
	u8 rxBuf[32];
	s16 iout_bits;
	s32 iout_ma;
	s32 iout_a;

	if(pmbus_read(IIC, supply->IIC_ADDR, READ_IOUT, rxBuf) != XST_SUCCESS) {
		xil_printf("Failed to read %s!\n", supply->supply_name);
	}else {
		iout_bits = (rxBuf[1] << 8 | rxBuf[0]);
		iout_ma = convert_bits_to_mamps(iout_bits);
		iout_a = iout_ma/1000;
		iout_ma -= iout_a*1000;
		//xil_printf("%s (0x%02X):\t\t 0x%04X\n", supply->supply_name, supply->IIC_ADDR, (u16) iout_bits);
		xil_printf("%s (0x%02X):\t\t %d.%dA\n", supply->supply_name, supply->IIC_ADDR, iout_a, iout_ma);
	}
}

int store_pmbus_current(u32 IIC, SupplyStats *supply) {
	u8 rxBuf[32];
	s16 iout_bits;
	s32 iout_ma;

	if(pmbus_read(IIC, supply->IIC_ADDR, READ_IOUT, rxBuf) != XST_SUCCESS) {
		xil_printf("Failed to read %s!\n", *supply->supply_name);
		return STORE_FAILED;
	} else {
		iout_bits = (rxBuf[1] << 8 | rxBuf[0]);
		iout_ma = convert_bits_to_mamps(iout_bits);

		supply->cur_iout = iout_ma;


		if(iout_ma > supply->max_iout && iout_ma > 0) //Positive current
			supply->max_iout = iout_ma;
		else if(iout_ma < supply->max_iout && iout_ma < 0) //Negative current
			supply->max_iout = iout_ma;

		supply->sum_iout += iout_ma;

		if(supply->sum_iout < iout_ma) {//overflow, rst sum/avg
			return STORE_OVERFLOW; //return early and do not write to BRAM
		} else { //Store avg/max values into BRAM
			supply->avg_iout = supply->sum_iout / num_samps;
			*((u32 *)(supply->BRAM_ADDR+MAX_IOUT_BRAM_OFFSET)) = supply->max_iout;
			*((u32 *)(supply->BRAM_ADDR+AVG_IOUT_BRAM_OFFSET)) = supply->avg_iout;
			*((u32 *)(supply->BRAM_ADDR+CUR_IOUT_BRAM_OFFSET)) = supply->cur_iout;

			return STORE_SUCCESS;
		}
	}
}

int reset_stats(u32 IIC, SupplyStats *supply) {
	supply->sum_iout = 0;

	return store_pmbus_current(IIC, supply);
}

int main ()
{
	u32 paused_time = 0;

	//Device addresses found in xparameters.h
	u32 IIC = XPAR_STATIC_REGION_BRD_MGMT_SCHEDULER_BOARD_MANAGEMENT_BOARD_I2C_CTRL_BASEADDR;
	u32 feature_rom = XPAR_STATIC_REGION_FEATURE_ROM_CTRL_S_AXI_BASEADDR;
	u32 gpio = XPAR_STATIC_REGION_BRD_MGMT_SCHEDULER_BOARD_MANAGEMENT_POWER_MONITOR_GPIO_BASEADDR;
	u32 bram = XPAR_STATIC_REGION_BRD_MGMT_SCHEDULER_BOARD_MANAGEMENT_POWER_MONITOR_CTRL_S_AXI_BASEADDR;

	//Get board info
	if(init_board_info(feature_rom) != 0) {
		xil_printf("Failed to initialize board! Exiting!\n");
		return -1;
	}

	//Initialize I2C
	XIic_DynInit(IIC);
	iic_mux_select (IIC, PMBUS_SEL);

	//Initialize GPIO
	XGpio_WriteReg(gpio, XGPIO_TRI_OFFSET, MB_GPIO_DIR_MASK);

	//Initialize structs
	u32 bram_addr = bram;
	for(u32 i=0; i<_board_info.num_supplies;i++,bram_addr+=NUM_BYTES_PER_SUPPLY) {
		_board_info.supplies[i].sum_iout = 0; _board_info.supplies[i].avg_iout = 0; _board_info.supplies[i].max_iout = 0;
		_board_info.supplies[i].BRAM_ADDR = bram_addr;
	}

	//Monitor Voltage/Current
	int gpio_value;
	int store_flag;
	u32 cntr = 0;
	while(1){
		cntr+=1;
		gpio_value = XGpio_ReadReg(gpio, XGPIO_DATA_OFFSET);

		//xil_printf("\n-----------------------------------------------------\n");

		switch (gpio_value) {
		case GPIO_CLEAR_BRAM: //Clear BRAM
			xil_printf("........Clearing BRAM ......\n");
			num_samps=0;
			for(int i=0;i<_board_info.num_supplies;i++) {
				if(_board_info.supplies[i].isPresent)
					bram_clear(&_board_info.supplies[i]);
			}
			bram_clear_checksum(bram);
			xil_printf("........Cleared GPIO.........\n");
			XGpio_WriteReg(gpio, XGPIO_DATA_OFFSET, GPIO_INACTIVE);
			break;

		case GPIO_DISPLAY_BRAM: //Print BRAM
			xil_printf(".........BRAM DATA .....\n");
			xil_printf("Num supplies: %d\n", _board_info.num_supplies);
			for(int i=0;i<_board_info.num_supplies;i++) {
				if(_board_info.supplies[i].isPresent)
					bram_display(&_board_info.supplies[i]);
			}
			xil_printf("........Cleared GPIO.........\n");
			XGpio_WriteReg(gpio, XGPIO_DATA_OFFSET, GPIO_INACTIVE);
			break;

		case GPIO_DRIVER_REQ: //Halt BRAM write and let driver read out contents. Timeout after 1s
			xil_printf(".........Waiting for driver.....\n");
			paused_time = 0;
			XGpio_WriteReg(gpio, XGPIO_DATA_OFFSET, GPIO_DRIVER_ACK);
			while(paused_time < DRIVER_TIMEOUT_US && gpio_value != GPIO_DRIVER_DONE) {
				usleep(10000);
				paused_time+=10000;
				gpio_value = XGpio_ReadReg(gpio, XGPIO_DATA_OFFSET);
			}
			xil_printf("........Cleared GPIO.........\n");
			XGpio_WriteReg(gpio, XGPIO_DATA_OFFSET, GPIO_INACTIVE);
			break;

		case GPIO_INACTIVE: //Write BRAM
			num_samps+=1;
			store_flag = STORE_SUCCESS;
			for(int i=0;i<_board_info.num_supplies;i++) {
				if(_board_info.supplies[i].isPresent) {
					//pmbus_print_voltage(IIC, &_board_info.supplies[i]);
					//pmbus_print_current(IIC, &_board_info.supplies[i]);
					store_flag = store_pmbus_current(IIC, &_board_info.supplies[i]);
					if(store_flag != STORE_SUCCESS)
						break;
				}
			}

			//Error handling
			if(store_flag != STORE_SUCCESS) {
				if(store_flag == STORE_OVERFLOW) {
					usleep(1000); //Pause before resetting stats
					num_samps=1;
					for(int i=0;i<_board_info.num_supplies;i++) {
						if(_board_info.supplies[i].isPresent) {
							store_flag = reset_stats(IIC, &_board_info.supplies[i]);
							if(store_flag != STORE_SUCCESS)
								break;
						}
					}
					if(store_flag != STORE_SUCCESS) { //Clear checksum if pmbus fails
						bram_clear_checksum(bram);
					} else
						bram_write_checksum(bram);
				}
				else { //Clear checksum if pmbus fails and reset iic bus
					bram_clear_checksum(bram);
				}
			} else
				bram_write_checksum(bram);
			break;
			
		case GPIO_STOP_MB: //Safely exit polling code so microblaze can be put into reset
			break;

		default: //Clear bad GPIO state
			xil_printf("Reached bad state!\n");
			XGpio_WriteReg(gpio, XGPIO_DATA_OFFSET, GPIO_INACTIVE);
			break;

		}
		//xil_printf("-----------------------------------------------------\n");
		
		if(gpio_value == GPIO_STOP_MB) { //Acknowledge stop
		   XGpio_WriteReg(gpio, XGPIO_DATA_OFFSET, GPIO_STOP_ACK); 
		   return 0;
		}

		//if(cntr % 1000 == 0)
		//	xil_printf("%d samples read\n", cntr);
	}

	return 0;
}
