-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Characters.Handling;
with Casing;
with CommandLineData;
with ContextManager;
with Dictionary;
with ErrorHandler;
with Error_Types;
with ExaminerConstants;
with E_Strings;
with FileSystem;
with LexTokenLists;
with LexTokenManager;
with Maths;
with ScreenEcho;
with SparkLex;
with SP_Expected_Symbols;
with SP_Symbols;
with XMLReport;

use type CommandLineData.Language_Profiles;
use type ErrorHandler.Error_Level;
use type LexTokenManager.Str_Comp_Result;
use type Maths.ErrorCode;
use type Maths.Value;
use type SP_Symbols.SP_Symbol;

package body ConfigFile
--# own State is The_Error_Context;
is

   The_Error_Context : ErrorHandler.Error_Contexts := ErrorHandler.Null_Error_Context;

   procedure Read_Config_File (Opened_OK : out Boolean;
                               No_Errors : out Boolean)
   --# global in     CommandLineData.Content;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SparkLex.Curr_Line;
   --#        in out SPARK_IO.File_Sys;
   --#           out The_Error_Context;
   --# derives Dictionary.Dict,
   --#         ErrorHandler.Error_Context,
   --#         LexTokenManager.State,
   --#         No_Errors,
   --#         SPARK_IO.File_Sys,
   --#         The_Error_Context          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys &
   --#         Opened_OK                  from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         SPARK_IO.File_Sys &
   --#         SparkLex.Curr_Line         from *,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys;
   is
      File_Opened_OK      : Boolean;
      File_Read_OK        : Boolean;
      Local_Config_File   : SPARK_IO.File_Type;
      Saved_Error_Context : ErrorHandler.Error_Contexts;

      procedure Open_File (The_File : out SPARK_IO.File_Type;
                           File_Ok  : out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --# derives File_Ok,
      --#         SPARK_IO.File_Sys,
      --#         The_File          from CommandLineData.Content,
      --#                                SPARK_IO.File_Sys;
      is
         File_Name        : E_Strings.T;
         File_Spec_Status : FileSystem.Typ_File_Spec_Status;
         File_Status      : SPARK_IO.File_Status;
         Local_File       : SPARK_IO.File_Type := SPARK_IO.Null_File;
      begin
         --# accept F, 10, File_Spec_Status, "File_Spec_Status unused here";
         FileSystem.Find_Full_File_Name
           (File_Spec      => CommandLineData.Content.Target_Config_File,
            File_Status    => File_Spec_Status,
            Full_File_Name => File_Name);
         --# end accept;
         E_Strings.Open
           (File         => Local_File,
            Mode_Of_File => SPARK_IO.In_File,
            Name_Of_File => File_Name,
            Form_Of_File => "",
            Status       => File_Status);
         if File_Status = SPARK_IO.Ok then
            File_Ok := True;
         else
            File_Ok := False;
            ScreenEcho.Put_String ("Cannot open file ");
            if CommandLineData.Content.Plain_Output then
               ScreenEcho.Put_ExaminerLine (FileSystem.Just_File (Fn  => File_Name,
                                                                  Ext => True));
            else
               ScreenEcho.Put_ExaminerLine (File_Name);
            end if;
         end if;
         The_File := Local_File;
         --# accept F, 33, File_Spec_Status, "File_Spec_Status unused here";
      end Open_File;

      procedure Close_File (The_File : in out SPARK_IO.File_Type)
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         The_File          from *,
      --#                                The_File;
      is
         File_Status : SPARK_IO.File_Status;
      begin
         --# accept F, 10, File_Status, "File_Status unused here";
         SPARK_IO.Close (The_File, File_Status);
         --# accept F, 33, File_Status, "File_Status unused here";
      end Close_File;

      procedure Process_Config_File (The_Config_File : in     SPARK_IO.File_Type;
                                     Overall_Status  :    out Boolean)
      --# global in     CommandLineData.Content;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#           out SparkLex.Curr_Line;
      --# derives Dictionary.Dict,
      --#         ErrorHandler.Error_Context,
      --#         LexTokenManager.State,
      --#         Overall_Status,
      --#         SparkLex.Curr_Line,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         SPARK_IO.File_Sys,
      --#                                         The_Config_File;
      is

         type Declaration_Type is (
                                   Dec_Named_Integer,
                                   Dec_Named_Real,
                                   Dec_Integer_Subtype,
                                   Dec_Integer_Type,
                                   Dec_Floating_Point_Type,
                                   Dec_Private,
                                   Dec_Typed_Constant);

         -- Depth of child package hierarchy that we support
         type Child_Package_Depth is range 1 .. 2;
         --# assert Child_Package_Depth'Base is Short_Short_Integer; -- for GNAT

         type Package_Hierarchy is array (Child_Package_Depth) of LexTokenManager.Lex_Value;

         type Declaration is record
            My_Type     : Declaration_Type;
            Enc_Package : Package_Hierarchy;
            Name        : LexTokenManager.Lex_Value;
            Base_Type   : LexTokenManager.Lex_Value;
            Low_Bound   : Maths.Value;
            High_Bound  : Maths.Value;
            Num_Digits  : Maths.Value;
            -- The value of a declartion can be numeric or an identifier.
            -- One of the following two fields is well-defined, depending on the
            -- value of My_Type
            Value     : Maths.Value;
            Value_Str : LexTokenManager.Lex_Value;
         end record;

         type Possible_Identifier_Range is range 1 .. 19;
         --# assert Possible_Identifier_Range'Base is Short_Short_Integer; -- for GNAT

         type Identifier_Strings is array (Possible_Identifier_Range) of LexTokenLists.Lists;

         -- acceptable packages
         type Library_Package_Range is range 1 .. 3;
         --# assert Library_Package_Range'Base is Short_Short_Integer; -- for GNAT

         type Child_Package_Range is range 1 .. 2;
         --# assert Child_Package_Range'Base is Short_Short_Integer; -- for GNAT

         type Library_Package_Strings is array (Library_Package_Range) of LexTokenManager.Lex_String;
         type Child_Package_Strings is array (Child_Package_Range) of LexTokenManager.Lex_String;

         type Syntax_Error_Types is (
                                     Exp_Symbol,
                                     Exp_Defn,
                                     Exp_Digits_Range,
                                     Exp_Minus_Real,
                                     Exp_Minus_Integer,
                                     Exp_Integer_Real);

         type Syntax_Errors is record
            The_Type        : Syntax_Error_Types;
            The_Symbol      : SP_Symbols.SP_Terminal;
            Expected_Symbol : SP_Symbols.SP_Terminal;
            The_Lex_Val     : LexTokenManager.Lex_Value;
         end record;

         Package_System : constant Library_Package_Range := 2;

         -- =====================================================================
         -- = LOCAL STATE
         -- =====================================================================

         Possible_Identifiers      : Identifier_Strings := Identifier_Strings'(others => LexTokenLists.Null_List);
         Possible_Library_Packages : Library_Package_Strings;
         Possible_Child_Packages   : Child_Package_Strings;

         -------------------
         -- Pseudo-constants
         -------------------

         Predefined_Scope : Dictionary.Scopes;

         ------------------------
         -- Important state items
         ------------------------

         Current_Scope       : Dictionary.Scopes;
         Valid_Package       : Boolean;
         Current_Declaration : Declaration;
         The_Token           : SP_Symbols.SP_Terminal;
         The_Lex_Val         : LexTokenManager.Lex_Value;
         Errors_Occurred     : Boolean;

         -- Flag to reparse current lexeme (after parser state change)
         Lookahead : Boolean;

         --------------------------------------------
         -- Integer expression parsing and evaluation
         --------------------------------------------

         Unary_Minus  : Boolean;
         Int_Subtract : Boolean;
         Int_Add      : Boolean;
         Int_Exponent : Boolean;

         Int_Val          : Maths.Value;
         Int_Exponent_Val : Maths.Value;
         Int_Add_Sub_Val  : Maths.Value;

         Expr_Int_Value : Maths.Value;

         ---------------------------------------
         -- Parsing and basic semantics checking
         ---------------------------------------

         type Parser_States is (
                                Config_Defn_Start,
                                Next_Config_Defn_Start,
                                Defn,
                                Type_Unknown,
                                Type_Private,
                                Type_FP,
                                Type_Int,
                                Subtype_Int,
                                Expr_Int,
                                Const_Unknown,
                                Config_Defn_End,
                                Found_Syntax_Error);
         subtype Parser_Stages is Positive range 1 .. 10;

         Parser_State        : Parser_States;
         Parser_Stage        : Parser_Stages;
         Saved_Parser_State  : Parser_States;
         Saved_Parser_Stage  : Parser_Stages;
         Next_Expected_State : Parser_States;

         Parsing_Ends : Boolean;

         Package_Full_Name, Package_End_Name : E_Strings.T := E_Strings.Empty_String;

         Package_Name_Depth : Child_Package_Depth := 1;

         Any_Priority_Sym       : Dictionary.Symbol := Dictionary.NullSymbol;
         Priority_Sym           : Dictionary.Symbol := Dictionary.NullSymbol;
         Interrupt_Priority_Sym : Dictionary.Symbol := Dictionary.NullSymbol;
         Lib_Package_Symbol     : Dictionary.Symbol := Dictionary.NullSymbol;

         ----------------------------------------------------------
         -- Semantic check on validity of 'new' package declaration
         ----------------------------------------------------------

         procedure State_Machine_Initialise
         --# global in     Dictionary.Dict;
         --#        in out Possible_Identifiers;
         --#           out Current_Declaration;
         --#           out Current_Scope;
         --#           out Errors_Occurred;
         --#           out Expr_Int_Value;
         --#           out Int_Add;
         --#           out Int_Add_Sub_Val;
         --#           out Int_Exponent;
         --#           out Int_Exponent_Val;
         --#           out Int_Subtract;
         --#           out Int_Val;
         --#           out Lookahead;
         --#           out Next_Expected_State;
         --#           out Parser_Stage;
         --#           out Parser_State;
         --#           out Parsing_Ends;
         --#           out Possible_Child_Packages;
         --#           out Possible_Library_Packages;
         --#           out Predefined_Scope;
         --#           out Saved_Parser_Stage;
         --#           out Saved_Parser_State;
         --#           out The_Lex_Val;
         --#           out The_Token;
         --#           out Unary_Minus;
         --#           out Valid_Package;
         --# derives Current_Declaration,
         --#         Errors_Occurred,
         --#         Expr_Int_Value,
         --#         Int_Add,
         --#         Int_Add_Sub_Val,
         --#         Int_Exponent,
         --#         Int_Exponent_Val,
         --#         Int_Subtract,
         --#         Int_Val,
         --#         Lookahead,
         --#         Next_Expected_State,
         --#         Parser_Stage,
         --#         Parser_State,
         --#         Parsing_Ends,
         --#         Possible_Child_Packages,
         --#         Possible_Library_Packages,
         --#         Saved_Parser_Stage,
         --#         Saved_Parser_State,
         --#         The_Lex_Val,
         --#         The_Token,
         --#         Unary_Minus,
         --#         Valid_Package             from  &
         --#         Current_Scope,
         --#         Predefined_Scope          from Dictionary.Dict &
         --#         Possible_Identifiers      from *;
         is
         begin
            -- integers
            LexTokenLists.Append (List => Possible_Identifiers (1),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (1),
                                  Item => LexTokenManager.Min_Int_Token);

            LexTokenLists.Append (List => Possible_Identifiers (2),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (2),
                                  Item => LexTokenManager.Max_Int_Token);

            LexTokenLists.Append (List => Possible_Identifiers (3),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (3),
                                  Item => LexTokenManager.Max_Binary_Modulus_Token);

            LexTokenLists.Append (List => Possible_Identifiers (4),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (4),
                                  Item => LexTokenManager.Max_Digits_Token);

            LexTokenLists.Append (List => Possible_Identifiers (5),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (5),
                                  Item => LexTokenManager.Max_Base_Digits_Token);

            LexTokenLists.Append (List => Possible_Identifiers (6),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (6),
                                  Item => LexTokenManager.Max_Mantissa_Token);

            LexTokenLists.Append (List => Possible_Identifiers (7),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (7),
                                  Item => LexTokenManager.Priority_Last_Token);

            LexTokenLists.Append (List => Possible_Identifiers (8),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (8),
                                  Item => LexTokenManager.Storage_Unit_Token);

            LexTokenLists.Append (List => Possible_Identifiers (9),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (9),
                                  Item => LexTokenManager.Word_Size_Token);

            -- reals
            LexTokenLists.Append (List => Possible_Identifiers (10),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (10),
                                  Item => LexTokenManager.Fine_Delta_Token);

            -- subtypes
            LexTokenLists.Append (List => Possible_Identifiers (11),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (11),
                                  Item => LexTokenManager.Any_Priority_Token);

            LexTokenLists.Append (List => Possible_Identifiers (12),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (12),
                                  Item => LexTokenManager.Priority_Token);

            LexTokenLists.Append (List => Possible_Identifiers (13),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (13),
                                  Item => LexTokenManager.Interrupt_Priority_Token);

            -- types
            LexTokenLists.Append (List => Possible_Identifiers (14),
                                  Item => LexTokenManager.Standard_Token);
            LexTokenLists.Append (List => Possible_Identifiers (14),
                                  Item => LexTokenManager.Integer_Token);

            LexTokenLists.Append (List => Possible_Identifiers (15),
                                  Item => LexTokenManager.Standard_Token);
            LexTokenLists.Append (List => Possible_Identifiers (15),
                                  Item => LexTokenManager.Float_Token);

            -- private type
            LexTokenLists.Append (List => Possible_Identifiers (16),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (16),
                                  Item => LexTokenManager.Address_Token);

            -- Ravenscar type
            LexTokenLists.Append (List => Possible_Identifiers (17),
                                  Item => LexTokenManager.Ada_Token);
            LexTokenLists.Append (List => Possible_Identifiers (17),
                                  Item => LexTokenManager.Real_Time_Token);
            LexTokenLists.Append (List => Possible_Identifiers (17),
                                  Item => LexTokenManager.Seconds_Count_Token);

            LexTokenLists.Append (List => Possible_Identifiers (18),
                                  Item => LexTokenManager.Ada_Token);
            LexTokenLists.Append (List => Possible_Identifiers (18),
                                  Item => LexTokenManager.Interrupts_Token);
            LexTokenLists.Append (List => Possible_Identifiers (18),
                                  Item => LexTokenManager.Interrupt_ID_Token);

            -- typed constants
            LexTokenLists.Append (List => Possible_Identifiers (19),
                                  Item => LexTokenManager.System_Token);
            LexTokenLists.Append (List => Possible_Identifiers (19),
                                  Item => LexTokenManager.Default_Bit_Order_Token);

            Possible_Library_Packages :=
              Library_Package_Strings'
              (1 => LexTokenManager.Standard_Token,
               2 => LexTokenManager.System_Token,
               3 => LexTokenManager.Ada_Token);
            Possible_Child_Packages   :=
              Child_Package_Strings'(1 => LexTokenManager.Real_Time_Token,
                                     2 => LexTokenManager.Interrupts_Token);

            Predefined_Scope    :=
              Dictionary.Set_Visibility
              (The_Visibility => Dictionary.Visible,
               The_Unit       => Dictionary.GetPredefinedPackageStandard);
            Current_Scope       := Predefined_Scope;
            The_Lex_Val         :=
              LexTokenManager.Lex_Value'
              (Position  => LexTokenManager.Null_Token_Position,
               Token_Str => LexTokenManager.Null_String);
            Current_Declaration :=
              Declaration'
              (Name        => The_Lex_Val,
               My_Type     => Declaration_Type'First,
               Enc_Package => Package_Hierarchy'(Child_Package_Depth => The_Lex_Val),
               Base_Type   => The_Lex_Val,
               Low_Bound   => Maths.NoValue,
               High_Bound  => Maths.NoValue,
               Num_Digits  => Maths.NoValue,
               Value       => Maths.NoValue,
               Value_Str   => The_Lex_Val);
            The_Token           := SP_Symbols.SPDEFAULT;
            Unary_Minus         := False;
            Int_Subtract        := False;
            Int_Add             := False;
            Int_Exponent        := False;
            Int_Val             := Maths.NoValue;
            Int_Exponent_Val    := Maths.NoValue;
            Int_Add_Sub_Val     := Maths.NoValue;
            Expr_Int_Value      := Maths.NoValue;
            Lookahead           := False;
            Parsing_Ends        := False;
            Valid_Package       := False;
            Saved_Parser_State  := Parser_States'First;
            Saved_Parser_Stage  := Parser_Stages'First;
            Next_Expected_State := Parser_States'First;
            Parser_State        := Parser_States'First;
            Parser_Stage        := Parser_Stages'First;
            Errors_Occurred     := False;
         end State_Machine_Initialise;

         procedure State_Machine_Iterate
         --# global in     CommandLineData.Content;
         --#        in     Possible_Child_Packages;
         --#        in     Possible_Identifiers;
         --#        in     Possible_Library_Packages;
         --#        in     Predefined_Scope;
         --#        in     The_Config_File;
         --#        in out Any_Priority_Sym;
         --#        in out Current_Declaration;
         --#        in out Current_Scope;
         --#        in out Dictionary.Dict;
         --#        in out ErrorHandler.Error_Context;
         --#        in out Errors_Occurred;
         --#        in out Expr_Int_Value;
         --#        in out Interrupt_Priority_Sym;
         --#        in out Int_Add;
         --#        in out Int_Add_Sub_Val;
         --#        in out Int_Exponent;
         --#        in out Int_Exponent_Val;
         --#        in out Int_Subtract;
         --#        in out Int_Val;
         --#        in out LexTokenManager.State;
         --#        in out Lib_Package_Symbol;
         --#        in out Lookahead;
         --#        in out Next_Expected_State;
         --#        in out Package_End_Name;
         --#        in out Package_Full_Name;
         --#        in out Package_Name_Depth;
         --#        in out Parser_Stage;
         --#        in out Parser_State;
         --#        in out Priority_Sym;
         --#        in out Saved_Parser_Stage;
         --#        in out Saved_Parser_State;
         --#        in out SparkLex.Curr_Line;
         --#        in out SPARK_IO.File_Sys;
         --#        in out The_Lex_Val;
         --#        in out The_Token;
         --#        in out Unary_Minus;
         --#        in out Valid_Package;
         --# derives Any_Priority_Sym,
         --#         Interrupt_Priority_Sym,
         --#         Priority_Sym               from Any_Priority_Sym,
         --#                                         CommandLineData.Content,
         --#                                         Current_Declaration,
         --#                                         Current_Scope,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Expr_Int_Value,
         --#                                         Interrupt_Priority_Sym,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Package_Full_Name,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Identifiers,
         --#                                         Priority_Sym,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         Valid_Package &
         --#         Current_Declaration        from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Expr_Int_Value,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token,
         --#                                         Unary_Minus &
         --#         Current_Scope              from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lib_Package_Symbol,
         --#                                         Lookahead,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Child_Packages,
         --#                                         Possible_Library_Packages,
         --#                                         Predefined_Scope,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token &
         --#         Dictionary.Dict            from *,
         --#                                         Any_Priority_Sym,
         --#                                         CommandLineData.Content,
         --#                                         Current_Declaration,
         --#                                         Current_Scope,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Expr_Int_Value,
         --#                                         Interrupt_Priority_Sym,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Package_Full_Name,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Identifiers,
         --#                                         Possible_Library_Packages,
         --#                                         Priority_Sym,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token,
         --#                                         Valid_Package &
         --#         ErrorHandler.Error_Context,
         --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
         --#                                         CommandLineData.Content,
         --#                                         Current_Declaration,
         --#                                         Current_Scope,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Expr_Int_Value,
         --#                                         Interrupt_Priority_Sym,
         --#                                         Int_Add,
         --#                                         Int_Add_Sub_Val,
         --#                                         Int_Exponent,
         --#                                         Int_Exponent_Val,
         --#                                         Int_Subtract,
         --#                                         Int_Val,
         --#                                         LexTokenManager.State,
         --#                                         Lib_Package_Symbol,
         --#                                         Lookahead,
         --#                                         Package_End_Name,
         --#                                         Package_Full_Name,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Child_Packages,
         --#                                         Possible_Identifiers,
         --#                                         Possible_Library_Packages,
         --#                                         Priority_Sym,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token,
         --#                                         Unary_Minus,
         --#                                         Valid_Package &
         --#         Errors_Occurred            from *,
         --#                                         Any_Priority_Sym,
         --#                                         CommandLineData.Content,
         --#                                         Current_Declaration,
         --#                                         Current_Scope,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Expr_Int_Value,
         --#                                         Interrupt_Priority_Sym,
         --#                                         LexTokenManager.State,
         --#                                         Lib_Package_Symbol,
         --#                                         Lookahead,
         --#                                         Package_End_Name,
         --#                                         Package_Full_Name,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Child_Packages,
         --#                                         Possible_Identifiers,
         --#                                         Possible_Library_Packages,
         --#                                         Priority_Sym,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token,
         --#                                         Unary_Minus,
         --#                                         Valid_Package &
         --#         Expr_Int_Value             from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Int_Add,
         --#                                         Int_Add_Sub_Val,
         --#                                         Int_Exponent,
         --#                                         Int_Exponent_Val,
         --#                                         Int_Subtract,
         --#                                         Int_Val,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Token,
         --#                                         Unary_Minus &
         --#         Int_Add,
         --#         Int_Exponent,
         --#         Int_Subtract,
         --#         Lookahead,
         --#         Package_Name_Depth,
         --#         Saved_Parser_Stage,
         --#         Saved_Parser_State,
         --#         Unary_Minus                from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Token &
         --#         Int_Add_Sub_Val,
         --#         Int_Val,
         --#         Package_End_Name           from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token &
         --#         Int_Exponent_Val           from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val &
         --#         LexTokenManager.State      from *,
         --#                                         Any_Priority_Sym,
         --#                                         CommandLineData.Content,
         --#                                         Current_Declaration,
         --#                                         Current_Scope,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         Expr_Int_Value,
         --#                                         Interrupt_Priority_Sym,
         --#                                         Lookahead,
         --#                                         Package_Full_Name,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Identifiers,
         --#                                         Priority_Sym,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token,
         --#                                         Unary_Minus,
         --#                                         Valid_Package &
         --#         Lib_Package_Symbol         from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Library_Packages,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token &
         --#         Next_Expected_State        from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Token,
         --#                                         Unary_Minus &
         --#         Package_Full_Name          from *,
         --#                                         CommandLineData.Content,
         --#                                         Current_Declaration,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Token &
         --#         Parser_Stage               from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Package_Name_Depth,
         --#                                         Parser_State,
         --#                                         Saved_Parser_Stage,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Token,
         --#                                         Unary_Minus &
         --#         Parser_State               from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         Next_Expected_State,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Saved_Parser_State,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Token,
         --#                                         Unary_Minus &
         --#         SparkLex.Curr_Line,
         --#         The_Lex_Val,
         --#         The_Token                  from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lookahead,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File &
         --#         Valid_Package              from *,
         --#                                         CommandLineData.Content,
         --#                                         Dictionary.Dict,
         --#                                         ErrorHandler.Error_Context,
         --#                                         LexTokenManager.State,
         --#                                         Lib_Package_Symbol,
         --#                                         Lookahead,
         --#                                         Package_Name_Depth,
         --#                                         Parser_Stage,
         --#                                         Parser_State,
         --#                                         Possible_Child_Packages,
         --#                                         Possible_Library_Packages,
         --#                                         SparkLex.Curr_Line,
         --#                                         SPARK_IO.File_Sys,
         --#                                         The_Config_File,
         --#                                         The_Lex_Val,
         --#                                         The_Token;
         is
            Unwanted_Punct_Token : Boolean;

            procedure Next_Stage
            --# global in out Parser_Stage;
            --# derives Parser_Stage from *;
            is
            begin
               Parser_Stage := Parser_Stage + 1;
            end Next_Stage;

            procedure New_State (The_Stage : in Parser_States)
            --# global out Parser_Stage;
            --#        out Parser_State;
            --# derives Parser_Stage from  &
            --#         Parser_State from The_Stage;
            is
            begin
               Parser_Stage := 1;
               Parser_State := The_Stage;
            end New_State;

            procedure Guarded_Literal_To_Value (Lex : in     LexTokenManager.Lex_Value;
                                                Val :    out Maths.Value)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     LexTokenManager.State;
            --#        in out ErrorHandler.Error_Context;
            --#        in out SPARK_IO.File_Sys;
            --# derives ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Lex,
            --#                                         LexTokenManager.State,
            --#                                         SPARK_IO.File_Sys &
            --#         Val                        from Lex,
            --#                                         LexTokenManager.State;
            is
               Local_Error_Code : Maths.ErrorCode;
            begin
               Maths.LiteralToValue (Lex.Token_Str, Val, Local_Error_Code);
               if Local_Error_Code /= Maths.NoError then
                  ErrorHandler.Semantic_Warning (Err_Num  => 200,
                                                 Position => Lex.Position,
                                                 Id_Str   => Lex.Token_Str);
               end if;
            end Guarded_Literal_To_Value;

            procedure Raise_Syntax_Error (The_Error : in Syntax_Errors)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out LexTokenManager.State;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#           out Errors_Occurred;
            --#           out Next_Expected_State;
            --#           out Parser_Stage;
            --# derives ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Error &
            --#         Errors_Occurred,
            --#         Parser_Stage,
            --#         Parser_State               from  &
            --#         LexTokenManager.State      from *,
            --#                                         The_Error &
            --#         Next_Expected_State        from Parser_State;
            is
               No_Of_Terminals : SP_Expected_Symbols.SP_Ess_Sym_Range;
               Terminal_List   : SP_Expected_Symbols.SP_Exp_Sym_List :=
                 SP_Expected_Symbols.SP_Exp_Sym_List'(others => SP_Symbols.SPDEFAULT);
            begin
               case The_Error.The_Type is
                  when Exp_Symbol =>
                     No_Of_Terminals   := 1;
                     Terminal_List (1) := The_Error.Expected_Symbol;
                  when Exp_Defn =>
                     No_Of_Terminals   := 3;
                     Terminal_List (1) := SP_Symbols.identifier;
                     Terminal_List (2) := SP_Symbols.RWtype;
                     Terminal_List (3) := SP_Symbols.RWsubtype;
                  when Exp_Digits_Range =>
                     No_Of_Terminals   := 3;
                     Terminal_List (1) := SP_Symbols.RWdigits;
                     Terminal_List (2) := SP_Symbols.RWrange;
                     Terminal_List (3) := SP_Symbols.RWprivate;
                  when Exp_Minus_Real =>
                     No_Of_Terminals   := 2;
                     Terminal_List (1) := SP_Symbols.minus;
                     Terminal_List (2) := SP_Symbols.real_number;
                  when Exp_Minus_Integer =>
                     No_Of_Terminals   := 2;
                     Terminal_List (1) := SP_Symbols.minus;
                     Terminal_List (2) := SP_Symbols.integer_number;
                  when Exp_Integer_Real =>
                     No_Of_Terminals   := 2;
                     Terminal_List (1) := SP_Symbols.integer_number;
                     Terminal_List (2) := SP_Symbols.real_number;
               end case;
               --# assert True;
               case Parser_State is
                  when Config_Defn_Start | Next_Config_Defn_Start =>
                     Next_Expected_State := Parser_State;
                  when others =>
                     Next_Expected_State := Defn;
               end case;
               New_State (The_Stage => Found_Syntax_Error);
               ErrorHandler.Syntax_Error
                 (Error_Item          => The_Error.The_Lex_Val,
                  Current_Sym         => The_Error.The_Symbol,
                  Entry_Symbol        => The_Error.The_Symbol,
                  No_Of_Terminals     => No_Of_Terminals,
                  No_Of_Non_Terminals => 0,
                  Terminal_List       => Terminal_List,
                  Non_Terminal_List   => Terminal_List);
               Errors_Occurred := True;
            end Raise_Syntax_Error;

            procedure Raise_Semantic_Error (Err_Num : in Natural;
                                            Lex_Val : in LexTokenManager.Lex_Value)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     LexTokenManager.State;
            --#        in out ErrorHandler.Error_Context;
            --#        in out SPARK_IO.File_Sys;
            --#           out Errors_Occurred;
            --# derives ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Err_Num,
            --#                                         LexTokenManager.State,
            --#                                         Lex_Val,
            --#                                         SPARK_IO.File_Sys &
            --#         Errors_Occurred            from ;
            is
            begin
               ErrorHandler.Semantic_Error
                 (Err_Num   => Err_Num,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Lex_Val.Position,
                  Id_Str    => Lex_Val.Token_Str);
               Errors_Occurred := True;
            end Raise_Semantic_Error;

            ----------------------------------------------
            -- Check semantic validity of new declaration,
            -- and update dictionary as appropriate
            ----------------------------------------------

            procedure Check_And_Process_Declaration (The_Decl : in Declaration)
            --# global in     CommandLineData.Content;
            --#        in     Current_Scope;
            --#        in     Package_Full_Name;
            --#        in     Possible_Identifiers;
            --#        in     Valid_Package;
            --#        in out Any_Priority_Sym;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out Interrupt_Priority_Sym;
            --#        in out LexTokenManager.State;
            --#        in out Priority_Sym;
            --#        in out SPARK_IO.File_Sys;
            --# derives Any_Priority_Sym,
            --#         Dictionary.Dict,
            --#         Errors_Occurred,
            --#         Interrupt_Priority_Sym,
            --#         LexTokenManager.State,
            --#         Priority_Sym               from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Decl,
            --#                                         Valid_Package &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Decl,
            --#                                         Valid_Package;
            is
               type Failures is (
                                 Not_Valid_Name,
                                 Wrong_Type_Found,
                                 Empty_Range,
                                 No_Failure,
                                 Not_Within_Valid_Package,
                                 Invalid_Redeclaration,
                                 Parent_Type_Does_Not_Exist,
                                 Subtype_Range_Mismatch,
                                 Parent_Type_Not_Integer,
                                 Parent_Type_Not_AP,
                                 Priority_Range_Insufficient,
                                 Any_Prio_First,
                                 Any_Prio_Last,
                                 Prio_Mid_Point,
                                 Must_Be_Private,
                                 Not_Positive_Power_Of_2,
                                 Bit_Order_Wrong_Type,
                                 Bit_Order_Wrong_Value);

               type Identifier_Type_Array is array (Possible_Identifier_Range) of Declaration_Type;

               Identifier_Types : constant Identifier_Type_Array :=
                 Identifier_Type_Array'
                 (1  => Dec_Named_Integer,
                  2  => Dec_Named_Integer,
                  3  => Dec_Named_Integer,
                  4  => Dec_Named_Integer,
                  5  => Dec_Named_Integer,
                  6  => Dec_Named_Integer,
                  7  => Dec_Named_Integer,
                  8  => Dec_Named_Integer,
                  9  => Dec_Named_Integer,
                  10 => Dec_Named_Real,
                  11 => Dec_Integer_Subtype,
                  12 => Dec_Integer_Subtype,
                  13 => Dec_Integer_Subtype,
                  14 => Dec_Integer_Type,
                  15 => Dec_Floating_Point_Type,
                  16 => Dec_Private,
                  17 => Dec_Integer_Type,
                  18 => Dec_Integer_Type,
                  19 => Dec_Typed_Constant);

               Failure_Reason        : Failures := No_Failure;
               Dotted_Name           : E_Strings.T;
               Name_Match            : Boolean  := False;
               Match_Type            : Declaration_Type;
               Parent_Type           : Dictionary.Symbol;
               Is_Any_Priority       : Boolean  := False;
               Is_Priority           : Boolean  := False;
               Is_Interrupt_Priority : Boolean  := False;
               Is_Address            : Boolean  := False;
               Is_Max_Binary_Modulus : Boolean  := False;
               Is_Predef_Integer     : Boolean  := False;
               Is_Predef_Float       : Boolean  := False;
               Is_Seconds_Count      : Boolean  := False;
               Is_Interrupt_ID       : Boolean  := False;
               Is_Default_Bit_Order  : Boolean  := False;

               procedure Match_Identifier_List
                 (ID       : in     LexTokenLists.Lists;
                  To_Match : in     E_Strings.T;
                  Position : in     LexTokenManager.Token_Position;
                  Matched  :    out Boolean)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     LexTokenManager.State;
               --#        in out ErrorHandler.Error_Context;
               --#        in out SPARK_IO.File_Sys;
               --# derives ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         ErrorHandler.Error_Context,
               --#                                         ID,
               --#                                         LexTokenManager.State,
               --#                                         Position,
               --#                                         SPARK_IO.File_Sys,
               --#                                         To_Match &
               --#         Matched                    from ID,
               --#                                         LexTokenManager.State,
               --#                                         To_Match;
               is
                  Start_Pos  : E_Strings.Positions        := E_Strings.Positions'First;
                  Str_Length : E_Strings.Lengths          := E_Strings.Lengths'First;
                  Str        : LexTokenManager.Lex_String := LexTokenManager.Null_String;
               begin
                  Matched := True;
                  for I in LexTokenLists.Positions range LexTokenLists.Positions'First .. LexTokenLists.Get_Length (List => ID)
                  loop
                     Str        := LexTokenLists.Get_Element (List => ID,
                                                              Pos  => I);
                     Str_Length := E_Strings.Get_Length (E_Str => LexTokenManager.Lex_String_To_String (Lex_Str => Str));
                     Matched    :=
                       LexTokenManager.Comp_Str_Case_Insensitive
                       (Str     => E_Strings.Section (E_Str     => To_Match,
                                                      Start_Pos => Start_Pos,
                                                      Length    => Str_Length),
                        Lex_Str => Str);
                     exit when not Matched;
                     Start_Pos := (Start_Pos + Str_Length) + 1;
                  end loop;
                  Matched := Matched and then Start_Pos - 2 = E_Strings.Get_Length (E_Str => To_Match);
                  if Matched then
                     Start_Pos := (Start_Pos - Str_Length) - 1;
                     Casing.Check_String_Casing
                       (Str      => E_Strings.Section (E_Str     => To_Match,
                                                       Start_Pos => Start_Pos,
                                                       Length    => Str_Length),
                        Lex_Str  => Str,
                        Position => Position);
                  end if;
               end Match_Identifier_List;

               function Ck_Failed return Boolean
               --# global in Failure_Reason;
               is
               begin
                  return Failure_Reason /= No_Failure;
               end Ck_Failed;

               procedure Do_Ck_Valid_Package
               --# global in     Valid_Package;
               --#        in out Failure_Reason;
               --# derives Failure_Reason from *,
               --#                             Valid_Package;
               is
               begin
                  if not Valid_Package then
                     Failure_Reason := Not_Within_Valid_Package;
                  end if;
               end Do_Ck_Valid_Package;

               procedure Do_Ck_Suffix_Name
               --# global in     Dotted_Name;
               --#        in     Failure_Reason;
               --#        in out Match_Type;
               --#        in out Name_Match;
               --# derives Match_Type,
               --#         Name_Match from *,
               --#                         Dotted_Name,
               --#                         Failure_Reason;
               is
                  Standard_Prefix : E_Strings.T;
                  Float_Suffix    : E_Strings.T;
                  Integer_Suffix  : E_Strings.T;

                  function Match_Prefix (Prefix   : E_Strings.T;
                                         To_Match : E_Strings.T) return Boolean is
                     Matched : Boolean;
                  begin
                     if E_Strings.Get_Length (E_Str => Prefix) > E_Strings.Get_Length (E_Str => To_Match) then
                        Matched := False;
                     else
                        Matched :=
                          E_Strings.Eq_String
                          (E_Str1 => Prefix,
                           E_Str2 => E_Strings.Section
                             (E_Str     => To_Match,
                              Start_Pos => 1,
                              Length    => E_Strings.Get_Length (E_Str => Prefix)));
                     end if;
                     return Matched;
                  end Match_Prefix;

                  function Match_Suffix (Suffix   : E_Strings.T;
                                         To_Match : E_Strings.T) return Boolean is
                     Matched : Boolean := True;
                  begin
                     if E_Strings.Get_Length (E_Str => Suffix) > E_Strings.Get_Length (E_Str => To_Match) then
                        Matched := False;
                     else
                        for I in Natural range 1 .. E_Strings.Get_Length (E_Str => Suffix) loop
                           if Ada.Characters.Handling.To_Lower
                             (Item => E_Strings.Get_Element
                                (E_Str => To_Match,
                                 Pos   => (E_Strings.Get_Length (E_Str => To_Match) - I) + 1)) /=
                             Ada.Characters.Handling.To_Lower
                             (Item => E_Strings.Get_Element
                                (E_Str => Suffix,
                                 Pos   => (E_Strings.Get_Length (E_Str => Suffix) - I) + 1)) then
                              Matched := False;
                           end if;
                           exit when not Matched;
                        end loop;
                     end if;
                     return Matched;
                  end Match_Suffix;

               begin
                  Standard_Prefix := E_Strings.Copy_String (Str => "Standard.");
                  Float_Suffix    := E_Strings.Copy_String (Str => "_Float");
                  Integer_Suffix  := E_Strings.Copy_String (Str => "_Integer");
                  if not Ck_Failed then
                     if Match_Prefix (Prefix   => Standard_Prefix,
                                      To_Match => Dotted_Name) then
                        if Match_Suffix (Suffix   => Integer_Suffix,
                                         To_Match => Dotted_Name) then
                           Name_Match := True;
                           Match_Type := Dec_Integer_Type;
                        elsif Match_Suffix (Suffix   => Float_Suffix,
                                            To_Match => Dotted_Name) then
                           Name_Match := True;
                           Match_Type := Dec_Floating_Point_Type;
                        end if;
                     end if;
                  end if;
               end Do_Ck_Suffix_Name;

               procedure Do_Ck_Def_Name_With_Priority_Ck
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     Dotted_Name;
               --#        in     LexTokenManager.State;
               --#        in     Name_Match;
               --#        in     Possible_Identifiers;
               --#        in     The_Decl;
               --#        in out ErrorHandler.Error_Context;
               --#        in out Failure_Reason;
               --#        in out Is_Any_Priority;
               --#        in out Is_Interrupt_Priority;
               --#        in out Is_Priority;
               --#        in out Match_Type;
               --#        in out SPARK_IO.File_Sys;
               --# derives ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         Dotted_Name,
               --#                                         ErrorHandler.Error_Context,
               --#                                         Failure_Reason,
               --#                                         LexTokenManager.State,
               --#                                         Name_Match,
               --#                                         Possible_Identifiers,
               --#                                         SPARK_IO.File_Sys,
               --#                                         The_Decl &
               --#         Failure_Reason,
               --#         Is_Any_Priority,
               --#         Is_Interrupt_Priority,
               --#         Is_Priority,
               --#         Match_Type                 from *,
               --#                                         Dotted_Name,
               --#                                         Failure_Reason,
               --#                                         LexTokenManager.State,
               --#                                         Name_Match,
               --#                                         Possible_Identifiers;
               is
                  Any_Priority_Subtype       : constant Possible_Identifier_Range := 11;
                  Priority_Subtype           : constant Possible_Identifier_Range := 12;
                  Interrupt_Priority_Subtype : constant Possible_Identifier_Range := 13;

                  Local_Name_Match  : Boolean := False;
                  Dotted_Name_Match : Boolean;
               begin
                  if not (Name_Match or Ck_Failed) then
                     Failure_Reason := Not_Valid_Name; -- initially
                     for I in Possible_Identifier_Range loop
                        Match_Identifier_List
                          (ID       => Possible_Identifiers (I),
                           To_Match => Dotted_Name,
                           Position => The_Decl.Name.Position,
                           Matched  => Dotted_Name_Match);
                        if Dotted_Name_Match then
                           Failure_Reason   := No_Failure;
                           Local_Name_Match := False;
                           case I is
                              when Any_Priority_Subtype =>
                                 Is_Any_Priority := True;
                              when Priority_Subtype =>
                                 Is_Priority := True;
                              when Interrupt_Priority_Subtype =>
                                 Is_Interrupt_Priority := True;
                              when others =>
                                 null;
                           end case;
                           Match_Type := Identifier_Types (I);
                        end if;
                        exit when Local_Name_Match;
                     end loop;
                  end if;
               end Do_Ck_Def_Name_With_Priority_Ck;

               procedure Do_Ck_Def_Name
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     Dotted_Name;
               --#        in     LexTokenManager.State;
               --#        in     Name_Match;
               --#        in     Possible_Identifiers;
               --#        in     The_Decl;
               --#        in out ErrorHandler.Error_Context;
               --#        in out Failure_Reason;
               --#        in out Is_Address;
               --#        in out Is_Default_Bit_Order;
               --#        in out Is_Interrupt_ID;
               --#        in out Is_Max_Binary_Modulus;
               --#        in out Is_Predef_Float;
               --#        in out Is_Predef_Integer;
               --#        in out Is_Seconds_Count;
               --#        in out Match_Type;
               --#        in out SPARK_IO.File_Sys;
               --# derives ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         Dotted_Name,
               --#                                         ErrorHandler.Error_Context,
               --#                                         Failure_Reason,
               --#                                         LexTokenManager.State,
               --#                                         Name_Match,
               --#                                         Possible_Identifiers,
               --#                                         SPARK_IO.File_Sys,
               --#                                         The_Decl &
               --#         Failure_Reason,
               --#         Is_Address,
               --#         Is_Default_Bit_Order,
               --#         Is_Interrupt_ID,
               --#         Is_Max_Binary_Modulus,
               --#         Is_Predef_Float,
               --#         Is_Predef_Integer,
               --#         Is_Seconds_Count,
               --#         Match_Type                 from *,
               --#                                         Dotted_Name,
               --#                                         Failure_Reason,
               --#                                         LexTokenManager.State,
               --#                                         Name_Match,
               --#                                         Possible_Identifiers;
               is
                  Max_Binary_Modulus : constant Possible_Identifier_Range := 3;
                  Predef_Integer     : constant Possible_Identifier_Range := 14;
                  Predef_Float       : constant Possible_Identifier_Range := 15;
                  System_Address     : constant Possible_Identifier_Range := 16;
                  Seconds_Count      : constant Possible_Identifier_Range := 17;
                  Interrupt_ID       : constant Possible_Identifier_Range := 18;
                  Default_Bit_Order  : constant Possible_Identifier_Range := 19;
                  Dotted_Name_Match  : Boolean;
               begin
                  if not (Name_Match or Ck_Failed) then
                     Failure_Reason := Not_Valid_Name; -- initially
                     for I in Possible_Identifier_Range loop
                        Match_Identifier_List
                          (ID       => Possible_Identifiers (I),
                           To_Match => Dotted_Name,
                           Position => The_Decl.Name.Position,
                           Matched  => Dotted_Name_Match);
                        if Dotted_Name_Match then
                           Is_Address            := I = System_Address;
                           Is_Max_Binary_Modulus := I = Max_Binary_Modulus;
                           Is_Predef_Integer     := I = Predef_Integer;
                           Is_Predef_Float       := I = Predef_Float;
                           Is_Seconds_Count      := I = Seconds_Count;
                           Is_Interrupt_ID       := I = Interrupt_ID;
                           Is_Default_Bit_Order  := I = Default_Bit_Order;
                           Failure_Reason        := No_Failure;
                           Match_Type            := Identifier_Types (I);
                           exit;
                        end if;
                     end loop;
                  end if;
               end Do_Ck_Def_Name;

               procedure Do_Ck_Existing_Decl
               --# global in     CommandLineData.Content;
               --#        in     Current_Scope;
               --#        in     Dictionary.Dict;
               --#        in     Is_Default_Bit_Order;
               --#        in     Is_Interrupt_ID;
               --#        in     Is_Seconds_Count;
               --#        in     LexTokenManager.State;
               --#        in     The_Decl;
               --#        in out Failure_Reason;
               --# derives Failure_Reason from *,
               --#                             CommandLineData.Content,
               --#                             Current_Scope,
               --#                             Dictionary.Dict,
               --#                             Is_Default_Bit_Order,
               --#                             Is_Interrupt_ID,
               --#                             Is_Seconds_Count,
               --#                             LexTokenManager.State,
               --#                             The_Decl;
               is
                  Check_Symbol : Dictionary.Symbol;
               begin
                  if not Ck_Failed then
                     Check_Symbol :=
                       Dictionary.LookupItem
                       (Name              => The_Decl.Name.Token_Str,
                        Scope             => Current_Scope,
                        Context           => Dictionary.ProgramContext,
                        Full_Package_Name => False);
                     if not Dictionary.Is_Null_Symbol (Check_Symbol) then
                        -- redeclaration of existing object ... but
                        if (Dictionary.IsTypeMark (Check_Symbol)
                              and then (Dictionary.IsPredefinedIntegerType (Check_Symbol)
                                          or else Dictionary.IsPredefinedFloatType (Check_Symbol)))
                          or else Is_Default_Bit_Order
                          or else Is_Seconds_Count
                          or else Is_Interrupt_ID then

                           if Is_Default_Bit_Order then
                              if not Dictionary.IsDeferredConstant (Check_Symbol) then
                                 Failure_Reason := Invalid_Redeclaration;
                              end if;
                           else
                              -- Must be a numeric constant
                              if not (Maths.HasNoValue
                                        (Maths.ValueRep
                                           (Dictionary.GetScalarAttributeValue
                                              (False,
                                               LexTokenManager.First_Token,
                                               Check_Symbol)))) then
                                 Failure_Reason := Invalid_Redeclaration;
                              end if;
                           end if;
                        else
                           Failure_Reason := Invalid_Redeclaration;
                        end if;
                     end if;
                  end if;
               end Do_Ck_Existing_Decl;

               procedure Do_Ck_Type
               --# global in     Match_Type;
               --#        in     The_Decl;
               --#        in out Failure_Reason;
               --# derives Failure_Reason from *,
               --#                             Match_Type,
               --#                             The_Decl;
               is
               begin
                  if not Ck_Failed then
                     if Match_Type /= The_Decl.My_Type then
                        if Match_Type = Dec_Private then
                           Failure_Reason := Must_Be_Private;
                        else
                           Failure_Reason := Wrong_Type_Found;
                        end if;
                     end if;
                  end if;

               end Do_Ck_Type;

               procedure Do_Ck_Bounds
               --# global in     The_Decl;
               --#        in out Failure_Reason;
               --# derives Failure_Reason from *,
               --#                             The_Decl;
               is
                  Temp_Value : Maths.Value;
                  Local_ME   : Maths.ErrorCode;
               begin
                  if not Ck_Failed then
                     --# accept F, 10, Local_ME, "Local_ME unused here";
                     Maths.Greater (The_Decl.Low_Bound, The_Decl.High_Bound, Temp_Value, Local_ME);
                     --# end accept;
                     -- ineffective assignment to Local_ME expected here
                     -- no possible error; lexical analysis confirms correct typing

                     if Temp_Value = Maths.BoolToValue (True) then
                        Failure_Reason := Empty_Range;
                     end if;
                  end if;
                  --# accept F, 33, Local_ME, "Local_ME unused here";
               end Do_Ck_Bounds;

               procedure Do_Ck_Base_Type
               --# global in     CommandLineData.Content;
               --#        in     Current_Scope;
               --#        in     Dictionary.Dict;
               --#        in     LexTokenManager.State;
               --#        in     The_Decl;
               --#        in out ErrorHandler.Error_Context;
               --#        in out Failure_Reason;
               --#        in out Parent_Type;
               --#        in out SPARK_IO.File_Sys;
               --# derives ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from CommandLineData.Content,
               --#                                         Current_Scope,
               --#                                         Dictionary.Dict,
               --#                                         ErrorHandler.Error_Context,
               --#                                         Failure_Reason,
               --#                                         LexTokenManager.State,
               --#                                         SPARK_IO.File_Sys,
               --#                                         The_Decl &
               --#         Failure_Reason,
               --#         Parent_Type                from *,
               --#                                         CommandLineData.Content,
               --#                                         Current_Scope,
               --#                                         Dictionary.Dict,
               --#                                         Failure_Reason,
               --#                                         LexTokenManager.State,
               --#                                         The_Decl;
               is
                  Check_Symbol : Dictionary.Symbol;
                  Temp_Value   : Maths.Value;
                  Local_ME     : Maths.ErrorCode;
               begin
                  if not Ck_Failed then
                     Check_Symbol :=
                       Dictionary.LookupItem
                       (Name              => The_Decl.Base_Type.Token_Str,
                        Scope             => Current_Scope,
                        Context           => Dictionary.ProgramContext,
                        Full_Package_Name => False);
                     if Dictionary.Is_Null_Symbol (Check_Symbol) then
                        -- parent type does not exist
                        Failure_Reason := Parent_Type_Does_Not_Exist;
                     else
                        --# accept F, 10, Local_ME, "Local_ME unused here";
                        Maths.Lesser
                          (The_Decl.Low_Bound,
                           Maths.ValueRep (Dictionary.GetScalarAttributeValue (False, LexTokenManager.First_Token, Check_Symbol)),
                           Temp_Value,
                           Local_ME);
                        --# end accept;
                        -- ineffective assignment to Local_ME expected here
                        -- no possible error; lexical analysis confirms correct typing

                        if Temp_Value = Maths.BoolToValue (True) then
                           -- type'first less than type'base'first
                           Failure_Reason := Subtype_Range_Mismatch;
                        else
                           --# accept F, 10, Local_ME, "Local_ME unused here";
                           Maths.Greater
                             (The_Decl.High_Bound,
                              Maths.ValueRep
                                (Dictionary.GetScalarAttributeValue (False, LexTokenManager.Last_Token, Check_Symbol)),
                              Temp_Value,
                              Local_ME);
                           --# end accept;
                           -- ineffective assignment to Local_ME expected here
                           -- no possible error; lexical analysis confirms correct typing
                           if Temp_Value = Maths.BoolToValue (True) then
                              -- type'last greater than type'base'last
                              Failure_Reason := Subtype_Range_Mismatch;
                           else
                              Casing.Check_Casing
                                (Lex_Str1 => Dictionary.GetSimpleName (Item => Check_Symbol),
                                 Lex_Str2 => The_Decl.Base_Type.Token_Str,
                                 Position => The_Decl.Base_Type.Position);
                              Parent_Type := Check_Symbol;
                           end if;
                        end if;
                     end if;
                  end if;
                  --# accept F, 33, Local_ME, "Local_ME unused here";
               end Do_Ck_Base_Type;

               procedure Do_Ck_System_Priority
               --# global in     Any_Priority_Sym;
               --#        in     Dictionary.Dict;
               --#        in     Interrupt_Priority_Sym;
               --#        in     Is_Any_Priority;
               --#        in     Is_Interrupt_Priority;
               --#        in     Is_Priority;
               --#        in     LexTokenManager.State;
               --#        in     Parent_Type;
               --#        in     Priority_Sym;
               --#        in     The_Decl;
               --#        in out Failure_Reason;
               --# derives Failure_Reason from *,
               --#                             Any_Priority_Sym,
               --#                             Dictionary.Dict,
               --#                             Interrupt_Priority_Sym,
               --#                             Is_Any_Priority,
               --#                             Is_Interrupt_Priority,
               --#                             Is_Priority,
               --#                             LexTokenManager.State,
               --#                             Parent_Type,
               --#                             Priority_Sym,
               --#                             The_Decl;
               is

                  procedure Check_Priority
                  --# global in     Any_Priority_Sym;
                  --#        in     Dictionary.Dict;
                  --#        in     Interrupt_Priority_Sym;
                  --#        in     LexTokenManager.State;
                  --#        in     Parent_Type;
                  --#        in     The_Decl;
                  --#        in out Failure_Reason;
                  --# derives Failure_Reason from *,
                  --#                             Any_Priority_Sym,
                  --#                             Dictionary.Dict,
                  --#                             Interrupt_Priority_Sym,
                  --#                             LexTokenManager.State,
                  --#                             Parent_Type,
                  --#                             The_Decl;
                  is
                     High_Minus_Low    : Maths.Value;
                     Last_Plus_One     : Maths.Value;
                     Boolean_Condition : Maths.Value;
                     Unwanted_ME       : Maths.ErrorCode;
                  begin
                     -- check parent type is Any_Priority
                     if Dictionary.Types_Are_Equal
                       (Left_Symbol        => Parent_Type,
                        Right_Symbol       => Any_Priority_Sym,
                        Full_Range_Subtype => False) then
                        -- check range of at least 30 values - LRM D.1(26)
                        --# accept F, 10, Unwanted_ME, "Unwanted_ME unused here";
                        Maths.Subtract (The_Decl.High_Bound, The_Decl.Low_Bound, High_Minus_Low, Unwanted_ME);
                        Maths.GreaterOrEqual (High_Minus_Low, Maths.IntegerToValue (29), Boolean_Condition, Unwanted_ME);
                        --# end accept;
                        -- ineffective assignment to Unwanted_ME expected here
                        -- no possible error; lexical analysis confirms correct typing
                        if Boolean_Condition = Maths.BoolToValue (False) then
                           Failure_Reason := Priority_Range_Insufficient;
                        else
                           -- check Priority'First = Any_Priority'First
                           if Maths.ValueRep
                             (Dictionary.GetScalarAttributeValue (False, LexTokenManager.First_Token, Any_Priority_Sym)) /=
                             The_Decl.Low_Bound then
                              Failure_Reason := Any_Prio_First;
                           else
                              if not Dictionary.Is_Null_Symbol (Interrupt_Priority_Sym) then
                                 -- check that Priority'Last + 1 = Interrupt_Priority'First;
                                 Last_Plus_One := The_Decl.High_Bound;
                                 --# accept F, 10, Unwanted_ME, "Unwanted_ME unused here";
                                 Maths.SuccOp (Last_Plus_One, Unwanted_ME);
                                 --# end accept;
                                 -- ineffective assignment to Unwanted_ME expected here
                                 -- no possible error; lexical analysis confirms correct typing
                                 if Maths.ValueRep
                                   (Dictionary.GetScalarAttributeValue
                                      (False,
                                       LexTokenManager.First_Token,
                                       Interrupt_Priority_Sym)) /=
                                   Last_Plus_One then
                                    Failure_Reason := Prio_Mid_Point;
                                 end if;
                              end if;
                           end if;
                        end if;
                     else
                        Failure_Reason := Parent_Type_Not_AP;
                     end if;
                     --# accept F, 33, Unwanted_ME, "Unwanted_ME unused here";
                  end Check_Priority;

                  procedure Check_Interrupt_Priority
                  --# global in     Any_Priority_Sym;
                  --#        in     Dictionary.Dict;
                  --#        in     LexTokenManager.State;
                  --#        in     Parent_Type;
                  --#        in     Priority_Sym;
                  --#        in     The_Decl;
                  --#        in out Failure_Reason;
                  --# derives Failure_Reason from *,
                  --#                             Any_Priority_Sym,
                  --#                             Dictionary.Dict,
                  --#                             LexTokenManager.State,
                  --#                             Parent_Type,
                  --#                             Priority_Sym,
                  --#                             The_Decl;
                  is
                     Last_Plus_One : Maths.Value;
                     Unwanted_ME   : Maths.ErrorCode;
                  begin
                     -- check parent type is Any_Priority
                     if Dictionary.Types_Are_Equal
                       (Left_Symbol        => Parent_Type,
                        Right_Symbol       => Any_Priority_Sym,
                        Full_Range_Subtype => False) then
                        -- there must be at least one value (LRM D.1(25)), but this is
                        -- checked by the bounds check in the general semantic checking
                        -- for a subtype...

                        -- check Interrupt_Priority'Last = Any_Priority'Last
                        if The_Decl.High_Bound /=
                          Maths.ValueRep
                          (Dictionary.GetScalarAttributeValue (False, LexTokenManager.Last_Token, Any_Priority_Sym)) then
                           -- some other semantic error
                           Failure_Reason := Any_Prio_Last;
                        else
                           if not Dictionary.Is_Null_Symbol (Priority_Sym) then
                              -- check that Priority'Last + 1 = Interrupt_Priority'First;
                              Last_Plus_One :=
                                Maths.ValueRep
                                (Dictionary.GetScalarAttributeValue (False, LexTokenManager.Last_Token, Priority_Sym));
                              --# accept F, 10, Unwanted_ME, "Unwanted_ME unused here";
                              Maths.SuccOp (Last_Plus_One, Unwanted_ME);
                              --# end accept;
                              -- ineffective assignment to Unwanted_ME expected here
                              -- no possible error; lexical analysis confirms correct typing
                              if The_Decl.Low_Bound /= Last_Plus_One then
                                 -- some other semantic error
                                 Failure_Reason := Prio_Mid_Point;
                              end if;
                           end if;
                        end if;
                     else
                        Failure_Reason := Parent_Type_Not_AP;
                     end if;
                     --# accept F, 33, Unwanted_ME, "Unwanted_ME unused here";
                  end Check_Interrupt_Priority;

               begin
                  if not Ck_Failed then
                     if Is_Any_Priority then
                        -- check parent type is Integer
                        if not Dictionary.IsPredefinedIntegerType (Parent_Type) then
                           Failure_Reason := Parent_Type_Not_Integer;
                        end if;
                     elsif Is_Priority then
                        Check_Priority;
                     elsif Is_Interrupt_Priority then
                        Check_Interrupt_Priority;
                     end if;
                  end if;
               end Do_Ck_System_Priority;

               procedure Do_Ck_Special_Cases
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     Is_Default_Bit_Order;
               --#        in     Is_Max_Binary_Modulus;
               --#        in     LexTokenManager.State;
               --#        in     The_Decl;
               --#        in out ErrorHandler.Error_Context;
               --#        in out Failure_Reason;
               --#        in out SPARK_IO.File_Sys;
               --# derives ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         ErrorHandler.Error_Context,
               --#                                         Failure_Reason,
               --#                                         Is_Default_Bit_Order,
               --#                                         Is_Max_Binary_Modulus,
               --#                                         LexTokenManager.State,
               --#                                         SPARK_IO.File_Sys,
               --#                                         The_Decl &
               --#         Failure_Reason             from *,
               --#                                         Is_Default_Bit_Order,
               --#                                         Is_Max_Binary_Modulus,
               --#                                         LexTokenManager.State,
               --#                                         The_Decl;
               is
               begin
                  if not Ck_Failed then
                     -- Max_Binary_Modulus must be a positive power of 2
                     if Is_Max_Binary_Modulus then

                        if not Maths.IsAPositivePowerOf2 (The_Decl.Value) then
                           Failure_Reason := Not_Positive_Power_Of_2;
                        end if;

                     elsif Is_Default_Bit_Order then

                        if LexTokenManager.Lex_String_Case_Insensitive_Compare
                          (Lex_Str1 => The_Decl.Base_Type.Token_Str,
                           Lex_Str2 => LexTokenManager.Bit_Order_Token) /=
                          LexTokenManager.Str_Eq then
                           Failure_Reason := Bit_Order_Wrong_Type;
                        else
                           Casing.Check_Casing
                             (Lex_Str1 => LexTokenManager.Bit_Order_Token,
                              Lex_Str2 => The_Decl.Base_Type.Token_Str,
                              Position => The_Decl.Base_Type.Position);
                           if LexTokenManager.Lex_String_Case_Insensitive_Compare
                             (Lex_Str1 => The_Decl.Value_Str.Token_Str,
                              Lex_Str2 => LexTokenManager.Low_Order_First_Token) /=
                             LexTokenManager.Str_Eq
                             and then LexTokenManager.Lex_String_Case_Insensitive_Compare
                             (Lex_Str1 => The_Decl.Value_Str.Token_Str,
                              Lex_Str2 => LexTokenManager.High_Order_First_Token) /=
                             LexTokenManager.Str_Eq then
                              Failure_Reason := Bit_Order_Wrong_Value;
                           else
                              if LexTokenManager.Lex_String_Case_Insensitive_Compare
                                (Lex_Str1 => The_Decl.Value_Str.Token_Str,
                                 Lex_Str2 => LexTokenManager.Low_Order_First_Token) =
                                LexTokenManager.Str_Eq then
                                 Casing.Check_Casing
                                   (Lex_Str1 => LexTokenManager.Low_Order_First_Token,
                                    Lex_Str2 => The_Decl.Value_Str.Token_Str,
                                    Position => The_Decl.Value_Str.Position);
                              else
                                 Casing.Check_Casing
                                   (Lex_Str1 => LexTokenManager.High_Order_First_Token,
                                    Lex_Str2 => The_Decl.Value_Str.Token_Str,
                                    Position => The_Decl.Value_Str.Position);
                              end if;
                           end if;
                        end if;
                     end if;
                  end if;
               end Do_Ck_Special_Cases;

               procedure Raise_Errors_And_Update_Dictionary
               --# global in     CommandLineData.Content;
               --#        in     Current_Scope;
               --#        in     Failure_Reason;
               --#        in     Is_Address;
               --#        in     Is_Default_Bit_Order;
               --#        in     Is_Interrupt_ID;
               --#        in     Is_Predef_Float;
               --#        in     Is_Predef_Integer;
               --#        in     Is_Priority;
               --#        in     Is_Seconds_Count;
               --#        in     Parent_Type;
               --#        in     The_Decl;
               --#        in out Dictionary.Dict;
               --#        in out ErrorHandler.Error_Context;
               --#        in out Errors_Occurred;
               --#        in out LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives Dictionary.Dict            from *,
               --#                                         CommandLineData.Content,
               --#                                         Current_Scope,
               --#                                         Failure_Reason,
               --#                                         Is_Address,
               --#                                         Is_Default_Bit_Order,
               --#                                         Is_Interrupt_ID,
               --#                                         Is_Predef_Float,
               --#                                         Is_Predef_Integer,
               --#                                         Is_Priority,
               --#                                         Is_Seconds_Count,
               --#                                         LexTokenManager.State,
               --#                                         Parent_Type,
               --#                                         The_Decl &
               --#         ErrorHandler.Error_Context from *,
               --#                                         CommandLineData.Content,
               --#                                         Current_Scope,
               --#                                         Dictionary.Dict,
               --#                                         Failure_Reason,
               --#                                         Is_Address,
               --#                                         Is_Priority,
               --#                                         LexTokenManager.State,
               --#                                         Parent_Type,
               --#                                         SPARK_IO.File_Sys,
               --#                                         The_Decl &
               --#         Errors_Occurred            from *,
               --#                                         Failure_Reason &
               --#         LexTokenManager.State      from *,
               --#                                         Failure_Reason,
               --#                                         Is_Interrupt_ID,
               --#                                         Is_Predef_Integer,
               --#                                         Is_Priority,
               --#                                         Is_Seconds_Count,
               --#                                         The_Decl &
               --#         SPARK_IO.File_Sys          from *,
               --#                                         CommandLineData.Content,
               --#                                         Current_Scope,
               --#                                         Dictionary.Dict,
               --#                                         ErrorHandler.Error_Context,
               --#                                         Failure_Reason,
               --#                                         Is_Address,
               --#                                         Is_Interrupt_ID,
               --#                                         Is_Predef_Float,
               --#                                         Is_Predef_Integer,
               --#                                         Is_Priority,
               --#                                         Is_Seconds_Count,
               --#                                         LexTokenManager.State,
               --#                                         Parent_Type,
               --#                                         The_Decl;
               is
                  Storage_Rep_Value      : LexTokenManager.Lex_String;
                  Storage_Rep_Low_Bound  : LexTokenManager.Lex_String;
                  Storage_Rep_High_Bound : LexTokenManager.Lex_String;
                  Storage_Rep_Digits     : LexTokenManager.Lex_String;
                  Integer_Type           : Dictionary.Symbol;
                  Float_Type             : Dictionary.Symbol;
                  Private_Type           : Dictionary.Symbol;
                  Lib_Package_Sym        : Dictionary.Symbol;
                  Constant_Sym           : Dictionary.Symbol;

                  procedure Add_Null_Address
                  --# global in     CommandLineData.Content;
                  --#        in     Current_Scope;
                  --#        in     LexTokenManager.State;
                  --#        in     The_Decl;
                  --#        in out Dictionary.Dict;
                  --#        in out ErrorHandler.Error_Context;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives Dictionary.Dict            from *,
                  --#                                         CommandLineData.Content,
                  --#                                         Current_Scope,
                  --#                                         LexTokenManager.State,
                  --#                                         The_Decl &
                  --#         ErrorHandler.Error_Context,
                  --#         SPARK_IO.File_Sys          from CommandLineData.Content,
                  --#                                         Current_Scope,
                  --#                                         Dictionary.Dict,
                  --#                                         ErrorHandler.Error_Context,
                  --#                                         LexTokenManager.State,
                  --#                                         SPARK_IO.File_Sys,
                  --#                                         The_Decl;
                  is
                     The_Constant : Dictionary.Symbol;
                     pragma Unreferenced (The_Constant);
                  begin
                     --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
                     Dictionary.Add_Deferred_Constant
                       (Name           => LexTokenManager.Null_Address_Token,
                        Type_Mark      => Dictionary.LookupItem
                          (Name              => The_Decl.Name.Token_Str,
                           Scope             => Current_Scope,
                           Context           => Dictionary.ProgramContext,
                           Full_Package_Name => False),
                        Type_Reference => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                               End_Position   => The_Decl.Name.Position),
                        Comp_Unit      => ContextManager.NullUnit,
                        Declaration    => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                               End_Position   => The_Decl.Name.Position),
                        The_Package    => Dictionary.GetRegion (Current_Scope),
                        TheConstant    => The_Constant);
                     --# end accept;
                     -- note to indicate implicit definition
                     ErrorHandler.Semantic_Note
                       (Err_Num  => 3,
                        Position => The_Decl.Name.Position,
                        Id_Str   => The_Decl.Name.Token_Str);
                     --# accept Flow, 33, The_Constant, "Expected to be neither referenced nor exported";
                  end Add_Null_Address;

                  procedure Add_Default_Priority
                  --# global in     CommandLineData.Content;
                  --#        in     Current_Scope;
                  --#        in     The_Decl;
                  --#        in out Dictionary.Dict;
                  --#        in out ErrorHandler.Error_Context;
                  --#        in out LexTokenManager.State;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives Dictionary.Dict            from *,
                  --#                                         CommandLineData.Content,
                  --#                                         Current_Scope,
                  --#                                         LexTokenManager.State,
                  --#                                         The_Decl &
                  --#         ErrorHandler.Error_Context,
                  --#         SPARK_IO.File_Sys          from CommandLineData.Content,
                  --#                                         Current_Scope,
                  --#                                         Dictionary.Dict,
                  --#                                         ErrorHandler.Error_Context,
                  --#                                         LexTokenManager.State,
                  --#                                         SPARK_IO.File_Sys,
                  --#                                         The_Decl &
                  --#         LexTokenManager.State      from *,
                  --#                                         The_Decl;
                  is
                     Temp_Val1, Temp_Val2 : Maths.Value;
                     Unwanted_ME          : Maths.ErrorCode;
                     Storage_Rep_Ave      : LexTokenManager.Lex_String;
                     The_Constant         : Dictionary.Symbol;
                     pragma Unreferenced (The_Constant);
                  begin
                     --# accept Flow, 10, Unwanted_ME, "Unwanted_ME unused here";
                     Maths.Add (The_Decl.Low_Bound, The_Decl.High_Bound, Temp_Val1, Unwanted_ME);
                     Maths.Divide (Temp_Val1, Maths.IntegerToValue (2), Temp_Val2, Unwanted_ME);
                     --# end accept;
                     Maths.StorageRep (Temp_Val2, Storage_Rep_Ave);
                     --# accept Flow, 10, The_Constant, "Expected ineffective assignment to OK";
                     Dictionary.Add_Constant_Declaration
                       (Name              => LexTokenManager.Default_Priority_Token,
                        Type_Mark         => Dictionary.LookupItem
                          (Name              => The_Decl.Name.Token_Str,
                           Scope             => Current_Scope,
                           Context           => Dictionary.ProgramContext,
                           Full_Package_Name => False),
                        Type_Reference    => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                  End_Position   => The_Decl.Name.Position),
                        Value             => Storage_Rep_Ave,
                        Exp_Is_Wellformed => True,
                        Exp_Node          => ExaminerConstants.RefType (0),
                        Static            => True,
                        Comp_Unit         => ContextManager.NullUnit,
                        Declaration       => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                  End_Position   => The_Decl.Name.Position),
                        Scope             => Current_Scope,
                        Context           => Dictionary.ProgramContext,
                        TheConstant       => The_Constant);
                     --# end accept;
                     -- note to indicate implicit definition
                     ErrorHandler.Semantic_Note
                       (Err_Num  => 4,
                        Position => The_Decl.Name.Position,
                        Id_Str   => The_Decl.Name.Token_Str);
                     --# accept Flow, 33, Unwanted_ME, "Unwanted_ME unused here" &
                     --#        Flow, 33, The_Constant, "Expected to be neither referenced nor exported";
                  end Add_Default_Priority;

               begin -- Raise_Errors_And_Update_Dictionary
                  case Failure_Reason is
                     when Invalid_Redeclaration =>
                        Raise_Semantic_Error (Err_Num => 10,
                                              Lex_Val => The_Decl.Name);
                     when Wrong_Type_Found =>
                        Raise_Semantic_Error (Err_Num => 42,
                                              Lex_Val => The_Decl.Name);
                     when Empty_Range =>
                        Raise_Semantic_Error (Err_Num => 409,
                                              Lex_Val => The_Decl.Name);
                     when Subtype_Range_Mismatch =>
                        Raise_Semantic_Error (Err_Num => 413,
                                              Lex_Val => The_Decl.Name);
                     when Not_Valid_Name | Not_Within_Valid_Package =>
                        Raise_Semantic_Error (Err_Num => 778,
                                              Lex_Val => The_Decl.Name);
                     when Parent_Type_Does_Not_Exist =>
                        Raise_Semantic_Error (Err_Num => 63,
                                              Lex_Val => The_Decl.Base_Type);
                     when Parent_Type_Not_AP =>
                        Raise_Semantic_Error (Err_Num => 771,
                                              Lex_Val => The_Decl.Base_Type);
                     when Priority_Range_Insufficient =>
                        Raise_Semantic_Error (Err_Num => 772,
                                              Lex_Val => The_Decl.Base_Type);
                     when Any_Prio_First =>
                        Raise_Semantic_Error (Err_Num => 773,
                                              Lex_Val => The_Decl.Base_Type);
                     when Any_Prio_Last =>
                        Raise_Semantic_Error (Err_Num => 775,
                                              Lex_Val => The_Decl.Base_Type);
                     when Prio_Mid_Point =>
                        Raise_Semantic_Error (Err_Num => 774,
                                              Lex_Val => The_Decl.Base_Type);
                     when Parent_Type_Not_Integer =>
                        Raise_Semantic_Error (Err_Num => 777,
                                              Lex_Val => The_Decl.Base_Type);
                     when Must_Be_Private =>
                        Raise_Semantic_Error (Err_Num => 780,
                                              Lex_Val => The_Decl.Name);
                     when Not_Positive_Power_Of_2 =>
                        Raise_Semantic_Error (Err_Num => 784,
                                              Lex_Val => The_Decl.Name);
                     when Bit_Order_Wrong_Type =>
                        Raise_Semantic_Error (Err_Num => 814,
                                              Lex_Val => The_Decl.Base_Type);
                     when Bit_Order_Wrong_Value =>
                        Raise_Semantic_Error (Err_Num => 815,
                                              Lex_Val => The_Decl.Value_Str);
                     when No_Failure =>
                        case The_Decl.My_Type is
                           when Dec_Named_Integer =>
                              Maths.StorageRep (The_Decl.Value, Storage_Rep_Value);
                              --# accept Flow, 10, Constant_Sym, "Expected ineffective assignment to OK";
                              Dictionary.Add_Constant_Declaration
                                (Name              => The_Decl.Name.Token_Str,
                                 Type_Mark         => Dictionary.GetUniversalIntegerType,
                                 Type_Reference    => Dictionary.Null_Location,
                                 Value             => Storage_Rep_Value,
                                 Exp_Is_Wellformed => True,
                                 Exp_Node          => ExaminerConstants.RefType (0),
                                 Static            => True,
                                 Comp_Unit         => ContextManager.NullUnit,
                                 Declaration       => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                           End_Position   => The_Decl.Name.Position),
                                 Scope             => Current_Scope,
                                 Context           => Dictionary.ProgramContext,
                                 TheConstant       => Constant_Sym);
                              --# end accept;
                           when Dec_Named_Real =>
                              Maths.StorageRep (The_Decl.Value, Storage_Rep_Value);
                              --# accept Flow, 10, Constant_Sym, "Expected ineffective assignment to OK";
                              Dictionary.Add_Constant_Declaration
                                (Name              => The_Decl.Name.Token_Str,
                                 Type_Mark         => Dictionary.GetUniversalRealType,
                                 Type_Reference    => Dictionary.Null_Location,
                                 Value             => Storage_Rep_Value,
                                 Exp_Is_Wellformed => True,
                                 Exp_Node          => ExaminerConstants.RefType (0),
                                 Static            => True,
                                 Comp_Unit         => ContextManager.NullUnit,
                                 Declaration       => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                           End_Position   => The_Decl.Name.Position),
                                 Scope             => Current_Scope,
                                 Context           => Dictionary.ProgramContext,
                                 TheConstant       => Constant_Sym);
                              --# end accept;
                           when Dec_Integer_Subtype =>
                              Maths.StorageRep (The_Decl.Low_Bound, Storage_Rep_Low_Bound);
                              Maths.StorageRep (The_Decl.High_Bound, Storage_Rep_High_Bound);
                              --# accept F, 10, Integer_Type, "Integer_Type unused here";
                              Dictionary.Add_Integer_Subtype
                                (Name             => The_Decl.Name.Token_Str,
                                 Static           => True,
                                 Parent           => Parent_Type,
                                 Parent_Reference => Dictionary.Null_Location,
                                 Lower            => Storage_Rep_Low_Bound,
                                 Upper            => Storage_Rep_High_Bound,
                                 Comp_Unit        => ContextManager.NullUnit,
                                 Declaration      => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                          End_Position   => The_Decl.Name.Position),
                                 Scope            => Current_Scope,
                                 Context          => Dictionary.ProgramContext,
                                 The_Subtype      => Integer_Type);
                              --# end accept;
                              if Is_Priority then
                                 Add_Default_Priority;
                              end if;
                           when Dec_Integer_Type =>
                              Maths.StorageRep (The_Decl.Low_Bound, Storage_Rep_Low_Bound);
                              Maths.StorageRep (The_Decl.High_Bound, Storage_Rep_High_Bound);

                              if Is_Interrupt_ID or else Is_Seconds_Count then
                                 -- An integer type which is pre-declared by
                                 -- Dictionary.Initialize, and has no dependent
                                 -- subtypes, so simply override the lower and upper bounds
                                 Integer_Type :=
                                   Dictionary.LookupItem
                                   (Name              => The_Decl.Name.Token_Str,
                                    Scope             => Current_Scope,
                                    Context           => Dictionary.ProgramContext,
                                    Full_Package_Name => False);
                                 Dictionary.AdjustTypeLowerBound (TypeMark => Integer_Type,
                                                                  NewBound => Storage_Rep_Low_Bound);
                                 Dictionary.AdjustTypeUpperBound (TypeMark => Integer_Type,
                                                                  NewBound => Storage_Rep_High_Bound);
                              elsif Is_Predef_Integer then
                                 -- Update the bounds for types Standard.Integer,
                                 -- Standard.Natural and Standard.Positive in the dictionary
                                 Integer_Type :=
                                   Dictionary.LookupItem
                                   (Name              => The_Decl.Name.Token_Str,
                                    Scope             => Current_Scope,
                                    Context           => Dictionary.ProgramContext,
                                    Full_Package_Name => False);
                                 Dictionary.AdjustTypeLowerBound (TypeMark => Integer_Type,
                                                                  NewBound => Storage_Rep_Low_Bound);
                                 Dictionary.AdjustTypeUpperBound (TypeMark => Integer_Type,
                                                                  NewBound => Storage_Rep_High_Bound);
                                 Maths.StorageRep (Maths.ZeroInteger, Storage_Rep_Low_Bound);
                                 Dictionary.AdjustTypeLowerBound (Dictionary.GetPredefinedNaturalSubtype, Storage_Rep_Low_Bound);
                                 Dictionary.AdjustTypeUpperBound (Dictionary.GetPredefinedNaturalSubtype, Storage_Rep_High_Bound);
                                 Maths.StorageRep (Maths.OneInteger, Storage_Rep_Low_Bound);
                                 Dictionary.AdjustTypeLowerBound (Dictionary.GetPredefinedPositiveSubtype, Storage_Rep_Low_Bound);
                                 Dictionary.AdjustTypeUpperBound (Dictionary.GetPredefinedPositiveSubtype, Storage_Rep_High_Bound);
                              else
                                 -- A new predefined integer type which _isn't_
                                 -- pre-declared by Dictionary.Initialize.
                                 -- For example: Short_Integer or Long_Long_Integer
                                 Dictionary.Add_Predef_Integer_Type
                                   (Name        => The_Decl.Name.Token_Str,
                                    Comp_Unit   => ContextManager.NullUnit,
                                    Declaration => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                        End_Position   => The_Decl.Name.Position),
                                    Lower       => Storage_Rep_Low_Bound,
                                    Upper       => Storage_Rep_High_Bound,
                                    Scope       => Current_Scope,
                                    Context     => Dictionary.ProgramContext);
                              end if;
                           when Dec_Floating_Point_Type =>
                              Maths.StorageRep (The_Decl.Low_Bound, Storage_Rep_Low_Bound);
                              Maths.StorageRep (The_Decl.High_Bound, Storage_Rep_High_Bound);
                              Maths.StorageRep (The_Decl.Num_Digits, Storage_Rep_Digits);
                              if Is_Predef_Float then
                                 -- A Floating-point type which is pre-declared by
                                 -- Dictionary.Initialize, and has no dependent
                                 -- subtypes, so simply override the accuracy,
                                 -- lower and upper bounds
                                 Float_Type :=
                                   Dictionary.LookupItem
                                   (Name              => The_Decl.Name.Token_Str,
                                    Scope             => Current_Scope,
                                    Context           => Dictionary.ProgramContext,
                                    Full_Package_Name => False);
                                 Dictionary.AdjustTypeLowerBound (TypeMark => Float_Type,
                                                                  NewBound => Storage_Rep_Low_Bound);
                                 Dictionary.AdjustTypeUpperBound (TypeMark => Float_Type,
                                                                  NewBound => Storage_Rep_High_Bound);
                                 Dictionary.AdjustTypeErrorBound (TypeMark => Float_Type,
                                                                  NewBound => Storage_Rep_Digits);
                              else
                                 -- A Floating-point type, which is _not_ pre-declared
                                 -- by Dictionary.Initialize - e.g. Short_Float or
                                 -- Long_Long_Float
                                 Dictionary.Add_Predef_Floating_Point_Type
                                   (Name        => The_Decl.Name.Token_Str,
                                    Comp_Unit   => ContextManager.NullUnit,
                                    Declaration => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                        End_Position   => The_Decl.Name.Position),
                                    Lower       => Storage_Rep_Low_Bound,
                                    Upper       => Storage_Rep_High_Bound,
                                    Error_Bound => Storage_Rep_Digits,
                                    Scope       => Current_Scope,
                                    Context     => Dictionary.ProgramContext);
                              end if;
                           when Dec_Private =>
                              --# accept Flow, 10, Private_Type, "Expected ineffective assignment to OK";
                              Dictionary.Add_Private_Type
                                (Name           => The_Decl.Name.Token_Str,
                                 Comp_Unit      => ContextManager.NullUnit,
                                 Declaration    => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                        End_Position   => The_Decl.Name.Position),
                                 The_Package    => Dictionary.GetRegion (Current_Scope),
                                 Is_Limited     => False,
                                 Is_Tagged_Type => False,
                                 Extends        => Dictionary.NullSymbol,
                                 The_Type       => Private_Type);
                              --# end accept;
                              if Is_Address then
                                 Add_Null_Address;
                              end if;
                           when Dec_Typed_Constant =>
                              if Is_Default_Bit_Order then
                                 Lib_Package_Sym :=
                                   Dictionary.LookupItem
                                   (Name              => LexTokenManager.System_Token,
                                    Scope             => Dictionary.GlobalScope,
                                    Context           => Dictionary.ProgramContext,
                                    Full_Package_Name => False);
                                 Constant_Sym    :=
                                   Dictionary.LookupItem
                                   (Name              => The_Decl.Name.Token_Str,
                                    Scope             => Current_Scope,
                                    Context           => Dictionary.ProgramContext,
                                    Full_Package_Name => False);

                                 -- In the Dictionary, we need to store the 'Pos of the
                                 -- value, so...
                                 if LexTokenManager.Lex_String_Case_Insensitive_Compare
                                   (Lex_Str1 => The_Decl.Value_Str.Token_Str,
                                    Lex_Str2 => LexTokenManager.High_Order_First_Token) =
                                   LexTokenManager.Str_Eq then
                                    Storage_Rep_Value := LexTokenManager.Zero_Value;
                                 else
                                    Storage_Rep_Value := LexTokenManager.One_Value;
                                 end if;
                                 Dictionary.Promote_Deferred_To_Full_Constant
                                   (Constant_Sym => Constant_Sym,
                                    Comp_Unit    => ContextManager.NullUnit,
                                    Declaration  => Dictionary.Location'(Start_Position => The_Decl.Name.Position,
                                                                         End_Position   => The_Decl.Name.Position),
                                    Value        => Storage_Rep_Value,
                                    Exp_Node     => ExaminerConstants.RefType (0),
                                    The_Package  => Lib_Package_Sym);
                              end if;
                        end case;
                  end case;
                  --# accept Flow, 33, Private_Type, "Expected Success to be neither referenced nor exported";
               end Raise_Errors_And_Update_Dictionary;

               procedure Update_Priority_Symbols
               --# global in     CommandLineData.Content;
               --#        in     Current_Scope;
               --#        in     Dictionary.Dict;
               --#        in     Failure_Reason;
               --#        in     Is_Any_Priority;
               --#        in     Is_Interrupt_Priority;
               --#        in     Is_Priority;
               --#        in     LexTokenManager.State;
               --#        in     The_Decl;
               --#        in out Any_Priority_Sym;
               --#        in out Interrupt_Priority_Sym;
               --#        in out Priority_Sym;
               --# derives Any_Priority_Sym       from *,
               --#                                     CommandLineData.Content,
               --#                                     Current_Scope,
               --#                                     Dictionary.Dict,
               --#                                     Failure_Reason,
               --#                                     Is_Any_Priority,
               --#                                     LexTokenManager.State,
               --#                                     The_Decl &
               --#         Interrupt_Priority_Sym from *,
               --#                                     CommandLineData.Content,
               --#                                     Current_Scope,
               --#                                     Dictionary.Dict,
               --#                                     Failure_Reason,
               --#                                     Is_Any_Priority,
               --#                                     Is_Interrupt_Priority,
               --#                                     Is_Priority,
               --#                                     LexTokenManager.State,
               --#                                     The_Decl &
               --#         Priority_Sym           from *,
               --#                                     CommandLineData.Content,
               --#                                     Current_Scope,
               --#                                     Dictionary.Dict,
               --#                                     Failure_Reason,
               --#                                     Is_Any_Priority,
               --#                                     Is_Priority,
               --#                                     LexTokenManager.State,
               --#                                     The_Decl;
               is
                  Temp_Sym : Dictionary.Symbol;
               begin
                  if not Ck_Failed then
                     Temp_Sym :=
                       Dictionary.LookupItem
                       (Name              => The_Decl.Name.Token_Str,
                        Scope             => Current_Scope,
                        Context           => Dictionary.ProgramContext,
                        Full_Package_Name => False);
                     if Is_Any_Priority then
                        Any_Priority_Sym := Temp_Sym;
                     elsif Is_Priority then
                        Priority_Sym := Temp_Sym;
                     elsif Is_Interrupt_Priority then
                        Interrupt_Priority_Sym := Temp_Sym;
                     end if;
                  end if;
               end Update_Priority_Symbols;

            begin -- Check_And_Process_Declaration

               -- get full name of declared item
               Dotted_Name := Package_Full_Name;
               E_Strings.Append_String (E_Str => Dotted_Name,
                                        Str   => ".");
               E_Strings.Append_Examiner_String
                 (E_Str1 => Dotted_Name,
                  E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => The_Decl.Name.Token_Str));
               Parent_Type := Dictionary.NullSymbol;
               Match_Type  := Declaration_Type'Last;

               -- perform whichever checks are appropriate
               case The_Decl.My_Type is
                  when Dec_Named_Integer =>
                     Do_Ck_Valid_Package;
                     Do_Ck_Def_Name;
                     Do_Ck_Existing_Decl;
                     Do_Ck_Type;
                     Do_Ck_Special_Cases;
                     Raise_Errors_And_Update_Dictionary;

                  when Dec_Named_Real =>
                     Do_Ck_Valid_Package;
                     Do_Ck_Def_Name;
                     Do_Ck_Existing_Decl;
                     Do_Ck_Type;
                     Do_Ck_Special_Cases;
                     Raise_Errors_And_Update_Dictionary;

                  when Dec_Integer_Subtype =>
                     Do_Ck_Valid_Package;
                     Do_Ck_Def_Name_With_Priority_Ck;
                     Do_Ck_Existing_Decl;
                     Do_Ck_Type;
                     Do_Ck_Bounds;
                     Do_Ck_Base_Type;
                     Do_Ck_System_Priority;
                     Do_Ck_Special_Cases;
                     Raise_Errors_And_Update_Dictionary;
                     Update_Priority_Symbols;

                  when Dec_Integer_Type =>
                     Do_Ck_Valid_Package;
                     Do_Ck_Suffix_Name;
                     Do_Ck_Def_Name;
                     Do_Ck_Existing_Decl;
                     Do_Ck_Type;
                     Do_Ck_Bounds;
                     Do_Ck_Special_Cases;
                     Raise_Errors_And_Update_Dictionary;

                  when Dec_Floating_Point_Type =>
                     Do_Ck_Valid_Package;
                     Do_Ck_Suffix_Name;
                     Do_Ck_Def_Name;
                     Do_Ck_Existing_Decl;
                     Do_Ck_Type;
                     Do_Ck_Bounds;
                     Do_Ck_Special_Cases;
                     Raise_Errors_And_Update_Dictionary;

                  when Dec_Private =>
                     Do_Ck_Valid_Package;
                     Do_Ck_Def_Name;
                     Do_Ck_Existing_Decl;
                     Do_Ck_Type;
                     Do_Ck_Special_Cases;
                     Raise_Errors_And_Update_Dictionary;

                  when Dec_Typed_Constant =>
                     Do_Ck_Valid_Package;
                     Do_Ck_Def_Name;
                     Do_Ck_Existing_Decl;
                     Do_Ck_Type;
                     Do_Ck_Special_Cases;
                     Raise_Errors_And_Update_Dictionary;
               end case;
            end Check_And_Process_Declaration;

            procedure Assert_Token (The_Token : in SP_Symbols.SP_Terminal;
                                    Should_Be : in SP_Symbols.SP_Terminal)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     The_Lex_Val;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out LexTokenManager.State;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --# derives ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         Should_Be,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Errors_Occurred,
            --#         Parser_Stage,
            --#         Parser_State               from *,
            --#                                         Should_Be,
            --#                                         The_Token &
            --#         LexTokenManager.State      from *,
            --#                                         Should_Be,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Next_Expected_State        from *,
            --#                                         Parser_State,
            --#                                         Should_Be,
            --#                                         The_Token;
            is
            begin
               if The_Token /= Should_Be then
                  Raise_Syntax_Error
                    (The_Error => Syntax_Errors'(The_Type        => Exp_Symbol,
                                                 The_Symbol      => The_Token,
                                                 Expected_Symbol => Should_Be,
                                                 The_Lex_Val     => The_Lex_Val));
               end if;
            end Assert_Token;

            procedure Begin_Int_Expr_Parse
            --# global in     Unary_Minus;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#           out Int_Add;
            --#           out Int_Exponent;
            --#           out Int_Subtract;
            --#           out Saved_Parser_Stage;
            --#           out Saved_Parser_State;
            --# derives Int_Add,
            --#         Int_Exponent,
            --#         Int_Subtract,
            --#         Parser_State       from  &
            --#         Parser_Stage       from Unary_Minus &
            --#         Saved_Parser_Stage from Parser_Stage &
            --#         Saved_Parser_State from Parser_State;
            is
            begin
               Saved_Parser_State := Parser_State;
               Saved_Parser_Stage := Parser_Stage;
               New_State (The_Stage => Expr_Int);
               if Unary_Minus then
                  -- cannot accept initial '-' if already had one
                  Next_Stage;
               end if;
               Int_Subtract := False;
               Int_Add      := False;
               Int_Exponent := False;
            end Begin_Int_Expr_Parse;

            procedure Do_Config_Defn_Start
            --# global in     CommandLineData.Content;
            --#        in     Possible_Child_Packages;
            --#        in     Possible_Library_Packages;
            --#        in     Predefined_Scope;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in out Current_Declaration;
            --#        in out Current_Scope;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out LexTokenManager.State;
            --#        in out Lib_Package_Symbol;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Package_Full_Name;
            --#        in out Package_Name_Depth;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Valid_Package;
            --# derives Current_Declaration        from *,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val &
            --#         Current_Scope              from *,
            --#                                         CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         LexTokenManager.State,
            --#                                         Lib_Package_Symbol,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Child_Packages,
            --#                                         Possible_Library_Packages,
            --#                                         Predefined_Scope,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Dictionary.Dict,
            --#         Lib_Package_Symbol         from *,
            --#                                         CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         LexTokenManager.State,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Library_Packages,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         Lib_Package_Symbol,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Child_Packages,
            --#                                         Possible_Library_Packages,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Errors_Occurred,
            --#         Valid_Package              from *,
            --#                                         CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         LexTokenManager.State,
            --#                                         Lib_Package_Symbol,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Child_Packages,
            --#                                         Possible_Library_Packages,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         LexTokenManager.State      from *,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Lookahead                  from *,
            --#                                         Parser_Stage &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Package_Full_Name          from *,
            --#                                         Current_Declaration,
            --#                                         LexTokenManager.State,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Token &
            --#         Package_Name_Depth,
            --#         Parser_Stage               from Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Token;
            is

               procedure Change_To_New_Package (Package_Name : in LexTokenManager.Lex_Value)
               --# global in     CommandLineData.Content;
               --#        in     Current_Declaration;
               --#        in     LexTokenManager.State;
               --#        in     Package_Name_Depth;
               --#        in     Possible_Child_Packages;
               --#        in     Possible_Library_Packages;
               --#        in     Predefined_Scope;
               --#        in out Current_Scope;
               --#        in out Dictionary.Dict;
               --#        in out ErrorHandler.Error_Context;
               --#        in out Errors_Occurred;
               --#        in out Lib_Package_Symbol;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Valid_Package;
               --# derives Current_Scope              from *,
               --#                                         CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         LexTokenManager.State,
               --#                                         Lib_Package_Symbol,
               --#                                         Package_Name,
               --#                                         Package_Name_Depth,
               --#                                         Possible_Child_Packages,
               --#                                         Possible_Library_Packages,
               --#                                         Predefined_Scope &
               --#         Dictionary.Dict,
               --#         Lib_Package_Symbol         from *,
               --#                                         CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         LexTokenManager.State,
               --#                                         Package_Name,
               --#                                         Package_Name_Depth,
               --#                                         Possible_Library_Packages &
               --#         ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from CommandLineData.Content,
               --#                                         Current_Declaration,
               --#                                         Dictionary.Dict,
               --#                                         ErrorHandler.Error_Context,
               --#                                         LexTokenManager.State,
               --#                                         Lib_Package_Symbol,
               --#                                         Package_Name,
               --#                                         Package_Name_Depth,
               --#                                         Possible_Child_Packages,
               --#                                         Possible_Library_Packages,
               --#                                         SPARK_IO.File_Sys,
               --#                                         Valid_Package &
               --#         Errors_Occurred,
               --#         Valid_Package              from *,
               --#                                         CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         LexTokenManager.State,
               --#                                         Lib_Package_Symbol,
               --#                                         Package_Name,
               --#                                         Package_Name_Depth,
               --#                                         Possible_Child_Packages,
               --#                                         Possible_Library_Packages,
               --#                                         Valid_Package;
               is
                  Package_Standard   : constant Library_Package_Range := 1;
                  Package_Ada        : constant Library_Package_Range := 3;
                  Package_Real_Time  : constant Child_Package_Range   := 1;
                  Package_Interrupts : constant Child_Package_Range   := 2;

                  Name_Match            : Boolean := False;
                  Local_Package_Symbol  : Dictionary.Symbol;
                  Ada_Package_Symbol    : Dictionary.Symbol;
                  Bit_Order_Type_Symbol : Dictionary.Symbol;
                  Unused_Symbol         : Dictionary.Symbol;
                  Package_Name_Match    : Boolean;

                  procedure Match_Identifier
                    (ID       : in     LexTokenManager.Lex_String;
                     To_Match : in     LexTokenManager.Lex_String;
                     Position : in     LexTokenManager.Token_Position;
                     Matched  :    out Boolean)
                  --# global in     CommandLineData.Content;
                  --#        in     Dictionary.Dict;
                  --#        in     LexTokenManager.State;
                  --#        in out ErrorHandler.Error_Context;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives ErrorHandler.Error_Context,
                  --#         SPARK_IO.File_Sys          from CommandLineData.Content,
                  --#                                         Dictionary.Dict,
                  --#                                         ErrorHandler.Error_Context,
                  --#                                         ID,
                  --#                                         LexTokenManager.State,
                  --#                                         Position,
                  --#                                         SPARK_IO.File_Sys,
                  --#                                         To_Match &
                  --#         Matched                    from ID,
                  --#                                         LexTokenManager.State,
                  --#                                         To_Match;
                  is
                  begin
                     Matched := LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => ID,
                                                                                     Lex_Str2 => To_Match) =
                       LexTokenManager.Str_Eq;
                     if Matched then
                        Casing.Check_Casing (Lex_Str1 => ID,
                                             Lex_Str2 => To_Match,
                                             Position => Position);
                     end if;
                  end Match_Identifier;

               begin -- Change_To_New_Package

                  -- Get the symbol for package Ada now since we need it more than once later.
                  -- For SPARK83 mode, this will return NullSymbol, but that doesn't matter
                  -- since we only use the symbol in code specific to SPARK95 onwards.
                  Ada_Package_Symbol :=
                    Dictionary.LookupItem
                    (Name              => LexTokenManager.Ada_Token,
                     Scope             => Dictionary.GlobalScope,
                     Context           => Dictionary.ProofContext,
                     Full_Package_Name => False);
                  case Package_Name_Depth is
                     when 1 => -- Checking Library packages
                        for I in Library_Package_Range loop
                           Match_Identifier
                             (ID       => Possible_Library_Packages (I),
                              To_Match => Package_Name.Token_Str,
                              Position => Package_Name.Position,
                              Matched  => Package_Name_Match);
                           if Package_Name_Match then
                              Name_Match := True;
                              case I is
                                 when Package_Standard =>
                                    Current_Scope      := Predefined_Scope;
                                    Lib_Package_Symbol := Dictionary.GetPredefinedPackageStandard;
                                    Valid_Package      := True;
                                 when Package_System =>
                                    --# accept F, 41, "Stable expression expected here";
                                    if CommandLineData.Content.Language_Profile = CommandLineData.SPARK83 then
                                       -- package System disallowed in Spark83 mode
                                       -- don't complain about the other declarations here;
                                       -- wood for trees etc.
                                       Raise_Semantic_Error (Err_Num => 779,
                                                             Lex_Val => Package_Name);
                                    end if;
                                    --# end accept;
                                    Lib_Package_Symbol :=
                                      Dictionary.LookupItem
                                      (Name              => Package_Name.Token_Str,
                                       Scope             => Dictionary.GlobalScope,
                                       Context           => Dictionary.ProofContext,
                                       Full_Package_Name => False);
                                    if not Dictionary.Is_Null_Symbol (Lib_Package_Symbol) then
                                       -- redeclaration of existing package
                                       Valid_Package := False;
                                       Raise_Semantic_Error (Err_Num => 10,
                                                             Lex_Val => Package_Name);
                                    else
                                       -- create package
                                       Dictionary.Add_Package
                                         (Name          => Package_Name.Token_Str,
                                          Comp_Unit     => ContextManager.NullUnit,
                                          Specification => Dictionary.Location'(Start_Position => Package_Name.Position,
                                                                                End_Position   => Package_Name.Position),
                                          Scope         => Dictionary.GlobalScope,
                                          ThePackage    => Lib_Package_Symbol);
                                       Current_Scope :=
                                         Dictionary.Set_Visibility
                                         (The_Visibility => Dictionary.Visible,
                                          The_Unit       => Lib_Package_Symbol);

                                       -- Add Bit_Order, its enumeration literals, and
                                       -- System.Default_Bit_Order as a deferred constant
                                       -- (for now...the actual value might be supplied and
                                       --  overridden in the config file).
                                       Dictionary.Add_Enumeration_Type
                                         (Name        => LexTokenManager.Bit_Order_Token,
                                          Comp_Unit   => ContextManager.NullUnit,
                                          Declaration => Dictionary.Location'(Start_Position => Package_Name.Position,
                                                                              End_Position   => Package_Name.Position),
                                          Scope       => Current_Scope,
                                          Context     => Dictionary.ProgramContext,
                                          The_Type    => Bit_Order_Type_Symbol);
                                       --# accept Flow, 10, Unused_Symbol, "Expected ineffective assignment to OK";
                                       Dictionary.AddEnumerationLiteral
                                         (Name                  => LexTokenManager.High_Order_First_Token,
                                          Comp_Unit             => ContextManager.NullUnit,
                                          Declaration           => Dictionary.Location'(Start_Position => Package_Name.Position,
                                                                                        End_Position   => Package_Name.Position),
                                          Position              => LexTokenManager.Zero_Value,
                                          The_Type              => Bit_Order_Type_Symbol,
                                          TheEnumerationLiteral => Unused_Symbol);
                                       Dictionary.AddEnumerationLiteral
                                         (Name                  => LexTokenManager.Low_Order_First_Token,
                                          Comp_Unit             => ContextManager.NullUnit,
                                          Declaration           => Dictionary.Location'(Start_Position => Package_Name.Position,
                                                                                        End_Position   => Package_Name.Position),
                                          Position              => LexTokenManager.One_Value,
                                          The_Type              => Bit_Order_Type_Symbol,
                                          TheEnumerationLiteral => Unused_Symbol);
                                       Dictionary.Add_Deferred_Constant
                                         (Name           => LexTokenManager.Default_Bit_Order_Token,
                                          Type_Mark      => Bit_Order_Type_Symbol,
                                          Type_Reference => Dictionary.Location'(Start_Position => Package_Name.Position,
                                                                                 End_Position   => Package_Name.Position),
                                          Comp_Unit      => ContextManager.NullUnit,
                                          Declaration    => Dictionary.Location'(Start_Position => Package_Name.Position,
                                                                                 End_Position   => Package_Name.Position),
                                          The_Package    => Lib_Package_Symbol,
                                          TheConstant    => Unused_Symbol);
                                       --# end accept;
                                       Valid_Package := True;
                                    end if;
                                 when Package_Ada =>
                                    --# accept F, 41, "Stable expression expected here";
                                    case CommandLineData.Content.Language_Profile is
                                       when CommandLineData.SPARK83 =>
                                          -- package Ada disallowed in Spark83 mode
                                          Raise_Semantic_Error (Err_Num => 779,
                                                                Lex_Val => Package_Name);
                                          Valid_Package := False;

                                       when CommandLineData.SPARK95_Onwards =>
                                          Lib_Package_Symbol := Ada_Package_Symbol;
                                          Current_Scope      :=
                                            Dictionary.Set_Visibility
                                            (The_Visibility => Dictionary.Visible,
                                             The_Unit       => Lib_Package_Symbol);
                                          Valid_Package      := True;
                                    end case;
                                    --# end accept;
                              end case;
                           end if;
                           exit when Name_Match;
                        end loop;

                     when 2 => -- Checking Child packages
                        case CommandLineData.Content.Language_Profile is
                           when CommandLineData.SPARK83 =>

                              -- Child packages not supported in Ada83 mode. Flag the error at the
                              -- library package name token, unless we have done so already.
                              if Valid_Package then -- No error flagged yet, so do it now.
                                 Raise_Semantic_Error (Err_Num => 779,
                                                       Lex_Val => Current_Declaration.Enc_Package (1));
                                 Valid_Package := False;
                              end if;
                              -- Don't rub it in by giving a second error (776) for an unmatched name. This
                              -- might be a bogus error message anyway since the name may be OK in Ada95.
                              Name_Match := True;

                           when CommandLineData.SPARK95_Onwards =>

                              for I in Child_Package_Range loop
                                 Match_Identifier
                                   (ID       => Possible_Child_Packages (I),
                                    To_Match => Package_Name.Token_Str,
                                    Position => Package_Name.Position,
                                    Matched  => Package_Name_Match);
                                 if Package_Name_Match then
                                    Name_Match := True;
                                    case I is
                                       when Package_Interrupts =>
                                          Local_Package_Symbol :=
                                            Dictionary.LookupSelectedItem
                                            (Prefix   => Lib_Package_Symbol,
                                             Selector => LexTokenManager.Interrupts_Token,
                                             Scope    => Dictionary.GlobalScope,
                                             Context  => Dictionary.ProofContext);
                                       when Package_Real_Time =>
                                          Local_Package_Symbol :=
                                            Dictionary.LookupSelectedItem
                                            (Prefix   => Lib_Package_Symbol,
                                             Selector => LexTokenManager.Real_Time_Token,
                                             Scope    => Dictionary.GlobalScope,
                                             Context  => Dictionary.ProofContext);
                                    end case;
                                    if Dictionary.Is_Null_Symbol (Local_Package_Symbol) then
                                       Valid_Package := False;
                                       --# accept F, 41, "Stable expression expected here";
                                       if not Dictionary.Packages_Are_Equal
                                         (Left_Symbol  => Lib_Package_Symbol,
                                          Right_Symbol => Ada_Package_Symbol) then
                                          -- The child package is a child of a package other than Ada, so
                                          -- name is not recognised.
                                          Name_Match := False;
                                       end if;
                                       --# end accept;
                                    else
                                       Current_Scope :=
                                         Dictionary.Set_Visibility
                                         (The_Visibility => Dictionary.Visible,
                                          The_Unit       => Local_Package_Symbol);
                                       Valid_Package := True;
                                    end if;
                                 end if;
                                 exit when Name_Match;
                              end loop;
                        end case;
                  end case;

                  --# assert True;
                  if not Name_Match then
                     Raise_Semantic_Error (Err_Num => 776,
                                           Lex_Val => Package_Name);
                     Valid_Package := False;
                  end if;
                  --# accept Flow, 33, Unused_Symbol, "Expected to be neither referenced nor exported";
               end Change_To_New_Package;

            begin
               case Parser_Stage is
                  when 1 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.RWpackage);
                     Package_Name_Depth := 1;
                     Next_Stage;
                  when 2 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.identifier);
                     Change_To_New_Package (Package_Name => The_Lex_Val);
                     Current_Declaration.Enc_Package (Package_Name_Depth) := The_Lex_Val;
                     Next_Stage;
                  when 3 =>
                     case The_Token is
                        when SP_Symbols.RWis =>
                           -- Evaluate and store the full name of the package
                           Package_Full_Name :=
                             LexTokenManager.Lex_String_To_String (Lex_Str => Current_Declaration.Enc_Package (1).Token_Str);
                           for I in Child_Package_Depth range 2 .. Package_Name_Depth loop
                              E_Strings.Append_String (E_Str => Package_Full_Name,
                                                       Str   => ".");
                              E_Strings.Append_Examiner_String
                                (E_Str1 => Package_Full_Name,
                                 E_Str2 => LexTokenManager.Lex_String_To_String
                                   (Lex_Str => Current_Declaration.Enc_Package (I).Token_Str));
                              --# assert True;
                           end loop;
                           Next_Stage;
                        when SP_Symbols.point => -- Child unit
                           if Package_Name_Depth = Child_Package_Depth'Last then
                              -- Too many child levels, or a spurious dot
                              Assert_Token (The_Token => The_Token,
                                            Should_Be => SP_Symbols.RWis);
                           else
                              Package_Name_Depth := Package_Name_Depth + 1;
                              Parser_Stage       := 2; -- Go back to process the child name
                           end if;
                        when others =>
                           -- expected a 'package <name> is'
                           Assert_Token (The_Token => The_Token,
                                         Should_Be => SP_Symbols.RWis);
                     end case;
                  when 4 =>
                     Lookahead := True;
                     New_State (The_Stage => Defn);
                  when others =>
                     null;
               end case;
            end Do_Config_Defn_Start;

            procedure Do_Next_Config_Defn_Start
            --# global in     CommandLineData.Content;
            --#        in     Possible_Child_Packages;
            --#        in     Possible_Library_Packages;
            --#        in     Predefined_Scope;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in out Current_Declaration;
            --#        in out Current_Scope;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out LexTokenManager.State;
            --#        in out Lib_Package_Symbol;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Package_Full_Name;
            --#        in out Package_Name_Depth;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Valid_Package;
            --# derives Current_Declaration        from *,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val &
            --#         Current_Scope              from *,
            --#                                         CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         LexTokenManager.State,
            --#                                         Lib_Package_Symbol,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Child_Packages,
            --#                                         Possible_Library_Packages,
            --#                                         Predefined_Scope,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Dictionary.Dict,
            --#         Lib_Package_Symbol         from *,
            --#                                         CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         LexTokenManager.State,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Library_Packages,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         Lib_Package_Symbol,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Child_Packages,
            --#                                         Possible_Library_Packages,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Errors_Occurred,
            --#         Valid_Package              from *,
            --#                                         CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         LexTokenManager.State,
            --#                                         Lib_Package_Symbol,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Child_Packages,
            --#                                         Possible_Library_Packages,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         LexTokenManager.State      from *,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Lookahead                  from *,
            --#                                         Parser_Stage &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Package_Full_Name          from *,
            --#                                         Current_Declaration,
            --#                                         LexTokenManager.State,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Token &
            --#         Package_Name_Depth,
            --#         Parser_Stage               from Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Token;
            is
            begin
               case Parser_Stage is
                  when 1 =>
                     case The_Token is
                        when SP_Symbols.RWpackage =>
                           Package_Name_Depth := 1;
                           Next_Stage;
                        when SP_Symbols.SPEND =>
                           null;
                        when others =>
                           -- expected a 'package ... is'
                           Assert_Token (The_Token => The_Token,
                                         Should_Be => SP_Symbols.RWpackage);
                     end case;
                  when others =>
                     Do_Config_Defn_Start;
               end case;
            end Do_Next_Config_Defn_Start;

            procedure Do_Defn
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in out Current_Declaration;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out LexTokenManager.State;
            --#        in out Next_Expected_State;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#           out Parser_Stage;
            --# derives Current_Declaration,
            --#         LexTokenManager.State      from *,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Errors_Occurred            from *,
            --#                                         The_Token &
            --#         Next_Expected_State        from *,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Parser_Stage,
            --#         Parser_State               from The_Token;
            is
            begin
               case The_Token is
                  when SP_Symbols.RWtype =>
                     New_State (The_Stage => Type_Unknown);
                  when SP_Symbols.RWsubtype =>
                     Current_Declaration.My_Type := Dec_Integer_Subtype;
                     New_State (The_Stage => Subtype_Int);
                  when SP_Symbols.identifier =>
                     Current_Declaration.Name := The_Lex_Val;
                     New_State (The_Stage => Const_Unknown);
                  when SP_Symbols.RWend =>
                     New_State (The_Stage => Config_Defn_End);
                  when others =>
                     -- expected a configuration definition
                     Raise_Syntax_Error
                       (The_Error => Syntax_Errors'(The_Type        => Exp_Defn,
                                                    The_Symbol      => The_Token,
                                                    Expected_Symbol => SP_Symbols.SPDEFAULT,
                                                    The_Lex_Val     => The_Lex_Val));
               end case;
            end Do_Defn;

            procedure Do_Type_Unknown
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in out Current_Declaration;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out LexTokenManager.State;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --# derives Current_Declaration,
            --#         LexTokenManager.State      from *,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         LexTokenManager.State,
            --#                                         Parser_Stage,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Errors_Occurred,
            --#         Parser_Stage,
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         The_Token &
            --#         Next_Expected_State        from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token;
            is
            begin
               case Parser_Stage is
                  when 1 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.identifier);
                     Current_Declaration.Name := The_Lex_Val;
                     Next_Stage;
                  when 2 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.RWis);
                     Next_Stage;
                  when 3 =>
                     case The_Token is
                        when SP_Symbols.RWprivate =>
                           Current_Declaration.My_Type := Dec_Private;
                           New_State (The_Stage => Type_Private);
                        when SP_Symbols.RWdigits =>
                           Current_Declaration.My_Type := Dec_Floating_Point_Type;
                           New_State (The_Stage => Type_FP);
                        when SP_Symbols.RWrange =>
                           Current_Declaration.My_Type := Dec_Integer_Type;
                           New_State (The_Stage => Type_Int);
                        when others =>
                           -- expected either 'digits', 'range' or 'private'
                           Raise_Syntax_Error
                             (The_Error => Syntax_Errors'(The_Type        => Exp_Digits_Range,
                                                          The_Symbol      => The_Token,
                                                          Expected_Symbol => SP_Symbols.SPDEFAULT,
                                                          The_Lex_Val     => The_Lex_Val));
                     end case;
                  when others =>
                     null;
               end case;
            end Do_Type_Unknown;

            procedure Do_Type_Private
            --# global in     CommandLineData.Content;
            --#        in     Current_Declaration;
            --#        in     Current_Scope;
            --#        in     Package_Full_Name;
            --#        in     Possible_Identifiers;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in     Valid_Package;
            --#        in out Any_Priority_Sym;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out Interrupt_Priority_Sym;
            --#        in out LexTokenManager.State;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out Priority_Sym;
            --#        in out SPARK_IO.File_Sys;
            --# derives Any_Priority_Sym,
            --#         Dictionary.Dict,
            --#         Interrupt_Priority_Sym,
            --#         Priority_Sym               from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         Valid_Package &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Errors_Occurred            from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         LexTokenManager.State      from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Lookahead                  from *,
            --#                                         Parser_Stage &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Parser_Stage               from *,
            --#                                         The_Token;
            is
            begin
               case Parser_Stage is
                  when 1 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.semicolon);
                     Next_Stage;
                  when 2 =>
                     Check_And_Process_Declaration (The_Decl => Current_Declaration);
                     Lookahead := True;
                     New_State (The_Stage => Defn);
                  when others =>
                     null;
               end case;
            end Do_Type_Private;

            procedure Do_Type_FP
            --# global in     CommandLineData.Content;
            --#        in     Current_Scope;
            --#        in     Package_Full_Name;
            --#        in     Possible_Identifiers;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in     Valid_Package;
            --#        in out Any_Priority_Sym;
            --#        in out Current_Declaration;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out Interrupt_Priority_Sym;
            --#        in out LexTokenManager.State;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out Priority_Sym;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Unary_Minus;
            --# derives Any_Priority_Sym,
            --#         Dictionary.Dict,
            --#         Interrupt_Priority_Sym,
            --#         Priority_Sym               from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         Valid_Package &
            --#         Current_Declaration        from *,
            --#                                         LexTokenManager.State,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Unary_Minus &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Unary_Minus,
            --#                                         Valid_Package &
            --#         Errors_Occurred            from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Token,
            --#                                         Unary_Minus,
            --#                                         Valid_Package &
            --#         LexTokenManager.State      from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Unary_Minus,
            --#                                         Valid_Package &
            --#         Lookahead                  from *,
            --#                                         Parser_Stage &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token,
            --#                                         Unary_Minus &
            --#         Parser_Stage,
            --#         Unary_Minus                from Parser_Stage,
            --#                                         The_Token,
            --#                                         Unary_Minus;
            is
            begin
               case Parser_Stage is
                  when 1 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.integer_number);
                     Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                               Val => Current_Declaration.Num_Digits);
                     Next_Stage;
                  when 2 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.RWrange);
                     Unary_Minus := False;
                     Next_Stage;
                  when 3 =>
                     case The_Token is
                        when SP_Symbols.real_number =>
                           Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                                     Val => Current_Declaration.Low_Bound);
                           if Unary_Minus then
                              Maths.Negate (Current_Declaration.Low_Bound);
                           end if;
                           Next_Stage;
                        when SP_Symbols.minus =>
                           if Unary_Minus then
                              -- expected a number
                              Assert_Token (The_Token => The_Token,
                                            Should_Be => SP_Symbols.real_number);
                           else
                              Unary_Minus := True;
                           end if;
                        when others =>
                           -- expected either either '-' or a number
                           Raise_Syntax_Error
                             (The_Error => Syntax_Errors'(The_Type        => Exp_Minus_Real,
                                                          The_Symbol      => The_Token,
                                                          Expected_Symbol => SP_Symbols.SPDEFAULT,
                                                          The_Lex_Val     => The_Lex_Val));
                     end case;
                  when 4 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.double_dot);
                     Unary_Minus := False;
                     Next_Stage;
                  when 5 =>
                     case The_Token is
                        when SP_Symbols.real_number =>
                           Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                                     Val => Current_Declaration.High_Bound);
                           if Unary_Minus then
                              Maths.Negate (Current_Declaration.High_Bound);
                           end if;
                           Next_Stage;
                        when SP_Symbols.minus =>
                           if Unary_Minus then
                              -- expected a number
                              Assert_Token (The_Token => The_Token,
                                            Should_Be => SP_Symbols.real_number);
                           else
                              Unary_Minus := True;
                           end if;
                        when others =>
                           -- expected either '-' or a number
                           Raise_Syntax_Error
                             (The_Error => Syntax_Errors'(The_Type        => Exp_Minus_Real,
                                                          The_Symbol      => The_Token,
                                                          Expected_Symbol => SP_Symbols.SPDEFAULT,
                                                          The_Lex_Val     => The_Lex_Val));
                     end case;
                  when 6 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.semicolon);
                     Next_Stage;
                  when 7 =>
                     Check_And_Process_Declaration (The_Decl => Current_Declaration);
                     Lookahead := True;
                     New_State (The_Stage => Defn);
                  when others =>
                     null;
               end case;

            end Do_Type_FP;

            procedure Do_Type_Int
            --# global in     CommandLineData.Content;
            --#        in     Current_Scope;
            --#        in     Expr_Int_Value;
            --#        in     Package_Full_Name;
            --#        in     Possible_Identifiers;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in     Valid_Package;
            --#        in out Any_Priority_Sym;
            --#        in out Current_Declaration;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out Interrupt_Priority_Sym;
            --#        in out Int_Add;
            --#        in out Int_Exponent;
            --#        in out Int_Subtract;
            --#        in out LexTokenManager.State;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out Priority_Sym;
            --#        in out Saved_Parser_Stage;
            --#        in out Saved_Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Unary_Minus;
            --# derives Any_Priority_Sym,
            --#         Dictionary.Dict,
            --#         Interrupt_Priority_Sym,
            --#         Priority_Sym               from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         Valid_Package &
            --#         Current_Declaration        from *,
            --#                                         Expr_Int_Value,
            --#                                         Parser_Stage &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Errors_Occurred            from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Int_Add,
            --#         Int_Exponent,
            --#         Int_Subtract,
            --#         Lookahead,
            --#         Saved_Parser_Stage,
            --#         Unary_Minus                from *,
            --#                                         Parser_Stage &
            --#         LexTokenManager.State      from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Parser_Stage               from *,
            --#                                         The_Token &
            --#         Saved_Parser_State         from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State;
            is
            begin
               case Parser_Stage is
                  when 1 =>
                     Lookahead := True;
                     Next_Stage;
                     Unary_Minus := False;
                     Begin_Int_Expr_Parse;
                  when 2 =>
                     -- got the low bound, last time
                     Current_Declaration.Low_Bound := Expr_Int_Value;
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.double_dot);
                     Next_Stage;
                  when 3 =>
                     Lookahead := True;
                     Next_Stage;
                     Unary_Minus := False;
                     Begin_Int_Expr_Parse;
                  when 4 =>
                     -- got the high bound, last time
                     Current_Declaration.High_Bound := Expr_Int_Value;
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.semicolon);
                     Next_Stage;
                  when 5 =>
                     Check_And_Process_Declaration (The_Decl => Current_Declaration);
                     Lookahead := True;
                     New_State (The_Stage => Defn);
                  when others =>
                     null;
               end case;
            end Do_Type_Int;

            procedure Do_Subtype_Int
            --# global in     CommandLineData.Content;
            --#        in     Current_Scope;
            --#        in     Expr_Int_Value;
            --#        in     Package_Full_Name;
            --#        in     Possible_Identifiers;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in     Valid_Package;
            --#        in out Any_Priority_Sym;
            --#        in out Current_Declaration;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out Interrupt_Priority_Sym;
            --#        in out Int_Add;
            --#        in out Int_Exponent;
            --#        in out Int_Subtract;
            --#        in out LexTokenManager.State;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out Priority_Sym;
            --#        in out Saved_Parser_Stage;
            --#        in out Saved_Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Unary_Minus;
            --# derives Any_Priority_Sym,
            --#         Dictionary.Dict,
            --#         Interrupt_Priority_Sym,
            --#         Priority_Sym               from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         Valid_Package &
            --#         Current_Declaration        from *,
            --#                                         Expr_Int_Value,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Errors_Occurred            from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Int_Add,
            --#         Int_Exponent,
            --#         Int_Subtract,
            --#         Lookahead,
            --#         Saved_Parser_Stage,
            --#         Unary_Minus                from *,
            --#                                         Parser_Stage &
            --#         LexTokenManager.State      from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Interrupt_Priority_Sym,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Valid_Package &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Parser_Stage               from *,
            --#                                         The_Token &
            --#         Saved_Parser_State         from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State;
            is
            begin
               case Parser_Stage is
                  when 1 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.identifier);
                     Current_Declaration.Name := The_Lex_Val;
                     Next_Stage;
                  when 2 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.RWis);
                     Next_Stage;
                  when 3 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.identifier);
                     Current_Declaration.Base_Type := The_Lex_Val;
                     Next_Stage;
                  when 4 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.RWrange);
                     Next_Stage;
                  when 5 =>
                     Lookahead := True;
                     Next_Stage;
                     Unary_Minus := False;
                     Begin_Int_Expr_Parse;
                  when 6 =>
                     -- got the low bound, last time
                     Current_Declaration.Low_Bound := Expr_Int_Value;
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.double_dot);
                     Next_Stage;
                  when 7 =>
                     Lookahead := True;
                     Next_Stage;
                     Unary_Minus := False;
                     Begin_Int_Expr_Parse;
                  when 8 =>
                     -- got the high bound, last time
                     Current_Declaration.High_Bound := Expr_Int_Value;
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.semicolon);
                     Next_Stage;
                  when 9 =>
                     Check_And_Process_Declaration (The_Decl => Current_Declaration);
                     Lookahead := True;
                     New_State (The_Stage => Defn);
                  when others =>
                     null;
               end case;
            end Do_Subtype_Int;

            procedure Do_Const_Unknown
            --# global in     CommandLineData.Content;
            --#        in     Current_Scope;
            --#        in     Expr_Int_Value;
            --#        in     Package_Full_Name;
            --#        in     Possible_Identifiers;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in     Valid_Package;
            --#        in out Any_Priority_Sym;
            --#        in out Current_Declaration;
            --#        in out Dictionary.Dict;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out Interrupt_Priority_Sym;
            --#        in out Int_Add;
            --#        in out Int_Exponent;
            --#        in out Int_Subtract;
            --#        in out LexTokenManager.State;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out Priority_Sym;
            --#        in out Saved_Parser_Stage;
            --#        in out Saved_Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Unary_Minus;
            --# derives Any_Priority_Sym,
            --#         Dictionary.Dict,
            --#         Interrupt_Priority_Sym,
            --#         Priority_Sym               from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Expr_Int_Value,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         Valid_Package &
            --#         Current_Declaration        from *,
            --#                                         Expr_Int_Value,
            --#                                         LexTokenManager.State,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Unary_Minus &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Expr_Int_Value,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Unary_Minus,
            --#                                         Valid_Package &
            --#         Errors_Occurred            from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Expr_Int_Value,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Token,
            --#                                         Unary_Minus,
            --#                                         Valid_Package &
            --#         Int_Add,
            --#         Int_Exponent,
            --#         Int_Subtract,
            --#         Lookahead,
            --#         Saved_Parser_Stage,
            --#         Unary_Minus                from *,
            --#                                         Parser_Stage,
            --#                                         The_Token &
            --#         LexTokenManager.State      from *,
            --#                                         Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Current_Scope,
            --#                                         Dictionary.Dict,
            --#                                         Expr_Int_Value,
            --#                                         Interrupt_Priority_Sym,
            --#                                         Package_Full_Name,
            --#                                         Parser_Stage,
            --#                                         Possible_Identifiers,
            --#                                         Priority_Sym,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Unary_Minus,
            --#                                         Valid_Package &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token,
            --#                                         Unary_Minus &
            --#         Parser_Stage               from *,
            --#                                         The_Token,
            --#                                         Unary_Minus &
            --#         Saved_Parser_State         from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token;
            is
            begin
               case Parser_Stage is
                  when 1 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.colon);
                     Next_Stage;
                  when 2 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.RWconstant);
                     Next_Stage;
                  when 3 =>
                     if The_Token = SP_Symbols.identifier then
                        Current_Declaration.My_Type   := Dec_Typed_Constant;
                        Current_Declaration.Base_Type := The_Lex_Val;
                        -- Stay in stage 3, since we now expect :=
                     else
                        Assert_Token (The_Token => The_Token,
                                      Should_Be => SP_Symbols.becomes);
                        Unary_Minus := False;
                        Next_Stage;
                     end if;
                  when 4 =>
                     case The_Token is
                        when SP_Symbols.real_number =>
                           Current_Declaration.My_Type := Dec_Named_Real;
                           Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                                     Val => Current_Declaration.Value);
                           if Unary_Minus then
                              Maths.Negate (Current_Declaration.Value);
                           end if;
                           Next_Stage;
                        when SP_Symbols.minus =>
                           if Unary_Minus then
                              -- expected a number
                              Raise_Syntax_Error
                                (The_Error => Syntax_Errors'(The_Type        => Exp_Integer_Real,
                                                             The_Symbol      => The_Token,
                                                             Expected_Symbol => SP_Symbols.SPDEFAULT,
                                                             The_Lex_Val     => The_Lex_Val));
                           else
                              Unary_Minus := True;
                           end if;
                        when SP_Symbols.identifier =>
                           Current_Declaration.Value_Str := The_Lex_Val;
                           Next_Stage;
                        when others =>
                           Current_Declaration.My_Type := Dec_Named_Integer;
                           Lookahead                   := True;
                           Next_Stage;
                           Begin_Int_Expr_Parse;
                     end case;
                  when 5 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.semicolon);
                     Next_Stage;
                  when 6 =>
                     if Current_Declaration.My_Type = Dec_Named_Integer then
                        Current_Declaration.Value := Expr_Int_Value;
                     end if;
                     Check_And_Process_Declaration (The_Decl => Current_Declaration);
                     Lookahead := True;
                     New_State (The_Stage => Defn);
                  when others =>
                     null;
               end case;
            end Do_Const_Unknown;

            procedure Do_Config_Defn_End
            --# global in     Any_Priority_Sym;
            --#        in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     Interrupt_Priority_Sym;
            --#        in     Package_Full_Name;
            --#        in     Package_Name_Depth;
            --#        in     Possible_Library_Packages;
            --#        in     Priority_Sym;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in out Current_Declaration;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out LexTokenManager.State;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Package_End_Name;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --# derives Current_Declaration        from *,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val &
            --#         ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
            --#                                         CommandLineData.Content,
            --#                                         Current_Declaration,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_End_Name,
            --#                                         Package_Full_Name,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Library_Packages,
            --#                                         Priority_Sym,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Errors_Occurred            from *,
            --#                                         Any_Priority_Sym,
            --#                                         Interrupt_Priority_Sym,
            --#                                         LexTokenManager.State,
            --#                                         Package_End_Name,
            --#                                         Package_Full_Name,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         Possible_Library_Packages,
            --#                                         Priority_Sym,
            --#                                         The_Token &
            --#         LexTokenManager.State      from *,
            --#                                         Package_Full_Name,
            --#                                         Package_Name_Depth,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Lookahead                  from *,
            --#                                         Parser_Stage &
            --#         Next_Expected_State,
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Package_End_Name           from *,
            --#                                         LexTokenManager.State,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Parser_Stage               from *,
            --#                                         The_Token;
            is

               procedure Check_Package_End
               --# global in     Any_Priority_Sym;
               --#        in     CommandLineData.Content;
               --#        in     Current_Declaration;
               --#        in     Dictionary.Dict;
               --#        in     Interrupt_Priority_Sym;
               --#        in     Package_End_Name;
               --#        in     Package_Full_Name;
               --#        in     Package_Name_Depth;
               --#        in     Possible_Library_Packages;
               --#        in     Priority_Sym;
               --#        in out ErrorHandler.Error_Context;
               --#        in out Errors_Occurred;
               --#        in out LexTokenManager.State;
               --#        in out SPARK_IO.File_Sys;
               --# derives ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from Any_Priority_Sym,
               --#                                         CommandLineData.Content,
               --#                                         Current_Declaration,
               --#                                         Dictionary.Dict,
               --#                                         ErrorHandler.Error_Context,
               --#                                         Interrupt_Priority_Sym,
               --#                                         LexTokenManager.State,
               --#                                         Package_End_Name,
               --#                                         Package_Full_Name,
               --#                                         Package_Name_Depth,
               --#                                         Possible_Library_Packages,
               --#                                         Priority_Sym,
               --#                                         SPARK_IO.File_Sys &
               --#         Errors_Occurred            from *,
               --#                                         Any_Priority_Sym,
               --#                                         Interrupt_Priority_Sym,
               --#                                         LexTokenManager.State,
               --#                                         Package_End_Name,
               --#                                         Package_Full_Name,
               --#                                         Package_Name_Depth,
               --#                                         Possible_Library_Packages,
               --#                                         Priority_Sym &
               --#         LexTokenManager.State      from *,
               --#                                         Package_Full_Name,
               --#                                         Package_Name_Depth;
               is
                  Full_Name_Token_Str : LexTokenManager.Lex_String;
               begin
                  if Package_Name_Depth = 1 then
                     Full_Name_Token_Str := Current_Declaration.Enc_Package (1).Token_Str;
                  else
                     -- We must construct a new lex string from the library and child package names
                     LexTokenManager.Insert_Examiner_String (Str     => Package_Full_Name,
                                                             Lex_Str => Full_Name_Token_Str);
                  end if;

                  -- Check that the (dotted) name after "end" matches the package name.
                  if not E_Strings.Eq_String (E_Str1 => Package_End_Name,
                                              E_Str2 => Package_Full_Name) then
                     -- don't use normal error handler here; need to pass different string from
                     -- position
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 58,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Current_Declaration.Name.Position,
                        Id_Str    => Full_Name_Token_Str);
                     Errors_Occurred := True;
                  else
                     Casing.Check_String_Casing
                       (Str      => Package_End_Name,
                        Lex_Str  => Full_Name_Token_Str,
                        Position => Current_Declaration.Name.Position);
                  end if;

                  -- Check that the full set of priority decls are given if we have Any_Priority
                  if LexTokenManager.Comp_Str_Case_Insensitive
                    (Str     => Package_Full_Name,
                     Lex_Str => Possible_Library_Packages (Package_System)) then
                     if not Dictionary.Is_Null_Symbol (Any_Priority_Sym)
                       and then (Dictionary.Is_Null_Symbol (Priority_Sym)
                                   or else Dictionary.Is_Null_Symbol (Interrupt_Priority_Sym)) then
                        Raise_Semantic_Error (Err_Num => 770,
                                              Lex_Val => Current_Declaration.Name);
                     end if;
                  end if;

               end Check_Package_End;

            begin
               case Parser_Stage is
                  when 1 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.identifier);
                     -- Build up the name after "end" in Package_End_Name
                     Package_End_Name         := LexTokenManager.Lex_String_To_String (Lex_Str => The_Lex_Val.Token_Str);
                     Current_Declaration.Name := The_Lex_Val;
                     Next_Stage;
                  when 2 =>
                     case The_Token is
                        when SP_Symbols.semicolon =>
                           Next_Stage;
                           -- Note that we don't need to parse the name too strictly here
                           -- e.g. we won't (yet) fail <id> <id> .   The string that is
                           -- built won't match the package name unless it is well-formed,
                           -- and so a badly-formed name will fail later in Check_Package_End.
                        when SP_Symbols.point =>
                           E_Strings.Append_String (E_Str => Package_End_Name,
                                                    Str   => ".");
                        when SP_Symbols.identifier =>
                           E_Strings.Append_Examiner_String
                             (E_Str1 => Package_End_Name,
                              E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => The_Lex_Val.Token_Str));
                        when others =>
                           -- expected a 'end <identifier>[.<identifier>] ;'
                           Assert_Token (The_Token => The_Token,
                                         Should_Be => SP_Symbols.semicolon);
                     end case;
                  when 3 =>
                     Lookahead := True;
                     Check_Package_End;
                     New_State (The_Stage => Next_Config_Defn_Start);
                  when others =>
                     null;
               end case;
            end Do_Config_Defn_End;

            procedure Do_Expr_Int
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     Saved_Parser_Stage;
            --#        in     Saved_Parser_State;
            --#        in     The_Lex_Val;
            --#        in     The_Token;
            --#        in out ErrorHandler.Error_Context;
            --#        in out Errors_Occurred;
            --#        in out Expr_Int_Value;
            --#        in out Int_Add;
            --#        in out Int_Add_Sub_Val;
            --#        in out Int_Exponent;
            --#        in out Int_Exponent_Val;
            --#        in out Int_Subtract;
            --#        in out Int_Val;
            --#        in out LexTokenManager.State;
            --#        in out Lookahead;
            --#        in out Next_Expected_State;
            --#        in out Parser_Stage;
            --#        in out Parser_State;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Unary_Minus;
            --# derives ErrorHandler.Error_Context,
            --#         SPARK_IO.File_Sys          from CommandLineData.Content,
            --#                                         Dictionary.Dict,
            --#                                         ErrorHandler.Error_Context,
            --#                                         Int_Add,
            --#                                         Int_Add_Sub_Val,
            --#                                         Int_Exponent,
            --#                                         Int_Exponent_Val,
            --#                                         Int_Subtract,
            --#                                         Int_Val,
            --#                                         LexTokenManager.State,
            --#                                         Parser_Stage,
            --#                                         SPARK_IO.File_Sys,
            --#                                         The_Lex_Val,
            --#                                         The_Token,
            --#                                         Unary_Minus &
            --#         Errors_Occurred,
            --#         Int_Add,
            --#         Int_Exponent,
            --#         Int_Subtract,
            --#         Lookahead,
            --#         Unary_Minus                from *,
            --#                                         Parser_Stage,
            --#                                         The_Token &
            --#         Expr_Int_Value             from *,
            --#                                         Int_Add,
            --#                                         Int_Add_Sub_Val,
            --#                                         Int_Exponent,
            --#                                         Int_Exponent_Val,
            --#                                         Int_Subtract,
            --#                                         Int_Val,
            --#                                         Parser_Stage,
            --#                                         The_Token,
            --#                                         Unary_Minus &
            --#         Int_Add_Sub_Val,
            --#         Int_Val,
            --#         LexTokenManager.State      from *,
            --#                                         LexTokenManager.State,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val,
            --#                                         The_Token &
            --#         Int_Exponent_Val           from *,
            --#                                         LexTokenManager.State,
            --#                                         Parser_Stage,
            --#                                         The_Lex_Val &
            --#         Next_Expected_State        from *,
            --#                                         Parser_Stage,
            --#                                         Parser_State,
            --#                                         The_Token &
            --#         Parser_Stage               from *,
            --#                                         Saved_Parser_Stage,
            --#                                         The_Token &
            --#         Parser_State               from *,
            --#                                         Parser_Stage,
            --#                                         Saved_Parser_State,
            --#                                         The_Token;
            is

               procedure End_Int_Expr_Parse
               --# global in     Saved_Parser_Stage;
               --#        in     Saved_Parser_State;
               --#           out Parser_Stage;
               --#           out Parser_State;
               --# derives Parser_Stage from Saved_Parser_Stage &
               --#         Parser_State from Saved_Parser_State;
               is
               begin
                  Parser_State := Saved_Parser_State;
                  Parser_Stage := Saved_Parser_Stage;
               end End_Int_Expr_Parse;

               procedure Calc_Int_Value (Result : out Maths.Value)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     Int_Add;
               --#        in     Int_Add_Sub_Val;
               --#        in     Int_Exponent;
               --#        in     Int_Exponent_Val;
               --#        in     Int_Subtract;
               --#        in     Int_Val;
               --#        in     LexTokenManager.State;
               --#        in     The_Lex_Val;
               --#        in     Unary_Minus;
               --#        in out ErrorHandler.Error_Context;
               --#        in out SPARK_IO.File_Sys;
               --# derives ErrorHandler.Error_Context,
               --#         SPARK_IO.File_Sys          from CommandLineData.Content,
               --#                                         Dictionary.Dict,
               --#                                         ErrorHandler.Error_Context,
               --#                                         Int_Add,
               --#                                         Int_Add_Sub_Val,
               --#                                         Int_Exponent,
               --#                                         Int_Exponent_Val,
               --#                                         Int_Subtract,
               --#                                         Int_Val,
               --#                                         LexTokenManager.State,
               --#                                         SPARK_IO.File_Sys,
               --#                                         The_Lex_Val,
               --#                                         Unary_Minus &
               --#         Result                     from Int_Add,
               --#                                         Int_Add_Sub_Val,
               --#                                         Int_Exponent,
               --#                                         Int_Exponent_Val,
               --#                                         Int_Subtract,
               --#                                         Int_Val,
               --#                                         Unary_Minus;
               is
                  So_Far, Temp_Val : Maths.Value;
                  Local_ME         : Maths.ErrorCode := Maths.NoError;
                  Had_An_Error     : Boolean         := False;
               begin
                  -- valid typing of the values is confirmed by correct lexical analysis,
                  -- and therefore the only Maths.ErrorCode that can arise from the calculations
                  -- here is Maths.Overflow; resulting in a semantic warning.
                  if Int_Exponent then
                     Maths.RaiseByPower (Int_Val, Int_Exponent_Val, So_Far, Local_ME);
                     if Local_ME /= Maths.NoError then
                        ErrorHandler.Semantic_Warning
                          (Err_Num  => 200,
                           Position => The_Lex_Val.Position,
                           Id_Str   => The_Lex_Val.Token_Str);
                        Had_An_Error := True;
                     end if;
                  else
                     So_Far := Int_Val;
                  end if;
                  if Unary_Minus then
                     Maths.Negate (So_Far);
                  end if;
                  --# assert True;
                  if Int_Subtract then
                     Maths.Subtract (So_Far, Int_Add_Sub_Val, Temp_Val, Local_ME);
                     So_Far := Temp_Val;
                  elsif Int_Add then
                     Maths.Add (So_Far, Int_Add_Sub_Val, Temp_Val, Local_ME);
                     So_Far := Temp_Val;
                  end if;
                  if Local_ME /= Maths.NoError and then not Had_An_Error then
                     ErrorHandler.Semantic_Warning
                       (Err_Num  => 200,
                        Position => The_Lex_Val.Position,
                        Id_Str   => The_Lex_Val.Token_Str);
                  end if;
                  Result := So_Far;
               end Calc_Int_Value;

            begin
               case Parser_Stage is
                  when 1 =>
                     case The_Token is
                        when SP_Symbols.minus =>
                           Unary_Minus  := True;
                           Parser_Stage := 2;
                        when SP_Symbols.integer_number =>
                           Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                                     Val => Int_Val);
                           Parser_Stage := 3;
                        when others =>
                           -- expected either '-' or an integer literal
                           Raise_Syntax_Error
                             (The_Error => Syntax_Errors'(The_Type        => Exp_Minus_Integer,
                                                          The_Symbol      => The_Token,
                                                          Expected_Symbol => SP_Symbols.SPDEFAULT,
                                                          The_Lex_Val     => The_Lex_Val));
                     end case;
                  when 2 =>
                     Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                               Val => Int_Val);
                     Parser_Stage := 3;
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.integer_number);
                  when 3 =>
                     case The_Token is
                        when SP_Symbols.minus =>
                           Int_Subtract := True;
                           Parser_Stage := 6;
                        when SP_Symbols.plus =>
                           Int_Add      := True;
                           Parser_Stage := 6;
                        when SP_Symbols.double_star =>
                           Int_Exponent := True;
                           Parser_Stage := 4;
                        when others =>
                           Lookahead := True;
                           Calc_Int_Value (Result => Expr_Int_Value);
                           End_Int_Expr_Parse;
                     end case;
                  when 4 =>
                     Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                               Val => Int_Exponent_Val);
                     Parser_Stage := 5;
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.integer_number);
                  when 5 =>
                     case The_Token is
                        when SP_Symbols.minus =>
                           Int_Subtract := True;
                           Parser_Stage := 6;
                        when SP_Symbols.plus =>
                           Int_Add      := True;
                           Parser_Stage := 6;
                        when others =>
                           Lookahead := True;
                           Calc_Int_Value (Result => Expr_Int_Value);
                           End_Int_Expr_Parse;
                     end case;
                  when 6 =>
                     Assert_Token (The_Token => The_Token,
                                   Should_Be => SP_Symbols.integer_number);
                     Guarded_Literal_To_Value (Lex => The_Lex_Val,
                                               Val => Int_Add_Sub_Val);
                     Next_Stage;
                  when 7 =>
                     Calc_Int_Value (Result => Expr_Int_Value);
                     Lookahead := True;
                     End_Int_Expr_Parse;
                  when others =>
                     null;
               end case;
            end Do_Expr_Int;

         begin
            if not Lookahead then
               --# accept F, 10, Unwanted_Punct_Token, "Unwanted_Punct_Token unused here";
               SparkLex.Examiner_Lex
                 (Prog_Text   => The_Config_File,
                  Token       => The_Token,
                  Lex_Val     => The_Lex_Val,
                  Punct_Token => Unwanted_Punct_Token);
               --# end accept;
            else
               Lookahead := False;
            end if;

            case Parser_State is

               when Config_Defn_Start =>
                  Do_Config_Defn_Start;
               when Next_Config_Defn_Start =>
                  Do_Next_Config_Defn_Start;
               when Defn =>
                  Do_Defn;
               when Type_Unknown =>
                  Do_Type_Unknown;
               when Type_Private =>
                  Do_Type_Private;
               when Type_FP =>
                  Do_Type_FP;
               when Type_Int =>
                  Do_Type_Int;
               when Subtype_Int =>
                  Do_Subtype_Int;
               when Const_Unknown =>
                  Do_Const_Unknown;
               when Config_Defn_End =>
                  Do_Config_Defn_End;
               when Expr_Int =>
                  Do_Expr_Int;
               when Found_Syntax_Error =>
                  if The_Token = SP_Symbols.semicolon then
                     New_State (The_Stage => Next_Expected_State);
                  end if;

            end case;
            --# accept F, 33, Unwanted_Punct_Token, "Unwanted_Punct_Token unused here";
         end State_Machine_Iterate;

      begin
         SparkLex.Clear_Line_Context;
         State_Machine_Initialise;
         loop
            State_Machine_Iterate;
            exit when The_Token = SP_Symbols.SPEND or else Parsing_Ends;
         end loop;
         Overall_Status := not Errors_Occurred;

      end Process_Config_File;

   begin
      File_Opened_OK := False;
      ErrorHandler.Get_Error_Context (Context => The_Error_Context);
      if CommandLineData.Content.Target_Config then
         Open_File (The_File => Local_Config_File,
                    File_Ok  => File_Opened_OK);
         if File_Opened_OK then
            if CommandLineData.Content.Echo and not CommandLineData.Content.Brief then
               ScreenEcho.New_Line (1);
               ScreenEcho.Put_Line ("           Reading target configuration file ...");
            end if;
            ErrorHandler.Get_Error_Context (Context => Saved_Error_Context);
            ErrorHandler.Error_Init
              (Source_File_Name => CommandLineData.Content.Target_Config_File,
               Echo             => CommandLineData.Content.Echo);
            Process_Config_File (The_Config_File => Local_Config_File,
                                 Overall_Status  => File_Read_OK);
            ErrorHandler.Get_Error_Context (Context => The_Error_Context);
            ErrorHandler.Set_Error_Context (Context => Saved_Error_Context);
            --# accept F, 10, Local_Config_File, "Local_Config_File unused here";
            Close_File (The_File => Local_Config_File);
            --# end accept;
         else
            File_Read_OK := False;
         end if;
      else
         File_Read_OK := True;
      end if;
      No_Errors := File_Read_OK;
      Opened_OK := File_Opened_OK;
   end Read_Config_File;

   procedure Output_Config_File (To_File : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     The_Error_Context;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out XMLReport.State;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys,
   --#         XMLReport.State            from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         The_Error_Context,
   --#                                         To_File,
   --#                                         XMLReport.State;
   is
      Saved_Error_Context : ErrorHandler.Error_Contexts;
      Error_Severity      : ErrorHandler.Error_Level;
      Target_Config_File  : E_Strings.T;

      procedure Check_And_Report_Errors
      --# global in     Error_Severity;
      --#        in     To_File;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.Error_Context from *,
      --#                                         Error_Severity &
      --#         SPARK_IO.File_Sys          from *,
      --#                                         Error_Severity,
      --#                                         To_File;
      is
      begin
         if Error_Severity /= ErrorHandler.No_Error then
            SPARK_IO.New_Line (To_File, 1);
            SPARK_IO.Put_Line (To_File, "Warning: analysis aborted due to errors in target configuration file", 0);
            ErrorHandler.Set_File_Open_Error;
         end if;
      end Check_And_Report_Errors;

   begin
      if CommandLineData.Content.Target_Config then
         if CommandLineData.Content.XML then
            XMLReport.Start_Section (Section => XMLReport.S_Target_Config,
                                     Report  => To_File);
            Target_Config_File := CommandLineData.Content.Target_Config_File;
            XMLReport.Filename (Plain_Output => CommandLineData.Content.Plain_Output,
                                File         => Target_Config_File);
            E_Strings.Put_String (File  => To_File,
                                  E_Str => Target_Config_File);
            ErrorHandler.Get_Error_Context (Context => Saved_Error_Context);
            ErrorHandler.Set_Error_Context (Context => The_Error_Context);
            ErrorHandler.Get_Error_Severity (Severity => Error_Severity);
            XMLReport.Start_Section (Section => XMLReport.S_Messages,
                                     Report  => To_File);
            ErrorHandler.AppendErrors (To_File, Error_Types.ForReportIndexedFiles);
            XMLReport.End_Section (Section => XMLReport.S_Messages,
                                   Report  => To_File);

            Check_And_Report_Errors;

            ErrorHandler.Set_Error_Context (Context => Saved_Error_Context);

            XMLReport.End_Section (Section => XMLReport.S_Target_Config,
                                   Report  => To_File);

         else
            SPARK_IO.New_Line (To_File, 2);
            SPARK_IO.Put_Line (To_File, "Target configuration file:", 0);
            ErrorHandler.Get_Error_Context (Context => Saved_Error_Context);
            ErrorHandler.Set_Error_Context (Context => The_Error_Context);
            ErrorHandler.Get_Error_Severity (Severity => Error_Severity);
            ErrorHandler.PrintErrors (To_File, Error_Types.ForReportIndexedFiles);

            Check_And_Report_Errors;

            ErrorHandler.Set_Error_Context (Context => Saved_Error_Context);
         end if;
      end if;
   end Output_Config_File;

end ConfigFile;
