(*
 * A "hash-cons" utility.
 *
 * ----------------------------------------------------------------
 *
 * @begin[license]
 * Copyright (C) 2005-2007 Mojave Group, Caltech and HRL Laboratories, LLC
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation,
 * version 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Additional permission is given to link this library with the
 * OpenSSL project's "OpenSSL" library, and with the OCaml runtime,
 * and you may distribute the linked executables.  See the file
 * LICENSE.libmojave for more details.
 *
 * Author: Jason Hickey @email{jyh@cs.caltech.edu}
 * Modified By: Aleksey Nogin @email{anogin@hrl.com} @email{nogin@metaprl.org}
 * @end[license]
 *)
open Lm_printf
open Lm_thread_core

open Lm_hash_sig

(************************************************************************
 * A generic hash module to make comparisons faster.
 * This version uses a state for hash-consing.
 *)
module MakeHash (Arg : HashArgSig)
: HashSig with type elt = Arg.t =
struct
   type elt = Arg.t

   (* %%MAGICBEGIN%% *)
   type t = int * elt
   (* %%MAGICEND%% *)

   let create x =
      Arg.hash x, x

   let get (_, x) =
      x

   let hash (i, _) =
      i

   let compare ((i1 : int), x1) ((i2 : int), x2) =
      if i1 = i2 then
         Arg.compare x1 x2
      else if i1 < i2 then
         -1
      else
         1
end

(************************************************************************
 * Table-based hashing.
 *)
module MakeHashCons (Arg : HashArgSig)
: HashConsSig
  with type elt = Arg.t
  with type hash = MakeHash(Arg).t =
struct
   (* %%MAGICBEGIN%% *)
   type elt = Arg.t
   type t = int

   module Key = MakeHash (Arg);;
   module KeyTable = Lm_map.LmMake (Key);;
   type hash = Key.t

   (*
    * We need both directions.
    *)
   type state =
      { mutable key_table : int KeyTable.t;
        mutable int_table : elt array
      }
   (* %%MAGICEND%% *)

   let create_state () =
      { key_table = KeyTable.empty;
        int_table = [||]
      }

   let length state =
      KeyTable.cardinal state.key_table

   let set state i x =
      let table = state.int_table in
      let len = Array.length table in
         if len = 0 then
            let table = Array.create 32 x in
               state.int_table <- table
         else if i = len then
            let table2 = Array.create (len * 2) x in
               Array.blit table 0 table2 0 len;
               state.int_table <- table2
         else
            table.(i) <- x

   let icreate state item =
      try KeyTable.find state.key_table item with
         Not_found ->
            let index = KeyTable.cardinal state.key_table in
               state.key_table <- KeyTable.add state.key_table item index;
               set state index (Key.get item);
               index

   let create state x =
      icreate state (Key.create x)

   let get state index =
      state.int_table.(index)

   let hash index =
      index

   let compare index1 index2 =
      index1 - index2

   let map_array f state =
      Array.mapi f (Array.sub state.int_table 0 (KeyTable.cardinal state.key_table))

   let fold f x state =
      let len = KeyTable.cardinal state.key_table in
      let rec fold i x =
         if i = len then
            x
         else
            fold (succ i) (f x i)
      in
         fold 0 x
end

(************************************************************************
 * Marshalable version.
 *
 * This takes a slightly different approach, wrapping the value in
 * a triple of a hash code and a dummy ref cell.  During marshaling,
 * the cell will point somewhere else, so we know that the value
 * must be rehashed.
 *)

(* %%MAGICBEGIN%% *)
type 'a hash_marshal_item =
   { mutable item_ref : unit ref;
     mutable item_val : 'a;
     item_hash        : int
   }
(* %%MAGICEND%% *)

(*
 * The reference in the current process.
 *)
let current_ref = ref ()

(*
 * Statistics.
 *)
type hash_stat =
   { hash_debug              : string;
     mutable hash_reintern   : int;
     mutable hash_compare    : int;
     mutable hash_collisions : int
   }

let hash_stats = ref []

let pp_print_stat buf stat =
   let { hash_debug      = debug;
         hash_reintern   = reintern;
         hash_compare    = compare;
         hash_collisions = collisions
       } = stat
   in
      fprintf buf "@[<hv 3>%s: reintern = %d, compare = %d, collisions = %d@]@\n" (**)
         debug reintern compare collisions

let pp_print_hash_stats buf =
   List.iter (pp_print_stat buf) !hash_stats

(*
 * let () =
 *    at_exit (fun () -> pp_print_hash_stats stderr)
 *)

(*
 * We need to work nicely with threads.
 *
 * Note that the reintern function may be recursive, so we need to account for cases,
 * where the current thread is already holding the lock.
 * Almost every accesses come from the main thread with very little if any contention from other
 * threads. This makes it more effiecient to use a single global lock (as opposed to having a
 * separate lock for each instance of the functor), so that mutually recursive reintern calls only
 * have to lock one lock, not all of them.
 *
 * Finally, we do not care about race conditions for the statistics
 *)
module Synchronize : sig
   val synchronize : ('a -> 'b) -> 'a -> 'b
end = struct
   let lock_mutex = MutexCore.create "Lm_hash.Synchronize"
   let lock_id = ref None

   let unsynchronize () =
      lock_id := None;
      MutexCore.unlock lock_mutex

   let synchronize f x =
      let id = ThreadCore.id (ThreadCore.self ()) in
         match !lock_id with
            Some id' when id = id' ->
               (*
                * We are already holding the lock. This means:
                *  - we do not have to do anything special
                *  - reading the shared lock_id ref could not have created a race condition
                *)
               f x
          | _ ->
               MutexCore.lock lock_mutex;
               lock_id := Some id;
               try
                  let res = f x in
                     unsynchronize();
                     res
               with exn ->
                  unsynchronize();
                  raise exn
end

let synchronize =
   if ThreadCore.enabled then
      Synchronize.synchronize
   else
      (fun f -> f)

(*
 * Make a hash item.
 *)
module MakeHashMarshal (Arg : HashMarshalArgSig) =
struct
   type elt = Arg.t
   type t = elt hash_marshal_item

   (* Keep a hash-cons table based on a weak comparison *)
   module WeakCompare =
   struct
      type t = elt hash_marshal_item

      let compare item1 item2 =
         let hash1 = item1.item_hash in
         let hash2 = item2.item_hash in
            if hash1 < hash2 then
               -1
            else if hash1 > hash2 then
               1
            else
               Arg.compare item1.item_val item2.item_val
   end;;

   module Table = Lm_map.LmMake (WeakCompare);;

   let table = ref Table.empty

   (*
    * Keep track of collisions for debugging.
    *)
   let stats =
      { hash_debug       = Arg.debug;
        hash_reintern    = 0;
        hash_compare     = 0;
        hash_collisions  = 0
      }

   let () =
      hash_stats := stats :: !hash_stats

   (*
    * When creating an item, look it up in the table.
    *)
   let create_core elt =
      let item =
         { item_ref  = current_ref;
           item_val  = elt;
           item_hash = Arg.hash elt
         }
      in
         try Table.find !table item with
            Not_found ->
               table := Table.add !table item item;
               item

   let create = synchronize create_core

   let intern elt =
      let item =
         { item_ref  = current_ref;
           item_val  = elt;
           item_hash = Arg.hash elt
         }
      in
         Table.find !table item

   (*
    * Reintern.  This will take an item that may-or-may-not be hashed
    * and produce a new one that is hashed.
    *)
   let reintern_core item1 =
      stats.hash_reintern <- succ stats.hash_reintern;
      try
         let item2 = Table.find !table item1 in
            if item2 != item1 then begin
               item1.item_val <- item2.item_val;
               item1.item_ref <- current_ref
            end;
            item2
      with
         Not_found ->
            item1.item_val <- Arg.reintern item1.item_val;
            item1.item_ref <- current_ref;
            table := Table.add !table item1 item1;
            item1

   let reintern = synchronize reintern_core

   (*
    * Access to the element.
    *)
   let get item =
      if item.item_ref == current_ref then
         item.item_val
      else
         (reintern item).item_val

   let hash item =
      item.item_hash

   (*
    * String pointer-based comparison.
    *)
   let compare item1 item2 =
      stats.hash_compare <- succ stats.hash_compare;
      let hash1 = item1.item_hash in
      let hash2 = item2.item_hash in
         if hash1 < hash2 then
            -1
         else if hash1 > hash2 then
            1
         else if item1.item_val == item2.item_val then
            0
         else
            let elt1 = get item1 in
            let elt2 = get item2 in
               if elt1 == elt2 then
                  0
               else begin
                  stats.hash_collisions <- succ stats.hash_collisions;
                  let cmp = Arg.compare elt1 elt2 in
                     if cmp = 0 then
                        invalid_arg "Lm_hash is broken@.";
                     cmp
               end

   let equal item1 item2 =
      (item1 == item2) || (item1.item_hash = item2.item_hash && get item1 == get item2)
end

(*
 * A version with two equalities.
 * The fine equality is used for cons-hashing, but the coarse
 * version is used for external comparisons.  The fine equality
 * must be a refinement of the coarse equality.
 *)
(* %%MAGICBEGIN%% *)
type 'a hash_marshal_eq_item = ('a * 'a hash_marshal_item) hash_marshal_item
(* %%MAGICEND%% *)

module MakeHashMarshalEq (Arg : HashMarshalEqArgSig) =
struct
   type elt = Arg.t
   type t = elt hash_marshal_eq_item

   module CoarseArg =
   struct
      type t     = Arg.t
      let debug  = Arg.debug ^ ":coarse"

      let hash     = Arg.coarse_hash
      let compare  = Arg.coarse_compare
      let reintern = Arg.reintern
   end;;

   module Coarse = MakeHashMarshal (CoarseArg);;

   (*
    * We fold the Coarse item into the fine
    * item only so we don't have to create three
    * modules (the final one being a pair of fine
    * and coarse).
    *)
   module FineArg =
   struct
      type t     = Arg.t * Coarse.t
      let debug  = Arg.debug ^ ":fine"

      (*
       * We're assuming that the fine hash is a
       * refinement of the coarse one.
       *)
      let hash (fine, _) =
         Arg.fine_hash fine

      let compare (fine1, _) (fine2, _) =
         Arg.fine_compare fine1 fine2

      let reintern ((fine, coarse) as item) =
         let fine' = Arg.reintern fine in
         let coarse' = Coarse.reintern coarse in
            if fine' == fine && coarse' == coarse then
               item
            else
               fine', coarse'
   end;;

   module Fine = MakeHashMarshal (FineArg);;

   let create x =
      Fine.create (x, Coarse.create x)

   let intern x =
      Fine.intern (x, Coarse.intern x)

   let get info =
      fst (Fine.get info)

   (*
    * The default functions are the coarse versions.
    *)
   let get_coarse info =
      snd (Fine.get info)

   let hash info =
      Coarse.hash (get_coarse info)

   let compare item1 item2 =
      Coarse.compare (get_coarse item1) (get_coarse item2)

   let equal item1 item2 =
      Coarse.equal (get_coarse item1) (get_coarse item2)

   (*
    * Also export the fine versions.
    *)
   let fine_hash = Fine.hash
   let fine_compare = Fine.compare
   let fine_equal = Fine.equal

   let reintern = Fine.reintern
end

(************************************************************************
 * Better-than usual hash functions.
 *
 * BUG: JYH: this hash is entirely ad-hoc.  The idea here is the following.
 * If we were able, we would use the message as an index into a random function.
 * We can't do that, so jump around in a smaller table.
 *
 * We construct the hash on-the-fly, using a table of random numbers.
 *)

(* %%MAGICBEGIN%% *)
let hash_length = 6229 (* Must be a prime *)
let digest_length = 16
let hash_data =
   [|0x04a018c6; 0x5ba7b0f2; 0x04dcf08b; 0x1e5a22cc; 0x2523b9ea; 0x4b92b691; 0x0bcaf015; 0x0a5d5109;
     0x64d3be71; 0x5fbc0769; 0x3ac35af5; 0x77136b9a; 0x64e16501; 0x24d1efb9; 0x3835ad79; 0x103ac74f;
     0x217edbb5; 0x33afeced; 0x1d9341e3; 0x116c3c4e; 0x01d3893a; 0x2657d18f; 0x324ea1ad; 0x20d56ec8;
     0x2361f48d; 0x50b71a60; 0x1ced96bf; 0x2c06ba13; 0x6a7d99e2; 0x213818ce; 0x182ab58e; 0x05964edf;
     0x049425a9; 0x2fa55545; 0x79c6238c; 0x33ee6060; 0x188796a1; 0x27650997; 0x48dbd56a; 0x51a7dedc;
     0x69b54b8e; 0x06135229; 0x1db1f87f; 0x0c72761c; 0x4974fbd3; 0x71657506; 0x2f8b6b68; 0x247cabda;
     0x71d47c35; 0x491e0036; 0x7da4c370; 0x5682e7c0; 0x5cf7c110; 0x0526f878; 0x262025fa; 0x41a19673;
     0x330e9047; 0x11980c36; 0x6ce02397; 0x0ceba603; 0x43a00345; 0x063c1716; 0x538cb4aa; 0x2f60b5d7;
     0x34943a0e; 0x56b7664c; 0x098cfa23; 0x1a64380e; 0x0ad1f7f3; 0x11a53f1f; 0x5b00d01e; 0x24bbd12d;
     0x2401aafe; 0x1149c2d3; 0x07021acf; 0x1f4ed7d2; 0x3d23c6bc; 0x38dfe52c; 0x3a8e4667; 0x6050bb6d;
     0x691f94eb; 0x537e771e; 0x2a9e04c1; 0x37050e1c; 0x73d70b51; 0x2c67f1ff; 0x6709d487; 0x7266f595;
     0x36fd1713; 0x17e3d678; 0x1751280b; 0x2003a7dd; 0x05bd861d; 0x6b538252; 0x4465c159; 0x3298cb7d;
     0x1489fbe8; 0x0833c615; 0x3aea7136; 0x2103f36d; 0x518acb8a; 0x0c4a0c83; 0x5418f7ed; 0x4ff40db4;
     0x6db84908; 0x2dd59f9d; 0x531a8372; 0x61d38c25; 0x3554dc6c; 0x073bb1c8; 0x10f82b8b; 0x574175dc;
     0x13966434; 0x7dfa6cd3; 0x69ed4f17; 0x62d4b072; 0x1890c649; 0x24052c29; 0x5b8dc934; 0x39833efb;
     0x79f5a355; 0x7d026da5; 0x1b1d310e; 0x211ecd0d; 0x2fdd9c20; 0x6ef9ebc6; 0x36dd158a; 0x10ffca06;
     0x52b38469; 0x74947de8; 0x6b3b1cbe; 0x680c4e82; 0x2657c2e2; 0x781e0f72; 0x145d923f; 0x6f321a9e;
     0x1f4402f5; 0x17392d1c; 0x352a5c33; 0x63840b4e; 0x4fd97839; 0x7a389dd1; 0x3343eb86; 0x04cc72c3;
     0x7cbba01e; 0x64cee4ad; 0x5c1f9282; 0x2de8158b; 0x2941291d; 0x30a3c118; 0x2070d8e8; 0x08503556;
     0x52694d55; 0x2de9ec28; 0x16667c6f; 0x2431a2c7; 0x71f2122e; 0x5b011553; 0x308d4cd1; 0x7db98b06;
     0x5d145510; 0x06040108; 0x5e078a62; 0x0d87bf4e; 0x4c20d8c2; 0x3171964d; 0x2a550059; 0x6a8b2d96;
     0x621079cb; 0x5f400e32; 0x5c78b895; 0x7863cc1f; 0x1857902c; 0x5ab77c01; 0x3922ad96; 0x3ec74313;
     0x090ff562; 0x7d8dc2d9; 0x77ccc196; 0x274aa8ec; 0x20e7d058; 0x5d451da4; 0x38411944; 0x16c92ac5;
     0x17c532ca; 0x387b6c9e; 0x04599060; 0x42d41b08; 0x50b480fe; 0x60b6a668; 0x59bf878b; 0x40ecd7ff;
     0x3c3a0e5f; 0x2226ba1f; 0x00b480d2; 0x27d243ce; 0x1b4fd4ac; 0x5452c80a; 0x6a055d72; 0x0e4f2f5f;
     0x73561052; 0x1e34c368; 0x5c3f2470; 0x0ada6aca; 0x28f33428; 0x54d04532; 0x71a49cc1; 0x4488adb3;
     0x620ea825; 0x6e3c834a; 0x3580610e; 0x1acc45ab; 0x7dcf080e; 0x396d3242; 0x2b6c8a54; 0x438e8578;
     0x506170bc; 0x02021c1c; 0x41d6f98c; 0x5d198360; 0x449a1a80; 0x47d51424; 0x046a749f; 0x68caf27a;
     0x15345f3a; 0x1edc6ace; 0x6b708fe1; 0x013dfa00; 0x7675676d; 0x4efa193c; 0x7f29b2b7; 0x55c10973;
     0x524894bb; 0x27b5ef49; 0x399ee9c8; 0x7570a9e8; 0x5ad98886; 0x68c17a3a; 0x70b75e37; 0x102fdd4f;
     0x678c6368; 0x6ab7b144; 0x781fe959; 0x45ba0108; 0x77746bf9; 0x0bb0a4a1; 0x63e3e01e; 0x08af1ae8;
     0x039963f7; 0x229bc81d; 0x4c123bf6; 0x2bd90b99; 0x0f813575; 0x29c6378f; 0x084561f2; 0x222f7d94;
     0x4a161957; 0x3fe15590; 0x616a3e1f; 0x6db1f066; 0x11b78394; 0x2791bafa; 0x0a634f2b; 0x7434ecb4;
     0x52b86ca4; 0x4c236a71; 0x7cb4bf8a; 0x1a449dcc; 0x4ccc5386; 0x45ff9435; 0x0f9c5572; 0x6c0a91bb;
     0x02182c23; 0x3e1fb322; 0x5bb9834e; 0x076ab97e; 0x3901cfcf; 0x398f5f59; 0x1f5e398e; 0x3423ba34;
     0x02531acd; 0x54aba737; 0x74826232; 0x195f5564; 0x6fb5bd70; 0x217a2270; 0x4453067b; 0x3867017e;
     0x2f5ea4ad; 0x5921c43f; 0x3699b3b4; 0x45314959; 0x121b34ca; 0x698f72f8; 0x79b7598e; 0x597b91b4;
     0x478c0c50; 0x298810dd; 0x7a55c741; 0x15d39d50; 0x79b6686a; 0x6fe06dd6; 0x216d27fa; 0x5d71cdad;
     0x02e1e6f7; 0x6055f976; 0x0477b854; 0x08cb4e67; 0x0c912f95; 0x7d015ba9; 0x17ac4ea1; 0x01f0a52e;
     0x0d87ec13; 0x5049eba7; 0x61605b56; 0x4385f27d; 0x00e75dd4; 0x684e563d; 0x50fa5caf; 0x244df9cb;
     0x421e740d; 0x54a3d875; 0x5a2c5f9c; 0x5b666fa8; 0x0e0b3042; 0x4c92c3c9; 0x494810c7; 0x6d48d5cc;
     0x2b60ad41; 0x3a861587; 0x5236f192; 0x68a7ad18; 0x570462b7; 0x597c7693; 0x5dc3a7c3; 0x6bd51a17;
     0x076979a9; 0x591bdb6b; 0x6f8ef6d3; 0x6c6fc11d; 0x43c409b6; 0x0e563262; 0x00c86b80; 0x19d39020;
     0x2f561416; 0x7c7fb3e0; 0x13fe5c59; 0x7ae78655; 0x6bc7d532; 0x487bf67d; 0x0f7bb1da; 0x43cd6ef9;
     0x203fdffa; 0x1ff4cd6e; 0x3d72805a; 0x1cabb100; 0x57978e79; 0x2d6697a4; 0x39207ced; 0x5b29cd55;
     0x6fad54be; 0x35471293; 0x45d2fe43; 0x222faf62; 0x6041ff94; 0x4fe452e1; 0x09fb3051; 0x5b3f1735;
     0x19f28b3a; 0x4969948b; 0x2bb5f504; 0x6e771990; 0x1a26d5b2; 0x01782693; 0x52c1cf7a; 0x5c9e1e0b;
     0x3f46071e; 0x1b10078b; 0x321fe12e; 0x0a8f1aad; 0x50c5b3b9; 0x244a53a6; 0x26e911e7; 0x5409ca95;
     0x0e562f01; 0x5b9dbb5b; 0x5103de78; 0x6c9532b9; 0x7e109d5c; 0x06df1ba6; 0x54809557; 0x0832d01a;
     0x2e9d33f5; 0x441ebc69; 0x5f1a46fa; 0x4769ef99; 0x53a10634; 0x564f986f; 0x2dbe50d2; 0x0e3b263e;
     0x5e4dd5f9; 0x6c6ce793; 0x3b7e7b2d; 0x6ace470e; 0x0bc364fb; 0x46567d32; 0x09d6c649; 0x713b537f;
     0x115cd25d; 0x311cbccb; 0x23906d48; 0x0ee8ebe6; 0x57f8f27c; 0x4effc46e; 0x62b69f26; 0x021cfce7;
     0x2f39c1dd; 0x11f3dff9; 0x31f19050; 0x3161d7be; 0x653ccb8e; 0x43ab514a; 0x58399317; 0x40b66d3a;
     0x32184fd8; 0x7be92684; 0x5e144054; 0x6a020658; 0x5c6ea9e3; 0x1a0d2a42; 0x6e07b3ab; 0x192dbb0e;
     0x5879ce29; 0x03c24bc9; 0x487cd04d; 0x082e47a3; 0x6256335a; 0x4628daea; 0x50b6cd01; 0x2486fc79;
     0x16f9eeca; 0x216fda6b; 0x765fa687; 0x60463443; 0x629d2b6e; 0x0482ec3c; 0x5a602c5f; 0x4da8f9a4;
     0x16daf805; 0x33445db9; 0x5c2c4b6c; 0x1e6ad5c0; 0x760fc9d0; 0x75054f5f; 0x7e146d0a; 0x2249fc81;
     0x6b58c2b4; 0x754da041; 0x5edd1713; 0x0f53eb43; 0x0f847468; 0x45b60961; 0x35066e17; 0x2d463f9b;
     0x5bff8738; 0x47fece41; 0x6b0df911; 0x2da46db3; 0x7d4df84e; 0x62524b98; 0x784689ba; 0x30d24182;
     0x79f07925; 0x1a60106c; 0x358d2986; 0x590bcace; 0x2319ec0f; 0x3c203ea9; 0x17b863ad; 0x3ff12821;
     0x72ab733a; 0x0a239828; 0x37f310ca; 0x1bd07b8d; 0x2c053d5d; 0x274de61c; 0x67fceabc; 0x6d64e17c;
     0x696c45e8; 0x65a36849; 0x0894492c; 0x25dcbccc; 0x0c1d6062; 0x45ab99bc; 0x3abe382f; 0x123ecab8;
     0x0d637cb3; 0x608dee33; 0x12519e90; 0x180a4c2c; 0x1412023f; 0x63ac36cb; 0x51c60552; 0x2ac4e2fe;
     0x5ebcae69; 0x25e393f3; 0x5741f8fc; 0x672b8977; 0x5cad6441; 0x61abf1a7; 0x0303ade0; 0x66fdb790;
     0x54403634; 0x386440c9; 0x5377f8ea; 0x2e811481; 0x41f9662c; 0x5b570ca7; 0x3fac5ce5; 0x7777b0a4;
     0x7477d864; 0x09b0ce27; 0x65f58514; 0x5e9d1753; 0x5df4b2c7; 0x1537c09a; 0x6986f1bf; 0x72bc0272;
     0x2a99a8fb; 0x42a78e20; 0x686b1a86; 0x4de3eaee; 0x1c64cfdc; 0x78255672; 0x53fbfa39; 0x6afffcdd;
     0x5307a4de; 0x5b4bfe26; 0x60cb0c0b; 0x5d257dc3; 0x02ac3709; 0x1b5cd6b9; 0x6ee55460; 0x45a09f22;
     0x7d08b346; 0x782699e8; 0x1ad44439; 0x152bcaed; 0x4337fcc7; 0x05d7932e; 0x71010380; 0x15b10b1b;
     0x06603391; 0x3502efb4; 0x2581aa16; 0x5a3a4867; 0x7af79303; 0x63e7f7aa; 0x1bcf76f1; 0x01217932;
     0x0b584102; 0x57d85778; 0x7b04b920; 0x5e7acac9; 0x69d85af9; 0x1fd3c6e7; 0x24c391ca; 0x7874e456;
     0x625de14d; 0x184a85a3; 0x12d8b176; 0x629dfb70; 0x0f1eece1; 0x647aa66b; 0x0c8ef991; 0x05d4698f;
     0x39bb53aa; 0x442a76b8; 0x6102de4f; 0x2d6dc958; 0x02b9e81e; 0x22def394; 0x1c63ef6f; 0x35aafbbc;
     0x65a9136c; 0x4b3a6456; 0x49931bd3; 0x1891ccd7; 0x052292a4; 0x5d74f10f; 0x3f0acada; 0x410ecca6;
     0x36a5b66d; 0x67e227f2; 0x5f914b7f; 0x25f7754e; 0x4fb7b36c; 0x074df71e; 0x0ff9b7d1; 0x2c5b5730;
     0x19a2d443; 0x3f0cffd8; 0x25bf69d3; 0x08607eb0; 0x5c3591bf; 0x7e1c83dd; 0x3596611d; 0x344b7b7f;
     0x51318748; 0x1b6bc7de; 0x3950cb93; 0x3150c3c9; 0x6c24ab7f; 0x4a835748; 0x488ff676; 0x344c9a37;
     0x667b4920; 0x04705d16; 0x4aa3a93c; 0x45c12f2b; 0x0e5748f4; 0x71e3436b; 0x02d937ec; 0x77254cff;
     0x65348109; 0x137ee1fc; 0x563f4c81; 0x591d8013; 0x4cb5c38f; 0x69dcae73; 0x2eea4c7f; 0x7574449e;
     0x430921b2; 0x64e96ada; 0x4bfeaea8; 0x1ce3ccd1; 0x7f322d02; 0x7b3f4495; 0x56dd60f2; 0x3fe87a94;
     0x460ef2a1; 0x74b24aea; 0x43910a0a; 0x2f327583; 0x521d6b68; 0x03e9a453; 0x5be1ed4f; 0x0ee37cf9;
     0x2b328f8d; 0x7a30339a; 0x1085ee1a; 0x5d8f806d; 0x77528625; 0x6336b586; 0x40a45acd; 0x654ed038;
     0x04f3d558; 0x0a9d1481; 0x69651f6d; 0x107b5ac2; 0x6d94fddb; 0x604126bd; 0x7c5ce529; 0x572cc18a;
     0x12202a13; 0x5f4fc004; 0x55feb730; 0x5123d263; 0x7a3944b9; 0x055edcbe; 0x214e18b9; 0x5e4f8962;
     0x3783d77f; 0x3ac9698e; 0x48672e1a; 0x7c67071c; 0x1b5e77e9; 0x48e40749; 0x2c30b199; 0x34740baa;
     0x0ace6efb; 0x69ae7350; 0x02431088; 0x453a1936; 0x49be39e8; 0x1fd1a764; 0x2beb8364; 0x511a0ba6;
     0x6c25e43e; 0x7a228cdc; 0x7efc9048; 0x6d5c3147; 0x325d87f6; 0x08ac5f8d; 0x1d1f46a2; 0x5580dbed;
     0x6da9f98a; 0x12d31e07; 0x09bcc9c9; 0x16f4522d; 0x724f739a; 0x0c4ea399; 0x3a47cdc4; 0x431165ed;
     0x45c5f59b; 0x64b7f58d; 0x056ff8a0; 0x1f2b9207; 0x4754c098; 0x1b29d81b; 0x6a3a2189; 0x5d6d59cd;
     0x41eadbb2; 0x6955a4ee; 0x5aee1cd8; 0x702a6ac9; 0x39c38ae2; 0x68236e2e; 0x77fe3bb4; 0x5574f7f6;
     0x7929bb55; 0x4c95d4bf; 0x540c36ad; 0x787cbbd2; 0x4cd8307d; 0x2f57f3f8; 0x554bad0e; 0x2744a1a2;
     0x0d54c124; 0x6189b2c8; 0x714f6ea7; 0x6db51214; 0x0b9f8122; 0x5cd106bd; 0x5b2e9625; 0x0d981aa2;
     0x0e6b2084; 0x4f935c8c; 0x0f4a3fc4; 0x6d7e8310; 0x3c28c81f; 0x419352ee; 0x5c6e8632; 0x55a7a369;
     0x5d608669; 0x16930920; 0x383c0c9c; 0x6d78c524; 0x41b4a572; 0x05206a3a; 0x091f55c4; 0x781e8cc5;
     0x73849939; 0x3069c3d6; 0x102105af; 0x4418aa9f; 0x106077c6; 0x74e3e663; 0x0bd966f0; 0x5a4e92c6;
     0x11949c7f; 0x7094747e; 0x64567f58; 0x03f166fa; 0x5c1ed154; 0x6e4d51e2; 0x0f68c753; 0x6b8bce2a;
     0x1f6bf7e0; 0x5da00969; 0x688983d1; 0x0f9411ea; 0x42c30c29; 0x5e6d5c3f; 0x72ee17f1; 0x5ab9c1d8;
     0x7b804d18; 0x67950752; 0x567ef044; 0x2c8f32f0; 0x398aa8d3; 0x52f5682f; 0x4fd896fc; 0x2c95de24;
     0x3d8b49c8; 0x0f3fb6fb; 0x3e4bd651; 0x3e1fbc15; 0x130c58e2; 0x5eb33a7b; 0x32e8c62c; 0x2db4eaed;
     0x2add624b; 0x48001129; 0x0bfee6fe; 0x52795045; 0x3decda0b; 0x242ca087; 0x768b17ba; 0x27b977b4;
     0x695a9796; 0x37c0d156; 0x1eae9da6; 0x20634d37; 0x591d8024; 0x3d2ea45a; 0x33a3c15a; 0x10be15f0;
     0x4a251182; 0x570af148; 0x174b1e42; 0x17e0b90a; 0x033ce1b3; 0x6c6d27e3; 0x1544431e; 0x3a96b38f;
     0x17937f7d; 0x2dfbb773; 0x400b4c81; 0x2ee1d467; 0x3b23808b; 0x3878c6ee; 0x2ccc3cc8; 0x43d8e836;
     0x2d99a2f4; 0x48aa1ad3; 0x415cfe75; 0x0186dcc1; 0x5c5c3971; 0x28844381; 0x44879b39; 0x600df749;
     0x01cbaf3b; 0x0501e48e; 0x4724e647; 0x1b94525c; 0x6cc4f0ad; 0x19ab8bce; 0x034a8a8f; 0x0d0cf0e6;
     0x48d34837; 0x1bfe75cd; 0x76e6e69c; 0x28b9bf16; 0x4ccf05bb; 0x460b20e3; 0x1aab81e6; 0x22871221;
     0x1bceba58; 0x3162c14f; 0x7a77fc7f; 0x6a96c008; 0x24bd9b86; 0x1eceffae; 0x71d3248a; 0x7dd7496e;
     0x6079b5fb; 0x46aa4663; 0x3d94dc97; 0x2b988744; 0x13894513; 0x54ceade5; 0x51ac6700; 0x3d106301;
     0x5f6ca51b; 0x1ecea974; 0x52398eb0; 0x052fd255; 0x416059c2; 0x496772e3; 0x4ba5eda5; 0x56a3c941;
     0x67c7a5d8; 0x7198fda9; 0x55cdeac1; 0x28da3caf; 0x253dbf27; 0x7e7917b2; 0x61a86d89; 0x50402de9;
     0x0fb8f61b; 0x2014709e; 0x2ef5cbe6; 0x445ad230; 0x23e915db; 0x266cff97; 0x69617cff; 0x00bbc4cc;
     0x3abe6841; 0x4b6f7c97; 0x1763cc8a; 0x50cade99; 0x1e161de1; 0x7ccfcd96; 0x6606f768; 0x70484358;
     0x759d61eb; 0x0cdfbbbc; 0x44e51f9b; 0x555b6bbe; 0x31809e0d; 0x7611c864; 0x0d2d06e8; 0x79daa60a;
     0x075cd420; 0x52976c7c; 0x60fd8034; 0x79576a57; 0x56643e64; 0x10fd1ba7; 0x34a29206; 0x1280e3be;
     0x2b102cac; 0x1de9c81a; 0x624ebbac; 0x015cb1bd; 0x2bf6865d; 0x49e6b9d9; 0x1dc0fa2d; 0x282867a4;
     0x23880f57; 0x49480b6c; 0x1453c615; 0x254f2d3d; 0x3fade2c4; 0x1d484333; 0x3b7ad09d; 0x241e0c3f;
     0x3f6bce27; 0x3a266bd8; 0x5c16a2c8; 0x2efc228a; 0x0171f038; 0x2df99611; 0x2d9f7cc3; 0x6607604d;
     0x2a339b56; 0x56451520; 0x064c5792; 0x005f8733; 0x014186ba; 0x2c453eec; 0x5b4c16bc; 0x166a91d9;
     0x3d6a1867; 0x4a9f32a0; 0x04992e7f; 0x5f25c7d1; 0x059c19da; 0x6050625c; 0x528f5dc5; 0x5aa7dae0;
     0x09c76d57; 0x1719c27b; 0x5fb89917; 0x7d7e1a63; 0x7132108b; 0x1b07e3dc; 0x3be9c866; 0x677d3f98;
     0x31dc4c07; 0x4681afde; 0x7f8593fc; 0x08a73130; 0x1510c9b1; 0x00fd5082; 0x7b19afa1; 0x3d48c17f;
     0x5d809454; 0x60607809; 0x2f32d01a; 0x5d2d79d9; 0x7358f349; 0x328de67b; 0x2783c512; 0x219128b8;
     0x5d6895ab; 0x6100781e; 0x1401caf2; 0x57d88484; 0x77ec205f; 0x09615075; 0x6fcede45; 0x6ccc9133;
     0x58b1da70; 0x0c62d9f3; 0x2fb2802a; 0x50b2767e; 0x14070e83; 0x366ff9a2; 0x08e33e81; 0x5a7501f2;
     0x6297cb91; 0x2b736f71; 0x4d2a6dbc; 0x11cbe4ac; 0x6796c025; 0x6696e255; 0x1a7db249; 0x05b7a730;
     0x1a564e20; 0x51396ea3; 0x476de965; 0x6ef27926; 0x72036c29; 0x14af38ef; 0x42a0bec5; 0x02b9f243;
     0x0914df62; 0x0a5f3eb8; 0x4c295107; 0x66bf8d61; 0x7bd10745; 0x46996fb2; 0x09f0744d; 0x52b6d2c4;
     0x047baaaa; 0x7793d0f2; 0x7bce7e8c; 0x2ceb5f22; 0x69a768bf; 0x041b2743; 0x12b1134f; 0x1b614c4f;
     0x50d0ad7f; 0x61aaf852; 0x31b19398; 0x3cd568f5; 0x7aff4e9a; 0x01be839b; 0x15e70b51; 0x4ddc10cc;
     0x41ac7cc6; 0x32278de9; 0x7c297ffa; 0x538b5777; 0x2f654e22; 0x206fc727; 0x36b61615; 0x3083f758;
     0x0a5ff376; 0x50e83ac5; 0x298f5119; 0x1be5047d; 0x4cacacba; 0x15729437; 0x30bffe29; 0x796454bc;
     0x464164c5; 0x6e1b52a4; 0x34a197b1; 0x2f6de3e4; 0x33ec8cf0; 0x7689183a; 0x25494ffd; 0x3c2e08b1;
     0x52aa9539; 0x202225e6; 0x48c9c982; 0x104b069a; 0x5bd3d6eb; 0x2a31eb11; 0x3f0ebf0f; 0x5db7a5b9;
     0x032e26b9; 0x7ce94a74; 0x1de43949; 0x60283bca; 0x446326a0; 0x182fec6c; 0x5d482618; 0x24a8136b;
     0x16b82f35; 0x35dd65bf; 0x3cdb68d0; 0x6fa2feff; 0x64f25d36; 0x714f7b92; 0x6474293c; 0x438b3310;
     0x38f1d054; 0x1c7cbe18; 0x6557eb33; 0x44f6f003; 0x001da352; 0x4d42f5b1; 0x26a86a29; 0x587aa516;
     0x21d2d6cc; 0x5ddedcd4; 0x04b27318; 0x3f1ddec5; 0x0ba998cc; 0x6fa3526b; 0x317ca40b; 0x64046062;
     0x2b988fe4; 0x4d0ca215; 0x49293924; 0x4aec68d1; 0x46987709; 0x224d68cd; 0x01da0c8c; 0x640ab4ad;
     0x2a0fde34; 0x258ff50d; 0x66cdf960; 0x0115127f; 0x31337e3c; 0x1f0c97ed; 0x431c7f85; 0x7420e6be;
     0x1503d04f; 0x266164cb; 0x34c41bca; 0x76ee24e8; 0x053c829a; 0x6919bd42; 0x470eecb2; 0x19e34d6b;
     0x1c3410ff; 0x78f1ce94; 0x002fa2d9; 0x148043e1; 0x1bd1b14e; 0x7a730d54; 0x2268f97f; 0x265c3cd4;
     0x035ab6ef; 0x5636dde3; 0x4b8ce2e7; 0x089993dd; 0x07d33f78; 0x47a06853; 0x08818463; 0x4d4d5a43;
     0x3d984089; 0x34048502; 0x710b3345; 0x7dd0b5cf; 0x4c1e989b; 0x223669f4; 0x191f26ac; 0x06d7c58d;
     0x4d3804c7; 0x58e5fad3; 0x12acb23c; 0x3fb31889; 0x53530ba7; 0x187f836e; 0x565d5fd1; 0x75916085;
     0x78cfee7d; 0x50d3414a; 0x4b08da32; 0x62b6b757; 0x22403cce; 0x31555807; 0x64b730e0; 0x2782f7ac;
     0x68c2d1f0; 0x03d8b1fb; 0x71ba690c; 0x13358eba; 0x289a8d55; 0x44451f4b; 0x4aaa7397; 0x6e8df8dc;
     0x5cc51ef8; 0x62ba14a0; 0x55247932; 0x1cb2c617; 0x38c4279a; 0x3e3fdb1d; 0x7923d01a; 0x48b5a0da;
     0x2ded66c5; 0x55d8bde2; 0x30e0f1b5; 0x0ef8ee5b; 0x26499f32; 0x59e46eea; 0x203238ae; 0x317d94be;
     0x22b66c78; 0x3d77c3e2; 0x3dbb7d80; 0x37d4cc3c; 0x3513c89f; 0x652c1aa5; 0x3e45f397; 0x18952cb9;
     0x715d715b; 0x5818afe4; 0x1b2c8936; 0x721b42a9; 0x3806eab6; 0x373fb803; 0x65cf04e6; 0x60158639;
     0x229fa6c7; 0x4ba93161; 0x45883ed7; 0x6acc130b; 0x292e1ba1; 0x7e32eb03; 0x28d74880; 0x6918953e;
     0x02e9e610; 0x7ecf2cd2; 0x055965b5; 0x1be309f7; 0x1a349a09; 0x50004d81; 0x3e142878; 0x4c8fc307;
     0x5b624ed3; 0x6a206d54; 0x511daea3; 0x28e07141; 0x76effaab; 0x3fb9e092; 0x56599aaf; 0x47c1a9f7;
     0x43ba2dd6; 0x76f6b718; 0x057227ba; 0x1d102f2d; 0x78950bf6; 0x354ed3ac; 0x6bb6949b; 0x1ac6de9c;
     0x70fd18b2; 0x24f64dab; 0x7abbe85b; 0x4b59634a; 0x457f3101; 0x69d0c293; 0x5f4b3b3a; 0x64b41643;
     0x091603bc; 0x53aca769; 0x06540870; 0x32206888; 0x0c8ddb7c; 0x65a2fe9e; 0x6ae56c5d; 0x35bc174a;
     0x0e19faaa; 0x6d968f8a; 0x0c255585; 0x70729843; 0x6eb4bfbc; 0x096ccfec; 0x4283f5d3; 0x18dbc0bb;
     0x0b8bd4d8; 0x3c7bc5d8; 0x36511e62; 0x3fcf9fcb; 0x5863f9e1; 0x0c93a134; 0x3868cb81; 0x09bd7098;
     0x4d6c4d3a; 0x743aa681; 0x452b22f1; 0x73110ba0; 0x30bab914; 0x482d48f4; 0x39aebe41; 0x4335e085;
     0x437220b0; 0x784cd569; 0x451cb08d; 0x35e65220; 0x5a5923c5; 0x23ea7282; 0x5b5ce8f6; 0x60a4b66f;
     0x004d4f86; 0x0a0da666; 0x23afa782; 0x41843bd0; 0x66e3c594; 0x09f01fff; 0x759e0327; 0x19c0d86a;
     0x6b45d72f; 0x1c030b2c; 0x1dcd8a65; 0x6bfbabca; 0x17605f56; 0x7e2f0134; 0x1b34fba5; 0x650f3665;
     0x3d96ae27; 0x59c19804; 0x3068e28a; 0x3f570a23; 0x435466c9; 0x5defe5b1; 0x2b2b2868; 0x5420bbe5;
     0x0d71526b; 0x7ed008cb; 0x404884f4; 0x2a1f25cb; 0x1b08266c; 0x6e3ab543; 0x39f5159e; 0x7f71867e;
     0x0af9051f; 0x11cbee38; 0x13efcd0b; 0x02320a8e; 0x67d2c7f1; 0x5ebca7e9; 0x55f5fbfc; 0x3c7e1f48;
     0x2e49d4c6; 0x1d4ea021; 0x48976b2d; 0x162b5749; 0x30a48718; 0x4df5ccb5; 0x4483dc7e; 0x495cc6a3;
     0x47e774d7; 0x33a166ea; 0x6c320c7e; 0x4ea77f68; 0x28f2a560; 0x2a8eb1b8; 0x4b2caa11; 0x76a4b342;
     0x7ea97d2c; 0x60149381; 0x3ffd3034; 0x0b5d8676; 0x2ed5dad2; 0x16e0f9bc; 0x53337770; 0x5fc683e4;
     0x6160eb15; 0x1a60fbb8; 0x75512e2c; 0x74485660; 0x24bbd3b7; 0x72e49a59; 0x67fac082; 0x696d6f1e;
     0x5210a35c; 0x3da4a2df; 0x026ae1db; 0x15a79c6a; 0x2b4cd21c; 0x2b624148; 0x61c25d04; 0x55496f5d;
     0x558722ad; 0x2a1273fb; 0x290e1223; 0x037c391e; 0x39d83696; 0x02301f08; 0x46e728de; 0x7c19a28d;
     0x331e1641; 0x2c8cc04f; 0x0317cb2f; 0x0bb5cb0e; 0x3c494b88; 0x7ad14516; 0x46670386; 0x35cd8c27;
     0x347493c4; 0x65c121af; 0x5a5af777; 0x24ec5406; 0x4efb1d22; 0x0706f58b; 0x1ec08da3; 0x477e8577;
     0x75e2b792; 0x6380f73b; 0x659dbbf4; 0x3c3d238c; 0x36149d5d; 0x7d803b71; 0x613eeaf4; 0x4a0eb13d;
     0x66935032; 0x5e34fc6d; 0x7f5e0e88; 0x6b58b524; 0x08e6c626; 0x13d7d6ab; 0x24b13685; 0x16ccaa95;
     0x30f916c1; 0x4d9b9d66; 0x362fa441; 0x42d7cfc3; 0x68fef43e; 0x0767accc; 0x0c9adb6d; 0x64294a9b;
     0x7de30f35; 0x0107a9eb; 0x51f769ba; 0x1ba83ee3; 0x43ab802a; 0x5c8e6383; 0x597f9fe2; 0x01a76dfd;
     0x71920459; 0x50dd63d1; 0x221243f9; 0x4c0daca6; 0x0f3dddf0; 0x7d1e3b58; 0x1a6b1765; 0x6b3e0df2;
     0x2f96ef48; 0x15d27e9b; 0x430fbf95; 0x21a38f7b; 0x77b096e8; 0x438799a8; 0x77c4e8ab; 0x37e67c07;
     0x7e0daa9a; 0x731a1cff; 0x15cc30d4; 0x738101a0; 0x472e465c; 0x5a9350a5; 0x56de970b; 0x10b61cbb;
     0x005cd8b0; 0x103b68f3; 0x3668b959; 0x1738935d; 0x14427f04; 0x573cdf89; 0x00291429; 0x48a290c9;
     0x7ec02ac9; 0x0b8f3442; 0x115573ed; 0x2d68388f; 0x08ad0c10; 0x1bcdaf7d; 0x307fb7a5; 0x47711635;
     0x79cb7e95; 0x2d95f62f; 0x5e1002b6; 0x30a6d954; 0x538f91f6; 0x29a90194; 0x3ea7ca32; 0x27139830;
     0x5fff5777; 0x3294e1c6; 0x1ff8aff2; 0x25271b4a; 0x6e81606b; 0x1c20f54d; 0x443c10d6; 0x71a68314;
     0x405df3c7; 0x01e035c0; 0x20af781f; 0x69a5a459; 0x47e41335; 0x5b96e3b5; 0x5e6cf1f2; 0x6af6bb79;
     0x354816a1; 0x54483619; 0x5e77e838; 0x36ff7b1d; 0x003d365e; 0x065217d6; 0x3c3c0a43; 0x16399e3d;
     0x10fc0ecc; 0x54706771; 0x11f8b664; 0x6beb65ed; 0x5b67f0f0; 0x1a5104af; 0x016c6ed5; 0x2b64b40e;
     0x21d7666b; 0x40182bee; 0x4150d06a; 0x0fa97fe1; 0x7e249e7a; 0x3a903dfc; 0x7a58a20a; 0x5413b746;
     0x52282b47; 0x5bd7770e; 0x6b7d2b6c; 0x0fdbec2b; 0x24a73ca0; 0x6b0ed62b; 0x2dc7bed2; 0x7cd08546;
     0x58cbb358; 0x115a485c; 0x0593a84c; 0x7d245728; 0x7e705081; 0x35ba77b9; 0x77eb1ff8; 0x390ed226;
     0x5b746195; 0x7a2f46ca; 0x173c2f51; 0x7f5ff48e; 0x0e5e0c24; 0x26885263; 0x539e6af3; 0x4975bf07;
     0x737ec724; 0x378c8526; 0x0548525b; 0x1e6aa7de; 0x3d3e23ca; 0x1bee9c7e; 0x459fdcc4; 0x168a4e12;
     0x300b753e; 0x224e3486; 0x18aaa7af; 0x240edda5; 0x6b1ce7c7; 0x39e54fc4; 0x68c0d007; 0x60a1237a;
     0x060af98e; 0x1c664afd; 0x60191e88; 0x5008ca70; 0x0729d4ea; 0x7af0fea1; 0x3e567b36; 0x636e2b36;
     0x6991ccde; 0x4e970d41; 0x7f4a23d4; 0x61ae9b21; 0x519c859b; 0x64755291; 0x6c821ce0; 0x6c092357;
     0x58d1d605; 0x643fdb00; 0x7af9d488; 0x78b375e4; 0x5c1b5fed; 0x19737d40; 0x25ca92f2; 0x484e53c3;
     0x32de5364; 0x67aca2f1; 0x348f85eb; 0x7bf18db4; 0x1e808be1; 0x27f8037d; 0x303ec858; 0x148276f9;
     0x783b394c; 0x549d9591; 0x7b903dfa; 0x7c744ec6; 0x48e33110; 0x355061b0; 0x2e9433e3; 0x4f7d56e5;
     0x266002ad; 0x349a2de8; 0x453d6d97; 0x19e2d185; 0x06b439d2; 0x4e6d973e; 0x5acd66d4; 0x47e4bb2e;
     0x0ffd3fd8; 0x33792f6b; 0x7b4dda07; 0x58b28d85; 0x42976852; 0x660be05d; 0x2f6daa6b; 0x443310de;
     0x3a8593a7; 0x343815d6; 0x65cb403e; 0x15f39284; 0x0eb91fed; 0x71b11b1e; 0x7ee61aec; 0x7b2361c4;
     0x0c80d611; 0x2d06e291; 0x7aa1b1fa; 0x3fc08590; 0x22dc6056; 0x190bc4b5; 0x3071efe8; 0x2936bb08;
     0x626654bf; 0x1a6a59ef; 0x4f39fd61; 0x6423b9f3; 0x25ae2918; 0x3649275a; 0x2bb09ddd; 0x146c8d63;
     0x01d7052c; 0x5700ffe5; 0x02195656; 0x491c1747; 0x332ccff2; 0x7b047ab4; 0x1cd7cff6; 0x610f9463;
     0x7d3626e7; 0x3ccab87c; 0x4056307b; 0x71652f78; 0x5026ecf1; 0x4cb33171; 0x1c2346ea; 0x46a481f8;
     0x2442e434; 0x272369d0; 0x178f0114; 0x2528634e; 0x7e601305; 0x5213ab95; 0x7aca98dd; 0x03a1f86b;
     0x179b39fe; 0x638fdf4c; 0x6e187033; 0x427d6e27; 0x7424600e; 0x686883bb; 0x43245a8f; 0x0c67e7fc;
     0x07ecca7b; 0x70d66cb9; 0x4b66d2ef; 0x50c96ebd; 0x7abd1121; 0x3b652ef4; 0x27b6181f; 0x04835d8a;
     0x006db332; 0x5c67fb8f; 0x2ef4823f; 0x2262f538; 0x79352f1e; 0x3dd506e1; 0x0e697922; 0x4398c0e3;
     0x17614e94; 0x42329b31; 0x1891be3f; 0x38876db4; 0x60c2645a; 0x297d710b; 0x4eaf2588; 0x633c61de;
     0x0e3ec224; 0x2b0f367f; 0x5783cd9b; 0x3012db44; 0x0dad035e; 0x75862c6b; 0x18a6a9e4; 0x457d54b8;
     0x0cfbbfc1; 0x626d0948; 0x1a11609a; 0x52db0720; 0x0d2e9949; 0x73592db9; 0x0836c12e; 0x01a3ca36;
     0x4eff3899; 0x54b688e2; 0x77cfd704; 0x0d16b76a; 0x39ce0419; 0x123ca410; 0x3feaac4e; 0x5ffd2dbe;
     0x7e5604cd; 0x7591dbcb; 0x4596e87b; 0x026d7e8d; 0x28ca14de; 0x17b18e89; 0x31935812; 0x4d23f884;
     0x15a1e507; 0x0e62eed6; 0x399b6464; 0x3dad0b13; 0x53c8a326; 0x5697042c; 0x78718a8b; 0x6b8a6385;
     0x383878b3; 0x5f65a31d; 0x43903e76; 0x5199544e; 0x79d5f067; 0x7bb7271b; 0x68d23870; 0x3a753b5f;
     0x18ff32dc; 0x0668a4d1; 0x1ee33a88; 0x1622a4d4; 0x3ccb06a2; 0x75a9daa8; 0x344b0695; 0x25b2f982;
     0x06b309c2; 0x6124d76d; 0x1f8566a2; 0x446e5001; 0x3fbb65c2; 0x666f84bc; 0x029cc5d4; 0x06fef632;
     0x3b1f652d; 0x2313b502; 0x51aa196a; 0x3becdbed; 0x77b388df; 0x5824fe17; 0x4f2bc8d9; 0x0b7e2cb5;
     0x0211418f; 0x4be80c5c; 0x6f598410; 0x01d70361; 0x3ef9cd78; 0x56e8a46d; 0x33f249ef; 0x708375da;
     0x10232b3e; 0x1bf28dc1; 0x7f248535; 0x70b11739; 0x703a6f57; 0x0a7d8ac4; 0x4366fb05; 0x3570ce08;
     0x6399d401; 0x523eb476; 0x6c9dc7e6; 0x1fe20c02; 0x5349e6e3; 0x5683a0ed; 0x4f45860a; 0x21b9c950;
     0x38b4c90a; 0x122a63c4; 0x00dcadc7; 0x5c685fc9; 0x579d4124; 0x742d8c77; 0x5d890bf0; 0x472af297;
     0x6524ca78; 0x2d85b3f4; 0x0661a51a; 0x2fc4c6ad; 0x0f732de0; 0x3f7db076; 0x25654f96; 0x32dfcc7b;
     0x5cff18f6; 0x51d24ec7; 0x3a849f02; 0x2d8cd529; 0x348a155f; 0x128ac16c; 0x51b2bfa3; 0x2f3742c6;
     0x6d6dee68; 0x3e1780e5; 0x71430228; 0x57be94a9; 0x649de755; 0x1dedc43a; 0x46dccc41; 0x029ecc67;
     0x0eda761a; 0x03f0b658; 0x23c73c16; 0x15e0a3b6; 0x7424f1e7; 0x6b2b752e; 0x25ee3cfe; 0x163b63b3;
     0x39455fc4; 0x5cf9ac67; 0x56f72a60; 0x0fdeb5c2; 0x11337b7c; 0x75580a63; 0x6ef3223f; 0x3530b515;
     0x60e52142; 0x40a0c952; 0x75ee0856; 0x46859913; 0x73b0e9be; 0x1fc694ec; 0x6df120aa; 0x4d21a11c;
     0x1778b9d5; 0x6b5ce9de; 0x31c6e7d4; 0x20b75a04; 0x28079b69; 0x2a017d5b; 0x75b8dc56; 0x131cae15;
     0x2b744638; 0x0b13383a; 0x48941fa6; 0x408fa934; 0x523b8aa7; 0x2c64c822; 0x386abe53; 0x5a5c8e4e;
     0x1ff099f1; 0x5a9d146f; 0x43735120; 0x0ba80217; 0x5d830ad8; 0x77dd1e3b; 0x6417a535; 0x3f9f1757;
     0x3cbce2b9; 0x7df926ee; 0x3d47799b; 0x736acbe8; 0x3e2b902d; 0x21b3f03f; 0x3fd3b242; 0x5cbc2690;
     0x1aa2d70e; 0x7a898e86; 0x2b2aeff9; 0x05fc5fa0; 0x688709bf; 0x2029696a; 0x6f2b92db; 0x69c593fb;
     0x32887339; 0x46b5c7ee; 0x01c7adef; 0x6c2c1d80; 0x4dc496b3; 0x439d5df7; 0x056a9c2c; 0x452faabc;
     0x4b5cf188; 0x14c30e37; 0x51d9a1ee; 0x32850029; 0x74b38012; 0x22c37992; 0x3eded38a; 0x13f31ecf;
     0x1b0c45de; 0x282ba3b0; 0x7d1a6816; 0x0a37ebaa; 0x4eb98141; 0x155bd048; 0x6909adee; 0x7c60985d;
     0x38b9986a; 0x25376392; 0x22468a80; 0x597bb549; 0x2c907676; 0x5b789d12; 0x1c09c2ca; 0x2ace1417;
     0x0f309694; 0x1be1622a; 0x4d303e18; 0x52f5cf6d; 0x75268055; 0x399b4ccb; 0x3a438c99; 0x3e2e9ff4;
     0x472d2559; 0x2e179f04; 0x029d0266; 0x4e4187ac; 0x7602c4f2; 0x3b015ab9; 0x0ec8ca63; 0x64f334fe;
     0x2fcda2cc; 0x55d36bd7; 0x55b8f6dd; 0x6e8afb12; 0x1519cd67; 0x6bb80efd; 0x0c487762; 0x4c11ff13;
     0x5f06a508; 0x706c098a; 0x51b16212; 0x736f8ed2; 0x0a77d16b; 0x6f8a2069; 0x5a0aa0d6; 0x7379f199;
     0x47df978a; 0x41f66f74; 0x33b3f10a; 0x326d3e81; 0x4cad3f0a; 0x4a7ea929; 0x1744d250; 0x0e57925f;
     0x2fbdd887; 0x228a587d; 0x795ab273; 0x5d524715; 0x35e405a4; 0x3c5d1502; 0x34375ada; 0x24284551;
     0x2d7b8657; 0x2f6895b4; 0x5e3ccd6e; 0x021b67ea; 0x05a0b1fe; 0x219af52e; 0x7a68af79; 0x7dbe4b05;
     0x10609660; 0x1716ca9c; 0x44c4c4de; 0x0c76fc6f; 0x6bee144c; 0x7e2f247c; 0x70e27c80; 0x246a8e8d;
     0x1d8a1923; 0x1332ef7f; 0x15623c32; 0x7b12aca5; 0x1ff63a96; 0x21ad6d66; 0x3f7c14bd; 0x3c47833a;
     0x2fbb3aa9; 0x4319e180; 0x2b52c07f; 0x27faf754; 0x5e1d447e; 0x475b7bbb; 0x078f8db8; 0x775b81f9;
     0x61c33237; 0x1f1eb3cd; 0x013f98b6; 0x72c234b7; 0x3499c3c1; 0x1508aeea; 0x45fc1b80; 0x6e7656d1;
     0x6e465c79; 0x02421332; 0x18d05641; 0x03b461a8; 0x2069c886; 0x6dd1d1fd; 0x4e2eb486; 0x267d6b90;
     0x70a73a9d; 0x485ba502; 0x513b8e12; 0x04cde3bd; 0x1b5842d5; 0x59f4839b; 0x00baf57b; 0x4fd0a7cd;
     0x5dcec214; 0x476c94b1; 0x24b49a8f; 0x02345c07; 0x70bf90a6; 0x4263c388; 0x2bc09fa9; 0x67920475;
     0x102db6de; 0x49477a59; 0x0d6bca31; 0x765d5c6d; 0x7641defc; 0x2382f738; 0x5f2d3847; 0x0813dad3;
     0x36110f2b; 0x5a0d70cb; 0x5a820c56; 0x2ab100d3; 0x314cc0b3; 0x10a63ea1; 0x21afc203; 0x1af7f635;
     0x523cadd3; 0x5848faac; 0x2009b1a4; 0x28edf346; 0x0e076f39; 0x40dd1dfc; 0x19454d7e; 0x536499e1;
     0x4cfe66a5; 0x740eec2d; 0x4389c1d1; 0x3b480d8f; 0x559fe63e; 0x3aa95d86; 0x02b063f1; 0x226d9525;
     0x3a8939c9; 0x20822a33; 0x17b7deae; 0x1374633e; 0x42f5c1e7; 0x2ee28717; 0x07bc5581; 0x12fb5ff1;
     0x0fc33713; 0x6835fba6; 0x34194bd0; 0x32f0c720; 0x52a22dc5; 0x275e1a4f; 0x398788ae; 0x73552ec8;
     0x0ccd1589; 0x0b565dfa; 0x45a85700; 0x33dfdd95; 0x6804b8ff; 0x15e0bc31; 0x0edf3044; 0x56c23414;
     0x2a904a5d; 0x0ff824f2; 0x3803521e; 0x38c96075; 0x418bafd5; 0x0b1751bf; 0x50de1f84; 0x1243e3ce;
     0x4142a3bb; 0x6bf5acd0; 0x4a98b49b; 0x17c380d7; 0x10eabb7f; 0x2e211ee9; 0x43559895; 0x120f8fe5;
     0x41444c79; 0x556ed854; 0x1e77d791; 0x6784a7c2; 0x49d63651; 0x61406b8c; 0x3eec4ed1; 0x2046aab6;
     0x135f339e; 0x5ff8d5ce; 0x45c31cae; 0x0fd112b3; 0x7bfe394b; 0x45815dd0; 0x2740b5b9; 0x34583ab6;
     0x6e16793a; 0x4476575a; 0x655bc32a; 0x5e8a512e; 0x2d69f885; 0x1a5df686; 0x3b8ef85c; 0x46f4866b;
     0x3b91267a; 0x2b224b1e; 0x7cdf7a3e; 0x1d30f22a; 0x4b7906db; 0x691afbe5; 0x4a81220e; 0x19b2cb5d;
     0x003ccfe1; 0x60aa8e47; 0x2e80e827; 0x00c10c0e; 0x7290e823; 0x3ab9193e; 0x59018e01; 0x091f1369;
     0x29457a4a; 0x1cb22871; 0x5eaa7421; 0x7d60120a; 0x2c5dc1d5; 0x221d0cbc; 0x7dcfc197; 0x41f27b22;
     0x42e081cd; 0x40f58eca; 0x0e329ebb; 0x2b02fbb5; 0x38eb6398; 0x1d302dee; 0x144bfab6; 0x47b45d2d;
     0x0afc5794; 0x4f7d7450; 0x77b637dc; 0x2f352f4f; 0x47aa8c6d; 0x192ee6f5; 0x54edb9e6; 0x5ad955dd;
     0x21bf10bb; 0x7882522d; 0x4ff3105b; 0x0e003402; 0x3914db07; 0x07206f09; 0x192bd68b; 0x48e097d6;
     0x7f738d03; 0x3548f239; 0x1f556559; 0x5a120699; 0x1de212c4; 0x44327121; 0x729735d7; 0x1df93727;
     0x6f30bf2e; 0x3582bbba; 0x2cfbdfc7; 0x762ebe19; 0x23b295e3; 0x2a45a0b6; 0x42adfcec; 0x7467b349;
     0x3bb6db6d; 0x4f1b9159; 0x52971f27; 0x78e4f308; 0x72bb223e; 0x6eaeb9dd; 0x41c75b76; 0x43075843;
     0x5f705c31; 0x18a40dc7; 0x483408f9; 0x67737627; 0x66e6f6f0; 0x6db5e1a0; 0x628b50c9; 0x5e498778;
     0x3e8504ad; 0x0ef8a5c5; 0x322e8288; 0x2850987d; 0x34bfd531; 0x09476c01; 0x0d800fb3; 0x00b31b4e;
     0x7b08d065; 0x3cc13245; 0x478aa452; 0x558307dc; 0x7ebdd476; 0x4ed7ebe5; 0x563856c7; 0x7c06186d;
     0x1b5a46a4; 0x3c2eab23; 0x2e597499; 0x2cce345b; 0x646da30b; 0x7207882b; 0x0e69b01b; 0x79425667;
     0x10aa8aaa; 0x420028f0; 0x548d5c62; 0x1acc1c75; 0x66150147; 0x25c6bbcd; 0x323eb521; 0x19b69fed;
     0x74968c5c; 0x45226a9f; 0x57e49b04; 0x37798ef2; 0x4314898a; 0x4ba3a806; 0x243fa728; 0x4f195e5d;
     0x725318ee; 0x0b5d4099; 0x40e0ab73; 0x4eff963c; 0x57150ce5; 0x10d4eb59; 0x4359dc39; 0x73f97336;
     0x7f73a89d; 0x0f31589a; 0x15788eab; 0x0fc34a01; 0x0f98c3fe; 0x5ab7a35f; 0x12fe6e3d; 0x1d34aee1;
     0x475a20ab; 0x05301b81; 0x3731b197; 0x141b061d; 0x02038eda; 0x3b2cf5a3; 0x22e8062b; 0x1ff7ba7f;
     0x5fd38a7d; 0x4644f9f6; 0x0718efae; 0x0f62d3f8; 0x7fc2f703; 0x7df801c0; 0x0423e00f; 0x5fcb8116;
     0x311c0a60; 0x2157315d; 0x05faed90; 0x5c6cacb3; 0x32ce6062; 0x795d50ba; 0x2a2331f4; 0x670dc594;
     0x2b05a2cc; 0x1958b6d5; 0x155a60f5; 0x21f0024a; 0x0e1159f7; 0x070d4827; 0x5e7a9944; 0x4ffd9be4;
     0x3fb2a1db; 0x5a519d74; 0x2b645342; 0x5f5fde78; 0x5728a0f0; 0x4b2177fe; 0x091bd43c; 0x6999e5f6;
     0x2056f645; 0x1a5fb634; 0x209f36ea; 0x50082395; 0x5663153d; 0x63808836; 0x13e1be53; 0x4efc5721;
     0x269c0fca; 0x6bf2ad30; 0x02b2e99a; 0x5f2657b4; 0x014c218d; 0x3e1a5eb1; 0x210d8e95; 0x3b32dbf5;
     0x2b6f19f6; 0x048a909a; 0x389fce76; 0x57048b12; 0x4b7c6d43; 0x363ede02; 0x41249b9a; 0x2fc0a4f2;
     0x31625d9f; 0x747ea635; 0x128b3a0e; 0x4700b3e2; 0x147fa1ca; 0x6449adcb; 0x6be45a59; 0x688c43ee;
     0x78ef5fdd; 0x3916622f; 0x03a4bb73; 0x3682c7e9; 0x371ab2d6; 0x5e87cefb; 0x2c0bb950; 0x1c14de12;
     0x607b2f61; 0x58fb476f; 0x5aa034fb; 0x76e509a0; 0x6d6a2508; 0x02e8d7c7; 0x6458a188; 0x1b8c7be8;
     0x09a7683c; 0x2fd349ec; 0x7630c840; 0x78fc6ac5; 0x75119dd2; 0x6e792bcd; 0x5e9b52f8; 0x1b16de3f;
     0x4fcd7094; 0x05aef895; 0x700ff99e; 0x20aa071c; 0x0fb4254f; 0x0db6fb1c; 0x220d58de; 0x75391864;
     0x58d59542; 0x7e4d0dab; 0x70cc4e3a; 0x0e2f2725; 0x5c5fd8cc; 0x7b3e3697; 0x62b63527; 0x4efc96f6;
     0x3f03c88b; 0x02bfdb72; 0x30248715; 0x5b6285e3; 0x5201d939; 0x34c35303; 0x65310694; 0x519fa01c;
     0x2839e9b5; 0x769b0f9e; 0x251ff9a0; 0x306103e4; 0x6cc53cba; 0x2d530753; 0x6c98e796; 0x2fd26a6b;
     0x2027f625; 0x5ec0e015; 0x5f2fbd3d; 0x36b6a8e5; 0x208dff84; 0x55667420; 0x4b169192; 0x59d85160;
     0x396783ab; 0x6b74b81d; 0x0740d9cf; 0x7de70e9a; 0x27b915d4; 0x4e2d92d9; 0x3b2e071e; 0x0cc2be38;
     0x32334433; 0x341533aa; 0x5d59ad71; 0x148ab82a; 0x0add2d85; 0x291d958f; 0x5fee44e2; 0x3e814418;
     0x1ab8fb28; 0x17f8565a; 0x3494e690; 0x642fe314; 0x4ea48a20; 0x4fce7b04; 0x7dc3232b; 0x0c838263;
     0x578bb576; 0x38701f33; 0x0b0a68c0; 0x71da5caf; 0x1986bfd9; 0x319e7547; 0x1380b906; 0x103d8ac2;
     0x33f4fd52; 0x614fe53e; 0x0fea2e1a; 0x67183781; 0x0ac0ec59; 0x3caa98fa; 0x38bec575; 0x05b18411;
     0x07e457e5; 0x19ecb3a2; 0x2a1a3af5; 0x0d7f9046; 0x76ff3352; 0x63510636; 0x204ea078; 0x7b237053;
     0x7f96c11a; 0x4785af01; 0x4dc92c50; 0x70ed7853; 0x7d4b5e2f; 0x4b5d3cd1; 0x497486e5; 0x6813b8c2;
     0x3e8d6ed9; 0x70b8825f; 0x71438e6d; 0x450c64b5; 0x5ad4d93a; 0x3fac9f4f; 0x5436658f; 0x226f4066;
     0x27b96a68; 0x5e87a5bf; 0x581afb62; 0x7414a28d; 0x70b32220; 0x13d3d7bc; 0x5d93920a; 0x305941f1;
     0x1958afbe; 0x285604d8; 0x783a3146; 0x6c5cf513; 0x76599d8a; 0x288377cd; 0x3ff643b3; 0x43d6dada;
     0x0919e8ba; 0x7b9a6fa2; 0x0e145ed2; 0x522a6e86; 0x7b6f40fc; 0x4b612a69; 0x2a21e223; 0x302e9211;
     0x16065716; 0x37ee58d3; 0x4c5237a4; 0x6ebca315; 0x35a3a4f6; 0x647d4526; 0x75c7c5e5; 0x2662e8f2;
     0x68cfc39c; 0x7f141e8c; 0x7f3d59b2; 0x713009ce; 0x3cfd65e2; 0x6d920399; 0x5df441da; 0x329a880c;
     0x58b0d566; 0x2025e86a; 0x5a599e7d; 0x59f7a6b5; 0x5bb520b8; 0x112dd707; 0x646e40ad; 0x15c53f46;
     0x16ddd3f6; 0x41772d65; 0x4d5063b5; 0x5ea895db; 0x4b7f922f; 0x60040783; 0x2b6decbf; 0x4f2155e5;
     0x0ebff517; 0x035eff40; 0x26d9e612; 0x2399883a; 0x440fcc24; 0x4cd2ab2c; 0x61348a3e; 0x72e1b0df;
     0x5f59bd5e; 0x151ffc78; 0x5d3737d1; 0x6808108c; 0x5fc2bac0; 0x163cca42; 0x370e9bdb; 0x416c5ff2;
     0x4751d730; 0x275d9319; 0x3afe320e; 0x5c4e9af4; 0x273d0d9f; 0x6df37842; 0x3b1bb279; 0x1a70c968;
     0x5ccaccdd; 0x5b11e53e; 0x6f09ab90; 0x33306bcc; 0x755ccc3b; 0x2d69f110; 0x45ce1ee4; 0x55b3560f;
     0x4b61cfa4; 0x4c86edcf; 0x37a63fd6; 0x7b89f077; 0x065fbc2f; 0x55dc5de6; 0x12416326; 0x1e6dd589;
     0x6816c403; 0x1edb61c3; 0x0d7eb97b; 0x0e14e002; 0x7087d1bd; 0x1474bd1e; 0x728abe4f; 0x3875af57;
     0x1a3e2433; 0x38aafecc; 0x1165fa66; 0x6e1f2b77; 0x34df02db; 0x34c5ace6; 0x16d21bd2; 0x2a576193;
     0x50e01de9; 0x4f4c2042; 0x28ccd193; 0x0118b0a8; 0x2d1e1194; 0x0b186e1b; 0x591b6182; 0x5b1669a3;
     0x3dc87988; 0x4155ad19; 0x0b7da189; 0x5d66a048; 0x1b8ffa9d; 0x3d9ab968; 0x215e567f; 0x4fcc1d06;
     0x3ecd0eda; 0x2b096b5b; 0x4b0e6ba8; 0x5d9941de; 0x24d44c38; 0x2815a91f; 0x12f4ffef; 0x4eb8a212;
     0x338e301c; 0x7c080c07; 0x24b00562; 0x1bd4920e; 0x383cf62e; 0x48564f90; 0x587a971c; 0x39bfcf07;
     0x4fd05db8; 0x29cc58c5; 0x21a5c4ea; 0x553b0b81; 0x0aee1786; 0x6b0db236; 0x3afc25df; 0x41e03234;
     0x77191114; 0x6cc3c2d8; 0x0aa3edfc; 0x099265e5; 0x132ab3fd; 0x72fbabb7; 0x7c520d2f; 0x29217994;
     0x5f50abbe; 0x240f8680; 0x4d42cf9a; 0x274f34ab; 0x61dfe105; 0x36ac1134; 0x7b135c87; 0x086fa1ae;
     0x5f251fff; 0x52bffc95; 0x78ef3d0f; 0x7f3319fa; 0x78cb6ae4; 0x6235bc0e; 0x0884d04a; 0x1f488ebb;
     0x228dd756; 0x417faca3; 0x1b602e81; 0x34edf07d; 0x2cbc229f; 0x5af39841; 0x07f861b7; 0x7d7497d6;
     0x2daedfd4; 0x43eda3ed; 0x5c715918; 0x5e4bd046; 0x405413a1; 0x3ebe920d; 0x6bfe6632; 0x13e7759b;
     0x11007468; 0x59275594; 0x0b500dc8; 0x1c720f77; 0x49774070; 0x3d71707c; 0x23e1afa9; 0x1afaca8b;
     0x24c4a16b; 0x605ba779; 0x59abca85; 0x537d1bd0; 0x41625cd7; 0x34f14a62; 0x6a706b97; 0x0dfa68ab;
     0x5354be5c; 0x01f76c34; 0x74e97d1d; 0x56803b57; 0x01b6d6b1; 0x7baa3355; 0x5af3a463; 0x394eb382;
     0x7d3a52a9; 0x1ce19b11; 0x13da49a4; 0x3330f7cb; 0x39947b74; 0x380fef57; 0x67ac0697; 0x725e425e;
     0x52069f4c; 0x5a91ca8c; 0x2c89989c; 0x270e7de1; 0x39e940ca; 0x467147c7; 0x02f161dc; 0x1cc1c7af;
     0x5870717a; 0x5a4841a6; 0x29b97166; 0x2155c4bf; 0x0fb22b7f; 0x30ebaf33; 0x75572e79; 0x71995cdf;
     0x635dc4f1; 0x47223f92; 0x7c1160be; 0x42c9e6f4; 0x65d93ca4; 0x5cb4b3cc; 0x59f452fd; 0x184cf6d2;
     0x0c29661c; 0x3c302af7; 0x427eb6e5; 0x73647e5f; 0x72fb07ab; 0x3f4fd2b6; 0x33285d8a; 0x071426dc;
     0x69bf1112; 0x4e8d1254; 0x5716e046; 0x3ebdaad9; 0x15bdcdf4; 0x4d5afbe9; 0x41c4891c; 0x7635ab8f;
     0x066d1c1a; 0x082ef13e; 0x3cdcfc51; 0x2e462b1c; 0x7855f1b0; 0x6496bb72; 0x01157f67; 0x54f7a611;
     0x62ca0f98; 0x4864d6e5; 0x78f4ff3f; 0x1941e5f8; 0x22ffc525; 0x76d259dd; 0x7b2db048; 0x2c46ddae;
     0x50fce5c8; 0x47e92e97; 0x6e5a3e1d; 0x20383925; 0x7623cd15; 0x42542320; 0x65269620; 0x4227b7dc;
     0x6fca8347; 0x2b4ac80f; 0x287e7607; 0x2356dd26; 0x53141027; 0x3da0576c; 0x4dfcedc9; 0x38cf58f2;
     0x3ac5337d; 0x4e091bc2; 0x62771f1c; 0x3022c185; 0x67be0e90; 0x20cbb7d5; 0x49e2e0c7; 0x4163191d;
     0x0d619edb; 0x1d62e802; 0x7cfe0882; 0x303d09e7; 0x6efab1d2; 0x07f7b2f0; 0x0c5168ec; 0x48db4cf0;
     0x47cf41b8; 0x41d12d22; 0x596dae08; 0x70225e74; 0x205d878c; 0x403d2c15; 0x1d84a46e; 0x74edbda1;
     0x22d550fb; 0x09330dcd; 0x369f48ec; 0x01ef6b42; 0x02ffb8d3; 0x236cfc79; 0x5838b102; 0x3baed070;
     0x2019d2c4; 0x40b28871; 0x0421ae21; 0x0e4a8cb3; 0x17f9a913; 0x576b1e6d; 0x0323e2e8; 0x36e74609;
     0x094b0597; 0x75c1ab66; 0x389e3214; 0x150b3fd6; 0x6c1480fb; 0x4758b86f; 0x45fc6e63; 0x7c36a108;
     0x2c9417bb; 0x20796197; 0x3dd882a4; 0x5dd2c328; 0x447a6d63; 0x1d50d626; 0x68a53e91; 0x687b34ec;
     0x297c784a; 0x17c15b30; 0x3eaf2496; 0x03fcd529; 0x111db78a; 0x1bc78587; 0x73b98903; 0x5124c094;
     0x0317085f; 0x22d73489; 0x6f0a0d7d; 0x4a3ea284; 0x72b3e3c1; 0x45540042; 0x2f1aff15; 0x1184de83;
     0x44efcf99; 0x100f10eb; 0x0a60989f; 0x393d3243; 0x5c69e55c; 0x0f053ae1; 0x1c36e3ad; 0x08220d7f;
     0x18eccf5b; 0x6956238f; 0x43009e99; 0x3eb6846c; 0x3ba8d2fc; 0x72176e61; 0x58f2b1c3; 0x4f09b0ef;
     0x4076272b; 0x0269c653; 0x6a750af3; 0x74f73602; 0x68f0404a; 0x64554782; 0x3bbd5eb3; 0x7a3b8098;
     0x70f70d13; 0x03767dfe; 0x7bcd3d70; 0x78c464b4; 0x5c1df256; 0x4b0f0b32; 0x0142017f; 0x5afaf3d9;
     0x7b34abc7; 0x3d305635; 0x0b3e6c87; 0x7ed8ce87; 0x526bb7d4; 0x0866e015; 0x0cf6bb75; 0x5d619a5e;
     0x7dd19740; 0x11951079; 0x4b7015b9; 0x73c60eed; 0x59bf64da; 0x4c139246; 0x06bce054; 0x552c2027;
     0x76498f60; 0x1716a8e1; 0x00bc683c; 0x0bc4d2c3; 0x38dfb361; 0x5290432e; 0x3533a783; 0x2b034708;
     0x13dc6bf1; 0x67adb4df; 0x454c1c6c; 0x7fa89e01; 0x66d75d01; 0x318cc638; 0x5c176262; 0x007a5160;
     0x55b1eb82; 0x36061fef; 0x4683bc01; 0x3bcb2864; 0x394e4d7d; 0x1bffb5ae; 0x3c33af44; 0x253641ba;
     0x76359470; 0x52f351d2; 0x37f0ce3a; 0x715b4006; 0x509867fc; 0x3ec1b864; 0x0c189992; 0x73719dbc;
     0x0f004be8; 0x2f731c15; 0x3b9de5d6; 0x4a9728ff; 0x29e41425; 0x791b4b80; 0x5b78b814; 0x695ab9e0;
     0x55c50a32; 0x53c1bbd3; 0x74386af5; 0x65f3ab2f; 0x6b167e16; 0x61191927; 0x10d36ee7; 0x714c72d7;
     0x486f1761; 0x508cb2b0; 0x5b118765; 0x7546714d; 0x7795161d; 0x16f1ad1c; 0x40d72d51; 0x138ac72a;
     0x71c1c32a; 0x09ca3bc2; 0x40c607ca; 0x60667159; 0x151bd20e; 0x3f16109e; 0x7f49a924; 0x2caff0d4;
     0x698a402f; 0x468590cf; 0x3c719bbe; 0x00228d4b; 0x7febf49f; 0x61ee0a75; 0x6321506f; 0x7e5a5ea7;
     0x1690b8ed; 0x7d7fe897; 0x6fb83917; 0x1610c69d; 0x469be633; 0x05878167; 0x77d2e01e; 0x140a18e5;
     0x3c7d25f1; 0x14d06d1f; 0x402cf8aa; 0x781e28ba; 0x3c5d3330; 0x26751668; 0x04fbfd67; 0x41d6587d;
     0x7850179e; 0x51b7ef33; 0x27752dbc; 0x799b69c9; 0x1af81a9e; 0x0fb7a7f4; 0x4ab85598; 0x09d68c8b;
     0x0b939aab; 0x4752a963; 0x1b900522; 0x3efc8779; 0x4fac56ae; 0x2aaf8789; 0x74e8d409; 0x35f9d081;
     0x036ee442; 0x05999340; 0x5b67e77a; 0x08062ba7; 0x29a208d5; 0x52a95082; 0x38d3d953; 0x5b3c66cd;
     0x6f593ef2; 0x471ff614; 0x28a60777; 0x6b55f098; 0x05fb5359; 0x7324cf10; 0x064465ee; 0x7b107ea3;
     0x1963cd39; 0x5ab248e3; 0x282ba636; 0x551e4b2a; 0x714c1e7c; 0x6587c6b4; 0x5e73045a; 0x78bdc91c;
     0x78e0ba73; 0x2298d184; 0x1ba9c9a1; 0x6ac60d43; 0x27812021; 0x3ffe2d17; 0x094593a6; 0x7de3404d;
     0x757fe17c; 0x5d952ee2; 0x5776a4ae; 0x361a1673; 0x104ee7ab; 0x1502c4d3; 0x74de8a10; 0x58e7e8b9;
     0x26a5b231; 0x59ce3889; 0x7629a6c2; 0x4464cb5d; 0x1afb39b9; 0x0afa2a10; 0x031a6122; 0x2b5e7c50;
     0x79d65f90; 0x5abfbac1; 0x4a6fbc57; 0x0cde04bf; 0x0bfde841; 0x3ff2dff9; 0x1ee54dd1; 0x0a65bcc6;
     0x62e15e0b; 0x122dc57b; 0x09b29d86; 0x20cf6d34; 0x0a5811dd; 0x0dadcc4e; 0x5f818b00; 0x20ca4cc3;
     0x5c444f38; 0x0a4e0e63; 0x1638e084; 0x5ec26021; 0x06d2c5ba; 0x04ea3064; 0x65638c12; 0x1e833e0d;
     0x0e5d320d; 0x49b44952; 0x3ee119d6; 0x6e7199ab; 0x6424104a; 0x61306969; 0x7819999b; 0x171a568a;
     0x797f1323; 0x2d94e279; 0x21a50a1e; 0x02d62334; 0x0fc63b20; 0x718fc024; 0x578d864e; 0x064affbd;
     0x50b68ff5; 0x7c4a4af6; 0x24319292; 0x602f25c2; 0x07a8557d; 0x3ad7bc40; 0x505ee2a3; 0x3f20e2ef;
     0x7c529cf1; 0x3015a272; 0x1be8807d; 0x665a9077; 0x270ff78b; 0x2b764596; 0x763c0fb1; 0x6650c8bd;
     0x1d9b2cc4; 0x5ecb33ae; 0x0a6724da; 0x36fe2e6a; 0x7a285123; 0x394717d8; 0x14047f9f; 0x6268a2d8;
     0x07ec7113; 0x04799ec6; 0x5ff7cd9a; 0x024d8545; 0x14308a4f; 0x20e7d1e1; 0x5ee6e563; 0x1c6fdae8;
     0x1946b79c; 0x26b2387a; 0x2a11cb55; 0x444c27c9; 0x69d8fe95; 0x4c611d51; 0x0816479c; 0x38ad41f8;
     0x5660def2; 0x6fdca975; 0x1b96c41d; 0x36441937; 0x36fd6ccc; 0x4ee7f8d6; 0x15dbf2c8; 0x4395bec3;
     0x382111dd; 0x71e67925; 0x7351f44e; 0x1068cdbe; 0x4cdbcc4c; 0x43b26fa2; 0x0c895704; 0x6ced025d;
     0x3bc0ae25; 0x4afb8f9b; 0x584613ef; 0x0777010d; 0x1e462994; 0x58d451eb; 0x387a8662; 0x251ed2fd;
     0x236b9150; 0x1d0197b7; 0x3f4acf93; 0x2014a7b5; 0x103273e6; 0x60304388; 0x41774766; 0x36729bb9;
     0x48431b97; 0x03f653d7; 0x22d3c70f; 0x5d880568; 0x1034119a; 0x3ac53697; 0x7598cc9e; 0x7b2c0d6f;
     0x01606ffe; 0x65ea5131; 0x3940dde1; 0x53015da3; 0x4153764a; 0x72e1c1bc; 0x7ebeb43b; 0x3c10cb7f;
     0x486fbacc; 0x4ff3a299; 0x2b777abb; 0x5cf9813c; 0x6843ced2; 0x1c9cbc25; 0x4d299ffa; 0x0d4367e5;
     0x78bb112e; 0x54eed40c; 0x77d674f1; 0x20dbd271; 0x58e10a25; 0x3124802b; 0x4aaf56dd; 0x04c34d25;
     0x1f96398b; 0x7b1af1db; 0x63df030a; 0x2b8c7ae1; 0x286f42de; 0x6627820d; 0x42cf7172; 0x5c7c0ca8;
     0x7389adcc; 0x643890f5; 0x36a6dc0d; 0x3eb800f4; 0x7a31883f; 0x4e9c2f98; 0x6e9080a5; 0x263297ae;
     0x3336f207; 0x4cae2aa7; 0x633082b9; 0x32f8e18b; 0x1f7dc004; 0x4ec4d4af; 0x092fd3dd; 0x3164eb5e;
     0x5d685a75; 0x32e3944d; 0x56e20ff6; 0x007b50fd; 0x2c2118fc; 0x5ad84c43; 0x3d17990c; 0x6872712f;
     0x45f9607f; 0x03ea6eec; 0x72d85d67; 0x139f72aa; 0x760c19e5; 0x34885dd0; 0x4252850d; 0x1a0da774;
     0x52e9369c; 0x1a1dd232; 0x6ccf489e; 0x009e1096; 0x46ced2a1; 0x3ba12484; 0x5c21425d; 0x34f5a7a3;
     0x385d748a; 0x28b29065; 0x4d8271e0; 0x1a2c9876; 0x3bf8c431; 0x31eb8f45; 0x2f77fad7; 0x5ca16376;
     0x526ef6a3; 0x5aa366af; 0x6a2c574c; 0x49d91d38; 0x03581d3b; 0x5e24d7fc; 0x69cdd94e; 0x5d0b41b3;
     0x0ed36ab3; 0x1bc8b563; 0x03d9d18a; 0x66713396; 0x45e5d08c; 0x7e0d9017; 0x50eaf619; 0x00d7576f;
     0x08a636e1; 0x0af0b667; 0x6613e753; 0x546a94a6; 0x43ad9c23; 0x7c4d9143; 0x1c81bc26; 0x6fbe92b6;
     0x1e851e23; 0x05bcc05b; 0x011b872d; 0x615cbcf1; 0x6e36b962; 0x33d4af29; 0x64b23d7e; 0x63fa8191;
     0x688ba7e2; 0x64b68c20; 0x667f0a0c; 0x3df65c9a; 0x6c6f6308; 0x697cc6bf; 0x16947a5c; 0x339b9d79;
     0x71a473e4; 0x0a670cbd; 0x041e1e18; 0x4a8fcf6b; 0x7e5a1cc2; 0x06db51a2; 0x430c2c36; 0x26efe2d5;
     0x392b1192; 0x4944f7cb; 0x57925973; 0x5354e955; 0x78101f3a; 0x5d7acf71; 0x111eb3cc; 0x67b2e0fe;
     0x4cb639b9; 0x34c731c5; 0x359669e3; 0x7be3d5a5; 0x2b255d76; 0x6dd9bc91; 0x37b134a8; 0x4023f2de;
     0x29010e92; 0x7f2e73e9; 0x4554253f; 0x55db9b84; 0x6e1fb800; 0x4d7c486f; 0x00c95b8e; 0x02fbbb08;
     0x37bd54b4; 0x4d4c466c; 0x03326852; 0x6f1c57f4; 0x650f5828; 0x257bc0ed; 0x59d563a7; 0x39cf6e30;
     0x77ed0220; 0x74072c80; 0x4ce4d321; 0x1e3b6b3b; 0x11b7cbe7; 0x00b58f25; 0x1fe92642; 0x5d88b299;
     0x74804042; 0x6e6294c3; 0x213a7157; 0x49868796; 0x1c00e2ff; 0x4ab648ca; 0x5255d948; 0x10e59486;
     0x3c62c097; 0x1e38337a; 0x432a7712; 0x749e9f34; 0x286ee75f; 0x491070cb; 0x547702e5; 0x4f087ec0;
     0x7512c0e1; 0x6a74ebc1; 0x3850161e; 0x40ef2da9; 0x4d415d52; 0x49260f3d; 0x36c63884; 0x0c9965d7;
     0x00d4cea3; 0x575c11b7; 0x38b779be; 0x6cb263e4; 0x25aed54a; 0x3cca9230; 0x0f394190; 0x38a43bb4;
     0x3a7df307; 0x2f75f3c8; 0x4535cd1a; 0x263373e3; 0x087c86f4; 0x4e6fc23a; 0x7615bdef; 0x7aba8e43;
     0x39ab54d3; 0x78938889; 0x0c8994eb; 0x161d551e; 0x1feb13cb; 0x6fa7235e; 0x7842e8dc; 0x5be265fb;
     0x3dc1bbd8; 0x2b42dd8b; 0x035c2fa0; 0x6c5bfd71; 0x6fb03d70; 0x5997ffb2; 0x22799227; 0x549eea90;
     0x63ad4ef2; 0x4e193331; 0x2048283b; 0x0a0f9bc2; 0x40333317; 0x6c2d81e9; 0x38086dfb; 0x00b4f714;
     0x576d2546; 0x53e24182; 0x081f2198; 0x53454457; 0x348b6001; 0x34da3a3d; 0x437b3e05; 0x3cc656f2;
     0x4824b0b7; 0x4f82fef9; 0x7774a615; 0x59f42d36; 0x1e874dff; 0x75923161; 0x3d4d19cd; 0x1fdd6839;
     0x2d85417c; 0x0668f243; 0x66943d31; 0x2f6893a1; 0x66d4a308; 0x4809c375; 0x1dddb6aa; 0x11283c04;
     0x5f3f4031; 0x25946ece; 0x610345ee; 0x46571018; 0x4af0b0ab; 0x2a4656cb; 0x6a0ed0aa; 0x3ca5d22c;
     0x05fa722c; 0x08687b89; 0x1deb2bc8; 0x2cbcbb12; 0x32ea36c9; 0x335aa1cb; 0x4d432e3c; 0x58590325;
     0x6528ba15; 0x7a8502d7; 0x5eddf67f; 0x05c7f2f9; 0x55c44f6e; 0x60eff105; 0x4e653ddf; 0x52443246;
     0x6d6c2302; 0x1d407d23; 0x385bb9f5; 0x76bb9251; 0x467748a6; 0x4f7598b9; 0x03951874; 0x1ce29e87;
     0x29f981be; 0x656475e8; 0x77e577bc; 0x38b592d4; 0x64bc9138; 0x1ce0e8cc; 0x5cc0dee4; 0x7bed28df;
     0x7d4b833e; 0x372538eb; 0x54f9ab5b; 0x60219684; 0x5edd67dd; 0x4152a814; 0x2a17dab3; 0x795c87e1;
     0x0318f771; 0x69212f49; 0x5ffc8f3c; 0x31b6efac; 0x0ddc5b0e; 0x59f2e97e; 0x77e19587; 0x702388ad;
     0x0fd80d57; 0x56481a14; 0x24798f46; 0x0e7cddcb; 0x7f5e87d5; 0x6b7f8044; 0x720a8827; 0x5dce12a9;
     0x37b27ab3; 0x3f85d2fd; 0x35bd5f1b; 0x28f31600; 0x6926167d; 0x0c92a104; 0x79d44637; 0x40b00545;
     0x364d1499; 0x4f973b47; 0x2707a4a9; 0x0fffef9d; 0x62e9f1b0; 0x1ccdb9a7; 0x0b269175; 0x41988ee1;
     0x6b019c43; 0x7c6a7660; 0x7fe50937; 0x2cf76d4f; 0x48a41b5b; 0x675ab10c; 0x3269e641; 0x508df653;
     0x25400cba; 0x01ae510d; 0x2abd4152; 0x7d5a624d; 0x18216fbb; 0x0404a3a4; 0x202d1f84; 0x257531b7;
     0x4358f8ee; 0x07cbca2c; 0x25d14f6d; 0x52eb267e; 0x20dc0c7c; 0x188d0058; 0x415d8702; 0x30ad0167;
     0x6fd805b2; 0x24efd4c4; 0x55d804a6; 0x63e75f27; 0x6f76851d; 0x1b04ff6c; 0x2914369a; 0x26854920;
     0x0d382bf2; 0x384f3fa6; 0x3a6e77b8; 0x5c91624d; 0x158280df; 0x18ebf672; 0x67944eaf; 0x473fb56a;
     0x4d627d87; 0x2240cf0c; 0x27dfee23; 0x6871444c; 0x4769f7b3; 0x6ac7a64c; 0x4813975a; 0x321345e2;
     0x7a654917; 0x00f2af9a; 0x6b7b4c8f; 0x4387212d; 0x29989b35; 0x62a22337; 0x4d91eeae; 0x7457b82c;
     0x18f8e953; 0x78517e81; 0x131ae695; 0x48fd27e2; 0x15776b81; 0x02658de4; 0x042c331c; 0x361cf324;
     0x5136c5ab; 0x1af02f55; 0x3f782342; 0x22e67f36; 0x2eee0c0b; 0x75d5214b; 0x7497be44; 0x750711a7;
     0x43c429e0; 0x66f25e7f; 0x3ab7b867; 0x4bb98074; 0x2110fc81; 0x7e9b9a25; 0x2ccabb7c; 0x3cca79d3;
     0x191683b3; 0x681287c5; 0x172312a8; 0x5a8ac1a4; 0x776faa04; 0x75d0373b; 0x2baa1086; 0x3a708f00;
     0x677f6df8; 0x19bb4d47; 0x03551e77; 0x004ec921; 0x3e197549; 0x3bbff427; 0x23f33763; 0x02b04e5e;
     0x34303c5a; 0x571ecc5e; 0x361ae0cd; 0x579d66fa; 0x3c0f23e5; 0x328119d6; 0x2acd416b; 0x118d3785;
     0x34e5f9a9; 0x1e97e844; 0x267728ed; 0x404665fd; 0x4eb07439; 0x3c3e0e3e; 0x47f30741; 0x0c2a8105;
     0x5f821242; 0x6a886e36; 0x4b060219; 0x7c033fc5; 0x6d8e88e7; 0x743facfe; 0x1311fb48; 0x4b1d6b5a;
     0x61c9e8f9; 0x247acfac; 0x6eebe842; 0x3a50620c; 0x6ba34d82; 0x61d707d9; 0x7a28073c; 0x6f3018bc;
     0x125edbf6; 0x600c6170; 0x1fa675df; 0x04052be2; 0x5cfd2fb6; 0x010b46d6; 0x7c6b88af; 0x42c44e47;
     0x3239873d; 0x6d55a8e4; 0x68d33f97; 0x59dd1f03; 0x3a1cd62f; 0x71413ee9; 0x0e04b8bc; 0x3fa9de50;
     0x1bfb9860; 0x1d320046; 0x0d5c5393; 0x1974b193; 0x20f9789e; 0x58512422; 0x029c6c03; 0x306d2867;
     0x6bfa436e; 0x4b8194a3; 0x1f2ea268; 0x30493c41; 0x394ccebc; 0x7f02ebf1; 0x2869c79d; 0x5ba3c913;
     0x522f913c; 0x135e0005; 0x317f53c2; 0x10ae0831; 0x2d1fb996; 0x47527af5; 0x73347852; 0x3059f561;
     0x736ac6ef; 0x245b55b5; 0x0aa92dca; 0x31468e40; 0x72c505ad; 0x5c75ccf4; 0x6e6a958c; 0x71f6fdcf;
     0x19d71bc2; 0x46cd9367; 0x55d27242; 0x00fc95cd; 0x59e4d26f; 0x6e13cda8; 0x7a0b8045; 0x149715b8;
     0x1c44d762; 0x5d241a53; 0x35633056; 0x4e6bf0fc; 0x371b31d7; 0x125e4978; 0x2b31a480; 0x2cce79eb;
     0x74365b5f; 0x0e61a254; 0x1b7667ff; 0x2e13c5e2; 0x21d18d30; 0x0dca0f44; 0x3c2d4239; 0x72674960;
     0x43572e2e; 0x1653b074; 0x67cd9cb2; 0x26997999; 0x284ca5a3; 0x45445635; 0x408a0fd7; 0x1182bec5;
     0x3908bc9f; 0x654aa774; 0x750525bf; 0x3935d809; 0x16503333; 0x12850179; 0x3bbe8863; 0x0fcff8e9;
     0x477ee5a8; 0x57112c2d; 0x7f692857; 0x5dd1fec4; 0x007535b1; 0x70e5a633; 0x38abdc8f; 0x794b02da;
     0x7c4e0ea5; 0x5c162e99; 0x2e6b7e2c; 0x150a36a8; 0x224a1db7; 0x310308fb; 0x758ca33c; 0x28ad8864;
     0x5c270857; 0x21451b86; 0x1fc6b1e7; 0x231cd080; 0x40c3518b; 0x6bd07585; 0x274185b2; 0x499f9e12;
     0x39c6fedf; 0x67f01bf1; 0x341821c9; 0x405f1855; 0x08f10b1b; 0x5f5f0262; 0x7f14fee8; 0x314b130b;
     0x4e87295b; 0x47b49179; 0x2eccaf5c; 0x142aa4cd; 0x44f779b8; 0x09fb922c; 0x05c93179; 0x52efc022;
     0x19ef07b2; 0x6de67c68; 0x7c4aa130; 0x75a1c155; 0x0eb4c80a; 0x2c25ae21; 0x18d20df4; 0x3e7935a8;
     0x11e9546b; 0x25537b53; 0x737910b3; 0x7f8dae88; 0x6076f39e; 0x28288087; 0x1b678118; 0x7078f694;
     0x010aca41; 0x09b15eb2; 0x6a1ca66f; 0x024f8085; 0x6b00405d; 0x7c4f0753; 0x3a111ddf; 0x03dcaa86;
     0x3bd5c6e7; 0x6240a6f3; 0x6c69c44e; 0x786c4cbc; 0x7d8c61a8; 0x716b6fe9; 0x51c27d60; 0x2450553a;
     0x7e333184; 0x49dd2551; 0x6ba8a2cf; 0x5b486e61; 0x0f3121c0; 0x0cc694fe; 0x7628f6ad; 0x556211d7;
     0x3e46d924; 0x6ce479fb; 0x3ca445df; 0x6142a947; 0x6a5a91f3; 0x44a51ebb; 0x33c35391; 0x4d8175a3;
     0x2fbb5cd2; 0x60ddf645; 0x060a9171; 0x59ec5f80; 0x0b3b392f; 0x134dba9d; 0x342f281f; 0x4c535e20;
     0x26f41f77; 0x477a4d16; 0x5a5b9810; 0x20ae0d21; 0x7e0ccb41; 0x535de6db; 0x7ab3e2eb; 0x13b8c347;
     0x5175a15c; 0x57538688; 0x0a25008a; 0x009ee4de; 0x4e2d4aa4; 0x2a159a8d; 0x3ab4eee7; 0x791419cc;
     0x1072bc3e; 0x4f1b9e62; 0x41dd7ae2; 0x44f993fb; 0x12ff1efe; 0x3c4afe27; 0x767576da; 0x1c176ee5;
     0x7d310e21; 0x61415445; 0x6e37b73d; 0x2482517f; 0x15868c3e; 0x04daf8ab; 0x7ca27925; 0x39536577;
     0x46185d24; 0x5a114757; 0x530a511f; 0x274130fd; 0x145e22f1; 0x258655cf; 0x7023c88b; 0x6ffba3f5;
     0x3118f256; 0x75ee4160; 0x6e985e61; 0x6729cdf7; 0x62294668; 0x6e38abde; 0x3f47f6f0; 0x44bb2221;
     0x050e166e; 0x7c1deea5; 0x5e8e9293; 0x03ab5438; 0x05e99c3f; 0x27dc38c6; 0x69217abd; 0x67c6c5d0;
     0x306eeba1; 0x151f12e9; 0x10fdc3a9; 0x43001a1e; 0x403cb6a0; 0x1c759bae; 0x739ff519; 0x18a2a2c0;
     0x77b62a26; 0x101dc187; 0x7259aa8a; 0x053f6d7d; 0x75cdf6d0; 0x34954961; 0x43c53d1e; 0x2f19d30b;
     0x31cb4386; 0x448c7574; 0x2b7d8cda; 0x43b17ae2; 0x6ed3a3e3; 0x662006ac; 0x048c5a87; 0x50f77cd3;
     0x59adf484; 0x04b82a8e; 0x0601786a; 0x69a80da7; 0x030b1305; 0x7f224b91; 0x554d6561; 0x0872fb31;
     0x32439cab; 0x6cfca818; 0x7e2e85d1; 0x1e237db8; 0x6b6eab85; 0x2c03a9a2; 0x056ce66f; 0x15ec2004;
     0x4c6334cf; 0x4fb34228; 0x2816a584; 0x0e4adfba; 0x470dce06; 0x74cc8552; 0x78634d90; 0x672ac1f9;
     0x2646f8e7; 0x7487b232; 0x3d72f037; 0x7798df83; 0x0068ccb4; 0x7867cb0f; 0x70ee0e84; 0x535d3313;
     0x2fef1f7f; 0x597c7553; 0x627aa78a; 0x1fcaf9d8; 0x20d1eef7; 0x5f21e7db; 0x57bed229; 0x453c3383;
     0x6c182950; 0x1ba95018; 0x49446cad; 0x6d2ef17b; 0x0b839ab0; 0x635f584e; 0x4aec70e7; 0x172f3e69;
     0x01c62d6a; 0x4ff4cdfb; 0x4481cadf; 0x265880c9; 0x63ea1aec; 0x17c9d8a3; 0x33901052; 0x4a2720fe;
     0x286cae78; 0x2c5b6d69; 0x4650f75e; 0x10205a46; 0x1f20ce1e; 0x649a139c; 0x74f7c9a0; 0x407ededf;
     0x12cc23a0; 0x59d3c025; 0x64f49735; 0x456d56c9; 0x1baefcaa; 0x188edc9b; 0x360aecc1; 0x722232bd;
     0x367882d2; 0x32e35078; 0x2e469dfc; 0x05401320; 0x4c524d2d; 0x5d0ae680; 0x75e7503b; 0x29cc2d30;
     0x1e71f7ee; 0x25549c64; 0x4bf6eed3; 0x17ae6dfe; 0x73022635; 0x0efccb06; 0x50b87ac7; 0x3c46da68;
     0x55b2ef85; 0x7d33dc30; 0x650d6431; 0x3daea8ad; 0x5b447185; 0x447d03e9; 0x657385e1; 0x6d8a45da;
     0x49f595be; 0x78cd8b86; 0x706abf47; 0x71451883; 0x7fc7794e; 0x391f364e; 0x0a0bbe4f; 0x6eca65ec;
     0x6963fabd; 0x6ce7b44a; 0x76c4c01d; 0x6584f0d2; 0x4c216813; 0x08f2e72f; 0x1ad03236; 0x1f77adce;
     0x1a6922cf; 0x59f39dca; 0x393e7956; 0x06331fa7; 0x4e77374a; 0x15519851; 0x12d3f1e0; 0x351f674a;
     0x44cf4d16; 0x4ef3af8e; 0x7d011e33; 0x01db9e9c; 0x567f44c1; 0x1a8b8075; 0x4f73fad5; 0x13d57e07;
     0x3ae94922; 0x710a944a; 0x6813cd59; 0x1ffc0cfd; 0x5fb9230f; 0x7999c2b5; 0x78c3cfdc; 0x784c7b75;
     0x025f4b50; 0x62df5793; 0x578bdead; 0x39e87c28; 0x1fdf4ccc; 0x44cd69e9; 0x105fd1ca; 0x019f50c2;
     0x60eadeb4; 0x5bf3b694; 0x4eacf638; 0x70c4f46e; 0x733e5315; 0x29ecb4b4; 0x4d876f1b; 0x57846793;
     0x402a0c6e; 0x6e8c1b7d; 0x4e023932; 0x2013ad66; 0x57427635; 0x77a4dfe3; 0x31f6f215; 0x22e0261b;
     0x2b76ccc8; 0x3e7f1ce7; 0x3d4eb95c; 0x1e2c3b97; 0x3693750b; 0x2b516ad7; 0x22618ad3; 0x2007c505;
     0x6d2253fb; 0x53e2b3d2; 0x2658f508; 0x14f15526; 0x156961c1; 0x5bd9aef5; 0x17fe663f; 0x13b5801f;
     0x2645bb43; 0x5bfc0896; 0x39b3ee37; 0x1882e0ae; 0x5494d6cb; 0x4f231480; 0x4f81b4b4; 0x4c6b8bcc;
     0x2a80a5fc; 0x0fc2863c; 0x7d0f5a9a; 0x12d391db; 0x63c62cae; 0x6dcd1abd; 0x56d756d2; 0x5ad69ec7;
     0x733316f3; 0x5d174dc0; 0x2b638370; 0x0791a88f; 0x3394a67f; 0x3b25e223; 0x54bfeab0; 0x4eede88b;
     0x2e4f18e1; 0x10a0cb69; 0x48bd6474; 0x142dba5c; 0x3c1fb91d; 0x20913df0; 0x4f222a10; 0x7eea83fa;
     0x143d9787; 0x234276eb; 0x21943fd8; 0x7afcf61a; 0x37cfc701; 0x0e91d3b8; 0x32bdd0ac; 0x2673b2b1;
     0x4f8080aa; 0x5a433931; 0x76272a40; 0x3be388d0; 0x4f138b43; 0x44c8cc46; 0x755e4526; 0x7d9bd3ef;
     0x655b571e; 0x4a2701cf; 0x606f42cd; 0x1a9ed0ab; 0x205da39e; 0x79ee468e; 0x42f18e3c; 0x00e6e10a;
     0x07594beb; 0x11ae0e7c; 0x6d130ff2; 0x6307d6f5; 0x65192e54; 0x3101acf1; 0x00948478; 0x55d579d3;
     0x5c02b3bd; 0x00ddadb1; 0x7402b81f; 0x5d86c02f; 0x61f95db2; 0x68c9131d; 0x225cd412; 0x06ed9363;
     0x0d171887; 0x0787ca99; 0x6e4720f4; 0x0b75bc96; 0x2a4c6f55; 0x0b731847; 0x65f47295; 0x697973a6;
     0x57d6ed42; 0x44e32a0c; 0x05df6ab1; 0x30ef124a; 0x4cdc50b5; 0x02a7b795; 0x48b89f58; 0x5a1e9b4e;
     0x0c1b9f86; 0x4853d297; 0x3b8bc64d; 0x4951ba00; 0x42f16a2e; 0x7d635dee; 0x2f56ada7; 0x30bfd935;
     0x575b1150; 0x54b60754; 0x28d04c1e; 0x1d24695b; 0x35684121; 0x19bda0b5; 0x5f0d06fb; 0x3bcc3882;
     0x56cf48ba; 0x2d9d19c7; 0x7a8ed977; 0x09804603; 0x21e2f1d6; 0x7e025120; 0x4c067df4; 0x6bf24efb;
     0x1dccf27e; 0x1587fded; 0x5520cf64; 0x22248ac8; 0x29e8c1f9; 0x6b201f69; 0x68ddcb5f; 0x35d6efcd;
     0x1d8be27a; 0x385f2231; 0x2faabb67; 0x2c15d7b7; 0x1ad06b4c; 0x63ab8f52; 0x58cbcb88; 0x22eddd1b;
     0x47c19e57; 0x325caf00; 0x3d838f8a; 0x51798027; 0x24783165; 0x030a63e9; 0x7381a5a8; 0x0da2c319;
     0x5152e6fc; 0x2ddc4572; 0x5606b7a4; 0x47957e28; 0x0c28840c; 0x510ba632; 0x6244bc12; 0x5fec5509;
     0x6978068c; 0x1f27ba34; 0x7d6bf406; 0x0b479ad0; 0x46d34493; 0x6a7a1924; 0x54e33d90; 0x013a999a;
     0x27133934; 0x73d9f31c; 0x2c8ad76a; 0x35dc746a; 0x1bf24116; 0x067e7042; 0x63bad7d5; 0x16261475;
     0x04e9e8cd; 0x7779d1d8; 0x6ccc682f; 0x32312028; 0x49c9917d; 0x1d5fd4ad; 0x196391bd; 0x6a38825e;
     0x12435a0d; 0x2f975905; 0x48c69a32; 0x22d1d253; 0x369008bd; 0x790bcf13; 0x3257fc0d; 0x0f14e48d;
     0x6c80d52b; 0x1a95edb9; 0x4c2d8230; 0x3459618c; 0x78d6d034; 0x6f425b82; 0x2b56daa7; 0x09a4965c;
     0x4738e054; 0x55550a95; 0x291e7b11; 0x4a725acb; 0x454bee1e; 0x2ad76572; 0x22f2da13; 0x6a3f6e38;
     0x4fd6f0a3; 0x0120446f; 0x0d532e8e; 0x135969d9; 0x30281a2f; 0x1237160a; 0x5fc81d86; 0x4a0efb08;
     0x05fcc265; 0x0c71fc77; 0x023020ef; 0x5fbcd58e; 0x500b9501; 0x46fb6317; 0x505b51bf; 0x70ebc1a3;
     0x3cb9a5ef; 0x4bd3e511; 0x284c87da; 0x5d531a40; 0x220d9231; 0x0c7c857f; 0x5733fad1; 0x2666f33d;
     0x6ea64545; 0x66ab684f; 0x5873e7d6; 0x2f2d3add; 0x381525f7; 0x41d24c04; 0x651173af; 0x5eba481c;
     0x314ec749; 0x772b3c71; 0x6ed78a03; 0x79a7fb37; 0x2032d4db; 0x511ce7c4; 0x363cef9e; 0x468b7aac;
     0x4751c0bf; 0x3fb2fcbf; 0x796da3c1; 0x217c1296; 0x45f87b04; 0x25200f29; 0x0895d813; 0x603f6086;
     0x099f1ac6; 0x49b93693; 0x10bcd12c; 0x38ea11fc; 0x67ec30f5; 0x17b565dd; 0x42b7fad8; 0x491f6d36;
     0x180d96da; 0x2d5a8222; 0x704f3401; 0x2f288917; 0x53eb82f1; 0x0a5fecfe; 0x63d03f7f; 0x0fe07be5;
     0x4af8cbc2; 0x1f2394fb; 0x0b618adf; 0x6f5559f4; 0x18f1e44d; 0x681a4a47; 0x431a3316; 0x2baf39c7;
     0x27c23482; 0x5db9f995; 0x42156bb3; 0x7866ac03; 0x01c3bfbc; 0x350a2a67; 0x1c5bfdbb; 0x292a5700;
     0x3e33186c; 0x1d33af1f; 0x7308e9b4; 0x765c61bc; 0x409e8f89; 0x14748453; 0x30e52509; 0x3f51a9be;
     0x7501b7ee; 0x04e762f9; 0x7e495118; 0x1463777a; 0x059500d6; 0x775552b0; 0x02673caf; 0x754212a5;
     0x78c34c06; 0x2121677b; 0x49438d47; 0x5087b578; 0x5bb7a20b; 0x2144ec27; 0x5aa22c1c; 0x54b84604;
     0x15619666; 0x5ff5ca94; 0x41b9f45a; 0x0034ddea; 0x5625be11; 0x7bcd62df; 0x0d16cbea; 0x58ef30f6;
     0x09fe0412; 0x0361b627; 0x1a7cf130; 0x649fa3b0; 0x0f7e0c6a; 0x5e0e4e10; 0x381cdd37; 0x64014603;
     0x540960ac; 0x1c38ff5c; 0x5bd27172; 0x7c0857b9; 0x34eeeb9e; 0x6d0b4699; 0x52d6e86d; 0x64128c00;
     0x7bda7fe1; 0x7bb8e215; 0x4d957340; 0x02acee12; 0x1884de8c; 0x5e6385ca; 0x334a34f7; 0x5795a7e4;
     0x214be206; 0x34600bed; 0x7c8c9411; 0x168aa5eb; 0x76e1f81e; 0x0a6e7e9c; 0x7f404bda; 0x7c204266;
     0x5aa70690; 0x3521e8fd; 0x7b7132ba; 0x030f3c8f; 0x4586a9cf; 0x18a8c511; 0x24ed1ba8; 0x0c5c200f;
     0x1a3006f9; 0x7bb6fa5a; 0x7acc15e5; 0x474e25a5; 0x39c79608; 0x16991d13; 0x0eb92dac; 0x4d4ebb78;
     0x35ae0456; 0x1471045b; 0x5d973841; 0x67ac9171; 0x56fb51a7; 0x51a11e37; 0x117061a3; 0x067ccf09;
     0x4b7ba5ec; 0x528b86c7; 0x40ded105; 0x5f72760f; 0x434e403c; 0x3af87594; 0x448e23e6; 0x3b691e33;
     0x10494bc0; 0x4733491c; 0x646d878f; 0x362ff651; 0x1fc40f74; 0x19daf33a; 0x47d8d4f4; 0x6e6f00fb;
     0x6b8a3dd5; 0x6eaee4b4; 0x5109c559; 0x28d105b7; 0x1afec99f; 0x2c9b84f7; 0x3f114f1c; 0x0b87f72b;
     0x2e14e12a; 0x6354cbdf; 0x4ec38c70; 0x63f08c96; 0x6cb346ca; 0x2432d028; 0x5df0d457; 0x4b76cc7d;
     0x3e38be0d; 0x70c5a954; 0x41550208; 0x680feb64; 0x2a82acc4; 0x7a9c85b7; 0x3324aff5; 0x5e141d55;
     0x70be8c9e; 0x68a30356; 0x5df89b7c; 0x3df6dfc3; 0x30f35754; 0x1f1fa925; 0x2207fd1d; 0x47effb71;
     0x108d3839; 0x7941b7a1; 0x302dc574; 0x05fef5a9; 0x4c342ec0; 0x04b0ce4c; 0x1309c9fd; 0x3f8e0c2f;
     0x79855054; 0x0eb3a21a; 0x684cd809; 0x290398b8; 0x3aa026c2; 0x17273476; 0x6fdf8b6b; 0x30ef722f;
     0x75eb6763; 0x0f8d430a; 0x39e14076; 0x652471fb; 0x0a45dcd1; 0x25e16b8e; 0x2fc8fdd6; 0x4826f750;
     0x6dc16eff; 0x67f3a5dd; 0x02ac5740; 0x7da61a61; 0x5fe9533b; 0x7e8c58e3; 0x797f3b58; 0x0ea20699;
     0x542e4c2a; 0x53e75ba3; 0x2ec97866; 0x3ac1ff3b; 0x22bfc07a; 0x45e73674; 0x0144065b; 0x3c0ec76e;
     0x78050acc; 0x47ecdb57; 0x1afeac12; 0x1542a90e; 0x26489bec; 0x30a168c6; 0x222c2688; 0x72431f1f;
     0x70c2d21c; 0x1e46a56b; 0x30869b17; 0x59de0a06; 0x40f9d6b7; 0x7b32d7e6; 0x444bc254; 0x64a5069b;
     0x799c4b5b; 0x3ca9163c; 0x3ef776dd; 0x62602e70; 0x00002679; 0x75569eed; 0x5b5f3f82; 0x5882a453;
     0x39c48214; 0x46525ef5; 0x657190e7; 0x5c948948; 0x749043c6; 0x06d4db2d; 0x2d1e29f9; 0x1f1b323b;
     0x0da6d6a9; 0x37639739; 0x1fc814b9; 0x379ff0ec; 0x3c5f9476; 0x1a5df0f1; 0x408a3f01; 0x2553af76;
     0x6c18addd; 0x2fd2d4d2; 0x14b1b50c; 0x1687dd88; 0x4a84da93; 0x2e83ab3c; 0x0b6ac942; 0x3ae9ecc6;
     0x266c8b7d; 0x756dd3d8; 0x4447e9f8; 0x62e25f8d; 0x0560a63f; 0x265c4749; 0x4e87260b; 0x3a0a430f;
     0x60671370; 0x4ff46bfa; 0x3372c0f0; 0x651f9d22; 0x7b1faf54; 0x76fcf0eb; 0x139847ac; 0x2a8e7a31;
     0x4fddde40; 0x2856cea6; 0x265b76ff; 0x469fd6fb; 0x44d9c077; 0x3e2d4256; 0x4733d4b0; 0x73d632ae;
     0x36e9999a; 0x5d9a4f8f; 0x21658e39; 0x2db4d86e; 0x767eec80; 0x6c36bfc7; 0x0dda4d1c; 0x6dda33ee;
     0x1b9261cd; 0x2f3a7795; 0x74d56676; 0x5184ef61; 0x5b25905d; 0x465b1aed; 0x3e766f73; 0x36b30a0d;
     0x45cc8798; 0x16d0ce43; 0x07286758; 0x3a59862c; 0x31d00cd7; 0x17d42673; 0x19bf81d3; 0x6385dc15;
     0x7e703d90; 0x0db0c974; 0x63203505; 0x1594160f; 0x029d1854; 0x0f0f9299; 0x3052e271; 0x6a7ffaf6;
     0x7e91fa69; 0x5f31bed4; 0x33682c5e; 0x44514cd0; 0x551e7aeb; 0x5bdb597a; 0x748b7da5; 0x7e7e077c;
     0x47c06929; 0x353e16b1; 0x3f2c2468; 0x5284cf94; 0x4f480e32; 0x4dffb244; 0x58066686; 0x7097fcba;
     0x09727855; 0x5a8a74c4; 0x2c83cf3f; 0x7693ffa4; 0x26691300; 0x12b882a7; 0x25530d92; 0x4623e215;
     0x691fbcb9; 0x2d8e824c; 0x2b895d52; 0x33de6ca4; 0x616cae4e; 0x22ae8643; 0x4053ac72; 0x5b01b5e7;
     0x1300cd17; 0x408fc301; 0x1f5e5442; 0x4a5dc893; 0x21434107; 0x245ae089; 0x60d679b5; 0x0d292a41;
     0x25480806; 0x220fdab4; 0x048af2c0; 0x1f8032d5; 0x268ff019; 0x09e38d40; 0x40a5a290; 0x21708e87;
     0x63da1685; 0x0ac301cc; 0x69d8796b; 0x6f1c97f8; 0x3c27dca5; 0x05e0a2b8; 0x3ade7473; 0x0422f0ef;
     0x31a7d37d; 0x5e830407; 0x4099d01e; 0x68fb3ad8; 0x216adf5e; 0x6849c8a2; 0x3a1707ac; 0x30e592e4;
     0x6f20f9a2; 0x7f853add; 0x58d4c511; 0x74efa634; 0x373e29b8; 0x69ef5143; 0x26fe16db; 0x6ca91698;
     0x347e987b; 0x2671f037; 0x67257324; 0x2da0d813; 0x17f9fa3d; 0x32b624ef; 0x5252c26c; 0x7c30d021;
     0x5cf6d6ae; 0x0bfc4352; 0x6ecffa4e; 0x7d400015; 0x347f286d; 0x4a0474d5; 0x2072a8f8; 0x2e79e95e;
     0x2e60bdda; 0x6961cf7e; 0x280c9b95; 0x0305a4d3; 0x31dc74bd; 0x1b20088e; 0x1b2c8594; 0x7911a714;
     0x273c3c43; 0x11848cd6; 0x6d71419e; 0x1d3983a6; 0x2ec1d7c6; 0x28cda532; 0x1d86d12f; 0x776f0e1a;
     0x79c4452d; 0x0a594f60; 0x04791d6a; 0x333f7931; 0x19b9ff9d; 0x2a57184b; 0x16af5b6d; 0x3fc82d59;
     0x01e272ce; 0x01257672; 0x3e9478ff; 0x26829a13; 0x3bcf5f33; 0x58b5d676; 0x074d58aa; 0x5adfa131;
     0x239c1681; 0x632d597e; 0x78f7d7ec; 0x7d9a7d3f; 0x711e4c48; 0x40f30a20; 0x16822483; 0x0d118aa6;
     0x083185a8; 0x50ada604; 0x2007523a; 0x72c56a6c; 0x130748ae; 0x53df84b1; 0x12b55f26; 0x11023f87;
     0x371a92b8; 0x6d631b98; 0x11ef6e51; 0x42964e17; 0x0e22ee39; 0x50de6086; 0x7d57e4dd; 0x2a410f1f;
     0x2b086094; 0x35a442d9; 0x1a964adf; 0x44508c96; 0x6d921dfe; 0x57ebbdd1; 0x07237a84; 0x5826b164;
     0x658bd900; 0x5f834ca7; 0x2274d894; 0x6f86aa44; 0x3b11adb5; 0x2cfb27fb; 0x56ffaa50; 0x4b4ca939;
     0x42a7622f; 0x59c04842; 0x7b0b42d1; 0x65c4f25b; 0x297d8a84; 0x45cbcada; 0x69ac322f; 0x598837fd;
     0x41817c20; 0x3b6e3329; 0x3b85f393; 0x15c6ff93; 0x679cc80a; 0x18f01b71; 0x77567276; 0x785b2bdd;
     0x7c406f95; 0x48fcf9bb; 0x71c2da2d; 0x3c58dc4f; 0x217c93bb; 0x41113a1f; 0x5ce74b9b; 0x57df20ed;
     0x5d265d55; 0x5e8dc8ba; 0x1392c19e; 0x1ed4290b; 0x6c3126f5; 0x180e41ca; 0x7cd48e32; 0x08121bc5;
     0x6e930d2e; 0x5743e0bf; 0x0822b61c; 0x54a6c82a; 0x2cc41a2b; 0x1a5ab09c; 0x410f6118; 0x32cac84c;
     0x1b740c70; 0x241e3ad7; 0x1b31e477; 0x0c9841ea; 0x73c21a51; 0x1c1a618c; 0x5478b0d6; 0x31a6a65c;
     0x6d0b7b39; 0x2f73b1b9; 0x2ddafe6b; 0x2245b5c4; 0x7ae15b42; 0x5731641b; 0x569a1be5; 0x56c6a297;
     0x57512a57; 0x1d8b40ff; 0x11cbd5c3; 0x29736a8d; 0x37bf91b2; 0x7eb93c76; 0x047bda38; 0x24535c0b;
     0x6b6a067a; 0x5f5e4d39; 0x6a5429d7; 0x0df1c0f7; 0x78efc153; 0x1f7f1964; 0x0d51cad3; 0x0d734b26;
     0x431077a3; 0x49a72bf0; 0x21070b95; 0x147665a5; 0x643ce158; 0x51757bd7; 0x33ebd27b; 0x0c921d18;
     0x3537839c; 0x0bb4d843; 0x181f31e9; 0x602ebd21; 0x41748069; 0x33d4c4e6; 0x47358ca8; 0x72d4cb6b;
     0x4ec0df43; 0x646073d8; 0x694f4eb9; 0x50368fcf; 0x5b700654; 0x21e80f55; 0x11f426f3; 0x40432412;
     0x762b3724; 0x2160aea0; 0x5c7fc768; 0x70e6bb52; 0x45dab133; 0x7f8f0890; 0x37205690; 0x4b193e9f;
     0x1633466d; 0x098481c1; 0x53dd0b0b; 0x07dea63f; 0x6d74cb9b; 0x4ceb70d3; 0x2bbddc09; 0x1782bef2;
     0x0ad6d41a; 0x035cd567; 0x11ba1685; 0x4d73a9c6; 0x7842ed43; 0x36e2e83e; 0x32ca3aa5; 0x0c89fd71;
     0x61bdc8ad; 0x02f008fe; 0x5a6ff452; 0x3efab8a9; 0x3907f9e9; 0x4e97ed88; 0x62a794b6; 0x2f480163;
     0x29d15968; 0x1dad0fea; 0x5d8cfb48; 0x42ff1c75; 0x13a0521b; 0x6754a0e8; 0x2aff406a; 0x50fea2b8;
     0x2e36c504; 0x3c4d85de; 0x0964b7d5; 0x5ccf49b5; 0x1af19321; 0x63707df0; 0x70cf185c; 0x50138647;
     0x06e8ff90; 0x38961917; 0x6a396953; 0x1cb90193; 0x0cbfe4d7; 0x64dd988f; 0x0736468f; 0x35f84f8e;
     0x7983aa23; 0x545740b1; 0x74237836; 0x76893adb; 0x3e50fe38; 0x6a3d937b; 0x6df4c840; 0x75ddd9dd;
     0x6dc91b59; 0x7e47c42d; 0x6f2d19fe; 0x7cf7fc32; 0x75185c96; 0x3e797881; 0x34f07631; 0x13bcd484;
     0x4569ddf1; 0x74090c7d; 0x29365247; 0x25c371eb; 0x44137e5b; 0x6cc51415; 0x30762aef; 0x28f11954;
     0x59d07259; 0x4a57d71d; 0x6a9896b9; 0x4dc11e4d; 0x7a7e9207; 0x0fe6ad68; 0x291203a1; 0x5d398557;
     0x7e8b5a9d; 0x3ffa702a; 0x6ca0c886; 0x7530b7f1; 0x6e9f6e0f; 0x6c1749fe; 0x790f3ceb; 0x009f2d67;
     0x1cf629f9; 0x19ae6ba7; 0x3407671b; 0x5b885f40; 0x0f55601e; 0x31080ba0; 0x71b7b0fa; 0x3e5f9391;
     0x3467665c; 0x6dbc7f38; 0x702046e9; 0x767da59f; 0x18eb9679; 0x0c3621d7; 0x2380e54f; 0x7bfe5b04;
     0x020dbba0; 0x550c7a2a; 0x43e7b76b; 0x7f44d8c6; 0x3c526d50; 0x453211d6; 0x361aaaa0; 0x129b23a8;
     0x1bd63dbc; 0x6cc63dde; 0x7ab77598; 0x786a95ce; 0x1e23af14; 0x43acad9c; 0x419372ec; 0x25411a86;
     0x5465bc2f; 0x20449f6d; 0x7fbd94be; 0x4411cf1f; 0x6dcf55fb; 0x64125ea6; 0x4298c20d; 0x67df4109;
     0x2c3ace7c; 0x09d3aad5; 0x023681ec; 0x1de1f9db; 0x69f0633f; 0x54e6ce94; 0x4ca11174; 0x50f72f47;
     0x7f1f98fe; 0x2d3e5659; 0x073736ae; 0x42877e3b; 0x2e4e2b89; 0x10b55972; 0x0af55b61; 0x5154e80b;
     0x6f6fe9af; 0x70d9bc90; 0x0e6833d1; 0x16de5a20; 0x0a481167; 0x1486e00c; 0x7b0fe5ca; 0x484a133f;
     0x65d733a9; 0x4f9e00ac; 0x38c14ecb; 0x6fe29b81; 0x69c5cfa9; 0x763d1873; 0x048ac156; 0x61967c50;
     0x779f1454; 0x7b7a663d; 0x5afb565c; 0x596dce7d; 0x22c0c326; 0x3a592474; 0x6f8edcdb; 0x2bc4d942;
     0x5e5ffc10; 0x5442141c; 0x5a0e2c27; 0x1b489c03; 0x2c23e85d; 0x58fa7b86; 0x5f73ad96; 0x02b51704;
     0x1327eb6e; 0x59a0061d; 0x118aa361; 0x1b6c6557; 0x15aa565e; 0x2814eb75; 0x5e53caa9; 0x25e2b857;
     0x68bf638a; 0x44f61364; 0x5ae8462b; 0x6d48d6d9; 0x6304ace4; 0x3b91c341; 0x2967b9fb; 0x45024c77;
     0x604a2270; 0x1c30ff53; 0x2c988d4c; 0x0fea9a0f; 0x4b1fccf7; 0x12bc8dcf; 0x5a1347d9; 0x15bf46df;
     0x24d2aa19; 0x473c3632; 0x19c231a0; 0x2be3fbfa; 0x18e8fa6c; 0x1f7712d1; 0x26e4406c; 0x4a600e60;
     0x075e839c; 0x37ec6e60; 0x41462304; 0x3c0595b2; 0x327ad663; 0x6be13716; 0x190f81d6; 0x7d1c6b98;
     0x72b9f8cd; 0x62c30f13; 0x0b34755b; 0x0fddfe79; 0x3e0cfe7b; 0x25d31d74; 0x1d44012f; 0x16da5de2;
     0x5ef01725; 0x7e13ca69; 0x3036ce30; 0x39fdbb34; 0x079678cc; 0x21c30525; 0x65deb91f; 0x26e9be6c;
     0x54e20fcc; 0x707829c0; 0x63323bb2; 0x5a1e6a08; 0x2ba16005; 0x16bc9631; 0x7475b468; 0x0407e507;
     0x6d24c699; 0x4c33ee1d; 0x62efce79; 0x50a2f4f6; 0x42229e31; 0x3f4c4961; 0x63a6b947; 0x16d7a8c6;
     0x0598c085; 0x1d91419c; 0x045b4a53; 0x284108f0; 0x577e69ab; 0x0d649e45; 0x24fae66d; 0x2eadf440;
     0x4d5f6417; 0x09e581b7; 0x2f447274; 0x624a0027; 0x41cae7bd; 0x3d06ae77; 0x1ee378a3; 0x268baefb;
     0x50035c8d; 0x17ae1c85; 0x15345501; 0x628acabb; 0x1567b406; 0x352917c7; 0x17d4a5a1; 0x05b648b7;
     0x4094686f; 0x2edaedaa; 0x0a32db9e; 0x2aa15578; 0x11addd06; 0x1b4e5a94; 0x7a53d077; 0x11748058;
     0x530be3f2; 0x10937921; 0x0469562b; 0x7dfa8fb6; 0x426d4252; 0x4c192f20; 0x04f8a476; 0x58fe9b14;
     0x77de7a4a; 0x51609341; 0x1e6c74b6; 0x74b38a0d; 0x085381d1; 0x60aac8ef; 0x0971a023; 0x45c5a95c;
     0x4da0686a; 0x5e2a4c5a; 0x14daa0a3; 0x4850a830; 0x23a319ce; 0x7c22c3de; 0x524ac220; 0x2f927599;
     0x4627750e; 0x5ec5db47; 0x331165ce; 0x3486b1db; 0x65652199; 0x6eebe579; 0x2d56adf6; 0x3b9bd897;
     0x7974d142; 0x36b3237f; 0x0cd82758; 0x648d3fe3; 0x5b694c94; 0x50322ec6; 0x700ee6ee; 0x562ec453;
     0x671f9758; 0x34e80c7d; 0x116d0859; 0x6a97c18f; 0x054f25a4; 0x0bc2a697; 0x6273ac62; 0x6b0975d7;
     0x444f5146; 0x0377f031; 0x418ba915; 0x7f857063; 0x2e93ab13; 0x7e7edbed; 0x7b8a437b; 0x132bbe00;
     0x456914b3; 0x1c5f90b9; 0x2359bbe7; 0x5f6278ce; 0x7802792d; 0x2c005dd4; 0x66c23472; 0x73bb8b8e;
     0x7cdba979; 0x1cc2485f; 0x5ad0931c; 0x7ad7445d; 0x566e9885; 0x5bb034c5; 0x3c630a22; 0x15548246;
     0x17673855; 0x3808bc81; 0x5e73733a; 0x751f927e; 0x112f572a; 0x15932165; 0x6b31c526; 0x2e9ecbac;
     0x2e10a087; 0x58f7e3bd; 0x01d382e6; 0x5814e1c9; 0x0a475c17; 0x2f32212f; 0x2f475f4c; 0x3c77f4f9;
     0x46cbd248; 0x2c808e7c; 0x36c1e12c; 0x5ece2924; 0x64093cb5; 0x7d6cbe86; 0x5b00438f; 0x6983fbff;
     0x00caef98; 0x2feb743f; 0x0d256f38; 0x739cf77d; 0x243aad7d; 0x083e20f7; 0x38f6b019; 0x38ea8331;
     0x2dc5b3ec; 0x4d5b0b11; 0x24bf5fbb; 0x46c8dcfc; 0x3ba9c518; 0x1a0f5d9a; 0x1f7648c9; 0x2a6dcdf7;
     0x0d829242; 0x60ca6248; 0x155a6a61; 0x43e42fde; 0x2087d99e; 0x38ba550d; 0x7f315e35; 0x6806f767;
     0x1f4c30d1; 0x4e1e8560; 0x362da316; 0x0d0acfae; 0x7bce9bf7; 0x20401a21; 0x08303e1e; 0x1bd9a513;
     0x64d11aa6; 0x64d0aa3b; 0x67026dc9; 0x61577eb8; 0x35d3d97d; 0x3d4cb88d; 0x54fccb61; 0x1eff9973;
     0x2cba2df8; 0x7b843eec; 0x2634a948; 0x394242c6; 0x416edcd1; 0x59650407; 0x4ed1b733; 0x4c9493e8;
     0x24959355; 0x4e2bac29; 0x6955df21; 0x3944d044; 0x7cfdfdb2; 0x703ae235; 0x51d67f32; 0x0992f427;
     0x4f224846; 0x20cee51c; 0x0570782c; 0x1cc8ff95; 0x5fcb8068; 0x1b6c2a00; 0x736dd8e1; 0x43cc41e3;
     0x3b63d6a3; 0x7770f5eb; 0x50afe778; 0x430e4dc6; 0x4668c374; 0x18c760ec; 0x18f22693; 0x58bcf051;
     0x74f56e60; 0x63f24409; 0x1d6191f8; 0x040bfe56; 0x131e5dcd; 0x5850edb6; 0x59a92d3b; 0x206885e7;
     0x43ca9bd6; 0x095458e6; 0x22a70ac0; 0x74685770; 0x372155d0; 0x40d09db7; 0x4cd36de1; 0x6d77d424;
     0x37a5d94c; 0x017b484e; 0x409f8957; 0x7e4c12db; 0x0c57a857; 0x6667b40d; 0x2157f12e; 0x7703635a;
     0x34a6829f; 0x1c407140; 0x08033dc8; 0x02f6dabf; 0x6e58c020; 0x0d5378a3; 0x7a911d20; 0x6c0eab72;
     0x0aabc581; 0x7224e8d1; 0x2cf6b087; 0x6ffbd3b0; 0x7e50ef20; 0x032cbb53; 0x5653caa8; 0x6cfa9e5f;
     0x29bd2155; 0x77c34326; 0x1d012344; 0x46a596b4; 0x513451d5; 0x5e7bb2b1; 0x1ac176a4; 0x6f49d87b;
     0x715f8c4d; 0x5b05e61e; 0x7c59a91f; 0x26ed0079; 0x1f4504b9; 0x63e87fab; 0x41fd273f; 0x24c53987;
     0x6df889b8; 0x5a683c24; 0x38e503fc; 0x3598f563; 0x5ba4542c; 0x471a740d; 0x676a34d1; 0x56a3b9f5;
     0x026dec82; 0x4091a3d7; 0x1f72d34d; 0x589b8169; 0x7e414787; 0x4bdfb8c9; 0x1674fcb9; 0x62427cf2;
     0x4beaab4d; 0x7c0a2a01; 0x33d9babe; 0x588a277e; 0x2d398e19; 0x4e54cd78; 0x02f857d9; 0x2afe1811;
     0x6a03cb87; 0x58267f5b; 0x613ea973; 0x463dea69; 0x5deeba8b; 0x590a37a4; 0x24d2db78; 0x27310d90;
     0x0f98c9ab; 0x48e31f0e; 0x527ee562; 0x253f21c6; 0x76872099; 0x36677eaf; 0x5ebb17f6; 0x4efc3013;
     0x4256eaa3; 0x0b2c1a4c; 0x2a8ca51a; 0x1b727c33; 0x55b6288c; 0x3b061e3d; 0x7684f682; 0x30cfb12a;
     0x23bef885; 0x67f9aba5; 0x3fde97c4; 0x555511f3; 0x4456819a; 0x43464142; 0x6de69bea; 0x5884ef58;
     0x05e4487e; 0x4f7dce1e; 0x0c90dd18; 0x6e62ce60; 0x33a739d6; 0x6036d72f; 0x310cf761; 0x68e624d3;
     0x29c6d2e5; 0x4be8b436; 0x625ae1f2; 0x33b93463; 0x61f91336; 0x1801620b; 0x5aaf48bc; 0x3b15d0f5;
     0x226f174c; 0x140d7c05; 0x41a55879; 0x112ac22c; 0x0c7aa6cb; 0x266e66df; 0x47effa31; 0x5e442f15;
     0x529db1fd; 0x7dbaf5c2; 0x261baabe; 0x6cc00620; 0x0f74584d; 0x56a51315; 0x025b0bcd; 0x436f02b3;
     0x70a42f3d; 0x3d7ac7e6; 0x1f28b9ed; 0x08e68d39; 0x01cc0726; 0x43eeab05; 0x35b8068e; 0x0c9b19c9;
     0x056a568e; 0x377ccfca; 0x2bab1e05; 0x04ab3472; 0x5699b826; 0x0d94ece0; 0x21d2de4f; 0x6077c505;
     0x779b0139; 0x74f0d050; 0x3434d0cb; 0x7d654ea2; 0x76e60e02; 0x41048a1d; 0x0a8a8674; 0x469df399;
     0x2b179d6d; 0x5ab96628; 0x7ee5d478; 0x3783a21e; 0x12fda64f; 0x28893262; 0x5dbeacb8; 0x20fc625e;
     0x4666a285; 0x0691a6a7; 0x7025e507; 0x130a45ad; 0x3d456ba2; 0x0e511fa0; 0x35d70cd3; 0x46e2dcd8;
     0x39db45e7; 0x35d727a1; 0x6e867631; 0x4a08d9d0; 0x0206774c; 0x100c02ce; 0x47a60b29; 0x48becb23;
     0x21b599cd; 0x4a22c3a7; 0x4c0b6018; 0x0c83d6d6; 0x4e813962; 0x3d739fcb; 0x41999c64; 0x164ced4b;
     0x16d8a84a; 0x4113fbfa; 0x6e4807c7; 0x28c3b3f6; 0x03071d36; 0x5159d26a; 0x087d88a4; 0x25fc1c06;
     0x64f922e8; 0x65c519e5; 0x5d8f42fe; 0x16e658ec; 0x3f78b13b; 0x66da2f46; 0x4ee1d53e; 0x4dce69f7;
     0x2479bf4f; 0x7af82284; 0x6a2175dd; 0x74a8a7b3; 0x438135f6; 0x0fd53da5; 0x2cf8856b; 0x0d0469a9;
     0x31fc6f6c; 0x715a1d09; 0x6055a1eb; 0x10835a34; 0x3740e967; 0x57b353ad; 0x4cd619dd; 0x6b6a3566;
     0x6f5a9dfe; 0x77c80a61; 0x2bfdee8b; 0x1816273d; 0x7cd74728; 0x553f44ce; 0x2a0e5ec1; 0x388c2068;
     0x319a9a08; 0x0c2af395; 0x2949f75e; 0x41f9e9b6; 0x6063f3cc; 0x7d4ea5af; 0x7861296b; 0x17e0f3cf;
     0x66012178; 0x043187bc; 0x5faa653d; 0x33cfc425; 0x53d378ca; 0x2155c31e; 0x712ab327; 0x6192c739;
     0x25f491ea; 0x54a3d8b3; 0x55b3d1bf; 0x762c4a3a; 0x75aebb59; 0x416fcfa8; 0x628a446b; 0x0d8457ce;
     0x79e603c1; 0x7cc575f0; 0x0a5ce0c5; 0x30b1ccfa; 0x00281d7e; 0x3cd2ed24; 0x7510c64c; 0x0be07209;
     0x73526859; 0x65946359; 0x2a546f4f; 0x783a8965; 0x34182cc1; 0x2616b0bc; 0x724ab52d; 0x278dd50a;
     0x71694e6f; 0x64f32ea7; 0x28ad21d9; 0x1c53c6ca; 0x6b0ba0a5; 0x2fe03f77; 0x188ac85a; 0x70b05fc0;
     0x0046bd3d; 0x7637b718; 0x661f49d3; 0x5b854049; 0x5d3ca349; 0x0a30b188; 0x2fd1462c; 0x1d26f321;
     0x5b8b7f1c; 0x45a48eb8; 0x5d86c8a6; 0x538daa6a; 0x2dc370a0; 0x17966abe; 0x45e629ed; 0x413c0ef8;
     0x4ca449b9; 0x7d193d45; 0x57244bc4; 0x31094ad3; 0x2582e257; 0x70a5e3fa; 0x18e190e7; 0x57dd2f9b;
     0x1c711f1d; 0x7bc6be06; 0x50d1b015; 0x051a7947; 0x06a92cf7; 0x5c2c30ba; 0x2c7a0ff4; 0x5805e415;
     0x4510ba6d; 0x5cae9dc3; 0x7a0c0b6c; 0x1741079e; 0x5d49d0ff; 0x6bbc8294; 0x77c2692d; 0x326bf8e2;
     0x54d4640e; 0x20012e23; 0x21013814; 0x30fddb5a; 0x75227b06; 0x1c20c1bb; 0x5f9bb584; 0x554ea171;
     0x6068384a; 0x2be5bcd2; 0x6971a336; 0x11144076; 0x0170e151; 0x20e24325; 0x74c6e58f; 0x5f393389;
     0x745070df; 0x76429400; 0x084a8659; 0x0c211b8b; 0x21faad9b; 0x540869f9; 0x6cbe4cf0; 0x1ba410b0;
     0x1f9c752d; 0x2a2c9ac3; 0x579bca4e; 0x1c1877d7; 0x755c2547; 0x7e1ae9f6; 0x288d06e2; 0x41f55262;
     0x6c85fae6; 0x2b95dd42; 0x5521b7ff; 0x4bc436c9; 0x4d41a34d; 0x73513ce9; 0x674ff0ca; 0x2a22cde2;
     0x5a837b4a; 0x55746f29; 0x0ac4f596; 0x29a81b75; 0x2fb343a1; 0x01ec05be; 0x30c0baab; 0x1078a8b9;
     0x54228b2e; 0x7f97ddef; 0x1fa2a455; 0x569e0dd2; 0x0c56fffd; 0x284ca4b9; 0x10103c78; 0x086c6758;
     0x30008cb4; 0x47a53b98; 0x6dec2457; 0x48987cf7; 0x4cd2d2c6; 0x1ab87a7b; 0x782fa417; 0x5fb16aba;
     0x05219cbe; 0x7d074008; 0x0b2728d9; 0x5285602d; 0x368d3435; 0x2278344c; 0x520ead89; 0x787e2018;
     0x45045ce3; 0x0f489fcd; 0x0abf96e8; 0x27935beb; 0x51cef0f2; 0x0bd1573c; 0x73197e33; 0x3f31c0da;
     0x31fc8b84; 0x3c4d7941; 0x64dc01ab; 0x187150a0; 0x76ab63a1; 0x78a7c190; 0x377ed885; 0x437d8255;
     0x6c39d9d7; 0x7b7edc68; 0x320890ce; 0x03c2a57a; 0x4b4e3a12; 0x7ef496ac; 0x7d1ab3c4; 0x3fa61f8d;
     0x1e9b72a3; 0x6c4a5021; 0x3860d04a; 0x51f7e47f; 0x4a0a633a; 0x15aff773; 0x7dba4f85; 0x7db57210;
     0x0e37413e; 0x59e776e6; 0x093907f5; 0x175e975c; 0x4105c34c; 0x1fb7280a; 0x3e376f3c; 0x31c02402;
     0x5304f8fd; 0x4c43eb12; 0x4b862a30; 0x2fb184c3; 0x06fc43a4; 0x059b463b; 0x5dd9b2c3; 0x0957fe28;
     0x4243df3a; 0x382f2a3a; 0x55048d94; 0x0248b7b6; 0x60e21a20; 0x6457da2d; 0x484d9d89; 0x6606cbd0;
     0x424e0d53; 0x3a5f3135; 0x6f12ea65; 0x77dbae6f; 0x52f147a4; 0x69ca4b9e; 0x35a4588a; 0x1dbc3d44;
     0x15ef5b44; 0x57f2e339; 0x4d4931b7; 0x35cc2530; 0x38df8918; 0x25cd233b; 0x041f0ffc; 0x4cf5974b;
     0x101783b8; 0x5de77ed1; 0x0599aff6; 0x69c913c8; 0x66e33f4e; 0x5eaa40bc; 0x2bae8234; 0x752eb641;
     0x5b0e4add; 0x2cf6a007; 0x1edab0aa; 0x19d48222; 0x32468cb1; 0x08e6bfe0; 0x3e312826; 0x148f3db6;
     0x3b729876; 0x149984e1; 0x71e39578; 0x197c67ed; 0x788b90a5; 0x2d508d8d; 0x4894022d; 0x06b89326;
     0x25e8c6aa; 0x16011f3a; 0x1a51adaf; 0x29b38486; 0x3c8dc319; 0x5f370371; 0x62107783; 0x1fac01b5;
     0x63b00ca0; 0x27a456dd; 0x514a48ae; 0x5882d384; 0x20715120; 0x78960a00; 0x51b4ee70; 0x72f6ce12;
     0x5f11e080; 0x2541e6f1; 0x386663d2; 0x01abfd90; 0x6bf859c3; 0x4268690e; 0x3c21c8cc; 0x3d9204f1;
     0x12da0f59; 0x2d50a1cd; 0x5b58291a; 0x1f9bc0c7; 0x0828411d; 0x2622389d; 0x2868b426; 0x112d1d34;
     0x1c755f8f; 0x3ebc4805; 0x425a40e7; 0x1520c6dd; 0x38253512; 0x4986bf19; 0x61fd9d93; 0x4a91d7dd;
     0x7e16c383; 0x61f6628a; 0x306e5fcc; 0x5c16d47c; 0x35384a14; 0x674e5bc2; 0x7bb151ba; 0x74517816;
     0x65b99a87; 0x59c9f9a2; 0x5e6cc4cc; 0x1125a977; 0x141614fd; 0x185c71d4; 0x3cfbc510; 0x131ab579;
     0x42d065bb; 0x58adedf1; 0x50615afd; 0x07a897a6; 0x3d2fb856; 0x25a1d19e; 0x6513d354; 0x10272655;
     0x148be972; 0x2462df8b; 0x2313aaaa; 0x65df0efa; 0x0a231d35; 0x040bebb9; 0x5643531d; 0x40eaecc9;
     0x383a4be5; 0x4eef678c; 0x7386d9bd; 0x249bee6c; 0x07fbccea; 0x3c649e71; 0x30dba936; 0x1a4e90b2;
     0x3044b430; 0x66e80661; 0x25c329ac; 0x2d1ad481; 0x55aa6fbe; 0x37393436; 0x1d50e1b4; 0x5cc71611;
     0x5d45e90b; 0x506fb608; 0x5b0750f2; 0x16e7da1f; 0x0e8d9bc1; 0x40c75f39; 0x00c42217; 0x65b81d97;
     0x26fe162a; 0x10c42d49; 0x68961a29; 0x0b216608; 0x19516141; 0x7391975a; 0x07b0171b; 0x2270be0d;
     0x4b56607d; 0x17abf6f4; 0x1d97d1f0; 0x094c7fdd; 0x2ad353a9; 0x11fa9c5e; 0x2f139267; 0x54bbe872;
     0x2812c620; 0x57f580dd; 0x1acda829; 0x21e4f841; 0x054a7560; 0x4b6a5f58; 0x435f35d1; 0x7e70d241;
     0x6bbca614; 0x2d7350bf; 0x2f8247ae; 0x5157f368; 0x0980275c; 0x381edf48; 0x00b7628b; 0x7f2244a7;
     0x699cb3f2; 0x64bcc872; 0x407ab854; 0x1f2d337e; 0x79774b1c; 0x5b87738e; 0x1123e837; 0x2c549098;
     0x1a7c30eb; 0x71033aa1; 0x7d90682e; 0x3e18a139; 0x17021dae; 0x7a4e00ee; 0x5dc05e30; 0x5b48d748;
     0x1cba2635; 0x01b2bd26; 0x19f8b8b8; 0x074d8011; 0x44001ee4; 0x45635de4; 0x6adfab30; 0x2922110e;
     0x0ce0e413; 0x7c5998b3; 0x49bc2d7e; 0x5e0c8700; 0x6c098f4b; 0x1a1f9f53; 0x0faa6ca2; 0x72bf68d6;
     0x7a0b7a4e; 0x12810ff8; 0x18d2e8f5; 0x67c0882f; 0x35283420; 0x77562a4c; 0x4a70b6d7; 0x4171b839;
     0x31ff85b8; 0x0244cf49; 0x0ca100ec; 0x18bb92b5; 0x16f8269e; 0x705a4c37; 0x034f02b1; 0x35eea06c;
     0x2fa25dda; 0x49e05f6d; 0x2bcf7b1c; 0x7cd1c0b7; 0x75df237e; 0x5499aaeb; 0x361679c8; 0x0b820950;
     0x25532980; 0x5be64b4a; 0x0df2ed58; 0x0776de7e; 0x120ab856; 0x262fd585; 0x079fcb96; 0x20515a07;
     0x222e6b0a; 0x4a326a89; 0x17726911; 0x7740772c; 0x4d43e0ae; 0x1062c95f; 0x4c288603; 0x04e751be;
     0x220cb789; 0x019e67d8; 0x0a4ae8e8; 0x4eabdfe8; 0x779a7125; 0x0abcba52; 0x4f1636ff; 0x180189ac;
     0x7aa37c64; 0x5b6d34e0; 0x17662f15; 0x0014bd21; 0x0c2bf0d5; 0x1cdb9601; 0x60111e36; 0x49d5ef9b;
     0x2ce1457a; 0x4dbd1236; 0x40dd7c03; 0x10d6ff53; 0x01f9e14b; 0x6359ecc2; 0x551b5368; 0x4ad84239;
     0x6b15d7e7; 0x2384a1e9; 0x03af4e12; 0x51a36df0; 0x5c012bf2; 0x789c09cc; 0x5706b0a3; 0x2b24f048;
     0x389b6ea4; 0x309619de; 0x550efe13; 0x17533d00; 0x1d7c3705; 0x0e44fcc3; 0x5d654e4f; 0x3b4f55e0;
     0x4e86ee08; 0x64ed41e9; 0x588f125d; 0x4fee7f8c; 0x6748d08c; 0x733e7edc; 0x01c5131c; 0x494cf6c7;
     0x45d3c79d; 0x70c43d4e; 0x775355ed; 0x2ac71136; 0x171986d8; 0x5f83c03b; 0x76f07660; 0x511ef21d;
     0x53083afb; 0x6a2485eb; 0x63a1826a; 0x45a58842; 0x2c0e6a52; 0x639a1f57; 0x3937ed98; 0x1915b44f;
     0x1a18140b; 0x73937457; 0x3407de32; 0x26b5f0e0; 0x1955df62; 0x7fec7c58; 0x0465e209; 0x36bd0139;
     0x7eea79aa; 0x308a8d4f; 0x7d1a4ff8; 0x49dc3f71; 0x1a637f96; 0x1106bd39; 0x093427bf; 0x13c5df63;
     0x40d8c0ea; 0x6901d06c; 0x7b3c8faf; 0x19c57b04; 0x31f722d1; 0x28906d5e; 0x5fcacd4f; 0x7220d1c2;
     0x7ca1e5f3; 0x3e357efd; 0x26b5ab7d; 0x776fa67f; 0x593e89d4; 0x60c55d19; 0x25a537f6; 0x4d212343;
     0x37d958a2; 0x355b5b7e; 0x7c8168c5; 0x4a40a28b; 0x64892e03; 0x5dbfa9f9; 0x4462be59; 0x1c280fd4;
     0x35ca1b1d; 0x428c16ed; 0x412b0065; 0x1b1d023c; 0x04f57182; 0x24c3532c; 0x057bc43d; 0x01b789f3;
     0x1091e70b; 0x62d54cd8; 0x3ff7b86a; 0x7b7536fe; 0x584b7537; 0x1706075b; 0x6b472e5b; 0x21479a34;
     0x487d73b3; 0x125d559a; 0x31fd960e; 0x210bc42a; 0x53736956; 0x506a54b6; 0x6166fbab; 0x5f4ef625;
     0x144cc0b6; 0x0dd8080a; 0x61f2463c; 0x5e55d215; 0x649a4ae9; 0x389715e9; 0x5b4de964; 0x75512f7b;
     0x57640cf1; 0x3aaf0948; 0x715e4549; 0x6b894fb4; 0x22073a23; 0x1e6e840a; 0x5168de1f; 0x084873a6;
     0x42b90605; 0x5e9a0e41; 0x71d0f075; 0x6fa752d4; 0x56497365; 0x5bc57ff8; 0x2e061440; 0x454831da;
     0x6b261bad; 0x4ea3796c; 0x1e67a05e; 0x2002265d; 0x12d8a4b9; 0x401251f5; 0x447f2086; 0x2adc1357;
     0x71b39152; 0x48575c94; 0x4dd14f36; 0x0f96f549; 0x77db980e; 0x352f423f; 0x7f0ed38a; 0x0fddf56b;
     0x7be6cf92; 0x41eab5b6; 0x57dfdd78; 0x6a5f34cf; 0x39c054b2; 0x7e2a627c; 0x7bb4839b; 0x2b1f4fd3;
     0x1e3d0271; 0x2d650bfe; 0x4d8f43e5; 0x6ff5dc13; 0x3e3fa0b2; 0x2140b2d3; 0x516da92e; 0x43fb1d3c;
     0x5b94011c; 0x153902f3; 0x2162c3d6; 0x712f52b5; 0x6eaecfb7; 0x59429c40; 0x5ddd6681; 0x626685c5;
     0x5dd22826; 0x4b2dc96a; 0x0438a69a; 0x239304fe; 0x7c868c07; 0x4eee5fa4; 0x120e5b8c; 0x4767aae0;
     0x648c36b1; 0x24bae5cc; 0x7b9ddf36; 0x4f94cb47; 0x1b7faeea; 0x73fa82ca; 0x4c32d990; 0x092d0c93;
     0x4315711b; 0x3fcc0592; 0x5d0a5111; 0x48d6ba90; 0x1e444b21; 0x40c16fcf; 0x6867f5b1; 0x57a476ea;
     0x175f2304; 0x1ffc5cfe; 0x6a7ba53a; 0x637b599c; 0x1404dd83; 0x0805aa32; 0x10d47daf; 0x2db94dc2;
     0x1a1f3371; 0x61cc8fb3; 0x3711c483; 0x28a61268; 0x2ed93c6e; 0x2e42ed8c; 0x1a387531; 0x6b1bd449;
     0x301efe32; 0x5dcf70d0; 0x3dd03c00; 0x1153313d; 0x129c2b48; 0x3b9d34b9; 0x2ba65435; 0x6e0426bb;
     0x1955c17b; 0x4f0e1b76; 0x028d41c9; 0x02d475d3; 0x2ef7c30f; 0x2ea73087; 0x357c92a5; 0x4141569c;
     |]
(* %%MAGICEND%% *)

let () =
   assert(Array.length hash_data >= hash_length + digest_length - 1)

(************************************************
 * Integer hashes.
 *)
(* %%MAGICBEGIN%% *)
module HashCode : HashCodeSig =
struct
   type t =
      { mutable hash_digest : int;
        mutable hash_code   : int
      }

   (*
    * New buffer.
    *)
   let create () =
      { hash_digest = 0;
        hash_code   = 0
      }

   (*
    * Add an integer.
    *)
   let add_truncated_bits buf i =
      let { hash_digest = digest;
            hash_code   = code
          } = buf
      in
      let code = (code + i + 1) mod hash_length in
         buf.hash_digest <- digest lxor (Array.unsafe_get hash_data code);
         buf.hash_code <- code

   let add_bits buf i =
      add_truncated_bits buf (i land 0x7ff)

   (*
    * Add the characters in a string.
    *)
   let add_string buf s =
      for i = 0 to pred (String.length s) do
         add_truncated_bits buf (Char.code (String.unsafe_get s i))
      done

   (*
    * Numbers.
    *)
   let add_int buf i =
      add_bits buf i;
      add_bits buf (i lsr 11);
      add_bits buf (i lsr 22)

   let add_int32 buf i =
      add_bits buf (Int32.to_int (Int32.shift_right_logical i 16));
      add_bits buf (Int32.to_int i)

   let add_int64 buf i =
      add_int buf (Int64.to_int (Int64.shift_right_logical i 48));
      add_int buf (Int64.to_int (Int64.shift_right_logical i 24));
      add_int buf (Int64.to_int i)

   let add_nativeint buf i =
      add_int buf (Nativeint.to_int (Nativeint.shift_right_logical i 48));
      add_int buf (Nativeint.to_int (Nativeint.shift_right_logical i 24));
      add_int buf (Nativeint.to_int i)

   let add_float buf x =
      add_int64 buf (Int64.bits_of_float x)

   (*
    * Extract the digest.
    *)
   let code buf =
      buf.hash_digest
end
(* %%MAGICEND%% *)

(************************************************
 * Digest-based hashes.
 *)
(* %%MAGICBEGIN%% *)
module HashDigest : HashDigestSig =
struct
   type t =
      { hash_digest         : int array;
        mutable hash_code   : int
      }

   (*
    * New buffer.
    *)
   let create () =
      { hash_digest = Array.create digest_length 0;
        hash_code   = 0
      }

   (*
    * Add an integer.
    *)
   let add_truncated_bits buf i =
      let { hash_digest = digest;
            hash_code   = code
          } = buf
      in
      let code = (code + digest_length + i) mod hash_length in
         for i = 0 to digest_length - 1 do
            digest.(i) <- digest.(i) lxor (Array.unsafe_get hash_data (code + i))
         done;
         buf.hash_code <- code

   let add_bits buf i =
      add_truncated_bits buf (i land 0x7ff)

   (*
    * Add the characters in a string.
    *)
   let add_char buf c =
      add_truncated_bits buf (Char.code c)

   let add_string buf s =
      for i = 0 to pred (String.length s) do
         add_char buf (String.unsafe_get s i)
      done

   let add_substring buf s off len =
      if off < 0 || len < 0 || off + len > String.length s then
         raise (Invalid_argument "Lm_hash.add_substring");
      for i = off to pred (off + len) do
         add_char buf (String.unsafe_get s i)
      done

   (*
    * Numbers.
    *)
   let add_bool buf b =
      add_truncated_bits buf (if b then 1 else 0)

   let add_int buf i =
      add_bits buf i;
      add_bits buf (i lsr 11);
      add_bits buf (i lsr 22)

   let add_int32 buf i =
      add_bits buf (Int32.to_int (Int32.shift_right_logical i 16));
      add_bits buf (Int32.to_int i)

   let add_int64 buf i =
      add_int buf (Int64.to_int (Int64.shift_right_logical i 48));
      add_int buf (Int64.to_int (Int64.shift_right_logical i 24));
      add_int buf (Int64.to_int i)

   let add_nativeint buf i =
      add_int buf (Nativeint.to_int (Nativeint.shift_right_logical i 48));
      add_int buf (Nativeint.to_int (Nativeint.shift_right_logical i 24));
      add_int buf (Nativeint.to_int i)

   let add_float buf x =
      let i = Int64.bits_of_float x in
         add_int buf (Int64.to_int (Int64.shift_right i 48));
         add_int buf (Int64.to_int (Int64.shift_right i 24));
         add_int buf (Int64.to_int i)

   (*
    * Extract the digest.
    *)
   let digest buf =
      let digest = buf.hash_digest in
      let s = String.create digest_length in
         for i = 0 to pred digest_length do
            s.[i] <- Char.chr (digest.(i) land 0xff)
         done;
         s
end;;
(* %%MAGICEND%% *)

(************************************************************************
 * Hash helper functions.
 *)

(*
 * The default function for combinding hash values.
 * XXX: JYH: we should try using a smarter hash function.
 *)
let hash_combine i1 i2 =
   (i1 lsl 2) lxor (i1 lsr 2) lxor i2

(*
 * Hash a list of integers.
 *)
let hash_int_list code l =
   let buf = HashCode.create () in
      HashCode.add_int buf code;
      List.iter (HashCode.add_int buf) l;
      HashCode.code buf

(*
 * Comparison utilities.
 *)
let rec compare_int_list (l1 : int list) (l2 : int list) =
   match l1, l2 with
      i1 :: l1, i2 :: l2 ->
         if i1 < i2 then
            -1
         else if i1 > i2 then
            1
         else
            compare_int_list l1 l2
    | [], _ ::_ ->
         -1
    | _ :: _, [] ->
         1
    | [], [] ->
         0

(*
 * -*-
 * Local Variables:
 * End:
 * -*-
 *)
