/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2007 Sun Microsystems, Inc.
 */
package org.netbeans.modules.cnd.completion.includes;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.netbeans.editor.BaseDocument;
import org.netbeans.modules.cnd.api.model.CsmFile;
import org.netbeans.modules.cnd.api.model.services.CsmFileInfoQuery;
import org.netbeans.modules.cnd.modelutil.CsmUtilities;
import org.netbeans.modules.cnd.utils.CndPathUtilitities;
import org.netbeans.modules.cnd.utils.FSPath;
import org.netbeans.modules.cnd.utils.FileObjectFilter;
import org.netbeans.modules.cnd.utils.MIMEExtensions;
import org.netbeans.modules.cnd.utils.MIMENames;
import org.netbeans.modules.cnd.utils.MIMESupport;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.ExtensionList;
import org.openide.util.Exceptions;

/**
 *
 * @author Vladimir Voskresensky
 */
public class CsmIncludeCompletionQuery {

    private static final Collection<String> EXCLUDED_DIR_NAMES = Arrays.asList(new String[]{
                "CVS", ".hg", "nbproject", "SCCS", "SunWS_cache"}); // NOI18N
    private Map<String, CsmIncludeCompletionItem> results;
    private final CsmFile file;

    public CsmIncludeCompletionQuery(CsmFile file) {
        this.file = file;
    }

    public Collection<CsmIncludeCompletionItem> query(BaseDocument doc, String childSubDir, int substitutionOffset, Boolean usrInclude, boolean showAll) {
        results = new HashMap<String, CsmIncludeCompletionItem>(100);
        CsmFile docFile = this.file;
        if (docFile == null) {
            docFile = CsmUtilities.getCsmFile(doc, false, false);
        }
        FileSystem docFileSystem;
        try {
            docFileSystem = docFile.getFileObject().getFileSystem();
        } catch (FileStateInvalidException ex) {
            Exceptions.printStackTrace(ex);
            return results.values();
        }
        Collection<FSPath> usrPaths = Collections.<FSPath>emptyList();
        Collection<FSPath> sysPaths = Collections.<FSPath>emptyList();
        if (CndPathUtilitities.isPathAbsolute(childSubDir)) {
            // special handling for absolute paths...
            addFolderItems(FSPath.toFSPath(docFileSystem.getRoot()),
                    "",
                    childSubDir, true, (usrInclude != null ? usrInclude : false),
                    true, substitutionOffset);
            return results.values();
        }
        if (docFile != null) {
            usrPaths = getFileIncludes(docFile, false);
            sysPaths = getFileIncludes(docFile, true);
        } else {
            FileObject baseFile = CsmUtilities.getFileObject(doc);
            if (baseFile == null || ! baseFile.isValid()) {
                // IZ#123039: NPE
                return Collections.<CsmIncludeCompletionItem>emptyList();
            }
        }
        FileObject usrDir = docFile.getFileObject().getParent();
        if (usrDir != null && usrDir.isValid()) {
            if (usrInclude == null || usrInclude == Boolean.TRUE) {
                addFolderItems(FSPath.toFSPath(usrDir), ".", childSubDir, false, false, true, substitutionOffset); // NOI18N
                if (showAll) {
                    for (FSPath usrPath : usrPaths) {
                        addFolderItems(usrPath, usrPath.getPath(), childSubDir, false, false, true, substitutionOffset);
                    }
                    for (FSPath sysPath : sysPaths) {
                        addFolderItems(sysPath, sysPath.getPath(), childSubDir, false, true, false, substitutionOffset);
                    }
                }
                if (usrDir.getParent() != null) {
                    addParentFolder(substitutionOffset, childSubDir, false);
                }
            } else {
                for (FSPath sysPath : sysPaths) {
                    addFolderItems(sysPath, sysPath.getPath(), childSubDir, false, true, false, substitutionOffset);
                }
                if (showAll) {
                    for (FSPath usrPath : usrPaths) {
                        addFolderItems(usrPath, usrPath.getPath(), childSubDir, false, false, true, substitutionOffset);
                    }
                    addFolderItems(FSPath.toFSPath(usrDir), ".", childSubDir, false, false, true, substitutionOffset); // NOI18N
                    if (usrDir.getParent() != null) {
                        addParentFolder(substitutionOffset, childSubDir, true);
                    }
                }
            }
        }
        return results.values();
    }

    private void addFolderItems(FSPath parentFolder, String parentFolderPresentation,
            String childSubDir, boolean highPriority, boolean system, boolean filtered, int substitutionOffset) {
        FileObject parentFO = parentFolder.getFileObject();
        if (parentFO != null) {
            FileObject dir = parentFO.getFileObject(childSubDir);
            if (dir != null && dir.isValid()) {
                FileObject[] list = filtered ? listFiles(dir, new HeadersFileFilter()) : listFiles(dir, new DefFileFilter());
                if (list != null) {
                    String relFileName;
                    for (FileObject curFile : list) {
                        relFileName = curFile.getNameExt();
                        CsmIncludeCompletionItem item = CsmIncludeCompletionItem.createItem(
                                substitutionOffset, relFileName, parentFolderPresentation, childSubDir,
                                system, highPriority, curFile.isFolder(), true);
                        if (!results.containsKey(relFileName)) {
                            results.put(relFileName, item);
                        }
                    }
                }
            }
        }
    }
    
    private FileObject[] listFiles(FileObject parent, FileObjectFilter filter) {
        FileObject[] children = parent.getChildren();
        if (children == null || children.length == 0) {
            return null;
        }
        List<FileObject> result = new ArrayList<FileObject>(children.length);
        for (FileObject child : children) {
            if (filter.accept(child)) {
                result.add(child);
            }
        }
        return result.toArray(new FileObject[result.size()]);
    }

    private void addParentFolder(int substitutionOffset, String childSubDir, boolean system) {
        // IZ#128044: Completion in #include should switch to 2-nd mode if there are no files in the list
        // doesn't append ".." item for empty lists
        if (!results.isEmpty()) {
            CsmIncludeCompletionItem item = CsmIncludeCompletionItem.createItem(
                    substitutionOffset, "..", ".", childSubDir, system, false, true, false); // NOI18N
            results.put("..", item);//NOI18N
        }
    }

    private Collection<FSPath> getFileIncludes(CsmFile file, boolean system) {        
        CsmFileInfoQuery query = CsmFileInfoQuery.getDefault();
        return system ? query.getSystemIncludePaths(file) : query.getUserIncludePaths(file);
    }

    private static final class DefFileFilter implements FileObjectFilter {
        @Override
        public boolean accept(FileObject fileObject) {
            return !specialFile(fileObject);
        }
    }

    private static final class HeadersFileFilter implements FileObjectFilter {

        private final ExtensionList exts;

        protected HeadersFileFilter() {
            exts = new ExtensionList();
            for (String ext : MIMEExtensions.get(MIMENames.HEADER_MIME_TYPE).getValues()) {
                exts.addExtension(ext);
            }
        }

        @Override
        public boolean accept(FileObject pathname) {
            return !specialFile(pathname) &&
                    (exts.isRegistered(pathname.getNameExt()) || pathname.isFolder() || isHeaderFileWoExt(pathname));
        }
    }

    private static boolean isHeaderFileWoExt(FileObject pathname) {
        if (FileUtil.getExtension(pathname.getNameExt()).length() == 0) {
            return MIMENames.HEADER_MIME_TYPE.equals(MIMESupport.getSourceFileMIMEType(pathname));
        }
        return false;
    }

    private static boolean specialFile(FileObject file) {
        String name = file.getNameExt();
        if (name.startsWith(".")) { // NOI18N
            return true;
        } else if (name.endsWith("~")) { // NOI18N
            return true;
        } else if (file.isFolder()) {
            if (EXCLUDED_DIR_NAMES.contains(name)) {
                return true;
            }
        }
        return false;
    }
}
