<?php

class Swift_ByteStream_FileByteStreamAcceptanceTest extends \PHPUnit\Framework\TestCase
{
    private $testFile;

    protected function setUp()
    {
        $this->testFile = sys_get_temp_dir().'/swift-test-file'.__CLASS__;
        file_put_contents($this->testFile, 'abcdefghijklm');
    }

    protected function tearDown()
    {
        unlink($this->testFile);
    }

    public function testFileDataCanBeRead()
    {
        $file = $this->createFileStream($this->testFile);
        $str = '';
        while (false !== $bytes = $file->read(8192)) {
            $str .= $bytes;
        }
        $this->assertEquals('abcdefghijklm', $str);
    }

    public function testFileDataCanBeReadSequentially()
    {
        $file = $this->createFileStream($this->testFile);
        $this->assertEquals('abcde', $file->read(5));
        $this->assertEquals('fghijklm', $file->read(8));
        $this->assertFalse($file->read(1));
    }

    public function testFilenameIsReturned()
    {
        $file = $this->createFileStream($this->testFile);
        $this->assertEquals($this->testFile, $file->getPath());
    }

    public function testFileCanBeWrittenTo()
    {
        $file = $this->createFileStream($this->testFile, true);
        $file->write('foobar');
        $this->assertEquals('foobar', $file->read(8192));
    }

    public function testReadingFromThenWritingToFile()
    {
        $file = $this->createFileStream($this->testFile, true);
        $file->write('foobar');
        $this->assertEquals('foobar', $file->read(8192));
        $file->write('zipbutton');
        $this->assertEquals('zipbutton', $file->read(8192));
    }

    public function testWritingToFileWithCanonicalization()
    {
        $file = $this->createFileStream($this->testFile, true);
        $file->addFilter($this->createFilter(["\r\n", "\r"], "\n"), 'allToLF');
        $file->write("foo\r\nbar\r");
        $file->write("\nzip\r\ntest\r");
        $file->flushBuffers();
        $this->assertEquals("foo\nbar\nzip\ntest\n", file_get_contents($this->testFile));
    }

    public function testWritingWithFulleMessageLengthOfAMultipleOf8192()
    {
        $file = $this->createFileStream($this->testFile, true);
        $file->addFilter($this->createFilter(["\r\n", "\r"], "\n"), 'allToLF');
        $file->write('');
        $file->flushBuffers();
        $this->assertEquals('', file_get_contents($this->testFile));
    }

    public function testBindingOtherStreamsMirrorsWriteOperations()
    {
        $file = $this->createFileStream($this->testFile, true);
        $is1 = $this->createMockInputStream();
        $is2 = $this->createMockInputStream();

        $is1->expects($this->exactly(2))
            ->method('write')
            ->withConsecutive(
                ['x'],
                ['y']
            );
        $is2->expects($this->exactly(2))
            ->method('write')
            ->withConsecutive(
                ['x'],
                ['y']
            );

        $file->bind($is1);
        $file->bind($is2);

        $file->write('x');
        $file->write('y');
    }

    public function testBindingOtherStreamsMirrorsFlushOperations()
    {
        $file = $this->createFileStream(
            $this->testFile, true
            );
        $is1 = $this->createMockInputStream();
        $is2 = $this->createMockInputStream();

        $is1->expects($this->once())
            ->method('flushBuffers');
        $is2->expects($this->once())
            ->method('flushBuffers');

        $file->bind($is1);
        $file->bind($is2);

        $file->flushBuffers();
    }

    public function testUnbindingStreamPreventsFurtherWrites()
    {
        $file = $this->createFileStream($this->testFile, true);
        $is1 = $this->createMockInputStream();
        $is2 = $this->createMockInputStream();

        $is1->expects($this->exactly(2))
            ->method('write')
            ->withConsecutive(
                ['x'],
                ['y']
            );
        $is2->expects($this->once())
            ->method('write')
            ->with('x');

        $file->bind($is1);
        $file->bind($is2);

        $file->write('x');

        $file->unbind($is2);

        $file->write('y');
    }

    private function createFilter($search, $replace)
    {
        return new Swift_StreamFilters_StringReplacementFilter($search, $replace);
    }

    private function createMockInputStream()
    {
        return $this->getMockBuilder('Swift_InputByteStream')->getMock();
    }

    private function createFileStream($file, $writable = false)
    {
        return new Swift_ByteStream_FileByteStream($file, $writable);
    }
}
