/* Copyright (C) 2000, 2001 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
   Contributed by Mark Kettenis <kettenis@gnu.org>, 2000.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <errno.h>
#include <netdb.h>
#include <nss.h>

#include <lwres/netdb.h>

static int copytobuf(struct hostent *, struct hostent *, char *, int);

static enum nss_status
internal_getipnodebyname_r (const char *name, int af, int flags,
			    struct hostent *result, char *buffer,
			    size_t buflen, int *errnop, int *herrnop)
{
  struct hostent *he;
  int mapped_flags = 0;
  int res;

  he = lwres_getipnodebyname (name, af, mapped_flags, herrnop);
  if (he == NULL)
    {
      switch (*herrnop)
	{
	case HOST_NOT_FOUND:
	case NO_ADDRESS:
	  return NSS_STATUS_NOTFOUND;
	default:
	  return NSS_STATUS_UNAVAIL;
	}
    }

  res = copytobuf (he, result, buffer, buflen);
  if (he != NULL)
    lwres_freehostent (he);
  if (res != 0)
    {
      *errnop = ERANGE;
      return NSS_STATUS_TRYAGAIN;
    }

  return NSS_STATUS_SUCCESS;
}

enum nss_status
_nss_lwres_gethostbyname2_r (const char *name, int af, struct hostent *result,
			     char *buffer, size_t buflen, int *errnop,
			     int *herrnop)
{
  return internal_getipnodebyname_r (name, af, 0, result, buffer, buflen,
				     errnop, herrnop);
}

enum nss_status
_nss_lwres_gethostbyname_r (const char *name, struct hostent *result,
			  char *buffer, size_t buflen, int *errnop,
			  int *herrnop)
{
  /* FIXME: I suppose we need some other options here.  */
  return internal_getipnodebyname_r (name, AF_INET, 0, result,
				     buffer, buflen, errnop, herrnop);
}

enum nss_status
_nss_lwres_gethostbyaddr_r (const char *addr, size_t len, int type,
			    struct hostent *result, char *buffer,
			    size_t buflen, int *errnop, int *herrnop)
{
  struct hostent *he;
  int res;
  
  he = lwres_getipnodebyaddr (addr, len, type, herrnop);
  if (he == NULL)
    {
      switch (*herrnop)
	{
	case HOST_NOT_FOUND:
	case NO_ADDRESS:
	  return NSS_STATUS_NOTFOUND;
	default:
	  return NSS_STATUS_UNAVAIL;
	}
    }

  res = copytobuf (he, result, buffer, buflen);
  if (he != NULL)
    lwres_freehostent (he);
  if (res != 0)
    {
      *errnop = ERANGE;
      return NSS_STATUS_TRYAGAIN;
    }

  return NSS_STATUS_SUCCESS;
}


/* The function on this page is copied from the file
   `lib/lwres/gethost.c' in the BIND9 distribution and has the
   following copyright: */

/*
 * Copyright (C) 2000  Internet Software Consortium.
 * 
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM DISCLAIMS
 * ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL INTERNET SOFTWARE
 * CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $BIND9Id: gethost.c,v 1.20 2000/06/27 23:13:26 bwelling Exp $ */

#define LWRES_ALIGNBYTES (sizeof(char *) - 1)
#define LWRES_ALIGN(p) \
	(((unsigned long)(p) + LWRES_ALIGNBYTES) &~ LWRES_ALIGNBYTES)

static int
copytobuf(struct hostent *he, struct hostent *hptr, char *buf, int buflen) {
        char *cp;
        char **ptr;
        int i, n;
        int nptr, len;

        /*
	 * Find out the amount of space required to store the answer.
	 */
        nptr = 2; /* NULL ptrs */
        len = (char *)LWRES_ALIGN(buf) - buf;
        for (i = 0; he->h_addr_list[i]; i++, nptr++) {
                len += he->h_length;
        }
        for (i = 0; he->h_aliases[i]; i++, nptr++) {
                len += strlen(he->h_aliases[i]) + 1;
        }
        len += strlen(he->h_name) + 1;
        len += nptr * sizeof(char*);
        
        if (len > buflen) {
                return (-1);
        }

        /*
	 * Copy address size and type.
	 */
        hptr->h_addrtype = he->h_addrtype;
        n = hptr->h_length = he->h_length;

        ptr = (char **)LWRES_ALIGN(buf);
        cp = (char *)LWRES_ALIGN(buf) + nptr * sizeof(char *);

        /*
	 * Copy address list.
	 */
        hptr->h_addr_list = ptr;
        for (i = 0; he->h_addr_list[i]; i++, ptr++) {
                memcpy(cp, he->h_addr_list[i], n);
                hptr->h_addr_list[i] = cp;
                cp += n;
        }
        hptr->h_addr_list[i] = NULL;
        ptr++;

        /*
	 * Copy official name.
	 */
        n = strlen(he->h_name) + 1;
        strcpy(cp, he->h_name);
        hptr->h_name = cp;
        cp += n;

        /*
	 * Copy aliases.
	 */
        hptr->h_aliases = ptr;
        for (i = 0; he->h_aliases[i]; i++) {
                n = strlen(he->h_aliases[i]) + 1;
                strcpy(cp, he->h_aliases[i]);
                hptr->h_aliases[i] = cp;
                cp += n;
        }
        hptr->h_aliases[i] = NULL;

        return (0);
}
