/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2002  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "kiss.h"

#define PREFIX "-xqifw="
#define TEMPLATE "/tmp/kiss-XXXXXX"

static int package_open= 0;
static char temp[]= TEMPLATE;

static void lha_case_fixup(void);

char *lha_open(const char *filename) {
  char options[]= PREFIX TEMPLATE;
  pid_t child;
  int status;

  lha_close();
  strcpy(temp, TEMPLATE); /* Must be replaced each time */
  mktemp(temp); /* safe in this context */
  if (mkdir(temp, 0700) != 0) {
    fprintf(stderr, _("Couldn't create \"%s\": %s\n"), temp, strerror(errno));
    gnome_app_message(GNOME_APP(app),
                        _("Unable to create temporary directory"));
    return NULL;
  }
  child = fork();
  if (child < 0) {
    gnome_app_message(GNOME_APP(app), _("Unable to run external programs"));
  } else if (child) {
    waitpid(child, &status, 0);
    package_open= 1;
    if (WEXITSTATUS(status) == 182) {
      gnome_app_message(GNOME_APP(app), _("You need LHA to unpack .LZH files"));
      lha_close();
      return NULL;
    } else if (WEXITSTATUS(status) == 1) {
      gnome_app_message(GNOME_APP(app),
              _("Can't load this .LZH file - perhaps it is corrupted?"));
      lha_close();
      return NULL;
    } else if (WEXITSTATUS(status) != 0) {
      gnome_app_message(GNOME_APP(app), _("Unknown error handling .LZH file"));
    }
  } else {
    snprintf(options, sizeof(options), PREFIX "%s", temp);
    execlp("lha", "lha", options, filename, NULL);
    _exit(182); /* it is important that this is _exit() not exit() */
  }
  lha_case_fixup();
  return g_strdup(temp);
}

void lha_extend(const char *filename) {
  char options[]= PREFIX TEMPLATE;
  pid_t child;
  int status;

  /* Can only extend an existing opened archive */
  if (package_open == 0) return;

  child = fork();
  if (child < 0) {
    gnome_app_message(GNOME_APP(app), _("Unable to run external programs"));
  } else if (child) {
    waitpid(child, &status, 0);
    if (WEXITSTATUS(status) == 182) {
      gnome_app_message(GNOME_APP(app), _("You need LHA to unpack .LZH files"));
    } else if (WEXITSTATUS(status) == 1) {
      gnome_app_message(GNOME_APP(app),
                     _("Can't load this .LZH file - perhaps it is corrupted?"));
    } else if (WEXITSTATUS(status) != 0) {
      gnome_app_message(GNOME_APP(app), _("Unknown error handling .LZH file"));
    }
  } else {
    snprintf(options, sizeof(options), PREFIX "%s", temp);
    execlp("lha", "lha", options, filename, NULL);
    _exit(182);
  }
  lha_case_fixup();
}

void lha_close() {
  DIR *dir;
  struct dirent *dirent;
  
  if (package_open == 0) return;
  dir= opendir(temp);
  if (dir == NULL) return;
  chdir(temp);

  while ((dirent= readdir(dir))) {
    if (dirent->d_name[0] != '.') {
      if ( unlink(dirent->d_name) );
    }
  }

  chdir("..");
  closedir(dir);
  rmdir(temp);
  package_open= 0;
}

static void lha_case_fixup(void) {
  DIR *dir;
  struct dirent *dirent;
  char *filename;
  size_t length;

  if (package_open == 0) return;
  dir= opendir(temp);
  if (dir == NULL) return;
  chdir(temp);

  while ((dirent= readdir(dir))) {
    length= strlen(dirent->d_name);
    filename= strdup(dirent->d_name);
    if (length > 3 && !strcmp(filename + length - 3, "CNF")) {
      strcpy(filename + length - 3, "cnf");
      rename(dirent->d_name, filename);
    }
  }
}
