/*
    SPDX-License-Identifier: GPL-2.0-or-later

    SPDX-FileCopyrightText: 2003 Brian Thomas <thomas@mail630.gsfc.nasa.gov>
    SPDX-FileCopyrightText: 2004-2022 Umbrello UML Modeller Authors <umbrello-devel@kde.org>
*/

#include "codeblock.h"

#include "codedocument.h"
#include "debug_utils.h"

#include <QTextStream>

/**
 * Constructor.
 * @param doc    the documentation text
 * @param body   the body text
 * */
CodeBlock::CodeBlock(CodeDocument * doc, const QString & body)
  : TextBlock(doc, body),
    m_contentType(AutoGenerated)
{
}

/**
 * Empty Destructor.
 */
CodeBlock::~CodeBlock()
{
}

/**
 * Set the value of m_contentType
 * specifies whether the content (text) of this object was generated by the code
 * generator or was supplied by the user.
 * @param new_var the new value of m_contentType
 */
void CodeBlock::setContentType(ContentType new_var)
{
    m_contentType = new_var;
}

/**
 * Get the value of m_contentType
 * specifies whether the content (text) of this object was generated by the code
 * generator or was supplied by the user.
 * @return the value of m_contentType
 */
CodeBlock::ContentType CodeBlock::contentType() const
{
    return m_contentType;
}

/**
 * Save the XMI representation of this object.
 * @param writer QXmlStreamWriter serialization target
 */
void CodeBlock::saveToXMI(QXmlStreamWriter& writer)
{
    writer.writeStartElement(QStringLiteral("codeblock"));
    // set attributes
    setAttributesOnNode(writer);
    writer.writeEndElement();
}

/**
 * Set attributes of the node that represents this class
 * in the XMI document.
 * @param writer QXmlStreamWriter serialization target
 */
void CodeBlock::setAttributesOnNode(QXmlStreamWriter& writer)
{
    // call super-class
    TextBlock::setAttributesOnNode(writer);
    // local attributes
    if (m_contentType != AutoGenerated)
        writer.writeAttribute(QStringLiteral("contentType"), QString::number(contentType()));
}

/**
 * Load params from the appropriate XMI element node.
 * @param root   the starting point to load from
 */
void CodeBlock::loadFromXMI(QDomElement & root)
{
    setAttributesFromNode(root);
}

/**
 * Set the class attributes of this object from
 * the passed element node.
 * @param elem   the xmi element from which to load
 */
void CodeBlock::setAttributesFromNode(QDomElement & elem)
{
    // set attributes from the XMI in super-class
    TextBlock::setAttributesFromNode(elem);
    // set local fields now
    setContentType(((ContentType) elem.attribute(QStringLiteral("contentType"), QStringLiteral("0")).toInt()));
}

/**
 * Set the class attributes from a passed object.
 * @param obj   text block from which the attributes are taken
 */
void CodeBlock::setAttributesFromObject(TextBlock * obj)
{
    TextBlock::setAttributesFromObject(obj);
    CodeBlock * cb = dynamic_cast<CodeBlock*>(obj);
    if (cb)
        setContentType(cb->contentType());
}

/**
 * Return a string representation of ContentType.
 *
 * @param  val  the enum value of the ContentType
 * @return      the string representation of the enum
 */
QString CodeBlock::enumToString(const ContentType& val)
{
    if (val == AutoGenerated) {
        return QStringLiteral("AutoGenerated");
    }
    else {
        return QStringLiteral("UserGenerated");
    }
}

QDebug operator<<(QDebug str, const CodeBlock& obj)
{
    str.nospace() << "CodeBlock: " << CodeBlock::enumToString(obj.contentType())
        << ", ..." << static_cast<TextBlock*>(const_cast<CodeBlock*>(&obj));
    return str.space();
}
