// ====================================================================
// Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
// Copyright (C) 2013 - Scilab Enterprises
//
// This file must be used under the terms of the CeCILL.
// This source file is licensed as described in the file COPYING, which
// you should have received as part of this distribution.  The terms
// are also available at
// http://www.cecill.info/licences/Licence_CeCILL_V2.1-en.txt
// =====================================================================

// Builds a Java package (JAR) from a set of Java sources (see SEP #116)
// - jarFilePath: is the JAR target file path
// - packageNames: names of packages to archive in the JAR
// - sourcePaths: paths to directories containing Java sources
// - classPaths (optional): paths to dependencies (JARs or directories)
// - manifestFilePath (optional) : file path to manifest
function ilib_build_jar(jarFilePath, packageNames, sourcePaths, classPaths, manifestFilePath)

    // Returns all the java source files contained in a path tree
    function javaFilePaths = findJavaFiles(path, javaFilePaths)
        fileNames = listfiles(path)';
        if (isempty(fileNames)) then
            return;
        end
        filePaths = fullfile(path, fileNames);

        // First, explore sub directories
        dirPaths = filePaths(find(isdir(filePaths)));
        for i = 1:size(dirPaths, "*")
            javaFilePaths = [javaFilePaths; findJavaFiles(dirPaths(i), [])];
        end

        // Then add Java files of that directory
        dirJavaFilePaths = filePaths(find(fileext(filePaths) == ".java"));
        javaFilePaths = [javaFilePaths; dirJavaFilePaths'];
    endfunction

    // Returns the JIMS build root path
    // It is the JIMS/bin folder in TMPDIR
    function jimsBuildPath = getJimsBuildPath()
        jimsBuildPath = fullfile(TMPDIR, "JIMS/bin");
    endfunction

    // Returns the package compilation path
    // It is the 'deepest' directory containing the built classes
    // if package is 'com.foo.package' the returned path will be:
    //   TMPDIR/JIMS/bin/com/foo/package
    function packageCompilePath = getPackageCompilePath(packageName)
        packageSubPath = strsubst(packageName, ".", filesep());
        packageCompilePath = fullfile(getJimsBuildPath(), packageSubPath);
    endfunction

    // Returns the jar root input path
    // It is the directory that will be jar-ed
    // if jar is 'com.foo.package' the returned path will be:
    //   TMPDIR/JIMS/bin/com.foo.package
    function jarInputRootPath = getJarInputRootPath(jarName)
        jarInputRootPath = fullfile(getJimsBuildPath(), "_jar_" + jarName);
        if isdir(jarInputRootPath) then
            removedir(jarInputRootPath);
        end
        mkdir(jarInputRootPath);
    endfunction

    // Returns the package path in the jar input path
    // It is the directory where the classes will be copied
    // if package is 'com.foo.package' the returned path will be:
    //   TMPDIR/JIMS/com.foo.package/bin/com/foo/package
    function jarInputPackagePath = getJarPackagePath(jarInputRootPath, packageName)
        if ~isempty(packageName) then
            packagePath = strsubst(packageName, ".", filesep());
            jarInputPackagePath = fullfile(jarInputRootPath, packagePath);
            mkdir(jarInputPackagePath);
        else
            jarInputPackagePath = jarInputRootPath;
        end
    endfunction

    // Add header instructions for loader and cleaner script (license info, change dir, etc..)
    function addHeaderToScript(scriptName, fd)
        mputl("// This file is released under the 3-clause BSD license. See COPYING-BSD.", fd);
        mputl("// Generated by builder.sce: Please, do not edit this file.", fd);
        mputl("// ------------------------------------------------------", fd);
        mputl("curdir = pwd();", fd);
        mputl(msprintf("scriptdir = get_file_path(''%s'');", scriptName), fd);
        mputl(msprintf("chdir(scriptdir);"), fd);
        mputl("// ------------------------------------------------------", fd);
    endfunction

    // Add footer instructions for loader and cleaner script (restore dir, etc..)
    function addFooterToScript(fd)
        mputl("// ------------------------------------------------------", fd);
        mputl("chdir(curdir);", fd);
        mputl("clear curdir;", fd);
    endfunction

    // Creates a loader script (loader.sce)
    // Does the javaclasspath on the created jar file
    function createLoaderScript(loaderScriptName, jarFilePath)
        fd = mopen(loaderScriptName, "wt");
        addHeaderToScript(loaderScriptName, fd);
        mputl(msprintf("jarFilePath = fullfile(scriptdir, ''%s'');", jarFilePath), fd);
        mputl("javaclasspath(fullpath(jarFilePath));", fd);
        addFooterToScript(fd);
        mclose(fd);
    endfunction

    // Creates a cleaner script (cleaner.sce)
    // Deletes the loader script and jar file
    function createCleanerScript(cleanerScriptName, loaderScriptName, jarFilePath)
        fd = mopen(cleanerScriptName, "wt");
        addHeaderToScript(cleanerScriptName, fd);
        mputl(msprintf("if fileinfo(''%s'') <> [] then", loaderScriptName), fd');
        mputl(msprintf("    mdelete(''%s'');", loaderScriptName), fd);
        mputl("end", fd);
        mputl("// ------------------------------------------------------", fd);
        mputl(msprintf("jarFilePath = fullfile(scriptdir, ''%s'');", jarFilePath), fd);
        mputl("if fileinfo(jarFilePath) <> [] then", fd);
        mputl("    mdelete(jarFilePath);", fd);
        mputl("end", fd);
        addFooterToScript(fd);
        mclose(fd);
    endfunction


    // ilib_build_jar body

    // ilib_build_jar needs Java, it is not usable in NWNI mode
    if (getscilabmode() == "NWNI")
        error(msprintf(_("%s: function not available in NWNI mode.\n"), "ilib_build_jar"));
        return;
    end

    // Check input arguments
    [lhs, rhs] = argn(0);
    if rhs < 3 then
        error(msprintf(_("%s: Wrong number of input argument(s): 3 to 5 expected.\n"), "ilib_build_jar"));
        return;
    end

    // Input argument 1: jar file path
    if type(jarFilePath) <> 10 then
        error(999, msprintf(_("%s: Wrong type for input argument #%d: A string expected.\n"), "ilib_build_jar", 1));
    end
    if size(jarFilePath, "*") <> 1 then
        error(999, msprintf(_("%s: Wrong size for input argument #%d: A string expected.\n"), "ilib_build_jar", 1));
        return;
    end

    // Input argument 2: package names
    if rhs > 2 then
        if type(packageNames) <> 10 then
            error(999, msprintf(_("%s: Wrong type for input argument #%d: A matrix of strings expected.\n"), "ilib_build_jar", 2));
            return;
        end
    end

    // Input argument 3: source paths
    if type(sourcePaths) <> 10 then
        error(999, msprintf(_("%s: Wrong type for input argument #%d: A matrix of strings expected.\n"), "ilib_build_jar", 3));
        return;
    end

    // Input argument 4 (optional): class paths
    if rhs > 3 then
        if type(classPaths) <> 10 then
            error(999, msprintf(_("%s: Wrong type for input argument #%d: A matrix of strings expected.\n"), "ilib_build_jar", 4));
            return;
        end
    else
        classPaths = "";
    end

    // Input argument 5 (optional): manifest file path
    if rhs > 4 then
        if type(manifestFilePath) <> 10 then
            error(999, msprintf(_("%s: Wrong type for input argument #%d: A matrix of strings expected.\n"), "ilib_build_jar", 5));
            return;
        end
    else
        manifestFilePath = "";
    end

    [jarDir, jarName] = fileparts(jarFilePath);

    if (ilib_verbose() <> 0) then
        mprintf(_("   Building JAR library %s\n"), jarName + ".jar");
    end

    // Create a directory for jar creation
    jarInputRootPath = getJarInputRootPath(jarName);
    if ~isdir(jarInputRootPath) then
        error(msprintf(_("Cannot create jar build dir %s"), jarInputRootPath));
        return;
    end

    // Dependencies
    if ~isempty(classPaths) then
        if (ilib_verbose() == 2) then
            mprintf(_("   Add dependency class paths:\n%s\n"), classPaths);
        end
        javaclasspath(classPaths);
    end

    nbPackages = size(packageNames, "*");
    for i = 1:nbPackages
        packageName = packageNames(i);
        if (ilib_verbose() == 2) then
            mprintf(_("   Build package %s\n"), packageName);
        end

        // Delete each package compilation directory if exists
        packageCompilePath = getPackageCompilePath(packageName);
        if isdir(packageCompilePath) then
            removedir(packageCompilePath);
        end

        // Find all Java sources for that package and compile
        sourcePath = sourcePaths(i);
        javaFilePaths = findJavaFiles(sourcePath, []);
        if javaFilePaths <> [] then
            if (ilib_verbose() == 2) then
                mprintf(_("   Compiling source files:\n"));
                disp(javaFilePaths);
            elseif (ilib_verbose() == 1) then
                mprintf(_("   Compiling Java sources in %s\n"), sourcePath);
            end
            jcompile(javaFilePaths);
        else
            if (ilib_verbose() <> 0) then
                warning(msprintf(_("No Java sources in %s to compile for package %s"), sourcePath, packageName));
            end
        end

        // Copy package compiled classes ...
        packageCompilePath = getPackageCompilePath(packageName);
        if isdir(packageCompilePath) then
            // ... to its location in JAR
            jarInputPackagePath = getJarPackagePath(jarInputRootPath, packageName);
            if ~isdir(jarInputPackagePath) then
                error(msprintf(_("Cannot create jar package directory %s"), jarInputRootPath));
            end

            if (ilib_verbose() == 2) then
                mprintf(_("   Copying compiled package from %s to %s\n"), packageCompilePath, jarInputPackagePath);
            end
            copyfile(packageCompilePath, jarInputPackagePath);
        else
            if (ilib_verbose() <> 0) then
                warning(msprintf(_("Cannot find compilation directory %s for package %s"), packageCompilePath, packageName));
            end
        end
    end

    // Delete target jar if already exists
    if isfile(jarFilePath) then
        deletefile(jarFilePath);
    end

    // Create jar
    if (ilib_verbose() <> 0) then
        mprintf(_("   Creating JAR archive %s\n"), jarFilePath);
    end
    jcreatejar(jarFilePath, jarInputRootPath, "", manifestFilePath);
    if ~isfile(jarFilePath) then
        error(msprintf(_("Cannot create JAR file %s"), jarFilePath));
    end

    // Creates script
    jarFileRelativePath = getrelativefilename(pwd(), jarFilePath);
    if (ilib_verbose() == 2) then
        mprintf(_("   Creating scripts for JAR relative path %s\n"), jarFileRelativePath);
    end

    // Creates loader script
    loaderScriptName = "loader.sce";
    if (ilib_verbose() <> 0) then
        mprintf(_("   Create loader script for Java %s\n"), loaderScriptName);
    end
    createLoaderScript(loaderScriptName, jarFileRelativePath);

    // Creates cleaner script
    cleanerScriptName = "cleaner.sce";
    if (ilib_verbose() <> 0) then
        mprintf(_("   Create cleaner script for Java %s\n"), cleanerScriptName);
    end
    createCleanerScript(cleanerScriptName, loaderScriptName, jarFileRelativePath);
endfunction
