/* 
 * Copyright (c) 2009, 2013, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "mforms/mforms.h"
#include "wf_view.h"
#include "wf_textentry.h"

#include "ConvUtils.h"

using namespace MySQL::Forms;

//--------------------------------------------------------------------------------------------------

bool TextEntryImpl::create(mforms::TextEntry *self, mforms::TextEntryType type)
{
  TextEntryImpl ^entry= gcnew TextEntryImpl(self);

  if (entry != nullptr)
  {
    TextBox ^tb = ViewImpl::create<TextBox>(self, entry);
    tb->TextChanged += gcnew EventHandler(entry, &TextEntryImpl::TextChanged);
    tb->KeyDown += gcnew KeyEventHandler(entry, &TextEntryImpl::KeyDown);
    tb->GotFocus += gcnew EventHandler(entry, &TextEntryImpl::GotFocus);
    tb->LostFocus += gcnew EventHandler(entry, &TextEntryImpl::LostFocus);
    tb->Multiline= false;
    tb->AutoSize = false;

    entry->_placeholder_color = SystemColors::GrayText;
    entry->_text_color = tb->ForeColor;

    switch (type)
    {
    case mforms::NormalEntry:
      break;
    case mforms::PasswordEntry:
      tb->PasswordChar= '*';
      break;
    case mforms::SearchEntry:
      break;
    }

    tb->Size = Size(100, tb->PreferredSize.Height); // DefaultSize is not accessible here. 
    return true;
  }

  return false;
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_text(mforms::TextEntry *self, const std::string &text)
{
  TextEntryImpl^ cb= (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
  {
    TextBox^ box= cb->get_control<TextBox>();
    box->Text= CppStringToNative(text);

    // Make sure the text box is never made taller than their preferred height.
    Size size = box->PreferredSize;
    size.Width = box->MaximumSize.Width;
    box->MaximumSize = size;

    if (!text.empty())
    {
      if (!cb->_has_real_text)
        cb->LostFocus(box, nullptr);
      cb->_has_real_text = true;
    }
    else
    {
      if (cb->_has_real_text)
        cb->GotFocus(box, nullptr);
      cb->_has_real_text = false;
    }
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_placeholder_text(mforms::TextEntry *self, const std::string &text)
{
  TextEntryImpl^ cb = (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
  {
    cb->_placeholder = CppStringToNative(text);
    if (!cb->get_control<TextBox>()->Focused && !cb->_has_real_text)
      cb->LostFocus(cb->get_control<TextBox>(), nullptr);
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_placeholder_color(mforms::TextEntry *self, const std::string &color)
{
  TextEntryImpl^ text_entry = (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (text_entry != nullptr)
  {
    text_entry->_placeholder_color = System::Drawing::ColorTranslator::FromHtml(CppStringToNative(color));
    text_entry->LostFocus(text_entry->get_control<TextBox>(), nullptr);
  }
}

//--------------------------------------------------------------------------------------------------

std::string TextEntryImpl::get_text(mforms::TextEntry *self)
{
  TextEntryImpl^ cb= (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr && cb->_has_real_text)
    return NativeToCppString(cb->get_control<TextBox>()->Text);

  return "";
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_max_length(mforms::TextEntry *self, int length)
{
  TextEntryImpl^ cb= (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
    cb->get_control<TextBox>()->MaxLength = length;
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::TextChanged(Object^ sender, EventArgs^ args)
{
  if (_changing_text)
    return;

  TextBox^ tb = (TextBox^) sender;

  _has_real_text = tb->Text->Length > 0;

  if (tb->Tag != nullptr)
  {
    mforms::TextEntry* box = ViewImpl::get_backend_control<mforms::TextEntry>(tb);
    if (box != NULL)
      box->callback(); 
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::KeyDown(Object^ sender, KeyEventArgs^ args)
{
  TextBox^ tb= (TextBox^) sender;

  if (tb->Tag != nullptr)
  {
    mforms::TextEntry* box= ViewImpl::get_backend_control<mforms::TextEntry>(tb);
    if (box != NULL)
    {
      switch (args->KeyCode)
      {
      case Keys::Return:
        box->action(mforms::EntryActivate);
        args->SuppressKeyPress= true;
        break;
      case Keys::Up:
        if (args->Control)
            box->action(mforms::EntryCKeyUp);
        else
            box->action(mforms::EntryKeyUp);
        args->SuppressKeyPress= true;
        break;
      case Keys::Down:
        if (args->Control)
            box->action(mforms::EntryCKeyDown);
        else
            box->action(mforms::EntryKeyDown);
        args->SuppressKeyPress= true;
        break;
      }
    }
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::GotFocus(Object^ sender, EventArgs^ args)
{
  if (!_has_real_text)
  {
    TextBox ^tb = (TextBox^)sender;
    tb->ForeColor = _text_color;
    _changing_text = true;
    tb->Text = "";
    _changing_text = false;
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::LostFocus(Object^ sender, EventArgs^ args)
{
  if (!_has_real_text && _placeholder != nullptr)
  {
    TextBox ^tb = (TextBox^)sender;

    tb->ForeColor = _placeholder_color;
    _changing_text = true;
    tb->Text = _placeholder;   
    _changing_text = false;
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_read_only(mforms::TextEntry *self, bool flag)
{
  TextEntryImpl^ cb= (TextEntryImpl^) ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
    cb->get_control<TextBox>()->ReadOnly= flag;
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_bordered(mforms::TextEntry *self, bool flag)
{
  TextEntryImpl^ text_box = (TextEntryImpl^) ObjectImpl::FromUnmanaged(self);

  if (text_box != nullptr)
    text_box->get_control<TextBox>()->BorderStyle = flag ? BorderStyle::Fixed3D : BorderStyle::None;
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_front_color(String ^color)
{
  __super::set_front_color(color);
  _text_color = get_control<TextBox>()->ForeColor;
}

//--------------------------------------------------------------------------------------------------

TextEntryImpl::TextEntryImpl(mforms::TextEntry *text)
  : ViewImpl(text)
{
}

//--------------------------------------------------------------------------------------------------
