# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Tests the artifacts UI helper."""

from typing import ClassVar

from django.test import RequestFactory

from debusine.db.models import Artifact, FileInArtifact
from debusine.db.playground import scenarios
from debusine.test.django import TestCase
from debusine.web.views.ui.artifacts import ArtifactUI, FileInArtifactUI


class ArtifactUITests(TestCase):
    """Test the :py:class:`ArtifactUI` class."""

    scenario = scenarios.DefaultContext(set_current=True)

    artifact: ClassVar[Artifact]

    @classmethod
    def setUpTestData(cls) -> None:
        """Set up common test data."""
        super().setUpTestData()
        cls.artifact, _ = cls.playground.create_artifact()

    def ui(self) -> ArtifactUI:
        """Instantiate the UI helper to test."""
        factory = RequestFactory()
        request = factory.get("/")
        return self.artifact.ui(request)

    def test_place(self) -> None:
        place = self.ui().place
        self.assertEqual(place.title, f"Artifact {self.artifact.get_label()}")
        self.assertEqual(place.url, self.artifact.get_absolute_url())
        self.assertEqual(place.icon, "file")
        self.assertEqual(
            place.description, "debusine:test artifact debusine:test"
        )
        self.assertEqual(place.breadcrumb, "debusine:test")
        assert place.parent is not None
        self.assertEqual(
            place.parent.url, self.scenario.workspace.get_absolute_url()
        )


class FileInArtifactUITests(TestCase):
    """Test the :py:class:`FileInArtifactUI` class."""

    scenario = scenarios.DefaultContext(set_current=True)

    file: ClassVar[FileInArtifact]

    @classmethod
    def setUpTestData(cls) -> None:
        """Set up common test data."""
        super().setUpTestData()
        artifact, _ = cls.playground.create_artifact(
            {"hello.deb": b"contents"}, create_files=True
        )
        cls.file = FileInArtifact.objects.get(
            artifact=artifact, path="hello.deb"
        )

    def ui(self) -> FileInArtifactUI:
        """Instantiate the UI helper to test."""
        factory = RequestFactory()
        request = factory.get("/")
        return self.file.ui(request)

    def test_place(self) -> None:
        place = self.ui().place
        self.assertEqual(place.title, self.file.path)
        self.assertEqual(place.url, self.file.get_absolute_url())
        self.assertIsNone(place.icon)
        self.assertIsNone(place.description)
        self.assertEqual(place.breadcrumb, "hello.deb")
        assert place.parent is not None
        self.assertEqual(
            place.parent.url, self.file.artifact.get_absolute_url()
        )

    def test_place_incomplete(self) -> None:
        self.file.complete = False
        place = self.ui().place
        self.assertEqual(place.title, self.file.path + " (incomplete)")
        self.assertEqual(place.url, self.file.get_absolute_url())
        self.assertIsNone(place.icon)
        self.assertIsNone(place.description)
        self.assertEqual(place.breadcrumb, "hello.deb")
        assert place.parent is not None
        self.assertEqual(
            place.parent.url, self.file.artifact.get_absolute_url()
        )

    def test_place_raw(self) -> None:
        place = self.ui().place_raw
        self.assertEqual(place.title, self.file.path)
        self.assertEqual(place.url, self.file.get_absolute_url_raw())
        self.assertIsNone(place.icon)
        self.assertIsNone(place.description)
        self.assertEqual(place.breadcrumb, "hello.deb")
        assert place.parent is not None
        self.assertEqual(
            place.parent.url, self.file.artifact.get_absolute_url()
        )

    def test_place_raw_incomplete(self) -> None:
        self.file.complete = False
        place = self.ui().place_raw
        self.assertEqual(place.title, self.file.path + " (incomplete)")
        self.assertEqual(place.url, self.file.get_absolute_url_raw())
        self.assertIsNone(place.icon)
        self.assertIsNone(place.description)
        self.assertEqual(place.breadcrumb, "hello.deb")
        assert place.parent is not None
        self.assertEqual(
            place.parent.url, self.file.artifact.get_absolute_url()
        )

    def test_place_download(self) -> None:
        place = self.ui().place_download
        self.assertEqual(place.title, self.file.path)
        self.assertEqual(place.url, self.file.get_absolute_url_download())
        self.assertIsNone(place.icon)
        self.assertIsNone(place.description)
        self.assertEqual(place.breadcrumb, "hello.deb")
        assert place.parent is not None
        self.assertEqual(
            place.parent.url, self.file.artifact.get_absolute_url()
        )

    def test_place_download_incomplete(self) -> None:
        self.file.complete = False
        place = self.ui().place_download
        self.assertEqual(place.title, self.file.path + " (incomplete)")
        self.assertEqual(place.url, self.file.get_absolute_url_download())
        self.assertIsNone(place.icon)
        self.assertIsNone(place.description)
        self.assertEqual(place.breadcrumb, "hello.deb")
        assert place.parent is not None
        self.assertEqual(
            place.parent.url, self.file.artifact.get_absolute_url()
        )
