#
# Copyright 2001 by Object Craft P/L, Melbourne, Australia.
#
# LICENCE - see LICENCE file distributed with this software for details.
#
# 
# Base class for testing Albatross functionality.
#
# Sub-classes are expected to define a class variable "template_dir" which
# tells albatross which directory to search for the html template files.
#
# There is also a class variable, self.ctx, available to classes derived
# from this which contains the current Albatross execution context.
#
# $Id: albatross_test.py 8878 2007-03-14 05:38:02Z andrewm $

import unittest
import albatross

import sys
import os

class AlbatrossTestError(Exception):
    pass

class TestContext(albatross.AppContext, albatross.NameRecorderMixin,
                  albatross.StubSessionMixin):
    def __init__(self, app):
        albatross.AppContext.__init__(self, app)
        albatross.NameRecorderMixin.__init__(self)

    def current_url(self):
        return 'here'

class TestApp(albatross.TemplateLoaderMixin, albatross.Application):
    def __init__(self, base_url):
        albatross.TemplateLoaderMixin.__init__(self, base_url)
        albatross.Application.__init__(self, base_url)

class AlbatrossTestCase(unittest.TestCase):
    def setUp(self):
        if not hasattr(self, "template_dir"):
            raise AlbatrossTestError("%s: template_dir not defined in class" % self.__class__)

        # This is because we may not be imported from the current directory
        base_dir = os.path.dirname(sys.modules[__name__].__file__)
        template_dir = os.path.join(base_dir, self.template_dir)

        # We create a new app instance for each test because macros and lookups
        # are cached on the app object.
        app = TestApp(template_dir)
        self.ctx = TestContext(app)

    def tearDown(self):
        pass

    def load_template_file(self, template):
        """
        Template from file
        """
        self.tmpl = self.ctx.load_template(template)

    def load_template_str(self, template):
        """
        Template from string
        """
        self.tmpl = albatross.Template(self.ctx, '<str>', template)

    def gen_html(self):
        """
        Run the template and capture the generated HTML
        """
        self.ctx.push_content_trap()
        self.tmpl.to_html(self.ctx)
        return self.ctx.pop_content_trap()

    def expect_html(self, expect):
        result = self.gen_html()
        self.assertEqual(result, expect, 
                         'expected "%s", got "%s"' % (expect, result))


    def html_test(self, template_file, expect):
        """ 
        Given a template file name and a string of expected results,
        compare the rendered template with the results
        """
        self.load_template_file(template_file)
        self.expect_html(expect)

    def test_str_html(self, template, expect):
        """ 
        Given a template in a string, and a string of expected results, 
        compare the rendered template with the results
        """
        self.load_template_str(template)
        self.expect_html(expect)

    def expect_raise(self, loader, template, exception=None):
        if exception is None:
            exception = albatross.ApplicationError
        try:
            loader(template)
            result = self.gen_html()
        except exception:
            pass
        else:
            if hasattr(exception, '__name__'):
                name = exception.__name__
            else: 
                name = str(exception)
            raise self.failureException('%s not raised, got "%s"' % 
                                        (name, result))

    def test_raise(self, template_file, exception=None):
        self.expect_raise(self.load_template_file, template_file, exception)

    def test_str_raise(self, template, exception=None):
        self.expect_raise(self.load_template_str, template, exception)
