"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerText2VizRoutes = registerText2VizRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _error_handler = require("./error_handler");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const inputSchema = _configSchema.schema.string({
  maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT,
  validate(value) {
    if (!value || value.trim().length === 0) {
      return "can't be empty or blank.";
    }
  }
});
function registerText2VizRoutes(router, assistantService) {
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2VEGA,
    validate: {
      body: _configSchema.schema.object({
        input_question: inputSchema,
        input_instruction: _configSchema.schema.maybe(_configSchema.schema.string({
          maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT
        })),
        ppl: _configSchema.schema.string(),
        dataSchema: _configSchema.schema.string(),
        sampleData: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const agentConfigName = req.body.input_instruction ? _llm.TEXT2VEGA_WITH_INSTRUCTIONS_AGENT_CONFIG_ID : _llm.TEXT2VEGA_RULE_BASED_AGENT_CONFIG_ID;
      const response = await assistantClient.executeAgentByConfigName(agentConfigName, {
        input_question: req.body.input_question,
        input_instruction: req.body.input_instruction,
        ppl: req.body.ppl,
        dataSchema: req.body.dataSchema,
        sampleData: req.body.sampleData
      });
      let textContent = response.body.inference_results[0].output[0].result;

      // extra content between tag <vega-lite></vega-lite>
      const startTag = '<vega-lite>';
      const endTag = '</vega-lite>';
      const startIndex = textContent.indexOf(startTag);
      const endIndex = textContent.indexOf(endTag);
      if (startIndex !== -1 && endIndex !== -1 && startIndex < endIndex) {
        // Extract the content between the tags
        textContent = textContent.substring(startIndex + startTag.length, endIndex).trim();
      }

      // extract json object
      const jsonMatch = textContent.match(/\{.*\}/s);
      if (jsonMatch) {
        let result = JSON.parse(jsonMatch[0]);
        // sometimes llm returns {response: <schema>} instead of <schema>
        if (result.response) {
          result = JSON.parse(result.response);
        }
        // Sometimes the response contains width and height which is not needed, here delete the these fields
        delete result.width;
        delete result.height;

        // make sure $schema field always been added, sometimes, LLM 'forgot' to add this field
        result.$schema = 'https://vega.github.io/schema/vega-lite/v5.json';
        return res.ok({
          body: result
        });
      }
      return res.badRequest();
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: inputSchema
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const response = await assistantClient.executeAgentByConfigName(_llm.TEXT2PPL_AGENT_CONFIG_ID, {
        question: req.body.question,
        index: req.body.index
      });
      const result = JSON.parse(response.body.inference_results[0].output[0].result);
      return res.ok({
        body: result
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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