#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2020 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Rayman Origins
# send your bug reports to vv221@dotslashplay.it
###

script_version=20251114.6

PLAYIT_COMPATIBILITY_LEVEL='2.33'

GAME_ID='rayman-origins'
GAME_NAME='Rayman Origins'

ARCHIVE_BASE_HUMBLE_0_NAME='RaymanOrigins_windows.zip'
ARCHIVE_BASE_HUMBLE_0_MD5='f9e657afbfac436fe2aea720cdc72196'
ARCHIVE_BASE_HUMBLE_0_VERSION='1.0.32504-humble'
ARCHIVE_BASE_HUMBLE_0_SIZE='2400000'
ARCHIVE_BASE_HUMBLE_0_URL='https://www.humblebundle.com/store/rayman-origins'

ARCHIVE_BASE_GOG_0_NAME='setup_rayman_origins_1.0.32504_(18757).exe'
ARCHIVE_BASE_GOG_0_MD5='a1021275180a433cd26ccb708c03dde4'
ARCHIVE_BASE_GOG_0_TYPE='innosetup'
ARCHIVE_BASE_GOG_0_PART1_NAME='setup_rayman_origins_1.0.32504_(18757)-1.bin'
ARCHIVE_BASE_GOG_0_PART1_MD5='813c51f290371869157b62b26abad411'
ARCHIVE_BASE_GOG_0_SIZE='2500000'
ARCHIVE_BASE_GOG_0_VERSION='1.0.32504-gog18757'
ARCHIVE_BASE_GOG_0_URL='https://www.gog.com/game/rayman_origins'

CONTENT_PATH_DEFAULT_GOG='app'
CONTENT_PATH_DEFAULT_HUMBLE='game'
## rayman origins.exe requirements
CONTENT_LIBS_BIN_FILES='
binkw32.dll'
CONTENT_GAME_BIN_FILES='
rayman origins.exe'
CONTENT_GAME_DATA_FILES='
gamedata/bundle_pc.ipk
gamedata/*.bik'
## These documentation files are not provided by the Humble Bundle archive.
CONTENT_DOC_DATA_RELATIVE_PATH='support'
CONTENT_DOC_DATA_FILES='
*.pdf
*.txt'

WINE_PERSISTENT_DIRECTORIES='
users/${USER}/Documents/My Games/Rayman Origins'
WINE_REGISTRY_INIT_L10N_FR='
[HKEY_LOCAL_MACHINE\Software\Ubisoft\RaymanOrigins\Settings]
"Language"=dword:00000001
'
WINE_REGISTRY_INIT_L10N_EN='
[HKEY_LOCAL_MACHINE\Software\Ubisoft\RaymanOrigins\Settings]
"Language"=dword:00000000
'
## Without this native library, the game is stuck on a black screen instead of rendering the intro video.
## The following error is shown a couple times:
##
## 00ec:err:d3dcompiler:D3DCompile2 Shader log:
## 00ec:err:d3dcompiler:D3DCompile2     <anonymous>:93:10: W5300: Implicit truncation of matrix type.
## 00ec:err:d3dcompiler:D3DCompile2     <anonymous>:154:10: W5300: Implicit truncation of matrix type.
## 00ec:err:d3dcompiler:D3DCompile2     <anonymous>:190:10: W5300: Implicit truncation of matrix type.
## 00ec:err:d3dcompiler:D3DCompile2     <anonymous>:612:10: W5300: Implicit truncation of matrix type.
##
## (WINE 10.0)
WINE_WINETRICKS_VERBS='d3dcompiler_47'

APP_MAIN_EXE='rayman origins.exe'

PACKAGES_LIST='
PKG_BIN
PKG_L10N_FR
PKG_L10N_EN
PKG_DATA'

PKG_BIN_ARCH='32'
PKG_BIN_DEPENDENCIES_SIBLINGS='
PKG_L10N
PKG_DATA'

PKG_L10N_ID="${GAME_ID}-l10n"
PKG_L10N_FR_ID="${PKG_L10N_ID}-fr"
PKG_L10N_EN_ID="${PKG_L10N_ID}-en"
PKG_L10N_PROVIDES="
$PKG_L10N_ID"
PKG_L10N_FR_PROVIDES="$PKG_L10N_PROVIDES"
PKG_L10N_EN_PROVIDES="$PKG_L10N_PROVIDES"
PKG_L10N_FR_DESCRIPTION='French localisation'
PKG_L10N_EN_DESCRIPTION='English localisation'

PKG_DATA_ID="${GAME_ID}-data"
PKG_DATA_DESCRIPTION='data'

# Set extra tools required to extract data from the Humble Bundle archive.

REQUIREMENTS_LIST_HUMBLE='
dd
truncate
unshield'

# Load common functions.

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run default initialisation actions.

initialization_default "$@"

# Extract game data.

archive_extraction_default
case "$(current_archive)" in
	('ARCHIVE_BASE_HUMBLE'*)
		## Extract cabinet archives from from Rayman Origins.exe.
		ARCHIVE_INNER_1_PATH="${PLAYIT_WORKDIR}/gamedata/RaymondOrigins_windows/Rayman Origins.exe"
		information_archive_data_extraction 'Rayman Origins.exe'
		extraction_log=$(archive_extraction_log_path)
		{
			dd_output="${PLAYIT_WORKDIR}/gamedata/data1.hdr"
			printf 'dd if="%s" of="%s" bs=3 skip=7740856 count=11107\n' "$ARCHIVE_INNER_1_PATH" "$dd_output"
			dd if="$ARCHIVE_INNER_1_PATH" of="$dd_output" bs=3 skip=7740856 count=11107 2>&1

			dd_output="${PLAYIT_WORKDIR}/gamedata/data1.cab"
			printf 'dd if="%s" of="%s" bs=8 skip=2655105 count=247706\n' "$ARCHIVE_INNER_1_PATH" "$dd_output"
			dd if="$ARCHIVE_INNER_1_PATH" of="$dd_output" bs=8 skip=2655105 count=247706 2>&1

			## The extraction of data2.cab is done in two steps with big block size values.
			## This is a big file that would take a lot of time to get using a small block size.
			dd_output="${PLAYIT_WORKDIR}/gamedata/data2.cab"
			printf 'dd if="%s" of="%s" bs=567219 skip=41\n' "$ARCHIVE_INNER_1_PATH" "$dd_output"
			dd if="$ARCHIVE_INNER_1_PATH" of="$dd_output" bs=567219 skip=41 2>&1
			rm "$ARCHIVE_INNER_1_PATH"
			printf 'truncate --io-blocks --size=2125482963 "%s"\n' "$dd_output"
			truncate --io-blocks --size=2125482963 "$dd_output"
		} >> "$extraction_log"

		## Extract game data from data1.cab + data2.cab.
		ARCHIVE_INNER_2_PATH="${PLAYIT_WORKDIR}/gamedata/data1.hdr"
		ARCHIVE_INNER_2_TYPE='installshield'
		archive_extraction 'ARCHIVE_INNER_2'
		rm \
			"${PLAYIT_WORKDIR}/gamedata/data1.hdr" \
			"${PLAYIT_WORKDIR}/gamedata/data1.cab" \
			"${PLAYIT_WORKDIR}/gamedata/data2.cab"
	;;
esac

# Include game data.

content_inclusion_icons 'PKG_DATA'
content_inclusion_default

# Write launchers.

## Set localisation of first launch.
for package in \
	'PKG_L10N_FR' \
	'PKG_L10N_EN'
do (
	set_current_package "$package"
	## TODO: Compatibility level 2.34 adds context support to WINE_REGISTRY_INIT.
	WINE_REGISTRY_INIT=$(context_value 'WINE_REGISTRY_INIT')
	# shellcheck disable=SC2218
	wine_registry_script_write "$package"
) done
wine_registry_script_write() { return 0; }
## A dummy value is set to trigger the integration of the registry script in the generated launcher.
WINE_REGISTRY_INIT='whatever'

## Install shipped libraries.
wineprefix_init_custom() {
	cat <<- EOF
	# Install shipped libraries.
	ln --force --symbolic $(path_libraries)/* "\${PATH_PREFIX}"
	EOF
}

launchers_generation

# Build packages.

packages_generation
case "$(messages_language)" in
	('fr')
		lang_string='version %s :'
		lang_fr='française'
		lang_en='anglaise'
	;;
	('en'|*)
		lang_string='%s version:'
		lang_fr='French'
		lang_en='English'
	;;
esac
printf '\n'
printf "$lang_string" "$lang_fr"
print_instructions 'PKG_DATA' 'PKG_BIN' 'PKG_L10N_FR'
printf "$lang_string" "$lang_en"
print_instructions 'PKG_DATA' 'PKG_BIN' 'PKG_L10N_EN'

# Clean up.

working_directory_cleanup

exit 0
